"fn.deform.estparam"<-
function(corrmat, geogsiteloc, sta.names = NULL, scalingloc = 1, variog, 
	initvariogpar, lamBE.S, sitesfix = c(1, 2), bindir = "", just.read = F
	)
{
# ===========================================================================
# Function to estimate the D-plane coordinates and variogram parameters
# for the deformation approach to modeling non-stationary spatial correlation
# Sampson and Guttorp, 1992; Guttorp and Sampson, 1994
# This code calls fortran code which uses a penalized weighted least squares
# estimation approach; 
# cf., Meiring, Guttorp and Sampson, 1997, Interface proceedings
# This function assumes that there are repeated observations from each of
# N monitoring locations in space, and that these observations have been
# used to estimate a NxN correlation matrix - which is supplied to this 
# function as corrmat.
# ===========================================================================
# REQUIRED arguments:
# ===================
#
# corrmat       : N by N correlation matrix 
#  
# geogsiteloc   : N by 2 matrix of site locations in same order as 
#                 rows of corrmat 
# 
# variog        : Form of D-plane isotropic variogram. 
#                 Initial choices: 
#                 "Exp" for exponential, ``Gaussian'' for Gaussian, 
#                 "Matern" for Matern.
# 
# initvariogpar : Initial variogram parameters for optimization.
#                 For the exponential and Gaussian variograms, the order 
#                 should be 
#                 c( nugget, parameter in the exponent ).
#
#                 For the Matern variogram, a vector of length three is
#                 required. The order of the parameters should be
#                 c( nugget, theta1, theta2 ) using the parameter notation of
#                 Handcock and Stein (1993) for theta1 and theta2.
#                
#                 Handcock and Stein (1993). A Bayesian analysis of kriging.
#                 Technometrics, 35: 403-410.
# 
# lamBE.S       : Weighting parameter for the bending energy in the penalized
#                 weighted least squares criterion.
# 
# ===========================================================================
# ===========================================================================
# OPTIONAL arguments:
# ===================
#
# scalingloc: Scalar indicating the number by which to divide each of the 
#             coordinates to aid computation of the bending energy matrix.
#             This should be changed if the function returns error messages
#             about a singular matrix in the step computing the bending 
#             energy matrix.
#             The default is 1.
#  
# sitesfix : Sites to fix at their geographic locations to constrain 
#            the rotation, scaling and translation of the D-plane.
#            The default is c(1,2).
# 
# sta.names : Vector of site indicators.
#             This is used for the deformation plot in fn.deform.plot.est. 
#             rep(1:nrow(geogsiteloc),1) will be used if this is null.
# ==========================================================================
# ============================================================================
# Compile the fortran code to create the executable NPmain.exe
#
# On the UW machines use the makefile
#
# "compilecode" sets the correct paths for the NCAR machines, but doesn't 
# work on student, so is currently commented out
#
# ============================================================================
# !compilecode
# ======================================================================
# Standardize the site locations, and compute the bending energy matrix
# ======================================================================
# =========================================================================
# Check variogram type to see if it is implemented
# =========================================================================
	if(variog == "Exp" | variog == "Exponential") {
		variogtype <- 1
	}
	if(variog == "Gaussian") {
		variogtype <- 2
	}
	if(variog == "Matern") {
		variogtype <- 3
	}
	if((variog != "Exp" & variog != "Exponential") & variog != "Gaussian" & 
		variog != "Matern") {
		error.message <- paste(
			"Error: current implementation is only for Exponential, Gaussian and ",
			"\nMatern D-plane variograms, please assign 'variog' as ",
			"\n'Exp' or 'Gaussian' or 'Matern'", sep = "")
		print(error.message)
		variogtype <- 0
		Dplane.crds <- NULL
		Dplane.variogpar <- NULL
		sampdisp <- NULL
	}
# ============================================================================
# Check initvariogpar to see if the length is correct for the chosen variogram
# 
# initvariogpar should be of length 2 for the exponential and Gaussian 
# variograms and of length 3 for the Matern variogram
# =============================================================================
	if(variogtype == 1 | variogtype == 2) {
		if(length(initvariogpar) != 2) {
			error.message <- paste(
				"Error: initvariogpar should be a vector of length 2 for",
				" \nthe Gaussian and exponential variograms", 
				sep = "")
			print(error.message)
			variogtype <- 0	
	# No further computation will be performed
			Dplane.crds <- NULL
			Dplane.variogpar <- NULL
			sampdisp <- NULL
		}
	}
	if(variogtype == 3) {
		if(length(initvariogpar) != 3) {
			error.message <- paste(
				"Error: initvariogpar should be a vector of length 3 for",
				"\nthe Matern variogram", sep = "")
			print(error.message)
			variogtype <- 0	
	# No further computation will be performed.
			Dplane.crds <- NULL
			Dplane.variogpar <- NULL
			sampdisp <- NULL
		}
	}
# =========================================================================
# Write all the needed data files for the Fortran executable code NPmain.exe
# ==========================================================================
	if(variogtype == 1 | variogtype == 2 | variogtype == 3) {
# --------------------------------------------------------------------
# Geographic locations: 
#  Standardize geogsiteloc, if necessary (default for scalingloc is 1)
#  Concatinate into one long vector
# --------------------------------------------------------------------
		gcrdsmat <- geogsiteloc/scalingloc	# Standardize site locations
		gcrds <- as.double(c(gcrdsmat[, 1], gcrdsmat[, 2]))	
	# ----------------------------------------------------------------------
# Bending energy matrix:     Fbenergy should be in this Splus directory, 
#                            as well as Fdist and FMPinv
# ----------------------------------------------------------------------
		BEPmat <- Fbenergy(gcrdsmat)
		BEMvect <- as.double(as.vector(BEPmat))	# -------------------
# Sample dispersions:
# -------------------
		sampdisp <- 2 * (1 - corrmat)
		disp <- as.double(sampdisp[row(sampdisp) < col(sampdisp)])	
	# -----------------------------------
# Write to data files for NPmain.exe:
# ----------------------------------- 
		if(!just.read) {
			write(gcrds, file = "Dcrds", ncolumns = 1, append = F)
			write(disp, file = "Ddisp", ncolumns = 1, append = F)
			write(as.double(initvariogpar), file = "Da0t0", 
				ncolumns = 1, append = F)
			write(BEMvect, file = "Dbem", ncolumns = 1, append = F)
			write(as.double(lamBE.S), file = "DlamBE", ncolumns = 1,
				append = F)
			write(variogtype, file = "Dmodel", ncolumns = 1, append
				 = F)
			write(sitesfix, file = "Dfixloc", ncolumns = 1, append
				 = F)	
	# ======================================================================
# Call the Fortran code to perform the estimation, save stdout in *.out
# ======================================================================
			unix("rm NPmain.out")	#
###
			cmd <- paste(bindir, "NPmain.exe > NPmain.out", sep = 
				"")
			unix(cmd)	###
		}
# =====================================================================
# Read in the results of the estimation
# =====================================================================
		Dplane.crds <- matrix(scan("NEWcrds"), ncol = 2, byrow = F)
		Dplane.variogpar <- scan("NEWa0t0")
	}
# End of if statement checking if variogtype = 1 or variogtype = 2 or
#                                 variogtype = 3
# ===========================================================================
# Return values in a form that corresponds to the input for 
# 'fn.deform.eval' and 'fn.deform.plot.est'
# ===========================================================================
	return(geogsiteloc, sta.names, scalingloc, variog, sampdisp, 
		initvariogpar, lamBE.S, sitesfix, Dplane.crds, Dplane.variogpar
		)
}
