# Thrown by WikiData classes
class WikiDataException(Exception): pass
class WikiWordNotFoundException(WikiDataException): pass
class WikiFileNotFoundException(WikiDataException): pass
class WikiDBExistsException(WikiDataException): pass

class NoPageAstException(Exception): pass

# For non-Windows systems
try:
    WindowsError
except NameError:
    class WindowsError(Exception): pass


class DbAccessError(Exception):
    """
    Base classes for read or write errors when acessing database
    where "database" also means wiki configuration and additional
    files.
    """
    def __init__(self, originalException):
        Exception.__init__(self, str(originalException))
        self.originalException = originalException
    
    def getOriginalException(self):
        return self.originalException

class DbReadAccessError(DbAccessError):
    """
    Impossible to read (and therefore also to write to) database
    """
    pass

class DbWriteAccessError(DbAccessError):
    """
    Impossible to write to database, reading may be possible
    """
    pass



class RenameWikiWordException(Exception):
    """
    Raised on problems with renaming multiple wikiwords at once.
    Constructed in 
    WikiDataManager.WikiDataManager.buildRenameSeqWithSubpages()
    """
    # Problems:
    # Multiple words should be renamed to same word
    PRB_RENAME_TO_SAME = 1
    # Word to rename to exist already
    PRB_TO_ALREADY_EXISTS = 2

    def __init__(self, affectedRenames):
        """
        affectedRenames -- list of tuples (fromWikiWord, toWikiWord, problem)
            where problem is one of the PRB_* constants of the class.
        """
        self.affectedRenames = affectedRenames
        
    def getAffectedRenames(self):
        return self.affectedRenames


    def getFlowText(self):
        """
        Return affectedRenames as multiple-line human readable text
        """
        # TODO Move definition outside (attn to i18n)
        PROBLEM_HR_DICT = {
                self.PRB_RENAME_TO_SAME: _(u"Multiple words rename to same word"),
                self.PRB_TO_ALREADY_EXISTS: _(u"Word already exists")
            }

        result = []
        for fromWikiWord, toWikiWord, problem in self.affectedRenames:
            result.append(u"%s -> %s: %s" % (fromWikiWord, toWikiWord,
                    PROBLEM_HR_DICT[problem]))
        
        return u"\n".join(result)




class InternalError(Exception): pass


class ExportException(Exception): pass
class ImportException(Exception): pass

# See Serialization.py
class SerializationException(Exception): pass
class VersioningException(Exception): pass

# See WikiDataManager.py. Thrown if requested handler for db backend isn't
#     available
class NoDbHandlerException(Exception): pass
class WrongDbHandlerException(Exception): pass
class DbHandlerNotAvailableException(Exception): pass
class UnknownDbHandlerException(Exception): pass


# See WikiDataManager.py. Thrown if requested handler for wiki language isn't
#     available
class UnknownWikiLanguageException(Exception): pass
class WrongWikiLanguageException(Exception): pass


class MissingConfigurationFileException(Exception): pass
class BadConfigurationFileException(Exception): pass
class LockedWikiException(Exception): pass


class NotCurrentThreadException(Exception): pass
class UserAbortException(Exception): pass

class DeadBlockPreventionTimeOutError(InternalError): pass

class BadFuncPageTagException(Exception): pass
