#ifndef THREADSTRUCTS_H
#define THREADSTRUCTS_H

#include "Attributes.h"
#include "ThreadEnums.h"

typedef unsigned long GroupId;          /* ThreadGroup Identifier */

  /* This structure allows for a linked list of group identifiers
     to be held in a thread. This may not really be a sensible approach,
     but we will leave it here for the moment.
     The question is how do we want to do group operations
     most frequently. Do we want find the the groups a thread is in
     or work on the elements of a group.
   */
typedef struct _ThreadGroupId {

  GroupId id;
  struct _ThreadGroupId *next;

} ThreadGroupId, *ThreadGroupList;


 /* We could have the EvalThread object itself in here, but one thread is not supposed
    to know about another and it is intended that we use the same methods to access
    other threads as an S user. In other words, we use messages. This avoids additional
    complexity of syncrhonizing access by a thread and the Evaluator Manager.
  */
typedef struct _ThreadGroupElement {
  s_index thread_id;
  struct _ThreadGroupElement *next;
 
} ThreadGroupElement, *ThreadGroup;


  /* For use in join */
typedef struct _TaskResult {
  s_object *expression; 
  s_object *value;
  struct _TaskResult *next;
} TaskResult, *TaskResultList;

typedef ThreadGroup ThreadList;


 /* This is the message that is sent to the manager and forwarded to another thread
    when the second thread is being requested to perform a task on behalf of the initial
    thread.
  */

typedef struct _EvalMessage {
    boolean answered;
    s_object *return_value;
    s_object *expression;     
    s_index target_thread;
    s_index source_thread;
    MUTEX *mtx;
    COND_VAR *cond_var;

    struct _EvalMessage *next;         /* Linked list */
} EvalMessage;

typedef struct _EvalMessage SThreadTask;


  /* This structure is a linked list of ThreadGroup elements
     which allows for easy maintanence of the list of elements
     in a group.
     ThreadGroups are interesting objects as they exist on the 
     S side also and should persist across  sessions as 
     descriptions of a collection of threads. However,
     the instance of the group does not persist across sessions
     and the threads disappear. 
     But we can use these two purposes simultaneously in the same
     object and store the pid and entry into the table of 
     groups in the S object.
   */

typedef struct {

  SCThreadAttributes attributes;  /* Attributes for this thread */
  Thread  thread;                 /* Primitive C level thread */

} CThread;


typedef 
struct _EvalThread_struct {

  CThread         *thread;          /* Corresponding C level thread that this extends. */

  s_evaluator   *evaluator;       /* Evaluator state/description */

  MUTEX           *task_queue_mtx;  /* These two fields are used by other threads */
  COND_VAR        *task_queue_cond; /* to pass expressions to this thread's evaluator */

  SThreadTask     *task_queue;      /* List of S expressions to be evaluated when finished
                                          with this thread's current "task"/expression 
                                     */
 
  long            queue_length;     /* Number of expressions waiting to be evaluated in 
                                       the expression queue 
                                     */

  s_object *frame;                  /* Default frame into which assignments are comitted */           
  s_object *value;                  /* The value of the last completed task for this thread. */

    /* We store the parent and children as keys in the thread table.
       This is horribly inefficient, but necessary to ensure synchronized access to the variables
       and complete control by the Evaluator Manager. For the most part, the inefficiency
       is circumvented by one-off lookups for a given expression or task.
     */
  s_index         self;             /* Index for this thread in the relevant hashtable */
  s_index         parent;           /* Parent identifier */
  
  MUTEX children_lock;
  ThreadGroup children;                /* The linked list of child threads spawned by this thread */
  int num_children;                    /* convenient way of getting number of children rather than
                                          traversing linked list. Usual problems of synchronization.
                                        */

  ThreadGroupList groups;              /* List of groups this thread is in */


   /* For some reason, if this is put under the definition of task_queue
      where _EvalMessage is defined, the self object becomes 0 for some 
      EvalThread objects and so send_task behaves differently. Specifically,
      S_self gets 0. Ok here.
    */
  SThreadTask *current_task;   /* Current task being operated on by this thread's evaluator */


   /* History list of tasks and return values for use in calls to join.
      The length of this list is governed by the thread attributes
      maxHistoryLength.
    */
  TaskResultList taskHistory;
  long taskHistoryLength;
  MUTEX task_history_lock;

} EvalThread;



 /*   This strucuture defines a table to contain thread objects.
      It extends the notion of a basic table which supports
      locking and notification throught the condition variable 
      and mutex.
  */
typedef struct {

 s_index_table   *table;
 unsigned long num_threads; /* Ever increasing count of the number 
                               of threads used in the process */

} ThreadTable;


typedef struct _EvalThreadElement {
  EvalThread *thread;
  struct _EvalThreadElement *next;
} EvalThreadElement, *EvalThreadList;


#define EVALUATOR_FRAME0(t)  (t)->evaluator->_thread_storage_frame
#define TOP_FRAME(evaluator)  (evaluator)->_thread_storage_frame

#endif
