
#include "apriori_entries.h"
#include "scan.h"
#include "tract.h"
#include "istree.h"
#include "symtab.h"
#include <stdlib.h>

#define BLKSIZE  256            /* block size for enlarging vectors */

APRIORI_API(APRIORI_ITEMSET*)
apriori_is_create()
{
  return (APRIORI_ITEMSET*)is_create();
}

APRIORI_API(void)
apriori_ist_delete(APRIORI_ISTREE* apriori_istree)
{
  ISTREE* istree = (ISTREE*)apriori_istree;
  ist_delete(istree);
}

APRIORI_API(void)
apriori_is_delete(APRIORI_ITEMSET* apriori_itemset)
{
  ITEMSET* itemset = (ITEMSET*)apriori_itemset;
  is_delete(itemset);
}

APRIORI_API(INT64)
apriori_is_cnt(APRIORI_ITEMSET* apriori_itemset)
{
  ITEMSET* itemset = (ITEMSET*)apriori_itemset;
  return is_cnt(itemset);
}

APRIORI_API(void)
apriori_read_transaction_start(APRIORI_ITEMSET* apriori_itemset)
{
  ITEMSET* itemset = (ITEMSET*)apriori_itemset;
  /* init items in the transaction */
  itemset->cnt = 0;
}

/* add item string to hash table, if not there. */
/* return 1 if ok, 0 if there was an error (memory error) */
APRIORI_API(INT64)
apriori_add_item_to_hash(APRIORI_ITEMSET* apriori_itemset, const char* item_string, INT64 add_new_item)
{
  ITEMSET* itemset = (ITEMSET*)apriori_itemset;
  /* look up the name in name/id map */
  ITEM *item = (ITEM*)nim_byname(itemset->nimap, item_string);
  if (!item) {
    /* don't want to add new items: ignore this item */
    if (!add_new_item) return 1;

    item = (ITEM*)nim_add(itemset->nimap, item_string, sizeof(ITEM));
    if (!item) {
      /* setError("out of memory reading items"); */
      return 0;
    }
    /* initialize the frequency counters */
    /* (occurrence and sum of t.a. sizes) */
    /* and set the appearance indicator */
    item->frq = item->xfq = 0;
    item->app = itemset->app;
  }
  return 1;
}

/* add item string to current transaction */
/* return 1 if ok, 0 if there was an error (memory error) */
APRIORI_API(INT64)
apriori_read_transaction_add_item(APRIORI_ITEMSET* apriori_itemset, const char* item_string, INT64 add_new_item)
{
  ITEMSET* itemset;
  INT64 ok;
  ITEM* item;
  INT64 size;
  INT64* vec;

  itemset = (ITEMSET*)apriori_itemset;

  ok = apriori_add_item_to_hash(apriori_itemset, item_string, add_new_item);
  if (!ok)
    return 0;

  /* look up item */
  item = (ITEM*)nim_byname(itemset->nimap, item_string);
  if (!item) {
    /* this must be a new item, and !add_new_item.  ignore this item */
    return 1;
  }

  /* expand item vector if it is full */
  size = itemset->vsz;
  if (itemset->cnt >= size) {
    size += (size > BLKSIZE) ? (size >> 1) : BLKSIZE;
    vec = (INT64*)realloc(itemset->items, (unsigned int)(size *sizeof(INT64)));
    if (!vec) {
      /* setError("out of memory expanding item vector"); */
      return 0;
    }
    /* set the new vector and its size */
    itemset->items = vec;
    itemset->vsz = size;
  }
  /* add item to item vector */
  itemset->items[itemset->cnt++] = item->id;
  return 1;
}

/* ends reading transaction */
/* sorts and uniquifies items, updates frequency counts */
/* returns number or items in transaction */
APRIORI_API(INT64)
apriori_read_transaction_end(APRIORI_ITEMSET* apriori_itemset)
{
  INT64 i;
  ITEMSET* itemset = (ITEMSET*)apriori_itemset;
  /* prepare the transaction */
  ta_sort(itemset->items, itemset->cnt);
  itemset->cnt = ta_unique(itemset->items, itemset->cnt);
  for (i = itemset->cnt; --i >= 0; ) {
    ITEM *item = (ITEM*)nim_byid(itemset->nimap, itemset->items[i]);
    /* count the item and sum the transaction sizes */
    item->frq += 1;
    /* sum the transaction sizes as an importance indicator */
    item->xfq += itemset->cnt;
  }

  return is_tsize(itemset);
}

/* add transaction to tree, updating counts */
APRIORI_API(void)
apriori_add_transaction_to_tree(APRIORI_ITEMSET* apriori_itemset, APRIORI_ISTREE* apriori_istree)
{
  ITEMSET* itemset = (ITEMSET*)apriori_itemset;
  ISTREE* istree = (ISTREE*)apriori_istree;
  ist_count(istree, is_tract(itemset), is_tsize(itemset));
}

APRIORI_API(INT64)
apriori_is_recode(APRIORI_ITEMSET* apriori_itemset, INT64 minfrq, INT64 sort_direction)
{
  ITEMSET* itemset;
  INT64 n;
  INT64* map;

  itemset = (ITEMSET*)apriori_itemset;
  /* --- filter, sort and recode items --- */
  map = (INT64*)malloc((unsigned int)(is_cnt(itemset) *sizeof(INT64)));
  if (!map) {
    /* setError("error allocating identifier map"); */
    return -1;
  }
  n = is_recode(itemset, minfrq, sort_direction, map);
  free(map);
  return n;
}

APRIORI_API(char)
apriori_is_getapp(APRIORI_ITEMSET* apriori_itemset, INT64 i)
{
  ITEMSET* itemset = (ITEMSET*)apriori_itemset;
  return (char)is_getapp(itemset, i);
}

APRIORI_API(APRIORI_ISTREE*)
apriori_ist_create(INT64 rule_support_both, INT64 cnt, INT64 supp, double conf, const char *apps)
{
  return (APRIORI_ISTREE*)ist_create((rule_support_both)? IST_BOTH : IST_BODY, cnt, supp, conf, apps);
}

APRIORI_API(void)
apriori_ist_setcnt(APRIORI_ISTREE *apriori_istree, INT64 item, INT64 cnt)
{
  ISTREE* istree = (ISTREE*)apriori_istree;
  ist_setcnt(istree, item, cnt);
}

APRIORI_API(INT64)
apriori_is_getfrq(APRIORI_ITEMSET* apriori_itemset, INT64 item)
{
  ITEMSET* itemset = (ITEMSET*)apriori_itemset;
  return is_getfrq(itemset, item);
}

APRIORI_API(void)
apriori_ist_settac(APRIORI_ISTREE *apriori_istree, INT64 cnt)
{
  ISTREE* istree = (ISTREE*)apriori_istree;
  ist_settac(istree, cnt);
}

APRIORI_API(INT64)
apriori_ist_check(APRIORI_ISTREE *apriori_istree, char *apps)
{
  ISTREE* istree = (ISTREE*)apriori_istree;
  return ist_check(istree, apps);
}

APRIORI_API(INT64)
apriori_ist_height(APRIORI_ISTREE *apriori_istree)
{
  ISTREE* istree = (ISTREE*)apriori_istree;
  return ist_height(istree);
}

APRIORI_API(INT64)
apriori_ist_addlvl(APRIORI_ISTREE *apriori_istree)
{
  ISTREE* istree = (ISTREE*)apriori_istree;
  return ist_addlvl(istree);
}

APRIORI_API(void)
apriori_ist_init(APRIORI_ISTREE *apriori_istree, INT64 minlen, INT64 arem, double minval)
{
  ISTREE* istree = (ISTREE*)apriori_istree;
  ist_init(istree, minlen, arem, minval);
}


APRIORI_API(INT64)
apriori_extract_rule(APRIORI_ISTREE *apriori_istree, APRIORI_ITEMSET* apriori_itemset,
		     INT64 *supp, double *conf, double *lift, double *aval,
		     INT64 *body_supp, INT64* head_supp, INT64* set_supp)
{
  ITEMSET* itemset = (ITEMSET*)apriori_itemset;
  ISTREE* istree = (ISTREE*)apriori_istree;
  INT64* set = is_tract(itemset);
  return ist_rule_ex(istree, set, supp, conf, lift, aval, 
		     body_supp, head_supp, set_supp);
}

APRIORI_API(const char*)
apriori_extract_rule_item(APRIORI_ITEMSET* apriori_itemset, INT64 rule_item)
{
  ITEMSET* itemset = (ITEMSET*)apriori_itemset;
  INT64* set = is_tract(itemset);
  const char* name = is_name(itemset, set[rule_item]);
  return name;
}

