/*
 IMLinearRegression.c implementation of interfaces for Linear Regression Model.
*/

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <memory.h>

#include "IMLinearRegression.h"

long IMCSRC_STDCALL IMinerLinearRegressionModel_isValid(const IMinerObject* pModel)
{
	if (!IMinerList_isValid(IMINER_LI_R_META_DATA(pModel))) return 0L;
	return IMinerVector_isValid(IMINER_LI_R_COEFFICIENTS(pModel));
}

/* alloc memory for type Linear Regression Model. */
long IMCSRC_STDCALL IMinerLinearRegressionModel_create(IMinerObject* pModel,  /* out: data object */
 IMinerObject* md,			    /* in:  meta data object */
 long nCoefficents,				/* in:  number of coefficients */
 double* pdCoefficents			/* in: if not NULL, elements of coefficents. */
)
{
	char *ppszMemberNames[] = {"metaData", "coefficients"};
	long nStatus=IMINER_SUCCESS;

	/* always have IMINER_LI_R_COEFFICIENTS_NUM+1 elements as list */
	nStatus = IMinerList_create(pModel, IMINER_LI_R_COEFFICIENTS_NUM+1, IMINER_MODE_MODEL); 
	if(nStatus != IMINER_SUCCESS)
		return nStatus;

	/* member: meta data */
	nStatus = IMinerList_clone(IMINER_LI_R_META_DATA(pModel), md);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;

	/* member: coefficients */
	nStatus = IMinerVector_create(IMINER_LI_R_COEFFICIENTS(pModel), nCoefficents, IMINER_MODE_DOUBLE, pdCoefficents);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;

	/*member 4 (special) : member names */
	nStatus = IMinerList_setNamesFromStrings(pModel, (const char**)ppszMemberNames);

	/* set the class ID */
	pModel->m_nMode = IMINER_MODE_MODEL;
	return nStatus;
}

/* free memory of type IMinerLinearRegressionModel. */
long IMCSRC_STDCALL IMinerLinearRegressionModel_destroy(IMinerObject* pModel )
{
	return IMinerVector_destroy(pModel);
}

/* write to stdout */
long IMCSRC_STDCALL IMinerLinearRegressionModel_print(const IMinerObject* pModel)
{
	long nStatus;

	printf("Meta Data:\n");
	if (IMinerMetaData_print(IMINER_LI_R_META_DATA(pModel))!=IMINER_SUCCESS) {
		return IMINER_BAD_INPUT_ARGS;
	}

	printf("Coefficients:\n");
	nStatus = IMinerVector_print(IMINER_LI_R_COEFFICIENTS(pModel));
	if(nStatus != IMINER_SUCCESS) return IMINER_BAD_INPUT_ARGS;

	return IMINER_SUCCESS;
}


/* predict cluster memberships */
long IMCSRC_STDCALL IMinerLinearRegressionModel_predict(IMinerObject* pdOutput,  /* out: output vector data of type double*/
 const IMinerObject* input,  /* in: input rectangular data */													 
 IMinerObject* pDescr,		 /* in: input description (if NULL, description will be 
                                    created from input data) */
 const IMinerObject* pModel  /* in: the model */
)
{
	long nStatus = IMINER_SUCCESS, nCoef, bIntercept;
	long i,j,k, nRows, nInputColumns;
	long outMode[] = {IMINER_MODE_DOUBLE};
	double dCoef;
	IMinerObject pInput, *pvCol=NULL;
	IMinerObject A, *pvCoefs;

	char *fit_name="PREDICT.fit", **columnNames;

	if(pdOutput==NULL || !IMinerDataSet_isValid(input) || !IMinerLinearRegressionModel_isValid(pModel))	
		return IMINER_BAD_INPUT_ARGS;

	/* convert input if needed*/
	nStatus = IMinerMetaData_InputConvert(&pInput, IMINER_LI_R_META_DATA(pModel), input, pDescr);
	if(nStatus != IMINER_SUCCESS) return IMINER_FAIL;

	pvCoefs		  = IMINER_LI_R_COEFFICIENTS(pModel);
	nInputColumns = IMINER_DATASET_NCOLUMNS(&pInput);
	nRows         = IMINER_DATASET_NROWS(&pInput);
	nCoef		  = IMINER_OBJECT_LENGTH(pvCoefs);

	bIntercept = (nCoef==nInputColumns+1);

	if(!(bIntercept && nCoef==nInputColumns+1) || (!bIntercept && nCoef!=nInputColumns))
		return IMINER_BAD_INPUT_ARGS;

	/* create a matrix double to store the input as double */
	nStatus = IMinerDoubleMatrix_create(&A, nRows, nCoef, NULL, NULL, NULL);
	if(nStatus != IMINER_SUCCESS) return IMINER_FAIL;

	/* Set the input data to the matrix */
	for(j=0L; j< nCoef; ++j) {
		if (bIntercept && j==0) {
			for(i=0L; i< nRows; ++i) IMINER_Aij(&A, i, j) = 1.0;
			j++;
		}

		k = ((bIntercept) ? j-1 : j);
		for(i=0L; i< nRows; ++i) {
			IMINER_Aij(&A, i, j) = IMinerDataSet_getNonStringValue(&pInput, k, i);
		}
	}

	/* create the output object of the same length number of rows */
	nStatus = IMinerDataSet_create(pdOutput, nRows, 1L, outMode);
	if(nStatus != IMINER_SUCCESS) return nStatus;
	columnNames = (char**)malloc(sizeof(char*));
	columnNames[0] = fit_name;
	nStatus = IMinerDataSet_setColumnNamesFromStrings(pdOutput, (const char**)columnNames);
	if(nStatus != IMINER_SUCCESS) return nStatus;
	free(columnNames);
	
	pvCol = IMINER_DATASET_COLUMN_PTR(pdOutput, 0);
	for(i=0L; i<nRows; ++i) IMINER_DOUBLE_VALUE(pvCol, i) = 0.0;

	/* Multiply the input matrix to the coefficents vector : v = c1*A1 + c2*A2 + ... + cn*An */
	for(j=0L; j<nCoef; ++j) {
		dCoef = IMINER_DOUBLE_VALUE(pvCoefs, j);
		for(i=0L; i<nRows; ++i)
			IMINER_DOUBLE_VALUE(pvCol, i) +=  dCoef * IMINER_Aij(&A, i, j);
	}

	IMinerObject_destroy(&pInput);
	IMinerObject_destroy(&A);

	return IMINER_SUCCESS;
}

