import java.util.Vector;

public class FamilyMember {

  /* This class is an example of creating and modifying a
     dynamic collection of instances using static methods.

     Example commands to use in S-PLUS:

     momId <- .JavaMethod("FamilyMember", "newFamilyMember", 
       "(Ljava/lang/String;Ljava/lang/String;)I", 
       "Sue", "Jones")

     dadId <- .JavaMethod("FamilyMember", "newFamilyMember", 
       "(Ljava/lang/String;Ljava/lang/String;)I", 
       "Tom", "Jones")

     sonId <- .JavaMethod("FamilyMember", "newFamilyMember", 
       "(Ljava/lang/String;Ljava/lang/String;)I", 
       "Skip", "Jones")

     .JavaMethod("FamilyMember", "setMother", "(II)Z", 
       sonId, momId)

     .JavaMethod("FamilyMember", "setFather", "(II)Z", 
       sonId, dadId)

     .JavaMethod("FamilyMember", "getInfo", 
       "(I)[Ljava/lang/String;", sonId)

     .JavaMethod("FamilyMember", "getInfo", 
       "(I)[Ljava/lang/String;", dadId)

  */

  // Track instances by keeping a static Vector of
  // instances.  We will add each FamilyMember object to
  // this Vector when it is created, and return its index
  // in this Vector as the key for accessing the object.

  static Vector members = new Vector();

  // Instance variables.  We get each person's first and
  // last name when the object is created.  Methods are then
  // used to specify their mother and father.  When the
  // person is specified as a mother or father, we know 
  // they have a child, which we also track.

  String firstName, lastName; 
  FamilyMember mother, father;
  Vector children = new Vector();

  /* Constructor */

  public FamilyMember (String first, String last){
    firstName = first;
    lastName = last;
  }

  /* Instance methods */
  /* Java routines would call these */

  // public methods to get names 

  public String getFirstName(){
    return firstName;
  }

  public String getLastName(){
    return lastName;
  }

  // public methods to set and get parents

  public void setMother(FamilyMember mom){
    mother = mom;
    mother.addChild(this);
  }

  public void setFather(FamilyMember dad){
    father = dad;
    father.addChild(this);
  }

  public FamilyMember getMother(){
    return mother;
  }

  public FamilyMember getFather(){
    return father;
  }

  // private method to add child when parent set
 
  private void addChild(FamilyMember kid){
    children.add(kid);
  }

  // public method to get children

  public Vector getChildren(){
    return children;
  }

  /* Static methods */
  /* S-PLUS would call these */

  // static method to create a family member and return 
  // an ID to track them

  public static int newFamilyMember(String first, 
      String last){
    FamilyMember newMember = new FamilyMember(first, last);

    // Add new instance to list of members
    members.add(newMember);

    // Use the position in the members vector as an ID
    return (members.size() -1);
  }

  // private method to check that ID in legal range

  private static boolean checkId(int id){
    boolean status = true;

    if (id < 0 || id > (members.size()-1)){
      // Could throw exception, but we'll just print a
      // message
      System.out.println("Error:  ID out of range");
      status = false;
    }
    return status;

  }

  // static methods to specify mother and father based on ID

  // The basic steps in these methods are: 
  //   1) Check that the ID is within the range of ID's.
  //   2) Get the object from the members Vector.
  //   3) Cast the object to a FamilyMember object.
  //   4) Apply the relevant non-static method to the object
  //
  // If the ID is out of range we return false.  Otherwise
  // we return true.

  public static boolean setMother(int personId, int momId){
    boolean status = true;
    if (checkId(personId) && checkId(momId)){
      FamilyMember person = 
        (FamilyMember) members.get(personId);
      FamilyMember mom = (FamilyMember) members.get(momId);
      person.setMother(mom);
    }
    else
      status = false;
    return status;
  }

  public static boolean setFather(int personId, int dadId){
    boolean status = true;
    if (checkId(personId) && checkId(dadId)){
      FamilyMember person = 
        (FamilyMember) members.get(personId);
      FamilyMember dad = (FamilyMember) members.get(dadId);
      person.setFather(dad);
    }
    else
      status = false;
    return status;
  }

  // static method to get information about a family member

  public static String [] getInfo(int id){
    if (!checkId(id))
      return new String [] {"Name: Unknown", 
        "Mother: Unknown", "Father: Unknown"};

    FamilyMember person = (FamilyMember) members.get(id);
    FamilyMember mom = person.getMother();
    FamilyMember dad = person.getFather();
    Vector kids = person.getChildren();

    String [] info = new String [3 + kids.size()];

    info[0] = "Name: " + person.getFirstName() + " " +
      person.getLastName();

    if (mom==null)
      info[1] = "Mother: Unknown";
    else
      info[1] = "Mother: " + mom.getFirstName() + " " + 
        mom.getLastName();

    if (dad==null)
      info[2] = "Father: Unknown";
    else
      info[2] = "Father: " + dad.getFirstName() + " " + 
        dad.getLastName();

    for (int i = 0; i < kids.size(); i++){
      FamilyMember aKid = (FamilyMember) kids.get(i);
      if (!(aKid==null)){
        info[3+i] = "Child: " + aKid.getFirstName() + " " +
          aKid.getLastName();
      }
    }

    return info;
  }
}
