import com.insightful.splus.*;
import com.insightful.splus.util.ResultSetUtilities;
import java.sql.*;
import java.io.*;
import java.math.BigDecimal;

/**
 * Example on how to use ResultSetUtilities to transfer data from a ResultSet
 * into a data.frame in S-PLUS.  
 * <P>
 * This class definition is very long as we have locally
 * defined implementations for the ResultSet and ResultSetMetaData interfaces
 * in order to make the example self-contained.  In typical usage the
 * ResultSet is obtained using other JDBC methods and the only code needed
 * is the code in the ResultSetExample() constructor.
 */

public class ResultSetExample {

    /**
     * Transfer a result set to S-PLUS.  Print the object to show it's there.
     */

    public ResultSetExample() {
	}

	public void splusExample() {
		// Example of accessing the data from S-PLUS.  Uses the
		// S-PLUS function javaGetResultSet() to create an S-PLUS
		// data.frame containing the data in the ResultSet.

		LocalResultSet theSet = new LocalResultSet();
	
		// Register the ResultSet with ResultSetUtilities.
	
		ResultSetUtilities.register("mySetId", theSet);
	
		// Save ResultSet data in S-PLUS as the object "test.df".
	
		// Specify client = T so ResultSet will be found in this JVM
		// when in client/server mode.
	
		System.out.println("Transferring the ResultSet to test.df in S-PLUS.");
	
		String expr = "assign(\"test.df\", javaGetResultSet(\"mySetId\", client = T), where = 1)\n";
	
	
		// NOTE: If using S-PLUS Server in client/server mode, set the
		// relevant properties before doing the SplusUserApp.eval().  For
		// example:
		//
		//    System.setProperty("splus.client.mode", "true");
		//    System.setProperty("splus.server.host", "myServerName");
		//    System.setProperty("splus.server.username", "myUserName");
		//    System.setProperty("splus.server.password", "myPassword");

		SplusDataResult result = SplusUserApp.eval(expr);
		if (result.hasError())
		    System.out.println(result.getError());
		if (result.hasWarning()) {
		    String [] warnings = result.getWarning();
		    for (int i = 0; i < warnings.length; i++)
			System.out.println(warnings[i]);
		}
	
		// Query S-PLUS to print test.df to show it's there
		
		System.out.println("Printing test.df in S-PLUS.");
	
		expr = "if (exists(\"test.df\")) print(test.df) else print(\"test.df does not exist\")\n";
	
		result = SplusUserApp.eval(expr, true, false, false, false, false);
		System.out.println(result.getOutput());
	
    }

	public void nosplusExample() {
		// Example using Java to retrieve ResultSet values.  Useful
		// for checking that the ResultSetUtilities Java code is working.

		LocalResultSet theSet = new LocalResultSet();
	
		// Register the ResultSet with ResultSetUtilities.
	
		ResultSetUtilities.register("mySetId", theSet);
	
		try {
			Object[] obj = ResultSetUtilities.getData("mySetId");
	
			String[] colNames = (String[]) obj[0];
			for (int i = 0; i < colNames.length; i++)
				System.out.println("colnames[" + i + "]=" + colNames[i]);
			
			String[] colTypeNames = (String[]) obj[1];
			int stringCol = -1;
			for (int i = 0; i < colTypeNames.length; i++) {
				if (colTypeNames[i].equals("String")) {
					stringCol = i;
					break;
				}
			}
			System.out.println("stringCol=" + stringCol);
			if (stringCol != -1) {
				Object[] colValues = (Object[]) obj[3];
				String[] arrStr = (String[]) colValues[stringCol];
				
				for (int i = 0; i < arrStr.length; i++)
					System.out.println("arrStr[" + i + "]=" + arrStr[i]);
			}
		}
		catch (SQLException sqle)
		{
			System.out.println("SQLException: " + sqle.getMessage());
		}
	
    }

    public static void main(String [] args) {
		
		ResultSetExample rse = new ResultSetExample();
		
		if (args.length > 0 && args[0].equals("nosplus"))
			rse.nosplusExample();
		else
			rse.splusExample();

		System.exit(0);
    }

    /**
     * Class implementing ResultSet to provide a self-contained example.
     * In actual usage, the ResultSet will be created using other JDBC classes.
     * <P> 
     * We only actually implement the methods used by ResultSetUtilities.
     * The remaining methods are left empty and are here because the 
     * interface requires them.
     */

    public class LocalResultSet implements ResultSet {

		int scrollType = ResultSet.TYPE_SCROLL_INSENSITIVE;
	
		// Current and last rows using 1-based indexing
		int curRow;
		int lastRow;
	
		String [] colNames;
		int [] colTypes;
		String [] colTypeNames;
	
		ResultSetMetaData metaData;
		
		// First index is column, second is row
	
		Object[][] dataVals;
	
		public LocalResultSet() {
		    lastRow = 3;
		    curRow = 0;
	
		    dataVals = new Object [][] {
				new Double[] {new Double(1.1), new Double(2.2), new Double(3.3)},
				new Integer[] {new Integer(1), new Integer(2), new Integer(3)},
				new String[] { "a1", "b2", "c3"},
				new Boolean[] {new Boolean(true), new Boolean(false), new Boolean(true)},
				new Timestamp[] { new Timestamp(10000), new Timestamp(20000),
						  new Timestamp(30000)}
		    };
		    
		    colNames = new String [] {"DoubleCol", "IntegerCol", "StringCol", 
					      "BooleanCol", "TimestampCol"};
			 	
		    colTypes = new int [] {Types.DOUBLE, Types.INTEGER, Types.CHAR,
					    Types.BIT, Types.TIMESTAMP};
	
		    colTypeNames = new String [] {"Double", "Integer", "String", 
						  "Boolean", "Timestamp"};
	
		    metaData = new LocalResultSetMetaData(colNames, colTypes,
							  colTypeNames);
		}
	
		/* Implemented methods */
	
	
		public boolean next() throws SQLException { 
		    if (curRow == lastRow){
				return false;
		    }
		    else {
				curRow += 1;
				return true;
		    }
		}
	
		public Object getObject(int i) throws SQLException { 
		    // ResultSet methods use 1-based indexing
		    return dataVals[i - 1][curRow - 1];
		}
	
		public void beforeFirst() throws SQLException { 
		    curRow = 0;
		}
	
		public ResultSetMetaData getMetaData() throws SQLException { 
		    return metaData;
		}
	
		public int getRow() throws SQLException { 
		    return curRow;
		}
	
		public int getType() throws SQLException { 
		    return scrollType;
		}
	
		public boolean last() throws SQLException { 
		    if (lastRow == 0)
				return false;
		    else {
				curRow = lastRow;
				return true;
		    }
		}
	
	
		/* Dummy methods required by the interface */
	
		public boolean absolute(int row) throws SQLException {
			if (row > 0)
				curRow = row;
			else
				curRow = lastRow + row;
			if (curRow <= 0) {
				curRow = 0;
				return false;
			} else if (curRow > lastRow) {
				curRow = lastRow + 1;
				return false;
			}
			return true;
		}
	
		public void afterLast() throws SQLException { curRow = lastRow + 1; };
		public void cancelRowUpdates() throws SQLException { };
		public void clearWarnings() throws SQLException { };
		public void close() throws SQLException { }
		public void deleteRow() throws SQLException { }
	
		public int findColumn(String str) throws SQLException { 
			for (int i = 0; i < colNames.length; i++) {
				if (colNames[i].equals(str))
					return (i + 1);
			}
		    return 0;
		}
	
		public boolean first() throws SQLException { 
		    if (lastRow == 0)
				return false;
		    else {
				curRow = 1;
				return true;
		    }
		}
	
		public Array getArray(String str) throws SQLException { 
			int idx = findColumn(str);
			if (idx > 0)
				return getArray(idx);
			else
		    	return null;
		}
	
		public Array getArray(int i) throws SQLException { 
		    return (Array) dataVals[i - 1][curRow - 1];
		}
	
		public InputStream getAsciiStream(String str) throws SQLException { 
		    return null;
		}
	
		public InputStream getAsciiStream(int i) throws SQLException { 
		    return null;
		}
	
		//
		public BigDecimal getBigDecimal(String str) throws SQLException { 
		    return null;
		}
	
		public BigDecimal getBigDecimal(int i) throws SQLException { 
		    return null;
		}
	
		public InputStream getBinaryStream(String str) throws SQLException { 
		    return null;
		}
	
		public InputStream getBinaryStream(int i) throws SQLException { 
		    return null;
		}
	
		public Blob getBlob(String str) throws SQLException { 
		    return null;
		}
	
		public Blob getBlob(int i) throws SQLException { 
		    return null;
		}
	
		public boolean getBoolean(String str) throws SQLException { 
			int idx = findColumn(str);
			if (idx > 0)
				return getBoolean(idx);
			else
		    	return false;
		}
	
		public boolean getBoolean(int i) throws SQLException { 
		    return ((Boolean)dataVals[i - 1][curRow - 1]).booleanValue();
		}
	
		public byte getByte(String str) throws SQLException { 
			int idx = findColumn(str);
			if (idx > 0)
				return getByte(idx);
			else
		    	return 0;
		}
	
		public byte getByte(int i) throws SQLException { 
		    return ((Number)dataVals[i - 1][curRow - 1]).byteValue();
		}
	
		public byte[] getBytes(String str) throws SQLException { 
		    return new byte[0];
		}
	
		public byte[] getBytes(int i) throws SQLException {  
		    return new byte[0];
		}
	
		public Reader getCharacterStream(String str) throws SQLException { 
		    return null;
		}
	
		public Reader getCharacterStream(int i) throws SQLException { 
		    return null;
		}
	
		public Clob getClob(String str) throws SQLException { 
		    return null;
		}
	
		public Clob getClob(int i) throws SQLException { 
		    return null;
		}
	
		public int getConcurrency() throws SQLException { 
		    return 0;
		}
	
		public String getCursorName() throws SQLException { 
		    return null;
		}
	
		public java.sql.Date getDate(String str) throws SQLException { 
			int idx = findColumn(str);
			if (idx > 0)
				return getDate(idx);
			else
		    	return null;
		}
	
		public java.sql.Date getDate(int i) throws SQLException { 
		    return (Date)dataVals[i - 1][curRow - 1];
		}
	
		public Date getDate(String str, java.util.Calendar cal) 
		    throws SQLException { 
		    return null;
		}
	
		public Date getDate(int i, java.util.Calendar cal) 
		    throws SQLException { 
		    return null;
		}
	
		public double getDouble(String str) throws SQLException { 
			int idx = findColumn(str);
			if (idx > 0)
				return getDouble(idx);
			else
		    	return 0;
		}
	
		public double getDouble(int i) throws SQLException { 
		    return ((Number) dataVals[i - 1][curRow - 1]).doubleValue();
		}
	
		public int getFetchDirection() throws SQLException { 
		    return 0;
		}
	
		public int getFetchSize() throws SQLException { 
		    return 0;
		}
	
		public float getFloat(String str) throws SQLException { 
			int idx = findColumn(str);
			if (idx > 0)
				return getFloat(idx);
			else
		    	return 0;
		}
	
		public float getFloat(int i) throws SQLException { 
		    return ((Number)dataVals[i - 1][curRow - 1]).floatValue();
		}
	
		public int getInt(String str) throws SQLException { 
			int idx = findColumn(str);
			if (idx > 0)
				return getInt(idx);
			else
		    	return 0;
		}
	
		public int getInt(int i) throws SQLException { 
		    return ((Number)dataVals[i - 1][curRow - 1]).intValue();
		}
	
		public long getLong(String str) throws SQLException { 
			int idx = findColumn(str);
			if (idx > 0)
				return getLong(idx);
			else
		    	return 0;
		}
	
		public long getLong(int i) throws SQLException { 
		    return ((Number)dataVals[i - 1][curRow - 1]).longValue();
		}
	
		public Object getObject(String str) throws SQLException { 
			int idx = findColumn(str);
			if (idx > 0)
				return getObject(idx);
			else
		    	return null;
		}
	
		public Object getObject(String str, java.util.Map map) 
		    throws SQLException { 
		    return null;
		}
	
		public Object getObject(int i, java.util.Map map) 
		    throws SQLException { 
		    return null;
		}
	
		public Ref getRef(String str) throws SQLException { 
		    return null;
		}
	
		public Ref getRef(int i) throws SQLException {
		    return null;
		}
	
		public short getShort(String str) throws SQLException { 
			int idx = findColumn(str);
			if (idx > 0)
				return getShort(idx);
			else
		    	return 0;
		}
	
		public short getShort(int i) throws SQLException { 
		    return ((Number)dataVals[i - 1][curRow - 1]).shortValue();
		}
	
		public Statement getStatement() throws SQLException { 
		    return null;
		}
	
		public String getString(String str) throws SQLException { 
			int idx = findColumn(str);
			if (idx > 0)
				return getString(idx);
			else
		    	return null;
		}
	
		public String getString(int i) throws SQLException { 
		    return (String)dataVals[i - 1][curRow - 1];
		}
	
		public Time getTime(String str) throws SQLException { 
			int idx = findColumn(str);
			if (idx > 0)
				return getTime(idx);
			else
		    	return null;
		}
	
		public Time getTime(int i) throws SQLException { 
		    return (Time)dataVals[i - 1][curRow - 1];
		}
	
		public Time getTime(String str, java.util.Calendar cal) 
		    throws SQLException { 
		    return null;
		}
		
		public Time getTime(int i, java.util.Calendar cal) 
		    throws SQLException { 
		    return null;
		}
	
		public Timestamp getTimestamp(String str) throws SQLException { 
			int idx = findColumn(str);
			if (idx > 0)
				return getTimestamp(idx);
			else
		    	return null;
		}
	
		public Timestamp getTimestamp(int i) throws SQLException { 
		    return (Timestamp)dataVals[i - 1][curRow - 1];
		}
	
		public Timestamp getTimestamp(String str, java.util.Calendar cal)
		    throws SQLException { 
		    return null;
		}
	
		public Timestamp getTimestamp(int i, java.util.Calendar cal) 
		    throws SQLException { 
		    return null;
		}
	
		public SQLWarning getWarnings() throws SQLException { 
		    return null;
		}
	
		public void insertRow() throws SQLException { }
	
		public boolean isAfterLast() throws SQLException { 
		    return (curRow > lastRow);
		}
	
		public boolean isBeforeFirst() throws SQLException { 
		    return (curRow == 0);
		}
	
		public boolean isFirst() throws SQLException { 
		    return (curRow == 1);
		}
	
		public boolean isLast() throws SQLException { 
		    return (curRow == lastRow);
		}
	
		public void moveToCurrentRow() throws SQLException { }
		public void moveToInsertRow() throws SQLException { }
	
		public boolean previous() throws SQLException {  
			curRow = curRow - 1;
			if (curRow <= 0) {
				curRow = 0;
				return false;
			}
			return true;
		}
	
		public void refreshRow() throws SQLException { }
	
		public boolean relative(int i) throws SQLException { 
			curRow = curRow + i;
			if (curRow <= 0) {
				curRow = 0;
				return false;
			} else if (curRow > lastRow) {
				curRow = lastRow + 1;
				return false;
			}
			return true;
		}
	
		public boolean rowDeleted() throws SQLException { 
		    return false;
		}
	
		public boolean rowInserted() throws SQLException { 
		    return false;
		}
	
		public boolean rowUpdated() throws SQLException { 
		    return false;
		}
	
		//new for jdk 1.4
		public void updateArray(int columnIndex, Array x) throws SQLException { }
		public void updateArray(String columnName, Array x) throws SQLException { }
		public void updateClob(int columnIndex, Clob x) throws SQLException { }
		public void updateClob(String columnName, Clob x) throws SQLException { }
		public void updateBlob(int columnIndex, Blob x) throws SQLException { }
		public void updateBlob(String columnName, Blob x) throws SQLException { }
		public void updateRef(int columnIndex, Ref x) throws SQLException { }
		public void updateRef(String columnName, Ref x) throws SQLException { }
		public java.net.URL getURL(int columnIndex) {return null;}
		public java.net.URL getURL(String columnName) {return null;}
	
		public void setFetchDirection(int i) throws SQLException { }
		public void setFetchSize(int i) throws SQLException { }
		public void updateAsciiStream(String str, InputStream x, int i) 
		    throws SQLException { }
		public void updateAsciiStream(int i, InputStream x, int j) 
		    throws SQLException { }
		public void updateBigDecimal(String str, BigDecimal x) 
		    throws SQLException { }
		public void updateBigDecimal(int i, BigDecimal x) 
		    throws SQLException { }
		public void updateBinaryStream(String str, InputStream x, int i) 
		    throws SQLException { }
		public void updateBinaryStream(int i, InputStream x, int j) 
		    throws SQLException { }
		public void updateBoolean(String str, boolean x) 
		    throws SQLException { }
		public void updateBoolean(int i, boolean x) throws SQLException { }
		public void updateByte(String str, byte x) throws SQLException { }
		public void updateByte(int i, byte x) throws SQLException { }
		public void updateBytes(String str, byte[] x) throws SQLException { }
		public void updateBytes(int i, byte[] x) throws SQLException { }
		public void updateCharacterStream(String str, Reader r, int i) 
		    throws SQLException { }
		public void updateCharacterStream(int i, Reader r, int j) 
		    throws SQLException { }
		public void updateDate(String str, Date x) throws SQLException { }
		public void updateDate(int i, Date x) throws SQLException { }
		public void updateDouble(String str, double x) throws SQLException { }
		public void updateDouble(int i, double x) throws SQLException { }
		public void updateFloat(String str, float x) throws SQLException { }
		public void updateFloat(int i, float x) throws SQLException { }
		public void updateInt(String str, int x) throws SQLException { }
		public void updateInt(int i, int x) throws SQLException { }
		public void updateLong(String str, long x) throws SQLException { }
		public void updateLong(int i, long x) throws SQLException { }
		public void updateNull(String str) throws SQLException { }
		public void updateNull(int i) throws SQLException { }
		public void updateObject(String str, Object x) throws SQLException { }
		public void updateObject(int i, Object x) throws SQLException { }
		public void updateObject(String str, Object x, int i) 
		    throws SQLException { }
		public void updateObject(int i, Object x, int j) 
		    throws SQLException { }
		public void updateRow() throws SQLException { }
		public void updateShort(String str, short x) throws SQLException { }
		public void updateShort(int i, short x) throws SQLException { }
		public void updateString(String str, String x) throws SQLException { }
		public void updateString(int i, String x) throws SQLException { }
		public void updateTime(String str, Time x) throws SQLException { }
		public void updateTime(int i, Time x) throws SQLException { }
		public void updateTimestamp(String str, Timestamp x) 
		    throws SQLException { }
		public void updateTimestamp(int i, Timestamp x) throws SQLException { }
	
		public boolean wasNull() throws SQLException { 
		    return false;
		}
	
		public BigDecimal getBigDecimal(String str, int i) 
		    throws SQLException { 
		    return null;
		}
	
		public BigDecimal getBigDecimal(int i, int x) throws SQLException { 
		    return null;
		}
	
		public InputStream getUnicodeStream(String str) throws SQLException { 
		    return null;
		}
	
		public InputStream getUnicodeStream(int i) throws SQLException { 
		    return null;
		}

    }

    /**
     * Class implementing ResultSetMetaData to provide a self-contained 
     * example.  In actual usage, the ResultSetMetaData will be created 
     * using other JDBC classes.
     * <P> 
     * We only actually implement the methods used by ResultSetUtilities.
     * The remaining methods are left empty and are here because the 
     * interface requires them.
     */

    public class LocalResultSetMetaData implements ResultSetMetaData {

		String [] colNames;
		int [] colTypes;
		String [] colTypeNames;
	
		public LocalResultSetMetaData(String [] names, int [] types,
					      String [] typeNames){
		    if (names.length != types.length ||
			names.length != typeNames.length) {
			throw new RuntimeException("All argument arrays must have the same length");
		    }
	
		    colNames = names;
		    colTypes = types;
		    colTypeNames = typeNames;
		}
	
		/* Implemented methods */
	
		public int getColumnCount() throws SQLException { 
		    return colNames.length;
		}
	
		public String getColumnName(int i) throws SQLException { 
		    return colNames[i - 1];
		}
	
		public int getColumnType(int i) throws SQLException { 
		    return colTypes[i-1];
		}
	
		public String getColumnTypeName(int i) throws SQLException { 
		    return colTypeNames[i-1];
		}
	
	
		/* Dummy methods requred for the interface */
	
		public String getCatalogName(int i) throws SQLException { 
		    return null;
		}
	
		public String getColumnClassName(int i) throws SQLException { 
		    return null;
		}
	
		public int getColumnDisplaySize(int i) throws SQLException { 
		    return 0;
		}
	
		public String getColumnLabel(int i) throws SQLException {
		    return null;
		}
	
		public int getPrecision(int i) throws SQLException { 
		    return 0;
		}
	
		public int getScale(int i) throws SQLException { 
		    return 0;
		}
	
		public String getSchemaName(int i) throws SQLException { 
		    return null;
		}
	
		public String getTableName(int i) throws SQLException { 
		    return null;
		}
	
		public boolean isAutoIncrement(int i) throws SQLException { 
		    return false;
		}
	
		public boolean isCaseSensitive(int i) throws SQLException { 
		    return false;
		}
	
		public boolean isCurrency(int i) throws SQLException { 
		    return false;
		}
	
		public boolean isDefinitelyWritable(int i) throws SQLException { 
		    return false;
		}
	
		public int isNullable(int i) throws SQLException { 
		    return 0;
		}
	
		public boolean isReadOnly(int i) throws SQLException { 
		    return false;
		}
	
		public boolean isSearchable(int i) throws SQLException { 
		    return false;
		}
	
		public boolean isSigned(int i) throws SQLException { 
		    return false;
		}
	
		public boolean isWritable(int i) throws SQLException { 
		    return false;
		}
    }
}





