#include "sconnect.h"
#include "gausssdl.h"

s_object* gaussSeidel(s_object* ps_A, s_object* ps_b)
/**********************************************************************
gaussSeidel solves a linear system using Gauss-Seidel iterative method.

REQUIRED ARGUMENTS: 
 ps_A and ps_b are numeric matrix and vector respectively.
 
VALUE: 
 a vector x, solution of A x = b 
 
S FUNCTION EQUIVALENT:  
 gaussSeidel() defined in gausssdl.ssc

Usage:
 n<-50
 A<-matrix(rnorm(n*n),nrow=n)
 diag(A)<-seq(ncol(A),ncol(A))            # Make it diagonally dominant
 b<-rnorm(ncol(A))
 source('gausssdl.ssc')                   # S version of gaussSeidel
 sys.time({x1<-gaussSeidel(A,b)})         # timing the S version
 sys.time({x2<-.Call('gaussSeidel',A,b)}) # timing the .Call version
 all.equal(x1,x2)                         # Should return T
*********************************************************************/
{	
  S_EVALUATOR
  try
  {
    //Hard-coded relative tolerance and max iterations
    double tol =1e-4; 
    long maxItr = 1000;
			
    //Constructing and validating C++ objects
    const CSPnumericMatrix A(ps_A);
    const CSPnumeric b(ps_b);

		long nRows= A.nrow();
		long nCols= A.ncol();

    if(nRows!=nCols || nCols!=b.length())
      PROBLEM "A.nrow()!=A.ncol() || A.ncol()!=b.length()" ERROR;
		
    //Begin Gauss-Seidel step
    CSPnumeric x=b;
    for(long k =1; k<= maxItr; k++)
    {
      CSPnumeric xOld = x;
      for(long i= 1; i <= nRows; i++)
      {    
        double s = A(i,i) * x(i);
        for(long j = 1; j <= nCols; j++)
          s  = s - A(i,j) * x(j);
        x(i) = (b(i)+s)/A(i,i);
      } 
      //Check convergence; continue if necessary
      if(Max(abs((x-xOld)/x)) < tol)
				return(x.Detach());
    }
    PROBLEM "Solution does not converge" WARN;
		return(x.Detach());
  }
  catch(...)
  {
  }
  return(blt_in_NULL); //return the build-in NULL object
}


