/*
	Connect/C++ : Copyright (c) 2001, 2006 Insightful Corp.
	All rights reserved.
	Version 6.0: 2001
*/

// spattach.cxx: implementation of the CSPattached class.
//
//////////////////////////////////////////////////////////////////////
#include <memory.h>
#include <string.h>

#include "spattach.h"
#include "spcall.h"
#include "spexcept.h"
#include "spchar.h"
#include "spengcon.h"
#include "speval.h"
#include "S_io.h"
#include "S_y_tab.h"


//////////////////////////////////////////////////////////////////////
// SObject iterator
CSPattached::CSObjectItem::CSObjectItem(): m_ps_object(NULL), m_pszName(NULL)
{
}

CSPattached::CSObjectItem::CSObjectItem(s_object* ps_object, const char* pszName)
{
	Set(ps_object, pszName);
}

BOOL CSPattached::CSObjectItem::IsValid() const
{
	return (!SPL_NotThere(m_ps_object) && m_pszName && strlen(m_pszName) > 0);
}

void CSPattached::CSObjectItem::Init()
{
	m_ps_object = NULL;
	m_pszName = NULL;
}

void CSPattached::CSObjectItem::Set(s_object* ps_object, const char* pszName)
{
	if ( SPL_NotThere(ps_object) )
		m_ps_object = NULL;
	else
		m_ps_object = ps_object;

	if ( pszName && strlen(pszName)>0 )
		m_pszName = pszName;
	else 
		m_pszName = NULL;
}

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

//////////////////////// CSPattached ////////////////////////
//Default constructor
CSPattached::CSPattached()
: CSPobject(),	m_uiFlags(0), m_pIterator(NULL)
{
}
//Copy constructor 
CSPattached::CSPattached(const CSPattached& sObject)
: CSPobject(),	m_uiFlags(0), m_pIterator(NULL)
{
	Attach(&sObject, sObject.GetTryToFreeOnDetach());
}
//Construct from a base class object
CSPattached::CSPattached(const CSPobject& sObject)
: CSPobject(),	m_uiFlags(0), m_pIterator(NULL)
{	
	Attach(&sObject, sObject.GetTryToFreeOnDetach());
}
//Construct from a valid S-expression
CSPattached::CSPattached(const char* pszExpression)
: CSPobject(),	m_uiFlags(0), m_pIterator(NULL)
{
	CSPevaluator sEvaluator;
	s_object* ps_value = sEvaluator.Eval(pszExpression);
	Attach(sEvaluator.CloneIfNeeded(ps_value));
}
//Construct from a valid S object
CSPattached::CSPattached(s_object* ps_object, BOOL bTryToFreeOnDetach)
: CSPobject(),	m_uiFlags(0), m_pIterator(NULL)
{
	Attach(ps_object, bTryToFreeOnDetach);
}
//Assigment from the same class
CSPattached& CSPattached::operator=(const CSPattached& sObject)
{
	Attach(&sObject, TRUE);
	return *this;
}
//Assigment from the base class
CSPattached& CSPattached::operator=(const CSPobject& sObject)
{
	Attach(&sObject, TRUE);
	return *this;
}
//Assigment from the S object
CSPattached& CSPattached::operator=(s_object* ps_object)
{
	Attach(ps_object, FALSE);
	return *this;
}
//The destructor
CSPattached::~CSPattached()
{
}
//////////////////// Other constructor/destructor and assignment operators

CSPattached::CSPattached(const char* pszName, s_db_purposes dbPurpose)
: CSPobject(),	m_uiFlags(0), m_pIterator(NULL)
{
	Attach(SPL_AttachedObjectFromName(pszName, dbPurpose));
}

CSPattached::CSPattached(long lPos, s_db_purposes dbPurpose)
: CSPobject(),	m_uiFlags(0), m_pIterator(NULL)
{
	Attach(SPL_AttachedObjectFromPos(lPos, dbPurpose));
}

s_object* CSPattached::Eval(void)
{
	return GetPtr();
}

BOOL CSPattached::IsValid() const
{
	if ( CSPobject::IsValid() )
		return IS(GetPtr(), MAKE_CLASS("attached"));

	return FALSE;
}

s_object* CSPattached::SDictionary(int iSysVal)
{
	// Engine allocates a character vector of length 1
	//  for the return value
	if( !S_evaluator->_eval_open )
		SCONNECT_ThrowException(SCONNECT_EVALUATOR_CLOSED);

	long iDBtype = ( GetPurpose() == META_DB ? 1 : 0 );
	s_object* ps_arglist = NEW_LIST(2);
	s_object* ps_int = NEW_INTEGER(1);
	INTEGER_POINTER(ps_int)[0] = iDBtype;
	LIST_POINTER(ps_arglist)[0] = GetPtr();
	LIST_POINTER(ps_arglist)[1] = ps_int;
	
	S_evaluator->_sys_index = iSysVal; 
	return S_dictionary(blt_in_NULL, ps_arglist, S_evaluator);
}

s_object* CSPattached::GetDatabaseName(BOOL bValidate)
{
	if ( bValidate && !IsValid() )
		return NULL;

	return LIST_POINTER(GetPtr())[ATC_NAME_SLOT];
}

s_object* CSPattached::GetDatabasePath(BOOL bValidate)
{
	if ( bValidate && !IsValid() )
		return NULL;

	s_object* ps_result = NULL;
	try
	{
		CSPevaluator s;
		s_object* ps_path = SDictionary(10);
		if ( !SPL_NotThere(ps_path) && IS_CHARACTER(ps_path) )
		{
			ps_result = ps_path;
			if ( S_is_env_removable_char() )
				S_copy_strings(ps_result, ps_result, S_evaluator);
			ps_result = s.CloneIfNeeded(ps_result);
		}
	}
	catch(...)
	{
	}
	return ps_result;
}

s_db_purposes CSPattached::GetPurpose(BOOL bValidate)
{
	if ( bValidate && !IsValid() )
		return MAX_DB_PURPOSE;

	return S_db_purpose(INTEGER_POINTER(LIST_POINTER(GetPtr())[ATC_ID_SLOT])[2]);
}

long CSPattached::GetCount(BOOL bValidate)
{
	if ( bValidate && !IsValid() )
		return -1;

	return S_db_count(GetPosition(), GetPurpose());
}

void CSPattached::DetachDatabase(BOOL bValidate)
{
	if ( bValidate && !IsValid() )
		return;

	try
	{
		s_object* ps_arglist;
		ps_arglist = NEW_LIST(1);

		LIST_POINTER(ps_arglist)[0] = GetPtr();

		S_evaluator->_sys_index = 1; 
		S_database(blt_in_NULL, ps_arglist,  S_evaluator);
	}
	catch(...)
	{
	}
}

long CSPattached::GetPosition(BOOL bValidate)
{
	if ( bValidate && !IsValid() )
		return -1;

	s_db_purposes purpose = GetPurpose();
	return search_position(GetPtr(), purpose, S_evaluator);
}

long CSPattached::GetTableIndex()
{
	if ( !IsValid() )
		return -1;

	s_boolean optional = S_FALSE;
	s_db_purposes purpose = GetPurpose();
	return database_position(GetPtr(), optional, purpose, S_evaluator)-1;
}

void CSPattached::SetStatus(Status status, BOOL bValidate)
{
	if ( bValidate && !IsValid() )
		return;
	if ( status != read && status != readwrite )
		return;

	try
	{
		CSPevaluator s;

		s_object* ps_arglist = NEW_LIST(2);
		s_object* ps_status = NEW_CHARACTER(1);
		if ( S_is_env_removable_char() )
		{
			CHARACTER_POINTER(ps_status)[0] = (string8*)(status==read?"read":"readwrite");
			S_copy_strings(ps_status, ps_status, S_evaluator);
		}
		else
		{
			CHARACTER_POINTER(ps_status)[0] = SPL_AllocateStringInFrame(ps_status, (status==read?"read":"readwrite")); 
		}
		LIST_POINTER(ps_arglist)[0] = GetDatabaseName();
		LIST_POINTER(ps_arglist)[1] = ps_status;
		
		S_evaluator->_sys_index = 20; 
		S_dictionary(blt_in_NULL, ps_arglist, S_evaluator);
	}
	catch(...)
	{
	}
}

BOOL CSPattached::DatabaseRequired(BOOL bValidate)
{
	if ( bValidate && !IsValid() )
		return FALSE;
	BOOL bRequired = FALSE;
	try
	{
		CSPevaluator s;
		s_object* ps_result = SDictionary(9);
		if ( !SPL_NotThere(ps_result) )
			bRequired = LOGICAL_VALUE(ps_result);
	}
	catch(...)
	{
	}
	return bRequired;
}

CSPattached::Status CSPattached::GetStatus(BOOL bValidate)
{
	if ( bValidate && !IsValid() )
		return statusError;

	const char* szStatus = CHARACTER_VALUE(SDictionary(3));
	if ( !szStatus )
		return statusError;

	if ( strncmp(szStatus,"readw", 5)==0 )
		return readwrite;

	if ( szStatus[0] == 'm' )
		return modified;
	
	if ( szStatus[0] == 'r' )
		return read;

	return statusError;
}

CSPattached::Type CSPattached::GetType(BOOL bValidate)
{
	if ( bValidate && !IsValid() )
		return typeError;

	const char* szStatus = CHARACTER_VALUE(SDictionary(1));
	if ( !szStatus )
		return typeError;

	if ( szStatus[0] == 'd' )
		return directory;

	if ( szStatus[0] == 'o' )
		return object;

	if ( szStatus[0] == 't' )
		return table;

	return typeError;
}

static void ModernizeClass(s_object* ps_object)
{
	if ( SPL_NotThere(ps_object) )
		return;

	// If it is mode STRUCTURE, then s_modernize_class will look for the 'class' attribute
	//	If it is mode NAME, then s_modernize_class will use ps_object->value.name
	// Otherwise, s_modernize_class will only use the mode.
	if ( (ps_object->mode == S_MODE_STRUCTURE || ps_object->mode==S_MODE_NAME) && LIST_POINTER(ps_object) == NULL )
		return;

	if ( ps_object->Class == make_s_class("COMMON", 0, S_evaluator) )
		s_modernize_class(ps_object, S_evaluator);

	if ( ps_object->mode != S_MODE_STRUCTURE && ps_object->mode != S_MODE_LIST )
		return;
	
	if ( LIST_POINTER(ps_object) == NULL )
		return;

	for ( long i=0; i<GET_LENGTH(ps_object); i++ )
		ModernizeClass(LIST_POINTER(ps_object)[i]);
}

const char* CSPattached::GetFileName(const char* pszName)
{
	if ( !pszName || !*pszName )
		return NULL;

	// unused: s_db_purposes purpose = GetPurpose();
	long index =  GetTableIndex();

	return true_file_name(pszName, index, TO_READ, S_evaluator);
}

s_object* CSPattached::GetSummary(const char* pszName, int iLevel, BOOL bKeep, BOOL bModernize)
{
	if ( iLevel <=0 && bKeep )
		SCONNECT_ThrowException("Must get all of the object (iLevel=1) in order to keep it.", iLevel);

	if ( bKeep && !S_evaluator->_eval_open )
		SCONNECT_ThrowException(SCONNECT_EVALUATOR_CLOSED);

	long lPos = GetTableIndex();
	int iHow = 0;
	s_db_purposes meta = GetPurpose();
	s_boolean lock = S_FALSE;
	s_boolean keep = (bKeep?S_TRUE:S_FALSE);
	// unused: long lFrame = (bKeep?-lPos-2:Nframe);

	s_object* ps_object = read_data(pszName, keep, iLevel, Nframe,	lPos, meta, &iHow,  S_evaluator, lock);
	if ( iLevel > 0 || SPL_NotThere(ps_object) || !bModernize || GET_CLASS(ps_object)!=make_s_class("COMMON",0,S_evaluator) )
		return ps_object;

	if ( iLevel == 0 )
	{
		// Detected SV3 Database. To convert the COMMON class we need one extra level to get the class attribute
		//  or the name component of a NAME.  This information is needed in modernized_class.
		SetInfoFlagBit(ATTACHED_SV3Database);
		if ( ps_object->mode == S_MODE_STRUCTURE || ps_object->mode == S_MODE_NAME )
			ps_object = read_data(pszName, keep, -1, Nframe, lPos, meta, &iHow,  S_evaluator, lock);
	}
	ModernizeClass(ps_object);

	return ps_object;
}

BOOL CSPattached::Exists(const char* pszName)
{
	BOOL bExists = FALSE;
	try
	{
		//  Linux compiler complains about next line - looks like a vestige
#ifdef WIN32
		CSPevaluator;
#endif
		bExists = (GetSummary(pszName, S_OBJECT_HEADER, FALSE, FALSE)!=NULL);
	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	return bExists;
}

s_object* CSPattached::Get(const char* pszName, int iType)
{
	if ( iType != S_OBJECT_DATA && iType != S_OBJECT_HEADER && iType != S_OBJECT_SUMMARY )
		iType = S_OBJECT_DATA;

	BOOL bKeep = (iType==S_OBJECT_DATA?TRUE:FALSE);
	if ( S_evaluator->_eval_open )
		bKeep = FALSE;

	return GetSummary(pszName, iType, bKeep, TRUE);
}

BOOL CSPattached::Remove(const char* pszName)
{
	if ( !pszName || !*pszName )
		return FALSE;
	if ( GetType() != directory || GetStatus() < read )
		return FALSE;

	// check to make sure we are successfull
	s_object* ps_object = Get(pszName, S_OBJECT_HEADER);
	if ( SPL_NotThere(ps_object) )
		return FALSE;

	BOOL bSuccess = FALSE;
	try 
	{
		s_name* p_name = make_name(pszName, S_evaluator);
		long lDBTablePos = GetTableIndex();
		do_rm(p_name, lDBTablePos, S_FALSE, S_evaluator);
		bSuccess = TRUE;
	}
	catch(...)
	{
	}
	return bSuccess;
}

#define ATT_BLOCK_SIZE 50 

s_object* CSPattached::GetNames(s_object* ps_classes, BOOL bExtends)
{
	if( !S_evaluator->_eval_open )
		SCONNECT_ThrowException(SCONNECT_EVALUATOR_CLOSED);		

	s_object* ps_names = SDictionary(5);
	if ( SPL_NotThere(ps_names) )
		return blt_in_NULL;
	
	if ( SPL_NotThere(ps_classes) )
		return ps_names;
	
	if ( !IS_CHARACTER(ps_classes) ) 
		SCONNECT_ThrowException("Class filter must be character");

	s_class** pFailed = NULL;
	s_class** pExtClasses = NULL;

	long l = 0, lExt = GET_LENGTH(ps_classes), lFail = 0, lExtSize=0, lFailSize=0;
	lExtSize = lExt+ATT_BLOCK_SIZE;
	pExtClasses = (s_class**)S_ok_malloc(lExtSize*sizeof(s_class*), S_evaluator);
	long i;
	for ( i=0; i<lExt; i++ )
		pExtClasses[i] = MAKE_CLASS(CHARACTER_POINTER(ps_classes)[i]);
	
	for ( i=0; i<GET_LENGTH(ps_names); i++ )
	{
		s_object* ps_header = GetSummary(CHARACTER_POINTER(ps_names)[i], S_OBJECT_HEADER, FALSE, TRUE);
		if ( SPL_NotThere(ps_header) )
			continue;
		BOOL bFound = FALSE, bFail = FALSE;
		s_class* pcl_class = GET_CLASS(ps_header);
		long j;
		for ( j=0; j<lFail; j++ )
		{
			if ( pFailed[j] == pcl_class )
			{
				bFail = TRUE;
				break;
			}
		}
		if ( bFail )
			continue;
		
		for ( j=0; j<lExt && !bFound; j++ )
		{
			if ( pExtClasses[j] == pcl_class )
			{
				if ( l++ < i )
				{
					CHARACTER_POINTER(ps_names)[l-1] = CHARACTER_POINTER(ps_names)[i];
					CHARACTER_POINTER(ps_names)[i] = EMPTY_STRING;
				}
				bFound = TRUE;
			}
		}
		if ( !bFound && bExtends )
		{
			s_object* ps_extends = SPL_ExtendsDirect(pcl_class->name);
			if ( !SPL_NotThere(ps_extends) && IS_CHARACTER(ps_extends) )
			{
				for ( j=1; j<GET_LENGTH(ps_extends) && !bFound; j++ )
				{
					s_class* pcl_extend = MAKE_CLASS(CHARACTER_POINTER(ps_extends)[j]);
					for ( long k=0; k<GET_LENGTH(ps_classes) && !bFound; k++ )
					{
						if ( pExtClasses[k] == pcl_extend )
						{
							if ( lExt >= lExtSize )
							{
								lExtSize += ATT_BLOCK_SIZE;
								pExtClasses = (s_class**)S_ok_realloc((void*)pExtClasses, lExtSize*sizeof(s_class*), S_evaluator);
							}
							if ( l++ < i )
							{
								CHARACTER_POINTER(ps_names)[l-1] = CHARACTER_POINTER(ps_names)[i];
								CHARACTER_POINTER(ps_names)[i] = EMPTY_STRING;
							}
							// zero-th element in ps_extends is the class name we are investigating (pszClass)
							pExtClasses[lExt++] = pcl_class;
							bFound = TRUE;
						}
					}
				}
			}
		}
		if ( !bFound )
		{
			if ( !pFailed )
			{
				lFailSize = ATT_BLOCK_SIZE;
				pFailed = (s_class**)S_ok_malloc(lFailSize*sizeof(s_class*), S_evaluator);
			}
			else if ( lFail >= lFailSize )
			{
				lFailSize += ATT_BLOCK_SIZE;
				pFailed = (s_class**)S_ok_realloc((void*)pFailed, lFailSize*sizeof(s_class*), S_evaluator);
			}
			pFailed[lFail++] = pcl_class;
		}
	}

	if ( l > 0 )
		SET_LENGTH(ps_names, l);
	else 
		ps_names = blt_in_NULL;

	if ( pFailed )
		S_ok_free((void*)pFailed, S_evaluator);
	if ( pExtClasses )
		S_ok_free((void*)pExtClasses, S_evaluator);
	
	if ( !SPL_NotThere(ps_names) && S_is_env_removable_char() )
		S_copy_strings(ps_names, ps_names, S_evaluator);
	
	return ps_names;
}

BOOL CSPattached::InitIterator(s_object* ps_classes, BOOL bExtends, BOOL bGetData)
{
	BOOL bSuccess = FALSE;
	try
	{
		if ( m_pIterator )
		{
			delete m_pIterator;
			m_pIterator = NULL;
		}
		m_sObjectItem.Init();

		if( !S_evaluator->_eval_open )
			SCONNECT_ThrowException(SCONNECT_EVALUATOR_CLOSED);

		if ( !SPL_NotThere(ps_classes) && !IS_CHARACTER(ps_classes) ) 
			SCONNECT_ThrowException("Class filter must be character");

		s_object* ps_names = SDictionary(5);
		if ( !SPL_NotThere(ps_names) && GET_LENGTH(ps_names)>0 )
		{
			m_pIterator = new TIterator;
			m_pIterator->ps_names = ps_names;
			m_pIterator->lPosition = 0;
			m_pIterator->bGetData = (bGetData?S_OBJECT_DATA:S_OBJECT_HEADER);
			m_pIterator->bExtends = bExtends;

			if ( !SPL_NotThere(ps_classes) )
			{
				m_pIterator->lExt = m_pIterator->lClass = GET_LENGTH(ps_classes); 
				m_pIterator->lExtSize = m_pIterator->lExt+ATT_BLOCK_SIZE;
				m_pIterator->pExtClasses = (s_class**)S_ok_malloc(m_pIterator->lExtSize*sizeof(s_class*), S_evaluator);
				for ( long i=0; i<m_pIterator->lExt; i++ )
					m_pIterator->pExtClasses[i] = MAKE_CLASS(CHARACTER_POINTER(ps_classes)[i]);
			}
			bSuccess = TRUE;
		}
	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	catch(...)
	{
		if ( m_pIterator )
		{
			delete m_pIterator;
			m_pIterator = NULL;
		}
	}
	return bSuccess;	
}

CSPattached::CSObjectItem& CSPattached::Next()
{
	m_sObjectItem.Init();
	try
	{
		if( !S_evaluator->_eval_open )
			SCONNECT_ThrowException(SCONNECT_EVALUATOR_CLOSED);

		if ( !m_pIterator )
			SCONNECT_ThrowException("Must initialize the object iterator");

		long nNames = (SPL_NotThere(m_pIterator->ps_names)?0:GET_LENGTH(m_pIterator->ps_names));

		while ( !m_sObjectItem.IsValid() && m_pIterator->lPosition<nNames )
		{
			const char* pszName = CHARACTER_POINTER(m_pIterator->ps_names)[m_pIterator->lPosition++];			
			if ( pszName == EMPTY_STRING )
				continue;
			s_object* ps_header = GetSummary(pszName, m_pIterator->bGetData, FALSE, TRUE);
			if ( SPL_NotThere(ps_header) )
				continue;
			
			if ( m_pIterator->lExt <= 0 )
			{
				m_sObjectItem.Set(ps_header, pszName);
				break;
			}
			BOOL bFound = FALSE, bFail = FALSE;
				
			s_class* pcl_class = GET_CLASS(ps_header);
			long j;
			for ( j=0; j<m_pIterator->lFail; j++ )
			{
				if ( m_pIterator->pFailed[j] == pcl_class )
				{
					bFail = TRUE;
					break;
				}
			}
			if ( bFail )
				continue;
			
			for ( j=0; j<m_pIterator->lExt && !bFound; j++ )
			{
				if ( m_pIterator->pExtClasses[j] == pcl_class )
				{
					m_sObjectItem.Set(ps_header, pszName);
					bFound = TRUE;
				}
			}
			if ( !bFound && m_pIterator->bExtends )
			{
				s_object* ps_extends = SPL_ExtendsDirect(pcl_class->name);
				if ( !SPL_NotThere(ps_extends) && IS_CHARACTER(ps_extends) )
				{
					for ( j=1; j<GET_LENGTH(ps_extends) && !bFound; j++ )
					{
						s_class* pcl_extend = MAKE_CLASS(CHARACTER_POINTER(ps_extends)[j]);
						for ( long k=0; k<m_pIterator->lClass && !bFound; k++ )
						{
							if ( m_pIterator->pExtClasses[k] == pcl_extend )
							{
								if ( m_pIterator->lExt >= m_pIterator->lExtSize )
								{
									m_pIterator->lExtSize += ATT_BLOCK_SIZE;
									m_pIterator->pExtClasses = (s_class**)S_ok_realloc((void*)m_pIterator->pExtClasses, m_pIterator->lExtSize, S_evaluator);
								}
								m_sObjectItem.Set(ps_header, pszName);
								bFound = TRUE;
								// zero-th element in ps_extends is the class name we are investigating (pszClass)
								m_pIterator->pExtClasses[m_pIterator->lExt++] = pcl_class;
								bFound = TRUE;
							}
						}
					}
				}
			}
			if ( !bFound )
			{
				if ( !m_pIterator->pFailed )
				{
					m_pIterator->lFailSize = ATT_BLOCK_SIZE;
					m_pIterator->pFailed = (s_class**)S_ok_malloc(ATT_BLOCK_SIZE*sizeof(s_class*), S_evaluator);
				}
				else if ( m_pIterator->lFail >= m_pIterator->lFailSize )
				{
					m_pIterator->lFailSize += ATT_BLOCK_SIZE;
					m_pIterator->pFailed = (s_class**)S_ok_realloc((void*)m_pIterator->pFailed, m_pIterator->lFailSize*sizeof(s_class*), S_evaluator);
				}
				m_pIterator->pFailed[m_pIterator->lFail++] = pcl_class;
			}
		}
	}
	catch(CSPexception& except)
	{
		except.Print();
		m_sObjectItem.Init();
	}
	catch(...)
	{
		m_sObjectItem.Init();
	}

	if ( !m_sObjectItem.IsValid() )
	{
		if ( m_pIterator )
		{
			delete m_pIterator;
			m_pIterator = NULL;
		}
	}

	return m_sObjectItem;
}

s_object* CSPattached::Assign(s_object* ps_object, const char* pszName)
{
	if ( !pszName || !*pszName )
		return blt_in_NULL;

	if ( SPL_NotThere(ps_object) )
		return blt_in_NULL;

	return db_put_data(pszName, GetTableIndex(), ps_object, S_evaluator);		
}

//////////////////////////////////////////////////////////////////////
// Compare helpers
//
BOOL CSPattached::operator == (const CSPobject& sobject) const
{
	try
	{
		if ( IsValid() && sobject.IsValid() )
			return LOGICAL_VALUE(s_identical_objects(GetPtr(),sobject.GetPtr()));
	}
	catch(...)
	{
	}
	return FALSE;
}

BOOL CSPattached::operator != (const CSPobject& sobject) const
{
	try
	{
		if ( IsValid() && sobject.IsValid() )
			return !LOGICAL_VALUE(s_identical_objects(GetPtr(),sobject.GetPtr()));
	}
	catch(...)
	{
	}
	return FALSE;
}
