/*
	Connect/C++ : Copyright (c) 2001, 2006 Insightful Corp.
	All rights reserved.
	Version 6.0: 2001
*/

// spnumeric.cxx: implementation of the CSPnumeric class.
//
//////////////////////////////////////////////////////////////////////

#include <memory.h>
#include "spnum.h"
#include "spchar.h"
#include "speval.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

//////////////////////// CSPnumeric ////////////////////////
//Default constructor
CSPnumeric::CSPnumeric()
: TSPvector<double, S_MODE_DOUBLE>()
{
}
//Copy constructor 
CSPnumeric::CSPnumeric(const CSPnumeric& sObject)
: TSPvector<double, S_MODE_DOUBLE>()
{
	Attach(&sObject, TRUE);
}
//Construct from object of a base class
CSPnumeric::CSPnumeric(const CSPobject& sObject)
: TSPvector<double, S_MODE_DOUBLE>()
{
	Attach(&sObject, sObject.GetTryToFreeOnDetach());
}
//Construct from a valid S-expression
CSPnumeric::CSPnumeric(const char* pszExpression)
: TSPvector<double, S_MODE_DOUBLE>(pszExpression)
{
}
//Construct from a valid S object
CSPnumeric::CSPnumeric(s_object* ps_object, BOOL bTryToFreeOnDetach)
: TSPvector<double, S_MODE_DOUBLE>()
{
	Attach(ps_object, bTryToFreeOnDetach);
}

//Assigment from the same class
CSPnumeric& CSPnumeric::operator=(const CSPnumeric& sObject)
{
	Attach(&sObject, TRUE);
	return *this;
}
//Assigment from the base class
CSPnumeric& CSPnumeric::operator=(const CSPobject& sObject)
{
	Attach(&sObject, TRUE);
	return *this;
}
//Assigment from the S object
CSPnumeric& CSPnumeric::operator=(s_object* ps_object)
{
	Attach(ps_object, FALSE);
	return *this;
}
//The destructor
CSPnumeric::~CSPnumeric()
{
}
//////////////////// Other constructor/destructor and assignment operators

CSPnumeric::CSPnumeric(double d)
: TSPvector<double, S_MODE_DOUBLE>(1L)
{
	(*this)->value.Double[0] = d;
}

//Construct a 'numeric' vector of length nLength
CSPnumeric::CSPnumeric(long lLength)
: TSPvector<double, S_MODE_DOUBLE>(lLength)
{
}

CSPnumeric::CSPnumeric(int nLength)
: TSPvector<double, S_MODE_DOUBLE>((long) nLength)
{
}

CSPnumeric::CSPnumeric(double* pdValues, long lLength)
: TSPvector<double, S_MODE_DOUBLE>(lLength)
{
	SetAtDirect(pdValues, 0L, lLength-1, FALSE);
}

CSPnumeric::CSPnumeric(double* pdValues, int nLength)
: TSPvector<double, S_MODE_DOUBLE>( (long) nLength)
{
	SetAtDirect(pdValues, 0L, (long)(nLength-1), FALSE);
}

CSPnumeric& CSPnumeric::operator = (double dRhs)
{
	DecrRef();
	Attach(NewMyType(1));
	(*this)->value.Double[0] = dRhs;
	return *this;
}

//Direct access to elements of character vector (ZERO-BASED INDEX). Use it carefully!
void CSPnumeric::SetAtDirect(long lIndex, double dElement, BOOL bValidate)
{
	if(bValidate && (!IsValid()) || (lIndex >= GetLength(FALSE)))
		SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);

	//Warning: no ref-count check.
	NUMERIC_POINTER(&(*this))[lIndex] = dElement; 
}

//Direct access to elements of character vector (ZERO-BASED INDEX). Use it carefully!
void CSPnumeric::SetAtDirect(double* pdValues, long lStartIndex, long lEndIndex, BOOL bValidate)
{
	if(bValidate  
	&& (!IsValid() || (lStartIndex > lEndIndex)
	    || (lStartIndex >= GetLength(FALSE)) || (lEndIndex >= GetLength(FALSE)))
	)
		SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);

	long j=0;

	if(pdValues != NULL)
	{
		for(long n=lStartIndex; n<= lEndIndex; n++)
			NUMERIC_POINTER(&(*this))[n] = pdValues[j++]; //pValues[j] better be valid!
	}
}

CSPnumeric CSPnumeric::abs(BOOL bValidate) const
{
	if(bValidate && !IsValid())
		SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);

	CSPnumeric sAns(Clone());
	long len = length(FALSE);
	for(long n=0; n<len ; n++)
		sAns->value.Double[n] = fabs(sAns->value.Double[n]);
	return sAns;			
}

CSPnumeric& CSPnumeric::operator=(const CSPnumeric::CProxy& proxyRhs)
{
	double element = (double) proxyRhs.m_vector.GetAt(proxyRhs.m_lIndex);
	s_object* ps_object = NewMyType(1L);
	MyTypeValue(ps_object, 0) = element;
	ReAttachAndAssign(ps_object);
	return *this;
}
CSPnumeric& CSPnumeric::operator+=(const double rhs)
{
	Validate();
	long len = GetLength(FALSE);
	if(IsSharing())
	{
		s_object* ps_lhs = CopyForWrite(FALSE);
		for(long i=0; i<len ; ++i)
			ps_lhs->value.Double[i] /= rhs;
		ReAttachAndAssign(ps_lhs);
	}
	else
	{
		for(long i=0; i<len ; ++i)
			(*this)->value.Double[i] += rhs;
	}
	return *this;
}
CSPnumeric& CSPnumeric::operator-=(const double rhs)
{
	Validate();
	long len = GetLength(FALSE);
	if(IsSharing())
	{
		s_object* ps_lhs = CopyForWrite(FALSE);
		for(long i=0; i<len ; ++i)
			ps_lhs->value.Double[i] /= rhs;
		ReAttachAndAssign(ps_lhs);
	}
	else
	{
		for(long i=0; i<len ; ++i)
			(*this)->value.Double[i] -= rhs;
	}
	return *this;
}
CSPnumeric& CSPnumeric::operator*=(const double rhs)
{
	Validate();
	long len = GetLength(FALSE);
	if(IsSharing())
	{
		s_object* ps_lhs = CopyForWrite(FALSE);
		for(long i=0; i<len ; ++i)
			ps_lhs->value.Double[i] /= rhs;
		ReAttachAndAssign(ps_lhs);
	}
	else
	{
		for(long i=0; i<len ; ++i)
			(*this)->value.Double[i] *= rhs;
	}
	return *this;
}

CSPnumeric& CSPnumeric::operator/=(const double rhs)
{
	Validate();
	long len = GetLength(FALSE);
	if(IsSharing())
	{
		s_object* ps_lhs = CopyForWrite(FALSE);
		for(long i=0; i<len ; ++i)
			ps_lhs->value.Double[i] /= rhs;
		ReAttachAndAssign(ps_lhs);
	}
	else
	{
		for(long i=0; i<len ; ++i)
			(*this)->value.Double[i] /= rhs;
	}
	return *this;
}

CSPnumeric& CSPnumeric::operator+=(const CSPnumeric& sRhs)
{
	Validate();
	long len = GetLength(FALSE);
	if(len!=sRhs.GetLength())
		SCONNECT_ThrowException("mismatch lengths for vector += operation");
	if(IsSharing())
	{
		s_object* ps_lhs = CopyForWrite(FALSE);
		for(long i=0; i<len ; ++i)
			ps_lhs->value.Double[i] += sRhs->value.Double[i];
		ReAttachAndAssign(ps_lhs);
	}
	else
	{
		for(long i=0; i<len ; ++i)
			(*this)->value.Double[i] += sRhs->value.Double[i] ;
	}
	return *this;
}
CSPnumeric& CSPnumeric::operator-=(const CSPnumeric& sRhs)
{
	Validate();
	long len = GetLength(FALSE);
	if(len!=sRhs.GetLength())
	if(GetLength()!=sRhs.GetLength())
		SCONNECT_ThrowException("mismatch lengths for vector -= operation");
	if(IsSharing())
	{
		s_object* ps_lhs = CopyForWrite(FALSE);
		for(long i=0; i<len ; ++i)
			ps_lhs->value.Double[i] -= sRhs->value.Double[i];
		ReAttachAndAssign(ps_lhs);
	}
	else
	{
		for(long i=0; i<len ; ++i)
			(*this)->value.Double[i] -= sRhs->value.Double[i] ;
	}
	return *this;
}
CSPnumeric& CSPnumeric::operator*=(const CSPnumeric& sRhs)
{
	Validate();
	long len = GetLength(FALSE);
	if(len!=sRhs.GetLength())
		SCONNECT_ThrowException("mismatch lengths for vector *= operation");
	if(IsSharing())
	{
		s_object* ps_lhs = CopyForWrite(FALSE);
		for(long i=0; i<len ; ++i)
			ps_lhs->value.Double[i] *= sRhs->value.Double[i] ;
		ReAttachAndAssign(ps_lhs);
	}
	else
	{
		for(long i=0; i<len ; ++i)
			(*this)->value.Double[i] *= sRhs->value.Double[i] ;
	}
	return *this;
}
CSPnumeric& CSPnumeric::operator/=(const CSPnumeric& sRhs)
{
	Validate();
	long len = GetLength(FALSE);
	if(len!=sRhs.GetLength())
		SCONNECT_ThrowException("mismatch lengths for vector /= operation");
	if(IsSharing())
	{
		s_object* ps_lhs = CopyForWrite(FALSE);
		for(long i=0; i<len ; ++i)
			ps_lhs->value.Double[i] /= sRhs->value.Double[i] ;
		ReAttachAndAssign(ps_lhs);
	}
	else
	{
		for(long i=0; i<len ; ++i)
			(*this)->value.Double[i] /= sRhs->value.Double[i] ;
	}
	return *this;
}

///////////////////////////
double CSPnumeric::Max(BOOL bValidate) const
{
	if(bValidate && !IsValid())
		SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);

	double* x = (*this)->value.Double;
	double dMax= x[0];
	for(long n=0; n< length(FALSE); n++)
	{
		if(dMax < x[n])
			dMax = x[n];
	}

	return dMax;
}

double CSPnumeric::Min(BOOL bValidate) const
{
	if(bValidate && !IsValid())
		SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);

	double* x = (*this)->value.Double;
	double dMin= x[0];
	for(long n=0; n< length(FALSE); n++)
	{
		if(dMin > x[n])
			dMin = x[n];
	}

	return dMin;
}

CSPnumeric abs(const CSPnumeric& sObject, BOOL bValidate)  
{
	return sObject.abs(bValidate);
}
double Max(const CSPnumeric& sObject, BOOL bValidate)  
{
	return sObject.Max(bValidate);
}
double Min(const CSPnumeric& sObject, BOOL bValidate)  
{
	return sObject.Min(bValidate);
}

//////////////////////////////////////////////////////
// Proxy implementation
CSPnumeric::CProxy::CProxy(CSPnumeric& sVector, long lZeroBasedIndex)
:TSPvector<double, S_MODE_DOUBLE>::CSPvectorProxy(sVector, lZeroBasedIndex)
{
}		
CSPnumeric::CProxy::CProxy(const CProxy& proxyRhs)
:TSPvector<double, S_MODE_DOUBLE>::CSPvectorProxy(proxyRhs.m_vector, proxyRhs.m_lIndex)
{
}		
CSPnumeric::CProxy& CSPnumeric::CProxy::operator =(const CSPnumeric::CProxy& proxyRhs)
{
	double element = (double) proxyRhs.m_vector.GetAt(proxyRhs.m_lIndex);
	m_vector.SetAt(m_lIndex, element, FALSE);
	return *this;
}
CSPnumeric::CProxy& CSPnumeric::CProxy::operator =(const double& rhs)
{
	m_vector.SetAt(m_lIndex, rhs, FALSE);
	return *this;
}
CSPnumeric::CProxy& CSPnumeric::CProxy::operator+=(const double& rhs)
{
	double element = m_vector.GetAt(m_lIndex);
	element += rhs;
	m_vector.SetAt(m_lIndex, element, FALSE);
	return *this;
}
CSPnumeric::CProxy& CSPnumeric::CProxy::operator-=(const double& rhs)
{
	double element = m_vector.GetAt(m_lIndex);
	element -= rhs;
	m_vector.SetAt(m_lIndex, element, FALSE);
	return *this;
}
CSPnumeric::CProxy& CSPnumeric::CProxy::operator*=(const double& rhs)
{
	double element = m_vector.GetAt(m_lIndex);
	element *= rhs;
	m_vector.SetAt(m_lIndex, element, FALSE);
	return *this;
}
CSPnumeric::CProxy& CSPnumeric::CProxy::operator/=(const double& rhs)
{
	double element = m_vector.GetAt(m_lIndex);
	element /= rhs;
	m_vector.SetAt(m_lIndex, element, FALSE);
	return *this;
}
