package Splus::Make;
use v5.8;

=head1 NAME

  Splus::Make - get and expand macro defintions from Makefiles

=head1 SYNOPSIS

  use Splus::Make
  $mk = Splus::Make->new() ;
  $mk->parse_makefile("m1.mak") ; # add macro definitions in m1.mak to hash
  $mk->parse_macrodef('FOO=value of foo is related to $N');
  $mk->print_makevars() ;
  print "Unexpanded FOO: \"" . $mk->get_makevar("FOO") . "\"\n" ;
  print "Expanded FOO:   \"" . $mk->expand_makevar("FOO") . "\"\n" ;
  
  The expanded FOO will be more interesting if m1.mak has
  a line defining N, like N=capital n.

  In the future we may add more make facilities.

=cut

sub new {
   my $class = shift ;
   my $self = { } ;
   $self->{"makevars"} = {} ;
   bless $self, $class ;
}

sub parse_macrodef {
   # Parse string of form name=value and hash it in %makevars.
   # Silently ignore nonconforming input strings.
   my $self = shift ;
   my $makevars = $self->{"makevars"} ;
   local($_) = @_ ;
   if ( my $retval = /^(\w+)\s*=\s*/ ) {
       my $name=$1 ; # stuff in pattern's parens, w+ (alphanumerics & underscores)
       my $value=$' ; # stuff after the match (after =whitespace*)
       $self->{"makevars"}->{$name} = $value ;
   }
   $retval
}

sub parse_makefile {
   my $self = shift ;
   my $makefile = shift ;
   open MAKEFILE, "< $makefile" || warn "Cannot open $makefile" ;
   while (<MAKEFILE>) {
      s/\r?\n$// ; # was chomp, but want to be able to deal with Windows files on Unix
      while ($_ =~ s/\\$//) { # backslash at end of line to continue it
         $_ = $_ . <MAKEFILE> ;
         s/\r?\n$// ; # was chomp
      }
      if ( /^(\w+)\s*=\s*/ ) {
         $self->parse_macrodef($_) ;
      }
   }
   close MAKEFILE;
   # print Dumper($self) ;
}

sub get_makevar {
   my $self = shift ;
   my $name = shift ;
   $self->{"makevars"}->{$name} ;
}

sub expand_makevar {
   my $self = shift ;
   my $name = shift ;
   my $makevars = $self->{"makevars"} ;
   my $value = $makevars->{$name} ;
   my $n_expansions=0;
   my $max_expansions=2000;
   # The following should expand $N or $(NAME) into the value of N or NAME.
   # Expand $NAME into the value of N followed by literal AME.
   while ( $value =~ s/\$\((\w+)\)/$makevars->{$1}/ || $value =~ s/\$(\w)/$makevars->{$1}/) {
       (++$n_expansions > $max_expansions) && die "Recursive definition of \"$name\"?  ($n_expansions expansions exceeds max_expansions)" ;
   }
   $value ;
}

# like expand_makevar, but you give it raw expression (value), not its name
# E.g., expand_macro("$(foo) $(bar)").
sub expand_macro {
    my $self = shift ;
    my $value = shift ;
    my $makevars = $self->{"makevars"} ;
    my $n_expansions=0;
    my $max_expansions=2000;
    while ( $value =~ s/\$\((\w+)\)/$makevars->{$1}/ || $value =~ s/\$(\w)/$makevars->{$1}/) {
        (++$n_expansions > $max_expansions) && die "Recursive definition of \"$name\"?  ($n_expansions expansions exceeds max_expansions)" ;
    }
    $value;
}

sub DESTROY {
   # This was to experiment with DESTROY method.
   # I think reference counting cleans up the hash table so
   # we don't really have to do anything here.
   my $self = shift ;
   $self->clear_makevars() ;
}

sub clear_makevars {
   # Clears hash table of all make macro definitions.
   # In OO version this probably is not needed  - just release Make object.
   # But it will let you reuse the Make object.
   my $self = shift ;
   my $makevars = $self->{"makevars"} ;
   while ( my ($key, $value) = each %$makevars ) { # delete each entry
      delete $makevars->{$key} ;
   }
}

sub print_makevars {
   # For debugging, print key, value, and expanded value of each variable
   my $self = shift ;
   my $makevars = $self->{"makevars"} ;
   $makevars || print "Empty makevars hash\n";
   my $count = 0;
   while ( my ($key, $value) = each %$makevars ) {
      print "$key=$value => " . $self->expand_makevar($key) . "\n";
   }
}

1 ;
