// Copyright 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

// Audio test utilities.

// Gathers |numSamples| samples at |frequency| number of times per second and
// calls back |callback| with an array with numbers in the [0, 32768] range.
function gatherAudioLevelSamples(peerConnection, numSamples, frequency,
                                 callback) {
  console.log('Gathering ' + numSamples + ' audio samples...');
  var audioLevelSamples = []
  var gatherSamples = setInterval(function() {
    peerConnection.getStats(function(response) {
      audioLevelSamples.push(getAudioLevelFromStats_(response));
      if (audioLevelSamples.length == numSamples) {
        console.log('Gathered all samples.');
        clearInterval(gatherSamples);
        callback(audioLevelSamples);
      }
    });
  }, 1000 / frequency);
}

// Tries to identify the beep-every-second signal generated by the fake audio
// media/audio/fake_audio_input_stream.cc. Fails the test if we can't see a
// signal.
function verifyAudioIsPlaying(samples) {
  var average = 0;
  for (var i = 0; i < samples.length; ++i)
    average += samples[i] / samples.length;

  var largest = 0;
  for (var i = 0; i < samples.length; ++i)
    largest = Math.max(largest, samples[i]);

  console.log('Average audio level: ' + average + ', largest: ' + largest);

  // TODO(phoglund): Make a more sophisticated curve-fitting algorithm. We want
  // to see a number of peaks with relative silence between them. The following
  // seems to work fine on a nexus 7.
  if (average < 3000 || average > 8000)
    failTest('Unexpected avg audio level: got ' + average + ', expected it ' +
             'to be 4000 < avg < 8000.');
  if (largest < 25000)
    failTest('Too low max audio level: got ' + largest + ', expected > 30000.');
}

// If silent (like when muted), we should get very near zero audio level.
function verifyIsSilent(samples) {
  var average = 0;
  for (var i = 0; i < samples.length; ++i)
    average += samples[i] / samples.length;

  console.log('Average audio level: ' + average);
  if (average > 500)
    failTest('Expected silence, but avg audio level was ' + average);
}

/**
 * @private
 */
function getAudioLevelFromStats_(response) {
  var reports = response.result();
  var audioOutputLevels = [];
  for (var i = 0; i < reports.length; ++i) {
    var report = reports[i];
    if (report.names().indexOf('audioOutputLevel') != -1) {
      audioOutputLevels.push(report.stat('audioOutputLevel'));
    }
  }
  // Should only be one audio level reported, otherwise we get confused.
  assertEquals(1, audioOutputLevels.length);

  return audioOutputLevels[0];
}
