/****************************************************************************
**
** Copyright (C) 2015 The Qt Company Ltd.
** Contact: http://www.qt.io/licensing/
**
** This file is part of the plugins of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL21$
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and The Qt Company. For licensing terms
** and conditions see http://www.qt.io/terms-conditions. For further
** information use the contact form at http://www.qt.io/contact-us.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 or version 3 as published by the Free
** Software Foundation and appearing in the file LICENSE.LGPLv21 and
** LICENSE.LGPLv3 included in the packaging of this file. Please review the
** following information to ensure the GNU Lesser General Public License
** requirements will be met: https://www.gnu.org/licenses/lgpl.html and
** http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** As a special exception, The Qt Company gives you certain additional
** rights. These rights are described in The Qt Company LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include <qpa/qwindowsysteminterface.h>
#include <QtPlatformSupport/private/qopenglcompositor_p.h>
#include <QtPlatformSupport/private/qopenglcompositorbackingstore_p.h>

#include "qeglplatformwindow_p.h"
#include "qeglplatformscreen_p.h"

QT_BEGIN_NAMESPACE

/*!
    \class QEGLPlatformWindow
    \brief Base class for EGL-based platform window implementations.
    \since 5.2
    \internal
    \ingroup qpa

    Lightweight class providing some basic platform window operations
    and interfacing with QOpenGLCompositorBackingStore.

    Almost no QPlatformWindow functions are implemented here. This is
    intentional because different platform plugins may use different
    strategies for their window management (some may force fullscreen
    windows, some may not, some may share the underlying native
    surface, some may not, etc.) and therefore it is not sensible to
    enforce anything for these functions.

    \note Subclasses are responsible for invoking this class'
    implementation of create() and are expected to utilize the window
    stack management functions (addWindow() etc.) in
    QOpenGLCompositor.
 */

QEGLPlatformWindow::QEGLPlatformWindow(QWindow *w)
    : QPlatformWindow(w),
      m_backingStore(0),
      m_raster(false),
      m_winId(0)
{
}

static WId newWId()
{
    static WId id = 0;

    if (id == std::numeric_limits<WId>::max())
        qWarning("QEGLPlatformWindow: Out of window IDs");

    return ++id;
}

void QEGLPlatformWindow::create()
{
    m_winId = newWId();

    // Save the original surface type before changing to OpenGLSurface.
    m_raster = (window()->surfaceType() == QSurface::RasterSurface);
    if (m_raster) // change to OpenGL, but not for RasterGLSurface
        window()->setSurfaceType(QSurface::OpenGLSurface);

    if (window()->type() == Qt::Desktop) {
        QRect fullscreenRect(QPoint(), screen()->availableGeometry().size());
        QPlatformWindow::setGeometry(fullscreenRect);
        QWindowSystemInterface::handleGeometryChange(window(), fullscreenRect);
        return;
    }
}

bool QEGLPlatformWindow::isRaster() const
{
    return m_raster || window()->surfaceType() == QSurface::RasterGLSurface;
}

QWindow *QEGLPlatformWindow::sourceWindow() const
{
    return window();
}

const QPlatformTextureList *QEGLPlatformWindow::textures() const
{
    if (m_backingStore)
        return m_backingStore->textures();

    return 0;
}

void QEGLPlatformWindow::endCompositing()
{
    if (m_backingStore)
        m_backingStore->notifyComposited();
}

WId QEGLPlatformWindow::winId() const
{
    return m_winId;
}

void QEGLPlatformWindow::setOpacity(qreal)
{
    if (!isRaster())
        qWarning("QEGLPlatformWindow: Cannot set opacity for non-raster windows");

    // Nothing to do here. The opacity is stored in the QWindow.
}

QT_END_NAMESPACE
