/* kvhash.c - key/value hash
 *
 * Copyright (c) 2010 Colin Watson <cjwatson@debian.org>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <limits.h>
#include <stdlib.h>
#include <stdbool.h>
#include <string.h>

#include "gl_rbtreehash_list.h"
#include "gl_xlist.h"
#include "xalloc.h"

#include "kvhash.h"

/* From Gnulib; see https://www.haible.de/bruno/hashfunc.html. */
static size_t kvhash_hashcode (const void *data)
{
    const struct kvelem *kvdata = data;
    const char *s;
    size_t h = 0;

    for (s = kvdata->key; *s; s++)
	h = *s + ((h << 9) | (h >> (sizeof (size_t) * CHAR_BIT - 9)));

    return h;
}

static int kvhash_compare (const void *a, const void *b)
{
    const struct kvelem *kva = a, *kvb = b;
    return strcmp (kva->key, kvb->key);
}

static bool kvhash_equals (const void *a, const void *b)
{
    return kvhash_compare (a, b) == 0;
}

void kvhash_dispose (void *data)
{
    struct kvelem *kvdata = data;
    free (kvdata->key);
    free (kvdata);
}

void kvhash_dispose_plain (void *data)
{
    struct kvelem *kvdata = data;
    free (kvdata->value);
    kvhash_dispose (data);
}

gl_list_t kvhash_initialize (gl_listelement_dispose_fn dispose_fn)
{
    return gl_list_create_empty (GL_RBTREEHASH_LIST,
				 kvhash_equals, kvhash_hashcode, dispose_fn,
				 false);
}

bool kvhash_exists (gl_list_t list, const char *key)
{
    struct kvelem entry;

    entry.key = (char *) key; /* short lifetime, so don't bother to copy */
    entry.value = NULL;

    return gl_sortedlist_search (list, kvhash_compare, &entry) != NULL;
}

void *kvhash_lookup (gl_list_t list, const char *key)
{
    struct kvelem entry;
    gl_list_node_t lookup;

    entry.key = (char *) key; /* short lifetime, so don't bother to copy */
    entry.value = NULL;

    lookup = gl_sortedlist_search (list, kvhash_compare, &entry);
    if (lookup)
	return ((struct kvelem *) gl_list_node_value (list, lookup))->value;
    else
	return NULL;
}

void *kvhash_insert (gl_list_t list, const char *key, const void *value)
{
    struct kvelem *entry;
    gl_list_node_t lookup;

    entry = xmalloc (sizeof *entry);
    entry->key = xstrdup (key);
    /* Caller should have duplicated value if necessary. */
    entry->value = (void *) value;

    lookup = gl_sortedlist_search (list, kvhash_compare, entry);
    if (lookup) {
	free (entry->key);
	free (entry);
	return ((struct kvelem *) gl_list_node_value (list, lookup))->value;
    } else {
	gl_sortedlist_add (list, kvhash_compare, entry);
	return entry->value;
    }
}

void *kvhash_delete (gl_list_t list, const char *key)
{
    struct kvelem entry;
    gl_list_node_t lookup;

    entry.key = (char *) key; /* short lifetime, so don't bother to copy */
    entry.value = NULL;

    lookup = gl_sortedlist_search (list, kvhash_compare, &entry);
    if (lookup) {
	void *value =
	    ((struct kvelem *) gl_list_node_value (list, lookup))->value;
	gl_sortedlist_remove (list, kvhash_compare, &entry);
	return value;
    } else
	return NULL;
}
