/***************************************************************************
 *   Copyright (C) 2005-2006 Gao Xianchao                                  *
 *                 2007 Gao Xianchao gnap_an linux_lyb ahlongxp            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <glib.h>
#include "UPnpNat.h"
#include "log.h"
#include "utils.h"


/***************************************************************
** General Defines                                             *
****************************************************************/
#define HTTP_OK "200 OK"
#define DEFAULT_HTTP_PORT 80
#define MAX_PORT_SIZE 6
#define SIZEOF_HTTP 7         /* size of "http://" string */
#define RECEIVE_TIMEOUT 10000
#define CONSECUTIVE_RECEIVE_TIMEOUT 500
#define DISCOVERY_TIMEOUT 1000


/***************************************************************
** Discovery/Description Defines                               *
****************************************************************/
#define NUM_UDP_ATTEMPTS 2

/* Address and port of an SSDP request used for discovery */
#define HTTPMU_HOST_ADDRESS "239.255.255.250"
#define HTTPMU_HOST_PORT 1900

#define SEARCH_REQUEST_DEVICE "urn:schemas-upnp-org:service:%s"

#define SEARCH_REQUEST_STRING "M-SEARCH * HTTP/1.1\r\n"            \
                              "HOST: 239.255.255.250:1900\r\n"     \
                              "ST:upnp:rootdevice\r\n"          \
                              "Man:\"ssdp:discover\"\r\n"  \
                              "MX: 3\r\n"                          \
                              "\r\n"

#define MAX_DISCOVERY_RECEIVE_SIZE 400
#define MAX_DESCRIPTION_RECEIVE_SIZE 7000
#define MAX_DESCRIPTION_HTTP_HEADER_SIZE 100


/******************************************************************
** Action Defines                                                 *
*******************************************************************/
#define HTTP_HEADER_ACTION "POST %s HTTP/1.1\r\n"                         \
                           "HOST: %s\r\n"                                  \
                           "SOAPACTION: "                                  \
                           "\"urn:schemas-upnp-org:"                       \
                           "service:%s#%s\"\r\n"                           \
                           "CONTENT-TYPE: text/xml ; charset=\"utf-8\"\r\n"\
                           "Content-Length: %" G_GSIZE_FORMAT "\r\n\r\n"

#define SOAP_ACTION  "<?xml version=\"1.0\" encoding=\"utf-8\"?>\r\n"     \
                     "<s:Envelope xmlns:s="                               \
                     "\"http://schemas.xmlsoap.org/soap/envelope/\" "     \
                     "s:encodingStyle="                                   \
                     "\"http://schemas.xmlsoap.org/soap/encoding/\">\r\n" \
                     "<s:Body>\r\n"                                       \
                     "<u:%s xmlns:u="                                     \
                     "\"urn:schemas-upnp-org:service:%s\">\r\n%s"         \
                     "</u:%s>\r\n"                                        \
                     "</s:Body>\r\n"                                      \
                     "</s:Envelope>\r\n"

#define PORT_MAPPING_LEASE_TIME "0"
#define PORT_MAPPING_DESCRIPTION "BitStorm"

#define ADD_PORT_MAPPING_PARAMS "<NewRemoteHost></NewRemoteHost>\r\n"      \
                                "<NewExternalPort>%i</NewExternalPort>\r\n"\
                                "<NewProtocol>%s</NewProtocol>\r\n"        \
                                "<NewInternalPort>%i</NewInternalPort>\r\n"\
                                "<NewInternalClient>%s"                    \
                                "</NewInternalClient>\r\n"                 \
                                "<NewEnabled>1</NewEnabled>\r\n"           \
                                "<NewPortMappingDescription>"              \
                                PORT_MAPPING_DESCRIPTION                   \
                                "</NewPortMappingDescription>\r\n"         \
                                "<NewLeaseDuration>"                       \
                                PORT_MAPPING_LEASE_TIME                    \
                                "</NewLeaseDuration>\r\n"

#define DELETE_PORT_MAPPING_PARAMS "<NewRemoteHost></NewRemoteHost>\r\n" \
                                   "<NewExternalPort>%i"                 \
                                   "</NewExternalPort>\r\n"              \
                                   "<NewProtocol>%s</NewProtocol>\r\n"
                                   
#define wanIP  "WANIPConnection:1"
#define wanPPP  "WANPPPConnection:1"                                   


// class CUPnpNatExplorer
CUPnpNatExplorer::CUPnpNatExplorer()
: _reactor(NULL)
, _eventHandler(NULL)
, _discoverTimerID(0)
, _found(false)
{
}

CUPnpNatExplorer::~CUPnpNatExplorer()
{
}

void CUPnpNatExplorer::setSocketReactor(ISocketReactor* reactor)
{
	_reactor = reactor;
}

void CUPnpNatExplorer::setEventHandler(IUPnpNatExplorerEventHandler* handler)
{
	_eventHandler = handler;
}

void CUPnpNatExplorer::start()
{
	LOG_DEBUG("CUPnpNatExplorer::start");
	
	_found = false;
	
	CSocket::setReactor(NULL);
	CSocket::close();
	CSocket::createUDPSocket();
	CSocket::setReactor(_reactor);	
	
	discover();
}

void CUPnpNatExplorer::stop()
{
	CSocket::setReactor(NULL);
	CSocket::close();	
}

void CUPnpNatExplorer::discover()
{
	struct sockaddr_in server;
	memset(&server, 0, sizeof(server));
	server.sin_family = AF_INET;
	server.sin_addr.s_addr = inet_addr(HTTPMU_HOST_ADDRESS);
	server.sin_port = htons(HTTPMU_HOST_PORT);
	
	std::string sendBuffer = SEARCH_REQUEST_STRING;
	
	CSocket::maskRead(true);
	sendto(getHandle(), sendBuffer.data(), sendBuffer.size(), 0, (const struct sockaddr*)&server, sizeof(server));		
	
	if(_discoverTimerID != 0)
	{
		_reactor->removeTimer(_discoverTimerID);
	}
	_discoverTimerID = _reactor->addTimer(this, 5000, true);	 
}

void CUPnpNatExplorer::parseDiscoverReponse(char* buffer, int len)
{
	LOG_DEBUG("CUPnpNatExplorer::parseDiscoverReponse");
	LOG_DEBUG("response = "<<buffer);
	
	gchar* buf = buffer;
	gchar* startDescURL;
	gchar* endDescURL;
	gchar* descURL;
	
	if(g_strstr_len(buf, strlen(buf), HTTP_OK) == NULL)
	{
		return;
	}
	
	if((startDescURL = g_strstr_len(buf, strlen(buf), "http://")) == NULL)
	{
		return;
	}
	
	endDescURL = g_strstr_len(startDescURL, strlen(startDescURL), "\r");
	if(endDescURL == NULL)
	{
		endDescURL = g_strstr_len(startDescURL, strlen(startDescURL), "\n");
		if(endDescURL == NULL)
		{
			return;
		} else if(endDescURL == startDescURL)
		{
			return;
	    }
	} else if(endDescURL == startDescURL)
	{
		return;
	}
	
	descURL = g_strndup(startDescURL, endDescURL-startDescURL);
	LOG_DEBUG("descURL="<<descURL);

	_found = true;
	if(_eventHandler != NULL)	
	{
		_eventHandler->OnGetDescriptionUrl(descURL);
	}

	g_free(descURL);	
}

int CUPnpNatExplorer::handleRead()
{
	struct sockaddr_in from;
	socklen_t length = sizeof(struct sockaddr_in);
	char* buffer = new char[MAX_DISCOVERY_RECEIVE_SIZE];
	int ret = 0;
	for(;;)
	{
		ret = recvfrom(getHandle(), buffer, MAX_DISCOVERY_RECEIVE_SIZE, 0, (struct sockaddr*)&from, &length);
		if(ret == -1
			&& errno == EINTR)
		{
			continue;
		}
		break;
	}
	
	if(ret > 0)
	{
		buffer[ret] = 0;
		parseDiscoverReponse(buffer, ret);	
	}
	
	delete[] buffer;
	
	return 0;
}

int CUPnpNatExplorer::handleWrite()
{
	return 0;
}

void CUPnpNatExplorer::handleClose()
{
	CSocket::setReactor(NULL);
	CSocket::close();
}

void CUPnpNatExplorer::onTimer(unsigned int id)
{
	if(id == _discoverTimerID)
	{
		if(!_found)
		{
			discover();
		}
	}
}


CUPnpNatDescriptionParser::CUPnpNatDescriptionParser()
: _reactor(NULL)
, _eventHandler(NULL)
, _timeoutTimerID(0)
, _state(DPS_INIT)
{
}

CUPnpNatDescriptionParser::~CUPnpNatDescriptionParser()
{
}

void CUPnpNatDescriptionParser::setSocketReactor(ISocketReactor* reactor)
{
	_reactor = reactor;
}

void CUPnpNatDescriptionParser::setDescriptionUrl(const char* url)
{
	_url = url;
}

void CUPnpNatDescriptionParser::setEventHandler(IUPnpNatDescriptionParserEventHandler* handler)
{
	_eventHandler = handler;
}

void CUPnpNatDescriptionParser::start()
{
	_timeoutTimerID = _reactor->addTimer(this, 5000, true);
	connect();
	_state = DPS_WORKING;
}

void CUPnpNatDescriptionParser::stop()
{
	if(_timeoutTimerID != 0)
	{
		_reactor->removeTimer(_timeoutTimerID);
		_timeoutTimerID = 0;
	}
	
	CSocket::setReactor(NULL);
	CSocket::close();
}

int CUPnpNatDescriptionParser::handleRead()
{
	char buffer[1024];
	for(;;)
	{
		int ret = recv(getHandle(), buffer, sizeof(buffer), 0);
		LOG_DEBUG("recv return "<<ret);
		if(ret == 0)
		{
			maskRead(false);
			return -1;
		}
		
		if(ret == -1)
		{
			if(errno == EINTR)
			{
				continue;
			}
			
			if(errno != EAGAIN)
			{
				maskRead(false);
				return -1;
			}
			break;
		}
		
		if(ret > 0)
		{
			_recvBuffer.append(buffer, ret);
		}
	}	
	
	return 0;
}

int CUPnpNatDescriptionParser::handleWrite()
{
	LOG_DEBUG("CUPnpNatDescriptionParser::handleWrite");
	
	sendRequest();
	maskWrite(false);
	
	return 0;
}

void CUPnpNatDescriptionParser::handleClose()
{
	LOG_DEBUG("CUPnpNatDescriptionParser::handleClose");
	
	bool parseResult = false;
	
	if(_recvBuffer.size() > 0)
	{
		char str[1024];
		std::string ip;
		std::string path;
		unsigned short port;
		parseUrl(_url.c_str(), ip, &port, path);		
		
		sprintf(str, "http://%s:%d", ip.c_str(), port);
		
		parseResult= parseResponse(_recvBuffer.c_str(), str, wanIP);
		if(!parseResult)
		{
			parseResult = parseResponse(_recvBuffer.c_str(), str, wanPPP);
		}
	}
	
	if(!parseResult)
	{
		_state = DPS_ERROR;
	}
	else
	{
		_state = DPS_OK;
	}
	
	CSocket::setReactor(NULL);
	CSocket::close();
}

void CUPnpNatDescriptionParser::onTimer(unsigned int id)
{
	if( id == _timeoutTimerID)
	{
		handleClose();
		_timeoutTimerID = 0;		
	}
}

void CUPnpNatDescriptionParser::connect()
{
	std::string ip;
	std::string path;
	unsigned short port;
	parseUrl(_url.c_str(), ip, &port, path);
	
	LOG_DEBUG("ip="<<ip<<" port="<<port<<" path="<<path);
	
	CSocket::createTCPSocket();
	CSocket::setReactor(_reactor);
	CSocket::connect(ip.c_str(), port);
	CSocket::maskWrite(true);		
}

void CUPnpNatDescriptionParser::sendRequest()
{
	std::string ip;
	std::string path;
	unsigned short port;
	parseUrl(_url.c_str(), ip, &port, path);
	
	gchar* httpRequest = g_strdup_printf("GET %s HTTP/1.1\r\nHost: %s:%d\r\n\r\n",
                    path.c_str(), ip.c_str(), port);
   LOG_DEBUG("httpRequest="<<httpRequest);
   send(getHandle(), httpRequest, strlen(httpRequest), 0);
   g_free(httpRequest);
   maskRead(true);	
}

bool CUPnpNatDescriptionParser::parseResponse(const char* response, const char* httpUrl, const char* serviceType)
{
  gchar* httpResponse = (gchar*)response;
  gchar* httpURL = (gchar*)httpUrl;
  gchar* xmlRoot;
  gchar* baseURL;
  gchar* controlURL;
  gchar* service;
  xmlnode* xmlRootNode;
  xmlnode* serviceTypeNode;
  xmlnode* controlURLNode;
  xmlnode* baseURLNode;
  
// LOG_DEBUG("response = \n"<<response);

  /* make sure we have a valid http response */
  if(g_strstr_len(httpResponse, strlen(httpResponse), HTTP_OK) == NULL)
  {
  	return false;
  }

  /* find the root of the xml document */
  if((xmlRoot = g_strstr_len(httpResponse, strlen(httpResponse), "<root")) == NULL)
  {
  	return false;
  }

  /* create the xml root node */
  if((xmlRootNode = xmlnode_from_str(xmlRoot, size_t(-1))) == NULL)
  {
  	return false;
  }

  /* get the baseURL of the device */
  if((baseURLNode = xmlnode_get_child(xmlRootNode, "URLBase")) != NULL)
  {
    baseURL = g_strdup(xmlnode_get_data(baseURLNode));
  } else
  {
    baseURL = g_strdup(httpURL);
  }

  /* get the serviceType child that has the service type as it's data */

  /* get urn:schemas-upnp-org:device:InternetGatewayDevice:1
     and it's devicelist */
  serviceTypeNode = xmlnode_get_child(xmlRootNode, "device");
  while(serviceTypeNode != NULL
  		&& !compareDevice(serviceTypeNode, "urn:schemas-upnp-org:device:InternetGatewayDevice:1"))
  {
    serviceTypeNode = xmlnode_get_next_twin(serviceTypeNode);
  }
  if(serviceTypeNode == NULL)
  {
  	return false;
  }
  serviceTypeNode = xmlnode_get_child(serviceTypeNode, "deviceList");
  if(serviceTypeNode == NULL)
  {
  	return false;
  }

  /* get urn:schemas-upnp-org:device:WANDevice:1
     and it's devicelist */
  serviceTypeNode = xmlnode_get_child(serviceTypeNode, "device");
  while(serviceTypeNode != NULL
  		&& !compareDevice(serviceTypeNode, "urn:schemas-upnp-org:device:WANDevice:1"))
  {
    serviceTypeNode = xmlnode_get_next_twin(serviceTypeNode);
  }
  if(serviceTypeNode == NULL)
  {
  	return false;
  }
  serviceTypeNode = xmlnode_get_child(serviceTypeNode, "deviceList");
  if(serviceTypeNode == NULL)
  {
  	return false;
  }

  /* get urn:schemas-upnp-org:device:WANConnectionDevice:1
     and it's servicelist */
  serviceTypeNode = xmlnode_get_child(serviceTypeNode, "device");
  while(serviceTypeNode != NULL
  		&& !compareDevice(serviceTypeNode, "urn:schemas-upnp-org:device:WANConnectionDevice:1"))
  {
    serviceTypeNode = xmlnode_get_next_twin(serviceTypeNode);
  }
  if(serviceTypeNode == NULL)
  {
  	return false;
  }
  serviceTypeNode = xmlnode_get_child(serviceTypeNode, "serviceList");
  if(serviceTypeNode == NULL)
  {
  	return false;
  }

  /* get the serviceType variable passed to this function */
  service = g_strdup_printf(SEARCH_REQUEST_DEVICE, serviceType);
  serviceTypeNode = xmlnode_get_child(serviceTypeNode, "service");
  while(serviceTypeNode != NULL
  		&& !compareService(serviceTypeNode, service))
  {
    serviceTypeNode = xmlnode_get_next_twin(serviceTypeNode);
  }

  g_free(service);
  if(serviceTypeNode == NULL)
  {
  	return false;
  }

  /* get the controlURL of the service */
  if((controlURLNode = xmlnode_get_child(serviceTypeNode, "controlURL")) == NULL)
  {
 	return false;
  }

  if(g_strstr_len(xmlnode_get_data(controlURLNode),
                  SIZEOF_HTTP, "http://") == NULL &&
     g_strstr_len(xmlnode_get_data(controlURLNode),
                  SIZEOF_HTTP, "HTTP://") == NULL) {
    controlURL = g_strdup_printf("%s%s", baseURL,
                                 xmlnode_get_data(controlURLNode));
  }else{
    controlURL = g_strdup(xmlnode_get_data(controlURLNode));
  }
  
  if(_eventHandler != NULL)	
  {
  	_eventHandler->OnGetControlUrl(controlURL, serviceType);
  }
  
  return true;
}

bool CUPnpNatDescriptionParser::compareService(xmlnode* service, const char* serviceType)
{
  xmlnode* serviceTypeNode = xmlnode_get_child(service, "serviceType");
  if(serviceTypeNode == NULL) {
    return FALSE;
  }
  return !g_ascii_strcasecmp(xmlnode_get_data(serviceTypeNode), serviceType);	
}

bool CUPnpNatDescriptionParser::compareDevice(xmlnode* device, const char* deviceType)
{
  xmlnode* deviceTypeNode = xmlnode_get_child(device, "deviceType");
  if(deviceTypeNode == NULL) {
    return FALSE;
  }
  return !g_ascii_strcasecmp(xmlnode_get_data(deviceTypeNode), deviceType);	
}

TDescriptionParserState CUPnpNatDescriptionParser::getState()
{
	return _state;
}




const char* CPnpNatController::ActionAdd = "AddPortMapping";
const char* CPnpNatController::ActionDelete = "DeletePortMapping";

CPnpNatController::CPnpNatController()
: _reactor(NULL)
, _state(CS_INIT)
{
}

CPnpNatController::~CPnpNatController()
{
}

void CPnpNatController::setSocketReactor(ISocketReactor* reactor)
{
	_reactor = reactor;
}

void CPnpNatController::setControlUrl(const char*controlUrl, const char* service)
{
	_controlUrl = controlUrl;
	_service = service;
}

void CPnpNatController::setAction(const char* action, unsigned short port, const char* protocol)
{
	_actionName = action;
	_port = port;
	_protocol = protocol;	
}

void CPnpNatController::start()
{
	std::string ip;
	std::string path;
	unsigned short port;
	parseUrl(_controlUrl.c_str(), ip, &port, path);
	
	_state = CS_WORKING;
	
	CSocket::createTCPSocket();
	CSocket::setReactor(_reactor);
	CSocket::maskWrite(true);
	CSocket::connect(ip.c_str(), port);	
	
	_timeoutTimerID = _reactor->addTimer(this, 5000, true);
}

void CPnpNatController::stop()
{
	LOG_DEBUG("CPnpNatController stopping");
	
	if(_timeoutTimerID != 0
		&& _reactor != NULL)
	{
		_reactor->removeTimer(_timeoutTimerID);
		_timeoutTimerID = 0;
	}
	
	CSocket::setReactor(NULL);
	CSocket::close();	
	
	LOG_DEBUG("CPnpNatController stoped");
}

TControllerState CPnpNatController::getState()
{
	return _state;
}

int CPnpNatController::handleRead()
{
	char buffer[1024];
	for(;;)
	{
		int ret = recv(getHandle(), buffer, sizeof(buffer), 0);
		LOG_DEBUG("recv return "<<ret);
		if(ret == 0)
		{
			maskRead(false);
			return -1;
		}
		
		if(ret == -1)
		{
			if(errno == EINTR)
			{
				continue;
			}
			
			if(errno != EAGAIN)
			{
				maskRead(false);
				return -1;
			}
			break;
		}
		
		if(ret > 0)
		{
			_recvBuffer.append(buffer, ret);
		}
	}	
	
	return 0;	
}

int CPnpNatController::handleWrite()
{
	//get local ip from socket handle
	struct sockaddr_in addr;
	socklen_t len = sizeof(addr);
	getsockname(getHandle(), (struct sockaddr*)&addr, &len);
	_localIP = inet_ntoa(addr.sin_addr);	
	
	LOG_DEBUG("localIP="<<_localIP);
	
	sendRequest();
	maskWrite(false);
	return 0;
}

void CPnpNatController::handleClose()
{
	if(_recvBuffer.find(HTTP_OK) != _recvBuffer.npos)
	{
		_state = CS_OK;
		LOG_DEBUG("port mapping ok");
	}
	else
	{
		_state = CS_ERROR;
		LOG_DEBUG("port mapping error");
	}

	CSocket::setReactor(NULL);
	CSocket::close();
}

void CPnpNatController::onTimer(unsigned int id)
{
	if(id == _timeoutTimerID
		&& _state == CS_WORKING)
	{
		handleClose();
		_timeoutTimerID = 0;
	}
}

void CPnpNatController::sendRequest()
{	
	gchar* actionParams = g_strdup_printf(ADD_PORT_MAPPING_PARAMS,
					_port,
					_protocol.c_str(),
					_port,
					_localIP.c_str());
	gchar* soapMessage = g_strdup_printf(SOAP_ACTION,
					_actionName.c_str(),
					_service.c_str(),
					actionParams,
					_actionName.c_str());
					
	std::string ip;
	std::string path;
	unsigned short port;
	parseUrl(_controlUrl.c_str(), ip, &port, path);				
	
	gchar* addressPortOfControl = g_strdup_printf("%s:%u", ip.c_str(), port);
	
	gchar* actionMessage = g_strdup_printf(HTTP_HEADER_ACTION,
					path.c_str(),
					addressPortOfControl,
					_service.c_str(),
					_actionName.c_str(),
					strlen(soapMessage));
					
	gchar* totalSendMessage = g_strdup_printf("%s%s", actionMessage, soapMessage);
	
	int ret = send(getHandle(), totalSendMessage, strlen(totalSendMessage), 0);	
	
	LOG_DEBUG("msg len="<<strlen(totalSendMessage)<<" send return "<<ret);
	LOG_DEBUG("totalSendMessage = \n"<<totalSendMessage);
	
	g_free(actionParams);
	g_free(soapMessage);
	g_free(addressPortOfControl);
	g_free(actionMessage);
	g_free(totalSendMessage);
	
	maskRead(true);												
}


CUPnpNat::CUPnpNat()
: _reactor(NULL)
, _state(NS_INIT)
, _discoverTimerID(0)
, _controlTimerID(0)
{
}

CUPnpNat::~CUPnpNat()
{
}

void CUPnpNat::setReactor(ISocketReactor* reactor)
{
	_reactor = reactor;
}

void CUPnpNat::start()
{
	_state = NS_DISCOVER;
	_natExplorer.setSocketReactor(_reactor);
	_natExplorer.setEventHandler(this);
	_natExplorer.start();
	_discoverTimerID = _reactor->addTimer(this, 10000, true);
	_controlTimerID = _reactor->addTimer(this, 2000, false);
}

void CUPnpNat::stop()
{
	if(_discoverTimerID != 0)
	{
		_reactor->removeTimer(_discoverTimerID);
		_discoverTimerID = 0;
	}

	if(_controlTimerID != 0)
	{
		_reactor->removeTimer(_controlTimerID);
		_controlTimerID = 0;
	}	
	
	_natExplorer.stop();
	_natParser.stop();	
	
	TNatPortInfoList::iterator iter = _portList.begin();
	for(; iter!=_portList.end(); ++iter)
	{
		if(iter->controller != NULL)
		{
			iter->controller->stop();
			delete iter->controller;
			iter->controller = NULL;
		}
	}	
	_portList.clear();
}

TUPnpNATState CUPnpNat::getState()
{
	return _state;
}

void CUPnpNat::addPortMapping(unsigned int port,const char* protocol)
{
	TNatPortInfo info;
	info.port = port;
	info.protocol = protocol;
		
	if(_state != NS_OK)
	{
		info.controller = NULL;
	}
	else
	{
		info.controller = new CPnpNatController();
		info.controller->setSocketReactor(_reactor);
		info.controller->setControlUrl(_controlUrl.c_str(), _service.c_str());
		info.controller->setAction(CPnpNatController::ActionAdd, port, protocol);
		info.controller->start();
	}
	
	_portList.push_back(info);
}

void CUPnpNat::removePortMapping(unsigned int port,const char* protocol)
{
}

void CUPnpNat::onTimer(unsigned int id)
{
	if(id == _discoverTimerID)
	{
		if(_state == NS_OK)
		{
			_discoverTimerID = 0;
		}
		else
		{
			LOG_INFO("NAT discover timeout");
			_natExplorer.stop();			
			start();			
		}
	}
	
	if(id == _controlTimerID)
	{
		if(_state == NS_OK)
		{
			TNatPortInfoList::iterator iter = _portList.begin();
			for(; iter!=_portList.end(); ++iter)
			{
				if(iter->controller == NULL)
				{
					iter->controller = new CPnpNatController();
					iter->controller->setSocketReactor(_reactor);
					iter->controller->setControlUrl(_controlUrl.c_str(), _service.c_str());
					iter->controller->setAction(CPnpNatController::ActionAdd, iter->port, iter->protocol.c_str());
					iter->controller->start();					
				}
			}			
		}
	}
}

void CUPnpNat::OnGetDescriptionUrl(const char* url)
{
	LOG_DEBUG("CUPnpNat::OnGetDescriptionUrl, url="<<url);
	
	if(_state == NS_GETDESCRIPTION)
	{
		return;
	}
	
	_state = NS_GETDESCRIPTION;
	
	_natParser.stop();
	_natParser.setSocketReactor(_reactor);
	_natParser.setEventHandler(this);
	_natParser.setDescriptionUrl(url);
	_natParser.start();
}

void CUPnpNat::OnGetControlUrl(const char* controlUrl, const char* service)
{
	LOG_DEBUG("controlUrl="<<controlUrl<<" service="<<service);
	
	_controlUrl = controlUrl;
	_service = service;
	
	_state = NS_OK;
}


