//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Device/Detector/SphericalDetector.h
//! @brief     Defines class SphericalDetector.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_DEVICE_DETECTOR_SPHERICALDETECTOR_H
#define BORNAGAIN_DEVICE_DETECTOR_SPHERICALDETECTOR_H

#include "Device/Detector/IDetector.h"

class IPixel;

//! A detector with coordinate axes along angles phi and alpha.

class SphericalDetector : public IDetector {
public:
    SphericalDetector(std::array<std::shared_ptr<Scale>, 2> axes);

    //! Returns a detector with given phi and alpha axes.
    //! @param n_phi number of phi-axis bins
    //! @param phi_min low edge of first phi-bin
    //! @param phi_max upper edge of last phi-bin
    //! @param n_alpha number of alpha-axis bins
    //! @param alpha_min low edge of first alpha-bin
    //! @param alpha_max upper edge of last alpha-bin
    SphericalDetector(size_t n_phi, double phi_min, double phi_max, size_t n_alpha,
                      double alpha_min, double alpha_max);

    //! Returns a detector with phi and alpha axes that have equal width and binning.
    //! @param n_bin number of bins per direction
    //! @param width full width of angular range in phi or alpha
    //! @param phi   central phi angle
    //! @param alpha central alpha angle
    SphericalDetector(size_t n_bin, double width, double phi, double alpha);

    SphericalDetector(const SphericalDetector& other);
    ~SphericalDetector() override = default;

    SphericalDetector* clone() const override;
    std::string className() const override { return "SphericalDetector"; }

    //! Returns default axes units
    Coords defaultCoords() const override { return Coords::RADIANS; }

    const CoordSystem2D* scatteringCoords(const Beam& beam) const override;

    //! Creates an IPixel for the given Datafield object and index
    IPixel* createPixel(size_t index) const override;

    //! Returns index of pixel that contains the specular wavevector.
    //! If no pixel contains this specular wavevector, the number of pixels is
    //! returned. This corresponds to an overflow index.
    size_t indexOfSpecular(const Beam& beam) const override;
};

#endif // BORNAGAIN_DEVICE_DETECTOR_SPHERICALDETECTOR_H
