#!/usr/bin/env python3
"""
Basic example for regular small-angle scattering (SAS).
Sample is a dilute assembly of ordered dodecahedra.
"""
import bornagain as ba
from bornagain import ba_plot as bp, deg, nm


def get_sample():
    vacuum = ba.RefractiveMaterial("Vacuum", 0, 0)
    material_solution = ba.RefractiveMaterial("Solution", 6e-6, 2e-8)
    material_particle = ba.RefractiveMaterial("Particle", 6e-4, 2e-8)

    # Finite sample layer, contains particles in solution:
    ff = ba.Dodecahedron(12*nm)
    particle = ba.Particle(material_particle, ff)
    layout = ba.ParticleLayout()
    layout.addParticle(particle)
    solution_layer = ba.Layer(material_solution, 1000*nm)
    # TODO: make intensity proportional to thickness,
    #       https://github.com/scgmlz/BornAgain/issues/1222
    solution_layer.addLayout(layout)

    # Flat sample layer sandwiched between semi-infinite vacuum layers:
    sample = ba.MultiLayer()
    sample.addLayer(ba.Layer(vacuum))
    sample.addLayer(solution_layer)
    sample.addLayer(ba.Layer(vacuum))
    return sample


def get_simulation(sample):
    # Beam from above (perpendicular to sample):
    beam = ba.Beam(1e9, 0.4*nm, 90*deg)

    # Detector opposite to source:
    detPos = 2000 # distance from sample center to detector in mm
    detWid = 500  # detector width in mm
    n = <%= sm ? 11 : 200 %>  # number of pixels per direction
    det = ba.RectangularDetector(n, detWid, n, detWid)
    det.setPerpendicularToDirectBeam(detPos, detWid/2, detWid/2)

    return ba.ScatteringSimulation(beam, sample, det)


if __name__ == '__main__':
    sample = get_sample()
    simulation = get_simulation(sample)
    result = simulation.simulate()
    bp.plot_simulation_result(result)
