/* -*- Mode: C++; tab-width: 4 -*- */
/* vi: set ts=4 sw=4 noexpandtab: */

/*
 * Copyright (C) 2010 Florian Weimer <fw@deneb.enyo.de>
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#ifndef DIAGNOSTICS__EXTENSIONS__STACKTRACE__RAW_PROCESS__INCLUDE_GUARD
#define DIAGNOSTICS__EXTENSIONS__STACKTRACE__RAW_PROCESS__INCLUDE_GUARD

#include <diagnostics/frame/namespace.hpp>
#include <diagnostics/extensions/stacktrace/namespace.hpp>

#include <fcntl.h> // for O_* constants
#include <spawn.h>

#include <diagnostics/extensions/stacktrace/posix.hpp>
#include <diagnostics/extensions/stacktrace/error.hpp>
#include <diagnostics/extensions/stacktrace/signal.hpp>

DIAGNOSTICS_NAMESPACE_BEGIN;
STACKTRACE_NAMESAPCE_BEGIN;
namespace POSIX {

class SpawnFileActions {
  posix_spawn_file_actions_t actions;

  SpawnFileActions(const SpawnFileActions &);
  SpawnFileActions &operator=(const SpawnFileActions &);
public:
  SpawnFileActions()
  {
    POSIX::Error::Unless(posix_spawn_file_actions_init(&actions) == 0);
  }

  ~SpawnFileActions()
  {
    posix_spawn_file_actions_destroy(&actions);
  }

  void AddClose(int fd)
  {
    POSIX::Error::Unless
      (posix_spawn_file_actions_addclose(&actions, fd) == 0);
  }

  void AddOpen(int fd, const char *path, int flag, mode_t mode)
  {
    POSIX::Error::Unless(posix_spawn_file_actions_addopen
                               (&actions, fd, path, flag, mode) == 0);
  }

  void AddDup2(int from, int to)
  {
    POSIX::Error::Unless(posix_spawn_file_actions_adddup2
                               (&actions, from, to) == 0);
  }

  posix_spawn_file_actions_t *GetRaw()
  {
    return &actions;
  }
};

class SpawnAttr {
  posix_spawnattr_t attr;

  SpawnAttr(const SpawnAttr &);
  SpawnAttr &operator=(const SpawnAttr &);

public:
  SpawnAttr()
  {
    POSIX::Error::Unless(posix_spawnattr_init(&attr) == 0);
  }

  ~SpawnAttr()
  {
    posix_spawnattr_destroy(&attr);
  }

  void SetFlags(short flags)
  {
    POSIX::Error::Unless(posix_spawnattr_setflags(&attr, flags) == 0);
  }

  void SetSigDefault(const Sigset &sig)
  {
    POSIX::Error::Unless(posix_spawnattr_setsigdefault
                               (&attr, sig.GetRaw()) == 0);
  }

  posix_spawnattr_t *GetRaw()
  {
    return &attr;
  }
};

} // POSIX
STACKTRACE_NAMESAPCE_END;
DIAGNOSTICS_NAMESPACE_END;

#endif

