/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file audio_gstreamer.c
 * \brief GStreamer audio plugin
 */

#include <ffgtk.h>
#include <preferences.h>
#include <gst/interfaces/propertyprobe.h>
#include <gst/app/gstappsrc.h>
#include <gst/app/gstappsink.h>
#include <gst/app/gstappbuffer.h>

/** Internal output device list */
static GList *psDeviceOutputList = NULL;
/** Internal input device list */
static GList *psDeviceInputList = NULL;
/** This flag informs about gstreamer initialization status */
static gboolean bAudioInitFailed = TRUE;
/** predefined backup values */
static gint nGstreamerChannels = 2;
static gint nGstreamerSampleRate = 8000;
static gint nGstreamerBitsPerSample = 16;

/** Device information, description and name */
struct sDeviceInfo {
	gchar nType;
	gchar *pnDescr;
	gchar *pnName;
};

/** Device test list */
struct sDeviceTest {
	gchar nType;
	gchar *pnPipe;
	gchar *pnTest;
} asDeviceTest[] = {
	{ 1, "pulsesink", "pulsesinkpresencetest" },
	{ 1, "alsasink", "alsasinkpresencetest" },
	{ 2, "pulsesrc", "pulsesrcpresencetest" },
	{ 2, "alsasrc", "alsasrcpresencetest" },
	{ 0, NULL, NULL }
};

struct sPipes {
	GstElement *psInPipe;
	GstElement *psOutPipe;
};

/** Get and set selected gstreamer output device */
CREATE_STRING_PREFS( gstreamer, SelectedOutputDevice, "/plugins/gstreamer/output" );
/** Get and set selected gstreamer input device */
CREATE_STRING_PREFS( gstreamer, SelectedInputDevice, "/plugins/gstreamer/input" );

/**
 * \brief Stop and clean gstreamer pipeline
 * \param psBus gstreamer bus
 * \param psMessage gstreamer message
 * \param pPipeline pipeline to pipeline we want to stop
 * \return TRUE if we received a EOS for the requested pipeline, otherwise FALSE
 */
static gboolean pipelineCleaner( GstBus *psBus, GstMessage *psMessage, gpointer pPipeline ) {
	gboolean bResult = TRUE;

	/* If we receive a End-Of-Stream message for the requested pipeline, stop the pipeline */
	if ( GST_MESSAGE_TYPE( psMessage ) == GST_MESSAGE_EOS && GST_MESSAGE_SRC( psMessage ) == GST_OBJECT_CAST( pPipeline ) ) {
		bResult = FALSE;
		gst_element_set_state( GST_ELEMENT( pPipeline ), GST_STATE_NULL );
		g_object_unref( pPipeline );
	}

	return bResult;
}

/**
 * \brief Detect supported audio devices
 * \return 0
 */
static int gstreamerAudioDetectDevices( void ) {
	GError *psError = NULL;
	int nDevice = 0;

	/* initialize gstreamer */
	bAudioInitFailed = !gst_init_check( NULL, NULL, &psError );
	if ( bAudioInitFailed == TRUE ) {
		Debug( KERN_WARNING, _( "GStreamer failed to initialized (%s)\n" ), psError ? psError -> message : "" );
		if ( psError != NULL ) {
			g_error_free( psError );
			psError = NULL;
		}
		return -1;
	}

	while ( asDeviceTest[ nDevice ].nType != 0 ) {
		GstElement *psElement = NULL;
		guint nIndex = 0;

		/* check if we have a alsa sink within gstreamer */
		psElement = gst_element_factory_make( asDeviceTest[ nDevice ].pnPipe, asDeviceTest[ nDevice ].pnTest );
		if ( psElement != NULL ) {
			GstPropertyProbe *psProbe = NULL;
			const GParamSpec *psSpec = NULL;
			GValueArray *psArray = NULL;

			/* Set playback mode to pause */
			gst_element_set_state( psElement, GST_STATE_PAUSED );
			psProbe = GST_PROPERTY_PROBE( psElement );
			psSpec = gst_property_probe_get_property( psProbe, "device" );

			/* now try to receive the device name and add it to the internal list */
			psArray = gst_property_probe_probe_and_get_values( psProbe, psSpec );
			if ( psArray != NULL ) {
				for ( nIndex = 0; nIndex < psArray -> n_values; nIndex++ ) {
					GValue *psDevice = NULL;
					gchar *pnName = NULL;

					psDevice = g_value_array_get_nth( psArray, nIndex );
					g_object_set_property( G_OBJECT( psElement ), "device", psDevice );
					if ( asDeviceTest[ nDevice ].nType == 1 ) {
						g_object_get( G_OBJECT( psElement ), "device-name", &pnName, NULL );
					}
					if ( pnName == NULL ) {
						g_object_get( G_OBJECT( psElement ), "device", &pnName, NULL );
					}
					if ( pnName != NULL ) {
						struct sDeviceInfo *psInfo = g_malloc( sizeof( *psInfo ) );
						psInfo -> pnDescr = g_strdup_printf( "%s device=%s", asDeviceTest[ nDevice ].pnPipe, g_value_get_string( psDevice ) );
						if ( nDevice == 0 ) {
							gchar *pnSpec = strrchr( pnName, '.' );
							if ( pnSpec != NULL ) {
								psInfo -> pnName = g_strdup_printf( "Pulseaudio (%s)", pnSpec + 1 );
							} else {
								psInfo -> pnName = g_strdup( "Pulseaudio" );
							}
						} else {
							psInfo -> pnName = g_strdup( pnName );
						}
						psInfo -> nType = asDeviceTest[ nDevice ].nType;

						if ( psInfo -> nType == 1 ) {
							psDeviceOutputList = g_list_append( psDeviceOutputList, psInfo );
						} else {
							psDeviceInputList = g_list_append( psDeviceInputList, psInfo );
						}
						g_free( pnName );
					}
				}
				g_value_array_free( psArray );
			}

			gst_element_set_state( psElement, GST_STATE_NULL );
			gst_object_unref( GST_OBJECT( psElement ) );
		}

		nDevice++;
	}

	return 0;
}

/**
 * \brief Set buffer size we want to use
 * \param nBufferSize requested buffer size
 */
static void gstreamerAudioSetBufferOutputSize( gpointer pPriv, unsigned nBufferSize ) {
	GstElement *psSrc = NULL;
	GstElement *psPipeline = pPriv;

	/* Try to receive pipeline bin name */
	psSrc = gst_bin_get_by_name( GST_BIN( psPipeline ), "ffgtk_src" );
	if ( psSrc != NULL ) {
		/* set blocksize */
		g_object_set( G_OBJECT( psSrc ), "blocksize", nBufferSize, NULL );
		g_object_unref( psSrc );
	}
}

static void gstreamerAudioSetBufferInputSize( gpointer pPriv, unsigned nBufferSize ) {
	GstElement *psSink = NULL;
	GstElement *psPipeline = pPriv;

	/* Try to receive pipeline bin name */
	psSink = gst_bin_get_by_name( GST_BIN( psPipeline ), "ffgtk_sink" );
	if ( psSink != NULL ) {
		/* set blocksize */
		g_object_set( G_OBJECT( psSink ), "blocksize", nBufferSize, NULL );
		g_object_unref( psSink );
	}
}

/**
 * \brief Initialize audio device
 * \param nChannels number of channels
 * \param nSampleRate sample rate
 * \param nBitsPerSample number of bits per samplerate
 * \return TRUE on success, otherwise error
 */
static int gstreamerAudioInit( unsigned char nChannels, unsigned short nSampleRate, unsigned char nBitsPerSample ) {
	/* TODO: Check if configuration is valid and usable */
	nGstreamerChannels = nChannels;
	nGstreamerSampleRate = nSampleRate;
	nGstreamerBitsPerSample = nBitsPerSample;

	return 0;
}

/**
 * \brief Open audio device
 * \return private data or NULL on error
 */
static void *gstreamerAudioOpen( void ) {
	gchar *pnCommand = NULL;
	GError *psError = NULL;
	GstState sCurrent;
	struct sPipes *psPipes = NULL;

	Debug( KERN_DEBUG, "started\n" );
	if ( gstreamerGetSelectedOutputDevice( getActiveProfile() ) == NULL ) {
		return NULL;
	}

	if ( gstreamerGetSelectedInputDevice( getActiveProfile() ) == NULL ) {
		return NULL;
	}

	psPipes = g_malloc( sizeof( struct sPipes ) );
	if ( psPipes == NULL ) {
		return NULL;
	}

	/* Create command for output pipeline */
	pnCommand = g_strdup_printf( "appsrc is-live=true name=ffgtk_src"
		" ! audio/x-raw-int"
		",rate=%d"
		",channels=%d"
		",width=%d"
		",depth=%d"
		",signed=true,endianness=1234"
		" ! %s",
		nGstreamerSampleRate, nGstreamerChannels, nGstreamerBitsPerSample, nGstreamerBitsPerSample,
		gstreamerGetSelectedOutputDevice( getActiveProfile() ) );
	Debug( KERN_DEBUG, "pnCommand %s\n", pnCommand );
	
	/* Start pipeline */
	psPipes -> psOutPipe = gst_parse_launch( pnCommand, &psError );
	g_free( pnCommand );
	if ( psError == NULL ) {
		/* Try to set pipeline state to playing */
		gst_element_set_state( psPipes -> psOutPipe, GST_STATE_PLAYING );

		/* Now read the current state and verify that we have a good pipeline */
		gst_element_get_state( psPipes -> psOutPipe, &sCurrent, NULL, GST_SECOND );

		if ( !( sCurrent == GST_STATE_PLAYING || sCurrent == GST_STATE_PAUSED ) ) {
			gst_element_set_state( psPipes -> psOutPipe, GST_STATE_NULL );
			gst_object_unref( GST_OBJECT( psPipes -> psOutPipe ) );
			psPipes -> psOutPipe = NULL;
			g_free( psPipes );
			psPipes = NULL;
			Debug( KERN_DEBUG, "Error: Could not start output pipe (%d)\n", sCurrent );
			return NULL;
		} else {
			gstreamerAudioSetBufferOutputSize( psPipes -> psOutPipe, 160 );

			//Debug( KERN_DEBUG, "Ok\n" );
		}
	} else {
		g_error_free( psError );
		psPipes -> psOutPipe = NULL;

		g_free( psPipes );
		psPipes = NULL;
		Debug( KERN_DEBUG, "Error: Could not launch output pipe\n" );

		return NULL;
	}

	/* Create command for input pipeline */
	pnCommand = g_strdup_printf( "%s ! appsink drop=true max_buffers=1"
		" caps=audio/x-raw-int"
		",rate=%d"
		",channels=%d"
		",width=%d"
		" name=ffgtk_sink",
		gstreamerGetSelectedInputDevice( getActiveProfile() ),
	    nGstreamerSampleRate, nGstreamerChannels, nGstreamerBitsPerSample );
	Debug( KERN_DEBUG, "pnCommand %s\n", pnCommand );
	
	/* Start pipeline */
	psPipes -> psInPipe = gst_parse_launch( pnCommand, &psError );
	if ( psError == NULL ) {
		/* Try to set pipeline state to playing */
		gst_element_set_state( psPipes -> psInPipe, GST_STATE_PLAYING );

		/* Now read the current state and verify that we have a good pipeline */
		gst_element_get_state( psPipes -> psInPipe, &sCurrent, NULL, GST_SECOND );

		if ( sCurrent != GST_STATE_PLAYING ) {
			gst_element_set_state( psPipes -> psInPipe, GST_STATE_NULL );
			gst_object_unref( GST_OBJECT( psPipes -> psInPipe ) );
			psPipes -> psInPipe = NULL;
			/* TODO: free output pipe */
			g_free( psPipes );
			psPipes = NULL;
			Debug( KERN_DEBUG, "Error: Could not start input pipe\n" );
		} else {
			gstreamerAudioSetBufferInputSize( psPipes -> psInPipe, 160 );

			//Debug( KERN_DEBUG, "Ok\n" );
		}
	} else {
		g_error_free( psError );
		psPipes -> psInPipe = NULL;

		g_free( psPipes );
		psPipes = NULL;
		Debug( KERN_DEBUG, "Error: Could not launch input pipe\n" );
	}

	g_free( pnCommand );

	return psPipes;
}

/**
 * \brief Write audio data
 * \param pPriv private data
 * \param pnData audio data
 * \param nSize size of audio data
 * \return bytes written or error code
 */
static int gstreamerAudioWrite( void *pPriv, unsigned char *pnData, unsigned int nSize ) {
	gchar *pnTmp = NULL;
	GstBuffer *psBuffer = NULL;
	GstElement *psSrc = NULL;
	struct sPipes *psPipes = pPriv;
	unsigned int nWritten = -1;

	if ( psPipes == NULL ) {
		return 0;
	}

	g_return_val_if_fail( GST_IS_BIN( psPipes -> psOutPipe ), -2 );

	psSrc = gst_bin_get_by_name( GST_BIN( psPipes -> psOutPipe ), "ffgtk_src" );
	if ( psSrc != NULL ) {
		pnTmp = ( gchar * ) g_malloc0( nSize );
		memcpy( ( char * ) pnTmp, ( char * ) pnData, nSize );
		psBuffer = gst_app_buffer_new( pnTmp, nSize, ( GstAppBufferFinalizeFunc ) g_free, pnTmp );
		gst_app_src_push_buffer( GST_APP_SRC( psSrc ), psBuffer );
		nWritten = nSize;
		g_object_unref( psSrc );
	}

	return nWritten;
}

static int gstreamerAudioRead( void *pPriv, unsigned char *pnData, unsigned int nSize ) {
	GstBuffer *psBuffer = NULL;
	GstElement *psSink = NULL;
	struct sPipes *psPipes = pPriv;
	unsigned int nRead =  0;

	if ( psPipes == NULL ) {
		return 0;
	}

	g_return_val_if_fail( GST_IS_BIN( psPipes -> psInPipe ), -2 );

	psSink = gst_bin_get_by_name( GST_BIN( psPipes -> psInPipe ), "ffgtk_sink" );
	if ( psSink != NULL ) {
		psBuffer = gst_app_sink_pull_buffer( GST_APP_SINK( psSink ) );
		if ( psBuffer != NULL ) {
			nRead = MIN( GST_BUFFER_SIZE( psBuffer ), nSize );
			memcpy( pnData, GST_BUFFER_DATA( psBuffer ), nRead );
			gst_buffer_unref( psBuffer );
		}
		g_object_unref( psSink );
	}

	return nRead;
}

/**
 * \brief Stop and remove pipeline
 * \param pPriv private data
 * \param bForce force quit
 * \return error code
 */
int gstreamerAudioClose( void *pPriv, gboolean bForce ) {
	struct sPipes *psPipes = pPriv;

	if ( psPipes == NULL ) {
		return 0;
	}

	if ( psPipes -> psOutPipe != NULL ) {
		GstElement *psSrc = gst_bin_get_by_name( GST_BIN( psPipes -> psOutPipe ), "ffgtk_src" );
		if ( psSrc != NULL ) {
			GstBus *psBus = gst_pipeline_get_bus( GST_PIPELINE( psPipes -> psOutPipe ) );
			gst_bus_add_watch( psBus, pipelineCleaner, psPipes -> psOutPipe );
			gst_app_src_end_of_stream( GST_APP_SRC( psSrc ) );
			gst_object_unref( psBus );
			psPipes -> psOutPipe = NULL;
		}
	}

	if ( psPipes -> psInPipe != NULL ) {
		gst_element_set_state( psPipes -> psInPipe, GST_STATE_NULL );
		gst_object_unref( psPipes -> psInPipe );
		psPipes -> psOutPipe = NULL;
	}

	free( psPipes );
	psPipes = NULL;

	return 0;
}

int gstreamerAudioDeinit( void ) {
	return 0;
}

/**
 * \brief Display gstreamer preferences window
 */
static void gstreamerPreferences( void ) {
	GtkWidget *psDialog;
	GtkWidget *psComboBoxOutput;
	GtkWidget *psComboBoxInput;
	GtkWidget *psLabelOutput;
	GtkWidget *psLabelInput;
	GtkWidget *psBox;
	int nIndexOutput = 0;
	int nIndexInput = 0;
	GList *psListOutput = psDeviceOutputList;
	GList *psListInput = psDeviceInputList;
	struct sProfile *psProfile = getActiveProfile();

	psDialog = gtk_dialog_new_with_buttons( _( "GStreamer Preferences" ), NULL, 0, GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, NULL );
	psComboBoxOutput = gtk_combo_box_new_text();
	psLabelOutput = gtk_label_new( "" );

	psComboBoxInput = gtk_combo_box_new_text();
	psLabelInput = gtk_label_new( "" );

	gtk_label_set_markup( GTK_LABEL( psLabelOutput ), _( "<b>Select output device:</b>" ) );
	psBox = gtk_dialog_get_content_area( GTK_DIALOG( psDialog ) );
	gtk_box_pack_start( GTK_BOX( psBox ), psLabelOutput, FALSE, TRUE, 10 );

	while ( psListOutput != NULL && psListOutput -> data != NULL ) {
		struct sDeviceInfo *psInfo = psListOutput -> data;

		gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBoxOutput ), psInfo -> pnName );

		if ( gstreamerGetSelectedOutputDevice( psProfile ) != NULL ) {
			if ( !strcmp( psInfo -> pnDescr, gstreamerGetSelectedOutputDevice( psProfile ) ) ) {
				gtk_combo_box_set_active( GTK_COMBO_BOX( psComboBoxOutput ), nIndexOutput );
			}
		} else {
			gtk_combo_box_set_active( GTK_COMBO_BOX( psComboBoxOutput ), 0 );
		}
		nIndexOutput++;

		psListOutput = psListOutput -> next;
	}

	gtk_box_pack_start( GTK_BOX( psBox ), psComboBoxOutput, FALSE, TRUE, 5 );

	gtk_label_set_markup( GTK_LABEL( psLabelInput ), _( "<b>Select input device:</b>" ) );
	gtk_box_pack_start( GTK_BOX( psBox ), psLabelInput, FALSE, TRUE, 10 );

	while ( psListInput != NULL && psListInput -> data != NULL ) {
		struct sDeviceInfo *psInfo = psListInput -> data;

		gtk_combo_box_append_text( GTK_COMBO_BOX( psComboBoxInput ), psInfo -> pnName );

		if ( gstreamerGetSelectedInputDevice( psProfile ) != NULL ) {
			if ( !strcmp( psInfo -> pnDescr, gstreamerGetSelectedInputDevice( psProfile ) ) ) {
				gtk_combo_box_set_active( GTK_COMBO_BOX( psComboBoxInput ), nIndexInput );
			}
		} else {
			gtk_combo_box_set_active( GTK_COMBO_BOX( psComboBoxInput ), 0 );
		}
		nIndexInput++;

		psListInput = psListInput -> next;
	}
	
	gtk_box_pack_start( GTK_BOX( psBox ), psComboBoxInput, FALSE, TRUE, 5 );

	gtk_widget_set_size_request( psDialog, 300, 250 );
	gtk_widget_show( GTK_WIDGET( psLabelOutput ) );
	gtk_widget_show( GTK_WIDGET( psComboBoxOutput ) );
	gtk_widget_show( GTK_WIDGET( psLabelInput ) );
	gtk_widget_show( GTK_WIDGET( psComboBoxInput ) );
	gtk_dialog_run( GTK_DIALOG( psDialog ) );

	prefsAddNone( getActiveProfile(), "/plugins/gstreamer" );

	psListOutput = psDeviceOutputList;
	nIndexOutput = 0;

	while ( psListOutput != NULL && psListOutput -> data != NULL ) {
		struct sDeviceInfo *psInfo = psListOutput -> data;

		if ( gtk_combo_box_get_active( GTK_COMBO_BOX( psComboBoxOutput ) ) == nIndexOutput ) {
			gstreamerSetSelectedOutputDevice( psProfile, psInfo -> pnDescr );
			break;
		}

		psListOutput = psListOutput -> next;
		nIndexOutput++;
	}

	psListInput = psDeviceInputList;
	nIndexInput = 0;

	while ( psListInput != NULL && psListInput -> data != NULL ) {
		struct sDeviceInfo *psInfo = psListInput -> data;

		if ( gtk_combo_box_get_active( GTK_COMBO_BOX( psComboBoxInput ) ) == nIndexInput ) {
			gstreamerSetSelectedInputDevice( psProfile, psInfo -> pnDescr );
			break;
		}

		psListInput = psListInput -> next;
		nIndexInput++;
	}

	gtk_widget_destroy( psDialog );
	SavePreferences( getActiveProfile() );

	// TODO: Reload driver
}

/** audio definition */
struct sAudio sGstreamer = {
	gstreamerAudioInit,
	gstreamerAudioOpen,
	gstreamerAudioWrite,
	gstreamerAudioRead,
	gstreamerAudioClose,
	gstreamerAudioDeinit,
	gstreamerPreferences
};

MODULE_INIT( PLUGIN_TYPE_AUDIO, _( "GStreamer" ), &sGstreamer, NULL, gstreamerAudioDetectDevices );
