#  Copyright (C) 2013-2016 Free Software Foundation, Inc.
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 3, or (at your option) any
# later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; see the file COPYING3.  If not see
# <http://www.gnu.org/licenses/>.

# This Awk script takes passes.def and writes pass-instances.def,
# counting the instances of each kind of pass, adding an instance number
# to everywhere that NEXT_PASS is used.
#
# For example, the single-instanced pass:
#     NEXT_PASS (pass_warn_unused_result);
# becomes this in the output:
#     NEXT_PASS (pass_warn_unused_result, 1);
#
# The various instances of
#   NEXT_PASS (pass_copy_prop);
# become:
#   NEXT_PASS (pass_copy_prop, 1);
# through:
#   NEXT_PASS (pass_copy_prop, 8);
# (currently there are 8 instances of that pass)

# Usage: awk -f gen-pass-instances.awk passes.def

BEGIN {
	print "/* This file is auto-generated by gen-pass-instances.awk";
	print "   from passes.def.  */";
}

function handle_line()
{
	line = $0;

	# Find call expression.
	call_starts_at = match(line, /NEXT_PASS \(.+\)/);
	if (call_starts_at == 0)
	{
		print line;
		return;
	}

	# Length of the call expression.
	len_of_call = RLENGTH;

	len_of_start = length("NEXT_PASS (");
	len_of_open = length("(");
	len_of_close = length(")");

	# Find arguments
	len_of_args = len_of_call - (len_of_start + len_of_close);
	args_start_at = call_starts_at + len_of_start;
	args_str = substr(line, args_start_at, len_of_args);
	split(args_str, args, ",");

	# Set pass_name argument, an optional with_arg argument
	pass_name = args[1];
	with_arg = args[2];

	# Find call expression prefix
	len_of_prefix = call_starts_at - 1;
	prefix = substr(line, 1, len_of_prefix);

	# Find call expression postfix
	postfix_starts_at = call_starts_at + len_of_call;
	postfix = substr(line, postfix_starts_at);

	# Set pass_counts
	if (pass_name in pass_counts)
		pass_counts[pass_name]++;
	else
		pass_counts[pass_name] = 1;

	pass_num = pass_counts[pass_name];

	# Print call expression with extra pass_num argument
	printf "%s", prefix;
	if (with_arg)
	{
		printf "NEXT_PASS_WITH_ARG";
	}
	else
	{
		printf "NEXT_PASS";
	}
	printf " (";
	printf "%s", pass_name;
	printf ", %s", pass_num;
	if (with_arg)
	{
		printf ", %s", with_arg;
	}
	printf ")%s\n", postfix;
}

{ handle_line() }

# Local Variables:
# mode:awk
# c-basic-offset:8
# End:
