/*
 * Copyright (C) 2014 Guitarix project MOD project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 * --------------------------------------------------------------------------
 */

#include <fftw3.h>
#include <assert.h>

////////////////////////////// LOCAL INCLUDES //////////////////////////

#include "gx_common.h"      // faust support and denormal protection (SSE)
#include "gx_detune.h"        // define struct PortIndex
#include "gx_pluginlv2.h"   // define struct PluginLV2
#include "gx_resampler.h"
#include "detune.cc"    // dsp class generated by faust -> dsp2cc

////////////////////////////// PLUG-IN CLASS ///////////////////////////

namespace detune {

class Gx_detune_
{
private:
  // pointer to buffer
  float*      output;
  float*      input;
  uint32_t    bufsize;
  float*      latency;
  float       latency_;
  bool        doit;
  volatile bool mode;
  // pointer to dsp class
  PluginLV2*  detune;
  LV2_URID_Map*                map;
  LV2_Worker_Schedule*         schedule;
  // private functions
  inline void do_work_mono();
  inline void run_dsp_(uint32_t n_samples);
  inline void connect_(uint32_t port,void* data);
  inline void init_dsp_(uint32_t rate, uint32_t bufsize_);
  inline void connect_all__ports(uint32_t port, void* data);
  inline void activate_f();
  inline void clean_up();
  inline void deactivate_f();

public:
  // LV2 Descriptor
  static const LV2_Descriptor descriptor;
  static const void* extension_data(const char* uri);
  // static wrapper to private functions
  static void deactivate(LV2_Handle instance);
  static void cleanup(LV2_Handle instance);
  static void run(LV2_Handle instance, uint32_t n_samples);
  static void activate(LV2_Handle instance);
  static void connect_port(LV2_Handle instance, uint32_t port, void* data);

  static LV2_Handle instantiate(const LV2_Descriptor* descriptor,
                                double rate, const char* bundle_path,
                                const LV2_Feature* const* features);

  static LV2_Worker_Status work(LV2_Handle                 instance,
                                LV2_Worker_Respond_Function respond,
                                LV2_Worker_Respond_Handle   handle,
                                uint32_t size, const void*    data);

  static LV2_Worker_Status work_response(LV2_Handle  instance,
                                         uint32_t    size,
                                         const void* data);

  Gx_detune_();
  ~Gx_detune_();
};

// constructor
Gx_detune_::Gx_detune_() :
  output(NULL),
  input(NULL),
  mode(false),
  detune(detune::plugin()) {};

// destructor
Gx_detune_::~Gx_detune_()
{
  // just to be sure the plug have given free the allocated mem
  // it didn't hurd if the mem is already given free by clean_up()
  if (detune->activate_plugin !=0)
    detune->activate_plugin(false, detune);
  // delete DSP class
  detune->delete_instance(detune);
};

///////////////////////// PRIVATE CLASS  FUNCTIONS /////////////////////

void Gx_detune_::do_work_mono()
{
    if (mode) detune::smbPitchShift::set_buffersize(detune, bufsize);
    detune::smbPitchShift::change_latency_static(detune);
}
void Gx_detune_::init_dsp_(uint32_t rate, uint32_t bufsize_)
{
  AVOIDDENORMALS(); // init the SSE denormal protection
  bufsize = bufsize_;
  detune::smbPitchShift::set_buffersize(detune, bufsize);
  detune->set_samplerate(rate, detune); // init the DSP class
}

// connect the Ports used by the plug-in class
void Gx_detune_::connect_(uint32_t port,void* data)
{
  switch ((PortIndex)port)
    {
    case EFFECTS_OUTPUT:
      output = static_cast<float*>(data);
      break;
    case EFFECTS_INPUT:
      input = static_cast<float*>(data);
      break;
	case LATENCY: 
		latency = static_cast<float*>(data); 
		break;
    default:
      break;
    }
}

void Gx_detune_::activate_f()
{
  // allocate the internal DSP mem
  if (detune->activate_plugin !=0)
    detune->activate_plugin(true, detune);
}

void Gx_detune_::clean_up()
{
  // delete the internal DSP mem
  if (detune->activate_plugin !=0)
    detune->activate_plugin(false, detune);
}

void Gx_detune_::deactivate_f()
{
  // delete the internal DSP mem
  if (detune->activate_plugin !=0)
    detune->activate_plugin(false, detune);
}

void Gx_detune_::run_dsp_(uint32_t n_samples)
{
  detune->mono_audio(static_cast<int>(n_samples), input, output, detune);
  if (*(latency) != latency_) {
      latency_ = *(latency) ;
      mode = false;
      schedule->schedule_work(schedule->handle, sizeof(bool), &doit);
  }
  if (bufsize != n_samples) {
      bufsize = n_samples;
      mode = true;
      schedule->schedule_work(schedule->handle, sizeof(bool), &doit);
  }
}

void Gx_detune_::connect_all__ports(uint32_t port, void* data)
{
  // connect the Ports used by the plug-in class
  connect_(port,data); 
  // connect the Ports used by the DSP class
  detune->connect_ports(port,  data, detune);
}

////////////////////// STATIC CLASS  FUNCTIONS  ////////////////////////

LV2_Worker_Status Gx_detune_::work(LV2_Handle                  instance,
     LV2_Worker_Respond_Function respond,
     LV2_Worker_Respond_Handle   handle,
     uint32_t                    size,
     const void*                 data)
{
  static_cast<Gx_detune_*>(instance)->do_work_mono();
  return LV2_WORKER_SUCCESS;
}

LV2_Worker_Status Gx_detune_::work_response(LV2_Handle  instance,
              uint32_t    size,
              const void* data)
{
  //printf("worker respose.\n");
  return LV2_WORKER_SUCCESS;
}

LV2_Handle 
Gx_detune_::instantiate(const LV2_Descriptor* descriptor,
                            double rate, const char* bundle_path,
                            const LV2_Feature* const* features)
{
  // init the plug-in class
  Gx_detune_ *self = new Gx_detune_();
  if (!self)
    {
      return NULL;
    }
const LV2_Options_Option* options  = NULL;
  uint32_t bufsize = 0;
  //printf(" %s\n",descriptor->URI);

  for (int32_t i = 0; features[i]; ++i)
    {
      if (!strcmp(features[i]->URI, LV2_URID__map))
        {
          self->map = (LV2_URID_Map*)features[i]->data;
        }
      else if (!strcmp(features[i]->URI, LV2_WORKER__schedule))
        {
          self->schedule = (LV2_Worker_Schedule*)features[i]->data;
        }
      else if (!strcmp(features[i]->URI, LV2_OPTIONS__options))
        {
          options = (const LV2_Options_Option*)features[i]->data;
        }
    }
  if (!self->schedule)
    {
      fprintf(stderr, "Missing feature work:schedule.\n");
      delete self;
      return NULL;
    }
  if (!self->map)
    {
      fprintf(stderr, "Missing feature uri:map.\n");
    }
  else if (!options)
    {
      fprintf(stderr, "Missing feature options.\n");
    }
  else
    {
      LV2_URID bufsz_max = self->map->map(self->map->handle, LV2_BUF_SIZE__maxBlockLength);
      LV2_URID atom_Int = self->map->map(self->map->handle, LV2_ATOM__Int);

      for (const LV2_Options_Option* o = options; o->key; ++o)
        {
          if (o->context == LV2_OPTIONS_INSTANCE &&
              o->key == bufsz_max &&
              o->type == atom_Int)
            {
              bufsize = *(const int32_t*)o->value;
            }
        }

      if (bufsize == 0)
        {
          fprintf(stderr, "No maximum buffer size given.\n");
        }
      printf("using block size: %d\n", bufsize);
    }

  self->init_dsp_((uint32_t)rate, bufsize);

  return (LV2_Handle)self;
}

void Gx_detune_::connect_port(LV2_Handle instance, 
                                   uint32_t port, void* data)
{
  // connect all ports
  static_cast<Gx_detune_*>(instance)->connect_all__ports(port, data);
}

void Gx_detune_::activate(LV2_Handle instance)
{
  // allocate needed mem
  static_cast<Gx_detune_*>(instance)->activate_f();
}

void Gx_detune_::run(LV2_Handle instance, uint32_t n_samples)
{
  // run dsp
  static_cast<Gx_detune_*>(instance)->run_dsp_(n_samples);
}

void Gx_detune_::deactivate(LV2_Handle instance)
{
  // free allocated mem
  static_cast<Gx_detune_*>(instance)->deactivate_f();
}

void Gx_detune_::cleanup(LV2_Handle instance)
{
  // well, clean up after us
  Gx_detune_* self = static_cast<Gx_detune_*>(instance);
  self->clean_up();
  delete self;
}

const void* Gx_detune_::extension_data(const char* uri)
{
  static const LV2_Worker_Interface worker = { work, work_response, NULL };
  if (!strcmp(uri, LV2_WORKER__interface))
    {
      return &worker;
    }
  return NULL;
}

const LV2_Descriptor Gx_detune_::descriptor =
{
  GXPLUGIN_URI "#_detune_",
  Gx_detune_::instantiate,
  Gx_detune_::connect_port,
  Gx_detune_::activate,
  Gx_detune_::run,
  Gx_detune_::deactivate,
  Gx_detune_::cleanup,
  Gx_detune_::extension_data
};


} // end namespace detune

////////////////////////// LV2 SYMBOL EXPORT ///////////////////////////

extern "C"
LV2_SYMBOL_EXPORT
const LV2_Descriptor*
lv2_descriptor(uint32_t index)
{
  switch (index)
    {
    case 0:
      return &detune::Gx_detune_::descriptor;
    default:
      return NULL;
    }
}

///////////////////////////// FIN //////////////////////////////////////
