#!/bin/bash

# USAGE
# This program is used to generate LemonLDAP::NG packages and some of
# its dependencies for Suse Linux

#Copyright (C) 2016  David Coutadeur / LINAGORA

# LICENCE
#This program is free software: you can redistribute it and/or modify
#it under the terms of the GNU General Public License as published by
#the Free Software Foundation, either version 3 of the License, or
#(at your option) any later version.

#This program is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GNU General Public License for more details.

#You should have received a copy of the GNU General Public License
#along with this program.  If not, see <http://www.gnu.org/licenses/>.






# debug mode (print all lines + expand variables)
#set -x

# Exit on error
set -e



################################################################################
# Variables
################################################################################

LEMON_DOWNLOAD_SITE="http://download.forge.ow2.org/lemonldap/"
CPAN2RPM="http://search.cpan.org/CPAN/authors/id/E/EC/ECALDER/cpan2rpm-2.028.tar.gz"

SRC_PACKAGE="lemonldap-ng-1.9.9-1.el7.src.rpm"

PKG_DEPENDENCIES="apache2 \
                  apache2-mod_perl \
                  apache2-mod_fcgid \
                  perl-ldap \
                  perl-XML-SAX \
                  perl-XML-NamespaceSupport \
                  perl-XML-Simple \
                  perl-XML-LibXML \
                  perl-Config-IniFiles \
                  perl-Digest-HMAC \
                  perl-Crypt-OpenSSL-Bignum \
                  perl-Crypt-OpenSSL-RSA \
                  perl-Authen-SASL \
                  perl-Unicode-String \
                  gd \
                  perl-Regexp-Assemble \
                  perl-Authen-Captcha \
                  perl-Cache-Cache \
                  perl-Apache-Session \
                  perl-CGI-Session \
                  perl-IO-String \
                  perl-MIME-Lite \
                  perl-SOAP-Lite \
                  perl-XML-LibXSLT \
                  perl-String-Random \
                  perl-Email-Date-Format \
                  perl-Crypt-Rijndael \
                  perl-HTML-Template \
                  perl-JSON \
                  perl-Crypt-OpenSSL-X509 \
                  perl-Crypt-DES \
                  perl-Class-Inspector \
                  perl-Test-MockObject \
                  perl-Clone \
                  perl-Net-CIDR-Lite \
                  perl-ExtUtils-MakeMaker \
                  perl-CGI \
                  perl-CGI-Session \
                  perl-HTML-Template \
                  perl-SOAP-Lite \
                  perl-IPC-ShareLite \
                  perl-Error \
                  perl-DBD-Pg \
                  perl-Plack \
                  perl-Regexp-Common \
                  perl-Crypt-OpenSSL-Bignum \
                  perl-FCGI-ProcManager"

BUILD_DEPENDENCIES="gd-devel \
                    libjpeg8-devel \
                    freetype-devel \
                    fontconfig-devel \
                    libXpm-devel \
                    libX11-devel \
                    libvpx-devel \
                    postgresql94-server \
                    postgresql94-devel \
                    libxslt-devel \
                    libopenssl-devel \
                    perl-Module-Build \
                    perl-Module-Build-XSUtil \
                    perl-Test-LeakTrace \
                    perl-Test-Requires \
                    perl-Test-Fatal \
                    perl-Test-Exception \
                    perl-Sub-Uplevel"
#this dependency is broken on native SLES12SP1 packages
#libtiff-devel


# Perl modules are not used by default
# Can be a quick way for installing all perl dependencies via CPAN
PERL_MODULES="Apache::Session \
              HTML::Template \
              Regexp::Assemble \
              Error \
              IPC::ShareLite \
              Cache::Cache \
              FreezeThaw \
              CGI \
              CGI::Session \
              DBD::Pg \
              BSD::Resource \
              Crypt::Rijndael \
              IO::String \
              XML::LibXSLT \
              SOAP::Lite \
              JSON \
              Digest::SHA \
              String::Random \
              MIME::Lite \
              Email::Date::Format \
              Clone \
              Log::Log4perl::Logger \
              Net::CIDR::Lite \
              Cache::Memcached \
              Convert::PEM \
              Mouse \
              MIME::Base64 \
              LWP::UserAgent \
              XML::Simple \
              Digest::HMAC \
              Crypt::OpenSSL::RSA \
              Crypt::OpenSSL::Bignum \
              Config::IniFiles \
              Net::CIDR \
              Authen::SASL \
              GD \
              Authen::Captcha \
              Crypt::OpenSSL::X509 \
              Module::Build"

PERL_NOTEST_MODULES="AuthCAS"

declare -A PERL_MODULES_TO_PACKAGE
PERL_MODULES_TO_PACKAGE=( [1]="Env" \
                          [2]="Digest-SHA" \
                          [3]="Crypt-DES_EDE3" \
                          [4]="Class-ErrorHandler" \
                          [5]="Convert-PEM" \
                          [6]="String-CRC32" \
                          [7]="Cache-Memcached" \
                          [8]="Mouse" )


################################################################################
# Functions
################################################################################


prerequisite()
{
  cd ~
  echo -e "\nInstalling RPMBUILD"
  zypper -n install rpmbuild createrepo

  echo -e "\nInstalling cpanminus"
  export PERL_MM_USE_DEFAULT=1
  cpan App::cpanminus

  echo -e "\nSearching for Devel packages"
  if zypper search libxslt-devel | grep -q libxslt-devel ; then
    echo "OK: Devel packages are present"
  else
    echo "KO: SUSE_Linux_Enterprise_Software_Development_Kit repository must be present"
    exit 1
  fi

  echo -e "\nAdding repository for many basic dependencies (please accept key)"
  if zypper lr | grep -q leap42 ; then
    echo "OK: leap42 repo already present"
  else
    zypper --gpg-auto-import-keys addrepo \
         http://download.opensuse.org/distribution/leap/42.1/repo/oss/suse/ leap42
    zypper --gpg-auto-import-keys refresh
  fi

  if zypper lr | grep -q "perl modules" ; then
    echo "OK: 'perl modules' repo already present"
  else
    zypper --gpg-auto-import-keys addrepo \
         http://download.opensuse.org/repositories/devel:languages:perl/SLE_12/devel:languages:perl.repo
    zypper --gpg-auto-import-keys refresh
  fi


  echo -e "\nSetting correct date with NTP\n"
  if systemctl list-unit-files | grep -q ntpd ; then
    service ntpd stop
    ntpdate 0.debian.pool.ntp.org
    service ntpd start
  fi

}


install_basic_dependencies()
{
  cd ~

  # Delete previous packages
  find /var/cache/zypp -name '*.rpm' | xargs -I {} rm -f {}

  # gather packages
  echo "Gather packages"
  mkdir -p ~/packages/dependencies/system-and-obs
  zypper mr -k -all
  ~/zypper-download ${PKG_DEPENDENCIES}
  find /var/cache/zypp -name '*.rpm' | xargs -I {} cp {} \
                                       ~/packages/dependencies/system-and-obs/
  # Install dependencies
  echo "Install dependencies"
  zypper -n install ${PKG_DEPENDENCIES}

}

install_build_dependencies()
{
  cd ~
  echo -e "\n\nInstalling build dependancies\n"
  zypper -n install ${BUILD_DEPENDENCIES}


  # Installing all modules with CPAN
  # -> pros: last version of package + dependencies are managed with cpan
  # -> cons: impossible to uninstall properly + hide dependency packages
  # *disabled* for all these reasons

  #echo -e "\n\nInstalling perl modules\n"
  #for MODULE in ${PERL_MODULES}; do
  #  cpanm ${MODULE}
  #done

  #echo -e "\nInstalling perl modules without tests\n"
  #for MODULE in ${PERL_NOTEST_MODULES}; do
  #  cpanm -n ${MODULE}
  #done

}


build_dependencies_packages()
{
  cd ~

  # get cpan2rpm for building RPM dependancies
  cpanm Pod::Simple
  cpanm Pod::Text
  cpanm Pod::PlainText
  wget -N ${CPAN2RPM}
  CPAN2RPM_ARCHIVE=${CPAN2RPM##*/}
  tar xvzf ${CPAN2RPM_ARCHIVE}
  cd ${CPAN2RPM_ARCHIVE%.tar.gz}
  # patch cpan2rpm for using Pod::PlainText instead
  sed -i 's/Pod::Text->new/Pod::PlainText->new/g' cpan2rpm
  sed -i '/^use Pod::Text;/a use Pod::PlainText;' cpan2rpm
  perl Makefile.PL
  make
  make install
  # patch cpan2rpm for getting correct path after package extraction
  perl -pe 's#\Q(\S+)/?#([^ \\t\\n\\/]+)/?#g' /usr/bin/cpan2rpm > /tmp/cpan2rpm
  cp /usr/bin/cpan2rpm /usr/bin/cpan2rpm.old
  mv /tmp/cpan2rpm /usr/bin/cpan2rpm
  chmod 555 /usr/bin/cpan2rpm
  # patch cpan2rpm for not testing perl dependency (failing)
  sed -i '/my $use = "use $_";/a \        next if $_ eq "perl";' /usr/bin/cpan2rpm
  # patch cpan2rpm for not being blocked by no perllocal.pod file found
  sed -i '/__cat/i \            if find %{buildroot} -name "perllocal.pod" | grep -q perllocal.pod ; then' /usr/bin/cpan2rpm
  sed -i '/adm\/perl-modules\/\%{name}/a \            fi' /usr/bin/cpan2rpm


  
  # Get last version of perl modules
  cd ~
  mkdir -p dependencies && cd dependencies
  echo -e "\n\nGet last version of perl modules\n"
  for (( K=1; K<=${#PERL_MODULES_TO_PACKAGE[@]}; K++ ))
  do
    # Get package name + perl module name
    PACKAGE_NAME=${PERL_MODULES_TO_PACKAGE[$K]}
    COMPLETE_PACKAGE_NAME="perl-${PACKAGE_NAME}"
    MODULE_NAME=$( echo ${PACKAGE_NAME} | sed 's/-/::/g' )

    if [ "${PACKAGE_NAME}" = "Class-ErrorHandler" ]; then
      # Class-ErrorHandler does not contain version, so give it directly
      VERSION="0.04"
    else
      # Get version of package through CPAN
      VERSION=$( get_cpan_module_version ${MODULE_NAME} )
    fi
    echo "${MODULE_NAME} $VERSION"
    cpan -g ${MODULE_NAME} >/dev/null 2>&1 || :
    PKG="$PACKAGE_NAME"
    PKG="${PKG}-${VERSION}.tar.gz"

    if [ "${PACKAGE_NAME}" = "Digest-SHA" ]; then
      # hook-Digest-SHA (the shasum binary is renamed in shasum2
      hook-digest-sha $PKG
    fi
    
    # Generate package
    cpan2rpm --author="Linagora" --version="${VERSION}" --no-sign --name="${PACKAGE_NAME}" ${PKG}

    # gather packages
    mkdir -p ~/packages/dependencies/additional
    PKG_PATH=$( find /usr/src/packages/RPMS -name "${COMPLETE_PACKAGE_NAME}*.rpm" )
    cp ${PKG_PATH} ~/packages/dependencies/additional/

    # gather src packages
    mkdir -p ~/packages/dependencies/additional-src
    SPKG_PATH=$( find /usr/src/packages/SRPMS -name "${COMPLETE_PACKAGE_NAME}*.src.rpm" )
    cp ${SPKG_PATH} ~/packages/dependencies/additional-src/

    # install package
    if zypper search ${COMPLETE_PACKAGE_NAME} | grep -E "\b${COMPLETE_PACKAGE_NAME}\b" | grep -E '^i' -q ; then
      echo "OK: package already installed"
    else
      echo "installing package ${COMPLETE_PACKAGE_NAME}"
      rpm -i ${PKG_PATH}
    fi

  done


}


get_src_package()
{
  cd ~
  wget -N ${LEMON_DOWNLOAD_SITE}${SRC_PACKAGE}
  rpm -i ${SRC_PACKAGE}
}


build_lemonldap_package()
{

  get_src_package

  cd /usr/src/packages

  # Adapt SPEC file
  echo -e "\nAdapt SPEC file\n\n"

  # Force Apache 2.4 version
  echo "Force Apache 2.4 version"
  sed -i 's/define apache_version .*/define apache_version 2.4/' \
                                                         SPECS/lemonldap-ng.spec

  # Force Apache vhosts dir
  echo "Force Apache vhosts dir"
  sed -i 's/define apache_confdir .*/define apache_confdir \/etc\/apache2\/vhosts.d/' \
                                                         SPECS/lemonldap-ng.spec

  # Fix some packages name
  echo "Fix perl-LDAP package name"
  sed -i 's/perl-LDAP/perl-ldap/g' SPECS/lemonldap-ng.spec

  echo "Fix perl-ExtUtims-MakeMaker"
  sed -i 's/perl-ExtUtims-MakeMaker/perl-ExtUtils-MakeMaker/g' SPECS/lemonldap-ng.spec

  # Add missing build dependencies
  echo "Add missing build dependencies"
  if ! grep -q -E '^BuildRequires: .*perl-Crypt-DES_EDE3' SPECS/lemonldap-ng.spec; then
    sed -i 's/\(^BuildRequires: .*perl-ldap.*$\)/\1 perl-Crypt-DES_EDE3 perl-Class-ErrorHandler perl-Convert-PEM perl-String-CRC32 perl-Plack/' SPECS/lemonldap-ng.spec
  fi

  # Add missing requirement dependencies
  echo "Add missing requirement dependencies"
  if ! grep -q -E '^Requires: .*perl-Mouse' SPECS/lemonldap-ng.spec; then
    sed -i 's/\(^Requires: .*perl-ldap.*$\)/\1 perl-Env perl-Digest-SHA perl-Crypt-DES_EDE3 perl-Class-ErrorHandler perl-Convert-PEM perl-String-CRC32 perl-Mouse perl-Net-CIDR-Lite perl-XML-LibXSLT perl-Clone perl-Crypt-OpenSSL-Bignum perl-Crypt-OpenSSL-RSA perl-Crypt-OpenSSL-X509 perl-Email-Date-Format perl-SOAP-Lite perl-String-Random perl-JSON perl-Plack/' SPECS/lemonldap-ng.spec
  fi

  # Automatically enable apache modules
  sed -i '/^\%post .*lemonldap-ng-conf/a a2enmod perl\na2enmod ssl\na2enmod rewrite\na2enmod proxy\na2enmod proxy_http\na2enmod headers\na2enmod mod_fcgid' \
                                                         SPECS/lemonldap-ng.spec

  rpmbuild -ba SPECS/lemonldap-ng.spec

  # gather packages
  mkdir -p ~/packages/lemonldap
  cp /usr/src/packages/RPMS/*/lemonldap*.rpm ~/packages/lemonldap/
  cp /usr/src/packages/RPMS/*/perl-Lemonldap*.rpm ~/packages/lemonldap/

  # gather src packages
  mkdir -p ~/packages/lemonldap-src
  cp /usr/src/packages/SRPMS/lemonldap-ng*.src.rpm ~/packages/lemonldap-src/
}


build_repository()
{
  cd ~
  LREPO="lemonldap-sles12-repository"

  if [ -d "$LREPO" ]; then
    # Repository already exists
    echo "The repository already exists"
    echo "Just update with new files "

    # create sub-directories anyway
    mkdir -p $LREPO/noarch
    mkdir -p $LREPO/x86_64
    mkdir -p $LREPO/src

    # move only new noarch dependency packages to repository
    for f in ~/packages/dependencies/additional/*.noarch.rpm; do
      if [ ! -e $LREPO/noarch/${f##*/} ]; then
        cp ~/packages/dependencies/additional/${f##*/} $LREPO/noarch
      fi
    done

    # move only new x86_64 dependency packages to repository
    for f in ~/packages/dependencies/additional/*.x86_64.rpm; do
      if [ ! -e $LREPO/noarch/${f##*/} ]; then
        cp ~/packages/dependencies/additional/${f##*/} $LREPO/x86_64
      fi
    done

    # move only new lemonldap packages to repository
    for f in ~/packages/lemonldap/*; do
      if [ ! -e $LREPO/noarch/${f##*/} ]; then
        cp ~/packages/lemonldap/${f##*/} $LREPO/noarch
      fi
    done

    # move only new src dependency packages to repository
    for f in ~/packages/dependencies/additional-src/*; do
      if [ ! -e $LREPO/src/${f##*/} ]; then
        cp ~/packages/dependencies/additional-src/${f##*/} $LREPO/src
      fi
    done

    # move only new src lemonldap packages to repository
    for f in ~/packages/lemonldap-src/*; do
      if [ ! -e $LREPO/src/${f##*/} ]; then
        cp ~/packages/lemonldap-src/${f##*/} $LREPO/src
      fi
    done

    # update repository with new packages
    createrepo --update ~/$LREPO

  else
    # Repository does not already exist
    echo "Repository does not already exist"
    echo "Create new one"

    # create sub-directories
    mkdir -p $LREPO/noarch
    mkdir -p $LREPO/x86_64
    mkdir -p $LREPO/src
    cp ~/packages/dependencies/additional/*.noarch.rpm $LREPO/noarch
    cp ~/packages/dependencies/additional/*.x86_64.rpm $LREPO/x86_64
    cp ~/packages/lemonldap/* $LREPO/noarch
    cp ~/packages/dependencies/additional-src/* $LREPO/src
    cp ~/packages/lemonldap-src/* $LREPO/src

    # create repository
    createrepo ~/$LREPO
  fi


  echo
  echo "OK: repository created in ~/$LREPO"
  echo "1. Expose it via a webserver"
  echo "2. Add new repository with a command like:"
  echo "   zypper addrepo http://host.tld/$LREPO $LREPO"
  echo
}


get_cpan_module_version()
{
  VERSION=$( cpan -D $1 2>/dev/null | grep CPAN: )
  VERSION=$( echo $VERSION | sed -e 's/CPAN:[ ]*//' )
  VERSION=$( echo $VERSION | sed -e 's/[ ]\+.*//g' )

  echo $VERSION
}


hook-digest-sha()
{
  tar xvzf $1
  dir=$( tar tvzf $1 | head -n 1 | sed -e 's/[ ]\+/ /g' | cut -d' ' -f6 )
  dir=$( echo $dir | sed -e 's/\/$//' )
  sed -i 's/shasum/shasum2/g' $dir/Makefile.PL
  sed -i 's/shasum/shasum2/g' $dir/MANIFEST
  mv $dir/shasum $dir/shasum2
  tar cvzf $1 $dir
  rm -rf ${dir}
}


print_help()
{
  echo "USAGE:"
  echo "$0 [pre] [iba] [ibu] [bde] [ble] [bre] [help]"
  echo "[pre] - install PRErequisites"
  echo "[iba] - Install BAsic dependencies"
  echo "[ibu] - Install BUild dependencies"
  echo "[bde] - Build DEpendencies packages"
  echo "[ble] - Build LEmonldap"
  echo "[bre] - Build REpository"
  echo "[help] - print this help"
  echo
}


################################################################################
# Entry point
################################################################################

PRE=0  # install PRErequisites
IBA=0  # Install BAsic dependencies
IBU=0  # Install BUild dependencies
BDE=0  # Build DEpendencies
BLE=0  # Build LEmonldap
BRE=0  # Build LEmonldap
HELP=0 # print HELP

# parse command line arguments
numargs=$#
for ((i=1 ; i <= $numargs ; i++))
do
  case "$1" in
    'pre')
       PRE=1
       shift
       continue
    ;;
    'iba')
       IBA=1
       shift
       continue
    ;;
    'ibu')
       IBU=1
       shift
       continue
    ;;
    'bde')
       BDE=1
       shift
       continue
    ;;
    'ble')
       BLE=1
       shift
       continue
    ;;
    'bre')
       BRE=1
       shift
       continue
    ;;
    *)
       HELP=1
       shift
       continue
    ;;
  esac

done


# do selected actions

if [ "$numargs" -eq "0" ]; then
  print_help
fi

if [ "$HELP" -eq "1" ]; then
  print_help
fi

if [ "$PRE" -eq "1" ]; then
  echo "[pre] - install PRErequisites"
  prerequisite
fi

if [ "$IBA" -eq "1" ]; then
  echo "[iba] - Install BAsic dependencies"
  install_basic_dependencies
fi

if [ "$IBU" -eq "1" ]; then
  echo "[ibu] - Install BUild dependencies"
  install_build_dependencies
fi

if [ "$BDE" -eq "1" ]; then
  echo "[bde] - Build DEpendencies packages"
  build_dependencies_packages
fi

if [ "$BLE" -eq "1" ]; then
  echo "[ble] - Build LEmonldap"
  build_lemonldap_package
fi

if [ "$BRE" -eq "1" ]; then
  echo "[bre] - Build REpository"
  build_repository
fi

