/***********************************************************************************

    Copyright (C) 2007-2018 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#include "lifeograph.hpp"
#include "app_window.hpp"
#include "printing.hpp"
#include "diary.hpp"
#include "chart_surface.hpp"


using namespace LIFEO;


void
PrintableText::render( const Glib::RefPtr< Gtk::PrintContext > &print_context )
{
    // Get a Cairo Context, which is used as a drawing board:
    Cairo::RefPtr< Cairo::Context > cr( print_context->get_cairo_context() );

    // We'll use black letters:
    cr->set_source_rgb( 0, 0, 0 );

    // Render Pango LayoutLines over the Cairo context:
#if( ( PANGOMM_MAJOR_VERSION > 2 ) || \
     ( ( PANGOMM_MAJOR_VERSION == 2 ) && ( PANGOMM_MINOR_VERSION >= 28 ) ) )
    Pango::LayoutIter iter( m_layout->get_iter() );
#else
    Pango::LayoutIter iter;
    m_layout->get_iter( iter );
#endif

    double start_pos( 0 );
    bool flag_1st( true );
    int line_index( 0 );
    do
    {
        if( line_index > m_line_end )
            break;
        else if( line_index >= m_line_begin )
        {
            Glib::RefPtr< Pango::LayoutLine > layout_line = iter.get_line();
            Pango::Rectangle logical_rect = iter.get_line_logical_extents();
            int baseline = iter.get_baseline();

            if( flag_1st )
            {
                start_pos = logical_rect.get_y() / Pango::SCALE - m_v_offset;
                flag_1st = false;
            }

            cr->move_to( logical_rect.get_x() / Pango::SCALE + PrintOpr::s_margin_h / 2,
                         baseline / Pango::SCALE - start_pos );

            layout_line->show_in_cairo_context( cr );
        }

        line_index++;
    }
    while( iter.next_line() );
}

void
PrintableImage::render( const Glib::RefPtr< Gtk::PrintContext > &print_context )
{
    Cairo::RefPtr<Cairo::Context> cr = print_context->get_cairo_context();

    int width( m_pixbuf->get_width() );
    int height( m_pixbuf->get_height() );

    Cairo::RefPtr< Cairo::ImageSurface > m_image_surface = Cairo::ImageSurface::create(
            m_pixbuf->get_has_alpha() ? Cairo::FORMAT_ARGB32 : Cairo::FORMAT_RGB24,
            width, height );
    Gdk::Cairo::set_source_pixbuf(
            cr, m_pixbuf, ( print_context->get_width() - width ) / 2.0, m_v_offset );
    cr->paint();
}

// PRINT OPERATION =================================================================================
double PrintOpr::s_margin_h{ 0 };
double PrintOpr::s_margin_v{ 0 };

PrintOpr::PrintOpr()
{
    m_refSettings = Gtk::PrintSettings::create();
    m_refPageSetup = Gtk::PageSetup::create();
    set_print_settings( m_refSettings );
    set_default_page_setup( m_refPageSetup );
    set_embed_page_setup( true );
    set_track_print_status();
}

PrintOpr::~PrintOpr()
{
}

void
PrintOpr::clear_content()
{
    for( Printables::iterator iter = m_content.begin(); iter != m_content.end(); ++iter )
        delete *iter;

    m_content.clear();
}

void
PrintOpr::init_variables( const Glib::RefPtr< Gtk::PrintContext >& print_context  )
{
    s_margin_h = print_context->get_dpi_x() * m_opt_margin; // 1 inch when full
    s_margin_v = print_context->get_dpi_y() * m_opt_margin; // 1 inch when full
    m_v_offset = s_margin_v / 2.0;

    m_n_pages = 1;
    m_last_chapter_date = 0;
    m_flag_chapter_round = DiaryElement::ET_CHAPTER;    // start with chapters

    m_chapters = Diary::d->get_current_chapter_ctg();
    m_iter_chapter = m_chapters->rbegin();

    m_parser.m_max_thumbnail_width = ( print_context->get_width() / 2 );

    // SETTING ENTRY ITERATORS
    m_iter_entry = Diary::d->get_entries().rbegin();
    if( m_opt_one_entry )
    {
        if( AppWindow::p->panel_main->get_cur_elem_type() != DiaryElement::ET_ENTRY )
            throw LIFEO::Error( "Current element is not an entry. Printing aborted." );

        m_iter_entry_end = EntryIterReverse( Diary::d->get_entries().find(
                AppWindow::p->panel_main->get_cur_elem()->get_date_t() ) );
        m_iter_entry = m_iter_entry_end;
        m_iter_entry--;
    }
    else
    {
        m_iter_entry_end = Diary::d->get_entries().rend();
    }
}

void
PrintOpr::set_hide_comments( bool flag_hide )
{
    m_parser.m_pango_opt_hide_comments = flag_hide;
}

Gtk::Widget*
PrintOpr::on_create_custom_widget()
{
    if( m_builder )
        m_builder.reset();

    m_builder = Gtk::Builder::create();
    Lifeograph::load_gui( m_builder, Lifeograph::UIDIR + "/print.ui" );

    m_builder->get_widget( "Bx_print", m_Bx_print );

    m_builder->get_widget( "RB_print_current", m_RB_current );
    m_builder->get_widget( "RB_print_all", m_RB_all );

    m_builder->get_widget( "FB_print", m_FB );

    m_builder->get_widget( "CB_print_theme_font", m_CB_theme_font );
    m_builder->get_widget( "CB_print_justify", m_CB_justify );

    m_builder->get_widget( "RB_print_margin_off", m_RB_margin_off );
    m_builder->get_widget( "RB_print_margin_half", m_RB_margin_half );

    return m_Bx_print;
}

void
PrintOpr::on_custom_widget_apply( Gtk::Widget* widget )
{
    m_opt_one_entry = m_RB_current->get_active();
    m_opt_entire_diary = m_RB_all->get_active();

    m_font = Pango::FontDescription( m_FB->get_font_name() );
    m_parser.m_pango_opt_use_theme_font = m_CB_theme_font->get_active();
    m_opt_justify = m_CB_justify->get_active();

    m_opt_margin = m_RB_margin_off->get_active() ? 0.0f :
                                ( m_RB_margin_half->get_active() ? 0.5f : 1.0f );
}

Glib::RefPtr< PrintOpr >
PrintOpr::create()
{
    return Glib::RefPtr< PrintOpr >( new PrintOpr() );
}

void
PrintOpr::on_done( Gtk::PrintOperationResult result )
{
    //Printing is "done" when the print data is spooled.

    if( result == Gtk::PRINT_OPERATION_RESULT_ERROR )
    {
        Gtk::MessageDialog err_dialog( *AppWindow::p, "Printing failed", false,
                Gtk::MESSAGE_ERROR, Gtk::BUTTONS_OK, true );
        err_dialog.run();
    }
    else
    if( result == Gtk::PRINT_OPERATION_RESULT_APPLY )
    {
        // Update PrintSettings with the ones used in this PrintOperation:
        m_refSettings = get_print_settings();
    }

   /* if( ! is_finished())
    {
    // We will connect to the status-changed signal to track status
    // and update a status bar. In addition, you can, for example,
    // keep a list of active print operations, or provide a progress dialog.
        signal_status_changed().connect( sigc::bind( sigc::mem_fun( this,
                    &PrintOpr::handle_print_status_changed ),
                operation ) );
    }*/
}

void
PrintOpr::on_status_changed()
{
    if( is_finished() )
    {
        print_info( "Print job completed" );
    }
    else
    {
        print_info( get_status_string() );
    }
}

void
PrintOpr::show_page_setup()
{
    // Show the page setup dialog, asking it to start with the existing settings:
    Glib::RefPtr< Gtk::PageSetup > new_page_setup =
            Gtk::run_page_setup_dialog( *AppWindow::p, m_refPageSetup, m_refSettings );

    //Save the chosen page setup dialog for use when printing, previewing, or
    //showing the page setup dialog again:
    m_refPageSetup = new_page_setup;
}

inline void
PrintOpr::add_page()
{
    m_content.push_back( new PrintablePageBreak );
    m_n_pages++;
    m_v_offset = ( s_margin_v / 2.0 );
}

void
PrintOpr::handle_text( const Glib::RefPtr< Gtk::PrintContext >& print_context )
{
    m_layout = print_context->create_pango_layout();

    const double width = print_context->get_width() - s_margin_h;
    const double height = print_context->get_height() - s_margin_v;
    double total_line_h = 0.0;

    m_layout->set_width( static_cast< int >( width * Pango::SCALE ) );
    m_layout->set_font_description( m_font );
    m_layout->set_markup( m_marked_up_text );
    // cover page is center aligned
    m_layout->set_alignment( m_flag_align_center ? Pango::ALIGN_CENTER : Pango::ALIGN_LEFT );
    m_layout->set_justify( m_opt_justify );

    PRINT_DEBUG( m_marked_up_text );

    const int last_line( m_layout->get_line_count() - 1 );
    int line_begin( 0 );

    for( int line = 0; line <= last_line; line++ )
    {
        Pango::Rectangle ink_rect, logical_rect;
        m_layout->get_line( line )->get_extents( ink_rect, logical_rect );
        const double line_height = logical_rect.get_height() / Pango::SCALE;

        if( ( m_v_offset + total_line_h + line_height ) > height )
        {
            line--;
            if( line < 0 )  // if first line is higher than the remaining space on the page
            {
                // at least one line should fit into the page
                if( m_v_offset == ( s_margin_v / 2.0 ) )
                {
                    print_error( "Cannot print, line is too big for the page" );
                    line++;
                }
                else
                    add_page();
                continue;
            }

            m_content.push_back( new PrintableText( m_layout, m_v_offset, line_begin, line ) );

            add_page();
            line_begin = line + 1;
            total_line_h = 0.0;
        }
        else if( line == last_line )
        {
            m_content.push_back(
                    new PrintableText( m_layout, m_v_offset, line_begin, line ) );
            m_v_offset += total_line_h;
        }
        else
            total_line_h += line_height;
    }
}

void
PrintOpr::handle_entry_break( const Glib::RefPtr< Gtk::PrintContext >& print_context )
{
    const double break_height = print_context->get_height() / 10;
    const double height = print_context->get_height();

    // TODO: also consider entry header height

    if( m_v_offset + break_height > height )
        add_page();
    else
    {
        m_content.push_back( new PrintableEntryBreak );
        m_v_offset += break_height;
    }
}

void
PrintOpr::handle_image( const Glib::RefPtr< Gtk::PrintContext >& print_context )
{
    const int image_height = m_parser.m_pango_pixbuf->get_height();
    const double height = print_context->get_height();

    if( m_v_offset + image_height > height )
        add_page();

    m_content.push_back( new PrintableImage( m_parser.m_pango_pixbuf, m_v_offset ) );

    m_v_offset += image_height;
}

Printable::Type
PrintOpr::get_more()
{
    Chapter* chapter( nullptr );
    Entry* entry( nullptr );
    Printable::Type return_type( Printable::PT_PAGE_BREAK );

    m_marked_up_text.clear();

    // diary title:
    if( m_content.empty() && m_opt_one_entry == false )
    {
        Date today( Date::get_today() );

        m_marked_up_text = Glib::ustring::compose(
                "\n\n\n<span size='xx-large'>%1</span>\n%2",
                Glib::Markup::escape_text( Diary::d->get_name() ),
                today.format_string() );
        m_flag_align_center = true;
        return Printable::PT_PAGE_BREAK;
    }
    else
        m_flag_align_center = false;

    // FIXME: empty chapters after the last entry are not printed
    while( m_iter_entry != m_iter_entry_end )
    {
        entry = m_iter_entry->second;
        m_parser.set( entry );

        // CHAPTER
        while( !m_opt_one_entry )
        {
            // CHAPTER CTG--TOPIC SWITCH
            if( m_iter_chapter == m_chapters->rend() )
            {
                if( m_flag_chapter_round == DiaryElement::ET_CHAPTER )
                {
                    m_chapters = Diary::d->get_groups();
                    m_iter_chapter = m_chapters->rbegin();
                    m_flag_chapter_round = DiaryElement::ET_GROUP;
                    continue;   // to check if rbegin() == rend()
                }
                else if( m_flag_chapter_round == DiaryElement::ET_GROUP )
                {
                    m_chapters = Diary::d->get_topics();
                    m_iter_chapter = m_chapters->rbegin();
                    m_flag_chapter_round = DiaryElement::ET_TOPIC;
                    continue;   // to check if rbegin() == rend()
                }
                else
                    break;
            }

            chapter = m_iter_chapter->second;

            if( entry->get_date() >= chapter->get_date() &&
                chapter->get_date_t() > m_last_chapter_date )
            {
                if( m_content.back()->get_type() != Printable::PT_PAGE_BREAK )
                    return Printable::PT_PAGE_BREAK; // page break before chapter

                if( ! chapter->get_date().is_hidden() )
                {
                    m_marked_up_text += Glib::ustring::compose(
                            "<big>%1</big>\n", chapter->get_date().format_string() );
                }

                m_marked_up_text += Glib::ustring::compose(
                        "<big><big><b>%1</b></big></big>\n\n\n",
                        Glib::Markup::escape_text( chapter->get_name() ) );

                m_last_chapter_date = chapter->get_date_t();
            }
            else
                break;

            ++m_iter_chapter;
        }

        // ENTRY
        if( m_opt_entire_diary || entry->get_filtered_out() == false || m_opt_one_entry )
        {
            return_type = m_parser.process();
            m_marked_up_text += m_parser.m_pango_text;
            if( return_type == Printable::PT_ENTRY_BREAK )
                ++m_iter_entry;

            return return_type;
        }

        ++m_iter_entry;
    }

    return Printable::PT_END;
}

void
PrintOpr::on_begin_print( const Glib::RefPtr< Gtk::PrintContext >& print_context )
{
    clear_content();
    init_variables( print_context );

    // PROCESSING THE MATERIAL TO PRINT
    bool more( true );
    do
    {
        switch( get_more() )
        {
//            case Printable::PT_TEXT:
//                handle_text( print_context );
//                break;
            case Printable::PT_ENTRY_BREAK:
                if( !m_marked_up_text.empty() )   // if there is any unprocessed text
                    handle_text( print_context );
                handle_entry_break( print_context );
                break;
            case Printable::PT_IMAGE:
                if( !m_marked_up_text.empty() )   // if there is any unprocessed text
                    handle_text( print_context );
                handle_image( print_context );
                break;
            case Printable::PT_PAGE_BREAK:
                if( !m_marked_up_text.empty() )   // if there is any unprocessed text
                    handle_text( print_context );
                add_page();
                break;
            case Printable::PT_END:
                // no unprocessed text should be left when this point is reached
                more = false;
                break;
            default:
                break;
        }
    }
    while( more );

    set_n_pages( m_n_pages );
}

void
PrintOpr::on_draw_page( const Glib::RefPtr< Gtk::PrintContext >& print_context, int page_no )
{
    int i( 0 );

    for( Printable* printable : m_content )
    {
        if( i == page_no )
            printable->render( print_context );

        if( printable->get_type() == Printable::PT_PAGE_BREAK )
            i++;

        if( i > page_no )
            break;
    }
}

// PANGO ENTRY PARSER ==============================================================================
void
EntryParserPango::set( const Entry* entry )
{
    m_pango_text.clear();
    if( m_ptr2entry != entry )
    {
        auto size{ entry->get_text()->size() };
        m_pango_begin = 0;  // do not reset when continuing
        m_ptr2entry = entry;
        m_format_map.clear();
        m_format_map.assign( size + 1, 0 ); // last zero (i.e. +1) is for the file end

        EntryParser::parse( 0, size );
    }
}

Printable::Type
EntryParserPango::process()
{
    Recipe::Id format_prev{ 0 };
    Recipe::Id format_curr{ 0 };

    m_pango_return = Printable::PT_ENTRY_BREAK;

    if( m_pango_begin == 0 )    // only at the beginning of the entry
    {
        // ENTRY DATE
        if( ! m_ptr2entry->get_date().is_hidden() )
            m_pango_text = Glib::ustring::compose(
                    "<b>%1</b>\n", m_ptr2entry->get_date().format_string() );

        // THEME
        if( m_pango_opt_use_theme_font && m_ptr2entry->get_theme_is_set() )
        {
            // TODO: health of the font should also be checked as it may crash the program
            m_pango_text += "<span font='";
            m_pango_text += m_ptr2entry->get_theme()->font.to_string();
            m_pango_text += "'>";
        }

        // ENTRY TO-DO STATUS
        if( m_ptr2entry->get_status() & ES::FILTER_TODO_PURE )
        {
            m_pango_text += "<big><big>";
            switch( m_ptr2entry->get_status() & ES::FILTER_TODO )
            {
                case ES::TODO:
                    m_pango_text += "[ ] ";
                    break;
                case ES::PROGRESSED:
                    m_pango_text += "[~] ";
                    break;
                case ES::DONE:
                    m_pango_text += "[+] ";
                    break;
                case ES::CANCELED:
                    m_pango_text += "[X] ";
                    break;
            }
            m_pango_text += "</big></big>";
        }
    }

    for( UstringSize i = m_pango_begin; i <= m_pos_end; i++ ) // <= is for handling the file end
    {
        format_curr = m_format_map[ i ];

        //Wchar ch = i < m_pos_end ? get_char_at( i ) : '\n' ;
        //PRINT_DEBUG( ">>>>>>>> ", ch, " - ", format_curr );

        if( format_curr != format_prev )
        {
            if( not( format_curr & RID_MARKUP ) )
            {
                // close last
                if( format_prev & RID_MONOSPACE )
                    m_pango_text += "</span>";
                if( format_prev & RID_LINK )
                    m_pango_text += "</span>";
                if( format_prev & RID_STRIKETHROUGH )
                    m_pango_text += "</s>";
                if( format_prev & RID_DONE )
                    m_pango_text += "</span>";
                if( format_prev & RID_HIGHLIGHT )
                    m_pango_text += "</span>";
                if( format_prev & RID_ITALIC )
                    m_pango_text += "</i>";
                if( format_prev & RID_BOLD )
                    m_pango_text += "</b>";
                if( format_prev & RID_COMMENT )
                    m_pango_text += "</sup></small>";
                if( format_prev & RID_SUBSUBHEADING )
                    m_pango_text += "</b></big>";
                if( format_prev & RID_SUBHEADING )
                    m_pango_text += "</b></big></big>";
                if( format_prev & RID_HEADING )
                    m_pango_text += "</b></big></big></big>";

                // open new (in reverse order)
                if( format_curr & RID_HEADING )
                    m_pango_text += "<big><big><big><b>";
                if( format_curr & RID_SUBHEADING )
                    m_pango_text += "<big><big><b>";
                if( format_curr & RID_SUBSUBHEADING )
                    m_pango_text += "<big><b>";
                if( format_curr & RID_COMMENT )
                    m_pango_text += "<small><sup>";
                if( format_curr & RID_BOLD )
                    m_pango_text += "<b>";
                if( format_curr & RID_ITALIC )
                    m_pango_text += "<i>";
                if( format_curr & RID_HIGHLIGHT )
                    m_pango_text += STR::compose( "<span bgcolor='",
                            convert_gdkcolor_to_html( ThemeSystem::get()->color_highlight ),
                            "'>" );
                if( format_curr & RID_DONE )
                    m_pango_text += STR::compose( "<span bgcolor='",
                            convert_gdkcolor_to_html( midtone( Theme::s_color_done,
                                     ThemeSystem::get()->color_base,
                                     0.35 ) ),
                            "'>" );
                if( format_curr & RID_STRIKETHROUGH )
                    m_pango_text += "<s>";
                if( format_curr & RID_LINK )
                    m_pango_text += "<span color='blue' underline='single'>";
                if( format_curr & RID_MONOSPACE )
                    m_pango_text += "<span font='Monospace'>";
            }

            // IMAGES
            if( format_curr & RID_IMAGE )
            {
                auto c{ get_format_string_size( i, RID_IMAGE ) };
                std::string link{ m_ptr2entry->get_text()->substr( i, c ) };
                PRINT_DEBUG( "process_RID_IMAGE: ", link );

                // HANDLE RELATIVE LINKS
                if( link.find( "rel://" ) == 0 )
                    link.replace( 0, 5, "file://" + Glib::path_get_dirname( Diary::d->get_path() ) );

                // we have already used try{} in apply_link() we don't "try" again:
                m_pango_pixbuf = Lifeograph::get_thumbnail( Glib::filename_from_uri( link ),
                                                            m_max_thumbnail_width );
                m_pango_return = Printable::PT_IMAGE;
                m_pango_begin = i+c;
                return m_pango_return;
            }
            // CHARTS
            else if( format_curr & RID_CHART )
            {
                auto c{ get_format_string_size( i, RID_CHART ) };
                Tag* tag{ Diary::d->get_tags()->get_tag(
                        m_ptr2entry->get_text()->substr( i + 6, c - 6 )  ) };
                        // "chart:" is 6 chars long
                if( tag )
                {
                    try
                    {
                        auto surface = new ChartSurface( tag, m_max_thumbnail_width*1.5 );

                        m_pango_pixbuf = surface->get_pixbuf();
                        m_pango_return = Printable::PT_IMAGE;
                        m_pango_begin = i+c;

                        delete surface;

                        return m_pango_return;
                    }
                    catch( ... )
                    {
                        PRINT_DEBUG( "Could not add chart" );
                    }
                }
            }
        }

        if( not( format_curr & RID_MARKUP ) && i < m_pos_end )
        {
            m_pango_text += Glib::Markup::escape_text( m_ptr2entry->get_text()->substr( i, 1 ) );

            format_prev = format_curr;
        }
    } // close loop

    // CLOSE THEME
    if( m_pango_opt_use_theme_font && m_ptr2entry->get_theme_is_set() )
        m_pango_text += "</span>";

    return m_pango_return;
}

void
EntryParserPango::apply_markup( Recipe::Id format )
{
    m_format_map[ m_recipe_curr->m_pos_start ] |= RID_MARKUP;
    m_format_map[ m_pos_curr ] |= RID_MARKUP;

    apply_format( format, m_recipe_curr->m_pos_start, m_pos_curr );
}

void
EntryParserPango::apply_heading( bool flag_non_empty )
{
    apply_format( ( m_ptr2entry->get_status() & ES::CANCELED ) ?
                        RID_HEADING|RID_STRIKETHROUGH : RID_HEADING, 0, get_para_end( 0 ) );
}

void
EntryParserPango::apply_subheading()
{
    apply_format( RID_SUBHEADING,
                  m_recipe_curr->m_pos_start, get_para_end( m_recipe_curr->m_pos_start ) );
}

void
EntryParserPango::apply_subsubheading()
{
    apply_format( RID_SUBSUBHEADING,
                  m_recipe_curr->m_pos_start, get_para_end( m_recipe_curr->m_pos_start ) );
}

void
EntryParserPango::apply_bold()
{
    apply_markup( RID_BOLD );
}

void
EntryParserPango::apply_italic()
{
    apply_markup( RID_ITALIC );
}

void
EntryParserPango::apply_strikethrough()
{
    apply_markup( RID_STRIKETHROUGH );
}

void
EntryParserPango::apply_highlight()
{
    apply_markup( RID_HIGHLIGHT );
}

void
EntryParserPango::apply_comment()
{
    apply_markup( m_pango_opt_hide_comments ? RID_MARKUP : RID_COMMENT );
}

void
EntryParserPango::apply_check_ccl()
{
    apply_format( RID_MONOSPACE, m_recipe_curr->m_pos_start, m_pos_curr );
    apply_format( RID_STRIKETHROUGH, m_pos_curr, get_para_end( m_pos_curr ) );
}

void
EntryParserPango::apply_check_unf()
{
    apply_format( RID_MONOSPACE, m_recipe_curr->m_pos_start, m_pos_curr );
}

void
EntryParserPango::apply_check_prg()
{
    apply_check_unf();
}

void
EntryParserPango::apply_check_fin()
{
    apply_format( RID_MONOSPACE|RID_DONE, m_recipe_curr->m_pos_start, m_pos_curr );
    apply_format( RID_DONE, m_pos_curr, get_para_end( m_pos_curr ) ); // TODO change color
}

void
EntryParserPango::apply_link()
{
    if( m_link_type_last & RID_DATE )
        return;

    auto begin{ m_recipe_curr->m_pos_start };
    Glib::RefPtr< Gdk::Pixbuf > pixbuf;
    bool flag_ends_para{ m_pos_curr == m_ptr2entry->get_text()->length() };
    if( !flag_ends_para )
        flag_ends_para = ( get_char_at( m_pos_curr ) == '\n' );

    // there can be no text on the same line as the image:
    if( begin > 0 && get_char_at( begin - 1 ) == '\n' && flag_ends_para )
    {
        std::string link{ m_ptr2entry->get_text()->substr( begin, m_pos_curr - begin ) };

        // HANDLE RELATIVE LINKS
        if( link.find( "rel://" ) == 0 )
            link.replace( 0, 5, "file://" + Glib::path_get_dirname( Diary::d->get_path() ) );

        if( link.find( "file:///" ) == 0 )
        {
            try
            {
                pixbuf = Lifeograph::get_thumbnail( Glib::filename_from_uri( link ),
                                                    m_max_thumbnail_width );
                PRINT_DEBUG( "pixbuf loaded" );
            }
            catch( ... )
            {
                PRINT_DEBUG( "Could not load image" );
            }
        }
    }

    if( pixbuf )
        apply_format( RID_MARKUP|RID_IMAGE, begin, m_pos_curr );
    else
        apply_format( RID_LINK, begin, m_pos_curr );
}

void
EntryParserPango::apply_link_hidden( )
{
    auto pos_label( m_recipe_curr->m_pos_middle + 1 );

    apply_format( RID_MARKUP, m_recipe_curr->m_pos_start, pos_label );
    apply_format( RID_MARKUP, m_pos_curr, m_pos_curr + 1 );
    apply_format( RID_LINK, pos_label, m_pos_curr );
}

void
EntryParserPango::apply_chart()
{
    apply_format( RID_CHART, m_recipe_curr->m_pos_start, m_pos_curr );
}

inline Wchar
EntryParserPango::get_char_at( int i )
{
    return m_ptr2entry->get_text()->at( i );
}

inline UstringSize
EntryParserPango::get_para_end( UstringSize begin )
{
    auto end{ m_ptr2entry->get_text()->find_first_of( '\n', begin ) };
    return( end == Ustring::npos ? end = m_ptr2entry->get_text()->length() : end );
}

inline void
EntryParserPango::apply_format( Recipe::Id format, UstringSize begin, UstringSize end )
{
    for( auto i = begin; i< end; i++ )
        m_format_map[ i ] |= format;
}

inline UstringSize
EntryParserPango::get_format_string_size( UstringSize begin, Recipe::Id format )
{
    UstringSize size{ 0 };
    for( ; ( begin + size ) < m_format_map.size(); size++ )
        if( not( m_format_map[ begin + size ] & format ) )
            break;

    return size;
}
