/*	ByteBuffer_InputStream

PIRL CVS ID: ByteBuffer_InputStream.java,v 1.4 2012/04/16 06:18:23 castalia Exp

Copyright (C) 2008-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package	PIRL.Utilities;

import	java.nio.Buffer;
import	java.nio.ByteBuffer;
import	java.nio.InvalidMarkException;
import	java.io.InputStream;
import	java.io.IOException;


/**	A <i>ByteBuffer_InputStream</i> presents a ByteBuffer as an InputStream.
<p>
	This class is thread safe: Methods that might modify the backing
	ByteBuffer are synchronized on the ByteBuffer.
<p>
	@author		Bradford Castalia, UA/PIRL
	@version	1.4
	@see		ByteBuffer
	@see		InputStream
*/
public class ByteBuffer_InputStream
	extends InputStream
{
/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Conductor.ByteBuffer_InputStream (1.4 2012/04/16 06:18:23)";

private ByteBuffer
	Source;


private static String
	NL;
static
{
if ((NL = System.getProperty ("line.separator")) == null)
	 NL = "\n";
}

/*==============================================================================
	Constructors
*/
/**	Construct a ByteBuffer_InputStream on a ByteBuffer.
<p>
	@param	source	The ByteBuffer to be used as the source of input data.
*/
public ByteBuffer_InputStream
	(
	ByteBuffer	source
	)
{
if ((Source = source) == null)
	throw new NullPointerException (ID + NL
		+ "A buffer must be provided.");
}

/**	A ByteBuffer_InputStream must have a ByteBuffer.
*/
private ByteBuffer_InputStream ()
{}

/*==============================================================================
	Accessors
*/
/**	Get the ByteBuffer backing this ByteBuffer_InputStream.
<p>
	@return	The ByteBuffer backing this ByteBuffer_InputStream.
*/
public ByteBuffer Source ()
{return Source;}

/*==============================================================================
	InputStream
*/
/**	Get the number of bytes remaining in the source.
<p>
	@return The number of bytes remaining in the {@link #Source() source}.
		 This is the difference between the buffer's current input
		{@link Buffer#position() position} and the input {@link
		Buffer#limit() limit}. Note that the limit may be less than the
		buffer {@link Buffer#capacity() capacity}.
*/
public int available ()
{return Source.remaining ();}

/**	Sets a mark at the current input position.
<p>
	The operation is synchronized on the source buffer.
<p>
	@param	read_limit	Ignored: The read limit is always the
		{@link #Source() source} limit.
*/
public void mark
	(
	int		read_limit
	)
{synchronized (Source) {Source.mark ();}}

/**	Test if stream {@link #mark(int) mark} and {@link #reset() reset} are
	supported.
<p>
	@return	Always returns true.
*/
public boolean markSupported ()
{return true;}

/**	Get the next datum from the source.
<p>
	The operation is synchronized on the source buffer.
<p>
	@return	The value of the next {@link #Source() source} byte (0-255) at
		the current buffer position, or -1 if the {@link #Source() source}
		position is at its {@link Buffer#limit() limit}.
*/
public int read ()
{
synchronized (Source)
	{
	if (Source.hasRemaining ())
		return ((int)Source.get () & 0xFF);
	}
return -1;
}

/**	Get some number of bytes from the source and store them into a
	byte array.
<p>
	This method simply returns
	<code>read (byte_array, 0, byte_array.length)</code>.
<p>
	@param	byte_array	A byte[] array where the source data is to be
		stored.
	@return	The number of bytes read into the array, or -1 if the
		{@link #Source() source} position is at its {@link Buffer#limit()
		limit}.
	@throws	NullPointerException	If the byte array is null.
	@see	#read(byte[], int, int)
*/
public int read
	(
	byte[]	byte_array
	)
{return read (byte_array, 0, byte_array.length);}

/**	Get some number of bytes from the source and store them into a
	byte array.
<p>
	If the length to read is less than or equal to zero, zero is
	returned. Otherwise the effective length is the lesser of the
	length and the amount of data {@link ByteBuffer#remaining()
	remaining} in the {@link #Source() source} buffer.
<p>
	The operation is synchronized on the source buffer.
<p>
	@param	byte_array	A byte[] array where the source data is to be
		stored.
	@param	offset	The index of the array where the first byte will be
		stored.
	@param	length	The maximum number of bytes to transfer from the
		{@link #Source() source} to the array.
	@return	The number of bytes read into the array, or -1 if the
		{@link #Source() source} position is at its {@link Buffer#limit()
		limit}.
	@throws	NullPointerException	If the byte array is null.
	@throws	IndexOutOfBoundsException	If the offset is negative or
		the offset plus the effective length is greater than the
		remaining space in the array.
*/
public int read
	(
	byte[]	byte_array,
	int		offset,
	int		length
	)
{
synchronized (Source)
	{
	if (! Source.hasRemaining ())
		return -1;
	if (byte_array == null)
		throw new NullPointerException (ID + NL
			+ "Can't read into a null array.");
	if (length <= 0)
		return 0;
	if (length > Source.remaining ())
		length = Source.remaining ();
	if (offset < 0 ||
		(offset + length) > byte_array.length)
		throw new IndexOutOfBoundsException (ID + NL
			+ "Can't read into the array of length " + byte_array.length
				+ " starting at offset " + offset + NL
			+ ((length == Source.remaining ()) ?
				"the remaining " : "")
				+ length + " buffer byte" + ((length == 1) ? "" : "s") + '.');
	Source.get (byte_array, offset, length);
	}
return length;
}

/**	Reset the {@link #Source() source} position to the most recent
	{@link #mark(int) mark} position.
<p>
	The operation is synchronized on the source buffer.
<p>
	@throws	IOException	If no mark has been set.
*/
public void reset ()
	throws IOException
{
try {synchronized (Source) {Source.reset ();}}
catch (InvalidMarkException exception)
	{
	throw new IOException (ID + NL
		+ "No mark set for reset.");
	}
}

/**	Skip over some number of input bytes.
<p>
	The {@link #Source() source} input position is moved to its
	current position plus the lesser of the amount specified or
	the amount available data {@link ByteBuffer#remaining() remaining}
	in the buffer.
<p>
	The operation is synchronized on the source buffer.
<p>
	@param	amount	The amount of data to skip.
	@return	The number of data bytes skipped.
*/
public long skip
	(
	long	amount
	)
{
synchronized (Source)
	{
	if (amount > Source.remaining ())
		amount = Source.remaining ();
	if (amount > 0)
		Source.position (Source.position () + (int)amount);
	}
return amount;
}


}
