///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
/*Prog:field
NAME: @code{field} -- plot a field 
@pindex field
@cindex plotting
@cindex plotting data
@cindex RHEOPATH environment variable
@clindex Float
@toindex @code{plotmtv}
@toindex @code{gnuplot}
@toindex @code{vtk}
@toindex @code{mayavi}
@fiindex @file{.field} field

SYNOPSIS:
  @example
  	field [-I@var{dir}] @var{filename}
  @end example
DESCRIPTION:       
  Read and output a finite element field from file.
EXAMPLE: 
  @example
	field square.field
	field square.field -black-and-white
	field box.field
	field box.field -volume
  @end example
INPUT FILE SPECIFICATION:
    @table @code

    @itemx -I@var{dir}
	add @var{dir} to the RHEOPATH search path.
    	See also @ref{geo class} for RHEOPATH mechanism. 

    @itemx @var{filename}
	specifies the name of the file containing
      	the input field.

    @itemx -
	read field on standard input instead on a file.

    @itemx -name
        when field data comes from standard input, the field name
        is not known and is set to "output" by default.
        This option allows to change this default.
        Useful when dealing with output formats (graphic, format conversion)
        that creates auxilliary files, based on this name.

    @end table

INPUT FORMAT OPTIONS:
  @table @code
  @itemx -input-field
        read the data in the @file{.field} format.
        This is the default.

@fiindex @file{.cemagref} cemagref topographic mesh
@cindex  @code{cemagref} topographic mesh

  @itemx -input-cemagref
        input data in cemagref file format.
        The field represents the topographic elevation
	and is assumed to be a @code{P1} approximation.
    @end table

OUTPUT FORMAT OPTIONS:
    @table @code
    @itemx -field
    @itemx -text
	output field on standard output stream in field text file format.

@fiindex @file{.bb} bamg field
    @itemx -bamg
        output field on standard output stream in bamg file format.

@fiindex @file{.bb} mmg3d field
    @itemx -mmg3d
        output field on standard output stream in mmg3d file format.

@fiindex @file{.mshdat} gmsh field
    @itemx -gmsh
        output field on standard output stream in gmsh file format.

@fiindex @file{.cemagref} cemagref topographic mesh
@cindex  @code{cemagref} topographic mesh

    @itemx -cemagref
        output field on standard output stream in cemagref file format.
        The field represents
        the topographic elevation and is assumed to be a @code{P1} approximation.
	This option is suitable for file format conversion:

    @itemx -ndigit @var{int}
	number of digits used to print floating point values.
	Default depends upon the machine precision associated to the
	@code{Float} type.

    @itemx -round
        Round the output, in order for the non-regression tests to
        be portable across platforms. Floating point are machine-dependent.

    @end table

FILE FORMAT CONVERSIONS:
    Input and output options can be combined for efficient
    file format conversion:
    @example
	    field -input-cemagref -field - < mytopo.cemagref > mytopo.field
	    field myfile.field -bamg > myfile.bb
    @end example
    can be performed in one pass:
    @example
	    field -input-cemagref -bamg  - < mytopo.cemagref > myfile.bb
    @end example

GETTING INFORMATION:
    @table @code
    @itemx -min
    @itemx -max
    	print the min (resp. max) value of the scalar field and then exit.
    @end table
RENDER OPTIONS:
    The default render could also specified by using the RHEOLEF_RENDER environment variable.
    @table @code

    @itemx -gnuplot
	use @code{gnuplot} tool.
	This is the default for monodimensional geometries.

    @itemx -mayavi
	use @code{mayavi} tool.
	This is the default for bi- and tridimensional geometries.
    
    @itemx -plotmtv
	use @code{plotmtv} plotting command, for bidimensional geometries.

    @itemx -vtk
	use @code{vtk} tool.
	This is the old (no more supported) tool for tridimensional geometries.
	Could be used in 2d also.

    @end table

RENDERING OPTIONS:
    @table @code
    @itemx -color
    @itemx -gray
    @itemx -black-and-white
	Use (color/gray scale/black and white) rendering.
	Color rendering is the default.
    @cindex anaglyph 3D stereo rendering
    @itemx -stereo
    @itemx -nostereo
  	rendering mode: suitable for red-blue anaglyph 3D stereoscopic glasses.
    @itemx -fill
	isoline intervals are filled with color.
	This is the default.
    @itemx -nofill
	draw isolines by using lines.

    @itemx -grid
	mesh edges appear also.

    @itemx -nogrid
	prevent mesh edges drawing.
	This is the default.

@cindex projection
    @itemx -proj
        Convert all selected fields to P1-continuous approximation by using a L2 projection.
	Usefull when input is a discontinuous field (e.g. P0 or P1d).

    @itemx -iso @var{float}
	extract an iso-value set of points (1d), lines (2d) 
	or surfaces (3d) for a specified value. Output
	the surface in text format (with @code{-text} option)
	or using a graphic driver.
	This option requires the @code{vtk} code.

    @itemx -noiso
	do not draw isosurface.

    @itemx -volume
    @itemx -novolume
	This is an experimental volume representation by using ray cast (mayavi and vtk graphics).
   
    @itemx -n-iso @var{int}
	For 2D visualizations, the isovalue table contains
	regularly spaced values from fmin to fmax, the bounds
        of the field.
@cindex vorticity
@cindex stream function
    @itemx -n-iso-negative @var{int}
	The isovalue table is splitted into negatives and positives values.
	Assume there is n_iso=15 isolines: if 4 is requested by this option,
	then, there will be 4 negatives isolines, regularly spaced
	from fmin to 0 and 11=15-4 positive isolines, regularly spaced
	from 0 to fmax.
	This option is usefull when plotting e.g. vorticity or stream
	functions, where the sign of the field is representative.

    @itemx -label
    @itemx -nolabel
    	Write or do not write labels for values on isolines on
	2d contour plots when using  @code{plotmtv} output.
@cindex elevation
    @itemx elevation
	For two dimensional field, represent values as elevation
	in the third dimension.

    @itemx noelevation
	Prevent from the @code{elevation} representation.
	This is the default.

    @itemx -scale @var{float}
	applies a multiplicative factor to the field.
	This is useful e.g. in conjonction with the @code{elevation} option.
	The default value is 1.

    @itemx -cut
	show a cut by a specified plane.
	The cutting plane is specified by its origin point and normal vector.
	This option requires the @code{vtk} code.

    @itemx -origin @var{float} [@var{float} [@var{float}]]
	set the origin of the cutting plane.
	Default is (0.5, 0.5, 0.5).
    
    @itemx -normal @var{float} [@var{float} [@var{float}]]
	set the normal of the cutting plane.
	Default is (1, 0, 0).

    @end table
OTHERS OPTIONS:
    @table @code
    @itemx -verbose
	print messages related to graphic files created and
       command system calls (this is the default).

    @itemx -noverbose
	does not print previous messages.

    @itemx -clean
	clear temporary graphic files (this is the default).
    
    @itemx -noclean
	does not clear temporary graphic files.

    @itemx -execute
	execute graphic command (this is the default).
    
    @itemx -noexecute
	does not execute graphic command. Generates only 
	graphic files. This is usefull in conjuction with the
	@code{-noclean} command.
    @end table
FIELD FILE FORMAT:
@noindent
It contains a header and a list values at degrees of freedom.
The header contains the @code{field}
keyword followed by a line containing a format version number (presently 1),
the number of degrees of freedom (i.e. the number of values listed),
the mesh file name without the @file{.geo} extension
the approximation (e.g. P1, P2, etc), and finaly the list of values:
@noindent
A sample field file (compatible with the sample mesh example presented in
command manual; see @ref{geo command}) writes:
@example
        field
        1 4
        square
        P1
        0.0
        1.0
        2.0
        3.0
@end example
EXAMPLES:
  @example
    field cube.field -cut -normal 0 1 0 -origin 0.5 0.5 0.5 -vtk
  @end example
  This command send to @code{vtk} the cutted 2d plane of the 3d field.
  @example
    field cube.field -cut -normal 0 1 0 -origin 0.5 0.5 0.5 -text > cube-cut.field
  @end example
  This command generates the cutted 2d field and its associated mesh.
  @example
    field cube.field -iso 0.5 -plotmtv
  @end example
  This command draws the isosurface.
  @example
    field cube.field -iso 0.5 -text > isosurf.geo
  @end example
  This command generates the isosurface as a 3d surface mesh in
  @file{.geo} format. This is suitable for others treatments.

AUTHOR: Pierre.Saramito@imag.fr
SEE ALSO:
    class "field(3)", "space"(3), class "geo"(3), class "base"(3)
DATE:
    7 july 1997
End:
*/
#include "rheolef/rheolef.h"
using namespace rheolef;
using namespace std;

void usage()
{
      cerr << "field: usage: field"
	   << " {-Igeodir}*"
	   << " [-name string]"
	   << " [-input-field|-input-bamg|-input-mmg3d|-input-cemagref]"
	   << " [-field|-text|-bamg|-mmg3d|-gmsh|-cemagref]"
	   << " [-gnuplot|-plotmtv|-vtk|-mayavi]"
	   << " [-color|-gray|-black-and-white]"
	   << " [-[no]stereo|-[no]grid|-[no]fill|-[no]cut|-[no]elevation|-[no]label|-[no]volume]"
	   << " [-noiso|-iso float]"
	   << " [-n-iso int]"
	   << " [-n-iso-negative int]"
           << " [-origin {float}+] [-normal {float}+]"
           << " [-scale float]"
           << " [-[no]clean] [-[no]execute] [-[no]verbose]"
	   << " [-proj]"
	   << " [-domains]"
	   << " [-round]"
           << " [-min|-max]"
	   << " -|filename"
	   << endl;
      exit (1);
}
// round error, for non-regression tests,
inline Float my_round (const Float& x) { return (fabs(x) < Float(1e-10)) ? Float(0) : x; }

typedef enum {
        input_field,
        input_bamg,
        input_mmg3d,
        input_cemagref
} input_type;

typedef enum {
        show_none,
        show_min,
        show_max,
} show_type;

typedef enum { 
	no_render, 
	vtk_render,
	plotmtv_render,
	mayavi_render,
	gnuplot_render,
	file_render
} render_type;

int main(int argc, char**argv)
{
    if (argc <= 1) usage();
    //
    // set default output options
    //
    string name = "output";
    cout << mayavi;
    cout << nogrid;
    cout << fill;
    input_type input_format = input_field;
    clog << verbose; bool bverbose = true;
    int digits10 = numeric_limits<Float>::digits10;
    bool do_round = false;
    bool do_proj = false;
    bool have_input = false;
    bool def_cut_opt = false;
    bool def_fill_opt = false;
    bool use_tmp = false;
    show_type show = show_none;
    render_type render = no_render;
    //
    // load geometry
    //
    field u;

    for (int i = 1; i < argc; i++) {

	if      (strcmp ("-plotmtv", argv [i]) == 0) { cout << plotmtv; render = plotmtv_render; }
	else if (strcmp ("-gnuplot", argv [i]) == 0) { cout << gnuplot; render = gnuplot_render; }
	else if (strcmp ("-vtk",     argv [i]) == 0) { cout << vtk;     render = vtk_render; }
	else if (strcmp ("-mayavi",  argv [i]) == 0) { cout << mayavi;  render = mayavi_render; }

	else if (strcmp ("-text",    argv [i]) == 0) { cout << rheo; render = file_render; }
	else if (strcmp ("-field",   argv [i]) == 0) { cout << rheo; render = file_render; }
	else if (strcmp ("-cemagref",argv [i]) == 0) { cout << cemagref; render = file_render; }
	else if (strcmp ("-bamg",    argv [i]) == 0) { cout << bamg; render = file_render; }
	else if (strcmp ("-mmg3d",   argv [i]) == 0) { cout << mmg3d; render = file_render; }
	else if (strcmp ("-gmsh",    argv [i]) == 0) { cout << gmsh; render = file_render; }

	else if (strcmp ("-tmp",     argv [i]) == 0) { use_tmp=true; }

	else if (strcmp ("-input-field",   argv [i]) == 0) input_format = input_field;
	else if (strcmp ("-input-bamg",    argv [i]) == 0) input_format = input_bamg;
	else if (strcmp ("-input-mmg3d",   argv [i]) == 0) input_format = input_mmg3d;
	else if (strcmp ("-input-cemagref",argv [i]) == 0) input_format = input_cemagref;
        else if (strcmp (argv[i], "-name") == 0) {
            if (i == argc-1) {
                cerr << "geo: -name option: argument missing" << endl;
                usage();
            }
            name = argv[++i];
        }
        else if (strcmp ("-ndigit",  argv [i]) == 0)  digits10 = atoi(argv[++i]);
        else if (strcmp ("-round",   argv [i]) == 0)  do_round = true;
        else if (strcmp ("-proj",    argv[i]) == 0)   do_proj = true;
	else if (strcmp ("-nogrid",  argv [i]) == 0)  cout << nogrid;
	else if (strcmp ("-grid",    argv [i]) == 0)  cout << grid;
	else if (strcmp ("-domains", argv [i]) == 0||
		 strcmp ("-dom",     argv [i]) == 0)  cout << domains;
	else if (strcmp ("-stereo",  argv [i]) == 0)  { cout << stereo; }
	else if (strcmp ("-nostereo",argv [i]) == 0)  { cout << nostereo; }
	else if (strcmp ("-fill",    argv [i]) == 0)  { cout << fill;    def_fill_opt = true; }
	else if (strcmp ("-nofill",  argv [i]) == 0)  { cout << nofill;  def_fill_opt = true; }
	else if (strcmp ("-fill",    argv [i]) == 0)  { cout << fill;    def_fill_opt = true; }
	else if (strcmp ("-nocut",   argv [i]) == 0)  { cout << nocut; def_cut_opt = true; }
	else if (strcmp ("-cut",     argv [i]) == 0)  { cout << cut;   def_cut_opt = true; }
	else if (strcmp ("-elevation",   argv [i]) == 0)  cout << elevation;
	else if (strcmp ("-color",   argv [i]) == 0)  cout << color;
	else if (strcmp ("-gray",   argv [i]) == 0)  cout << gray;
	else if (strcmp ("-black-and-white",   argv [i]) == 0)  cout << black_and_white;
	else if (strcmp ("-noelevation", argv [i]) == 0)  cout << noelevation;

	else if (strcmp (argv[i], "-noclean") == 0)   clog << noclean;
	else if (strcmp (argv[i], "-clean") == 0)     clog << clean;
        else if (strcmp (argv[i], "-noexecute") == 0) clog << noexecute;
        else if (strcmp (argv[i], "-execute") == 0)   clog << execute;
        else if (strcmp (argv[i], "-noverbose") == 0) { bverbose = false; clog << noverbose; }
        else if (strcmp (argv[i], "-verbose") == 0)   { bverbose = true; clog << verbose; }
        else if (strcmp (argv[i], "-nolabel") == 0)   { cout << nolabel; }
        else if (strcmp (argv[i], "-label") == 0)     { cout << label; }
        else if (strcmp (argv[i], "-novolume") == 0)  { cout << novolume; }
        else if (strcmp (argv[i], "-volume") == 0)    { cout << volume; }
        else if (strcmp (argv[i], "-min") == 0)       { show = show_min; }
        else if (strcmp (argv[i], "-max") == 0)       { show = show_max; }
        else if (strcmp (argv[i], "-noiso") == 0)     { cout << noiso; }
        else if (strcmp (argv[i], "-iso") == 0)   {

	    if (i+1 == argc || !is_float(argv[i+1])) usage();
	    Float iso_value = to_float (argv[++i]);
	    cout << iso << setisovalue(iso_value);

	} else if (strcmp (argv[i], "-n-iso") == 0)   {

	    if (i+1 == argc || !isdigit(argv[i+1][0])) usage();
	    size_t idx = atoi (argv[++i]);
	    cout << setn_isovalue(idx);

	} else if (strcmp (argv[i], "-n-iso-negative") == 0)   {

	    if (i+1 == argc || !isdigit(argv[i+1][0])) usage();
	    size_t idx = atoi (argv[++i]);
	    cout << setn_isovalue_negative(idx);

        } else if (strcmp (argv[i], "-scale") == 0)   {

	    if (i+1 == argc || !is_float(argv[i+1])) usage();
	    Float scale = to_float (argv[++i]);
	    cout << setvectorscale (scale);

        } else if ((strcmp (argv[i], "-origin") == 0) || (strcmp (argv[i], "-normal") == 0))   {

	    point x;
	    unsigned int io = i;
	    if (i+1 == argc || !is_float(argv[i+1])) {
	    	warning_macro ("invalid argument to `" << argv[i] << "'");
	    	usage();
	    }
	    x[0] = to_float (argv[++i]);
	    if (i+1 < argc && is_float(argv[i+1])) {
	        x[1] = to_float (argv[++i]);
	        if (i+1 < argc && is_float(argv[i+1])) {
	            x[2] = to_float (argv[++i]);
	        }	
	    }	
            if (strcmp (argv[io], "-origin") == 0)   {
		cout << setorigin(x);
	    } else {
		cout << setnormal(x);
	    }

	} else if (argv [i][0] == '-' && argv [i][1] == 'I') {
	    
	    append_dir_to_rheo_path (argv[i]+2);
        
	} else if (strcmp (argv [i], "-") == 0) {

	    // input field on standard input
	    cin >> setbasename (name);
            if (bverbose) clog << "! load field on stdin" << endl;
            if      (input_format == input_cemagref) cin >> cemagref;
            else if (input_format == input_bamg)  {
		 // concatenate mesh and data at input
		 geo g;
		 cin >> bamg >> g;
		 space V (g, "P1");  // bamg field is always P1
		 u = field (V);	     // field may be allocated for bamg input !
	    }
            else if (input_format == input_mmg3d)  {
		 // concatenate mesh and data at input
		 geo g;
		 cin >> mmg3d >> g;
		 space V (g, "P1");  // bamg field is always P1
		 u = field (V);	     // field may be allocated for bamg input !
	    }
	    cin >> u;
    	    have_input = true;

	} else if (argv [i][0] == '-') {
    
	    usage();

	} else {

            // input on file
            name = get_basename(delete_suffix (delete_suffix (argv[i], "gz"), "field"));
            u = field (argv[i]);
    	    have_input = true;
        }
    }
    if (!have_input) {
	cerr << "field: no input specifield." << endl;
	usage();
    }
    if (use_tmp) { name="/tmp/" + name; cerr << "tmp "<< name; }
    if (render == no_render) {
        if (u.dimension() == 1) {
            cout << gnuplot;
            render == gnuplot_render;
        } else {
	    char* env_render = getenv("RHEOLEF_RENDER");
	    if (env_render!=NULL) 
	    {
		if (strcmp (env_render, "mayavi") == 0)    	
			{ cout << mayavi; render = mayavi_render; }
		else if (strcmp (env_render, "gnuplot") == 0)	
			{ cout << gnuplot; render = gnuplot_render; }
		else if (strcmp (env_render, "vtk") == 0)	
			{ cout << vtk; render = vtk_render; }
		else if (strcmp (env_render, "plotmtv") == 0)	
			{ cout << plotmtv; render = plotmtv_render; }
		else { render = mayavi_render; cout << mayavi; }
	    }
	    else { render = mayavi_render; cout << mayavi; }
        }
    }
    if (u.dimension() == 3 && render != file_render) {
	if (!def_cut_opt) cout << cut;
	if (!def_fill_opt) cout << nofill;
    }
    if (do_round) {
	  numeric_flags<Float>::output_as_double(true);
          u = transform(u,my_round);
    }
    if (do_proj) { // proj to P1-C0 approx
      const space& V = u.get_space();
      space V0 (u.get_geo(), "P1", u.get_valued());
      form m  (V0, V0, "mass");
      form p  (V,  V0, "mass");
      ssk<Float> fact_m = ldlt(m.uu);
      field u_p1 (V0);
      u_p1.u = fact_m.solve((p*u).u);
      u = u_p1;
    }
    if (show == show_min) {
	  cout << u.min() << endl;
	  return 0;
    }
    if (show == show_max) {
	  cout << u.max() << endl;
	  return 0;
    }
    //
    // then output field
    //
    cout << setbasename(name) 
	 << setprecision(digits10) 
	 << u;
    return 0;
}
