# Glossary

### Artifact

An *artifact* is the file or set of files created as a result of the
compilation process. This includes linkable libraries, executable binaries,
and generated documentation.

### Cargo

*Cargo* is the Rust [*package manager*](#package-manager), and the primary
topic of this book.

### Cargo.lock

See [*lock file*](#lock-file).

### Cargo.toml

See [*manifest*](#manifest).

### Crate

A Rust *crate* is either a library or an executable program, referred to as
either a *library crate* or a *binary crate*, respectively.

Every [target](#target) defined for a Cargo [package](#package) is a *crate*.

Loosely, the term *crate* may refer to either the source code of the target or
to the compiled artifact that the target produces. It may also refer to a
compressed package fetched from a [registry](#registry).

The source code for a given crate may be subdivided into [*modules*](#module).

### Edition

A *Rust edition* is a developmental landmark of the Rust language. The
[edition of a package][edition-field] is specified in the `Cargo.toml`
[manifest](#manifest), and individual targets can specify which edition they
use. See the [Edition Guide] for more information.

### Feature

The meaning of *feature* depends on the context:

- A [*feature*][feature] is a named flag which allows for conditional
  compilation. A feature can refer to an optional dependency, or an arbitrary
  name defined in a `Cargo.toml` [manifest](#manifest) that can be checked
  within source code.

- Cargo has [*unstable feature flags*][cargo-unstable] which can be used to
  enable experimental behavior of Cargo itself.

- The Rust compiler and Rustdoc have their own unstable feature flags (see
  [The Unstable Book][unstable-book] and [The Rustdoc
  Book][rustdoc-unstable]).

- CPU targets have [*target features*][target-feature] which specify
  capabilities of a CPU.

### Index

The *index* is the searchable list of [*crates*](#crate) in a
[*registry*](#registry).

### Lock file

The `Cargo.lock` *lock file* is a file that captures the exact version of
every dependency used in a [*workspace*](#workspace) or
[*package*](#package). It is automatically generated by Cargo. See
[Cargo.toml vs Cargo.lock].

### Manifest

A [*manifest*][manifest] is a description of a [package](#package) or a
[workspace](#workspace) in a file named `Cargo.toml`.

A [*virtual manifest*][virtual] is a `Cargo.toml` file that only describes a
workspace, and does not include a package.

### Member

A *member* is a [*package*](#package) that belongs to a
[*workspace*](#workspace).

### Module

Rust's module system is used to organize code into logical units called
*modules*, which provide isolated namespaces within the code.

The source code for a given [crate](#crate) may be subdivided into one or more
separate modules. This is usually done to organize the code into areas of
related functionality or to control the visible scope (public/private) of
symbols within the source (structs, functions, and so on).

A [`Cargo.toml`](#manifest) file is primarily concerned with the
[package](#package) it defines, its crates, and the packages of the crates on
which they depend. Nevertheless, you will see the term "module" often when
working with Rust, so you should understand its relationship to a given crate.

### Package

A *package* is a collection of source files and a `Cargo.toml`
[*manifest*](#manifest) file which describes the package. A package has a name
and version which is used for specifying dependencies between packages.

A package contains multiple [*targets*](#target), each of which is a
[*crate*](#crate). The `Cargo.toml` file describes the type of the crates
(binary or library) within the package, along with some metadata about each
one --- how each is to be built, what their direct dependencies are, etc., as
described throughout this book.

The *package root* is the directory where the package's `Cargo.toml` manifest
is located. (Compare with [*workspace root*](#workspace).)

The [*package ID specification*][pkgid-spec], or *SPEC*, is a string used to
uniquely reference a specific version of a package from a specific source.

Small to medium sized Rust projects will only need a single package, though it
is common for them to have multiple crates.

Larger projects may involve multiple packages, in which case Cargo
[*workspaces*](#workspace) can be used to manage common dependencies and other
related metadata between the packages.

### Package manager

Broadly speaking, a *package manager* is a program (or collection of related
programs) in a software ecosystem that automates the process of obtaining,
installing, and upgrading artifacts. Within a programming language ecosystem,
a package manager is a developer-focused tool whose primary functionality is
to download library artifacts and their dependencies from some central
repository; this capability is often combined with the ability to perform
software builds (by invoking the language-specific compiler).

[*Cargo*](#cargo) is the package manager within the Rust ecosystem. Cargo
downloads your Rust [package](#package)’s dependencies
([*artifacts*](#artifact) known as [*crates*](#crate)), compiles your
packages, makes distributable packages, and (optionally) uploads them to
[crates.io][], the Rust community’s [*package registry*](#registry).

### Package registry

See [*registry*](#registry).

### Project

Another name for a [package](#package).

### Registry

A *registry* is a service that contains a collection of downloadable
[*crates*](#crate) that can be installed or used as dependencies for a
[*package*](#package). The default registry in the Rust ecosystem is
[crates.io](https://crates.io). The registry has an [*index*](#index) which
contains a list of all crates, and tells Cargo how to download the crates that
are needed.

### Source

A *source* is a provider that contains [*crates*](#crate) that may be included
as dependencies for a [*package*](#package). There are several kinds of
sources:

- **Registry source** --- See [registry](#registry).
- **Local registry source** --- A set of crates stored as compressed files on
  the filesystem. See [Local Registry Sources].
- **Directory source** --- A set of crates stored as uncompressed files on the
  filesystem. See [Directory Sources].
- **Path source** --- An individual package located on the filesystem (such as a
  [path dependency]) or a set of multiple packages (such as [path overrides]).
- **Git source** --- Packages located in a git repository (such as a [git
  dependency] or [git source]).

See [Source Replacement] for more information.

### Spec

See [package ID specification](#package).

### Target

The meaning of the term *target* depends on the context:

- **Cargo Target** --- Cargo [*packages*](#package) consist of *targets* which
  correspond to [*artifacts*](#artifact) that will be produced. Packages can
  have library, binary, example, test, and benchmark targets. The
  [list of targets][targets] are configured in the `Cargo.toml`
  [*manifest*](#manifest), often inferred automatically by the [directory
  layout] of the source files.
- **Target Directory** --- Cargo places all built artifacts and intermediate
  files in the *target* directory. By default this is a directory named
  `target` at the [*workspace*](#workspace) root, or the package root if not
  using a workspace. The directory may be changed with the `--target-dir`
  command-line option, the `CARGO_TARGET_DIR` [environment variable], or the
  `build.target-dir` [config option].
- **Target Architecture** --- The OS and machine architecture for the built
  artifacts are typically referred to as a *target*.
- **Target Triple** --- A triple is a specific format for specifying a target
  architecture. Triples may be referred to as a *target triple* which is the
  architecture for the artifact produced, and the *host triple* which is the
  architecture that the compiler is running on. The target triple can be
  specified with the `--target` command-line option or the `build.target`
  [config option]. The general format of the triple is
  `<arch><sub>-<vendor>-<sys>-<abi>` where:

  - `arch` = The base CPU architecture, for example `x86_64`, `i686`, `arm`,
    `thumb`, `mips`, etc.
  - `sub` = The CPU sub-architecture, for example `arm` has `v7`, `v7s`,
    `v5te`, etc.
  - `vendor` = The vendor, for example `unknown`, `apple`, `pc`, `nvidia`, etc.
  - `sys` = The system name, for example `linux`, `windows`, `darwin`, etc.
    `none` is typically used for bare-metal without an OS.
  - `abi` = The ABI, for example `gnu`, `android`, `eabi`, etc.

  Some parameters may be omitted. Run `rustc --print target-list` for a list of
  supported targets.

### Test Targets

Cargo *test targets* generate binaries which help verify proper operation and
correctness of code. There are two types of test artifacts:

* **Unit test** --- A *unit test* is an executable binary compiled directly from
  a library or a binary target. It contains the entire contents of the library
  or binary code, and runs `#[test]` annotated functions, intended to verify
  individual units of code.
* **Integration test target** --- An [*integration test
  target*][integration-tests] is an executable binary compiled from a *test
  target* which is a distinct [*crate*](#crate) whose source is located in the
  `tests` directory or specified by the [`[[test]]` table][targets] in the
  `Cargo.toml` [*manifest*](#manifest). It is intended to only test the public
  API of a library, or execute a binary to verify its operation.

### Workspace

A [*workspace*][workspace] is a collection of one or more
[*packages*](#package) that share common dependency resolution (with a shared
`Cargo.lock` [*lock file*](#lock-file)), output directory, and various
settings such as profiles.

A [*virtual workspace*][virtual] is a workspace where the root `Cargo.toml`
[*manifest*](#manifest) does not define a package, and only lists the
workspace [*members*](#member).

The *workspace root* is the directory where the workspace's `Cargo.toml`
manifest is located. (Compare with [*package root*](#package).)


[Cargo.toml vs Cargo.lock]: ../guide/cargo-toml-vs-cargo-lock.md
[Directory Sources]: ../reference/source-replacement.md#directory-sources
[Local Registry Sources]: ../reference/source-replacement.md#local-registry-sources
[Source Replacement]: ../reference/source-replacement.md
[cargo-unstable]: ../reference/unstable.md
[config option]: ../reference/config.md
[crates.io]: https://crates.io/
[directory layout]: ../guide/project-layout.md
[edition guide]: ../../edition-guide/index.html
[edition-field]: ../reference/manifest.md#the-edition-field
[environment variable]: ../reference/environment-variables.md
[feature]: ../reference/features.md
[git dependency]: ../reference/specifying-dependencies.md#specifying-dependencies-from-git-repositories
[git source]: ../reference/source-replacement.md
[integration-tests]: ../reference/cargo-targets.md#integration-tests
[manifest]: ../reference/manifest.md
[path dependency]: ../reference/specifying-dependencies.md#specifying-path-dependencies
[path overrides]: ../reference/overriding-dependencies.md#paths-overrides
[pkgid-spec]: ../reference/pkgid-spec.md
[rustdoc-unstable]: https://doc.rust-lang.org/nightly/rustdoc/unstable-features.html
[target-feature]: ../../reference/attributes/codegen.html#the-target_feature-attribute
[targets]: ../reference/cargo-targets.md#configuring-a-target
[unstable-book]: https://doc.rust-lang.org/nightly/unstable-book/index.html
[virtual]: ../reference/workspaces.md
[workspace]: ../reference/workspaces.md
