#ifndef RESOLVERTHREAD_H
#define RESOLVERTHREAD_H

#include <sigxconfig.h>
#ifdef SIGC_MSC
#include <Winsock2.h>
// in_addr_t is not defined on windows
typedef ULONG in_addr_t;
#else
#include <netinet/in.h> // in_addr
#endif
#include <sigc++/functors/slot.h>
#include <glibmm/thread.h> // Glib::Private
#include <sigx/glib_threadable.h>
#include <sigx/request_f.h>
#include <sigx/signal_f.h>


/*	Resolves IP addresses to host names.
 *	
 *	IP addresses are resolved by using gethostbyaddr().
 *	You can feed the resolver with addresses by calling resolve(). Results
 *	are signaled after resolving with signal_resolved() to which you can 
 *	connect.
 *	@attention Do not use two IPResolverThreads but only one instance because 
 *	gethostbyaddr() might not be threadsafe on some systems.
 */
class IPResolverThread: public sigx::glib_threadable
{
public:
	// convenience sigc::slot typedefs
	typedef sigc::slot<void> slot_resolving_stopped_t;
	typedef sigc::slot<void> slot_finished_t;
	typedef sigc::slot<void, const std::string&, in_addr_t, int> slot_resolved_t;

protected:
	typedef sigc::signal<void> signal_resolving_stopped_t;
	typedef sigc::signal<void> signal_finished_t;
	typedef sigc::signal<void, const std::string&, in_addr_t, int /*error*/> signal_resolved_t;


public:
	IPResolverThread();


protected:
	// virtuals from sigx::glib_threadable
	virtual void on_startup();
	virtual void on_cleanup();

	/**	@short marshalls the resolving requests and optimizes them
	 *	@note handler for resolve.
	 */
	void on_marshall_resolving(in_addr_t nIP);

	/**	@short resolves an ip address to the hostname and emits a signal. @see signal_resolved
	 */
	bool resolve_next();

	/**	@short Resolves the specified ip address.
	 *	@return Resolving error.
	 */
	int resolve_this(in_addr addr);

	/**	@short stops resolving and emits signal_resolving_stopped() when stopped.
	 *	@note handler for stop_resolving().
	 */
	void on_stop_resolving();

public:
	/**	@name request interface
	 *	All the requests you can send to this thread.
	 */
	//@{
	/**	@short resolves the provided ip.
	 *	
	 *	the result is broadcast with signal_resolved().
	 *	
	 *	@note asynchronous
	 */
	sigx::request_f<in_addr_t> resolve;

	/**	@short stops resolving.
	 *	
	 *	broadcasts signal_resolving_stopped() when stopped.
	 *	
	 *	@note asynchronous
	 */
	sigx::request_f<> stop_resolving;
	//@}

	/**	@name signal interface
	 *	All the messages you can connect to.
	 */
	//@{
	/**	
	 *	@ingroup signal_api
	 */
	sigx::signal_f<signal_resolving_stopped_t> signal_resolving_stopped;

	/**	
	 *	@ingroup signal_api
	 */
	sigx::signal_f<signal_finished_t> signal_finished;

	/**	emitted if resolving ended, no matter whether successful or not.
	 *	host constains an emtpy string if not successful.
	 *	for type of signal and slot @see signal_resolved_t
	 */
	sigx::signal_f<signal_resolved_t> signal_resolved;
	//@}

	
private:
	struct ThreadData;
	Glib::Private<ThreadData> m_ThreadData;
};

#endif
