/* SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Copyright:
 *   2020      Evan Nemerson <evan@nemerson.com>
 *   2020      Hidayat Khan <huk2209@gmail.com>
 *   2020      Christopher Moore <moore@free.fr>
 */

#define SIMDE_TEST_X86_AVX512_INSN broadcast

#include <test/x86/avx512/test-avx512.h>
#include <simde/x86/avx512/broadcast.h>

static int
test_simde_mm256_broadcast_f32x2 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float32 a[4];
    const simde_float32 r[8];
  } test_vec[] = {
    { { SIMDE_FLOAT32_C(  -751.49), SIMDE_FLOAT32_C(  -275.35), SIMDE_FLOAT32_C(   260.73), SIMDE_FLOAT32_C(    40.02) },
      { SIMDE_FLOAT32_C(  -751.49), SIMDE_FLOAT32_C(  -275.35), SIMDE_FLOAT32_C(  -751.49), SIMDE_FLOAT32_C(  -275.35),
        SIMDE_FLOAT32_C(  -751.49), SIMDE_FLOAT32_C(  -275.35), SIMDE_FLOAT32_C(  -751.49), SIMDE_FLOAT32_C(  -275.35) } },
    { { SIMDE_FLOAT32_C(   629.63), SIMDE_FLOAT32_C(   163.39), SIMDE_FLOAT32_C(   167.23), SIMDE_FLOAT32_C(   652.38) },
      { SIMDE_FLOAT32_C(   629.63), SIMDE_FLOAT32_C(   163.39), SIMDE_FLOAT32_C(   629.63), SIMDE_FLOAT32_C(   163.39),
        SIMDE_FLOAT32_C(   629.63), SIMDE_FLOAT32_C(   163.39), SIMDE_FLOAT32_C(   629.63), SIMDE_FLOAT32_C(   163.39) } },
    { { SIMDE_FLOAT32_C(   574.73), SIMDE_FLOAT32_C(  -529.99), SIMDE_FLOAT32_C(   389.79), SIMDE_FLOAT32_C(  -875.04) },
      { SIMDE_FLOAT32_C(   574.73), SIMDE_FLOAT32_C(  -529.99), SIMDE_FLOAT32_C(   574.73), SIMDE_FLOAT32_C(  -529.99),
        SIMDE_FLOAT32_C(   574.73), SIMDE_FLOAT32_C(  -529.99), SIMDE_FLOAT32_C(   574.73), SIMDE_FLOAT32_C(  -529.99) } },
    { { SIMDE_FLOAT32_C(  -790.15), SIMDE_FLOAT32_C(     7.90), SIMDE_FLOAT32_C(   834.33), SIMDE_FLOAT32_C(   549.92) },
      { SIMDE_FLOAT32_C(  -790.15), SIMDE_FLOAT32_C(     7.90), SIMDE_FLOAT32_C(  -790.15), SIMDE_FLOAT32_C(     7.90),
        SIMDE_FLOAT32_C(  -790.15), SIMDE_FLOAT32_C(     7.90), SIMDE_FLOAT32_C(  -790.15), SIMDE_FLOAT32_C(     7.90) } },
    { { SIMDE_FLOAT32_C(   494.62), SIMDE_FLOAT32_C(  -875.96), SIMDE_FLOAT32_C(  -221.96), SIMDE_FLOAT32_C(  -519.70) },
      { SIMDE_FLOAT32_C(   494.62), SIMDE_FLOAT32_C(  -875.96), SIMDE_FLOAT32_C(   494.62), SIMDE_FLOAT32_C(  -875.96),
        SIMDE_FLOAT32_C(   494.62), SIMDE_FLOAT32_C(  -875.96), SIMDE_FLOAT32_C(   494.62), SIMDE_FLOAT32_C(  -875.96) } },
    { { SIMDE_FLOAT32_C(  -583.03), SIMDE_FLOAT32_C(  -938.00), SIMDE_FLOAT32_C(   973.38), SIMDE_FLOAT32_C(  -468.70) },
      { SIMDE_FLOAT32_C(  -583.03), SIMDE_FLOAT32_C(  -938.00), SIMDE_FLOAT32_C(  -583.03), SIMDE_FLOAT32_C(  -938.00),
        SIMDE_FLOAT32_C(  -583.03), SIMDE_FLOAT32_C(  -938.00), SIMDE_FLOAT32_C(  -583.03), SIMDE_FLOAT32_C(  -938.00) } },
    { { SIMDE_FLOAT32_C(   521.04), SIMDE_FLOAT32_C(  -960.21), SIMDE_FLOAT32_C(  -215.76), SIMDE_FLOAT32_C(  -218.82) },
      { SIMDE_FLOAT32_C(   521.04), SIMDE_FLOAT32_C(  -960.21), SIMDE_FLOAT32_C(   521.04), SIMDE_FLOAT32_C(  -960.21),
        SIMDE_FLOAT32_C(   521.04), SIMDE_FLOAT32_C(  -960.21), SIMDE_FLOAT32_C(   521.04), SIMDE_FLOAT32_C(  -960.21) } },
    { { SIMDE_FLOAT32_C(   315.04), SIMDE_FLOAT32_C(   872.51), SIMDE_FLOAT32_C(   318.60), SIMDE_FLOAT32_C(   720.27) },
      { SIMDE_FLOAT32_C(   315.04), SIMDE_FLOAT32_C(   872.51), SIMDE_FLOAT32_C(   315.04), SIMDE_FLOAT32_C(   872.51),
        SIMDE_FLOAT32_C(   315.04), SIMDE_FLOAT32_C(   872.51), SIMDE_FLOAT32_C(   315.04), SIMDE_FLOAT32_C(   872.51) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128 a = simde_mm_loadu_ps(test_vec[i].a);
    simde__m256 r = simde_mm256_broadcast_f32x2(a);
    simde_test_x86_assert_equal_f32x8(r, simde_mm256_loadu_ps(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm256_mask_broadcast_f32x2 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float32 src[8];
    const simde__mmask8 k;
    const simde_float32 a[4];
    const simde_float32 r[8];
  } test_vec[] = {
    { { SIMDE_FLOAT32_C(  -155.94), SIMDE_FLOAT32_C(  -965.17), SIMDE_FLOAT32_C(   378.08), SIMDE_FLOAT32_C(   365.29),
        SIMDE_FLOAT32_C(  -495.97), SIMDE_FLOAT32_C(   311.10), SIMDE_FLOAT32_C(   575.79), SIMDE_FLOAT32_C(  -655.57) },
      UINT8_C( 85),
      { SIMDE_FLOAT32_C(   963.37), SIMDE_FLOAT32_C(  -596.05), SIMDE_FLOAT32_C(   183.95), SIMDE_FLOAT32_C(  -410.87) },
      { SIMDE_FLOAT32_C(   963.37), SIMDE_FLOAT32_C(  -965.17), SIMDE_FLOAT32_C(   963.37), SIMDE_FLOAT32_C(   365.29),
        SIMDE_FLOAT32_C(   963.37), SIMDE_FLOAT32_C(   311.10), SIMDE_FLOAT32_C(   963.37), SIMDE_FLOAT32_C(  -655.57) } },
    { { SIMDE_FLOAT32_C(   431.64), SIMDE_FLOAT32_C(   613.27), SIMDE_FLOAT32_C(  -834.97), SIMDE_FLOAT32_C(   711.68),
        SIMDE_FLOAT32_C(  -862.98), SIMDE_FLOAT32_C(   -74.52), SIMDE_FLOAT32_C(  -451.05), SIMDE_FLOAT32_C(  -751.41) },
      UINT8_C(193),
      { SIMDE_FLOAT32_C(   -39.01), SIMDE_FLOAT32_C(   325.90), SIMDE_FLOAT32_C(  -543.82), SIMDE_FLOAT32_C(    50.30) },
      { SIMDE_FLOAT32_C(   -39.01), SIMDE_FLOAT32_C(   613.27), SIMDE_FLOAT32_C(  -834.97), SIMDE_FLOAT32_C(   711.68),
        SIMDE_FLOAT32_C(  -862.98), SIMDE_FLOAT32_C(   -74.52), SIMDE_FLOAT32_C(   -39.01), SIMDE_FLOAT32_C(   325.90) } },
    { { SIMDE_FLOAT32_C(  -570.27), SIMDE_FLOAT32_C(  -600.03), SIMDE_FLOAT32_C(  -713.28), SIMDE_FLOAT32_C(   -16.45),
        SIMDE_FLOAT32_C(  -512.72), SIMDE_FLOAT32_C(   640.13), SIMDE_FLOAT32_C(   632.82), SIMDE_FLOAT32_C(  -156.53) },
      UINT8_C(110),
      { SIMDE_FLOAT32_C(   351.05), SIMDE_FLOAT32_C(    39.68), SIMDE_FLOAT32_C(   822.74), SIMDE_FLOAT32_C(  -140.05) },
      { SIMDE_FLOAT32_C(  -570.27), SIMDE_FLOAT32_C(    39.68), SIMDE_FLOAT32_C(   351.05), SIMDE_FLOAT32_C(    39.68),
        SIMDE_FLOAT32_C(  -512.72), SIMDE_FLOAT32_C(    39.68), SIMDE_FLOAT32_C(   351.05), SIMDE_FLOAT32_C(  -156.53) } },
    { { SIMDE_FLOAT32_C(   219.95), SIMDE_FLOAT32_C(   765.90), SIMDE_FLOAT32_C(   464.19), SIMDE_FLOAT32_C(  -363.72),
        SIMDE_FLOAT32_C(   978.16), SIMDE_FLOAT32_C(   -55.83), SIMDE_FLOAT32_C(  -268.61), SIMDE_FLOAT32_C(  -471.94) },
      UINT8_C(194),
      { SIMDE_FLOAT32_C(   300.83), SIMDE_FLOAT32_C(   122.56), SIMDE_FLOAT32_C(  -137.37), SIMDE_FLOAT32_C(  -830.55) },
      { SIMDE_FLOAT32_C(   219.95), SIMDE_FLOAT32_C(   122.56), SIMDE_FLOAT32_C(   464.19), SIMDE_FLOAT32_C(  -363.72),
        SIMDE_FLOAT32_C(   978.16), SIMDE_FLOAT32_C(   -55.83), SIMDE_FLOAT32_C(   300.83), SIMDE_FLOAT32_C(   122.56) } },
    { { SIMDE_FLOAT32_C(  -993.95), SIMDE_FLOAT32_C(   735.37), SIMDE_FLOAT32_C(  -715.04), SIMDE_FLOAT32_C(   363.48),
        SIMDE_FLOAT32_C(   997.38), SIMDE_FLOAT32_C(   957.48), SIMDE_FLOAT32_C(   411.04), SIMDE_FLOAT32_C(   318.40) },
      UINT8_C(  0),
      { SIMDE_FLOAT32_C(   944.29), SIMDE_FLOAT32_C(   688.98), SIMDE_FLOAT32_C(  -319.61), SIMDE_FLOAT32_C(   391.33) },
      { SIMDE_FLOAT32_C(  -993.95), SIMDE_FLOAT32_C(   735.37), SIMDE_FLOAT32_C(  -715.04), SIMDE_FLOAT32_C(   363.48),
        SIMDE_FLOAT32_C(   997.38), SIMDE_FLOAT32_C(   957.48), SIMDE_FLOAT32_C(   411.04), SIMDE_FLOAT32_C(   318.40) } },
    { { SIMDE_FLOAT32_C(  -917.62), SIMDE_FLOAT32_C(  -406.65), SIMDE_FLOAT32_C(  -532.97), SIMDE_FLOAT32_C(   298.17),
        SIMDE_FLOAT32_C(  -598.91), SIMDE_FLOAT32_C(   107.47), SIMDE_FLOAT32_C(   214.95), SIMDE_FLOAT32_C(   587.62) },
      UINT8_C(159),
      { SIMDE_FLOAT32_C(  -173.39), SIMDE_FLOAT32_C(  -170.67), SIMDE_FLOAT32_C(  -483.21), SIMDE_FLOAT32_C(   718.07) },
      { SIMDE_FLOAT32_C(  -173.39), SIMDE_FLOAT32_C(  -170.67), SIMDE_FLOAT32_C(  -173.39), SIMDE_FLOAT32_C(  -170.67),
        SIMDE_FLOAT32_C(  -173.39), SIMDE_FLOAT32_C(   107.47), SIMDE_FLOAT32_C(   214.95), SIMDE_FLOAT32_C(  -170.67) } },
    { { SIMDE_FLOAT32_C(   526.28), SIMDE_FLOAT32_C(  -786.80), SIMDE_FLOAT32_C(   286.87), SIMDE_FLOAT32_C(  -560.33),
        SIMDE_FLOAT32_C(   596.72), SIMDE_FLOAT32_C(   991.58), SIMDE_FLOAT32_C(  -572.23), SIMDE_FLOAT32_C(   587.29) },
      UINT8_C( 79),
      { SIMDE_FLOAT32_C(   221.82), SIMDE_FLOAT32_C(   117.18), SIMDE_FLOAT32_C(  -624.10), SIMDE_FLOAT32_C(   727.41) },
      { SIMDE_FLOAT32_C(   221.82), SIMDE_FLOAT32_C(   117.18), SIMDE_FLOAT32_C(   221.82), SIMDE_FLOAT32_C(   117.18),
        SIMDE_FLOAT32_C(   596.72), SIMDE_FLOAT32_C(   991.58), SIMDE_FLOAT32_C(   221.82), SIMDE_FLOAT32_C(   587.29) } },
    { { SIMDE_FLOAT32_C(  -473.57), SIMDE_FLOAT32_C(   647.70), SIMDE_FLOAT32_C(  -174.14), SIMDE_FLOAT32_C(  -701.99),
        SIMDE_FLOAT32_C(  -317.30), SIMDE_FLOAT32_C(  -833.25), SIMDE_FLOAT32_C(  -470.85), SIMDE_FLOAT32_C(   426.74) },
      UINT8_C(169),
      { SIMDE_FLOAT32_C(  -800.29), SIMDE_FLOAT32_C(  -506.53), SIMDE_FLOAT32_C(   682.63), SIMDE_FLOAT32_C(   942.35) },
      { SIMDE_FLOAT32_C(  -800.29), SIMDE_FLOAT32_C(   647.70), SIMDE_FLOAT32_C(  -174.14), SIMDE_FLOAT32_C(  -506.53),
        SIMDE_FLOAT32_C(  -317.30), SIMDE_FLOAT32_C(  -506.53), SIMDE_FLOAT32_C(  -470.85), SIMDE_FLOAT32_C(  -506.53) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256 src = simde_mm256_loadu_ps(test_vec[i].src);
    simde__m128 a = simde_mm_loadu_ps(test_vec[i].a);
    simde__m256 r = simde_mm256_mask_broadcast_f32x2(src, test_vec[i].k, a);
    simde_test_x86_assert_equal_f32x8(r, simde_mm256_loadu_ps(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm256_maskz_broadcast_f32x2 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde__mmask8 k;
    const simde_float32 a[4];
    const simde_float32 r[8];
  } test_vec[] = {
    { UINT8_C(167),
      { SIMDE_FLOAT32_C(   -73.48), SIMDE_FLOAT32_C(  -950.66), SIMDE_FLOAT32_C(   265.90), SIMDE_FLOAT32_C(  -988.50) },
      { SIMDE_FLOAT32_C(   -73.48), SIMDE_FLOAT32_C(  -950.66), SIMDE_FLOAT32_C(   -73.48), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -950.66), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -950.66) } },
    { UINT8_C(122),
      { SIMDE_FLOAT32_C(   490.14), SIMDE_FLOAT32_C(  -286.45), SIMDE_FLOAT32_C(  -424.27), SIMDE_FLOAT32_C(  -754.18) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -286.45), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -286.45),
        SIMDE_FLOAT32_C(   490.14), SIMDE_FLOAT32_C(  -286.45), SIMDE_FLOAT32_C(   490.14), SIMDE_FLOAT32_C(     0.00) } },
    { UINT8_C( 66),
      { SIMDE_FLOAT32_C(  -622.52), SIMDE_FLOAT32_C(  -691.02), SIMDE_FLOAT32_C(    48.53), SIMDE_FLOAT32_C(  -368.74) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -691.02), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -622.52), SIMDE_FLOAT32_C(     0.00) } },
    { UINT8_C(140),
      { SIMDE_FLOAT32_C(   336.37), SIMDE_FLOAT32_C(  -709.34), SIMDE_FLOAT32_C(    65.79), SIMDE_FLOAT32_C(  -200.10) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   336.37), SIMDE_FLOAT32_C(  -709.34),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -709.34) } },
    { UINT8_C(151),
      { SIMDE_FLOAT32_C(   450.42), SIMDE_FLOAT32_C(   257.72), SIMDE_FLOAT32_C(  -507.45), SIMDE_FLOAT32_C(  -644.25) },
      { SIMDE_FLOAT32_C(   450.42), SIMDE_FLOAT32_C(   257.72), SIMDE_FLOAT32_C(   450.42), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(   450.42), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   257.72) } },
    { UINT8_C( 11),
      { SIMDE_FLOAT32_C(  -161.31), SIMDE_FLOAT32_C(   845.16), SIMDE_FLOAT32_C(   584.32), SIMDE_FLOAT32_C(   641.28) },
      { SIMDE_FLOAT32_C(  -161.31), SIMDE_FLOAT32_C(   845.16), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   845.16),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00) } },
    { UINT8_C( 82),
      { SIMDE_FLOAT32_C(   565.26), SIMDE_FLOAT32_C(   325.20), SIMDE_FLOAT32_C(  -344.79), SIMDE_FLOAT32_C(  -940.47) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   325.20), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(   565.26), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   565.26), SIMDE_FLOAT32_C(     0.00) } },
    { UINT8_C(152),
      { SIMDE_FLOAT32_C(   715.85), SIMDE_FLOAT32_C(  -726.67), SIMDE_FLOAT32_C(   812.36), SIMDE_FLOAT32_C(  -643.19) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -726.67),
        SIMDE_FLOAT32_C(   715.85), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -726.67) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128 a = simde_mm_loadu_ps(test_vec[i].a);
    simde__m256 r = simde_mm256_maskz_broadcast_f32x2(test_vec[i].k, a);
    simde_test_x86_assert_equal_f32x8(r, simde_mm256_loadu_ps(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm512_broadcast_f32x2 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float32 a[4];
    const simde_float32 r[16];
  } test_vec[] = {
    { { SIMDE_FLOAT32_C(  -125.63), SIMDE_FLOAT32_C(   601.06), SIMDE_FLOAT32_C(    20.21), SIMDE_FLOAT32_C(  -317.28) },
      { SIMDE_FLOAT32_C(  -125.63), SIMDE_FLOAT32_C(   601.06), SIMDE_FLOAT32_C(  -125.63), SIMDE_FLOAT32_C(   601.06),
        SIMDE_FLOAT32_C(  -125.63), SIMDE_FLOAT32_C(   601.06), SIMDE_FLOAT32_C(  -125.63), SIMDE_FLOAT32_C(   601.06),
        SIMDE_FLOAT32_C(  -125.63), SIMDE_FLOAT32_C(   601.06), SIMDE_FLOAT32_C(  -125.63), SIMDE_FLOAT32_C(   601.06),
        SIMDE_FLOAT32_C(  -125.63), SIMDE_FLOAT32_C(   601.06), SIMDE_FLOAT32_C(  -125.63), SIMDE_FLOAT32_C(   601.06) } },
    { { SIMDE_FLOAT32_C(  -590.78), SIMDE_FLOAT32_C(   832.90), SIMDE_FLOAT32_C(   590.84), SIMDE_FLOAT32_C(   180.72) },
      { SIMDE_FLOAT32_C(  -590.78), SIMDE_FLOAT32_C(   832.90), SIMDE_FLOAT32_C(  -590.78), SIMDE_FLOAT32_C(   832.90),
        SIMDE_FLOAT32_C(  -590.78), SIMDE_FLOAT32_C(   832.90), SIMDE_FLOAT32_C(  -590.78), SIMDE_FLOAT32_C(   832.90),
        SIMDE_FLOAT32_C(  -590.78), SIMDE_FLOAT32_C(   832.90), SIMDE_FLOAT32_C(  -590.78), SIMDE_FLOAT32_C(   832.90),
        SIMDE_FLOAT32_C(  -590.78), SIMDE_FLOAT32_C(   832.90), SIMDE_FLOAT32_C(  -590.78), SIMDE_FLOAT32_C(   832.90) } },
    { { SIMDE_FLOAT32_C(  -605.74), SIMDE_FLOAT32_C(  -713.02), SIMDE_FLOAT32_C(   218.93), SIMDE_FLOAT32_C(  -470.99) },
      { SIMDE_FLOAT32_C(  -605.74), SIMDE_FLOAT32_C(  -713.02), SIMDE_FLOAT32_C(  -605.74), SIMDE_FLOAT32_C(  -713.02),
        SIMDE_FLOAT32_C(  -605.74), SIMDE_FLOAT32_C(  -713.02), SIMDE_FLOAT32_C(  -605.74), SIMDE_FLOAT32_C(  -713.02),
        SIMDE_FLOAT32_C(  -605.74), SIMDE_FLOAT32_C(  -713.02), SIMDE_FLOAT32_C(  -605.74), SIMDE_FLOAT32_C(  -713.02),
        SIMDE_FLOAT32_C(  -605.74), SIMDE_FLOAT32_C(  -713.02), SIMDE_FLOAT32_C(  -605.74), SIMDE_FLOAT32_C(  -713.02) } },
    { { SIMDE_FLOAT32_C(    61.13), SIMDE_FLOAT32_C(  -592.59), SIMDE_FLOAT32_C(   423.81), SIMDE_FLOAT32_C(   987.29) },
      { SIMDE_FLOAT32_C(    61.13), SIMDE_FLOAT32_C(  -592.59), SIMDE_FLOAT32_C(    61.13), SIMDE_FLOAT32_C(  -592.59),
        SIMDE_FLOAT32_C(    61.13), SIMDE_FLOAT32_C(  -592.59), SIMDE_FLOAT32_C(    61.13), SIMDE_FLOAT32_C(  -592.59),
        SIMDE_FLOAT32_C(    61.13), SIMDE_FLOAT32_C(  -592.59), SIMDE_FLOAT32_C(    61.13), SIMDE_FLOAT32_C(  -592.59),
        SIMDE_FLOAT32_C(    61.13), SIMDE_FLOAT32_C(  -592.59), SIMDE_FLOAT32_C(    61.13), SIMDE_FLOAT32_C(  -592.59) } },
    { { SIMDE_FLOAT32_C(  -608.09), SIMDE_FLOAT32_C(   -99.23), SIMDE_FLOAT32_C(   300.10), SIMDE_FLOAT32_C(  -254.94) },
      { SIMDE_FLOAT32_C(  -608.09), SIMDE_FLOAT32_C(   -99.23), SIMDE_FLOAT32_C(  -608.09), SIMDE_FLOAT32_C(   -99.23),
        SIMDE_FLOAT32_C(  -608.09), SIMDE_FLOAT32_C(   -99.23), SIMDE_FLOAT32_C(  -608.09), SIMDE_FLOAT32_C(   -99.23),
        SIMDE_FLOAT32_C(  -608.09), SIMDE_FLOAT32_C(   -99.23), SIMDE_FLOAT32_C(  -608.09), SIMDE_FLOAT32_C(   -99.23),
        SIMDE_FLOAT32_C(  -608.09), SIMDE_FLOAT32_C(   -99.23), SIMDE_FLOAT32_C(  -608.09), SIMDE_FLOAT32_C(   -99.23) } },
    { { SIMDE_FLOAT32_C(  -727.78), SIMDE_FLOAT32_C(   285.14), SIMDE_FLOAT32_C(   318.61), SIMDE_FLOAT32_C(   956.19) },
      { SIMDE_FLOAT32_C(  -727.78), SIMDE_FLOAT32_C(   285.14), SIMDE_FLOAT32_C(  -727.78), SIMDE_FLOAT32_C(   285.14),
        SIMDE_FLOAT32_C(  -727.78), SIMDE_FLOAT32_C(   285.14), SIMDE_FLOAT32_C(  -727.78), SIMDE_FLOAT32_C(   285.14),
        SIMDE_FLOAT32_C(  -727.78), SIMDE_FLOAT32_C(   285.14), SIMDE_FLOAT32_C(  -727.78), SIMDE_FLOAT32_C(   285.14),
        SIMDE_FLOAT32_C(  -727.78), SIMDE_FLOAT32_C(   285.14), SIMDE_FLOAT32_C(  -727.78), SIMDE_FLOAT32_C(   285.14) } },
    { { SIMDE_FLOAT32_C(   704.27), SIMDE_FLOAT32_C(   738.40), SIMDE_FLOAT32_C(   301.28), SIMDE_FLOAT32_C(  -459.90) },
      { SIMDE_FLOAT32_C(   704.27), SIMDE_FLOAT32_C(   738.40), SIMDE_FLOAT32_C(   704.27), SIMDE_FLOAT32_C(   738.40),
        SIMDE_FLOAT32_C(   704.27), SIMDE_FLOAT32_C(   738.40), SIMDE_FLOAT32_C(   704.27), SIMDE_FLOAT32_C(   738.40),
        SIMDE_FLOAT32_C(   704.27), SIMDE_FLOAT32_C(   738.40), SIMDE_FLOAT32_C(   704.27), SIMDE_FLOAT32_C(   738.40),
        SIMDE_FLOAT32_C(   704.27), SIMDE_FLOAT32_C(   738.40), SIMDE_FLOAT32_C(   704.27), SIMDE_FLOAT32_C(   738.40) } },
    { { SIMDE_FLOAT32_C(   379.79), SIMDE_FLOAT32_C(  -819.17), SIMDE_FLOAT32_C(   172.39), SIMDE_FLOAT32_C(  -722.17) },
      { SIMDE_FLOAT32_C(   379.79), SIMDE_FLOAT32_C(  -819.17), SIMDE_FLOAT32_C(   379.79), SIMDE_FLOAT32_C(  -819.17),
        SIMDE_FLOAT32_C(   379.79), SIMDE_FLOAT32_C(  -819.17), SIMDE_FLOAT32_C(   379.79), SIMDE_FLOAT32_C(  -819.17),
        SIMDE_FLOAT32_C(   379.79), SIMDE_FLOAT32_C(  -819.17), SIMDE_FLOAT32_C(   379.79), SIMDE_FLOAT32_C(  -819.17),
        SIMDE_FLOAT32_C(   379.79), SIMDE_FLOAT32_C(  -819.17), SIMDE_FLOAT32_C(   379.79), SIMDE_FLOAT32_C(  -819.17) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128 a = simde_mm_loadu_ps(test_vec[i].a);
    simde__m512 r = simde_mm512_broadcast_f32x2(a);
    simde_test_x86_assert_equal_f32x16(r, simde_mm512_loadu_ps(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm512_mask_broadcast_f32x2 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float32 src[16];
    const simde__mmask16 k;
    const simde_float32 a[4];
    const simde_float32 r[16];
  } test_vec[] = {
    { { SIMDE_FLOAT32_C(    16.97), SIMDE_FLOAT32_C(  -724.36), SIMDE_FLOAT32_C(  -251.03), SIMDE_FLOAT32_C(   955.86),
        SIMDE_FLOAT32_C(  -884.86), SIMDE_FLOAT32_C(    79.30), SIMDE_FLOAT32_C(   805.27), SIMDE_FLOAT32_C(   217.58),
        SIMDE_FLOAT32_C(   919.33), SIMDE_FLOAT32_C(  -770.42), SIMDE_FLOAT32_C(  -363.93), SIMDE_FLOAT32_C(  -528.80),
        SIMDE_FLOAT32_C(   387.46), SIMDE_FLOAT32_C(     8.94), SIMDE_FLOAT32_C(   238.55), SIMDE_FLOAT32_C(  -769.11) },
      UINT16_C(26495),
      { SIMDE_FLOAT32_C(  -701.54), SIMDE_FLOAT32_C(  -832.82), SIMDE_FLOAT32_C(   858.15), SIMDE_FLOAT32_C(   988.45) },
      { SIMDE_FLOAT32_C(  -701.54), SIMDE_FLOAT32_C(  -832.82), SIMDE_FLOAT32_C(  -701.54), SIMDE_FLOAT32_C(  -832.82),
        SIMDE_FLOAT32_C(  -701.54), SIMDE_FLOAT32_C(  -832.82), SIMDE_FLOAT32_C(  -701.54), SIMDE_FLOAT32_C(   217.58),
        SIMDE_FLOAT32_C(  -701.54), SIMDE_FLOAT32_C(  -832.82), SIMDE_FLOAT32_C(  -701.54), SIMDE_FLOAT32_C(  -528.80),
        SIMDE_FLOAT32_C(   387.46), SIMDE_FLOAT32_C(  -832.82), SIMDE_FLOAT32_C(  -701.54), SIMDE_FLOAT32_C(  -769.11) } },
    { { SIMDE_FLOAT32_C(   886.30), SIMDE_FLOAT32_C(   115.75), SIMDE_FLOAT32_C(  -627.06), SIMDE_FLOAT32_C(  -987.33),
        SIMDE_FLOAT32_C(  -126.79), SIMDE_FLOAT32_C(   964.00), SIMDE_FLOAT32_C(  -128.64), SIMDE_FLOAT32_C(   -75.15),
        SIMDE_FLOAT32_C(   949.72), SIMDE_FLOAT32_C(  -114.82), SIMDE_FLOAT32_C(   286.01), SIMDE_FLOAT32_C(  -995.38),
        SIMDE_FLOAT32_C(   721.81), SIMDE_FLOAT32_C(  -531.94), SIMDE_FLOAT32_C(  -379.35), SIMDE_FLOAT32_C(   301.40) },
      UINT16_C(55066),
      { SIMDE_FLOAT32_C(  -112.99), SIMDE_FLOAT32_C(   933.42), SIMDE_FLOAT32_C(   -66.18), SIMDE_FLOAT32_C(  -307.32) },
      { SIMDE_FLOAT32_C(   886.30), SIMDE_FLOAT32_C(   933.42), SIMDE_FLOAT32_C(  -627.06), SIMDE_FLOAT32_C(   933.42),
        SIMDE_FLOAT32_C(  -112.99), SIMDE_FLOAT32_C(   964.00), SIMDE_FLOAT32_C(  -128.64), SIMDE_FLOAT32_C(   -75.15),
        SIMDE_FLOAT32_C(  -112.99), SIMDE_FLOAT32_C(   933.42), SIMDE_FLOAT32_C(  -112.99), SIMDE_FLOAT32_C(  -995.38),
        SIMDE_FLOAT32_C(  -112.99), SIMDE_FLOAT32_C(  -531.94), SIMDE_FLOAT32_C(  -112.99), SIMDE_FLOAT32_C(   933.42) } },
    { { SIMDE_FLOAT32_C(   858.06), SIMDE_FLOAT32_C(  -630.09), SIMDE_FLOAT32_C(    82.49), SIMDE_FLOAT32_C(   401.49),
        SIMDE_FLOAT32_C(  -226.24), SIMDE_FLOAT32_C(  -448.63), SIMDE_FLOAT32_C(  -200.28), SIMDE_FLOAT32_C(  -144.91),
        SIMDE_FLOAT32_C(   574.72), SIMDE_FLOAT32_C(  -647.66), SIMDE_FLOAT32_C(   850.68), SIMDE_FLOAT32_C(  -645.45),
        SIMDE_FLOAT32_C(  -136.23), SIMDE_FLOAT32_C(   385.26), SIMDE_FLOAT32_C(  -998.08), SIMDE_FLOAT32_C(  -718.84) },
      UINT16_C(39639),
      { SIMDE_FLOAT32_C(  -394.96), SIMDE_FLOAT32_C(   -89.93), SIMDE_FLOAT32_C(   511.24), SIMDE_FLOAT32_C(   328.98) },
      { SIMDE_FLOAT32_C(  -394.96), SIMDE_FLOAT32_C(   -89.93), SIMDE_FLOAT32_C(  -394.96), SIMDE_FLOAT32_C(   401.49),
        SIMDE_FLOAT32_C(  -394.96), SIMDE_FLOAT32_C(  -448.63), SIMDE_FLOAT32_C(  -394.96), SIMDE_FLOAT32_C(   -89.93),
        SIMDE_FLOAT32_C(   574.72), SIMDE_FLOAT32_C(   -89.93), SIMDE_FLOAT32_C(   850.68), SIMDE_FLOAT32_C(   -89.93),
        SIMDE_FLOAT32_C(  -394.96), SIMDE_FLOAT32_C(   385.26), SIMDE_FLOAT32_C(  -998.08), SIMDE_FLOAT32_C(   -89.93) } },
    { { SIMDE_FLOAT32_C(  -783.73), SIMDE_FLOAT32_C(  -210.92), SIMDE_FLOAT32_C(  -991.67), SIMDE_FLOAT32_C(   979.95),
        SIMDE_FLOAT32_C(    49.71), SIMDE_FLOAT32_C(  -489.71), SIMDE_FLOAT32_C(  -591.16), SIMDE_FLOAT32_C(   388.37),
        SIMDE_FLOAT32_C(  -622.36), SIMDE_FLOAT32_C(    45.42), SIMDE_FLOAT32_C(  -553.07), SIMDE_FLOAT32_C(   498.54),
        SIMDE_FLOAT32_C(   904.46), SIMDE_FLOAT32_C(  -795.68), SIMDE_FLOAT32_C(  -943.60), SIMDE_FLOAT32_C(   933.59) },
      UINT16_C(44422),
      { SIMDE_FLOAT32_C(   213.33), SIMDE_FLOAT32_C(  -541.90), SIMDE_FLOAT32_C(   310.55), SIMDE_FLOAT32_C(  -596.77) },
      { SIMDE_FLOAT32_C(  -783.73), SIMDE_FLOAT32_C(  -541.90), SIMDE_FLOAT32_C(   213.33), SIMDE_FLOAT32_C(   979.95),
        SIMDE_FLOAT32_C(    49.71), SIMDE_FLOAT32_C(  -489.71), SIMDE_FLOAT32_C(  -591.16), SIMDE_FLOAT32_C(  -541.90),
        SIMDE_FLOAT32_C(   213.33), SIMDE_FLOAT32_C(    45.42), SIMDE_FLOAT32_C(   213.33), SIMDE_FLOAT32_C(  -541.90),
        SIMDE_FLOAT32_C(   904.46), SIMDE_FLOAT32_C(  -541.90), SIMDE_FLOAT32_C(  -943.60), SIMDE_FLOAT32_C(  -541.90) } },
    { { SIMDE_FLOAT32_C(     4.43), SIMDE_FLOAT32_C(   378.61), SIMDE_FLOAT32_C(  -660.44), SIMDE_FLOAT32_C(   -60.44),
        SIMDE_FLOAT32_C(   265.90), SIMDE_FLOAT32_C(   922.57), SIMDE_FLOAT32_C(  -447.45), SIMDE_FLOAT32_C(  -208.75),
        SIMDE_FLOAT32_C(  -386.55), SIMDE_FLOAT32_C(  -791.16), SIMDE_FLOAT32_C(   993.63), SIMDE_FLOAT32_C(  -107.89),
        SIMDE_FLOAT32_C(   758.84), SIMDE_FLOAT32_C(  -215.37), SIMDE_FLOAT32_C(   198.46), SIMDE_FLOAT32_C(  -486.35) },
      UINT16_C(19819),
      { SIMDE_FLOAT32_C(   413.19), SIMDE_FLOAT32_C(   527.77), SIMDE_FLOAT32_C(   286.90), SIMDE_FLOAT32_C(   -50.52) },
      { SIMDE_FLOAT32_C(   413.19), SIMDE_FLOAT32_C(   527.77), SIMDE_FLOAT32_C(  -660.44), SIMDE_FLOAT32_C(   527.77),
        SIMDE_FLOAT32_C(   265.90), SIMDE_FLOAT32_C(   527.77), SIMDE_FLOAT32_C(   413.19), SIMDE_FLOAT32_C(  -208.75),
        SIMDE_FLOAT32_C(   413.19), SIMDE_FLOAT32_C(  -791.16), SIMDE_FLOAT32_C(   413.19), SIMDE_FLOAT32_C(   527.77),
        SIMDE_FLOAT32_C(   758.84), SIMDE_FLOAT32_C(  -215.37), SIMDE_FLOAT32_C(   413.19), SIMDE_FLOAT32_C(  -486.35) } },
    { { SIMDE_FLOAT32_C(   968.23), SIMDE_FLOAT32_C(  -877.74), SIMDE_FLOAT32_C(  -102.63), SIMDE_FLOAT32_C(  -954.86),
        SIMDE_FLOAT32_C(  -411.69), SIMDE_FLOAT32_C(   708.12), SIMDE_FLOAT32_C(  -635.17), SIMDE_FLOAT32_C(   743.77),
        SIMDE_FLOAT32_C(   622.65), SIMDE_FLOAT32_C(   851.75), SIMDE_FLOAT32_C(  -569.83), SIMDE_FLOAT32_C(   908.51),
        SIMDE_FLOAT32_C(  -674.71), SIMDE_FLOAT32_C(   173.61), SIMDE_FLOAT32_C(  -162.66), SIMDE_FLOAT32_C(   200.03) },
      UINT16_C(57825),
      { SIMDE_FLOAT32_C(  -696.94), SIMDE_FLOAT32_C(   529.84), SIMDE_FLOAT32_C(  -942.89), SIMDE_FLOAT32_C(   880.87) },
      { SIMDE_FLOAT32_C(  -696.94), SIMDE_FLOAT32_C(  -877.74), SIMDE_FLOAT32_C(  -102.63), SIMDE_FLOAT32_C(  -954.86),
        SIMDE_FLOAT32_C(  -411.69), SIMDE_FLOAT32_C(   529.84), SIMDE_FLOAT32_C(  -696.94), SIMDE_FLOAT32_C(   529.84),
        SIMDE_FLOAT32_C(  -696.94), SIMDE_FLOAT32_C(   851.75), SIMDE_FLOAT32_C(  -569.83), SIMDE_FLOAT32_C(   908.51),
        SIMDE_FLOAT32_C(  -674.71), SIMDE_FLOAT32_C(   529.84), SIMDE_FLOAT32_C(  -696.94), SIMDE_FLOAT32_C(   529.84) } },
    { { SIMDE_FLOAT32_C(   733.15), SIMDE_FLOAT32_C(    63.36), SIMDE_FLOAT32_C(   903.02), SIMDE_FLOAT32_C(  -977.76),
        SIMDE_FLOAT32_C(   704.77), SIMDE_FLOAT32_C(   985.75), SIMDE_FLOAT32_C(  -492.96), SIMDE_FLOAT32_C(   872.57),
        SIMDE_FLOAT32_C(  -697.69), SIMDE_FLOAT32_C(   -32.06), SIMDE_FLOAT32_C(  -826.65), SIMDE_FLOAT32_C(   423.95),
        SIMDE_FLOAT32_C(  -668.70), SIMDE_FLOAT32_C(  -777.46), SIMDE_FLOAT32_C(  -794.02), SIMDE_FLOAT32_C(   931.91) },
      UINT16_C(22885),
      { SIMDE_FLOAT32_C(   241.78), SIMDE_FLOAT32_C(  -340.95), SIMDE_FLOAT32_C(  -411.67), SIMDE_FLOAT32_C(  -904.01) },
      { SIMDE_FLOAT32_C(   241.78), SIMDE_FLOAT32_C(    63.36), SIMDE_FLOAT32_C(   241.78), SIMDE_FLOAT32_C(  -977.76),
        SIMDE_FLOAT32_C(   704.77), SIMDE_FLOAT32_C(  -340.95), SIMDE_FLOAT32_C(   241.78), SIMDE_FLOAT32_C(   872.57),
        SIMDE_FLOAT32_C(   241.78), SIMDE_FLOAT32_C(   -32.06), SIMDE_FLOAT32_C(  -826.65), SIMDE_FLOAT32_C(  -340.95),
        SIMDE_FLOAT32_C(   241.78), SIMDE_FLOAT32_C(  -777.46), SIMDE_FLOAT32_C(   241.78), SIMDE_FLOAT32_C(   931.91) } },
    { { SIMDE_FLOAT32_C(   377.61), SIMDE_FLOAT32_C(   543.54), SIMDE_FLOAT32_C(  -676.81), SIMDE_FLOAT32_C(   796.04),
        SIMDE_FLOAT32_C(  -952.55), SIMDE_FLOAT32_C(   439.69), SIMDE_FLOAT32_C(  -139.34), SIMDE_FLOAT32_C(   103.48),
        SIMDE_FLOAT32_C(  -782.74), SIMDE_FLOAT32_C(   562.99), SIMDE_FLOAT32_C(   161.99), SIMDE_FLOAT32_C(   620.38),
        SIMDE_FLOAT32_C(   696.86), SIMDE_FLOAT32_C(    88.47), SIMDE_FLOAT32_C(   998.69), SIMDE_FLOAT32_C(  -955.66) },
      UINT16_C(13591),
      { SIMDE_FLOAT32_C(  -395.69), SIMDE_FLOAT32_C(  -372.87), SIMDE_FLOAT32_C(  -839.61), SIMDE_FLOAT32_C(   668.17) },
      { SIMDE_FLOAT32_C(  -395.69), SIMDE_FLOAT32_C(  -372.87), SIMDE_FLOAT32_C(  -395.69), SIMDE_FLOAT32_C(   796.04),
        SIMDE_FLOAT32_C(  -395.69), SIMDE_FLOAT32_C(   439.69), SIMDE_FLOAT32_C(  -139.34), SIMDE_FLOAT32_C(   103.48),
        SIMDE_FLOAT32_C(  -395.69), SIMDE_FLOAT32_C(   562.99), SIMDE_FLOAT32_C(  -395.69), SIMDE_FLOAT32_C(   620.38),
        SIMDE_FLOAT32_C(  -395.69), SIMDE_FLOAT32_C(  -372.87), SIMDE_FLOAT32_C(   998.69), SIMDE_FLOAT32_C(  -955.66) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512 src = simde_mm512_loadu_ps(test_vec[i].src);
    simde__m128 a = simde_mm_loadu_ps(test_vec[i].a);
    simde__m512 r = simde_mm512_mask_broadcast_f32x2(src, test_vec[i].k, a);
    simde_test_x86_assert_equal_f32x16(r, simde_mm512_loadu_ps(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm512_maskz_broadcast_f32x2 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde__mmask16 k;
    const simde_float32 a[4];
    const simde_float32 r[16];
  } test_vec[] = {
            { UINT16_C(18884),
      { SIMDE_FLOAT32_C(   545.10), SIMDE_FLOAT32_C(  -550.17), SIMDE_FLOAT32_C(  -710.41), SIMDE_FLOAT32_C(   204.85) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   545.10), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   545.10), SIMDE_FLOAT32_C(  -550.17),
        SIMDE_FLOAT32_C(   545.10), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -550.17),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   545.10), SIMDE_FLOAT32_C(     0.00) } },
    { UINT16_C(16968),
      { SIMDE_FLOAT32_C(    51.85), SIMDE_FLOAT32_C(  -493.14), SIMDE_FLOAT32_C(  -214.52), SIMDE_FLOAT32_C(   484.86) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -493.14),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    51.85), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -493.14), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    51.85), SIMDE_FLOAT32_C(     0.00) } },
    { UINT16_C(55493),
      { SIMDE_FLOAT32_C(  -547.31), SIMDE_FLOAT32_C(  -681.83), SIMDE_FLOAT32_C(   567.76), SIMDE_FLOAT32_C(   376.14) },
      { SIMDE_FLOAT32_C(  -547.31), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -547.31), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -547.31), SIMDE_FLOAT32_C(  -681.83),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -681.83),
        SIMDE_FLOAT32_C(  -547.31), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -547.31), SIMDE_FLOAT32_C(  -681.83) } },
    { UINT16_C( 1280),
      { SIMDE_FLOAT32_C(   358.99), SIMDE_FLOAT32_C(  -507.35), SIMDE_FLOAT32_C(  -959.80), SIMDE_FLOAT32_C(   688.48) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(   358.99), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   358.99), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00) } },
    { UINT16_C(16569),
      { SIMDE_FLOAT32_C(  -988.71), SIMDE_FLOAT32_C(   789.03), SIMDE_FLOAT32_C(  -740.57), SIMDE_FLOAT32_C(  -739.46) },
      { SIMDE_FLOAT32_C(  -988.71), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   789.03),
        SIMDE_FLOAT32_C(  -988.71), SIMDE_FLOAT32_C(   789.03), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   789.03),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -988.71), SIMDE_FLOAT32_C(     0.00) } },
    { UINT16_C(26242),
      { SIMDE_FLOAT32_C(  -555.34), SIMDE_FLOAT32_C(   402.79), SIMDE_FLOAT32_C(  -274.64), SIMDE_FLOAT32_C(   159.53) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   402.79), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   402.79),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   402.79), SIMDE_FLOAT32_C(  -555.34), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   402.79), SIMDE_FLOAT32_C(  -555.34), SIMDE_FLOAT32_C(     0.00) } },
    { UINT16_C(39055),
      { SIMDE_FLOAT32_C(   -25.84), SIMDE_FLOAT32_C(  -228.90), SIMDE_FLOAT32_C(   813.40), SIMDE_FLOAT32_C(   762.90) },
      { SIMDE_FLOAT32_C(   -25.84), SIMDE_FLOAT32_C(  -228.90), SIMDE_FLOAT32_C(   -25.84), SIMDE_FLOAT32_C(  -228.90),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -228.90),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -228.90),
        SIMDE_FLOAT32_C(   -25.84), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -228.90) } },
    { UINT16_C(53187),
      { SIMDE_FLOAT32_C(  -400.08), SIMDE_FLOAT32_C(  -173.64), SIMDE_FLOAT32_C(  -349.66), SIMDE_FLOAT32_C(  -663.64) },
      { SIMDE_FLOAT32_C(  -400.08), SIMDE_FLOAT32_C(  -173.64), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -400.08), SIMDE_FLOAT32_C(  -173.64),
        SIMDE_FLOAT32_C(  -400.08), SIMDE_FLOAT32_C(  -173.64), SIMDE_FLOAT32_C(  -400.08), SIMDE_FLOAT32_C(  -173.64),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -400.08), SIMDE_FLOAT32_C(  -173.64) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128 a = simde_mm_loadu_ps(test_vec[i].a);
    simde__m512 r = simde_mm512_maskz_broadcast_f32x2(test_vec[i].k, a);
    simde_test_x86_assert_equal_f32x16(r, simde_mm512_loadu_ps(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm512_broadcast_f32x8 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float32 a[8];
    const simde_float32 r[16];
  } test_vec[] = {
    { { SIMDE_FLOAT32_C(  -416.45), SIMDE_FLOAT32_C(   711.04), SIMDE_FLOAT32_C(   494.08), SIMDE_FLOAT32_C(    55.06),
        SIMDE_FLOAT32_C(  -527.80), SIMDE_FLOAT32_C(  -810.11), SIMDE_FLOAT32_C(   486.30), SIMDE_FLOAT32_C(  -695.23) },
      { SIMDE_FLOAT32_C(  -416.45), SIMDE_FLOAT32_C(   711.04), SIMDE_FLOAT32_C(   494.08), SIMDE_FLOAT32_C(    55.06),
        SIMDE_FLOAT32_C(  -527.80), SIMDE_FLOAT32_C(  -810.11), SIMDE_FLOAT32_C(   486.30), SIMDE_FLOAT32_C(  -695.23),
        SIMDE_FLOAT32_C(  -416.45), SIMDE_FLOAT32_C(   711.04), SIMDE_FLOAT32_C(   494.08), SIMDE_FLOAT32_C(    55.06),
        SIMDE_FLOAT32_C(  -527.80), SIMDE_FLOAT32_C(  -810.11), SIMDE_FLOAT32_C(   486.30), SIMDE_FLOAT32_C(  -695.23) } },
    { { SIMDE_FLOAT32_C(  -800.88), SIMDE_FLOAT32_C(  -452.72), SIMDE_FLOAT32_C(  -904.66), SIMDE_FLOAT32_C(  -614.99),
        SIMDE_FLOAT32_C(  -172.17), SIMDE_FLOAT32_C(   311.84), SIMDE_FLOAT32_C(  -833.25), SIMDE_FLOAT32_C(  -503.53) },
      { SIMDE_FLOAT32_C(  -800.88), SIMDE_FLOAT32_C(  -452.72), SIMDE_FLOAT32_C(  -904.66), SIMDE_FLOAT32_C(  -614.99),
        SIMDE_FLOAT32_C(  -172.17), SIMDE_FLOAT32_C(   311.84), SIMDE_FLOAT32_C(  -833.25), SIMDE_FLOAT32_C(  -503.53),
        SIMDE_FLOAT32_C(  -800.88), SIMDE_FLOAT32_C(  -452.72), SIMDE_FLOAT32_C(  -904.66), SIMDE_FLOAT32_C(  -614.99),
        SIMDE_FLOAT32_C(  -172.17), SIMDE_FLOAT32_C(   311.84), SIMDE_FLOAT32_C(  -833.25), SIMDE_FLOAT32_C(  -503.53) } },
    { { SIMDE_FLOAT32_C(  -875.06), SIMDE_FLOAT32_C(   874.51), SIMDE_FLOAT32_C(  -123.24), SIMDE_FLOAT32_C(   657.48),
        SIMDE_FLOAT32_C(   309.07), SIMDE_FLOAT32_C(   484.03), SIMDE_FLOAT32_C(  -839.17), SIMDE_FLOAT32_C(    10.32) },
      { SIMDE_FLOAT32_C(  -875.06), SIMDE_FLOAT32_C(   874.51), SIMDE_FLOAT32_C(  -123.24), SIMDE_FLOAT32_C(   657.48),
        SIMDE_FLOAT32_C(   309.07), SIMDE_FLOAT32_C(   484.03), SIMDE_FLOAT32_C(  -839.17), SIMDE_FLOAT32_C(    10.32),
        SIMDE_FLOAT32_C(  -875.06), SIMDE_FLOAT32_C(   874.51), SIMDE_FLOAT32_C(  -123.24), SIMDE_FLOAT32_C(   657.48),
        SIMDE_FLOAT32_C(   309.07), SIMDE_FLOAT32_C(   484.03), SIMDE_FLOAT32_C(  -839.17), SIMDE_FLOAT32_C(    10.32) } },
    { { SIMDE_FLOAT32_C(  -515.09), SIMDE_FLOAT32_C(   924.58), SIMDE_FLOAT32_C(  -659.21), SIMDE_FLOAT32_C(   676.36),
        SIMDE_FLOAT32_C(  -421.41), SIMDE_FLOAT32_C(  -682.12), SIMDE_FLOAT32_C(  -306.00), SIMDE_FLOAT32_C(  -939.89) },
      { SIMDE_FLOAT32_C(  -515.09), SIMDE_FLOAT32_C(   924.58), SIMDE_FLOAT32_C(  -659.21), SIMDE_FLOAT32_C(   676.36),
        SIMDE_FLOAT32_C(  -421.41), SIMDE_FLOAT32_C(  -682.12), SIMDE_FLOAT32_C(  -306.00), SIMDE_FLOAT32_C(  -939.89),
        SIMDE_FLOAT32_C(  -515.09), SIMDE_FLOAT32_C(   924.58), SIMDE_FLOAT32_C(  -659.21), SIMDE_FLOAT32_C(   676.36),
        SIMDE_FLOAT32_C(  -421.41), SIMDE_FLOAT32_C(  -682.12), SIMDE_FLOAT32_C(  -306.00), SIMDE_FLOAT32_C(  -939.89) } },
    { { SIMDE_FLOAT32_C(  -812.70), SIMDE_FLOAT32_C(   906.23), SIMDE_FLOAT32_C(  -979.37), SIMDE_FLOAT32_C(  -275.20),
        SIMDE_FLOAT32_C(   664.08), SIMDE_FLOAT32_C(  -809.85), SIMDE_FLOAT32_C(   934.39), SIMDE_FLOAT32_C(   280.51) },
      { SIMDE_FLOAT32_C(  -812.70), SIMDE_FLOAT32_C(   906.23), SIMDE_FLOAT32_C(  -979.37), SIMDE_FLOAT32_C(  -275.20),
        SIMDE_FLOAT32_C(   664.08), SIMDE_FLOAT32_C(  -809.85), SIMDE_FLOAT32_C(   934.39), SIMDE_FLOAT32_C(   280.51),
        SIMDE_FLOAT32_C(  -812.70), SIMDE_FLOAT32_C(   906.23), SIMDE_FLOAT32_C(  -979.37), SIMDE_FLOAT32_C(  -275.20),
        SIMDE_FLOAT32_C(   664.08), SIMDE_FLOAT32_C(  -809.85), SIMDE_FLOAT32_C(   934.39), SIMDE_FLOAT32_C(   280.51) } },
    { { SIMDE_FLOAT32_C(   461.56), SIMDE_FLOAT32_C(  -484.84), SIMDE_FLOAT32_C(  -776.35), SIMDE_FLOAT32_C(   -37.28),
        SIMDE_FLOAT32_C(  -552.72), SIMDE_FLOAT32_C(   358.22), SIMDE_FLOAT32_C(   561.82), SIMDE_FLOAT32_C(   465.10) },
      { SIMDE_FLOAT32_C(   461.56), SIMDE_FLOAT32_C(  -484.84), SIMDE_FLOAT32_C(  -776.35), SIMDE_FLOAT32_C(   -37.28),
        SIMDE_FLOAT32_C(  -552.72), SIMDE_FLOAT32_C(   358.22), SIMDE_FLOAT32_C(   561.82), SIMDE_FLOAT32_C(   465.10),
        SIMDE_FLOAT32_C(   461.56), SIMDE_FLOAT32_C(  -484.84), SIMDE_FLOAT32_C(  -776.35), SIMDE_FLOAT32_C(   -37.28),
        SIMDE_FLOAT32_C(  -552.72), SIMDE_FLOAT32_C(   358.22), SIMDE_FLOAT32_C(   561.82), SIMDE_FLOAT32_C(   465.10) } },
    { { SIMDE_FLOAT32_C(   996.67), SIMDE_FLOAT32_C(  -908.09), SIMDE_FLOAT32_C(  -292.64), SIMDE_FLOAT32_C(  -421.79),
        SIMDE_FLOAT32_C(  -984.50), SIMDE_FLOAT32_C(  -529.88), SIMDE_FLOAT32_C(   228.67), SIMDE_FLOAT32_C(  -756.34) },
      { SIMDE_FLOAT32_C(   996.67), SIMDE_FLOAT32_C(  -908.09), SIMDE_FLOAT32_C(  -292.64), SIMDE_FLOAT32_C(  -421.79),
        SIMDE_FLOAT32_C(  -984.50), SIMDE_FLOAT32_C(  -529.88), SIMDE_FLOAT32_C(   228.67), SIMDE_FLOAT32_C(  -756.34),
        SIMDE_FLOAT32_C(   996.67), SIMDE_FLOAT32_C(  -908.09), SIMDE_FLOAT32_C(  -292.64), SIMDE_FLOAT32_C(  -421.79),
        SIMDE_FLOAT32_C(  -984.50), SIMDE_FLOAT32_C(  -529.88), SIMDE_FLOAT32_C(   228.67), SIMDE_FLOAT32_C(  -756.34) } },
    { { SIMDE_FLOAT32_C(   236.36), SIMDE_FLOAT32_C(   442.90), SIMDE_FLOAT32_C(  -175.57), SIMDE_FLOAT32_C(  -799.66),
        SIMDE_FLOAT32_C(    97.65), SIMDE_FLOAT32_C(  -822.08), SIMDE_FLOAT32_C(  -738.45), SIMDE_FLOAT32_C(   923.13) },
      { SIMDE_FLOAT32_C(   236.36), SIMDE_FLOAT32_C(   442.90), SIMDE_FLOAT32_C(  -175.57), SIMDE_FLOAT32_C(  -799.66),
        SIMDE_FLOAT32_C(    97.65), SIMDE_FLOAT32_C(  -822.08), SIMDE_FLOAT32_C(  -738.45), SIMDE_FLOAT32_C(   923.13),
        SIMDE_FLOAT32_C(   236.36), SIMDE_FLOAT32_C(   442.90), SIMDE_FLOAT32_C(  -175.57), SIMDE_FLOAT32_C(  -799.66),
        SIMDE_FLOAT32_C(    97.65), SIMDE_FLOAT32_C(  -822.08), SIMDE_FLOAT32_C(  -738.45), SIMDE_FLOAT32_C(   923.13) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256 a = simde_mm256_loadu_ps(test_vec[i].a);
    simde__m512 r = simde_mm512_broadcast_f32x8(a);
    simde_test_x86_assert_equal_f32x16(r, simde_mm512_loadu_ps(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm512_mask_broadcast_f32x8 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float32 src[16];
    const simde__mmask16 k;
    const simde_float32 a[8];
    const simde_float32 r[16];
  } test_vec[] = {
    { { SIMDE_FLOAT32_C(   280.29), SIMDE_FLOAT32_C(   838.38), SIMDE_FLOAT32_C(   622.29), SIMDE_FLOAT32_C(   762.17),
        SIMDE_FLOAT32_C(  -281.25), SIMDE_FLOAT32_C(   985.78), SIMDE_FLOAT32_C(    78.74), SIMDE_FLOAT32_C(  -555.08),
        SIMDE_FLOAT32_C(   759.89), SIMDE_FLOAT32_C(  -557.22), SIMDE_FLOAT32_C(   754.50), SIMDE_FLOAT32_C(   954.59),
        SIMDE_FLOAT32_C(  -153.57), SIMDE_FLOAT32_C(   932.38), SIMDE_FLOAT32_C(   449.83), SIMDE_FLOAT32_C(   378.57) },
      UINT16_C(36924),
      { SIMDE_FLOAT32_C(   598.39), SIMDE_FLOAT32_C(  -917.42), SIMDE_FLOAT32_C(   853.85), SIMDE_FLOAT32_C(   635.72),
        SIMDE_FLOAT32_C(   497.82), SIMDE_FLOAT32_C(   880.65), SIMDE_FLOAT32_C(  -930.36), SIMDE_FLOAT32_C(  -512.19) },
      { SIMDE_FLOAT32_C(   280.29), SIMDE_FLOAT32_C(   838.38), SIMDE_FLOAT32_C(   853.85), SIMDE_FLOAT32_C(   635.72),
        SIMDE_FLOAT32_C(   497.82), SIMDE_FLOAT32_C(   880.65), SIMDE_FLOAT32_C(    78.74), SIMDE_FLOAT32_C(  -555.08),
        SIMDE_FLOAT32_C(   759.89), SIMDE_FLOAT32_C(  -557.22), SIMDE_FLOAT32_C(   754.50), SIMDE_FLOAT32_C(   954.59),
        SIMDE_FLOAT32_C(   497.82), SIMDE_FLOAT32_C(   932.38), SIMDE_FLOAT32_C(   449.83), SIMDE_FLOAT32_C(  -512.19) } },
    { { SIMDE_FLOAT32_C(  -437.09), SIMDE_FLOAT32_C(  -187.22), SIMDE_FLOAT32_C(  -573.53), SIMDE_FLOAT32_C(   628.55),
        SIMDE_FLOAT32_C(    16.28), SIMDE_FLOAT32_C(  -343.67), SIMDE_FLOAT32_C(    13.33), SIMDE_FLOAT32_C(    92.74),
        SIMDE_FLOAT32_C(   617.88), SIMDE_FLOAT32_C(   659.02), SIMDE_FLOAT32_C(   114.72), SIMDE_FLOAT32_C(    86.74),
        SIMDE_FLOAT32_C(   -78.46), SIMDE_FLOAT32_C(  -669.19), SIMDE_FLOAT32_C(   913.81), SIMDE_FLOAT32_C(   480.88) },
      UINT16_C(25166),
      { SIMDE_FLOAT32_C(  -761.34), SIMDE_FLOAT32_C(   162.88), SIMDE_FLOAT32_C(  -410.95), SIMDE_FLOAT32_C(  -918.77),
        SIMDE_FLOAT32_C(   294.07), SIMDE_FLOAT32_C(   489.11), SIMDE_FLOAT32_C(   466.01), SIMDE_FLOAT32_C(   281.28) },
      { SIMDE_FLOAT32_C(  -437.09), SIMDE_FLOAT32_C(   162.88), SIMDE_FLOAT32_C(  -410.95), SIMDE_FLOAT32_C(  -918.77),
        SIMDE_FLOAT32_C(    16.28), SIMDE_FLOAT32_C(  -343.67), SIMDE_FLOAT32_C(   466.01), SIMDE_FLOAT32_C(    92.74),
        SIMDE_FLOAT32_C(   617.88), SIMDE_FLOAT32_C(   162.88), SIMDE_FLOAT32_C(   114.72), SIMDE_FLOAT32_C(    86.74),
        SIMDE_FLOAT32_C(   -78.46), SIMDE_FLOAT32_C(   489.11), SIMDE_FLOAT32_C(   466.01), SIMDE_FLOAT32_C(   480.88) } },
    { { SIMDE_FLOAT32_C(  -606.28), SIMDE_FLOAT32_C(   188.60), SIMDE_FLOAT32_C(  -142.85), SIMDE_FLOAT32_C(  -814.99),
        SIMDE_FLOAT32_C(   440.56), SIMDE_FLOAT32_C(   576.44), SIMDE_FLOAT32_C(   238.85), SIMDE_FLOAT32_C(   303.69),
        SIMDE_FLOAT32_C(   150.34), SIMDE_FLOAT32_C(   808.69), SIMDE_FLOAT32_C(  -362.83), SIMDE_FLOAT32_C(  -158.08),
        SIMDE_FLOAT32_C(  -803.96), SIMDE_FLOAT32_C(  -196.75), SIMDE_FLOAT32_C(  -727.89), SIMDE_FLOAT32_C(   308.53) },
      UINT16_C(23787),
      { SIMDE_FLOAT32_C(   944.26), SIMDE_FLOAT32_C(   110.45), SIMDE_FLOAT32_C(   407.09), SIMDE_FLOAT32_C(    45.91),
        SIMDE_FLOAT32_C(  -335.37), SIMDE_FLOAT32_C(  -560.84), SIMDE_FLOAT32_C(     3.97), SIMDE_FLOAT32_C(   760.14) },
      { SIMDE_FLOAT32_C(   944.26), SIMDE_FLOAT32_C(   110.45), SIMDE_FLOAT32_C(  -142.85), SIMDE_FLOAT32_C(    45.91),
        SIMDE_FLOAT32_C(   440.56), SIMDE_FLOAT32_C(  -560.84), SIMDE_FLOAT32_C(     3.97), SIMDE_FLOAT32_C(   760.14),
        SIMDE_FLOAT32_C(   150.34), SIMDE_FLOAT32_C(   808.69), SIMDE_FLOAT32_C(   407.09), SIMDE_FLOAT32_C(    45.91),
        SIMDE_FLOAT32_C(  -335.37), SIMDE_FLOAT32_C(  -196.75), SIMDE_FLOAT32_C(     3.97), SIMDE_FLOAT32_C(   308.53) } },
    { { SIMDE_FLOAT32_C(  -278.78), SIMDE_FLOAT32_C(   517.15), SIMDE_FLOAT32_C(  -283.92), SIMDE_FLOAT32_C(   114.05),
        SIMDE_FLOAT32_C(   798.05), SIMDE_FLOAT32_C(   868.23), SIMDE_FLOAT32_C(   258.92), SIMDE_FLOAT32_C(  -367.27),
        SIMDE_FLOAT32_C(  -720.23), SIMDE_FLOAT32_C(  -836.19), SIMDE_FLOAT32_C(   163.28), SIMDE_FLOAT32_C(   201.97),
        SIMDE_FLOAT32_C(   461.48), SIMDE_FLOAT32_C(    33.48), SIMDE_FLOAT32_C(   752.68), SIMDE_FLOAT32_C(   274.33) },
      UINT16_C( 9614),
      { SIMDE_FLOAT32_C(  -353.42), SIMDE_FLOAT32_C(    72.45), SIMDE_FLOAT32_C(  -313.79), SIMDE_FLOAT32_C(    54.95),
        SIMDE_FLOAT32_C(  -482.32), SIMDE_FLOAT32_C(  -268.09), SIMDE_FLOAT32_C(   146.77), SIMDE_FLOAT32_C(   772.72) },
      { SIMDE_FLOAT32_C(  -278.78), SIMDE_FLOAT32_C(    72.45), SIMDE_FLOAT32_C(  -313.79), SIMDE_FLOAT32_C(    54.95),
        SIMDE_FLOAT32_C(   798.05), SIMDE_FLOAT32_C(   868.23), SIMDE_FLOAT32_C(   258.92), SIMDE_FLOAT32_C(   772.72),
        SIMDE_FLOAT32_C(  -353.42), SIMDE_FLOAT32_C(  -836.19), SIMDE_FLOAT32_C(  -313.79), SIMDE_FLOAT32_C(   201.97),
        SIMDE_FLOAT32_C(   461.48), SIMDE_FLOAT32_C(  -268.09), SIMDE_FLOAT32_C(   752.68), SIMDE_FLOAT32_C(   274.33) } },
    { { SIMDE_FLOAT32_C(  -894.15), SIMDE_FLOAT32_C(    -6.16), SIMDE_FLOAT32_C(   455.15), SIMDE_FLOAT32_C(  -216.19),
        SIMDE_FLOAT32_C(   419.21), SIMDE_FLOAT32_C(  -283.83), SIMDE_FLOAT32_C(  -341.07), SIMDE_FLOAT32_C(  -431.79),
        SIMDE_FLOAT32_C(   825.19), SIMDE_FLOAT32_C(  -956.94), SIMDE_FLOAT32_C(   688.79), SIMDE_FLOAT32_C(   509.40),
        SIMDE_FLOAT32_C(  -511.22), SIMDE_FLOAT32_C(   -14.80), SIMDE_FLOAT32_C(  -763.30), SIMDE_FLOAT32_C(  -769.02) },
      UINT16_C(57357),
      { SIMDE_FLOAT32_C(  -152.14), SIMDE_FLOAT32_C(  -951.21), SIMDE_FLOAT32_C(   936.35), SIMDE_FLOAT32_C(  -713.46),
        SIMDE_FLOAT32_C(   933.97), SIMDE_FLOAT32_C(  -738.03), SIMDE_FLOAT32_C(     3.91), SIMDE_FLOAT32_C(  -225.68) },
      { SIMDE_FLOAT32_C(  -152.14), SIMDE_FLOAT32_C(    -6.16), SIMDE_FLOAT32_C(   936.35), SIMDE_FLOAT32_C(  -713.46),
        SIMDE_FLOAT32_C(   419.21), SIMDE_FLOAT32_C(  -283.83), SIMDE_FLOAT32_C(  -341.07), SIMDE_FLOAT32_C(  -431.79),
        SIMDE_FLOAT32_C(   825.19), SIMDE_FLOAT32_C(  -956.94), SIMDE_FLOAT32_C(   688.79), SIMDE_FLOAT32_C(   509.40),
        SIMDE_FLOAT32_C(  -511.22), SIMDE_FLOAT32_C(  -738.03), SIMDE_FLOAT32_C(     3.91), SIMDE_FLOAT32_C(  -225.68) } },
    { { SIMDE_FLOAT32_C(   958.35), SIMDE_FLOAT32_C(   959.55), SIMDE_FLOAT32_C(  -771.84), SIMDE_FLOAT32_C(  -312.71),
        SIMDE_FLOAT32_C(   261.02), SIMDE_FLOAT32_C(  -965.72), SIMDE_FLOAT32_C(  -898.55), SIMDE_FLOAT32_C(    98.86),
        SIMDE_FLOAT32_C(  -506.78), SIMDE_FLOAT32_C(   475.13), SIMDE_FLOAT32_C(  -561.78), SIMDE_FLOAT32_C(   145.04),
        SIMDE_FLOAT32_C(  -310.71), SIMDE_FLOAT32_C(  -100.99), SIMDE_FLOAT32_C(   656.93), SIMDE_FLOAT32_C(   955.62) },
      UINT16_C(55637),
      { SIMDE_FLOAT32_C(    64.66), SIMDE_FLOAT32_C(   704.14), SIMDE_FLOAT32_C(   421.81), SIMDE_FLOAT32_C(  -620.94),
        SIMDE_FLOAT32_C(  -124.06), SIMDE_FLOAT32_C(   858.04), SIMDE_FLOAT32_C(  -855.91), SIMDE_FLOAT32_C(   691.15) },
      { SIMDE_FLOAT32_C(    64.66), SIMDE_FLOAT32_C(   959.55), SIMDE_FLOAT32_C(   421.81), SIMDE_FLOAT32_C(  -312.71),
        SIMDE_FLOAT32_C(  -124.06), SIMDE_FLOAT32_C(  -965.72), SIMDE_FLOAT32_C(  -855.91), SIMDE_FLOAT32_C(    98.86),
        SIMDE_FLOAT32_C(    64.66), SIMDE_FLOAT32_C(   475.13), SIMDE_FLOAT32_C(  -561.78), SIMDE_FLOAT32_C(  -620.94),
        SIMDE_FLOAT32_C(  -124.06), SIMDE_FLOAT32_C(  -100.99), SIMDE_FLOAT32_C(  -855.91), SIMDE_FLOAT32_C(   691.15) } },
    { { SIMDE_FLOAT32_C(   165.52), SIMDE_FLOAT32_C(  -117.15), SIMDE_FLOAT32_C(  -914.50), SIMDE_FLOAT32_C(   -48.64),
        SIMDE_FLOAT32_C(   429.74), SIMDE_FLOAT32_C(   612.18), SIMDE_FLOAT32_C(   933.85), SIMDE_FLOAT32_C(  -778.14),
        SIMDE_FLOAT32_C(  -214.40), SIMDE_FLOAT32_C(   623.77), SIMDE_FLOAT32_C(  -288.84), SIMDE_FLOAT32_C(  -541.76),
        SIMDE_FLOAT32_C(   699.14), SIMDE_FLOAT32_C(   473.09), SIMDE_FLOAT32_C(  -762.45), SIMDE_FLOAT32_C(  -518.42) },
      UINT16_C(63181),
      { SIMDE_FLOAT32_C(   188.68), SIMDE_FLOAT32_C(  -923.58), SIMDE_FLOAT32_C(  -542.98), SIMDE_FLOAT32_C(   193.71),
        SIMDE_FLOAT32_C(  -319.51), SIMDE_FLOAT32_C(    46.76), SIMDE_FLOAT32_C(   -44.67), SIMDE_FLOAT32_C(  -768.90) },
      { SIMDE_FLOAT32_C(   188.68), SIMDE_FLOAT32_C(  -117.15), SIMDE_FLOAT32_C(  -542.98), SIMDE_FLOAT32_C(   193.71),
        SIMDE_FLOAT32_C(   429.74), SIMDE_FLOAT32_C(   612.18), SIMDE_FLOAT32_C(   -44.67), SIMDE_FLOAT32_C(  -768.90),
        SIMDE_FLOAT32_C(  -214.40), SIMDE_FLOAT32_C(  -923.58), SIMDE_FLOAT32_C(  -542.98), SIMDE_FLOAT32_C(  -541.76),
        SIMDE_FLOAT32_C(  -319.51), SIMDE_FLOAT32_C(    46.76), SIMDE_FLOAT32_C(   -44.67), SIMDE_FLOAT32_C(  -768.90) } },
    { { SIMDE_FLOAT32_C(  -857.07), SIMDE_FLOAT32_C(  -775.77), SIMDE_FLOAT32_C(  -351.82), SIMDE_FLOAT32_C(   984.69),
        SIMDE_FLOAT32_C(  -320.14), SIMDE_FLOAT32_C(  -636.62), SIMDE_FLOAT32_C(   297.63), SIMDE_FLOAT32_C(   186.04),
        SIMDE_FLOAT32_C(   780.35), SIMDE_FLOAT32_C(  -693.20), SIMDE_FLOAT32_C(  -589.12), SIMDE_FLOAT32_C(   731.33),
        SIMDE_FLOAT32_C(  -601.90), SIMDE_FLOAT32_C(  -195.41), SIMDE_FLOAT32_C(  -239.98), SIMDE_FLOAT32_C(   675.16) },
      UINT16_C(63687),
      { SIMDE_FLOAT32_C(   751.41), SIMDE_FLOAT32_C(   926.41), SIMDE_FLOAT32_C(   149.18), SIMDE_FLOAT32_C(  -662.14),
        SIMDE_FLOAT32_C(  -649.07), SIMDE_FLOAT32_C(  -858.90), SIMDE_FLOAT32_C(   465.33), SIMDE_FLOAT32_C(   831.66) },
      { SIMDE_FLOAT32_C(   751.41), SIMDE_FLOAT32_C(   926.41), SIMDE_FLOAT32_C(   149.18), SIMDE_FLOAT32_C(   984.69),
        SIMDE_FLOAT32_C(  -320.14), SIMDE_FLOAT32_C(  -636.62), SIMDE_FLOAT32_C(   465.33), SIMDE_FLOAT32_C(   831.66),
        SIMDE_FLOAT32_C(   780.35), SIMDE_FLOAT32_C(  -693.20), SIMDE_FLOAT32_C(  -589.12), SIMDE_FLOAT32_C(  -662.14),
        SIMDE_FLOAT32_C(  -649.07), SIMDE_FLOAT32_C(  -858.90), SIMDE_FLOAT32_C(   465.33), SIMDE_FLOAT32_C(   831.66) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512 src = simde_mm512_loadu_ps(test_vec[i].src);
    simde__m256 a = simde_mm256_loadu_ps(test_vec[i].a);
    simde__m512 r = simde_mm512_mask_broadcast_f32x8(src, test_vec[i].k, a);
    simde_test_x86_assert_equal_f32x16(r, simde_mm512_loadu_ps(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm512_maskz_broadcast_f32x8 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde__mmask16 k;
    const simde_float32 a[8];
    const simde_float32 r[16];
  } test_vec[] = {
    { UINT16_C(49062),
      { SIMDE_FLOAT32_C(   -67.12), SIMDE_FLOAT32_C(  -144.98), SIMDE_FLOAT32_C(  -693.09), SIMDE_FLOAT32_C(  -717.03),
        SIMDE_FLOAT32_C(   833.33), SIMDE_FLOAT32_C(  -297.62), SIMDE_FLOAT32_C(  -166.55), SIMDE_FLOAT32_C(   748.74) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -144.98), SIMDE_FLOAT32_C(  -693.09), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -297.62), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   748.74),
        SIMDE_FLOAT32_C(   -67.12), SIMDE_FLOAT32_C(  -144.98), SIMDE_FLOAT32_C(  -693.09), SIMDE_FLOAT32_C(  -717.03),
        SIMDE_FLOAT32_C(   833.33), SIMDE_FLOAT32_C(  -297.62), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   748.74) } },
    { UINT16_C( 6586),
      { SIMDE_FLOAT32_C(  -140.91), SIMDE_FLOAT32_C(  -189.72), SIMDE_FLOAT32_C(  -663.50), SIMDE_FLOAT32_C(   613.12),
        SIMDE_FLOAT32_C(   772.89), SIMDE_FLOAT32_C(   -76.35), SIMDE_FLOAT32_C(   859.08), SIMDE_FLOAT32_C(   595.36) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -189.72), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   613.12),
        SIMDE_FLOAT32_C(   772.89), SIMDE_FLOAT32_C(   -76.35), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   595.36),
        SIMDE_FLOAT32_C(  -140.91), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   613.12),
        SIMDE_FLOAT32_C(   772.89), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00) } },
    { UINT16_C(41380),
      { SIMDE_FLOAT32_C(  -218.39), SIMDE_FLOAT32_C(  -397.45), SIMDE_FLOAT32_C(    20.87), SIMDE_FLOAT32_C(   703.15),
        SIMDE_FLOAT32_C(  -126.69), SIMDE_FLOAT32_C(   776.77), SIMDE_FLOAT32_C(  -820.00), SIMDE_FLOAT32_C(   252.00) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    20.87), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   776.77), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   252.00),
        SIMDE_FLOAT32_C(  -218.39), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   776.77), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   252.00) } },
    { UINT16_C(14746),
      { SIMDE_FLOAT32_C(   488.59), SIMDE_FLOAT32_C(  -333.19), SIMDE_FLOAT32_C(    82.99), SIMDE_FLOAT32_C(   818.76),
        SIMDE_FLOAT32_C(   927.98), SIMDE_FLOAT32_C(   586.60), SIMDE_FLOAT32_C(   933.90), SIMDE_FLOAT32_C(    84.47) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -333.19), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   818.76),
        SIMDE_FLOAT32_C(   927.98), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    84.47),
        SIMDE_FLOAT32_C(   488.59), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   818.76),
        SIMDE_FLOAT32_C(   927.98), SIMDE_FLOAT32_C(   586.60), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00) } },
    { UINT16_C(22430),
      { SIMDE_FLOAT32_C(  -788.60), SIMDE_FLOAT32_C(    -2.38), SIMDE_FLOAT32_C(   -57.26), SIMDE_FLOAT32_C(  -363.40),
        SIMDE_FLOAT32_C(   348.91), SIMDE_FLOAT32_C(   172.83), SIMDE_FLOAT32_C(   816.49), SIMDE_FLOAT32_C(   677.29) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    -2.38), SIMDE_FLOAT32_C(   -57.26), SIMDE_FLOAT32_C(  -363.40),
        SIMDE_FLOAT32_C(   348.91), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   677.29),
        SIMDE_FLOAT32_C(  -788.60), SIMDE_FLOAT32_C(    -2.38), SIMDE_FLOAT32_C(   -57.26), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(   348.91), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   816.49), SIMDE_FLOAT32_C(     0.00) } },
    { UINT16_C(53747),
      { SIMDE_FLOAT32_C(   -74.10), SIMDE_FLOAT32_C(   628.20), SIMDE_FLOAT32_C(   176.40), SIMDE_FLOAT32_C(   789.58),
        SIMDE_FLOAT32_C(   434.02), SIMDE_FLOAT32_C(   537.30), SIMDE_FLOAT32_C(   360.66), SIMDE_FLOAT32_C(  -306.64) },
      { SIMDE_FLOAT32_C(   -74.10), SIMDE_FLOAT32_C(   628.20), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(   434.02), SIMDE_FLOAT32_C(   537.30), SIMDE_FLOAT32_C(   360.66), SIMDE_FLOAT32_C(  -306.64),
        SIMDE_FLOAT32_C(   -74.10), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(   434.02), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   360.66), SIMDE_FLOAT32_C(  -306.64) } },
    { UINT16_C(57660),
      { SIMDE_FLOAT32_C(   529.43), SIMDE_FLOAT32_C(   185.72), SIMDE_FLOAT32_C(  -666.37), SIMDE_FLOAT32_C(   372.37),
        SIMDE_FLOAT32_C(   420.53), SIMDE_FLOAT32_C(   -76.09), SIMDE_FLOAT32_C(  -764.18), SIMDE_FLOAT32_C(   472.62) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -666.37), SIMDE_FLOAT32_C(   372.37),
        SIMDE_FLOAT32_C(   420.53), SIMDE_FLOAT32_C(   -76.09), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(   529.43), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   -76.09), SIMDE_FLOAT32_C(  -764.18), SIMDE_FLOAT32_C(   472.62) } },
    { UINT16_C(60506),
      { SIMDE_FLOAT32_C(  -796.21), SIMDE_FLOAT32_C(   148.32), SIMDE_FLOAT32_C(   781.59), SIMDE_FLOAT32_C(   218.77),
        SIMDE_FLOAT32_C(   802.35), SIMDE_FLOAT32_C(  -915.03), SIMDE_FLOAT32_C(  -953.21), SIMDE_FLOAT32_C(  -530.25) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   148.32), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   218.77),
        SIMDE_FLOAT32_C(   802.35), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -953.21), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   781.59), SIMDE_FLOAT32_C(   218.77),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -915.03), SIMDE_FLOAT32_C(  -953.21), SIMDE_FLOAT32_C(  -530.25) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256 a = simde_mm256_loadu_ps(test_vec[i].a);
    simde__m512 r = simde_mm512_maskz_broadcast_f32x8(test_vec[i].k, a);
    simde_test_x86_assert_equal_f32x16(r, simde_mm512_loadu_ps(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm512_broadcast_f64x2 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float64 a[2];
    const simde_float64 r[8];
  } test_vec[] = {
    { { SIMDE_FLOAT64_C(   715.32), SIMDE_FLOAT64_C(   317.58) },
      { SIMDE_FLOAT64_C(   715.32), SIMDE_FLOAT64_C(   317.58), SIMDE_FLOAT64_C(   715.32), SIMDE_FLOAT64_C(   317.58),
        SIMDE_FLOAT64_C(   715.32), SIMDE_FLOAT64_C(   317.58), SIMDE_FLOAT64_C(   715.32), SIMDE_FLOAT64_C(   317.58) } },
    { { SIMDE_FLOAT64_C(  -404.76), SIMDE_FLOAT64_C(  -835.80) },
      { SIMDE_FLOAT64_C(  -404.76), SIMDE_FLOAT64_C(  -835.80), SIMDE_FLOAT64_C(  -404.76), SIMDE_FLOAT64_C(  -835.80),
        SIMDE_FLOAT64_C(  -404.76), SIMDE_FLOAT64_C(  -835.80), SIMDE_FLOAT64_C(  -404.76), SIMDE_FLOAT64_C(  -835.80) } },
    { { SIMDE_FLOAT64_C(   653.97), SIMDE_FLOAT64_C(  -774.97) },
      { SIMDE_FLOAT64_C(   653.97), SIMDE_FLOAT64_C(  -774.97), SIMDE_FLOAT64_C(   653.97), SIMDE_FLOAT64_C(  -774.97),
        SIMDE_FLOAT64_C(   653.97), SIMDE_FLOAT64_C(  -774.97), SIMDE_FLOAT64_C(   653.97), SIMDE_FLOAT64_C(  -774.97) } },
    { { SIMDE_FLOAT64_C(  -843.04), SIMDE_FLOAT64_C(  -900.71) },
      { SIMDE_FLOAT64_C(  -843.04), SIMDE_FLOAT64_C(  -900.71), SIMDE_FLOAT64_C(  -843.04), SIMDE_FLOAT64_C(  -900.71),
        SIMDE_FLOAT64_C(  -843.04), SIMDE_FLOAT64_C(  -900.71), SIMDE_FLOAT64_C(  -843.04), SIMDE_FLOAT64_C(  -900.71) } },
    { { SIMDE_FLOAT64_C(  -197.71), SIMDE_FLOAT64_C(  -989.91) },
      { SIMDE_FLOAT64_C(  -197.71), SIMDE_FLOAT64_C(  -989.91), SIMDE_FLOAT64_C(  -197.71), SIMDE_FLOAT64_C(  -989.91),
        SIMDE_FLOAT64_C(  -197.71), SIMDE_FLOAT64_C(  -989.91), SIMDE_FLOAT64_C(  -197.71), SIMDE_FLOAT64_C(  -989.91) } },
    { { SIMDE_FLOAT64_C(   515.43), SIMDE_FLOAT64_C(   879.19) },
      { SIMDE_FLOAT64_C(   515.43), SIMDE_FLOAT64_C(   879.19), SIMDE_FLOAT64_C(   515.43), SIMDE_FLOAT64_C(   879.19),
        SIMDE_FLOAT64_C(   515.43), SIMDE_FLOAT64_C(   879.19), SIMDE_FLOAT64_C(   515.43), SIMDE_FLOAT64_C(   879.19) } },
    { { SIMDE_FLOAT64_C(   610.61), SIMDE_FLOAT64_C(   540.00) },
      { SIMDE_FLOAT64_C(   610.61), SIMDE_FLOAT64_C(   540.00), SIMDE_FLOAT64_C(   610.61), SIMDE_FLOAT64_C(   540.00),
        SIMDE_FLOAT64_C(   610.61), SIMDE_FLOAT64_C(   540.00), SIMDE_FLOAT64_C(   610.61), SIMDE_FLOAT64_C(   540.00) } },
    { { SIMDE_FLOAT64_C(  -234.86), SIMDE_FLOAT64_C(   751.29) },
      { SIMDE_FLOAT64_C(  -234.86), SIMDE_FLOAT64_C(   751.29), SIMDE_FLOAT64_C(  -234.86), SIMDE_FLOAT64_C(   751.29),
        SIMDE_FLOAT64_C(  -234.86), SIMDE_FLOAT64_C(   751.29), SIMDE_FLOAT64_C(  -234.86), SIMDE_FLOAT64_C(   751.29) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128d a = simde_mm_loadu_pd(test_vec[i].a);
    simde__m512d r = simde_mm512_broadcast_f64x2(a);
    simde_test_x86_assert_equal_f64x8(r, simde_mm512_loadu_pd(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm512_mask_broadcast_f64x2 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float64 src[8];
    const simde__mmask8 k;
    const simde_float64 a[2];
    const simde_float64 r[8];
  } test_vec[] = {
    { { SIMDE_FLOAT64_C(   749.13), SIMDE_FLOAT64_C(   675.39), SIMDE_FLOAT64_C(  -739.63), SIMDE_FLOAT64_C(  -916.53),
        SIMDE_FLOAT64_C(   -70.94), SIMDE_FLOAT64_C(  -224.46), SIMDE_FLOAT64_C(  -485.72), SIMDE_FLOAT64_C(   433.96) },
      UINT8_C(250),
      { SIMDE_FLOAT64_C(   424.19), SIMDE_FLOAT64_C(  -720.98) },
      { SIMDE_FLOAT64_C(   749.13), SIMDE_FLOAT64_C(  -720.98), SIMDE_FLOAT64_C(  -739.63), SIMDE_FLOAT64_C(  -720.98),
        SIMDE_FLOAT64_C(   424.19), SIMDE_FLOAT64_C(  -720.98), SIMDE_FLOAT64_C(   424.19), SIMDE_FLOAT64_C(  -720.98) } },
    { { SIMDE_FLOAT64_C(   461.33), SIMDE_FLOAT64_C(  -402.24), SIMDE_FLOAT64_C(  -437.75), SIMDE_FLOAT64_C(   785.96),
        SIMDE_FLOAT64_C(  -372.46), SIMDE_FLOAT64_C(   110.74), SIMDE_FLOAT64_C(  -831.39), SIMDE_FLOAT64_C(   846.99) },
      UINT8_C( 78),
      { SIMDE_FLOAT64_C(  -572.48), SIMDE_FLOAT64_C(   394.61) },
      { SIMDE_FLOAT64_C(   461.33), SIMDE_FLOAT64_C(   394.61), SIMDE_FLOAT64_C(  -572.48), SIMDE_FLOAT64_C(   394.61),
        SIMDE_FLOAT64_C(  -372.46), SIMDE_FLOAT64_C(   110.74), SIMDE_FLOAT64_C(  -572.48), SIMDE_FLOAT64_C(   846.99) } },
    { { SIMDE_FLOAT64_C(   215.35), SIMDE_FLOAT64_C(  -616.54), SIMDE_FLOAT64_C(  -262.30), SIMDE_FLOAT64_C(  -426.39),
        SIMDE_FLOAT64_C(  -336.22), SIMDE_FLOAT64_C(  -839.02), SIMDE_FLOAT64_C(   672.49), SIMDE_FLOAT64_C(   589.70) },
      UINT8_C(163),
      { SIMDE_FLOAT64_C(  -982.23), SIMDE_FLOAT64_C(  -416.77) },
      { SIMDE_FLOAT64_C(  -982.23), SIMDE_FLOAT64_C(  -416.77), SIMDE_FLOAT64_C(  -262.30), SIMDE_FLOAT64_C(  -426.39),
        SIMDE_FLOAT64_C(  -336.22), SIMDE_FLOAT64_C(  -416.77), SIMDE_FLOAT64_C(   672.49), SIMDE_FLOAT64_C(  -416.77) } },
    { { SIMDE_FLOAT64_C(  -578.35), SIMDE_FLOAT64_C(  -267.73), SIMDE_FLOAT64_C(   242.90), SIMDE_FLOAT64_C(   449.74),
        SIMDE_FLOAT64_C(   714.62), SIMDE_FLOAT64_C(   671.90), SIMDE_FLOAT64_C(   577.25), SIMDE_FLOAT64_C(   -88.86) },
      UINT8_C(222),
      { SIMDE_FLOAT64_C(   379.16), SIMDE_FLOAT64_C(   573.95) },
      { SIMDE_FLOAT64_C(  -578.35), SIMDE_FLOAT64_C(   573.95), SIMDE_FLOAT64_C(   379.16), SIMDE_FLOAT64_C(   573.95),
        SIMDE_FLOAT64_C(   379.16), SIMDE_FLOAT64_C(   671.90), SIMDE_FLOAT64_C(   379.16), SIMDE_FLOAT64_C(   573.95) } },
    { { SIMDE_FLOAT64_C(   428.10), SIMDE_FLOAT64_C(  -969.60), SIMDE_FLOAT64_C(  -117.58), SIMDE_FLOAT64_C(  -121.88),
        SIMDE_FLOAT64_C(  -513.12), SIMDE_FLOAT64_C(   -67.52), SIMDE_FLOAT64_C(  -880.81), SIMDE_FLOAT64_C(   257.25) },
      UINT8_C( 35),
      { SIMDE_FLOAT64_C(   -71.92), SIMDE_FLOAT64_C(  -682.64) },
      { SIMDE_FLOAT64_C(   -71.92), SIMDE_FLOAT64_C(  -682.64), SIMDE_FLOAT64_C(  -117.58), SIMDE_FLOAT64_C(  -121.88),
        SIMDE_FLOAT64_C(  -513.12), SIMDE_FLOAT64_C(  -682.64), SIMDE_FLOAT64_C(  -880.81), SIMDE_FLOAT64_C(   257.25) } },
    { { SIMDE_FLOAT64_C(   858.06), SIMDE_FLOAT64_C(  -576.56), SIMDE_FLOAT64_C(  -199.04), SIMDE_FLOAT64_C(   741.89),
        SIMDE_FLOAT64_C(   940.66), SIMDE_FLOAT64_C(  -320.73), SIMDE_FLOAT64_C(  -519.45), SIMDE_FLOAT64_C(  -359.73) },
      UINT8_C( 14),
      { SIMDE_FLOAT64_C(  -260.24), SIMDE_FLOAT64_C(   150.09) },
      { SIMDE_FLOAT64_C(   858.06), SIMDE_FLOAT64_C(   150.09), SIMDE_FLOAT64_C(  -260.24), SIMDE_FLOAT64_C(   150.09),
        SIMDE_FLOAT64_C(   940.66), SIMDE_FLOAT64_C(  -320.73), SIMDE_FLOAT64_C(  -519.45), SIMDE_FLOAT64_C(  -359.73) } },
    { { SIMDE_FLOAT64_C(   508.76), SIMDE_FLOAT64_C(   671.76), SIMDE_FLOAT64_C(   188.22), SIMDE_FLOAT64_C(  -524.84),
        SIMDE_FLOAT64_C(   958.74), SIMDE_FLOAT64_C(  -408.21), SIMDE_FLOAT64_C(  -756.34), SIMDE_FLOAT64_C(   260.63) },
      UINT8_C( 48),
      { SIMDE_FLOAT64_C(  -287.86), SIMDE_FLOAT64_C(   -66.95) },
      { SIMDE_FLOAT64_C(   508.76), SIMDE_FLOAT64_C(   671.76), SIMDE_FLOAT64_C(   188.22), SIMDE_FLOAT64_C(  -524.84),
        SIMDE_FLOAT64_C(  -287.86), SIMDE_FLOAT64_C(   -66.95), SIMDE_FLOAT64_C(  -756.34), SIMDE_FLOAT64_C(   260.63) } },
    { { SIMDE_FLOAT64_C(   741.62), SIMDE_FLOAT64_C(   389.31), SIMDE_FLOAT64_C(  -806.05), SIMDE_FLOAT64_C(   761.48),
        SIMDE_FLOAT64_C(   242.55), SIMDE_FLOAT64_C(   550.14), SIMDE_FLOAT64_C(   214.54), SIMDE_FLOAT64_C(  -176.03) },
      UINT8_C( 79),
      { SIMDE_FLOAT64_C(   639.90), SIMDE_FLOAT64_C(   881.52) },
      { SIMDE_FLOAT64_C(   639.90), SIMDE_FLOAT64_C(   881.52), SIMDE_FLOAT64_C(   639.90), SIMDE_FLOAT64_C(   881.52),
        SIMDE_FLOAT64_C(   242.55), SIMDE_FLOAT64_C(   550.14), SIMDE_FLOAT64_C(   639.90), SIMDE_FLOAT64_C(  -176.03) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512d src = simde_mm512_loadu_pd(test_vec[i].src);
    simde__m128d a = simde_mm_loadu_pd(test_vec[i].a);
    simde__m512d r = simde_mm512_mask_broadcast_f64x2(src, test_vec[i].k, a);
    simde_test_x86_assert_equal_f64x8(r, simde_mm512_loadu_pd(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm512_maskz_broadcast_f64x2 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde__mmask8 k;
    const simde_float64 a[2];
    const simde_float64 r[8];
  } test_vec[] = {
    { UINT8_C( 32),
      { SIMDE_FLOAT64_C(    95.43), SIMDE_FLOAT64_C(  -111.80) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00),
        SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(  -111.80), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00) } },
    { UINT8_C(212),
      { SIMDE_FLOAT64_C(   159.26), SIMDE_FLOAT64_C(   721.63) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(   159.26), SIMDE_FLOAT64_C(     0.00),
        SIMDE_FLOAT64_C(   159.26), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(   159.26), SIMDE_FLOAT64_C(   721.63) } },
    { UINT8_C(232),
      { SIMDE_FLOAT64_C(   -41.02), SIMDE_FLOAT64_C(   592.81) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(   592.81),
        SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(   592.81), SIMDE_FLOAT64_C(   -41.02), SIMDE_FLOAT64_C(   592.81) } },
    { UINT8_C(112),
      { SIMDE_FLOAT64_C(    80.26), SIMDE_FLOAT64_C(   969.51) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00),
        SIMDE_FLOAT64_C(    80.26), SIMDE_FLOAT64_C(   969.51), SIMDE_FLOAT64_C(    80.26), SIMDE_FLOAT64_C(     0.00) } },
    { UINT8_C(215),
      { SIMDE_FLOAT64_C(   905.16), SIMDE_FLOAT64_C(  -968.55) },
      { SIMDE_FLOAT64_C(   905.16), SIMDE_FLOAT64_C(  -968.55), SIMDE_FLOAT64_C(   905.16), SIMDE_FLOAT64_C(     0.00),
        SIMDE_FLOAT64_C(   905.16), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(   905.16), SIMDE_FLOAT64_C(  -968.55) } },
    { UINT8_C(135),
      { SIMDE_FLOAT64_C(   140.43), SIMDE_FLOAT64_C(   267.82) },
      { SIMDE_FLOAT64_C(   140.43), SIMDE_FLOAT64_C(   267.82), SIMDE_FLOAT64_C(   140.43), SIMDE_FLOAT64_C(     0.00),
        SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(   267.82) } },
    { UINT8_C(192),
      { SIMDE_FLOAT64_C(  -853.88), SIMDE_FLOAT64_C(   811.68) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00),
        SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(  -853.88), SIMDE_FLOAT64_C(   811.68) } },
    { UINT8_C( 17),
      { SIMDE_FLOAT64_C(  -661.24), SIMDE_FLOAT64_C(   561.84) },
      { SIMDE_FLOAT64_C(  -661.24), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00),
        SIMDE_FLOAT64_C(  -661.24), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128d a = simde_mm_loadu_pd(test_vec[i].a);
    simde__m512d r = simde_mm512_maskz_broadcast_f64x2(test_vec[i].k, a);
    simde_test_x86_assert_equal_f64x8(r, simde_mm512_loadu_pd(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm256_broadcast_f32x4 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float32 a[4];
    const simde_float32 r[8];
  } test_vec[] = {
    { { SIMDE_FLOAT32_C(  -558.39), SIMDE_FLOAT32_C(  -943.50), SIMDE_FLOAT32_C(   652.52), SIMDE_FLOAT32_C(   945.52) },
      { SIMDE_FLOAT32_C(  -558.39), SIMDE_FLOAT32_C(  -943.50), SIMDE_FLOAT32_C(   652.52), SIMDE_FLOAT32_C(   945.52),
        SIMDE_FLOAT32_C(  -558.39), SIMDE_FLOAT32_C(  -943.50), SIMDE_FLOAT32_C(   652.52), SIMDE_FLOAT32_C(   945.52) } },
    { { SIMDE_FLOAT32_C(  -577.06), SIMDE_FLOAT32_C(  -623.59), SIMDE_FLOAT32_C(  -742.48), SIMDE_FLOAT32_C(  -807.52) },
      { SIMDE_FLOAT32_C(  -577.06), SIMDE_FLOAT32_C(  -623.59), SIMDE_FLOAT32_C(  -742.48), SIMDE_FLOAT32_C(  -807.52),
        SIMDE_FLOAT32_C(  -577.06), SIMDE_FLOAT32_C(  -623.59), SIMDE_FLOAT32_C(  -742.48), SIMDE_FLOAT32_C(  -807.52) } },
    { { SIMDE_FLOAT32_C(     0.46), SIMDE_FLOAT32_C(  -222.35), SIMDE_FLOAT32_C(   965.41), SIMDE_FLOAT32_C(  -320.94) },
      { SIMDE_FLOAT32_C(     0.46), SIMDE_FLOAT32_C(  -222.35), SIMDE_FLOAT32_C(   965.41), SIMDE_FLOAT32_C(  -320.94),
        SIMDE_FLOAT32_C(     0.46), SIMDE_FLOAT32_C(  -222.35), SIMDE_FLOAT32_C(   965.41), SIMDE_FLOAT32_C(  -320.94) } },
    { { SIMDE_FLOAT32_C(    34.85), SIMDE_FLOAT32_C(  -238.64), SIMDE_FLOAT32_C(  -834.61), SIMDE_FLOAT32_C(   763.48) },
      { SIMDE_FLOAT32_C(    34.85), SIMDE_FLOAT32_C(  -238.64), SIMDE_FLOAT32_C(  -834.61), SIMDE_FLOAT32_C(   763.48),
        SIMDE_FLOAT32_C(    34.85), SIMDE_FLOAT32_C(  -238.64), SIMDE_FLOAT32_C(  -834.61), SIMDE_FLOAT32_C(   763.48) } },
    { { SIMDE_FLOAT32_C(  -215.99), SIMDE_FLOAT32_C(  -214.29), SIMDE_FLOAT32_C(   432.66), SIMDE_FLOAT32_C(  -222.94) },
      { SIMDE_FLOAT32_C(  -215.99), SIMDE_FLOAT32_C(  -214.29), SIMDE_FLOAT32_C(   432.66), SIMDE_FLOAT32_C(  -222.94),
        SIMDE_FLOAT32_C(  -215.99), SIMDE_FLOAT32_C(  -214.29), SIMDE_FLOAT32_C(   432.66), SIMDE_FLOAT32_C(  -222.94) } },
    { { SIMDE_FLOAT32_C(  -994.85), SIMDE_FLOAT32_C(  -413.17), SIMDE_FLOAT32_C(  -100.86), SIMDE_FLOAT32_C(   836.37) },
      { SIMDE_FLOAT32_C(  -994.85), SIMDE_FLOAT32_C(  -413.17), SIMDE_FLOAT32_C(  -100.86), SIMDE_FLOAT32_C(   836.37),
        SIMDE_FLOAT32_C(  -994.85), SIMDE_FLOAT32_C(  -413.17), SIMDE_FLOAT32_C(  -100.86), SIMDE_FLOAT32_C(   836.37) } },
    { { SIMDE_FLOAT32_C(   809.63), SIMDE_FLOAT32_C(  -520.84), SIMDE_FLOAT32_C(   265.00), SIMDE_FLOAT32_C(  -111.67) },
      { SIMDE_FLOAT32_C(   809.63), SIMDE_FLOAT32_C(  -520.84), SIMDE_FLOAT32_C(   265.00), SIMDE_FLOAT32_C(  -111.67),
        SIMDE_FLOAT32_C(   809.63), SIMDE_FLOAT32_C(  -520.84), SIMDE_FLOAT32_C(   265.00), SIMDE_FLOAT32_C(  -111.67) } },
    { { SIMDE_FLOAT32_C(  -855.41), SIMDE_FLOAT32_C(  -875.73), SIMDE_FLOAT32_C(  -447.77), SIMDE_FLOAT32_C(   263.25) },
      { SIMDE_FLOAT32_C(  -855.41), SIMDE_FLOAT32_C(  -875.73), SIMDE_FLOAT32_C(  -447.77), SIMDE_FLOAT32_C(   263.25),
        SIMDE_FLOAT32_C(  -855.41), SIMDE_FLOAT32_C(  -875.73), SIMDE_FLOAT32_C(  -447.77), SIMDE_FLOAT32_C(   263.25) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128 a = simde_mm_loadu_ps(test_vec[i].a);
    simde__m256 r = simde_mm256_broadcast_f32x4(a);
    simde_test_x86_assert_equal_f32x8(r, simde_mm256_loadu_ps(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm256_mask_broadcast_f32x4 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float32 src[8];
    const simde__mmask8 k;
    const simde_float32 a[4];
    const simde_float32 r[8];
  } test_vec[] = {
    { { SIMDE_FLOAT32_C(   113.43), SIMDE_FLOAT32_C(   410.75), SIMDE_FLOAT32_C(  -451.88), SIMDE_FLOAT32_C(  -725.60),
        SIMDE_FLOAT32_C(   921.94), SIMDE_FLOAT32_C(  -987.53), SIMDE_FLOAT32_C(   590.45), SIMDE_FLOAT32_C(  -298.17) },
      UINT8_C( 50),
      { SIMDE_FLOAT32_C(    44.31), SIMDE_FLOAT32_C(   797.52), SIMDE_FLOAT32_C(  -107.60), SIMDE_FLOAT32_C(  -484.17) },
      { SIMDE_FLOAT32_C(   113.43), SIMDE_FLOAT32_C(   797.52), SIMDE_FLOAT32_C(  -451.88), SIMDE_FLOAT32_C(  -725.60),
        SIMDE_FLOAT32_C(    44.31), SIMDE_FLOAT32_C(   797.52), SIMDE_FLOAT32_C(   590.45), SIMDE_FLOAT32_C(  -298.17) } },
    { { SIMDE_FLOAT32_C(   556.86), SIMDE_FLOAT32_C(  -797.02), SIMDE_FLOAT32_C(   402.24), SIMDE_FLOAT32_C(   441.25),
        SIMDE_FLOAT32_C(   142.97), SIMDE_FLOAT32_C(   883.64), SIMDE_FLOAT32_C(  -635.48), SIMDE_FLOAT32_C(  -488.89) },
      UINT8_C(165),
      { SIMDE_FLOAT32_C(  -333.03), SIMDE_FLOAT32_C(   703.87), SIMDE_FLOAT32_C(   -69.82), SIMDE_FLOAT32_C(   527.07) },
      { SIMDE_FLOAT32_C(  -333.03), SIMDE_FLOAT32_C(  -797.02), SIMDE_FLOAT32_C(   -69.82), SIMDE_FLOAT32_C(   441.25),
        SIMDE_FLOAT32_C(   142.97), SIMDE_FLOAT32_C(   703.87), SIMDE_FLOAT32_C(  -635.48), SIMDE_FLOAT32_C(   527.07) } },
    { { SIMDE_FLOAT32_C(   425.48), SIMDE_FLOAT32_C(   960.83), SIMDE_FLOAT32_C(   698.87), SIMDE_FLOAT32_C(  -175.48),
        SIMDE_FLOAT32_C(   789.83), SIMDE_FLOAT32_C(   633.19), SIMDE_FLOAT32_C(    85.22), SIMDE_FLOAT32_C(   351.45) },
      UINT8_C(206),
      { SIMDE_FLOAT32_C(   362.09), SIMDE_FLOAT32_C(  -387.94), SIMDE_FLOAT32_C(   -58.09), SIMDE_FLOAT32_C(  -381.37) },
      { SIMDE_FLOAT32_C(   425.48), SIMDE_FLOAT32_C(  -387.94), SIMDE_FLOAT32_C(   -58.09), SIMDE_FLOAT32_C(  -381.37),
        SIMDE_FLOAT32_C(   789.83), SIMDE_FLOAT32_C(   633.19), SIMDE_FLOAT32_C(   -58.09), SIMDE_FLOAT32_C(  -381.37) } },
    { { SIMDE_FLOAT32_C(   385.81), SIMDE_FLOAT32_C(   368.14), SIMDE_FLOAT32_C(  -607.80), SIMDE_FLOAT32_C(   623.02),
        SIMDE_FLOAT32_C(  -955.44), SIMDE_FLOAT32_C(  -138.05), SIMDE_FLOAT32_C(  -245.78), SIMDE_FLOAT32_C(  -750.22) },
      UINT8_C(110),
      { SIMDE_FLOAT32_C(   548.54), SIMDE_FLOAT32_C(  -618.32), SIMDE_FLOAT32_C(  -113.43), SIMDE_FLOAT32_C(  -437.94) },
      { SIMDE_FLOAT32_C(   385.81), SIMDE_FLOAT32_C(  -618.32), SIMDE_FLOAT32_C(  -113.43), SIMDE_FLOAT32_C(  -437.94),
        SIMDE_FLOAT32_C(  -955.44), SIMDE_FLOAT32_C(  -618.32), SIMDE_FLOAT32_C(  -113.43), SIMDE_FLOAT32_C(  -750.22) } },
    { { SIMDE_FLOAT32_C(  -510.40), SIMDE_FLOAT32_C(  -247.29), SIMDE_FLOAT32_C(  -272.50), SIMDE_FLOAT32_C(   154.15),
        SIMDE_FLOAT32_C(   745.34), SIMDE_FLOAT32_C(   865.17), SIMDE_FLOAT32_C(   893.80), SIMDE_FLOAT32_C(    79.97) },
      UINT8_C(108),
      { SIMDE_FLOAT32_C(  -178.61), SIMDE_FLOAT32_C(    31.69), SIMDE_FLOAT32_C(   669.52), SIMDE_FLOAT32_C(   693.51) },
      { SIMDE_FLOAT32_C(  -510.40), SIMDE_FLOAT32_C(  -247.29), SIMDE_FLOAT32_C(   669.52), SIMDE_FLOAT32_C(   693.51),
        SIMDE_FLOAT32_C(   745.34), SIMDE_FLOAT32_C(    31.69), SIMDE_FLOAT32_C(   669.52), SIMDE_FLOAT32_C(    79.97) } },
    { { SIMDE_FLOAT32_C(  -127.96), SIMDE_FLOAT32_C(  -619.72), SIMDE_FLOAT32_C(   284.07), SIMDE_FLOAT32_C(   372.86),
        SIMDE_FLOAT32_C(   649.51), SIMDE_FLOAT32_C(   278.96), SIMDE_FLOAT32_C(   407.00), SIMDE_FLOAT32_C(   484.63) },
      UINT8_C( 35),
      { SIMDE_FLOAT32_C(  -266.56), SIMDE_FLOAT32_C(  -110.85), SIMDE_FLOAT32_C(  -976.05), SIMDE_FLOAT32_C(  -446.86) },
      { SIMDE_FLOAT32_C(  -266.56), SIMDE_FLOAT32_C(  -110.85), SIMDE_FLOAT32_C(   284.07), SIMDE_FLOAT32_C(   372.86),
        SIMDE_FLOAT32_C(   649.51), SIMDE_FLOAT32_C(  -110.85), SIMDE_FLOAT32_C(   407.00), SIMDE_FLOAT32_C(   484.63) } },
    { { SIMDE_FLOAT32_C(  -413.34), SIMDE_FLOAT32_C(   993.71), SIMDE_FLOAT32_C(  -725.95), SIMDE_FLOAT32_C(   912.24),
        SIMDE_FLOAT32_C(    38.79), SIMDE_FLOAT32_C(  -113.15), SIMDE_FLOAT32_C(   355.83), SIMDE_FLOAT32_C(   489.44) },
      UINT8_C(174),
      { SIMDE_FLOAT32_C(   271.71), SIMDE_FLOAT32_C(   611.34), SIMDE_FLOAT32_C(   750.31), SIMDE_FLOAT32_C(   445.31) },
      { SIMDE_FLOAT32_C(  -413.34), SIMDE_FLOAT32_C(   611.34), SIMDE_FLOAT32_C(   750.31), SIMDE_FLOAT32_C(   445.31),
        SIMDE_FLOAT32_C(    38.79), SIMDE_FLOAT32_C(   611.34), SIMDE_FLOAT32_C(   355.83), SIMDE_FLOAT32_C(   445.31) } },
    { { SIMDE_FLOAT32_C(   394.72), SIMDE_FLOAT32_C(    -2.71), SIMDE_FLOAT32_C(   433.21), SIMDE_FLOAT32_C(   979.88),
        SIMDE_FLOAT32_C(   870.25), SIMDE_FLOAT32_C(   239.46), SIMDE_FLOAT32_C(   664.36), SIMDE_FLOAT32_C(   -21.11) },
      UINT8_C(236),
      { SIMDE_FLOAT32_C(    20.21), SIMDE_FLOAT32_C(  -364.92), SIMDE_FLOAT32_C(   870.25), SIMDE_FLOAT32_C(   218.91) },
      { SIMDE_FLOAT32_C(   394.72), SIMDE_FLOAT32_C(    -2.71), SIMDE_FLOAT32_C(   870.25), SIMDE_FLOAT32_C(   218.91),
        SIMDE_FLOAT32_C(   870.25), SIMDE_FLOAT32_C(  -364.92), SIMDE_FLOAT32_C(   870.25), SIMDE_FLOAT32_C(   218.91) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256 src = simde_mm256_loadu_ps(test_vec[i].src);
    simde__m128 a = simde_mm_loadu_ps(test_vec[i].a);
    simde__m256 r = simde_mm256_mask_broadcast_f32x4(src, test_vec[i].k, a);
    simde_test_x86_assert_equal_f32x8(r, simde_mm256_loadu_ps(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm256_maskz_broadcast_f32x4 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde__mmask8 k;
    const simde_float32 a[4];
    const simde_float32 r[8];
  } test_vec[] = {
    { UINT8_C(233),
      { SIMDE_FLOAT32_C(   749.31), SIMDE_FLOAT32_C(  -425.85), SIMDE_FLOAT32_C(   752.50), SIMDE_FLOAT32_C(  -794.87) },
      { SIMDE_FLOAT32_C(   749.31), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -794.87),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -425.85), SIMDE_FLOAT32_C(   752.50), SIMDE_FLOAT32_C(  -794.87) } },
    { UINT8_C(237),
      { SIMDE_FLOAT32_C(   236.00), SIMDE_FLOAT32_C(   493.54), SIMDE_FLOAT32_C(  -992.91), SIMDE_FLOAT32_C(   213.78) },
      { SIMDE_FLOAT32_C(   236.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -992.91), SIMDE_FLOAT32_C(   213.78),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   493.54), SIMDE_FLOAT32_C(  -992.91), SIMDE_FLOAT32_C(   213.78) } },
    { UINT8_C(229),
      { SIMDE_FLOAT32_C(   572.59), SIMDE_FLOAT32_C(  -505.20), SIMDE_FLOAT32_C(  -888.69), SIMDE_FLOAT32_C(  -168.99) },
      { SIMDE_FLOAT32_C(   572.59), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -888.69), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -505.20), SIMDE_FLOAT32_C(  -888.69), SIMDE_FLOAT32_C(  -168.99) } },
    { UINT8_C(115),
      { SIMDE_FLOAT32_C(   961.78), SIMDE_FLOAT32_C(   587.15), SIMDE_FLOAT32_C(   162.08), SIMDE_FLOAT32_C(   131.99) },
      { SIMDE_FLOAT32_C(   961.78), SIMDE_FLOAT32_C(   587.15), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(   961.78), SIMDE_FLOAT32_C(   587.15), SIMDE_FLOAT32_C(   162.08), SIMDE_FLOAT32_C(     0.00) } },
    { UINT8_C(107),
      { SIMDE_FLOAT32_C(   722.82), SIMDE_FLOAT32_C(   519.77), SIMDE_FLOAT32_C(  -160.36), SIMDE_FLOAT32_C(   908.34) },
      { SIMDE_FLOAT32_C(   722.82), SIMDE_FLOAT32_C(   519.77), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   908.34),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   519.77), SIMDE_FLOAT32_C(  -160.36), SIMDE_FLOAT32_C(     0.00) } },
    { UINT8_C(148),
      { SIMDE_FLOAT32_C(   251.18), SIMDE_FLOAT32_C(  -347.86), SIMDE_FLOAT32_C(  -514.92), SIMDE_FLOAT32_C(  -206.57) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -514.92), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(   251.18), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -206.57) } },
    { UINT8_C(156),
      { SIMDE_FLOAT32_C(   874.47), SIMDE_FLOAT32_C(  -711.75), SIMDE_FLOAT32_C(  -458.03), SIMDE_FLOAT32_C(  -188.74) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -458.03), SIMDE_FLOAT32_C(  -188.74),
        SIMDE_FLOAT32_C(   874.47), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -188.74) } },
    { UINT8_C( 78),
      { SIMDE_FLOAT32_C(  -804.36), SIMDE_FLOAT32_C(  -844.65), SIMDE_FLOAT32_C(   -82.05), SIMDE_FLOAT32_C(  -986.67) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -844.65), SIMDE_FLOAT32_C(   -82.05), SIMDE_FLOAT32_C(  -986.67),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   -82.05), SIMDE_FLOAT32_C(     0.00) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128 a = simde_mm_loadu_ps(test_vec[i].a);
    simde__m256 r = simde_mm256_maskz_broadcast_f32x4(test_vec[i].k, a);
    simde_test_x86_assert_equal_f32x8(r, simde_mm256_loadu_ps(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm256_broadcast_f64x2 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float64 a[2];
    const simde_float64 r[4];
  } test_vec[] = {
    { { SIMDE_FLOAT64_C(  -818.06), SIMDE_FLOAT64_C(   862.04) },
      { SIMDE_FLOAT64_C(  -818.06), SIMDE_FLOAT64_C(   862.04), SIMDE_FLOAT64_C(  -818.06), SIMDE_FLOAT64_C(   862.04) } },
    { { SIMDE_FLOAT64_C(   251.28), SIMDE_FLOAT64_C(  -807.49) },
      { SIMDE_FLOAT64_C(   251.28), SIMDE_FLOAT64_C(  -807.49), SIMDE_FLOAT64_C(   251.28), SIMDE_FLOAT64_C(  -807.49) } },
    { { SIMDE_FLOAT64_C(   489.47), SIMDE_FLOAT64_C(   521.73) },
      { SIMDE_FLOAT64_C(   489.47), SIMDE_FLOAT64_C(   521.73), SIMDE_FLOAT64_C(   489.47), SIMDE_FLOAT64_C(   521.73) } },
    { { SIMDE_FLOAT64_C(   697.15), SIMDE_FLOAT64_C(  -943.39) },
      { SIMDE_FLOAT64_C(   697.15), SIMDE_FLOAT64_C(  -943.39), SIMDE_FLOAT64_C(   697.15), SIMDE_FLOAT64_C(  -943.39) } },
    { { SIMDE_FLOAT64_C(   397.38), SIMDE_FLOAT64_C(   769.24) },
      { SIMDE_FLOAT64_C(   397.38), SIMDE_FLOAT64_C(   769.24), SIMDE_FLOAT64_C(   397.38), SIMDE_FLOAT64_C(   769.24) } },
    { { SIMDE_FLOAT64_C(   607.10), SIMDE_FLOAT64_C(  -411.28) },
      { SIMDE_FLOAT64_C(   607.10), SIMDE_FLOAT64_C(  -411.28), SIMDE_FLOAT64_C(   607.10), SIMDE_FLOAT64_C(  -411.28) } },
    { { SIMDE_FLOAT64_C(  -417.96), SIMDE_FLOAT64_C(  -732.77) },
      { SIMDE_FLOAT64_C(  -417.96), SIMDE_FLOAT64_C(  -732.77), SIMDE_FLOAT64_C(  -417.96), SIMDE_FLOAT64_C(  -732.77) } },
    { { SIMDE_FLOAT64_C(   409.47), SIMDE_FLOAT64_C(   -49.18) },
      { SIMDE_FLOAT64_C(   409.47), SIMDE_FLOAT64_C(   -49.18), SIMDE_FLOAT64_C(   409.47), SIMDE_FLOAT64_C(   -49.18) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128d a = simde_mm_loadu_pd(test_vec[i].a);
    simde__m256d r = simde_mm256_broadcast_f64x2(a);
    simde_test_x86_assert_equal_f64x4(r, simde_mm256_loadu_pd(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm256_mask_broadcast_f64x2 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float64 src[4];
    const simde__mmask8 k;
    const simde_float64 a[2];
    const simde_float64 r[4];
  } test_vec[] = {
    { { SIMDE_FLOAT64_C(  -338.03), SIMDE_FLOAT64_C(   731.04), SIMDE_FLOAT64_C(   652.28), SIMDE_FLOAT64_C(   868.31) },
      UINT8_C(176),
      { SIMDE_FLOAT64_C(  -240.09), SIMDE_FLOAT64_C(   738.97) },
      { SIMDE_FLOAT64_C(  -338.03), SIMDE_FLOAT64_C(   731.04), SIMDE_FLOAT64_C(   652.28), SIMDE_FLOAT64_C(   868.31) } },
    { { SIMDE_FLOAT64_C(  -161.99), SIMDE_FLOAT64_C(  -539.33), SIMDE_FLOAT64_C(  -491.52), SIMDE_FLOAT64_C(   960.24) },
      UINT8_C( 23),
      { SIMDE_FLOAT64_C(   782.07), SIMDE_FLOAT64_C(   293.17) },
      { SIMDE_FLOAT64_C(   782.07), SIMDE_FLOAT64_C(   293.17), SIMDE_FLOAT64_C(   782.07), SIMDE_FLOAT64_C(   960.24) } },
    { { SIMDE_FLOAT64_C(  -948.97), SIMDE_FLOAT64_C(   718.70), SIMDE_FLOAT64_C(  -833.55), SIMDE_FLOAT64_C(   519.24) },
      UINT8_C(166),
      { SIMDE_FLOAT64_C(   879.34), SIMDE_FLOAT64_C(  -863.77) },
      { SIMDE_FLOAT64_C(  -948.97), SIMDE_FLOAT64_C(  -863.77), SIMDE_FLOAT64_C(   879.34), SIMDE_FLOAT64_C(   519.24) } },
    { { SIMDE_FLOAT64_C(   136.25), SIMDE_FLOAT64_C(   -99.23), SIMDE_FLOAT64_C(   178.08), SIMDE_FLOAT64_C(  -929.05) },
      UINT8_C( 20),
      { SIMDE_FLOAT64_C(  -614.75), SIMDE_FLOAT64_C(   -70.42) },
      { SIMDE_FLOAT64_C(   136.25), SIMDE_FLOAT64_C(   -99.23), SIMDE_FLOAT64_C(  -614.75), SIMDE_FLOAT64_C(  -929.05) } },
    { { SIMDE_FLOAT64_C(  -617.52), SIMDE_FLOAT64_C(  -721.29), SIMDE_FLOAT64_C(  -762.54), SIMDE_FLOAT64_C(    70.31) },
      UINT8_C(  5),
      { SIMDE_FLOAT64_C(  -322.15), SIMDE_FLOAT64_C(  -417.60) },
      { SIMDE_FLOAT64_C(  -322.15), SIMDE_FLOAT64_C(  -721.29), SIMDE_FLOAT64_C(  -322.15), SIMDE_FLOAT64_C(    70.31) } },
    { { SIMDE_FLOAT64_C(  -577.36), SIMDE_FLOAT64_C(   298.63), SIMDE_FLOAT64_C(  -985.58), SIMDE_FLOAT64_C(  -562.98) },
      UINT8_C(167),
      { SIMDE_FLOAT64_C(   -39.73), SIMDE_FLOAT64_C(   262.95) },
      { SIMDE_FLOAT64_C(   -39.73), SIMDE_FLOAT64_C(   262.95), SIMDE_FLOAT64_C(   -39.73), SIMDE_FLOAT64_C(  -562.98) } },
    { { SIMDE_FLOAT64_C(   943.89), SIMDE_FLOAT64_C(  -108.91), SIMDE_FLOAT64_C(  -463.93), SIMDE_FLOAT64_C(   675.74) },
      UINT8_C(200),
      { SIMDE_FLOAT64_C(  -918.41), SIMDE_FLOAT64_C(   364.14) },
      { SIMDE_FLOAT64_C(   943.89), SIMDE_FLOAT64_C(  -108.91), SIMDE_FLOAT64_C(  -463.93), SIMDE_FLOAT64_C(   364.14) } },
    { { SIMDE_FLOAT64_C(   -90.94), SIMDE_FLOAT64_C(  -345.61), SIMDE_FLOAT64_C(  -599.08), SIMDE_FLOAT64_C(  -818.15) },
      UINT8_C(109),
      { SIMDE_FLOAT64_C(  -714.87), SIMDE_FLOAT64_C(  -771.51) },
      { SIMDE_FLOAT64_C(  -714.87), SIMDE_FLOAT64_C(  -345.61), SIMDE_FLOAT64_C(  -714.87), SIMDE_FLOAT64_C(  -771.51) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256d src = simde_mm256_loadu_pd(test_vec[i].src);
    simde__m128d a = simde_mm_loadu_pd(test_vec[i].a);
    simde__m256d r = simde_mm256_mask_broadcast_f64x2(src, test_vec[i].k, a);
    simde_test_x86_assert_equal_f64x4(r, simde_mm256_loadu_pd(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm256_maskz_broadcast_f64x2 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde__mmask8 k;
    const simde_float64 a[2];
    const simde_float64 r[4];
  } test_vec[] = {
    { UINT8_C(197),
      { SIMDE_FLOAT64_C(  -215.62), SIMDE_FLOAT64_C(    35.19) },
      { SIMDE_FLOAT64_C(  -215.62), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(  -215.62), SIMDE_FLOAT64_C(     0.00) } },
    { UINT8_C(198),
      { SIMDE_FLOAT64_C(   716.52), SIMDE_FLOAT64_C(   473.89) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(   473.89), SIMDE_FLOAT64_C(   716.52), SIMDE_FLOAT64_C(     0.00) } },
    { UINT8_C( 85),
      { SIMDE_FLOAT64_C(   312.77), SIMDE_FLOAT64_C(   715.13) },
      { SIMDE_FLOAT64_C(   312.77), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(   312.77), SIMDE_FLOAT64_C(     0.00) } },
    { UINT8_C(212),
      { SIMDE_FLOAT64_C(   527.96), SIMDE_FLOAT64_C(  -502.50) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(   527.96), SIMDE_FLOAT64_C(     0.00) } },
    { UINT8_C( 50),
      { SIMDE_FLOAT64_C(  -571.65), SIMDE_FLOAT64_C(   248.58) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(   248.58), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00) } },
    { UINT8_C(207),
      { SIMDE_FLOAT64_C(   234.22), SIMDE_FLOAT64_C(   607.13) },
      { SIMDE_FLOAT64_C(   234.22), SIMDE_FLOAT64_C(   607.13), SIMDE_FLOAT64_C(   234.22), SIMDE_FLOAT64_C(   607.13) } },
    { UINT8_C(  5),
      { SIMDE_FLOAT64_C(  -229.19), SIMDE_FLOAT64_C(   -58.91) },
      { SIMDE_FLOAT64_C(  -229.19), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(  -229.19), SIMDE_FLOAT64_C(     0.00) } },
    { UINT8_C(104),
      { SIMDE_FLOAT64_C(    -8.77), SIMDE_FLOAT64_C(   682.18) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(   682.18) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128d a = simde_mm_loadu_pd(test_vec[i].a);
    simde__m256d r = simde_mm256_maskz_broadcast_f64x2(test_vec[i].k, a);
    simde_test_x86_assert_equal_f64x4(r, simde_mm256_loadu_pd(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm512_broadcast_f32x4(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128 a;
    simde__m512 r;
  } test_vec[8] = {
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   241.63), SIMDE_FLOAT32_C(   962.32), SIMDE_FLOAT32_C(  -223.53), SIMDE_FLOAT32_C(  -221.69)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   241.63), SIMDE_FLOAT32_C(   962.32), SIMDE_FLOAT32_C(  -223.53), SIMDE_FLOAT32_C(  -221.69),
                         SIMDE_FLOAT32_C(   241.63), SIMDE_FLOAT32_C(   962.32), SIMDE_FLOAT32_C(  -223.53), SIMDE_FLOAT32_C(  -221.69),
                         SIMDE_FLOAT32_C(   241.63), SIMDE_FLOAT32_C(   962.32), SIMDE_FLOAT32_C(  -223.53), SIMDE_FLOAT32_C(  -221.69),
                         SIMDE_FLOAT32_C(   241.63), SIMDE_FLOAT32_C(   962.32), SIMDE_FLOAT32_C(  -223.53), SIMDE_FLOAT32_C(  -221.69)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   115.71), SIMDE_FLOAT32_C(  -206.04), SIMDE_FLOAT32_C(  -581.48), SIMDE_FLOAT32_C(   670.36)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   115.71), SIMDE_FLOAT32_C(  -206.04), SIMDE_FLOAT32_C(  -581.48), SIMDE_FLOAT32_C(   670.36),
                         SIMDE_FLOAT32_C(   115.71), SIMDE_FLOAT32_C(  -206.04), SIMDE_FLOAT32_C(  -581.48), SIMDE_FLOAT32_C(   670.36),
                         SIMDE_FLOAT32_C(   115.71), SIMDE_FLOAT32_C(  -206.04), SIMDE_FLOAT32_C(  -581.48), SIMDE_FLOAT32_C(   670.36),
                         SIMDE_FLOAT32_C(   115.71), SIMDE_FLOAT32_C(  -206.04), SIMDE_FLOAT32_C(  -581.48), SIMDE_FLOAT32_C(   670.36)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   297.45), SIMDE_FLOAT32_C(   193.39), SIMDE_FLOAT32_C(  -163.24), SIMDE_FLOAT32_C(  -775.87)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   297.45), SIMDE_FLOAT32_C(   193.39), SIMDE_FLOAT32_C(  -163.24), SIMDE_FLOAT32_C(  -775.87),
                         SIMDE_FLOAT32_C(   297.45), SIMDE_FLOAT32_C(   193.39), SIMDE_FLOAT32_C(  -163.24), SIMDE_FLOAT32_C(  -775.87),
                         SIMDE_FLOAT32_C(   297.45), SIMDE_FLOAT32_C(   193.39), SIMDE_FLOAT32_C(  -163.24), SIMDE_FLOAT32_C(  -775.87),
                         SIMDE_FLOAT32_C(   297.45), SIMDE_FLOAT32_C(   193.39), SIMDE_FLOAT32_C(  -163.24), SIMDE_FLOAT32_C(  -775.87)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(  -675.37), SIMDE_FLOAT32_C(   853.20), SIMDE_FLOAT32_C(  -377.67), SIMDE_FLOAT32_C(   233.14)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -675.37), SIMDE_FLOAT32_C(   853.20), SIMDE_FLOAT32_C(  -377.67), SIMDE_FLOAT32_C(   233.14),
                         SIMDE_FLOAT32_C(  -675.37), SIMDE_FLOAT32_C(   853.20), SIMDE_FLOAT32_C(  -377.67), SIMDE_FLOAT32_C(   233.14),
                         SIMDE_FLOAT32_C(  -675.37), SIMDE_FLOAT32_C(   853.20), SIMDE_FLOAT32_C(  -377.67), SIMDE_FLOAT32_C(   233.14),
                         SIMDE_FLOAT32_C(  -675.37), SIMDE_FLOAT32_C(   853.20), SIMDE_FLOAT32_C(  -377.67), SIMDE_FLOAT32_C(   233.14)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(  -156.08), SIMDE_FLOAT32_C(  -209.26), SIMDE_FLOAT32_C(    48.51), SIMDE_FLOAT32_C(  -627.76)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -156.08), SIMDE_FLOAT32_C(  -209.26), SIMDE_FLOAT32_C(    48.51), SIMDE_FLOAT32_C(  -627.76),
                         SIMDE_FLOAT32_C(  -156.08), SIMDE_FLOAT32_C(  -209.26), SIMDE_FLOAT32_C(    48.51), SIMDE_FLOAT32_C(  -627.76),
                         SIMDE_FLOAT32_C(  -156.08), SIMDE_FLOAT32_C(  -209.26), SIMDE_FLOAT32_C(    48.51), SIMDE_FLOAT32_C(  -627.76),
                         SIMDE_FLOAT32_C(  -156.08), SIMDE_FLOAT32_C(  -209.26), SIMDE_FLOAT32_C(    48.51), SIMDE_FLOAT32_C(  -627.76)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   108.40), SIMDE_FLOAT32_C(   970.37), SIMDE_FLOAT32_C(   934.72), SIMDE_FLOAT32_C(  -932.81)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   108.40), SIMDE_FLOAT32_C(   970.37), SIMDE_FLOAT32_C(   934.72), SIMDE_FLOAT32_C(  -932.81),
                         SIMDE_FLOAT32_C(   108.40), SIMDE_FLOAT32_C(   970.37), SIMDE_FLOAT32_C(   934.72), SIMDE_FLOAT32_C(  -932.81),
                         SIMDE_FLOAT32_C(   108.40), SIMDE_FLOAT32_C(   970.37), SIMDE_FLOAT32_C(   934.72), SIMDE_FLOAT32_C(  -932.81),
                         SIMDE_FLOAT32_C(   108.40), SIMDE_FLOAT32_C(   970.37), SIMDE_FLOAT32_C(   934.72), SIMDE_FLOAT32_C(  -932.81)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   690.58), SIMDE_FLOAT32_C(   836.42), SIMDE_FLOAT32_C(  -952.66), SIMDE_FLOAT32_C(    22.35)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   690.58), SIMDE_FLOAT32_C(   836.42), SIMDE_FLOAT32_C(  -952.66), SIMDE_FLOAT32_C(    22.35),
                         SIMDE_FLOAT32_C(   690.58), SIMDE_FLOAT32_C(   836.42), SIMDE_FLOAT32_C(  -952.66), SIMDE_FLOAT32_C(    22.35),
                         SIMDE_FLOAT32_C(   690.58), SIMDE_FLOAT32_C(   836.42), SIMDE_FLOAT32_C(  -952.66), SIMDE_FLOAT32_C(    22.35),
                         SIMDE_FLOAT32_C(   690.58), SIMDE_FLOAT32_C(   836.42), SIMDE_FLOAT32_C(  -952.66), SIMDE_FLOAT32_C(    22.35)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   740.10), SIMDE_FLOAT32_C(   159.65), SIMDE_FLOAT32_C(   -65.49), SIMDE_FLOAT32_C(   946.83)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   740.10), SIMDE_FLOAT32_C(   159.65), SIMDE_FLOAT32_C(   -65.49), SIMDE_FLOAT32_C(   946.83),
                         SIMDE_FLOAT32_C(   740.10), SIMDE_FLOAT32_C(   159.65), SIMDE_FLOAT32_C(   -65.49), SIMDE_FLOAT32_C(   946.83),
                         SIMDE_FLOAT32_C(   740.10), SIMDE_FLOAT32_C(   159.65), SIMDE_FLOAT32_C(   -65.49), SIMDE_FLOAT32_C(   946.83),
                         SIMDE_FLOAT32_C(   740.10), SIMDE_FLOAT32_C(   159.65), SIMDE_FLOAT32_C(   -65.49), SIMDE_FLOAT32_C(   946.83)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_broadcast_f32x4(test_vec[i].a);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_mask_broadcast_f32x4(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512 src;
    simde__mmask16 k;
    simde__m128 a;
    simde__m512 r;
  } test_vec[8] = {
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -476.82), SIMDE_FLOAT32_C(   687.27), SIMDE_FLOAT32_C(   239.12), SIMDE_FLOAT32_C(  -622.96),
                         SIMDE_FLOAT32_C(   479.82), SIMDE_FLOAT32_C(  -652.18), SIMDE_FLOAT32_C(   585.66), SIMDE_FLOAT32_C(  -840.39),
                         SIMDE_FLOAT32_C(  -680.47), SIMDE_FLOAT32_C(  -211.69), SIMDE_FLOAT32_C(   879.50), SIMDE_FLOAT32_C(   245.88),
                         SIMDE_FLOAT32_C(   689.68), SIMDE_FLOAT32_C(   107.64), SIMDE_FLOAT32_C(  -872.56), SIMDE_FLOAT32_C(  -586.10)),
      UINT16_C(63721),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   337.98), SIMDE_FLOAT32_C(  -931.30), SIMDE_FLOAT32_C(   -93.71), SIMDE_FLOAT32_C(   492.43)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   337.98), SIMDE_FLOAT32_C(  -931.30), SIMDE_FLOAT32_C(   -93.71), SIMDE_FLOAT32_C(   492.43),
                         SIMDE_FLOAT32_C(   337.98), SIMDE_FLOAT32_C(  -652.18), SIMDE_FLOAT32_C(   585.66), SIMDE_FLOAT32_C(  -840.39),
                         SIMDE_FLOAT32_C(   337.98), SIMDE_FLOAT32_C(  -931.30), SIMDE_FLOAT32_C(   -93.71), SIMDE_FLOAT32_C(   245.88),
                         SIMDE_FLOAT32_C(   337.98), SIMDE_FLOAT32_C(   107.64), SIMDE_FLOAT32_C(  -872.56), SIMDE_FLOAT32_C(   492.43)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   792.34), SIMDE_FLOAT32_C(  -828.98), SIMDE_FLOAT32_C(   152.82), SIMDE_FLOAT32_C(   261.49),
                         SIMDE_FLOAT32_C(  -674.96), SIMDE_FLOAT32_C(  -626.70), SIMDE_FLOAT32_C(  -365.50), SIMDE_FLOAT32_C(   522.39),
                         SIMDE_FLOAT32_C(   659.15), SIMDE_FLOAT32_C(   204.13), SIMDE_FLOAT32_C(   487.20), SIMDE_FLOAT32_C(   790.92),
                         SIMDE_FLOAT32_C(  -372.23), SIMDE_FLOAT32_C(  -362.18), SIMDE_FLOAT32_C(   725.62), SIMDE_FLOAT32_C(   817.00)),
      UINT16_C(44067),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -858.39), SIMDE_FLOAT32_C(   608.18), SIMDE_FLOAT32_C(   129.78), SIMDE_FLOAT32_C(  -779.98)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -858.39), SIMDE_FLOAT32_C(  -828.98), SIMDE_FLOAT32_C(   129.78), SIMDE_FLOAT32_C(   261.49),
                         SIMDE_FLOAT32_C(  -858.39), SIMDE_FLOAT32_C(   608.18), SIMDE_FLOAT32_C(  -365.50), SIMDE_FLOAT32_C(   522.39),
                         SIMDE_FLOAT32_C(   659.15), SIMDE_FLOAT32_C(   204.13), SIMDE_FLOAT32_C(   129.78), SIMDE_FLOAT32_C(   790.92),
                         SIMDE_FLOAT32_C(  -372.23), SIMDE_FLOAT32_C(  -362.18), SIMDE_FLOAT32_C(   129.78), SIMDE_FLOAT32_C(  -779.98)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   433.05), SIMDE_FLOAT32_C(   405.44), SIMDE_FLOAT32_C(   652.04), SIMDE_FLOAT32_C(  -453.75),
                         SIMDE_FLOAT32_C(    56.24), SIMDE_FLOAT32_C(   506.86), SIMDE_FLOAT32_C(  -127.57), SIMDE_FLOAT32_C(  -230.83),
                         SIMDE_FLOAT32_C(  -815.89), SIMDE_FLOAT32_C(   351.22), SIMDE_FLOAT32_C(  -739.81), SIMDE_FLOAT32_C(  -104.33),
                         SIMDE_FLOAT32_C(   331.38), SIMDE_FLOAT32_C(   749.42), SIMDE_FLOAT32_C(   151.95), SIMDE_FLOAT32_C(   -25.90)),
      UINT16_C(12331),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -159.95), SIMDE_FLOAT32_C(  -519.57), SIMDE_FLOAT32_C(   -66.62), SIMDE_FLOAT32_C(  -690.93)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   433.05), SIMDE_FLOAT32_C(   405.44), SIMDE_FLOAT32_C(   -66.62), SIMDE_FLOAT32_C(  -690.93),
                         SIMDE_FLOAT32_C(    56.24), SIMDE_FLOAT32_C(   506.86), SIMDE_FLOAT32_C(  -127.57), SIMDE_FLOAT32_C(  -230.83),
                         SIMDE_FLOAT32_C(  -815.89), SIMDE_FLOAT32_C(   351.22), SIMDE_FLOAT32_C(   -66.62), SIMDE_FLOAT32_C(  -104.33),
                         SIMDE_FLOAT32_C(  -159.95), SIMDE_FLOAT32_C(   749.42), SIMDE_FLOAT32_C(   -66.62), SIMDE_FLOAT32_C(  -690.93)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   417.50), SIMDE_FLOAT32_C(   245.21), SIMDE_FLOAT32_C(   960.01), SIMDE_FLOAT32_C(  -303.61),
                         SIMDE_FLOAT32_C(  -550.57), SIMDE_FLOAT32_C(   665.98), SIMDE_FLOAT32_C(  -521.00), SIMDE_FLOAT32_C(   239.39),
                         SIMDE_FLOAT32_C(   798.32), SIMDE_FLOAT32_C(   251.37), SIMDE_FLOAT32_C(  -596.78), SIMDE_FLOAT32_C(   840.69),
                         SIMDE_FLOAT32_C(  -684.92), SIMDE_FLOAT32_C(    87.08), SIMDE_FLOAT32_C(   734.84), SIMDE_FLOAT32_C(  -854.89)),
      UINT16_C(52021),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -116.62), SIMDE_FLOAT32_C(   -17.97), SIMDE_FLOAT32_C(   229.99), SIMDE_FLOAT32_C(  -771.72)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -116.62), SIMDE_FLOAT32_C(   -17.97), SIMDE_FLOAT32_C(   960.01), SIMDE_FLOAT32_C(  -303.61),
                         SIMDE_FLOAT32_C(  -116.62), SIMDE_FLOAT32_C(   665.98), SIMDE_FLOAT32_C(   229.99), SIMDE_FLOAT32_C(  -771.72),
                         SIMDE_FLOAT32_C(   798.32), SIMDE_FLOAT32_C(   251.37), SIMDE_FLOAT32_C(   229.99), SIMDE_FLOAT32_C(  -771.72),
                         SIMDE_FLOAT32_C(  -684.92), SIMDE_FLOAT32_C(   -17.97), SIMDE_FLOAT32_C(   734.84), SIMDE_FLOAT32_C(  -771.72)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -616.45), SIMDE_FLOAT32_C(   914.70), SIMDE_FLOAT32_C(  -963.67), SIMDE_FLOAT32_C(  -935.61),
                         SIMDE_FLOAT32_C(   106.52), SIMDE_FLOAT32_C(   367.48), SIMDE_FLOAT32_C(   -10.30), SIMDE_FLOAT32_C(   543.55),
                         SIMDE_FLOAT32_C(   142.17), SIMDE_FLOAT32_C(  -844.51), SIMDE_FLOAT32_C(  -959.58), SIMDE_FLOAT32_C(   913.58),
                         SIMDE_FLOAT32_C(  -227.61), SIMDE_FLOAT32_C(  -979.09), SIMDE_FLOAT32_C(  -746.95), SIMDE_FLOAT32_C(   363.67)),
      UINT16_C(46395),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -319.95), SIMDE_FLOAT32_C(  -241.48), SIMDE_FLOAT32_C(  -416.05), SIMDE_FLOAT32_C(  -700.83)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -319.95), SIMDE_FLOAT32_C(   914.70), SIMDE_FLOAT32_C(  -416.05), SIMDE_FLOAT32_C(  -700.83),
                         SIMDE_FLOAT32_C(   106.52), SIMDE_FLOAT32_C(  -241.48), SIMDE_FLOAT32_C(   -10.30), SIMDE_FLOAT32_C(  -700.83),
                         SIMDE_FLOAT32_C(   142.17), SIMDE_FLOAT32_C(  -844.51), SIMDE_FLOAT32_C(  -416.05), SIMDE_FLOAT32_C(  -700.83),
                         SIMDE_FLOAT32_C(  -319.95), SIMDE_FLOAT32_C(  -979.09), SIMDE_FLOAT32_C(  -416.05), SIMDE_FLOAT32_C(  -700.83)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   810.23), SIMDE_FLOAT32_C(  -571.66), SIMDE_FLOAT32_C(  -313.94), SIMDE_FLOAT32_C(   812.08),
                         SIMDE_FLOAT32_C(   905.89), SIMDE_FLOAT32_C(    95.84), SIMDE_FLOAT32_C(  -942.64), SIMDE_FLOAT32_C(   490.95),
                         SIMDE_FLOAT32_C(   432.01), SIMDE_FLOAT32_C(  -989.57), SIMDE_FLOAT32_C(  -908.07), SIMDE_FLOAT32_C(   843.06),
                         SIMDE_FLOAT32_C(  -567.12), SIMDE_FLOAT32_C(   561.55), SIMDE_FLOAT32_C(  -316.58), SIMDE_FLOAT32_C(  -224.94)),
      UINT16_C(28510),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   608.47), SIMDE_FLOAT32_C(   502.71), SIMDE_FLOAT32_C(   524.73), SIMDE_FLOAT32_C(  -206.66)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   810.23), SIMDE_FLOAT32_C(   502.71), SIMDE_FLOAT32_C(   524.73), SIMDE_FLOAT32_C(   812.08),
                         SIMDE_FLOAT32_C(   608.47), SIMDE_FLOAT32_C(   502.71), SIMDE_FLOAT32_C(   524.73), SIMDE_FLOAT32_C(  -206.66),
                         SIMDE_FLOAT32_C(   432.01), SIMDE_FLOAT32_C(   502.71), SIMDE_FLOAT32_C(  -908.07), SIMDE_FLOAT32_C(  -206.66),
                         SIMDE_FLOAT32_C(   608.47), SIMDE_FLOAT32_C(   502.71), SIMDE_FLOAT32_C(   524.73), SIMDE_FLOAT32_C(  -224.94)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -770.27), SIMDE_FLOAT32_C(  -598.61), SIMDE_FLOAT32_C(   672.88), SIMDE_FLOAT32_C(  -504.06),
                         SIMDE_FLOAT32_C(   481.78), SIMDE_FLOAT32_C(  -154.88), SIMDE_FLOAT32_C(  -363.51), SIMDE_FLOAT32_C(  -643.93),
                         SIMDE_FLOAT32_C(  -973.84), SIMDE_FLOAT32_C(  -599.20), SIMDE_FLOAT32_C(   230.44), SIMDE_FLOAT32_C(  -713.35),
                         SIMDE_FLOAT32_C(  -554.88), SIMDE_FLOAT32_C(  -858.98), SIMDE_FLOAT32_C(   -21.09), SIMDE_FLOAT32_C(  -441.11)),
      UINT16_C( 6749),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   687.60), SIMDE_FLOAT32_C(   681.66), SIMDE_FLOAT32_C(  -362.35), SIMDE_FLOAT32_C(  -482.20)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -770.27), SIMDE_FLOAT32_C(  -598.61), SIMDE_FLOAT32_C(   672.88), SIMDE_FLOAT32_C(  -482.20),
                         SIMDE_FLOAT32_C(   687.60), SIMDE_FLOAT32_C(  -154.88), SIMDE_FLOAT32_C(  -362.35), SIMDE_FLOAT32_C(  -643.93),
                         SIMDE_FLOAT32_C(  -973.84), SIMDE_FLOAT32_C(   681.66), SIMDE_FLOAT32_C(   230.44), SIMDE_FLOAT32_C(  -482.20),
                         SIMDE_FLOAT32_C(   687.60), SIMDE_FLOAT32_C(   681.66), SIMDE_FLOAT32_C(   -21.09), SIMDE_FLOAT32_C(  -482.20)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -454.36), SIMDE_FLOAT32_C(  -172.69), SIMDE_FLOAT32_C(   256.23), SIMDE_FLOAT32_C(   682.27),
                         SIMDE_FLOAT32_C(   -43.91), SIMDE_FLOAT32_C(  -300.48), SIMDE_FLOAT32_C(   916.93), SIMDE_FLOAT32_C(  -592.77),
                         SIMDE_FLOAT32_C(   939.83), SIMDE_FLOAT32_C(  -553.88), SIMDE_FLOAT32_C(  -796.09), SIMDE_FLOAT32_C(  -515.91),
                         SIMDE_FLOAT32_C(   623.85), SIMDE_FLOAT32_C(   359.37), SIMDE_FLOAT32_C(  -557.79), SIMDE_FLOAT32_C(   595.65)),
      UINT16_C( 8287),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -705.53), SIMDE_FLOAT32_C(   238.42), SIMDE_FLOAT32_C(   504.37), SIMDE_FLOAT32_C(   296.48)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -454.36), SIMDE_FLOAT32_C(  -172.69), SIMDE_FLOAT32_C(   504.37), SIMDE_FLOAT32_C(   682.27),
                         SIMDE_FLOAT32_C(   -43.91), SIMDE_FLOAT32_C(  -300.48), SIMDE_FLOAT32_C(   916.93), SIMDE_FLOAT32_C(  -592.77),
                         SIMDE_FLOAT32_C(   939.83), SIMDE_FLOAT32_C(   238.42), SIMDE_FLOAT32_C(  -796.09), SIMDE_FLOAT32_C(   296.48),
                         SIMDE_FLOAT32_C(  -705.53), SIMDE_FLOAT32_C(   238.42), SIMDE_FLOAT32_C(   504.37), SIMDE_FLOAT32_C(   296.48)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_mask_broadcast_f32x4(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_maskz_broadcast_f32x4(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask16 k;
    simde__m128 a;
    simde__m512 r;
  } test_vec[8] = {
    { UINT16_C(12860),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   -93.71), SIMDE_FLOAT32_C(   137.99), SIMDE_FLOAT32_C(   492.43), SIMDE_FLOAT32_C(   420.83)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   492.43), SIMDE_FLOAT32_C(   420.83),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   492.43), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   492.43), SIMDE_FLOAT32_C(   420.83),
                         SIMDE_FLOAT32_C(   -93.71), SIMDE_FLOAT32_C(   137.99), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C(63770),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -652.18), SIMDE_FLOAT32_C(  -872.56), SIMDE_FLOAT32_C(   585.66), SIMDE_FLOAT32_C(  -586.10)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -652.18), SIMDE_FLOAT32_C(  -872.56), SIMDE_FLOAT32_C(   585.66), SIMDE_FLOAT32_C(  -586.10),
                         SIMDE_FLOAT32_C(  -652.18), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -586.10),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -586.10),
                         SIMDE_FLOAT32_C(  -652.18), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   585.66), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C(26030),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   700.39), SIMDE_FLOAT32_C(   129.78), SIMDE_FLOAT32_C(   708.98), SIMDE_FLOAT32_C(  -779.98)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   129.78), SIMDE_FLOAT32_C(   708.98), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   129.78), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -779.98),
                         SIMDE_FLOAT32_C(   700.39), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   708.98), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   700.39), SIMDE_FLOAT32_C(   129.78), SIMDE_FLOAT32_C(   708.98), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C(41122),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -362.18), SIMDE_FLOAT32_C(  -626.70), SIMDE_FLOAT32_C(   725.62), SIMDE_FLOAT32_C(  -365.50)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -362.18), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   725.62), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(  -362.18), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   725.62), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   725.62), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C(49851),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -519.57), SIMDE_FLOAT32_C(  -632.83), SIMDE_FLOAT32_C(   -66.62), SIMDE_FLOAT32_C(  -181.94)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -519.57), SIMDE_FLOAT32_C(  -632.83), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   -66.62), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(  -519.57), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   -66.62), SIMDE_FLOAT32_C(  -181.94),
                         SIMDE_FLOAT32_C(  -519.57), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   -66.62), SIMDE_FLOAT32_C(  -181.94)) },
    { UINT16_C(41826),
      simde_mm_set_ps(SIMDE_FLOAT32_C(    56.24), SIMDE_FLOAT32_C(   749.42), SIMDE_FLOAT32_C(   506.86), SIMDE_FLOAT32_C(   151.95)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(    56.24), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   506.86), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   506.86), SIMDE_FLOAT32_C(   151.95),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   749.42), SIMDE_FLOAT32_C(   506.86), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   506.86), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C(19285),
      simde_mm_set_ps(SIMDE_FLOAT32_C(    57.37), SIMDE_FLOAT32_C(   -17.97), SIMDE_FLOAT32_C(   347.13), SIMDE_FLOAT32_C(   229.99)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   -17.97), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(    57.37), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   347.13), SIMDE_FLOAT32_C(   229.99),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   -17.97), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   229.99),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   -17.97), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   229.99)) },
    { UINT16_C(48133),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -684.92), SIMDE_FLOAT32_C(  -550.57), SIMDE_FLOAT32_C(    87.08), SIMDE_FLOAT32_C(   665.98)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -684.92), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    87.08), SIMDE_FLOAT32_C(   665.98),
                         SIMDE_FLOAT32_C(  -684.92), SIMDE_FLOAT32_C(  -550.57), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -550.57), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   665.98)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_maskz_broadcast_f32x4(test_vec[i].k, test_vec[i].a);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_broadcast_f64x4(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256d a;
    simde__m512d r;
  } test_vec[8] = {
    { simde_mm256_set_pd(SIMDE_FLOAT64_C(  241.63), SIMDE_FLOAT64_C(  962.32),
                         SIMDE_FLOAT64_C( -223.53), SIMDE_FLOAT64_C( -221.69)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  241.63), SIMDE_FLOAT64_C(  962.32),
                         SIMDE_FLOAT64_C( -223.53), SIMDE_FLOAT64_C( -221.69),
                         SIMDE_FLOAT64_C(  241.63), SIMDE_FLOAT64_C(  962.32),
                         SIMDE_FLOAT64_C( -223.53), SIMDE_FLOAT64_C( -221.69)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C(  115.71), SIMDE_FLOAT64_C( -206.04),
                         SIMDE_FLOAT64_C( -581.48), SIMDE_FLOAT64_C(  670.36)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  115.71), SIMDE_FLOAT64_C( -206.04),
                         SIMDE_FLOAT64_C( -581.48), SIMDE_FLOAT64_C(  670.36),
                         SIMDE_FLOAT64_C(  115.71), SIMDE_FLOAT64_C( -206.04),
                         SIMDE_FLOAT64_C( -581.48), SIMDE_FLOAT64_C(  670.36)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C(  297.45), SIMDE_FLOAT64_C(  193.39),
                         SIMDE_FLOAT64_C( -163.24), SIMDE_FLOAT64_C( -775.87)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  297.45), SIMDE_FLOAT64_C(  193.39),
                         SIMDE_FLOAT64_C( -163.24), SIMDE_FLOAT64_C( -775.87),
                         SIMDE_FLOAT64_C(  297.45), SIMDE_FLOAT64_C(  193.39),
                         SIMDE_FLOAT64_C( -163.24), SIMDE_FLOAT64_C( -775.87)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C( -675.37), SIMDE_FLOAT64_C(  853.20),
                         SIMDE_FLOAT64_C( -377.67), SIMDE_FLOAT64_C(  233.14)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -675.37), SIMDE_FLOAT64_C(  853.20),
                         SIMDE_FLOAT64_C( -377.67), SIMDE_FLOAT64_C(  233.14),
                         SIMDE_FLOAT64_C( -675.37), SIMDE_FLOAT64_C(  853.20),
                         SIMDE_FLOAT64_C( -377.67), SIMDE_FLOAT64_C(  233.14)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C( -156.08), SIMDE_FLOAT64_C( -209.26),
                         SIMDE_FLOAT64_C(   48.51), SIMDE_FLOAT64_C( -627.76)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -156.08), SIMDE_FLOAT64_C( -209.26),
                         SIMDE_FLOAT64_C(   48.51), SIMDE_FLOAT64_C( -627.76),
                         SIMDE_FLOAT64_C( -156.08), SIMDE_FLOAT64_C( -209.26),
                         SIMDE_FLOAT64_C(   48.51), SIMDE_FLOAT64_C( -627.76)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C(  108.40), SIMDE_FLOAT64_C(  970.37),
                         SIMDE_FLOAT64_C(  934.72), SIMDE_FLOAT64_C( -932.81)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  108.40), SIMDE_FLOAT64_C(  970.37),
                         SIMDE_FLOAT64_C(  934.72), SIMDE_FLOAT64_C( -932.81),
                         SIMDE_FLOAT64_C(  108.40), SIMDE_FLOAT64_C(  970.37),
                         SIMDE_FLOAT64_C(  934.72), SIMDE_FLOAT64_C( -932.81)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C(  690.58), SIMDE_FLOAT64_C(  836.42),
                         SIMDE_FLOAT64_C( -952.66), SIMDE_FLOAT64_C(   22.35)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  690.58), SIMDE_FLOAT64_C(  836.42),
                         SIMDE_FLOAT64_C( -952.66), SIMDE_FLOAT64_C(   22.35),
                         SIMDE_FLOAT64_C(  690.58), SIMDE_FLOAT64_C(  836.42),
                         SIMDE_FLOAT64_C( -952.66), SIMDE_FLOAT64_C(   22.35)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C(  740.10), SIMDE_FLOAT64_C(  159.65),
                         SIMDE_FLOAT64_C(  -65.49), SIMDE_FLOAT64_C(  946.83)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  740.10), SIMDE_FLOAT64_C(  159.65),
                         SIMDE_FLOAT64_C(  -65.49), SIMDE_FLOAT64_C(  946.83),
                         SIMDE_FLOAT64_C(  740.10), SIMDE_FLOAT64_C(  159.65),
                         SIMDE_FLOAT64_C(  -65.49), SIMDE_FLOAT64_C(  946.83)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_broadcast_f64x4(test_vec[i].a);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_mask_broadcast_f64x4(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512d src;
    simde__mmask8 k;
    simde__m256d a;
    simde__m512d r;
  } test_vec[8] = {
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -396.88), SIMDE_FLOAT64_C(  354.04),
                         SIMDE_FLOAT64_C(  268.06), SIMDE_FLOAT64_C( -972.10),
                         SIMDE_FLOAT64_C( -213.85), SIMDE_FLOAT64_C( -574.68),
                         SIMDE_FLOAT64_C(  137.99), SIMDE_FLOAT64_C(  420.83)),
      UINT8_C( 60),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  337.98), SIMDE_FLOAT64_C( -931.30),
                         SIMDE_FLOAT64_C(  -93.71), SIMDE_FLOAT64_C(  492.43)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -396.88), SIMDE_FLOAT64_C(  354.04),
                         SIMDE_FLOAT64_C(  -93.71), SIMDE_FLOAT64_C(  492.43),
                         SIMDE_FLOAT64_C(  337.98), SIMDE_FLOAT64_C( -931.30),
                         SIMDE_FLOAT64_C(  137.99), SIMDE_FLOAT64_C(  420.83)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -680.47), SIMDE_FLOAT64_C( -211.69),
                         SIMDE_FLOAT64_C(  879.50), SIMDE_FLOAT64_C(  245.88),
                         SIMDE_FLOAT64_C(  689.68), SIMDE_FLOAT64_C(  107.64),
                         SIMDE_FLOAT64_C( -872.56), SIMDE_FLOAT64_C( -586.10)),
      UINT8_C( 26),
      simde_mm256_set_pd(SIMDE_FLOAT64_C( -622.96), SIMDE_FLOAT64_C(  479.82),
                         SIMDE_FLOAT64_C( -652.18), SIMDE_FLOAT64_C(  585.66)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -680.47), SIMDE_FLOAT64_C( -211.69),
                         SIMDE_FLOAT64_C(  879.50), SIMDE_FLOAT64_C(  585.66),
                         SIMDE_FLOAT64_C( -622.96), SIMDE_FLOAT64_C(  107.64),
                         SIMDE_FLOAT64_C( -652.18), SIMDE_FLOAT64_C( -586.10)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  408.02), SIMDE_FLOAT64_C(  662.99),
                         SIMDE_FLOAT64_C( -491.44), SIMDE_FLOAT64_C( -586.97),
                         SIMDE_FLOAT64_C( -858.39), SIMDE_FLOAT64_C(  608.18),
                         SIMDE_FLOAT64_C(  129.78), SIMDE_FLOAT64_C( -779.98)),
      UINT8_C(174),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  469.58), SIMDE_FLOAT64_C( -229.18),
                         SIMDE_FLOAT64_C(  700.39), SIMDE_FLOAT64_C(  708.98)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  469.58), SIMDE_FLOAT64_C(  662.99),
                         SIMDE_FLOAT64_C(  700.39), SIMDE_FLOAT64_C( -586.97),
                         SIMDE_FLOAT64_C(  469.58), SIMDE_FLOAT64_C( -229.18),
                         SIMDE_FLOAT64_C(  700.39), SIMDE_FLOAT64_C( -779.98)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -491.65), SIMDE_FLOAT64_C(  792.34),
                         SIMDE_FLOAT64_C( -828.98), SIMDE_FLOAT64_C(  152.82),
                         SIMDE_FLOAT64_C(  261.49), SIMDE_FLOAT64_C( -674.96),
                         SIMDE_FLOAT64_C( -626.70), SIMDE_FLOAT64_C( -365.50)),
      UINT8_C(162),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  790.92), SIMDE_FLOAT64_C( -372.23),
                         SIMDE_FLOAT64_C( -362.18), SIMDE_FLOAT64_C(  725.62)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  790.92), SIMDE_FLOAT64_C(  792.34),
                         SIMDE_FLOAT64_C( -362.18), SIMDE_FLOAT64_C(  152.82),
                         SIMDE_FLOAT64_C(  261.49), SIMDE_FLOAT64_C( -674.96),
                         SIMDE_FLOAT64_C( -362.18), SIMDE_FLOAT64_C( -365.50)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -230.83), SIMDE_FLOAT64_C( -480.11),
                         SIMDE_FLOAT64_C(  511.94), SIMDE_FLOAT64_C(  614.74),
                         SIMDE_FLOAT64_C(  794.95), SIMDE_FLOAT64_C( -331.37),
                         SIMDE_FLOAT64_C( -632.83), SIMDE_FLOAT64_C( -181.94)),
      UINT8_C(187),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  125.71), SIMDE_FLOAT64_C( -159.95),
                         SIMDE_FLOAT64_C( -519.57), SIMDE_FLOAT64_C(  -66.62)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  125.71), SIMDE_FLOAT64_C( -480.11),
                         SIMDE_FLOAT64_C( -519.57), SIMDE_FLOAT64_C(  -66.62),
                         SIMDE_FLOAT64_C(  125.71), SIMDE_FLOAT64_C( -331.37),
                         SIMDE_FLOAT64_C( -519.57), SIMDE_FLOAT64_C(  -66.62)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  870.57), SIMDE_FLOAT64_C( -815.89),
                         SIMDE_FLOAT64_C(  351.22), SIMDE_FLOAT64_C( -739.81),
                         SIMDE_FLOAT64_C( -104.33), SIMDE_FLOAT64_C(  331.38),
                         SIMDE_FLOAT64_C(  749.42), SIMDE_FLOAT64_C(  151.95)),
      UINT8_C( 98),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  652.04), SIMDE_FLOAT64_C( -453.75),
                         SIMDE_FLOAT64_C(   56.24), SIMDE_FLOAT64_C(  506.86)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  870.57), SIMDE_FLOAT64_C( -453.75),
                         SIMDE_FLOAT64_C(   56.24), SIMDE_FLOAT64_C( -739.81),
                         SIMDE_FLOAT64_C( -104.33), SIMDE_FLOAT64_C(  331.38),
                         SIMDE_FLOAT64_C(   56.24), SIMDE_FLOAT64_C(  151.95)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -854.89), SIMDE_FLOAT64_C(  376.16),
                         SIMDE_FLOAT64_C( -846.26), SIMDE_FLOAT64_C(  817.65),
                         SIMDE_FLOAT64_C( -403.95), SIMDE_FLOAT64_C( -116.62),
                         SIMDE_FLOAT64_C(  -17.97), SIMDE_FLOAT64_C(  229.99)),
      UINT8_C( 85),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  -65.83), SIMDE_FLOAT64_C( -494.87),
                         SIMDE_FLOAT64_C(   57.37), SIMDE_FLOAT64_C(  347.13)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -854.89), SIMDE_FLOAT64_C( -494.87),
                         SIMDE_FLOAT64_C( -846.26), SIMDE_FLOAT64_C(  347.13),
                         SIMDE_FLOAT64_C( -403.95), SIMDE_FLOAT64_C( -494.87),
                         SIMDE_FLOAT64_C(  -17.97), SIMDE_FLOAT64_C(  347.13)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -700.83), SIMDE_FLOAT64_C( -289.50),
                         SIMDE_FLOAT64_C(  417.50), SIMDE_FLOAT64_C(  245.21),
                         SIMDE_FLOAT64_C(  960.01), SIMDE_FLOAT64_C( -303.61),
                         SIMDE_FLOAT64_C( -550.57), SIMDE_FLOAT64_C(  665.98)),
      UINT8_C(  5),
      simde_mm256_set_pd(SIMDE_FLOAT64_C( -596.78), SIMDE_FLOAT64_C(  840.69),
                         SIMDE_FLOAT64_C( -684.92), SIMDE_FLOAT64_C(   87.08)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -700.83), SIMDE_FLOAT64_C( -289.50),
                         SIMDE_FLOAT64_C(  417.50), SIMDE_FLOAT64_C(  245.21),
                         SIMDE_FLOAT64_C(  960.01), SIMDE_FLOAT64_C(  840.69),
                         SIMDE_FLOAT64_C( -550.57), SIMDE_FLOAT64_C(   87.08)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_mask_broadcast_f64x4(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_maskz_broadcast_f64x4(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask8 k;
    simde__m256d a;
    simde__m512d r;
  } test_vec[8] = {
    { UINT8_C( 25),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  -93.71), SIMDE_FLOAT64_C(  137.99),
                         SIMDE_FLOAT64_C(  492.43), SIMDE_FLOAT64_C(  420.83)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(  420.83),
                         SIMDE_FLOAT64_C(  -93.71), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(  420.83)) },
    { UINT8_C(223),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  354.04), SIMDE_FLOAT64_C( -261.67),
                         SIMDE_FLOAT64_C(  268.06), SIMDE_FLOAT64_C(  648.56)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  354.04), SIMDE_FLOAT64_C( -261.67),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(  648.56),
                         SIMDE_FLOAT64_C(  354.04), SIMDE_FLOAT64_C( -261.67),
                         SIMDE_FLOAT64_C(  268.06), SIMDE_FLOAT64_C(  648.56)) },
    { UINT8_C(191),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  107.64), SIMDE_FLOAT64_C( -652.18),
                         SIMDE_FLOAT64_C( -872.56), SIMDE_FLOAT64_C(  585.66)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  107.64), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C( -872.56), SIMDE_FLOAT64_C(  585.66),
                         SIMDE_FLOAT64_C(  107.64), SIMDE_FLOAT64_C( -652.18),
                         SIMDE_FLOAT64_C( -872.56), SIMDE_FLOAT64_C(  585.66)) },
    { UINT8_C( 77),
      simde_mm256_set_pd(SIMDE_FLOAT64_C( -476.82), SIMDE_FLOAT64_C( -211.69),
                         SIMDE_FLOAT64_C(  687.27), SIMDE_FLOAT64_C(  879.50)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C( -211.69),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C( -476.82), SIMDE_FLOAT64_C( -211.69),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(  879.50)) },
    { UINT8_C(216),
      simde_mm256_set_pd(SIMDE_FLOAT64_C( -229.18), SIMDE_FLOAT64_C(  608.18),
                         SIMDE_FLOAT64_C(  700.39), SIMDE_FLOAT64_C(  129.78)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -229.18), SIMDE_FLOAT64_C(  608.18),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(  129.78),
                         SIMDE_FLOAT64_C( -229.18), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00)) },
    { UINT8_C(196),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  408.02), SIMDE_FLOAT64_C( -213.85),
                         SIMDE_FLOAT64_C(  662.99), SIMDE_FLOAT64_C(  346.52)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  408.02), SIMDE_FLOAT64_C( -213.85),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C( -213.85),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00)) },
    { UINT8_C(125),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  261.49), SIMDE_FLOAT64_C( -372.23),
                         SIMDE_FLOAT64_C( -674.96), SIMDE_FLOAT64_C( -362.18)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C( -372.23),
                         SIMDE_FLOAT64_C( -674.96), SIMDE_FLOAT64_C( -362.18),
                         SIMDE_FLOAT64_C(  261.49), SIMDE_FLOAT64_C( -372.23),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C( -362.18)) },
    { UINT8_C( 95),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  475.10), SIMDE_FLOAT64_C( -491.65),
                         SIMDE_FLOAT64_C(  659.15), SIMDE_FLOAT64_C(  792.34)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C( -491.65),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(  792.34),
                         SIMDE_FLOAT64_C(  475.10), SIMDE_FLOAT64_C( -491.65),
                         SIMDE_FLOAT64_C(  659.15), SIMDE_FLOAT64_C(  792.34)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_maskz_broadcast_f64x4(test_vec[i].k, test_vec[i].a);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_broadcast_i32x4(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm_set_epi32(INT32_C( 1322912216), INT32_C( -192131569), INT32_C(  457247766), INT32_C( 1585478853)),
      simde_mm512_set_epi32(INT32_C( 1322912216), INT32_C( -192131569), INT32_C(  457247766), INT32_C( 1585478853),
                            INT32_C( 1322912216), INT32_C( -192131569), INT32_C(  457247766), INT32_C( 1585478853),
                            INT32_C( 1322912216), INT32_C( -192131569), INT32_C(  457247766), INT32_C( 1585478853),
                            INT32_C( 1322912216), INT32_C( -192131569), INT32_C(  457247766), INT32_C( 1585478853)) },
    { simde_mm_set_epi32(INT32_C(  455358584), INT32_C( -549958328), INT32_C( 1779282555), INT32_C(-1938144165)),
      simde_mm512_set_epi32(INT32_C(  455358584), INT32_C( -549958328), INT32_C( 1779282555), INT32_C(-1938144165),
                            INT32_C(  455358584), INT32_C( -549958328), INT32_C( 1779282555), INT32_C(-1938144165),
                            INT32_C(  455358584), INT32_C( -549958328), INT32_C( 1779282555), INT32_C(-1938144165),
                            INT32_C(  455358584), INT32_C( -549958328), INT32_C( 1779282555), INT32_C(-1938144165)) },
    { simde_mm_set_epi32(INT32_C(   35244693), INT32_C( -163894097), INT32_C(  -32854349), INT32_C(-1300832792)),
      simde_mm512_set_epi32(INT32_C(   35244693), INT32_C( -163894097), INT32_C(  -32854349), INT32_C(-1300832792),
                            INT32_C(   35244693), INT32_C( -163894097), INT32_C(  -32854349), INT32_C(-1300832792),
                            INT32_C(   35244693), INT32_C( -163894097), INT32_C(  -32854349), INT32_C(-1300832792),
                            INT32_C(   35244693), INT32_C( -163894097), INT32_C(  -32854349), INT32_C(-1300832792)) },
    { simde_mm_set_epi32(INT32_C( 1137728540), INT32_C( 1602744474), INT32_C( -610393021), INT32_C(-1810116300)),
      simde_mm512_set_epi32(INT32_C( 1137728540), INT32_C( 1602744474), INT32_C( -610393021), INT32_C(-1810116300),
                            INT32_C( 1137728540), INT32_C( 1602744474), INT32_C( -610393021), INT32_C(-1810116300),
                            INT32_C( 1137728540), INT32_C( 1602744474), INT32_C( -610393021), INT32_C(-1810116300),
                            INT32_C( 1137728540), INT32_C( 1602744474), INT32_C( -610393021), INT32_C(-1810116300)) },
    { simde_mm_set_epi32(INT32_C(-1023450780), INT32_C(  840494259), INT32_C(-1087383364), INT32_C(-1604779562)),
      simde_mm512_set_epi32(INT32_C(-1023450780), INT32_C(  840494259), INT32_C(-1087383364), INT32_C(-1604779562),
                            INT32_C(-1023450780), INT32_C(  840494259), INT32_C(-1087383364), INT32_C(-1604779562),
                            INT32_C(-1023450780), INT32_C(  840494259), INT32_C(-1087383364), INT32_C(-1604779562),
                            INT32_C(-1023450780), INT32_C(  840494259), INT32_C(-1087383364), INT32_C(-1604779562)) },
    { simde_mm_set_epi32(INT32_C( 1284866833), INT32_C(   27132707), INT32_C(-1597877982), INT32_C(-1252321438)),
      simde_mm512_set_epi32(INT32_C( 1284866833), INT32_C(   27132707), INT32_C(-1597877982), INT32_C(-1252321438),
                            INT32_C( 1284866833), INT32_C(   27132707), INT32_C(-1597877982), INT32_C(-1252321438),
                            INT32_C( 1284866833), INT32_C(   27132707), INT32_C(-1597877982), INT32_C(-1252321438),
                            INT32_C( 1284866833), INT32_C(   27132707), INT32_C(-1597877982), INT32_C(-1252321438)) },
    { simde_mm_set_epi32(INT32_C( -165954025), INT32_C(  878840386), INT32_C( -802596544), INT32_C( 1574139347)),
      simde_mm512_set_epi32(INT32_C( -165954025), INT32_C(  878840386), INT32_C( -802596544), INT32_C( 1574139347),
                            INT32_C( -165954025), INT32_C(  878840386), INT32_C( -802596544), INT32_C( 1574139347),
                            INT32_C( -165954025), INT32_C(  878840386), INT32_C( -802596544), INT32_C( 1574139347),
                            INT32_C( -165954025), INT32_C(  878840386), INT32_C( -802596544), INT32_C( 1574139347)) },
    { simde_mm_set_epi32(INT32_C( -602275056), INT32_C(-1823359312), INT32_C( 1232365699), INT32_C(  345237769)),
      simde_mm512_set_epi32(INT32_C( -602275056), INT32_C(-1823359312), INT32_C( 1232365699), INT32_C(  345237769),
                            INT32_C( -602275056), INT32_C(-1823359312), INT32_C( 1232365699), INT32_C(  345237769),
                            INT32_C( -602275056), INT32_C(-1823359312), INT32_C( 1232365699), INT32_C(  345237769),
                            INT32_C( -602275056), INT32_C(-1823359312), INT32_C( 1232365699), INT32_C(  345237769)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_broadcast_i32x4(test_vec[i].a);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_broadcast_i32x4(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i src;
    simde__mmask16 k;
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C( 1479802474), INT32_C(  587294539), INT32_C( -174751528), INT32_C( 1465222154),
                            INT32_C( 1625882140), INT32_C(-1283973275), INT32_C(  567394727), INT32_C( 1808136008),
                            INT32_C(  324921956), INT32_C(-1888780980), INT32_C( -262803011), INT32_C( 2131227345),
                            INT32_C( -161180317), INT32_C( -569391310), INT32_C(  471197581), INT32_C( 2029664703)),
      UINT16_C(12686),
      simde_mm_set_epi32(INT32_C(-1929654026), INT32_C(-1217014324), INT32_C(  230292224), INT32_C( 1361651453)),
      simde_mm512_set_epi32(INT32_C( 1479802474), INT32_C(  587294539), INT32_C(  230292224), INT32_C( 1361651453),
                            INT32_C( 1625882140), INT32_C(-1283973275), INT32_C(  567394727), INT32_C( 1361651453),
                            INT32_C(-1929654026), INT32_C(-1888780980), INT32_C( -262803011), INT32_C( 2131227345),
                            INT32_C(-1929654026), INT32_C(-1217014324), INT32_C(  230292224), INT32_C( 2029664703)) },
    { simde_mm512_set_epi32(INT32_C( 1958214116), INT32_C( 2124258263), INT32_C(-1603442041), INT32_C(-1137458903),
                            INT32_C( -291704812), INT32_C( -523349105), INT32_C( -769676631), INT32_C(  359038153),
                            INT32_C( -860324016), INT32_C(  142975746), INT32_C( 1871956670), INT32_C(-2122929741),
                            INT32_C( 1007202856), INT32_C(-1693638626), INT32_C(-1497430440), INT32_C(  766142674)),
      UINT16_C( 3460),
      simde_mm_set_epi32(INT32_C(-1801778632), INT32_C(  793094568), INT32_C(  739597071), INT32_C( 1855829690)),
      simde_mm512_set_epi32(INT32_C( 1958214116), INT32_C( 2124258263), INT32_C(-1603442041), INT32_C(-1137458903),
                            INT32_C(-1801778632), INT32_C(  793094568), INT32_C( -769676631), INT32_C( 1855829690),
                            INT32_C(-1801778632), INT32_C(  142975746), INT32_C( 1871956670), INT32_C(-2122929741),
                            INT32_C( 1007202856), INT32_C(  793094568), INT32_C(-1497430440), INT32_C(  766142674)) },
    { simde_mm512_set_epi32(INT32_C( -491998875), INT32_C( -465346847), INT32_C( 1096008422), INT32_C( -151618100),
                            INT32_C( -483382033), INT32_C(-1500806456), INT32_C(  175505846), INT32_C( -698441328),
                            INT32_C( -515513970), INT32_C( 1679973349), INT32_C(-1523347194), INT32_C(   91392241),
                            INT32_C( -561919749), INT32_C( -634254878), INT32_C( -625316172), INT32_C(  -17019235)),
      UINT16_C(25030),
      simde_mm_set_epi32(INT32_C( -839244820), INT32_C(-1678825378), INT32_C(  464598558), INT32_C(-1198702193)),
      simde_mm512_set_epi32(INT32_C( -491998875), INT32_C(-1678825378), INT32_C(  464598558), INT32_C( -151618100),
                            INT32_C( -483382033), INT32_C(-1500806456), INT32_C(  175505846), INT32_C(-1198702193),
                            INT32_C( -839244820), INT32_C(-1678825378), INT32_C(-1523347194), INT32_C(   91392241),
                            INT32_C( -561919749), INT32_C(-1678825378), INT32_C(  464598558), INT32_C(  -17019235)) },
    { simde_mm512_set_epi32(INT32_C( 1319681857), INT32_C(  649867282), INT32_C(-1955467744), INT32_C(-1687114005),
                            INT32_C(-1950655074), INT32_C(-2040429697), INT32_C( 1764915437), INT32_C(  813475409),
                            INT32_C(-1622276195), INT32_C(  614665853), INT32_C( -661145222), INT32_C(  -43416876),
                            INT32_C(  954392932), INT32_C(-1003825870), INT32_C( -858676034), INT32_C( 1589986539)),
      UINT16_C(29308),
      simde_mm_set_epi32(INT32_C(-1945617369), INT32_C( -313192838), INT32_C( -614227976), INT32_C(  -73637500)),
      simde_mm512_set_epi32(INT32_C( 1319681857), INT32_C( -313192838), INT32_C( -614227976), INT32_C(  -73637500),
                            INT32_C(-1950655074), INT32_C(-2040429697), INT32_C( -614227976), INT32_C(  813475409),
                            INT32_C(-1622276195), INT32_C( -313192838), INT32_C( -614227976), INT32_C(  -73637500),
                            INT32_C(-1945617369), INT32_C( -313192838), INT32_C( -858676034), INT32_C( 1589986539)) },
    { simde_mm512_set_epi32(INT32_C(  482652005), INT32_C( 1083073699), INT32_C( -547163888), INT32_C(-1439583577),
                            INT32_C( -836573741), INT32_C(-2032318592), INT32_C( 1307381638), INT32_C( 2027662416),
                            INT32_C( 2001285861), INT32_C( 1074543972), INT32_C(-2107097596), INT32_C(-2025611729),
                            INT32_C(  962055101), INT32_C( 1886777199), INT32_C( 1689643613), INT32_C(-1874481648)),
      UINT16_C(45428),
      simde_mm_set_epi32(INT32_C(  110278011), INT32_C(-1940227644), INT32_C(-1803195700), INT32_C( 1287862649)),
      simde_mm512_set_epi32(INT32_C(  110278011), INT32_C( 1083073699), INT32_C(-1803195700), INT32_C( 1287862649),
                            INT32_C( -836573741), INT32_C(-2032318592), INT32_C( 1307381638), INT32_C( 1287862649),
                            INT32_C( 2001285861), INT32_C(-1940227644), INT32_C(-1803195700), INT32_C( 1287862649),
                            INT32_C(  962055101), INT32_C(-1940227644), INT32_C( 1689643613), INT32_C(-1874481648)) },
    { simde_mm512_set_epi32(INT32_C(  485695865), INT32_C( 1704586743), INT32_C(-1227241134), INT32_C(  279727823),
                            INT32_C( -480355834), INT32_C( 1374909005), INT32_C(-1706379633), INT32_C( 1300025155),
                            INT32_C( 1901096153), INT32_C(-1845297076), INT32_C(  188971064), INT32_C( 1903842318),
                            INT32_C(-1221674473), INT32_C(-1332164211), INT32_C(   23564349), INT32_C(-2098316192)),
      UINT16_C(21964),
      simde_mm_set_epi32(INT32_C(-1820692848), INT32_C( -830585945), INT32_C( 1667959054), INT32_C(-1758734041)),
      simde_mm512_set_epi32(INT32_C(  485695865), INT32_C( -830585945), INT32_C(-1227241134), INT32_C(-1758734041),
                            INT32_C( -480355834), INT32_C( -830585945), INT32_C(-1706379633), INT32_C(-1758734041),
                            INT32_C(-1820692848), INT32_C( -830585945), INT32_C(  188971064), INT32_C( 1903842318),
                            INT32_C(-1820692848), INT32_C( -830585945), INT32_C(   23564349), INT32_C(-2098316192)) },
    { simde_mm512_set_epi32(INT32_C(-1876069406), INT32_C( 1820341222), INT32_C(  987166931), INT32_C(-1021572249),
                            INT32_C(-1046533173), INT32_C(-1808511518), INT32_C( -283777637), INT32_C( -168486656),
                            INT32_C( 1250903497), INT32_C( 1175614584), INT32_C(  204391673), INT32_C( -667659280),
                            INT32_C( 2035348040), INT32_C( -596829354), INT32_C(-1607289004), INT32_C( -670488239)),
      UINT16_C(31159),
      simde_mm_set_epi32(INT32_C(-1492076939), INT32_C( 1502879171), INT32_C( 1497885207), INT32_C(-1325620059)),
      simde_mm512_set_epi32(INT32_C(-1876069406), INT32_C( 1502879171), INT32_C( 1497885207), INT32_C(-1325620059),
                            INT32_C(-1492076939), INT32_C(-1808511518), INT32_C( -283777637), INT32_C(-1325620059),
                            INT32_C(-1492076939), INT32_C( 1175614584), INT32_C( 1497885207), INT32_C(-1325620059),
                            INT32_C( 2035348040), INT32_C( 1502879171), INT32_C( 1497885207), INT32_C(-1325620059)) },
    { simde_mm512_set_epi32(INT32_C(-1346174896), INT32_C( 1223712250), INT32_C( 2029339086), INT32_C( 2108949315),
                            INT32_C(-1822742445), INT32_C( -343433299), INT32_C(-1626119528), INT32_C( 1735301543),
                            INT32_C(  766111295), INT32_C(  -80424103), INT32_C( 1232059506), INT32_C(-1681875170),
                            INT32_C( 1819208351), INT32_C( -734074357), INT32_C(   61937468), INT32_C(-1403575087)),
      UINT16_C(37926),
      simde_mm_set_epi32(INT32_C( 1656599178), INT32_C( 1293315993), INT32_C( -728433677), INT32_C( -125533424)),
      simde_mm512_set_epi32(INT32_C( 1656599178), INT32_C( 1223712250), INT32_C( 2029339086), INT32_C( -125533424),
                            INT32_C(-1822742445), INT32_C( 1293315993), INT32_C(-1626119528), INT32_C( 1735301543),
                            INT32_C(  766111295), INT32_C(  -80424103), INT32_C( -728433677), INT32_C(-1681875170),
                            INT32_C( 1819208351), INT32_C( 1293315993), INT32_C( -728433677), INT32_C(-1403575087)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_broadcast_i32x4(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_broadcast_i32x4(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask16 k;
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { UINT16_C(57503),
      simde_mm_set_epi32(INT32_C(  913371223), INT32_C( 1946242675), INT32_C(-1851162974), INT32_C(-1090004303)),
      simde_mm512_set_epi32(INT32_C(  913371223), INT32_C( 1946242675), INT32_C(-1851162974), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(  913371223), INT32_C(          0), INT32_C(          0), INT32_C(-1090004303),
                            INT32_C(  913371223), INT32_C( 1946242675), INT32_C(-1851162974), INT32_C(-1090004303)) },
    { UINT16_C( 9830),
      simde_mm_set_epi32(INT32_C( -754702866), INT32_C(   59910169), INT32_C(-1421684089), INT32_C( 1688249563)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(-1421684089), INT32_C(          0),
                            INT32_C(          0), INT32_C(   59910169), INT32_C(-1421684089), INT32_C(          0),
                            INT32_C(          0), INT32_C(   59910169), INT32_C(-1421684089), INT32_C(          0),
                            INT32_C(          0), INT32_C(   59910169), INT32_C(-1421684089), INT32_C(          0)) },
    { UINT16_C(54973),
      simde_mm_set_epi32(INT32_C( 1295192258), INT32_C( 2064350366), INT32_C(-1387191485), INT32_C( 1585557386)),
      simde_mm512_set_epi32(INT32_C( 1295192258), INT32_C( 2064350366), INT32_C(          0), INT32_C( 1585557386),
                            INT32_C(          0), INT32_C( 2064350366), INT32_C(-1387191485), INT32_C(          0),
                            INT32_C( 1295192258), INT32_C(          0), INT32_C(-1387191485), INT32_C( 1585557386),
                            INT32_C( 1295192258), INT32_C( 2064350366), INT32_C(          0), INT32_C( 1585557386)) },
    { UINT16_C( 2571),
      simde_mm_set_epi32(INT32_C(  273665101), INT32_C( -889778981), INT32_C(  888851167), INT32_C(  342766140)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(  273665101), INT32_C(          0), INT32_C(  888851167), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(  273665101), INT32_C(          0), INT32_C(  888851167), INT32_C(  342766140)) },
    { UINT16_C(34156),
      simde_mm_set_epi32(INT32_C(  809684493), INT32_C( -666403540), INT32_C(-1117073828), INT32_C(-1916337185)),
      simde_mm512_set_epi32(INT32_C(  809684493), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C( -666403540), INT32_C(          0), INT32_C(-1916337185),
                            INT32_C(          0), INT32_C( -666403540), INT32_C(-1117073828), INT32_C(          0),
                            INT32_C(  809684493), INT32_C( -666403540), INT32_C(          0), INT32_C(          0)) },
    { UINT16_C( 6544),
      simde_mm_set_epi32(INT32_C( 1692879261), INT32_C( -671588299), INT32_C( -258764942), INT32_C(-1633977409)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(-1633977409),
                            INT32_C( 1692879261), INT32_C(          0), INT32_C(          0), INT32_C(-1633977409),
                            INT32_C( 1692879261), INT32_C(          0), INT32_C(          0), INT32_C(-1633977409),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0)) },
    { UINT16_C(45909),
      simde_mm_set_epi32(INT32_C(  472486650), INT32_C( 1238366490), INT32_C(-1084360471), INT32_C(  686181072)),
      simde_mm512_set_epi32(INT32_C(  472486650), INT32_C(          0), INT32_C(-1084360471), INT32_C(  686181072),
                            INT32_C(          0), INT32_C(          0), INT32_C(-1084360471), INT32_C(  686181072),
                            INT32_C(          0), INT32_C( 1238366490), INT32_C(          0), INT32_C(  686181072),
                            INT32_C(          0), INT32_C( 1238366490), INT32_C(          0), INT32_C(  686181072)) },
    { UINT16_C(56653),
      simde_mm_set_epi32(INT32_C( 1655322598), INT32_C( -841418169), INT32_C( -643403227), INT32_C(-1868778842)),
      simde_mm512_set_epi32(INT32_C( 1655322598), INT32_C( -841418169), INT32_C(          0), INT32_C(-1868778842),
                            INT32_C( 1655322598), INT32_C( -841418169), INT32_C(          0), INT32_C(-1868778842),
                            INT32_C(          0), INT32_C( -841418169), INT32_C(          0), INT32_C(          0),
                            INT32_C( 1655322598), INT32_C( -841418169), INT32_C(          0), INT32_C(-1868778842)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_broadcast_i32x4(test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_broadcast_i64x4(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm256_set_epi64x(INT64_C( 2067253863170152603), INT64_C( 7322969156688688496),
                             INT64_C(-3040413397780943697), INT64_C( -347515311309491350)),
      simde_mm512_set_epi64(INT64_C( 2067253863170152603), INT64_C( 7322969156688688496),
                            INT64_C(-3040413397780943697), INT64_C( -347515311309491350),
                            INT64_C( 2067253863170152603), INT64_C( 7322969156688688496),
                            INT64_C(-3040413397780943697), INT64_C( -347515311309491350)) },
    { simde_mm256_set_epi64x(INT64_C(-8775907405261856642), INT64_C( 2994184764454707691),
                             INT64_C( 5740004668815682638), INT64_C(-6479861669953478300)),
      simde_mm512_set_epi64(INT64_C(-8775907405261856642), INT64_C( 2994184764454707691),
                            INT64_C( 5740004668815682638), INT64_C(-6479861669953478300),
                            INT64_C(-8775907405261856642), INT64_C( 2994184764454707691),
                            INT64_C( 5740004668815682638), INT64_C(-6479861669953478300)) },
    { simde_mm256_set_epi64x(INT64_C(-1508734178901937051), INT64_C(-9017252864562564261),
                             INT64_C( -273279204292504060), INT64_C(  619750219118375084)),
      simde_mm512_set_epi64(INT64_C(-1508734178901937051), INT64_C(-9017252864562564261),
                            INT64_C( -273279204292504060), INT64_C(  619750219118375084),
                            INT64_C(-1508734178901937051), INT64_C(-9017252864562564261),
                            INT64_C( -273279204292504060), INT64_C(  619750219118375084)) },
    { simde_mm256_set_epi64x(INT64_C( 5726987144774798582), INT64_C(-5242976599564634972),
                             INT64_C(-2397121704692329659), INT64_C( 8619348224440898856)),
      simde_mm512_set_epi64(INT64_C( 5726987144774798582), INT64_C(-5242976599564634972),
                            INT64_C(-2397121704692329659), INT64_C( 8619348224440898856),
                            INT64_C( 5726987144774798582), INT64_C(-5242976599564634972),
                            INT64_C(-2397121704692329659), INT64_C( 8619348224440898856)) },
    { simde_mm256_set_epi64x(INT64_C( 3770039990400590046), INT64_C(-4228023324121815234),
                             INT64_C(-2554402032947045809), INT64_C(-5734730006803594733)),
      simde_mm512_set_epi64(INT64_C( 3770039990400590046), INT64_C(-4228023324121815234),
                            INT64_C(-2554402032947045809), INT64_C(-5734730006803594733),
                            INT64_C( 3770039990400590046), INT64_C(-4228023324121815234),
                            INT64_C(-2554402032947045809), INT64_C(-5734730006803594733)) },
    { simde_mm256_set_epi64x(INT64_C(-7969300362390541280), INT64_C( 5131273406597805369),
                             INT64_C( 3164578103377175393), INT64_C( -896289702737256643)),
      simde_mm512_set_epi64(INT64_C(-7969300362390541280), INT64_C( 5131273406597805369),
                            INT64_C( 3164578103377175393), INT64_C( -896289702737256643),
                            INT64_C(-7969300362390541280), INT64_C( 5131273406597805369),
                            INT64_C( 3164578103377175393), INT64_C( -896289702737256643)) },
    { simde_mm256_set_epi64x(INT64_C( 6358202424481672256), INT64_C(-2088789378195753898),
                             INT64_C(-3832720361616382569), INT64_C(-1395499602347228816)),
      simde_mm512_set_epi64(INT64_C( 6358202424481672256), INT64_C(-2088789378195753898),
                            INT64_C(-3832720361616382569), INT64_C(-1395499602347228816),
                            INT64_C( 6358202424481672256), INT64_C(-2088789378195753898),
                            INT64_C(-3832720361616382569), INT64_C(-1395499602347228816)) },
    { simde_mm256_set_epi64x(INT64_C(-7005415045902450329), INT64_C(  454800303112400674),
                             INT64_C(  120562593220559221), INT64_C(-9183341893829321065)),
      simde_mm512_set_epi64(INT64_C(-7005415045902450329), INT64_C(  454800303112400674),
                            INT64_C(  120562593220559221), INT64_C(-9183341893829321065),
                            INT64_C(-7005415045902450329), INT64_C(  454800303112400674),
                            INT64_C(  120562593220559221), INT64_C(-9183341893829321065)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_broadcast_i64x4(test_vec[i].a);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_broadcast_i64x4(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i src;
    simde__mmask8 k;
    simde__m256i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C(-6314317108894035774), INT64_C( 8866317312363406147),
                            INT64_C( 6809917121524389565), INT64_C(-3241424127607560167),
                            INT64_C(-6106086665810303781), INT64_C(  633642393017577559),
                            INT64_C( 8359048641648361122), INT64_C(-4681532830833057633)),
      UINT8_C( 60),
      simde_mm256_set_epi64x(INT64_C( 3477568421141904684), INT64_C(-4797795556098898977),
                             INT64_C( 3208117008747973709), INT64_C(-3821571623174354209)),
      simde_mm512_set_epi64(INT64_C(-6314317108894035774), INT64_C( 8866317312363406147),
                            INT64_C( 3208117008747973709), INT64_C(-3821571623174354209),
                            INT64_C( 3477568421141904684), INT64_C(-4797795556098898977),
                            INT64_C( 8359048641648361122), INT64_C(-4681532830833057633)) },
    { simde_mm512_set_epi64(INT64_C( 1306125493676423142), INT64_C(-3613863514463636955),
                            INT64_C(-8026344006176744115), INT64_C( 2029314710784964890),
                            INT64_C(-4657292759333975344), INT64_C( 4825522705097247133),
                            INT64_C(-2884449776545067150), INT64_C(-7017879531382302320)),
      UINT8_C(  0),
      simde_mm256_set_epi64x(INT64_C(-4405193415265233332), INT64_C( 7250935849068321562),
                             INT64_C(-6027293339582699304), INT64_C(-3733599027822978693)),
      simde_mm512_set_epi64(INT64_C( 1306125493676423142), INT64_C(-3613863514463636955),
                            INT64_C(-8026344006176744115), INT64_C( 2029314710784964890),
                            INT64_C(-4657292759333975344), INT64_C( 4825522705097247133),
                            INT64_C(-2884449776545067150), INT64_C(-7017879531382302320)) },
    { simde_mm512_set_epi64(INT64_C( 4688717956956220153), INT64_C(-1915316091557446787),
                            INT64_C( 1577347929723399506), INT64_C(-7813885322626023749),
                            INT64_C(-6811547529988353683), INT64_C( 2997984888778655645),
                            INT64_C( 3443124806434765346), INT64_C( 5852240145563215278)),
      UINT8_C(246),
      simde_mm256_set_epi64x(INT64_C(-1891210360757244537), INT64_C( 6167039147883013727),
                             INT64_C( 3386552444698298512), INT64_C( 7545310155849572514)),
      simde_mm512_set_epi64(INT64_C(-1891210360757244537), INT64_C( 6167039147883013727),
                            INT64_C( 3386552444698298512), INT64_C( 7545310155849572514),
                            INT64_C(-6811547529988353683), INT64_C( 6167039147883013727),
                            INT64_C( 3386552444698298512), INT64_C( 5852240145563215278)) },
    { simde_mm512_set_epi64(INT64_C( 5038277295705077786), INT64_C(-8704670477732479640),
                            INT64_C(-4548397220420700343), INT64_C( 8046739269734052975),
                            INT64_C( 7094379553694909752), INT64_C( 4795143479989329521),
                            INT64_C(-4501545483124413586), INT64_C(-3553418787378740418)),
      UINT8_C( 91),
      simde_mm256_set_epi64x(INT64_C(-1193819960890806229), INT64_C( 1698145641448748604),
                             INT64_C(-5983907472113043464), INT64_C( 2399871967268573321)),
      simde_mm512_set_epi64(INT64_C( 5038277295705077786), INT64_C( 1698145641448748604),
                            INT64_C(-4548397220420700343), INT64_C( 2399871967268573321),
                            INT64_C(-1193819960890806229), INT64_C( 4795143479989329521),
                            INT64_C(-5983907472113043464), INT64_C( 2399871967268573321)) },
    { simde_mm512_set_epi64(INT64_C(-7015430497800685262), INT64_C( 6395476272833483099),
                            INT64_C(-7658177893206805688), INT64_C( 8616202346974378134),
                            INT64_C( 4658965153462790469), INT64_C(-8694270525310808014),
                            INT64_C(-6021620893121233714), INT64_C(-2734912706905093379)),
      UINT8_C(200),
      simde_mm256_set_epi64x(INT64_C(-1469383970610000896), INT64_C( 2906056864364420569),
                             INT64_C(-8420208282727167471), INT64_C(-2445653243165948933)),
      simde_mm512_set_epi64(INT64_C(-1469383970610000896), INT64_C( 2906056864364420569),
                            INT64_C(-7658177893206805688), INT64_C( 8616202346974378134),
                            INT64_C(-1469383970610000896), INT64_C(-8694270525310808014),
                            INT64_C(-6021620893121233714), INT64_C(-2734912706905093379)) },
    { simde_mm512_set_epi64(INT64_C(  -14573144697473529), INT64_C( 8194534140513027918),
                            INT64_C( 2864848388614962181), INT64_C(-8899252041456864412),
                            INT64_C( 6379752944219310901), INT64_C(-1860193003353627344),
                            INT64_C(-6904865090556452860), INT64_C( 3719036040063860682)),
      UINT8_C(242),
      simde_mm256_set_epi64x(INT64_C(-5869124324801971655), INT64_C( 4548184433513821860),
                             INT64_C( -866976878921007676), INT64_C( 2203520398864570966)),
      simde_mm512_set_epi64(INT64_C(-5869124324801971655), INT64_C( 4548184433513821860),
                            INT64_C( -866976878921007676), INT64_C( 2203520398864570966),
                            INT64_C( 6379752944219310901), INT64_C(-1860193003353627344),
                            INT64_C( -866976878921007676), INT64_C( 3719036040063860682)) },
    { simde_mm512_set_epi64(INT64_C( 6933317985964373307), INT64_C(-7912084547370987750),
                            INT64_C( 1434122569595023374), INT64_C(  372849821895528123),
                            INT64_C( -797096709674116855), INT64_C( 7124042714150240897),
                            INT64_C(  192820077199458500), INT64_C( 2333974304098521090)),
      UINT8_C(243),
      simde_mm256_set_epi64x(INT64_C( 9216760499566437432), INT64_C(-3611239802138142732),
                             INT64_C(-4586686018735308980), INT64_C(-4383556822793463465)),
      simde_mm512_set_epi64(INT64_C( 9216760499566437432), INT64_C(-3611239802138142732),
                            INT64_C(-4586686018735308980), INT64_C(-4383556822793463465),
                            INT64_C( -797096709674116855), INT64_C( 7124042714150240897),
                            INT64_C(-4586686018735308980), INT64_C(-4383556822793463465)) },
    { simde_mm512_set_epi64(INT64_C(-1447537183271280169), INT64_C( 3992622506060288146),
                            INT64_C(-4043997837551953925), INT64_C( 6303477149728220498),
                            INT64_C( 7148655265583700891), INT64_C(-2780283900793463061),
                            INT64_C( 3296623181868458839), INT64_C( 3808941703531633947)),
      UINT8_C(191),
      simde_mm256_set_epi64x(INT64_C( 4775871390633368548), INT64_C( 1184569154591270183),
                             INT64_C(-1750343127516454914), INT64_C( 3950749388527391085)),
      simde_mm512_set_epi64(INT64_C( 4775871390633368548), INT64_C( 3992622506060288146),
                            INT64_C(-1750343127516454914), INT64_C( 3950749388527391085),
                            INT64_C( 4775871390633368548), INT64_C( 1184569154591270183),
                            INT64_C(-1750343127516454914), INT64_C( 3950749388527391085)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_broadcast_i64x4(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_broadcast_i64x4(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask8 k;
    simde__m256i a;
    simde__m512i r;
  } test_vec[8] = {
    { UINT8_C( 81),
      simde_mm256_set_epi64x(INT64_C(-3226888659503117201), INT64_C( 7490209482650655404),
                             INT64_C(-9179276487306987344), INT64_C( 7055682156038845095)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C( 7490209482650655404),
                            INT64_C(                   0), INT64_C( 7055682156038845095),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C( 7055682156038845095)) },
    { UINT8_C(230),
      simde_mm256_set_epi64x(INT64_C( 6952848743567724070), INT64_C( 6398498157984007660),
                             INT64_C(-7276216502972313781), INT64_C( 4842545408380684085)),
      simde_mm512_set_epi64(INT64_C( 6952848743567724070), INT64_C( 6398498157984007660),
                            INT64_C(-7276216502972313781), INT64_C(                   0),
                            INT64_C(                   0), INT64_C( 6398498157984007660),
                            INT64_C(-7276216502972313781), INT64_C(                   0)) },
    { UINT8_C(115),
      simde_mm256_set_epi64x(INT64_C( -147426939517817059), INT64_C(-3374766540151601501),
                             INT64_C( 9013437962204473886), INT64_C( 2290211861166994880)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(-3374766540151601501),
                            INT64_C( 9013437962204473886), INT64_C( 2290211861166994880),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C( 9013437962204473886), INT64_C( 2290211861166994880)) },
    { UINT8_C(102),
      simde_mm256_set_epi64x(INT64_C(-8700458333795307779), INT64_C(-9147297996573979024),
                             INT64_C(-3649385965919135635), INT64_C( 1818037113458506686)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(-9147297996573979024),
                            INT64_C(-3649385965919135635), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(-9147297996573979024),
                            INT64_C(-3649385965919135635), INT64_C(                   0)) },
    { UINT8_C( 59),
      simde_mm256_set_epi64x(INT64_C( 8763762661767364639), INT64_C(-7194784414741958081),
                             INT64_C(-1605849263772874289), INT64_C(-2187551180549076287)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                   0),
                            INT64_C(-1605849263772874289), INT64_C(-2187551180549076287),
                            INT64_C( 8763762661767364639), INT64_C(                   0),
                            INT64_C(-1605849263772874289), INT64_C(-2187551180549076287)) },
    { UINT8_C(119),
      simde_mm256_set_epi64x(INT64_C( 3282428208913039389), INT64_C(-2887297167729747289),
                             INT64_C( 6938672003976555894), INT64_C(-3765766577293323049)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(-2887297167729747289),
                            INT64_C( 6938672003976555894), INT64_C(-3765766577293323049),
                            INT64_C(                   0), INT64_C(-2887297167729747289),
                            INT64_C( 6938672003976555894), INT64_C(-3765766577293323049)) },
    { UINT8_C( 25),
      simde_mm256_set_epi64x(INT64_C(-4802008903577488206), INT64_C(-3983516919532966210),
                             INT64_C(-4702094198572773446), INT64_C( -958715043139892800)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C( -958715043139892800),
                            INT64_C(-4802008903577488206), INT64_C(                   0),
                            INT64_C(                   0), INT64_C( -958715043139892800)) },
    { UINT8_C(207),
      simde_mm256_set_epi64x(INT64_C( 2289318697780797186), INT64_C(-4515948424499803858),
                             INT64_C( 7316310196690749623), INT64_C( 4937967944726422430)),
      simde_mm512_set_epi64(INT64_C( 2289318697780797186), INT64_C(-4515948424499803858),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C( 2289318697780797186), INT64_C(-4515948424499803858),
                            INT64_C( 7316310196690749623), INT64_C( 4937967944726422430)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_broadcast_i64x4(test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_broadcastd_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm_set_epi32(INT32_C(-1051270324), INT32_C(-1977183446), INT32_C( -548195640), INT32_C(-1363461466)),
      simde_mm512_set_epi32(INT32_C(-1363461466), INT32_C(-1363461466), INT32_C(-1363461466), INT32_C(-1363461466),
                            INT32_C(-1363461466), INT32_C(-1363461466), INT32_C(-1363461466), INT32_C(-1363461466),
                            INT32_C(-1363461466), INT32_C(-1363461466), INT32_C(-1363461466), INT32_C(-1363461466),
                            INT32_C(-1363461466), INT32_C(-1363461466), INT32_C(-1363461466), INT32_C(-1363461466)) },
    { simde_mm_set_epi32(INT32_C(  979094891), INT32_C(  416506319), INT32_C( 2123490297), INT32_C(  200388421)),
      simde_mm512_set_epi32(INT32_C(  200388421), INT32_C(  200388421), INT32_C(  200388421), INT32_C(  200388421),
                            INT32_C(  200388421), INT32_C(  200388421), INT32_C(  200388421), INT32_C(  200388421),
                            INT32_C(  200388421), INT32_C(  200388421), INT32_C(  200388421), INT32_C(  200388421),
                            INT32_C(  200388421), INT32_C(  200388421), INT32_C(  200388421), INT32_C(  200388421)) },
    { simde_mm_set_epi32(INT32_C( 1927260635), INT32_C( 1201458882), INT32_C(-1448742498), INT32_C(-1111904220)),
      simde_mm512_set_epi32(INT32_C(-1111904220), INT32_C(-1111904220), INT32_C(-1111904220), INT32_C(-1111904220),
                            INT32_C(-1111904220), INT32_C(-1111904220), INT32_C(-1111904220), INT32_C(-1111904220),
                            INT32_C(-1111904220), INT32_C(-1111904220), INT32_C(-1111904220), INT32_C(-1111904220),
                            INT32_C(-1111904220), INT32_C(-1111904220), INT32_C(-1111904220), INT32_C(-1111904220)) },
    { simde_mm_set_epi32(INT32_C( -976455818), INT32_C(  542613123), INT32_C(  -15911923), INT32_C( -562895064)),
      simde_mm512_set_epi32(INT32_C( -562895064), INT32_C( -562895064), INT32_C( -562895064), INT32_C( -562895064),
                            INT32_C( -562895064), INT32_C( -562895064), INT32_C( -562895064), INT32_C( -562895064),
                            INT32_C( -562895064), INT32_C( -562895064), INT32_C( -562895064), INT32_C( -562895064),
                            INT32_C( -562895064), INT32_C( -562895064), INT32_C( -562895064), INT32_C( -562895064)) },
    { simde_mm_set_epi32(INT32_C(  836747087), INT32_C(-1431045412), INT32_C(-1356396683), INT32_C( 1489138473)),
      simde_mm512_set_epi32(INT32_C( 1489138473), INT32_C( 1489138473), INT32_C( 1489138473), INT32_C( 1489138473),
                            INT32_C( 1489138473), INT32_C( 1489138473), INT32_C( 1489138473), INT32_C( 1489138473),
                            INT32_C( 1489138473), INT32_C( 1489138473), INT32_C( 1489138473), INT32_C( 1489138473),
                            INT32_C( 1489138473), INT32_C( 1489138473), INT32_C( 1489138473), INT32_C( 1489138473)) },
    { simde_mm_set_epi32(INT32_C(-1783426961), INT32_C( -263517415), INT32_C(-1697630001), INT32_C( 2025142863)),
      simde_mm512_set_epi32(INT32_C( 2025142863), INT32_C( 2025142863), INT32_C( 2025142863), INT32_C( 2025142863),
                            INT32_C( 2025142863), INT32_C( 2025142863), INT32_C( 2025142863), INT32_C( 2025142863),
                            INT32_C( 2025142863), INT32_C( 2025142863), INT32_C( 2025142863), INT32_C( 2025142863),
                            INT32_C( 2025142863), INT32_C( 2025142863), INT32_C( 2025142863), INT32_C( 2025142863)) },
    { simde_mm_set_epi32(INT32_C(  300619496), INT32_C( -659754204), INT32_C(-1019736463), INT32_C( 1022872166)),
      simde_mm512_set_epi32(INT32_C( 1022872166), INT32_C( 1022872166), INT32_C( 1022872166), INT32_C( 1022872166),
                            INT32_C( 1022872166), INT32_C( 1022872166), INT32_C( 1022872166), INT32_C( 1022872166),
                            INT32_C( 1022872166), INT32_C( 1022872166), INT32_C( 1022872166), INT32_C( 1022872166),
                            INT32_C( 1022872166), INT32_C( 1022872166), INT32_C( 1022872166), INT32_C( 1022872166)) },
    { simde_mm_set_epi32(INT32_C( -274893610), INT32_C(  171227717), INT32_C( 1187872667), INT32_C( -590903223)),
      simde_mm512_set_epi32(INT32_C( -590903223), INT32_C( -590903223), INT32_C( -590903223), INT32_C( -590903223),
                            INT32_C( -590903223), INT32_C( -590903223), INT32_C( -590903223), INT32_C( -590903223),
                            INT32_C( -590903223), INT32_C( -590903223), INT32_C( -590903223), INT32_C( -590903223),
                            INT32_C( -590903223), INT32_C( -590903223), INT32_C( -590903223), INT32_C( -590903223)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_broadcastd_epi32(test_vec[i].a);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_broadcastd_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i src;
    simde__mmask16 k;
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C( 1638944021), INT32_C( -385149059), INT32_C(  852916680), INT32_C(-1839015366),
                            INT32_C( 1146921463), INT32_C(  765234486), INT32_C( -388218844), INT32_C(-1402803832),
                            INT32_C( 1245942358), INT32_C( 2001202713), INT32_C(  868062804), INT32_C(-1988191751),
                            INT32_C(  807099340), INT32_C(  -38350755), INT32_C( -443928971), INT32_C( -432554813)),
      UINT16_C(24594),
      simde_mm_set_epi32(INT32_C( -255909174), INT32_C(-1302917278), INT32_C(  327520540), INT32_C(  176606543)),
      simde_mm512_set_epi32(INT32_C( 1638944021), INT32_C(  176606543), INT32_C(  176606543), INT32_C(-1839015366),
                            INT32_C( 1146921463), INT32_C(  765234486), INT32_C( -388218844), INT32_C(-1402803832),
                            INT32_C( 1245942358), INT32_C( 2001202713), INT32_C(  868062804), INT32_C(  176606543),
                            INT32_C(  807099340), INT32_C(  -38350755), INT32_C(  176606543), INT32_C( -432554813)) },
    { simde_mm512_set_epi32(INT32_C( -115460801), INT32_C( 1889676725), INT32_C(    2912775), INT32_C(-1289469215),
                            INT32_C( 1033489041), INT32_C(  147853139), INT32_C(  706073024), INT32_C( -130092746),
                            INT32_C( -799642653), INT32_C(-1439962375), INT32_C(-1798405841), INT32_C( 1190396108),
                            INT32_C(-1013986568), INT32_C(  994541610), INT32_C(-1127995400), INT32_C( 1108325476)),
      UINT16_C(40849),
      simde_mm_set_epi32(INT32_C(  250706831), INT32_C( -936079925), INT32_C(-1129184131), INT32_C(  803417186)),
      simde_mm512_set_epi32(INT32_C(  803417186), INT32_C( 1889676725), INT32_C(    2912775), INT32_C(  803417186),
                            INT32_C(  803417186), INT32_C(  803417186), INT32_C(  803417186), INT32_C(  803417186),
                            INT32_C(  803417186), INT32_C(-1439962375), INT32_C(-1798405841), INT32_C(  803417186),
                            INT32_C(-1013986568), INT32_C(  994541610), INT32_C(-1127995400), INT32_C(  803417186)) },
    { simde_mm512_set_epi32(INT32_C(  357625867), INT32_C( -157238200), INT32_C(  909767636), INT32_C( 1422277073),
                            INT32_C( 2123935701), INT32_C(-1040550911), INT32_C(  686758291), INT32_C(-2090356905),
                            INT32_C( -362358815), INT32_C( -482453842), INT32_C(  117787421), INT32_C( 1300554279),
                            INT32_C(-1085613264), INT32_C( -109297466), INT32_C(-1230203271), INT32_C(-1731521429)),
      UINT16_C(53728),
      simde_mm_set_epi32(INT32_C( -707786971), INT32_C( 1712040202), INT32_C(-2012675757), INT32_C(-1396559749)),
      simde_mm512_set_epi32(INT32_C(-1396559749), INT32_C(-1396559749), INT32_C(  909767636), INT32_C(-1396559749),
                            INT32_C( 2123935701), INT32_C(-1040550911), INT32_C(  686758291), INT32_C(-1396559749),
                            INT32_C(-1396559749), INT32_C(-1396559749), INT32_C(-1396559749), INT32_C( 1300554279),
                            INT32_C(-1085613264), INT32_C( -109297466), INT32_C(-1230203271), INT32_C(-1731521429)) },
    { simde_mm512_set_epi32(INT32_C( 2041534605), INT32_C( 1255681923), INT32_C( 1220121473), INT32_C( 1819952522),
                            INT32_C(-1737362693), INT32_C(  712438877), INT32_C(-1234448370), INT32_C(  217554028),
                            INT32_C(-1878093154), INT32_C( -741869417), INT32_C(  943666007), INT32_C(  622675686),
                            INT32_C( -269910912), INT32_C(  137195559), INT32_C(  469574756), INT32_C( 1490101689)),
      UINT16_C(50038),
      simde_mm_set_epi32(INT32_C( -272719467), INT32_C( -594597983), INT32_C( -820913821), INT32_C(  345700481)),
      simde_mm512_set_epi32(INT32_C(  345700481), INT32_C(  345700481), INT32_C( 1220121473), INT32_C( 1819952522),
                            INT32_C(-1737362693), INT32_C(  712438877), INT32_C(  345700481), INT32_C(  345700481),
                            INT32_C(-1878093154), INT32_C(  345700481), INT32_C(  345700481), INT32_C(  345700481),
                            INT32_C( -269910912), INT32_C(  345700481), INT32_C(  345700481), INT32_C( 1490101689)) },
    { simde_mm512_set_epi32(INT32_C(  605201121), INT32_C(    2188130), INT32_C( -956406632), INT32_C(-1144421408),
                            INT32_C(-2008693903), INT32_C( 1823632430), INT32_C( 2043624683), INT32_C(  457225971),
                            INT32_C( 1484257119), INT32_C(  719932227), INT32_C( 1722430058), INT32_C(  916001650),
                            INT32_C(  553469699), INT32_C(-2003831430), INT32_C(-1834906502), INT32_C(  225358926)),
      UINT16_C(22657),
      simde_mm_set_epi32(INT32_C(  290541765), INT32_C( -479926223), INT32_C( 2079119915), INT32_C( -331512500)),
      simde_mm512_set_epi32(INT32_C(  605201121), INT32_C( -331512500), INT32_C( -956406632), INT32_C( -331512500),
                            INT32_C( -331512500), INT32_C( 1823632430), INT32_C( 2043624683), INT32_C(  457225971),
                            INT32_C( -331512500), INT32_C(  719932227), INT32_C( 1722430058), INT32_C(  916001650),
                            INT32_C(  553469699), INT32_C(-2003831430), INT32_C(-1834906502), INT32_C( -331512500)) },
    { simde_mm512_set_epi32(INT32_C( -545987817), INT32_C(-1146550995), INT32_C(  963048631), INT32_C( -701605919),
                            INT32_C(  432096480), INT32_C(-2030393254), INT32_C(-1236899565), INT32_C(-1697034971),
                            INT32_C( -998012960), INT32_C(-1579141793), INT32_C( 1664269708), INT32_C( -667117157),
                            INT32_C( -708117814), INT32_C(   85211107), INT32_C(  909670673), INT32_C( 1616737139)),
      UINT16_C( 4531),
      simde_mm_set_epi32(INT32_C( -503580732), INT32_C(-1790221512), INT32_C(-1663970343), INT32_C( 1633501790)),
      simde_mm512_set_epi32(INT32_C( -545987817), INT32_C(-1146550995), INT32_C(  963048631), INT32_C( 1633501790),
                            INT32_C(  432096480), INT32_C(-2030393254), INT32_C(-1236899565), INT32_C( 1633501790),
                            INT32_C( 1633501790), INT32_C(-1579141793), INT32_C( 1633501790), INT32_C( 1633501790),
                            INT32_C( -708117814), INT32_C(   85211107), INT32_C( 1633501790), INT32_C( 1633501790)) },
    { simde_mm512_set_epi32(INT32_C(-1668661089), INT32_C( 1895031925), INT32_C( 2107029353), INT32_C(-1915428586),
                            INT32_C(  963718296), INT32_C( 1878898594), INT32_C( -403168746), INT32_C(  502390291),
                            INT32_C( 1855826407), INT32_C(-1442018177), INT32_C( -244961355), INT32_C( 1777042193),
                            INT32_C(  373997996), INT32_C( -684064874), INT32_C(  930695451), INT32_C(-1073438864)),
      UINT16_C(53861),
      simde_mm_set_epi32(INT32_C( 1599859635), INT32_C(  543659234), INT32_C(-1222091200), INT32_C(  817594139)),
      simde_mm512_set_epi32(INT32_C(  817594139), INT32_C(  817594139), INT32_C( 2107029353), INT32_C(  817594139),
                            INT32_C(  963718296), INT32_C( 1878898594), INT32_C(  817594139), INT32_C(  502390291),
                            INT32_C( 1855826407), INT32_C(  817594139), INT32_C(  817594139), INT32_C( 1777042193),
                            INT32_C(  373997996), INT32_C(  817594139), INT32_C(  930695451), INT32_C(  817594139)) },
    { simde_mm512_set_epi32(INT32_C( -831807470), INT32_C( -591553083), INT32_C( -492649784), INT32_C(-1394371521),
                            INT32_C(-1760655625), INT32_C( 2135736563), INT32_C(-2075134444), INT32_C( -933317766),
                            INT32_C( -731013025), INT32_C(-2091361347), INT32_C( 1562364760), INT32_C( -612070110),
                            INT32_C( 1365385309), INT32_C( -121237183), INT32_C( 1543044931), INT32_C(-1490381593)),
      UINT16_C(20921),
      simde_mm_set_epi32(INT32_C(-1466503600), INT32_C(  824864478), INT32_C(-1491396230), INT32_C(-1907140086)),
      simde_mm512_set_epi32(INT32_C( -831807470), INT32_C(-1907140086), INT32_C( -492649784), INT32_C(-1907140086),
                            INT32_C(-1760655625), INT32_C( 2135736563), INT32_C(-2075134444), INT32_C(-1907140086),
                            INT32_C(-1907140086), INT32_C(-2091361347), INT32_C(-1907140086), INT32_C(-1907140086),
                            INT32_C(-1907140086), INT32_C( -121237183), INT32_C( 1543044931), INT32_C(-1907140086)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_broadcastd_epi32(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_broadcastd_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask16 k;
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { UINT16_C(21274),
      simde_mm_set_epi32(INT32_C( 1459257075), INT32_C(  587801532), INT32_C( 1631678564), INT32_C(  715337051)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(  715337051), INT32_C(          0), INT32_C(  715337051),
                            INT32_C(          0), INT32_C(          0), INT32_C(  715337051), INT32_C(  715337051),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(  715337051),
                            INT32_C(  715337051), INT32_C(          0), INT32_C(  715337051), INT32_C(          0)) },
    { UINT16_C(59357),
      simde_mm_set_epi32(INT32_C(-2022546688), INT32_C( 2145084340), INT32_C(   29275255), INT32_C( -827125259)),
      simde_mm512_set_epi32(INT32_C( -827125259), INT32_C( -827125259), INT32_C( -827125259), INT32_C(          0),
                            INT32_C(          0), INT32_C( -827125259), INT32_C( -827125259), INT32_C( -827125259),
                            INT32_C( -827125259), INT32_C( -827125259), INT32_C(          0), INT32_C( -827125259),
                            INT32_C( -827125259), INT32_C( -827125259), INT32_C(          0), INT32_C( -827125259)) },
    { UINT16_C(34446),
      simde_mm_set_epi32(INT32_C(  973425906), INT32_C( -935954345), INT32_C( 1285315081), INT32_C( 2142489532)),
      simde_mm512_set_epi32(INT32_C( 2142489532), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C( 2142489532), INT32_C( 2142489532), INT32_C(          0),
                            INT32_C( 2142489532), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C( 2142489532), INT32_C( 2142489532), INT32_C( 2142489532), INT32_C(          0)) },
    { UINT16_C(33955),
      simde_mm_set_epi32(INT32_C(-1114656122), INT32_C( 1221674060), INT32_C( -740975665), INT32_C( 2132760332)),
      simde_mm512_set_epi32(INT32_C( 2132760332), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C( 2132760332), INT32_C(          0), INT32_C(          0),
                            INT32_C( 2132760332), INT32_C(          0), INT32_C( 2132760332), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C( 2132760332), INT32_C( 2132760332)) },
    { UINT16_C(52572),
      simde_mm_set_epi32(INT32_C( -724774954), INT32_C( -166426332), INT32_C(-1571631693), INT32_C( -124417294)),
      simde_mm512_set_epi32(INT32_C( -124417294), INT32_C( -124417294), INT32_C(          0), INT32_C(          0),
                            INT32_C( -124417294), INT32_C( -124417294), INT32_C(          0), INT32_C( -124417294),
                            INT32_C(          0), INT32_C( -124417294), INT32_C(          0), INT32_C( -124417294),
                            INT32_C( -124417294), INT32_C( -124417294), INT32_C(          0), INT32_C(          0)) },
    { UINT16_C(38931),
      simde_mm_set_epi32(INT32_C(-1992244525), INT32_C( -292982508), INT32_C( -691380397), INT32_C(-1292068161)),
      simde_mm512_set_epi32(INT32_C(-1292068161), INT32_C(          0), INT32_C(          0), INT32_C(-1292068161),
                            INT32_C(-1292068161), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(-1292068161),
                            INT32_C(          0), INT32_C(          0), INT32_C(-1292068161), INT32_C(-1292068161)) },
    { UINT16_C(32377),
      simde_mm_set_epi32(INT32_C( -766689829), INT32_C(-1724046912), INT32_C( 1799018744), INT32_C(  623047724)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(  623047724), INT32_C(  623047724), INT32_C(  623047724),
                            INT32_C(  623047724), INT32_C(  623047724), INT32_C(  623047724), INT32_C(          0),
                            INT32_C(          0), INT32_C(  623047724), INT32_C(  623047724), INT32_C(  623047724),
                            INT32_C(  623047724), INT32_C(          0), INT32_C(          0), INT32_C(  623047724)) },
    { UINT16_C(18782),
      simde_mm_set_epi32(INT32_C(-2020669200), INT32_C( -170583969), INT32_C( -628885190), INT32_C(  818636447)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(  818636447), INT32_C(          0), INT32_C(          0),
                            INT32_C(  818636447), INT32_C(          0), INT32_C(          0), INT32_C(  818636447),
                            INT32_C(          0), INT32_C(  818636447), INT32_C(          0), INT32_C(  818636447),
                            INT32_C(  818636447), INT32_C(  818636447), INT32_C(  818636447), INT32_C(          0)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_broadcastd_epi32(test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_broadcastq_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm_set_epi64x(INT64_C(-4515171658517540054), INT64_C(-2354482342678283610)),
      simde_mm512_set_epi64(INT64_C(-2354482342678283610), INT64_C(-2354482342678283610),
                            INT64_C(-2354482342678283610), INT64_C(-2354482342678283610),
                            INT64_C(-2354482342678283610), INT64_C(-2354482342678283610),
                            INT64_C(-2354482342678283610), INT64_C(-2354482342678283610)) },
    { simde_mm_set_epi64x(INT64_C( 4205180536942191055), INT64_C( 9120321379188715333)),
      simde_mm512_set_epi64(INT64_C( 9120321379188715333), INT64_C( 9120321379188715333),
                            INT64_C( 9120321379188715333), INT64_C( 9120321379188715333),
                            INT64_C( 9120321379188715333), INT64_C( 9120321379188715333),
                            INT64_C( 9120321379188715333), INT64_C( 9120321379188715333)) },
    { simde_mm_set_epi64x(INT64_C( 8277521399394651842), INT64_C(-6222301646052282332)),
      simde_mm512_set_epi64(INT64_C(-6222301646052282332), INT64_C(-6222301646052282332),
                            INT64_C(-6222301646052282332), INT64_C(-6222301646052282332),
                            INT64_C(-6222301646052282332), INT64_C(-6222301646052282332),
                            INT64_C(-6222301646052282332), INT64_C(-6222301646052282332)) },
    { simde_mm_set_epi64x(INT64_C(-4193845803756315005), INT64_C(  -68341185169397976)),
      simde_mm512_set_epi64(INT64_C(  -68341185169397976), INT64_C(  -68341185169397976),
                            INT64_C(  -68341185169397976), INT64_C(  -68341185169397976),
                            INT64_C(  -68341185169397976), INT64_C(  -68341185169397976),
                            INT64_C(  -68341185169397976), INT64_C(  -68341185169397976)) },
    { simde_mm_set_epi64x(INT64_C( 3593801376552188636), INT64_C(-5825679392398740695)),
      simde_mm512_set_epi64(INT64_C(-5825679392398740695), INT64_C(-5825679392398740695),
                            INT64_C(-5825679392398740695), INT64_C(-5825679392398740695),
                            INT64_C(-5825679392398740695), INT64_C(-5825679392398740695),
                            INT64_C(-5825679392398740695), INT64_C(-5825679392398740695)) },
    { simde_mm_set_epi64x(INT64_C(-7659760468268217575), INT64_C(-7291265332978304433)),
      simde_mm512_set_epi64(INT64_C(-7291265332978304433), INT64_C(-7291265332978304433),
                            INT64_C(-7291265332978304433), INT64_C(-7291265332978304433),
                            INT64_C(-7291265332978304433), INT64_C(-7291265332978304433),
                            INT64_C(-7291265332978304433), INT64_C(-7291265332978304433)) },
    { simde_mm_set_epi64x(INT64_C( 1291150907495215908), INT64_C(-4379734758100841882)),
      simde_mm512_set_epi64(INT64_C(-4379734758100841882), INT64_C(-4379734758100841882),
                            INT64_C(-4379734758100841882), INT64_C(-4379734758100841882),
                            INT64_C(-4379734758100841882), INT64_C(-4379734758100841882),
                            INT64_C(-4379734758100841882), INT64_C(-4379734758100841882)) },
    { simde_mm_set_epi64x(INT64_C(-1180659064658150843), INT64_C( 5101874260281362505)),
      simde_mm512_set_epi64(INT64_C( 5101874260281362505), INT64_C( 5101874260281362505),
                            INT64_C( 5101874260281362505), INT64_C( 5101874260281362505),
                            INT64_C( 5101874260281362505), INT64_C( 5101874260281362505),
                            INT64_C( 5101874260281362505), INT64_C( 5101874260281362505)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_broadcastq_epi64(test_vec[i].a);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_broadcastq_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i src;
    simde__mmask8 k;
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C( 7039210974079555453), INT64_C( 3663249249268849210),
                            INT64_C( 4925990175430708534), INT64_C(-1667387235778762360),
                            INT64_C( 5351281682312326681), INT64_C( 3728301356360833529),
                            INT64_C( 3466465274179801181), INT64_C(-1906660408329519933)),
      UINT8_C( 18),
      simde_mm_set_epi64x(INT64_C(-5595987098075819748), INT64_C(  758519329013942001)),
      simde_mm512_set_epi64(INT64_C( 7039210974079555453), INT64_C( 3663249249268849210),
                            INT64_C( 4925990175430708534), INT64_C(  758519329013942001),
                            INT64_C( 5351281682312326681), INT64_C( 3728301356360833529),
                            INT64_C(  758519329013942001), INT64_C(-1906660408329519933)) },
    { simde_mm512_set_epi64(INT64_C( 8116099733890298375), INT64_C(-5538228106590303599),
                            INT64_C(  635024397322015168), INT64_C( -558744086021510173),
                            INT64_C(-6184591305598926545), INT64_C( 5112712356426664696),
                            INT64_C( 4271523692628158456), INT64_C( 4760221676782691018)),
      UINT8_C( 63),
      simde_mm_set_epi64x(INT64_C(-4849808913003762590), INT64_C( 7901381612815228817)),
      simde_mm512_set_epi64(INT64_C( 8116099733890298375), INT64_C(-5538228106590303599),
                            INT64_C( 7901381612815228817), INT64_C( 7901381612815228817),
                            INT64_C( 7901381612815228817), INT64_C( 7901381612815228817),
                            INT64_C( 7901381612815228817), INT64_C( 7901381612815228817)) },
    { simde_mm512_set_epi64(INT64_C( 3907422245001509329), INT64_C( 9122234377856250881),
                            INT64_C( 2949604402306461527), INT64_C(-1556319256029800786),
                            INT64_C(  505893122375737895), INT64_C(-4662673460798144314),
                            INT64_C(-5283682813813779349), INT64_C( 1076777643387686347)),
      UINT8_C( 72),
      simde_mm_set_epi64x(INT64_C(-5998178448496319999), INT64_C( 7927410529462710283)),
      simde_mm512_set_epi64(INT64_C( 3907422245001509329), INT64_C( 7927410529462710283),
                            INT64_C( 2949604402306461527), INT64_C(-1556319256029800786),
                            INT64_C( 7927410529462710283), INT64_C(-4662673460798144314),
                            INT64_C(-5283682813813779349), INT64_C( 1076777643387686347)) },
    { simde_mm512_set_epi64(INT64_C( 7816636564820325115), INT64_C( 3059901680174485518),
                            INT64_C(  934387437789942430), INT64_C(-3186304882973920425),
                            INT64_C( 2674371711409421440), INT64_C(  589250439531013220),
                            INT64_C( 6399938025556543269), INT64_C( 7353156679309525331)),
      UINT8_C(129),
      simde_mm_set_epi64x(INT64_C(-3132262719190613130), INT64_C( 8768324363382960003)),
      simde_mm512_set_epi64(INT64_C( 8768324363382960003), INT64_C( 3059901680174485518),
                            INT64_C(  934387437789942430), INT64_C(-3186304882973920425),
                            INT64_C( 2674371711409421440), INT64_C(  589250439531013220),
                            INT64_C( 6399938025556543269), INT64_C( 8768324363382960003)) },
    { simde_mm512_set_epi64(INT64_C(-8627274619235963858), INT64_C( 8777301179240593139),
                            INT64_C( 6374835785680112451), INT64_C( 7397780769673384818),
                            INT64_C( 2377134258823099770), INT64_C(-7880863417082399666),
                            INT64_C(-1171321188047181919), INT64_C(-3525798013683697535)),
      UINT8_C(224),
      simde_mm_set_epi64x(INT64_C(-5077429793204296991), INT64_C(    9397950127957144)),
      simde_mm512_set_epi64(INT64_C(    9397950127957144), INT64_C(    9397950127957144),
                            INT64_C(    9397950127957144), INT64_C( 7397780769673384818),
                            INT64_C( 2377134258823099770), INT64_C(-7880863417082399666),
                            INT64_C(-1171321188047181919), INT64_C(-3525798013683697535)) },
    { simde_mm512_set_epi64(INT64_C(-8720472620890953453), INT64_C(-7288709697316354080),
                            INT64_C(-6782362355017532020), INT64_C(-2865246368328647990),
                            INT64_C(  365978918730627345), INT64_C( 6943833138524147909),
                            INT64_C(-2061267430198683093), INT64_C(-1423835345422209809)),
      UINT8_C(224),
      simde_mm_set_epi64x(INT64_C(-2344999814881016531), INT64_C( 4136262378195933153)),
      simde_mm512_set_epi64(INT64_C( 4136262378195933153), INT64_C( 4136262378195933153),
                            INT64_C( 4136262378195933153), INT64_C(-2865246368328647990),
                            INT64_C(  365978918730627345), INT64_C( 6943833138524147909),
                            INT64_C(-2061267430198683093), INT64_C(-1423835345422209809)) },
    { simde_mm512_set_epi64(INT64_C(-1731596578336940525), INT64_C( 7970713727971134591),
                            INT64_C(-1052101006731803887), INT64_C( 1606309165200441238),
                            INT64_C( 3997306527802498928), INT64_C(-2162862772330994888),
                            INT64_C(-7146698203065400738), INT64_C(-4784261768320577101)),
      UINT8_C(162),
      simde_mm_set_epi64x(INT64_C( 8139100144857954153), INT64_C(-8226703133729805160)),
      simde_mm512_set_epi64(INT64_C(-8226703133729805160), INT64_C( 7970713727971134591),
                            INT64_C(-8226703133729805160), INT64_C( 1606309165200441238),
                            INT64_C( 3997306527802498928), INT64_C(-2162862772330994888),
                            INT64_C(-8226703133729805160), INT64_C(-4784261768320577101)) },
    { simde_mm512_set_epi64(INT64_C(-4008569278181826465), INT64_C(-8982328587921142952),
                            INT64_C(-2628821103943737251), INT64_C( -520709734501122237),
                            INT64_C(-6401140198895522893), INT64_C( 2334998633271287360),
                            INT64_C( 3511540092651127844), INT64_C( 3766647997225123999)),
      UINT8_C( 20),
      simde_mm_set_epi64x(INT64_C(-2115914707760868289), INT64_C(-7561958326757703437)),
      simde_mm512_set_epi64(INT64_C(-4008569278181826465), INT64_C(-8982328587921142952),
                            INT64_C(-2628821103943737251), INT64_C(-7561958326757703437),
                            INT64_C(-6401140198895522893), INT64_C(-7561958326757703437),
                            INT64_C( 3511540092651127844), INT64_C( 3766647997225123999)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_broadcastq_epi64(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_broadcastq_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask8 k;
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { UINT8_C( 26),
      simde_mm_set_epi64x(INT64_C( 2524588358110376036), INT64_C( 3072349241054123220)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C( 3072349241054123220),
                            INT64_C( 3072349241054123220), INT64_C(                   0),
                            INT64_C( 3072349241054123220), INT64_C(                   0)) },
    { UINT8_C(243),
      simde_mm_set_epi64x(INT64_C(  125736266274902517), INT64_C( 4529119523676940253)),
      simde_mm512_set_epi64(INT64_C( 4529119523676940253), INT64_C( 4529119523676940253),
                            INT64_C( 4529119523676940253), INT64_C( 4529119523676940253),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C( 4529119523676940253), INT64_C( 4529119523676940253)) },
    { UINT8_C(180),
      simde_mm_set_epi64x(INT64_C( 9201922475629043961), INT64_C(-5256397243355602176)),
      simde_mm512_set_epi64(INT64_C(-5256397243355602176), INT64_C(                   0),
                            INT64_C(-5256397243355602176), INT64_C(-5256397243355602176),
                            INT64_C(                   0), INT64_C(-5256397243355602176),
                            INT64_C(                   0), INT64_C(                   0)) },
    { UINT8_C(  9),
      simde_mm_set_epi64x(INT64_C( 4382010425855345827), INT64_C( 4180832434708183127)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C( 4180832434708183127), INT64_C(                   0),
                            INT64_C(                   0), INT64_C( 4180832434708183127)) },
    { UINT8_C( 12),
      simde_mm_set_epi64x(INT64_C(-1597707644585397626), INT64_C( 5247050137625533391)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C( 5247050137625533391), INT64_C( 5247050137625533391),
                            INT64_C(                   0), INT64_C(                   0)) },
    { UINT8_C(150),
      simde_mm_set_epi64x(INT64_C(-3112884720261363420), INT64_C(-6750106718621562126)),
      simde_mm512_set_epi64(INT64_C(-6750106718621562126), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(-6750106718621562126),
                            INT64_C(                   0), INT64_C(-6750106718621562126),
                            INT64_C(-6750106718621562126), INT64_C(                   0)) },
    { UINT8_C( 19),
      simde_mm_set_epi64x(INT64_C(-1258350286556471469), INT64_C(-5549390491787734701)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(-5549390491787734701),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(-5549390491787734701), INT64_C(-5549390491787734701)) },
    { UINT8_C(211),
      simde_mm_set_epi64x(INT64_C( 7726726670994043948), INT64_C( 5635717459582615161)),
      simde_mm512_set_epi64(INT64_C( 5635717459582615161), INT64_C( 5635717459582615161),
                            INT64_C(                   0), INT64_C( 5635717459582615161),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C( 5635717459582615161), INT64_C( 5635717459582615161)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_broadcastq_epi64(test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_broadcastss_ps(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128 a;
    simde__m512 r;
  } test_vec[8] = {
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   104.48), SIMDE_FLOAT32_C(   410.97), SIMDE_FLOAT32_C(   548.32), SIMDE_FLOAT32_C(   631.04)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04),
                         SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04),
                         SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04),
                         SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   668.18), SIMDE_FLOAT32_C(  -107.97), SIMDE_FLOAT32_C(  -627.99), SIMDE_FLOAT32_C(  -347.00)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00),
                         SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00),
                         SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00),
                         SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   236.12), SIMDE_FLOAT32_C(  -776.74), SIMDE_FLOAT32_C(   643.82), SIMDE_FLOAT32_C(  -941.79)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79),
                         SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79),
                         SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79),
                         SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(  -401.52), SIMDE_FLOAT32_C(   338.53), SIMDE_FLOAT32_C(  -725.48), SIMDE_FLOAT32_C(   387.06)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06),
                         SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06),
                         SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06),
                         SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   119.04), SIMDE_FLOAT32_C(   263.81), SIMDE_FLOAT32_C(   717.18), SIMDE_FLOAT32_C(  -996.30)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30),
                         SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30),
                         SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30),
                         SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(  -278.68), SIMDE_FLOAT32_C(   120.15), SIMDE_FLOAT32_C(   751.98), SIMDE_FLOAT32_C(   536.33)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33),
                         SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33),
                         SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33),
                         SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(  -143.65), SIMDE_FLOAT32_C(   810.77), SIMDE_FLOAT32_C(  -448.76), SIMDE_FLOAT32_C(   234.43)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43),
                         SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43),
                         SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43),
                         SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   -42.20), SIMDE_FLOAT32_C(  -923.83), SIMDE_FLOAT32_C(   357.03), SIMDE_FLOAT32_C(  -933.51)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51),
                         SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51),
                         SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51),
                         SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_broadcastss_ps(test_vec[i].a);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_mask_broadcastss_ps(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512 src;
    simde__mmask16 k;
    simde__m128 a;
    simde__m512 r;
  } test_vec[8] = {
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -227.30), SIMDE_FLOAT32_C(   999.04), SIMDE_FLOAT32_C(   956.07), SIMDE_FLOAT32_C(  -270.40),
                         SIMDE_FLOAT32_C(   132.00), SIMDE_FLOAT32_C(   480.19), SIMDE_FLOAT32_C(  -107.97), SIMDE_FLOAT32_C(  -347.00),
                         SIMDE_FLOAT32_C(  -927.52), SIMDE_FLOAT32_C(   -67.87), SIMDE_FLOAT32_C(   891.86), SIMDE_FLOAT32_C(  -870.50),
                         SIMDE_FLOAT32_C(   932.69), SIMDE_FLOAT32_C(   244.86), SIMDE_FLOAT32_C(  -621.59), SIMDE_FLOAT32_C(    36.25)),
      UINT16_C(30253),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   134.75), SIMDE_FLOAT32_C(   871.12), SIMDE_FLOAT32_C(   104.48), SIMDE_FLOAT32_C(   548.32)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -227.30), SIMDE_FLOAT32_C(   548.32), SIMDE_FLOAT32_C(   548.32), SIMDE_FLOAT32_C(   548.32),
                         SIMDE_FLOAT32_C(   132.00), SIMDE_FLOAT32_C(   548.32), SIMDE_FLOAT32_C(   548.32), SIMDE_FLOAT32_C(  -347.00),
                         SIMDE_FLOAT32_C(  -927.52), SIMDE_FLOAT32_C(   -67.87), SIMDE_FLOAT32_C(   548.32), SIMDE_FLOAT32_C(  -870.50),
                         SIMDE_FLOAT32_C(   548.32), SIMDE_FLOAT32_C(   548.32), SIMDE_FLOAT32_C(  -621.59), SIMDE_FLOAT32_C(   548.32)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(   389.31), SIMDE_FLOAT32_C(   670.35), SIMDE_FLOAT32_C(   396.13),
                         SIMDE_FLOAT32_C(  -971.67), SIMDE_FLOAT32_C(   528.69), SIMDE_FLOAT32_C(   275.37), SIMDE_FLOAT32_C(   338.53),
                         SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(    29.64), SIMDE_FLOAT32_C(   199.34), SIMDE_FLOAT32_C(  -686.40),
                         SIMDE_FLOAT32_C(   717.18), SIMDE_FLOAT32_C(   416.06), SIMDE_FLOAT32_C(   645.78), SIMDE_FLOAT32_C(  -990.79)),
      UINT16_C(37933),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   442.28), SIMDE_FLOAT32_C(   811.14), SIMDE_FLOAT32_C(  -767.79), SIMDE_FLOAT32_C(   236.12)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   236.12), SIMDE_FLOAT32_C(   389.31), SIMDE_FLOAT32_C(   670.35), SIMDE_FLOAT32_C(   236.12),
                         SIMDE_FLOAT32_C(  -971.67), SIMDE_FLOAT32_C(   236.12), SIMDE_FLOAT32_C(   275.37), SIMDE_FLOAT32_C(   338.53),
                         SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(    29.64), SIMDE_FLOAT32_C(   236.12), SIMDE_FLOAT32_C(  -686.40),
                         SIMDE_FLOAT32_C(   236.12), SIMDE_FLOAT32_C(   236.12), SIMDE_FLOAT32_C(   645.78), SIMDE_FLOAT32_C(   236.12)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   810.77), SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   101.39), SIMDE_FLOAT32_C(  -366.10),
                         SIMDE_FLOAT32_C(   855.96), SIMDE_FLOAT32_C(   -55.56), SIMDE_FLOAT32_C(   896.89), SIMDE_FLOAT32_C(   697.60),
                         SIMDE_FLOAT32_C(   120.15), SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(  -156.71), SIMDE_FLOAT32_C(  -331.13),
                         SIMDE_FLOAT32_C(  -143.65), SIMDE_FLOAT32_C(  -448.76), SIMDE_FLOAT32_C(  -628.22), SIMDE_FLOAT32_C(   318.72)),
      UINT16_C(19701),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -248.84), SIMDE_FLOAT32_C(   566.99), SIMDE_FLOAT32_C(  -650.08), SIMDE_FLOAT32_C(  -460.40)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   810.77), SIMDE_FLOAT32_C(  -460.40), SIMDE_FLOAT32_C(   101.39), SIMDE_FLOAT32_C(  -366.10),
                         SIMDE_FLOAT32_C(  -460.40), SIMDE_FLOAT32_C(  -460.40), SIMDE_FLOAT32_C(   896.89), SIMDE_FLOAT32_C(   697.60),
                         SIMDE_FLOAT32_C(  -460.40), SIMDE_FLOAT32_C(  -460.40), SIMDE_FLOAT32_C(  -460.40), SIMDE_FLOAT32_C(  -460.40),
                         SIMDE_FLOAT32_C(  -143.65), SIMDE_FLOAT32_C(  -460.40), SIMDE_FLOAT32_C(  -628.22), SIMDE_FLOAT32_C(  -460.40)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   779.71), SIMDE_FLOAT32_C(   196.66), SIMDE_FLOAT32_C(    -0.50), SIMDE_FLOAT32_C(  -273.06),
                         SIMDE_FLOAT32_C(   429.50), SIMDE_FLOAT32_C(   650.80), SIMDE_FLOAT32_C(   509.10), SIMDE_FLOAT32_C(   709.57),
                         SIMDE_FLOAT32_C(  -561.64), SIMDE_FLOAT32_C(  -923.83), SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -304.13),
                         SIMDE_FLOAT32_C(   728.72), SIMDE_FLOAT32_C(  -511.49), SIMDE_FLOAT32_C(   144.42), SIMDE_FLOAT32_C(   848.91)),
      UINT16_C(27468),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   949.19), SIMDE_FLOAT32_C(  -102.63), SIMDE_FLOAT32_C(    87.04), SIMDE_FLOAT32_C(   914.16)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   779.71), SIMDE_FLOAT32_C(   914.16), SIMDE_FLOAT32_C(   914.16), SIMDE_FLOAT32_C(  -273.06),
                         SIMDE_FLOAT32_C(   914.16), SIMDE_FLOAT32_C(   650.80), SIMDE_FLOAT32_C(   914.16), SIMDE_FLOAT32_C(   914.16),
                         SIMDE_FLOAT32_C(  -561.64), SIMDE_FLOAT32_C(   914.16), SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -304.13),
                         SIMDE_FLOAT32_C(   914.16), SIMDE_FLOAT32_C(   914.16), SIMDE_FLOAT32_C(   144.42), SIMDE_FLOAT32_C(   848.91)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -960.76), SIMDE_FLOAT32_C(  -613.57), SIMDE_FLOAT32_C(   864.92), SIMDE_FLOAT32_C(   278.02),
                         SIMDE_FLOAT32_C(   573.37), SIMDE_FLOAT32_C(   393.40), SIMDE_FLOAT32_C(  -782.91), SIMDE_FLOAT32_C(  -933.90),
                         SIMDE_FLOAT32_C(  -291.87), SIMDE_FLOAT32_C(   382.75), SIMDE_FLOAT32_C(   -62.73), SIMDE_FLOAT32_C(   163.52),
                         SIMDE_FLOAT32_C(    87.09), SIMDE_FLOAT32_C(  -486.60), SIMDE_FLOAT32_C(  -157.79), SIMDE_FLOAT32_C(  -247.69)),
      UINT16_C(56353),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   -97.06), SIMDE_FLOAT32_C(    -2.41), SIMDE_FLOAT32_C(   418.81), SIMDE_FLOAT32_C(  -141.42)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -141.42), SIMDE_FLOAT32_C(  -141.42), SIMDE_FLOAT32_C(   864.92), SIMDE_FLOAT32_C(  -141.42),
                         SIMDE_FLOAT32_C(  -141.42), SIMDE_FLOAT32_C(  -141.42), SIMDE_FLOAT32_C(  -782.91), SIMDE_FLOAT32_C(  -933.90),
                         SIMDE_FLOAT32_C(  -291.87), SIMDE_FLOAT32_C(   382.75), SIMDE_FLOAT32_C(  -141.42), SIMDE_FLOAT32_C(   163.52),
                         SIMDE_FLOAT32_C(    87.09), SIMDE_FLOAT32_C(  -486.60), SIMDE_FLOAT32_C(  -157.79), SIMDE_FLOAT32_C(  -141.42)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -490.98), SIMDE_FLOAT32_C(  -718.54), SIMDE_FLOAT32_C(  -172.08), SIMDE_FLOAT32_C(   476.19),
                         SIMDE_FLOAT32_C(  -825.45), SIMDE_FLOAT32_C(  -528.02), SIMDE_FLOAT32_C(  -604.26), SIMDE_FLOAT32_C(  -201.78),
                         SIMDE_FLOAT32_C(  -105.47), SIMDE_FLOAT32_C(   619.70), SIMDE_FLOAT32_C(   603.28), SIMDE_FLOAT32_C(  -553.28),
                         SIMDE_FLOAT32_C(   787.83), SIMDE_FLOAT32_C(  -945.21), SIMDE_FLOAT32_C(  -786.09), SIMDE_FLOAT32_C(   628.77)),
      UINT16_C(51486),
      simde_mm_set_ps(SIMDE_FLOAT32_C(    54.48), SIMDE_FLOAT32_C(   679.92), SIMDE_FLOAT32_C(  -550.45), SIMDE_FLOAT32_C(  -482.87)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -482.87), SIMDE_FLOAT32_C(  -482.87), SIMDE_FLOAT32_C(  -172.08), SIMDE_FLOAT32_C(   476.19),
                         SIMDE_FLOAT32_C(  -482.87), SIMDE_FLOAT32_C(  -528.02), SIMDE_FLOAT32_C(  -604.26), SIMDE_FLOAT32_C(  -482.87),
                         SIMDE_FLOAT32_C(  -105.47), SIMDE_FLOAT32_C(   619.70), SIMDE_FLOAT32_C(   603.28), SIMDE_FLOAT32_C(  -482.87),
                         SIMDE_FLOAT32_C(  -482.87), SIMDE_FLOAT32_C(  -482.87), SIMDE_FLOAT32_C(  -482.87), SIMDE_FLOAT32_C(   628.77)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -124.63), SIMDE_FLOAT32_C(  -948.04), SIMDE_FLOAT32_C(  -344.11), SIMDE_FLOAT32_C(  -424.86),
                         SIMDE_FLOAT32_C(   640.76), SIMDE_FLOAT32_C(  -243.42), SIMDE_FLOAT32_C(   962.71), SIMDE_FLOAT32_C(   314.11),
                         SIMDE_FLOAT32_C(   599.88), SIMDE_FLOAT32_C(  -844.53), SIMDE_FLOAT32_C(  -530.48), SIMDE_FLOAT32_C(   563.54),
                         SIMDE_FLOAT32_C(   165.16), SIMDE_FLOAT32_C(   384.17), SIMDE_FLOAT32_C(   149.22), SIMDE_FLOAT32_C(   712.14)),
      UINT16_C(53759),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -284.59), SIMDE_FLOAT32_C(  -286.48), SIMDE_FLOAT32_C(  -340.65), SIMDE_FLOAT32_C(   563.88)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   563.88), SIMDE_FLOAT32_C(   563.88), SIMDE_FLOAT32_C(  -344.11), SIMDE_FLOAT32_C(   563.88),
                         SIMDE_FLOAT32_C(   640.76), SIMDE_FLOAT32_C(  -243.42), SIMDE_FLOAT32_C(   962.71), SIMDE_FLOAT32_C(   563.88),
                         SIMDE_FLOAT32_C(   563.88), SIMDE_FLOAT32_C(   563.88), SIMDE_FLOAT32_C(   563.88), SIMDE_FLOAT32_C(   563.88),
                         SIMDE_FLOAT32_C(   563.88), SIMDE_FLOAT32_C(   563.88), SIMDE_FLOAT32_C(   563.88), SIMDE_FLOAT32_C(   563.88)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -434.52), SIMDE_FLOAT32_C(   678.42), SIMDE_FLOAT32_C(   -65.20), SIMDE_FLOAT32_C(  -319.19),
                         SIMDE_FLOAT32_C(   664.97), SIMDE_FLOAT32_C(     9.01), SIMDE_FLOAT32_C(  -334.08), SIMDE_FLOAT32_C(  -870.44),
                         SIMDE_FLOAT32_C(   269.08), SIMDE_FLOAT32_C(  -345.75), SIMDE_FLOAT32_C(  -732.77), SIMDE_FLOAT32_C(   374.12),
                         SIMDE_FLOAT32_C(  -491.24), SIMDE_FLOAT32_C(   525.54), SIMDE_FLOAT32_C(  -178.26), SIMDE_FLOAT32_C(  -733.62)),
      UINT16_C(50870),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -300.63), SIMDE_FLOAT32_C(  -396.75), SIMDE_FLOAT32_C(   745.02), SIMDE_FLOAT32_C(   369.43)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   369.43), SIMDE_FLOAT32_C(   369.43), SIMDE_FLOAT32_C(   -65.20), SIMDE_FLOAT32_C(  -319.19),
                         SIMDE_FLOAT32_C(   664.97), SIMDE_FLOAT32_C(   369.43), SIMDE_FLOAT32_C(   369.43), SIMDE_FLOAT32_C(  -870.44),
                         SIMDE_FLOAT32_C(   369.43), SIMDE_FLOAT32_C(  -345.75), SIMDE_FLOAT32_C(   369.43), SIMDE_FLOAT32_C(   369.43),
                         SIMDE_FLOAT32_C(  -491.24), SIMDE_FLOAT32_C(   369.43), SIMDE_FLOAT32_C(   369.43), SIMDE_FLOAT32_C(  -733.62)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_mask_broadcastss_ps(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_maskz_broadcastss_ps(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask16 k;
    simde__m128 a;
    simde__m512 r;
  } test_vec[8] = {
    { UINT16_C(25371),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   104.48), SIMDE_FLOAT32_C(   410.97), SIMDE_FLOAT32_C(   548.32), SIMDE_FLOAT32_C(   631.04)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   631.04),
                         SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04)) },
    { UINT16_C(49342),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -559.08), SIMDE_FLOAT32_C(   480.19), SIMDE_FLOAT32_C(   668.18), SIMDE_FLOAT32_C(  -107.97)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -107.97), SIMDE_FLOAT32_C(  -107.97), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(  -107.97), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -107.97), SIMDE_FLOAT32_C(  -107.97),
                         SIMDE_FLOAT32_C(  -107.97), SIMDE_FLOAT32_C(  -107.97), SIMDE_FLOAT32_C(  -107.97), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C(24820),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   811.14), SIMDE_FLOAT32_C(  -333.00), SIMDE_FLOAT32_C(  -767.79), SIMDE_FLOAT32_C(   825.12)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   825.12), SIMDE_FLOAT32_C(   825.12), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   825.12), SIMDE_FLOAT32_C(   825.12), SIMDE_FLOAT32_C(   825.12), SIMDE_FLOAT32_C(   825.12),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   825.12), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C(45881),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -610.29), SIMDE_FLOAT32_C(  -971.67), SIMDE_FLOAT32_C(   997.86), SIMDE_FLOAT32_C(   528.69)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   528.69), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   528.69), SIMDE_FLOAT32_C(   528.69),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   528.69), SIMDE_FLOAT32_C(   528.69),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   528.69), SIMDE_FLOAT32_C(   528.69),
                         SIMDE_FLOAT32_C(   528.69), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   528.69)) },
    { UINT16_C(28771),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -248.84), SIMDE_FLOAT32_C(   102.57), SIMDE_FLOAT32_C(   566.99), SIMDE_FLOAT32_C(   900.54)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   900.54), SIMDE_FLOAT32_C(   900.54), SIMDE_FLOAT32_C(   900.54),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   900.54), SIMDE_FLOAT32_C(   900.54), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   900.54), SIMDE_FLOAT32_C(   900.54)) },
    { UINT16_C(61611),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   318.72), SIMDE_FLOAT32_C(  -366.10), SIMDE_FLOAT32_C(   625.17), SIMDE_FLOAT32_C(   855.96)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   855.96), SIMDE_FLOAT32_C(   855.96), SIMDE_FLOAT32_C(   855.96), SIMDE_FLOAT32_C(   855.96),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   855.96), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   855.96), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   855.96), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   855.96), SIMDE_FLOAT32_C(   855.96)) },
    { UINT16_C(55548),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   324.92), SIMDE_FLOAT32_C(  -304.13), SIMDE_FLOAT32_C(   949.19), SIMDE_FLOAT32_C(   617.60)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   617.60), SIMDE_FLOAT32_C(   617.60), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   617.60),
                         SIMDE_FLOAT32_C(   617.60), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   617.60), SIMDE_FLOAT32_C(   617.60), SIMDE_FLOAT32_C(   617.60), SIMDE_FLOAT32_C(   617.60),
                         SIMDE_FLOAT32_C(   617.60), SIMDE_FLOAT32_C(   617.60), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C(15841),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   144.42), SIMDE_FLOAT32_C(    -0.50), SIMDE_FLOAT32_C(   848.91), SIMDE_FLOAT32_C(  -273.06)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -273.06), SIMDE_FLOAT32_C(  -273.06),
                         SIMDE_FLOAT32_C(  -273.06), SIMDE_FLOAT32_C(  -273.06), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -273.06),
                         SIMDE_FLOAT32_C(  -273.06), SIMDE_FLOAT32_C(  -273.06), SIMDE_FLOAT32_C(  -273.06), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -273.06)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_maskz_broadcastss_ps(test_vec[i].k, test_vec[i].a);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_broadcastsd_pd(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128d a;
    simde__m512d r;
  } test_vec[8] = {
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  912.41), SIMDE_FLOAT64_C(  842.49)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  842.49), SIMDE_FLOAT64_C(  842.49),
                         SIMDE_FLOAT64_C(  842.49), SIMDE_FLOAT64_C(  842.49),
                         SIMDE_FLOAT64_C(  842.49), SIMDE_FLOAT64_C(  842.49),
                         SIMDE_FLOAT64_C(  842.49), SIMDE_FLOAT64_C(  842.49)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  -93.61), SIMDE_FLOAT64_C( -903.55)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -903.55), SIMDE_FLOAT64_C( -903.55),
                         SIMDE_FLOAT64_C( -903.55), SIMDE_FLOAT64_C( -903.55),
                         SIMDE_FLOAT64_C( -903.55), SIMDE_FLOAT64_C( -903.55),
                         SIMDE_FLOAT64_C( -903.55), SIMDE_FLOAT64_C( -903.55)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C( -219.93), SIMDE_FLOAT64_C( -754.32)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -754.32), SIMDE_FLOAT64_C( -754.32),
                         SIMDE_FLOAT64_C( -754.32), SIMDE_FLOAT64_C( -754.32),
                         SIMDE_FLOAT64_C( -754.32), SIMDE_FLOAT64_C( -754.32),
                         SIMDE_FLOAT64_C( -754.32), SIMDE_FLOAT64_C( -754.32)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  137.51), SIMDE_FLOAT64_C(  527.47)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  527.47), SIMDE_FLOAT64_C(  527.47),
                         SIMDE_FLOAT64_C(  527.47), SIMDE_FLOAT64_C(  527.47),
                         SIMDE_FLOAT64_C(  527.47), SIMDE_FLOAT64_C(  527.47),
                         SIMDE_FLOAT64_C(  527.47), SIMDE_FLOAT64_C(  527.47)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  596.82), SIMDE_FLOAT64_C(  365.41)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  365.41), SIMDE_FLOAT64_C(  365.41),
                         SIMDE_FLOAT64_C(  365.41), SIMDE_FLOAT64_C(  365.41),
                         SIMDE_FLOAT64_C(  365.41), SIMDE_FLOAT64_C(  365.41),
                         SIMDE_FLOAT64_C(  365.41), SIMDE_FLOAT64_C(  365.41)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C( -229.59), SIMDE_FLOAT64_C( -642.88)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -642.88), SIMDE_FLOAT64_C( -642.88),
                         SIMDE_FLOAT64_C( -642.88), SIMDE_FLOAT64_C( -642.88),
                         SIMDE_FLOAT64_C( -642.88), SIMDE_FLOAT64_C( -642.88),
                         SIMDE_FLOAT64_C( -642.88), SIMDE_FLOAT64_C( -642.88)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  308.04), SIMDE_FLOAT64_C( -958.64)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -958.64), SIMDE_FLOAT64_C( -958.64),
                         SIMDE_FLOAT64_C( -958.64), SIMDE_FLOAT64_C( -958.64),
                         SIMDE_FLOAT64_C( -958.64), SIMDE_FLOAT64_C( -958.64),
                         SIMDE_FLOAT64_C( -958.64), SIMDE_FLOAT64_C( -958.64)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  991.16), SIMDE_FLOAT64_C( -172.14)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -172.14), SIMDE_FLOAT64_C( -172.14),
                         SIMDE_FLOAT64_C( -172.14), SIMDE_FLOAT64_C( -172.14),
                         SIMDE_FLOAT64_C( -172.14), SIMDE_FLOAT64_C( -172.14),
                         SIMDE_FLOAT64_C( -172.14), SIMDE_FLOAT64_C( -172.14)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_broadcastsd_pd(test_vec[i].a);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_mask_broadcastsd_pd(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512d src;
    simde__mmask8 k;
    simde__m128d a;
    simde__m512d r;
  } test_vec[8] = {
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -746.75), SIMDE_FLOAT64_C(  634.39),
                         SIMDE_FLOAT64_C( -651.68), SIMDE_FLOAT64_C( -903.55),
                         SIMDE_FLOAT64_C(  689.73), SIMDE_FLOAT64_C(  178.89),
                         SIMDE_FLOAT64_C( -342.04), SIMDE_FLOAT64_C( -292.58)),
      UINT8_C(162),
      simde_mm_set_pd(SIMDE_FLOAT64_C(   63.63), SIMDE_FLOAT64_C(  912.41)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  912.41), SIMDE_FLOAT64_C(  634.39),
                         SIMDE_FLOAT64_C(  912.41), SIMDE_FLOAT64_C( -903.55),
                         SIMDE_FLOAT64_C(  689.73), SIMDE_FLOAT64_C(  178.89),
                         SIMDE_FLOAT64_C(  912.41), SIMDE_FLOAT64_C( -292.58)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -287.03), SIMDE_FLOAT64_C(  851.62),
                         SIMDE_FLOAT64_C(  765.97), SIMDE_FLOAT64_C(  137.51),
                         SIMDE_FLOAT64_C( -457.60), SIMDE_FLOAT64_C(  815.46),
                         SIMDE_FLOAT64_C(  365.41), SIMDE_FLOAT64_C(  250.27)),
      UINT8_C( 66),
      simde_mm_set_pd(SIMDE_FLOAT64_C( -149.06), SIMDE_FLOAT64_C( -899.78)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -287.03), SIMDE_FLOAT64_C( -899.78),
                         SIMDE_FLOAT64_C(  765.97), SIMDE_FLOAT64_C(  137.51),
                         SIMDE_FLOAT64_C( -457.60), SIMDE_FLOAT64_C(  815.46),
                         SIMDE_FLOAT64_C( -899.78), SIMDE_FLOAT64_C(  250.27)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -958.64), SIMDE_FLOAT64_C(  523.36),
                         SIMDE_FLOAT64_C( -361.34), SIMDE_FLOAT64_C( -153.87),
                         SIMDE_FLOAT64_C( -642.88), SIMDE_FLOAT64_C(  573.19),
                         SIMDE_FLOAT64_C(  308.04), SIMDE_FLOAT64_C(  -38.88)),
      UINT8_C(115),
      simde_mm_set_pd(SIMDE_FLOAT64_C( -758.12), SIMDE_FLOAT64_C(   12.83)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -958.64), SIMDE_FLOAT64_C(   12.83),
                         SIMDE_FLOAT64_C(   12.83), SIMDE_FLOAT64_C(   12.83),
                         SIMDE_FLOAT64_C( -642.88), SIMDE_FLOAT64_C(  573.19),
                         SIMDE_FLOAT64_C(   12.83), SIMDE_FLOAT64_C(   12.83)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -221.94), SIMDE_FLOAT64_C(  499.58),
                         SIMDE_FLOAT64_C(   49.04), SIMDE_FLOAT64_C( -205.69),
                         SIMDE_FLOAT64_C(  991.16), SIMDE_FLOAT64_C( -984.94),
                         SIMDE_FLOAT64_C(  224.44), SIMDE_FLOAT64_C(  644.01)),
      UINT8_C(  4),
      simde_mm_set_pd(SIMDE_FLOAT64_C(   53.80), SIMDE_FLOAT64_C( -691.82)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -221.94), SIMDE_FLOAT64_C(  499.58),
                         SIMDE_FLOAT64_C(   49.04), SIMDE_FLOAT64_C( -205.69),
                         SIMDE_FLOAT64_C(  991.16), SIMDE_FLOAT64_C( -691.82),
                         SIMDE_FLOAT64_C(  224.44), SIMDE_FLOAT64_C(  644.01)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -225.42), SIMDE_FLOAT64_C( -144.08),
                         SIMDE_FLOAT64_C( -549.59), SIMDE_FLOAT64_C(  465.78),
                         SIMDE_FLOAT64_C( -316.69), SIMDE_FLOAT64_C( -133.94),
                         SIMDE_FLOAT64_C( -646.50), SIMDE_FLOAT64_C(  160.17)),
      UINT8_C(172),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  841.59), SIMDE_FLOAT64_C(  843.47)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  843.47), SIMDE_FLOAT64_C( -144.08),
                         SIMDE_FLOAT64_C(  843.47), SIMDE_FLOAT64_C(  465.78),
                         SIMDE_FLOAT64_C(  843.47), SIMDE_FLOAT64_C(  843.47),
                         SIMDE_FLOAT64_C( -646.50), SIMDE_FLOAT64_C(  160.17)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -560.31), SIMDE_FLOAT64_C( -882.49),
                         SIMDE_FLOAT64_C(  -54.78), SIMDE_FLOAT64_C( -896.38),
                         SIMDE_FLOAT64_C(  607.65), SIMDE_FLOAT64_C( -296.43),
                         SIMDE_FLOAT64_C(  124.51), SIMDE_FLOAT64_C( -913.38)),
      UINT8_C(201),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  625.18), SIMDE_FLOAT64_C(   54.43)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   54.43), SIMDE_FLOAT64_C(   54.43),
                         SIMDE_FLOAT64_C(  -54.78), SIMDE_FLOAT64_C( -896.38),
                         SIMDE_FLOAT64_C(   54.43), SIMDE_FLOAT64_C( -296.43),
                         SIMDE_FLOAT64_C(  124.51), SIMDE_FLOAT64_C(   54.43)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  349.28), SIMDE_FLOAT64_C( -559.74),
                         SIMDE_FLOAT64_C( -116.49), SIMDE_FLOAT64_C(  342.49),
                         SIMDE_FLOAT64_C( -608.07), SIMDE_FLOAT64_C(  778.83),
                         SIMDE_FLOAT64_C( -284.17), SIMDE_FLOAT64_C( -113.81)),
      UINT8_C(234),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  312.83), SIMDE_FLOAT64_C(  -27.64)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  -27.64), SIMDE_FLOAT64_C(  -27.64),
                         SIMDE_FLOAT64_C(  -27.64), SIMDE_FLOAT64_C(  342.49),
                         SIMDE_FLOAT64_C(  -27.64), SIMDE_FLOAT64_C(  778.83),
                         SIMDE_FLOAT64_C(  -27.64), SIMDE_FLOAT64_C( -113.81)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  364.84), SIMDE_FLOAT64_C(   86.19),
                         SIMDE_FLOAT64_C( -699.29), SIMDE_FLOAT64_C(  244.26),
                         SIMDE_FLOAT64_C( -206.27), SIMDE_FLOAT64_C( -921.17),
                         SIMDE_FLOAT64_C(  483.42), SIMDE_FLOAT64_C( -935.00)),
      UINT8_C( 12),
      simde_mm_set_pd(SIMDE_FLOAT64_C( -328.98), SIMDE_FLOAT64_C(  803.91)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  364.84), SIMDE_FLOAT64_C(   86.19),
                         SIMDE_FLOAT64_C( -699.29), SIMDE_FLOAT64_C(  244.26),
                         SIMDE_FLOAT64_C(  803.91), SIMDE_FLOAT64_C(  803.91),
                         SIMDE_FLOAT64_C(  483.42), SIMDE_FLOAT64_C( -935.00)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_mask_broadcastsd_pd(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_maskz_broadcastsd_pd(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask8 k;
    simde__m128d a;
    simde__m512d r;
  } test_vec[8] = {
    { UINT8_C(128),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  912.41), SIMDE_FLOAT64_C(  842.49)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  842.49), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00)) },
    { UINT8_C(  2),
      simde_mm_set_pd(SIMDE_FLOAT64_C( -651.68), SIMDE_FLOAT64_C(  -93.61)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  -93.61), SIMDE_FLOAT64_C(    0.00)) },
    { UINT8_C(216),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  881.57), SIMDE_FLOAT64_C( -899.78)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -899.78), SIMDE_FLOAT64_C( -899.78),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C( -899.78),
                         SIMDE_FLOAT64_C( -899.78), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00)) },
    { UINT8_C(183),
      simde_mm_set_pd(SIMDE_FLOAT64_C( -292.94), SIMDE_FLOAT64_C(  765.97)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  765.97), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  765.97), SIMDE_FLOAT64_C(  765.97),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(  765.97),
                         SIMDE_FLOAT64_C(  765.97), SIMDE_FLOAT64_C(  765.97)) },
    { UINT8_C(169),
      simde_mm_set_pd(SIMDE_FLOAT64_C( -758.12), SIMDE_FLOAT64_C(  593.03)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  593.03), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  593.03), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  593.03), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(  593.03)) },
    { UINT8_C(243),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  523.36), SIMDE_FLOAT64_C(  761.91)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  761.91), SIMDE_FLOAT64_C(  761.91),
                         SIMDE_FLOAT64_C(  761.91), SIMDE_FLOAT64_C(  761.91),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  761.91), SIMDE_FLOAT64_C(  761.91)) },
    { UINT8_C(109),
      simde_mm_set_pd(SIMDE_FLOAT64_C( -984.94), SIMDE_FLOAT64_C(   53.80)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(   53.80),
                         SIMDE_FLOAT64_C(   53.80), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(   53.80), SIMDE_FLOAT64_C(   53.80),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(   53.80)) },
    { UINT8_C(168),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  644.01), SIMDE_FLOAT64_C(  499.58)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  499.58), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  499.58), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  499.58), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_maskz_broadcastsd_pd(test_vec[i].k, test_vec[i].a);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_broadcastb_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm_set_epi8(INT8_C( -17), INT8_C(  88), INT8_C(-122), INT8_C(-119),
                        INT8_C( 111), INT8_C(  87), INT8_C( -76), INT8_C(  27),
                        INT8_C( -93), INT8_C(  -8), INT8_C( -17), INT8_C(  24),
                        INT8_C(  69), INT8_C( 116), INT8_C( -78), INT8_C(-124)),
      simde_mm512_set_epi8(INT8_C(-124), INT8_C(-124), INT8_C(-124), INT8_C(-124),
                           INT8_C(-124), INT8_C(-124), INT8_C(-124), INT8_C(-124),
                           INT8_C(-124), INT8_C(-124), INT8_C(-124), INT8_C(-124),
                           INT8_C(-124), INT8_C(-124), INT8_C(-124), INT8_C(-124),
                           INT8_C(-124), INT8_C(-124), INT8_C(-124), INT8_C(-124),
                           INT8_C(-124), INT8_C(-124), INT8_C(-124), INT8_C(-124),
                           INT8_C(-124), INT8_C(-124), INT8_C(-124), INT8_C(-124),
                           INT8_C(-124), INT8_C(-124), INT8_C(-124), INT8_C(-124),
                           INT8_C(-124), INT8_C(-124), INT8_C(-124), INT8_C(-124),
                           INT8_C(-124), INT8_C(-124), INT8_C(-124), INT8_C(-124),
                           INT8_C(-124), INT8_C(-124), INT8_C(-124), INT8_C(-124),
                           INT8_C(-124), INT8_C(-124), INT8_C(-124), INT8_C(-124),
                           INT8_C(-124), INT8_C(-124), INT8_C(-124), INT8_C(-124),
                           INT8_C(-124), INT8_C(-124), INT8_C(-124), INT8_C(-124),
                           INT8_C(-124), INT8_C(-124), INT8_C(-124), INT8_C(-124),
                           INT8_C(-124), INT8_C(-124), INT8_C(-124), INT8_C(-124)) },
    { simde_mm_set_epi8(INT8_C(  -5), INT8_C(-114), INT8_C( -86), INT8_C(  -2),
                        INT8_C(  33), INT8_C(  90), INT8_C( -50), INT8_C(  28),
                        INT8_C(  15), INT8_C(  12), INT8_C(  32), INT8_C(  54),
                        INT8_C( -15), INT8_C( -57), INT8_C(  36), INT8_C( -50)),
      simde_mm512_set_epi8(INT8_C( -50), INT8_C( -50), INT8_C( -50), INT8_C( -50),
                           INT8_C( -50), INT8_C( -50), INT8_C( -50), INT8_C( -50),
                           INT8_C( -50), INT8_C( -50), INT8_C( -50), INT8_C( -50),
                           INT8_C( -50), INT8_C( -50), INT8_C( -50), INT8_C( -50),
                           INT8_C( -50), INT8_C( -50), INT8_C( -50), INT8_C( -50),
                           INT8_C( -50), INT8_C( -50), INT8_C( -50), INT8_C( -50),
                           INT8_C( -50), INT8_C( -50), INT8_C( -50), INT8_C( -50),
                           INT8_C( -50), INT8_C( -50), INT8_C( -50), INT8_C( -50),
                           INT8_C( -50), INT8_C( -50), INT8_C( -50), INT8_C( -50),
                           INT8_C( -50), INT8_C( -50), INT8_C( -50), INT8_C( -50),
                           INT8_C( -50), INT8_C( -50), INT8_C( -50), INT8_C( -50),
                           INT8_C( -50), INT8_C( -50), INT8_C( -50), INT8_C( -50),
                           INT8_C( -50), INT8_C( -50), INT8_C( -50), INT8_C( -50),
                           INT8_C( -50), INT8_C( -50), INT8_C( -50), INT8_C( -50),
                           INT8_C( -50), INT8_C( -50), INT8_C( -50), INT8_C( -50),
                           INT8_C( -50), INT8_C( -50), INT8_C( -50), INT8_C( -50)) },
    { simde_mm_set_epi8(INT8_C( -49), INT8_C( -76), INT8_C( -62), INT8_C( 118),
                        INT8_C(  -4), INT8_C( -25), INT8_C( -58), INT8_C( 126),
                        INT8_C(-115), INT8_C( 126), INT8_C(-104), INT8_C( 127),
                        INT8_C(  15), INT8_C(  41), INT8_C(  68), INT8_C(  31)),
      simde_mm512_set_epi8(INT8_C(  31), INT8_C(  31), INT8_C(  31), INT8_C(  31),
                           INT8_C(  31), INT8_C(  31), INT8_C(  31), INT8_C(  31),
                           INT8_C(  31), INT8_C(  31), INT8_C(  31), INT8_C(  31),
                           INT8_C(  31), INT8_C(  31), INT8_C(  31), INT8_C(  31),
                           INT8_C(  31), INT8_C(  31), INT8_C(  31), INT8_C(  31),
                           INT8_C(  31), INT8_C(  31), INT8_C(  31), INT8_C(  31),
                           INT8_C(  31), INT8_C(  31), INT8_C(  31), INT8_C(  31),
                           INT8_C(  31), INT8_C(  31), INT8_C(  31), INT8_C(  31),
                           INT8_C(  31), INT8_C(  31), INT8_C(  31), INT8_C(  31),
                           INT8_C(  31), INT8_C(  31), INT8_C(  31), INT8_C(  31),
                           INT8_C(  31), INT8_C(  31), INT8_C(  31), INT8_C(  31),
                           INT8_C(  31), INT8_C(  31), INT8_C(  31), INT8_C(  31),
                           INT8_C(  31), INT8_C(  31), INT8_C(  31), INT8_C(  31),
                           INT8_C(  31), INT8_C(  31), INT8_C(  31), INT8_C(  31),
                           INT8_C(  31), INT8_C(  31), INT8_C(  31), INT8_C(  31),
                           INT8_C(  31), INT8_C(  31), INT8_C(  31), INT8_C(  31)) },
    { simde_mm_set_epi8(INT8_C( -30), INT8_C( -23), INT8_C( -42), INT8_C( -27),
                        INT8_C(-102), INT8_C(  -5), INT8_C( -87), INT8_C(  98),
                        INT8_C(  33), INT8_C(  73), INT8_C( 125), INT8_C( 120),
                        INT8_C( -70), INT8_C(  59), INT8_C( 124), INT8_C(  46)),
      simde_mm512_set_epi8(INT8_C(  46), INT8_C(  46), INT8_C(  46), INT8_C(  46),
                           INT8_C(  46), INT8_C(  46), INT8_C(  46), INT8_C(  46),
                           INT8_C(  46), INT8_C(  46), INT8_C(  46), INT8_C(  46),
                           INT8_C(  46), INT8_C(  46), INT8_C(  46), INT8_C(  46),
                           INT8_C(  46), INT8_C(  46), INT8_C(  46), INT8_C(  46),
                           INT8_C(  46), INT8_C(  46), INT8_C(  46), INT8_C(  46),
                           INT8_C(  46), INT8_C(  46), INT8_C(  46), INT8_C(  46),
                           INT8_C(  46), INT8_C(  46), INT8_C(  46), INT8_C(  46),
                           INT8_C(  46), INT8_C(  46), INT8_C(  46), INT8_C(  46),
                           INT8_C(  46), INT8_C(  46), INT8_C(  46), INT8_C(  46),
                           INT8_C(  46), INT8_C(  46), INT8_C(  46), INT8_C(  46),
                           INT8_C(  46), INT8_C(  46), INT8_C(  46), INT8_C(  46),
                           INT8_C(  46), INT8_C(  46), INT8_C(  46), INT8_C(  46),
                           INT8_C(  46), INT8_C(  46), INT8_C(  46), INT8_C(  46),
                           INT8_C(  46), INT8_C(  46), INT8_C(  46), INT8_C(  46),
                           INT8_C(  46), INT8_C(  46), INT8_C(  46), INT8_C(  46)) },
    { simde_mm_set_epi8(INT8_C( -18), INT8_C(  28), INT8_C( -19), INT8_C( -73),
                        INT8_C( -19), INT8_C(  67), INT8_C(  79), INT8_C( -45),
                        INT8_C(-124), INT8_C(  80), INT8_C(-101), INT8_C(-122),
                        INT8_C( -54), INT8_C(  30), INT8_C( -16), INT8_C(  55)),
      simde_mm512_set_epi8(INT8_C(  55), INT8_C(  55), INT8_C(  55), INT8_C(  55),
                           INT8_C(  55), INT8_C(  55), INT8_C(  55), INT8_C(  55),
                           INT8_C(  55), INT8_C(  55), INT8_C(  55), INT8_C(  55),
                           INT8_C(  55), INT8_C(  55), INT8_C(  55), INT8_C(  55),
                           INT8_C(  55), INT8_C(  55), INT8_C(  55), INT8_C(  55),
                           INT8_C(  55), INT8_C(  55), INT8_C(  55), INT8_C(  55),
                           INT8_C(  55), INT8_C(  55), INT8_C(  55), INT8_C(  55),
                           INT8_C(  55), INT8_C(  55), INT8_C(  55), INT8_C(  55),
                           INT8_C(  55), INT8_C(  55), INT8_C(  55), INT8_C(  55),
                           INT8_C(  55), INT8_C(  55), INT8_C(  55), INT8_C(  55),
                           INT8_C(  55), INT8_C(  55), INT8_C(  55), INT8_C(  55),
                           INT8_C(  55), INT8_C(  55), INT8_C(  55), INT8_C(  55),
                           INT8_C(  55), INT8_C(  55), INT8_C(  55), INT8_C(  55),
                           INT8_C(  55), INT8_C(  55), INT8_C(  55), INT8_C(  55),
                           INT8_C(  55), INT8_C(  55), INT8_C(  55), INT8_C(  55),
                           INT8_C(  55), INT8_C(  55), INT8_C(  55), INT8_C(  55)) },
    { simde_mm_set_epi8(INT8_C(   6), INT8_C(  -5), INT8_C(  37), INT8_C( -97),
                        INT8_C(  16), INT8_C(  -5), INT8_C( -18), INT8_C(  14),
                        INT8_C(-120), INT8_C( -59), INT8_C( -43), INT8_C( -97),
                        INT8_C( -71), INT8_C( -73), INT8_C( -73), INT8_C( -50)),
      simde_mm512_set_epi8(INT8_C( -50), INT8_C( -50), INT8_C( -50), INT8_C( -50),
                           INT8_C( -50), INT8_C( -50), INT8_C( -50), INT8_C( -50),
                           INT8_C( -50), INT8_C( -50), INT8_C( -50), INT8_C( -50),
                           INT8_C( -50), INT8_C( -50), INT8_C( -50), INT8_C( -50),
                           INT8_C( -50), INT8_C( -50), INT8_C( -50), INT8_C( -50),
                           INT8_C( -50), INT8_C( -50), INT8_C( -50), INT8_C( -50),
                           INT8_C( -50), INT8_C( -50), INT8_C( -50), INT8_C( -50),
                           INT8_C( -50), INT8_C( -50), INT8_C( -50), INT8_C( -50),
                           INT8_C( -50), INT8_C( -50), INT8_C( -50), INT8_C( -50),
                           INT8_C( -50), INT8_C( -50), INT8_C( -50), INT8_C( -50),
                           INT8_C( -50), INT8_C( -50), INT8_C( -50), INT8_C( -50),
                           INT8_C( -50), INT8_C( -50), INT8_C( -50), INT8_C( -50),
                           INT8_C( -50), INT8_C( -50), INT8_C( -50), INT8_C( -50),
                           INT8_C( -50), INT8_C( -50), INT8_C( -50), INT8_C( -50),
                           INT8_C( -50), INT8_C( -50), INT8_C( -50), INT8_C( -50),
                           INT8_C( -50), INT8_C( -50), INT8_C( -50), INT8_C( -50)) },
    { simde_mm_set_epi8(INT8_C( 119), INT8_C(  60), INT8_C(  63), INT8_C( -26),
                        INT8_C(  50), INT8_C(  56), INT8_C(  40), INT8_C(  -7),
                        INT8_C(  68), INT8_C( -11), INT8_C( -21), INT8_C( -77),
                        INT8_C(  56), INT8_C(-109), INT8_C(-118), INT8_C(-108)),
      simde_mm512_set_epi8(INT8_C(-108), INT8_C(-108), INT8_C(-108), INT8_C(-108),
                           INT8_C(-108), INT8_C(-108), INT8_C(-108), INT8_C(-108),
                           INT8_C(-108), INT8_C(-108), INT8_C(-108), INT8_C(-108),
                           INT8_C(-108), INT8_C(-108), INT8_C(-108), INT8_C(-108),
                           INT8_C(-108), INT8_C(-108), INT8_C(-108), INT8_C(-108),
                           INT8_C(-108), INT8_C(-108), INT8_C(-108), INT8_C(-108),
                           INT8_C(-108), INT8_C(-108), INT8_C(-108), INT8_C(-108),
                           INT8_C(-108), INT8_C(-108), INT8_C(-108), INT8_C(-108),
                           INT8_C(-108), INT8_C(-108), INT8_C(-108), INT8_C(-108),
                           INT8_C(-108), INT8_C(-108), INT8_C(-108), INT8_C(-108),
                           INT8_C(-108), INT8_C(-108), INT8_C(-108), INT8_C(-108),
                           INT8_C(-108), INT8_C(-108), INT8_C(-108), INT8_C(-108),
                           INT8_C(-108), INT8_C(-108), INT8_C(-108), INT8_C(-108),
                           INT8_C(-108), INT8_C(-108), INT8_C(-108), INT8_C(-108),
                           INT8_C(-108), INT8_C(-108), INT8_C(-108), INT8_C(-108),
                           INT8_C(-108), INT8_C(-108), INT8_C(-108), INT8_C(-108)) },
    { simde_mm_set_epi8(INT8_C(-112), INT8_C(  65), INT8_C(  26), INT8_C( -90),
                        INT8_C( -77), INT8_C(  72), INT8_C(   2), INT8_C(   4),
                        INT8_C( -52), INT8_C( -82), INT8_C( -18), INT8_C( -66),
                        INT8_C(-118), INT8_C( -10), INT8_C(  52), INT8_C( -40)),
      simde_mm512_set_epi8(INT8_C( -40), INT8_C( -40), INT8_C( -40), INT8_C( -40),
                           INT8_C( -40), INT8_C( -40), INT8_C( -40), INT8_C( -40),
                           INT8_C( -40), INT8_C( -40), INT8_C( -40), INT8_C( -40),
                           INT8_C( -40), INT8_C( -40), INT8_C( -40), INT8_C( -40),
                           INT8_C( -40), INT8_C( -40), INT8_C( -40), INT8_C( -40),
                           INT8_C( -40), INT8_C( -40), INT8_C( -40), INT8_C( -40),
                           INT8_C( -40), INT8_C( -40), INT8_C( -40), INT8_C( -40),
                           INT8_C( -40), INT8_C( -40), INT8_C( -40), INT8_C( -40),
                           INT8_C( -40), INT8_C( -40), INT8_C( -40), INT8_C( -40),
                           INT8_C( -40), INT8_C( -40), INT8_C( -40), INT8_C( -40),
                           INT8_C( -40), INT8_C( -40), INT8_C( -40), INT8_C( -40),
                           INT8_C( -40), INT8_C( -40), INT8_C( -40), INT8_C( -40),
                           INT8_C( -40), INT8_C( -40), INT8_C( -40), INT8_C( -40),
                           INT8_C( -40), INT8_C( -40), INT8_C( -40), INT8_C( -40),
                           INT8_C( -40), INT8_C( -40), INT8_C( -40), INT8_C( -40),
                           INT8_C( -40), INT8_C( -40), INT8_C( -40), INT8_C( -40)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_broadcastb_epi8(test_vec[i].a);
    simde_assert_m512i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_broadcastb_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i src;
    simde__mmask64 k;
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi8(INT8_C(  65), INT8_C( -68), INT8_C( 102), INT8_C(-122),
                           INT8_C(  40), INT8_C(  19), INT8_C(-111), INT8_C(   8),
                           INT8_C( -58), INT8_C(-120), INT8_C( 111), INT8_C(  10),
                           INT8_C( -54), INT8_C(-100), INT8_C(  36), INT8_C(  27),
                           INT8_C(-106), INT8_C(-123), INT8_C( 120), INT8_C(  43),
                           INT8_C( -31), INT8_C(   4), INT8_C(  10), INT8_C(  96),
                           INT8_C( -40), INT8_C(  23), INT8_C(  31), INT8_C(  73),
                           INT8_C( -51), INT8_C(  91), INT8_C(  68), INT8_C( -23),
                           INT8_C(-108), INT8_C(  52), INT8_C(  23), INT8_C( 115),
                           INT8_C(  -4), INT8_C(  24), INT8_C( 106), INT8_C( -92),
                           INT8_C( 126), INT8_C(  -6), INT8_C(  16), INT8_C( 102),
                           INT8_C( -47), INT8_C(-116), INT8_C(  -4), INT8_C(  33),
                           INT8_C( -25), INT8_C(-108), INT8_C(-115), INT8_C(-104),
                           INT8_C( -39), INT8_C(  49), INT8_C(  72), INT8_C(  44),
                           INT8_C( -17), INT8_C( -66), INT8_C( -61), INT8_C( -68),
                           INT8_C( 124), INT8_C(  20), INT8_C(  64), INT8_C( -49)),
      UINT64_C(          2970261907),
      simde_mm_set_epi8(INT8_C( -78), INT8_C( -60), INT8_C(-122), INT8_C( -66),
                        INT8_C(   8), INT8_C( -42), INT8_C(  44), INT8_C(  45),
                        INT8_C(  37), INT8_C(  -9), INT8_C( -14), INT8_C(  38),
                        INT8_C( -85), INT8_C(  35), INT8_C(   8), INT8_C(-127)),
      simde_mm512_set_epi8(INT8_C(  65), INT8_C( -68), INT8_C( 102), INT8_C(-122),
                           INT8_C(  40), INT8_C(  19), INT8_C(-111), INT8_C(   8),
                           INT8_C( -58), INT8_C(-120), INT8_C( 111), INT8_C(  10),
                           INT8_C( -54), INT8_C(-100), INT8_C(  36), INT8_C(  27),
                           INT8_C(-106), INT8_C(-123), INT8_C( 120), INT8_C(  43),
                           INT8_C( -31), INT8_C(   4), INT8_C(  10), INT8_C(  96),
                           INT8_C( -40), INT8_C(  23), INT8_C(  31), INT8_C(  73),
                           INT8_C( -51), INT8_C(  91), INT8_C(  68), INT8_C( -23),
                           INT8_C(-127), INT8_C(  52), INT8_C(-127), INT8_C(-127),
                           INT8_C(  -4), INT8_C(  24), INT8_C( 106), INT8_C(-127),
                           INT8_C( 126), INT8_C(  -6), INT8_C(  16), INT8_C( 102),
                           INT8_C(-127), INT8_C(-116), INT8_C(-127), INT8_C(  33),
                           INT8_C(-127), INT8_C(-108), INT8_C(-115), INT8_C(-127),
                           INT8_C(-127), INT8_C(  49), INT8_C(  72), INT8_C(-127),
                           INT8_C(-127), INT8_C( -66), INT8_C( -61), INT8_C(-127),
                           INT8_C( 124), INT8_C(  20), INT8_C(-127), INT8_C(-127)) },
    { simde_mm512_set_epi8(INT8_C( -76), INT8_C(  58), INT8_C(  70), INT8_C(-106),
                           INT8_C( 120), INT8_C(  72), INT8_C(  -4), INT8_C( -60),
                           INT8_C( 104), INT8_C( 101), INT8_C(  53), INT8_C(-121),
                           INT8_C( 102), INT8_C(-115), INT8_C(  90), INT8_C(  31),
                           INT8_C(  11), INT8_C(  68), INT8_C(  48), INT8_C(   4),
                           INT8_C(  55), INT8_C( -83), INT8_C(  75), INT8_C( -60),
                           INT8_C( -54), INT8_C(  57), INT8_C(  70), INT8_C(-114),
                           INT8_C(  51), INT8_C( -72), INT8_C( -71), INT8_C(  17),
                           INT8_C(  48), INT8_C(  40), INT8_C(-108), INT8_C( -12),
                           INT8_C( -11), INT8_C( -71), INT8_C(-114), INT8_C( -36),
                           INT8_C( -92), INT8_C( 101), INT8_C(  30), INT8_C(  10),
                           INT8_C(  43), INT8_C(-116), INT8_C( -45), INT8_C(-104),
                           INT8_C(  99), INT8_C( 108), INT8_C(  90), INT8_C(   7),
                           INT8_C( 112), INT8_C(  86), INT8_C(-125), INT8_C(  88),
                           INT8_C(  27), INT8_C(  40), INT8_C(  10), INT8_C(-105),
                           INT8_C(  76), INT8_C(-101), INT8_C(  87), INT8_C( 112)),
      UINT64_C(           458960523),
      simde_mm_set_epi8(INT8_C(  70), INT8_C( -73), INT8_C( -42), INT8_C( -19),
                        INT8_C( 110), INT8_C( -58), INT8_C(-117), INT8_C(-100),
                        INT8_C(  52), INT8_C( -63), INT8_C( -88), INT8_C( -55),
                        INT8_C(  90), INT8_C( -15), INT8_C( -11), INT8_C( -21)),
      simde_mm512_set_epi8(INT8_C( -76), INT8_C(  58), INT8_C(  70), INT8_C(-106),
                           INT8_C( 120), INT8_C(  72), INT8_C(  -4), INT8_C( -60),
                           INT8_C( 104), INT8_C( 101), INT8_C(  53), INT8_C(-121),
                           INT8_C( 102), INT8_C(-115), INT8_C(  90), INT8_C(  31),
                           INT8_C(  11), INT8_C(  68), INT8_C(  48), INT8_C(   4),
                           INT8_C(  55), INT8_C( -83), INT8_C(  75), INT8_C( -60),
                           INT8_C( -54), INT8_C(  57), INT8_C(  70), INT8_C(-114),
                           INT8_C(  51), INT8_C( -72), INT8_C( -71), INT8_C(  17),
                           INT8_C(  48), INT8_C(  40), INT8_C(-108), INT8_C( -21),
                           INT8_C( -21), INT8_C( -71), INT8_C( -21), INT8_C( -21),
                           INT8_C( -92), INT8_C( -21), INT8_C(  30), INT8_C( -21),
                           INT8_C( -21), INT8_C(-116), INT8_C( -21), INT8_C( -21),
                           INT8_C(  99), INT8_C( 108), INT8_C( -21), INT8_C(   7),
                           INT8_C( -21), INT8_C( -21), INT8_C( -21), INT8_C(  88),
                           INT8_C( -21), INT8_C(  40), INT8_C(  10), INT8_C(-105),
                           INT8_C( -21), INT8_C(-101), INT8_C( -21), INT8_C( -21)) },
    { simde_mm512_set_epi8(INT8_C( -65), INT8_C(  -9), INT8_C( -93), INT8_C(-113),
                           INT8_C( -10), INT8_C(  74), INT8_C(  39), INT8_C(  57),
                           INT8_C(  91), INT8_C( -48), INT8_C(  11), INT8_C( -15),
                           INT8_C(  21), INT8_C( -88), INT8_C(  91), INT8_C(  87),
                           INT8_C(-120), INT8_C(-105), INT8_C( -47), INT8_C(  85),
                           INT8_C( -98), INT8_C(  22), INT8_C(-124), INT8_C(-124),
                           INT8_C(   2), INT8_C(-104), INT8_C(  27), INT8_C(  96),
                           INT8_C( -89), INT8_C(  31), INT8_C(  20), INT8_C(  31),
                           INT8_C( -95), INT8_C(  13), INT8_C(  37), INT8_C(  31),
                           INT8_C( -72), INT8_C(  83), INT8_C(  94), INT8_C(  52),
                           INT8_C(  41), INT8_C(  25), INT8_C( -42), INT8_C(-109),
                           INT8_C(  31), INT8_C(  88), INT8_C( -71), INT8_C( -89),
                           INT8_C( 103), INT8_C( -85), INT8_C( -29), INT8_C(  86),
                           INT8_C(  71), INT8_C(  28), INT8_C( -23), INT8_C(  28),
                           INT8_C( -53), INT8_C( -82), INT8_C(  58), INT8_C( -12),
                           INT8_C(  63), INT8_C(  39), INT8_C( -32), INT8_C( -94)),
      UINT64_C(          1058428392),
      simde_mm_set_epi8(INT8_C(  85), INT8_C( -11), INT8_C( -21), INT8_C(  66),
                        INT8_C(  72), INT8_C(  -7), INT8_C( -18), INT8_C(-121),
                        INT8_C(  56), INT8_C(  51), INT8_C( 101), INT8_C(  91),
                        INT8_C( -85), INT8_C( -32), INT8_C( -40), INT8_C( -81)),
      simde_mm512_set_epi8(INT8_C( -65), INT8_C(  -9), INT8_C( -93), INT8_C(-113),
                           INT8_C( -10), INT8_C(  74), INT8_C(  39), INT8_C(  57),
                           INT8_C(  91), INT8_C( -48), INT8_C(  11), INT8_C( -15),
                           INT8_C(  21), INT8_C( -88), INT8_C(  91), INT8_C(  87),
                           INT8_C(-120), INT8_C(-105), INT8_C( -47), INT8_C(  85),
                           INT8_C( -98), INT8_C(  22), INT8_C(-124), INT8_C(-124),
                           INT8_C(   2), INT8_C(-104), INT8_C(  27), INT8_C(  96),
                           INT8_C( -89), INT8_C(  31), INT8_C(  20), INT8_C(  31),
                           INT8_C( -95), INT8_C(  13), INT8_C( -81), INT8_C( -81),
                           INT8_C( -81), INT8_C( -81), INT8_C( -81), INT8_C( -81),
                           INT8_C(  41), INT8_C(  25), INT8_C( -42), INT8_C( -81),
                           INT8_C(  31), INT8_C( -81), INT8_C( -81), INT8_C( -89),
                           INT8_C( 103), INT8_C( -81), INT8_C( -29), INT8_C( -81),
                           INT8_C(  71), INT8_C( -81), INT8_C( -23), INT8_C( -81),
                           INT8_C( -81), INT8_C( -81), INT8_C( -81), INT8_C( -12),
                           INT8_C( -81), INT8_C(  39), INT8_C( -32), INT8_C( -94)) },
    { simde_mm512_set_epi8(INT8_C(  85), INT8_C(  18), INT8_C(-117), INT8_C( -50),
                           INT8_C(  -8), INT8_C( 126), INT8_C( 103), INT8_C( -42),
                           INT8_C( 107), INT8_C( -60), INT8_C( -85), INT8_C( 123),
                           INT8_C( -11), INT8_C(  41), INT8_C(  98), INT8_C( 115),
                           INT8_C(  14), INT8_C(  34), INT8_C(  89), INT8_C( 101),
                           INT8_C(  39), INT8_C(  26), INT8_C( 121), INT8_C(  70),
                           INT8_C( -20), INT8_C( -34), INT8_C( -11), INT8_C(  72),
                           INT8_C(   8), INT8_C( -24), INT8_C(-104), INT8_C(  61),
                           INT8_C(-108), INT8_C( -43), INT8_C( 102), INT8_C( 100),
                           INT8_C( -29), INT8_C( -21), INT8_C(  70), INT8_C( -28),
                           INT8_C( -21), INT8_C( -82), INT8_C( -18), INT8_C(   9),
                           INT8_C(  94), INT8_C( -32), INT8_C(  97), INT8_C( -86),
                           INT8_C(  87), INT8_C(  62), INT8_C(-118), INT8_C(  17),
                           INT8_C(  18), INT8_C(-126), INT8_C(  74), INT8_C( -83),
                           INT8_C( -46), INT8_C(-103), INT8_C( -21), INT8_C( 108),
                           INT8_C( -58), INT8_C(-126), INT8_C( -28), INT8_C(-112)),
      UINT64_C(           923153287),
      simde_mm_set_epi8(INT8_C(  73), INT8_C( -73), INT8_C( -11), INT8_C(  36),
                        INT8_C( -17), INT8_C(  70), INT8_C(-102), INT8_C(-111),
                        INT8_C(  27), INT8_C( -97), INT8_C(  -6), INT8_C(  -7),
                        INT8_C(  28), INT8_C( -52), INT8_C( -54), INT8_C( -50)),
      simde_mm512_set_epi8(INT8_C(  85), INT8_C(  18), INT8_C(-117), INT8_C( -50),
                           INT8_C(  -8), INT8_C( 126), INT8_C( 103), INT8_C( -42),
                           INT8_C( 107), INT8_C( -60), INT8_C( -85), INT8_C( 123),
                           INT8_C( -11), INT8_C(  41), INT8_C(  98), INT8_C( 115),
                           INT8_C(  14), INT8_C(  34), INT8_C(  89), INT8_C( 101),
                           INT8_C(  39), INT8_C(  26), INT8_C( 121), INT8_C(  70),
                           INT8_C( -20), INT8_C( -34), INT8_C( -11), INT8_C(  72),
                           INT8_C(   8), INT8_C( -24), INT8_C(-104), INT8_C(  61),
                           INT8_C(-108), INT8_C( -43), INT8_C( -50), INT8_C( -50),
                           INT8_C( -29), INT8_C( -50), INT8_C( -50), INT8_C( -50),
                           INT8_C( -21), INT8_C( -82), INT8_C( -18), INT8_C(   9),
                           INT8_C(  94), INT8_C( -50), INT8_C( -50), INT8_C( -86),
                           INT8_C(  87), INT8_C(  62), INT8_C( -50), INT8_C( -50),
                           INT8_C(  18), INT8_C(-126), INT8_C( -50), INT8_C( -50),
                           INT8_C( -50), INT8_C(-103), INT8_C( -21), INT8_C( 108),
                           INT8_C( -58), INT8_C( -50), INT8_C( -50), INT8_C( -50)) },
    { simde_mm512_set_epi8(INT8_C(  67), INT8_C(-107), INT8_C(  82), INT8_C(  55),
                           INT8_C(  64), INT8_C(  72), INT8_C( -53), INT8_C(  66),
                           INT8_C( -50), INT8_C( 103), INT8_C( -13), INT8_C(  78),
                           INT8_C(  15), INT8_C(  32), INT8_C(  76), INT8_C(  78),
                           INT8_C(  28), INT8_C( -98), INT8_C(-128), INT8_C(  80),
                           INT8_C( 106), INT8_C( -45), INT8_C(  79), INT8_C( 116),
                           INT8_C(  23), INT8_C(  31), INT8_C( 117), INT8_C( -12),
                           INT8_C( -59), INT8_C( -16), INT8_C(  98), INT8_C( -49),
                           INT8_C( 116), INT8_C( -82), INT8_C(  92), INT8_C(   1),
                           INT8_C(  30), INT8_C(-100), INT8_C(  61), INT8_C( -14),
                           INT8_C(  26), INT8_C( -40), INT8_C( -78), INT8_C( -85),
                           INT8_C( -24), INT8_C( -47), INT8_C( -93), INT8_C(  -1),
                           INT8_C(  21), INT8_C(  82), INT8_C( 119), INT8_C(  64),
                           INT8_C(  74), INT8_C( -53), INT8_C(  58), INT8_C(  33),
                           INT8_C(  14), INT8_C( 114), INT8_C(  35), INT8_C( 109),
                           INT8_C( -74), INT8_C( -59), INT8_C( -81), INT8_C(  16)),
      UINT64_C(           594368556),
      simde_mm_set_epi8(INT8_C(  26), INT8_C( -78), INT8_C(  32), INT8_C(  10),
                        INT8_C(-126), INT8_C(  64), INT8_C(  35), INT8_C( -54),
                        INT8_C( -42), INT8_C( -70), INT8_C( 114), INT8_C( 111),
                        INT8_C( 111), INT8_C(  11), INT8_C( 104), INT8_C(  39)),
      simde_mm512_set_epi8(INT8_C(  67), INT8_C(-107), INT8_C(  82), INT8_C(  55),
                           INT8_C(  64), INT8_C(  72), INT8_C( -53), INT8_C(  66),
                           INT8_C( -50), INT8_C( 103), INT8_C( -13), INT8_C(  78),
                           INT8_C(  15), INT8_C(  32), INT8_C(  76), INT8_C(  78),
                           INT8_C(  28), INT8_C( -98), INT8_C(-128), INT8_C(  80),
                           INT8_C( 106), INT8_C( -45), INT8_C(  79), INT8_C( 116),
                           INT8_C(  23), INT8_C(  31), INT8_C( 117), INT8_C( -12),
                           INT8_C( -59), INT8_C( -16), INT8_C(  98), INT8_C( -49),
                           INT8_C( 116), INT8_C( -82), INT8_C(  39), INT8_C(   1),
                           INT8_C(  30), INT8_C(-100), INT8_C(  39), INT8_C(  39),
                           INT8_C(  26), INT8_C(  39), INT8_C(  39), INT8_C( -85),
                           INT8_C(  39), INT8_C(  39), INT8_C( -93), INT8_C(  39),
                           INT8_C(  21), INT8_C(  39), INT8_C( 119), INT8_C(  39),
                           INT8_C(  39), INT8_C( -53), INT8_C(  58), INT8_C(  33),
                           INT8_C(  14), INT8_C( 114), INT8_C(  39), INT8_C( 109),
                           INT8_C(  39), INT8_C(  39), INT8_C( -81), INT8_C(  16)) },
    { simde_mm512_set_epi8(INT8_C( 124), INT8_C(  71), INT8_C(-128), INT8_C( 110),
                           INT8_C(-123), INT8_C( -14), INT8_C( 123), INT8_C( -42),
                           INT8_C(  94), INT8_C(  60), INT8_C( 116), INT8_C( -89),
                           INT8_C(  73), INT8_C( -61), INT8_C(  -3), INT8_C(-114),
                           INT8_C( -92), INT8_C( -78), INT8_C(  90), INT8_C(  44),
                           INT8_C( -84), INT8_C( -33), INT8_C( 116), INT8_C(  -6),
                           INT8_C( -44), INT8_C( 126), INT8_C( -26), INT8_C(  80),
                           INT8_C( -91), INT8_C(-125), INT8_C(  72), INT8_C(  -8),
                           INT8_C( -16), INT8_C(  95), INT8_C( -25), INT8_C( -16),
                           INT8_C( -52), INT8_C( 116), INT8_C( -23), INT8_C(-102),
                           INT8_C( 119), INT8_C( -76), INT8_C(  48), INT8_C(  26),
                           INT8_C(-128), INT8_C(  43), INT8_C(  99), INT8_C( -34),
                           INT8_C(-103), INT8_C( -40), INT8_C(  47), INT8_C(-112),
                           INT8_C(-117), INT8_C( 111), INT8_C(-126), INT8_C(-115),
                           INT8_C(  65), INT8_C( -55), INT8_C(  49), INT8_C(  37),
                           INT8_C(-110), INT8_C(-124), INT8_C( 126), INT8_C(  -2)),
      UINT64_C(          1610616610),
      simde_mm_set_epi8(INT8_C( -95), INT8_C(  29), INT8_C( -58), INT8_C( -87),
                        INT8_C(  73), INT8_C(  12), INT8_C( -29), INT8_C(  41),
                        INT8_C( -96), INT8_C( 122), INT8_C( -95), INT8_C( -33),
                        INT8_C(-128), INT8_C(   2), INT8_C( 115), INT8_C( 108)),
      simde_mm512_set_epi8(INT8_C( 124), INT8_C(  71), INT8_C(-128), INT8_C( 110),
                           INT8_C(-123), INT8_C( -14), INT8_C( 123), INT8_C( -42),
                           INT8_C(  94), INT8_C(  60), INT8_C( 116), INT8_C( -89),
                           INT8_C(  73), INT8_C( -61), INT8_C(  -3), INT8_C(-114),
                           INT8_C( -92), INT8_C( -78), INT8_C(  90), INT8_C(  44),
                           INT8_C( -84), INT8_C( -33), INT8_C( 116), INT8_C(  -6),
                           INT8_C( -44), INT8_C( 126), INT8_C( -26), INT8_C(  80),
                           INT8_C( -91), INT8_C(-125), INT8_C(  72), INT8_C(  -8),
                           INT8_C( -16), INT8_C( 108), INT8_C( 108), INT8_C( -16),
                           INT8_C( -52), INT8_C( 116), INT8_C( -23), INT8_C(-102),
                           INT8_C( 119), INT8_C( -76), INT8_C(  48), INT8_C(  26),
                           INT8_C(-128), INT8_C(  43), INT8_C(  99), INT8_C( -34),
                           INT8_C(-103), INT8_C( -40), INT8_C(  47), INT8_C(-112),
                           INT8_C( 108), INT8_C( 108), INT8_C( 108), INT8_C( 108),
                           INT8_C(  65), INT8_C( -55), INT8_C( 108), INT8_C(  37),
                           INT8_C(-110), INT8_C(-124), INT8_C( 108), INT8_C(  -2)) },
    { simde_mm512_set_epi8(INT8_C(  73), INT8_C( -95), INT8_C( -44), INT8_C( 123),
                           INT8_C( -34), INT8_C(-122), INT8_C( 105), INT8_C( -63),
                           INT8_C( -13), INT8_C( -78), INT8_C(  -7), INT8_C(  88),
                           INT8_C(-101), INT8_C(  60), INT8_C(  29), INT8_C( -15),
                           INT8_C(  87), INT8_C( -77), INT8_C(  65), INT8_C(  71),
                           INT8_C( 113), INT8_C(-124), INT8_C( -41), INT8_C( -18),
                           INT8_C(  37), INT8_C( -20), INT8_C( 112), INT8_C(  70),
                           INT8_C(  36), INT8_C( -80), INT8_C( 122), INT8_C( -28),
                           INT8_C( -45), INT8_C(-113), INT8_C(  68), INT8_C(  23),
                           INT8_C(  84), INT8_C(  56), INT8_C( -44), INT8_C( -61),
                           INT8_C( -78), INT8_C(   6), INT8_C(-108), INT8_C(  73),
                           INT8_C( -22), INT8_C( -71), INT8_C(   1), INT8_C(   7),
                           INT8_C(  47), INT8_C(  18), INT8_C(-127), INT8_C( 127),
                           INT8_C( -16), INT8_C( -48), INT8_C( -39), INT8_C( 106),
                           INT8_C(  27), INT8_C(  40), INT8_C( -58), INT8_C( -56),
                           INT8_C( -27), INT8_C(  17), INT8_C(  29), INT8_C( -46)),
      UINT64_C(          2168160586),
      simde_mm_set_epi8(INT8_C(  45), INT8_C(  89), INT8_C( -40), INT8_C(  94),
                        INT8_C( -55), INT8_C( -34), INT8_C(-119), INT8_C(-109),
                        INT8_C(   3), INT8_C(-117), INT8_C(-101), INT8_C(  63),
                        INT8_C( 122), INT8_C(  -4), INT8_C(-100), INT8_C( -84)),
      simde_mm512_set_epi8(INT8_C(  73), INT8_C( -95), INT8_C( -44), INT8_C( 123),
                           INT8_C( -34), INT8_C(-122), INT8_C( 105), INT8_C( -63),
                           INT8_C( -13), INT8_C( -78), INT8_C(  -7), INT8_C(  88),
                           INT8_C(-101), INT8_C(  60), INT8_C(  29), INT8_C( -15),
                           INT8_C(  87), INT8_C( -77), INT8_C(  65), INT8_C(  71),
                           INT8_C( 113), INT8_C(-124), INT8_C( -41), INT8_C( -18),
                           INT8_C(  37), INT8_C( -20), INT8_C( 112), INT8_C(  70),
                           INT8_C(  36), INT8_C( -80), INT8_C( 122), INT8_C( -28),
                           INT8_C( -84), INT8_C(-113), INT8_C(  68), INT8_C(  23),
                           INT8_C(  84), INT8_C(  56), INT8_C( -44), INT8_C( -84),
                           INT8_C( -78), INT8_C(   6), INT8_C( -84), INT8_C( -84),
                           INT8_C( -84), INT8_C( -71), INT8_C( -84), INT8_C( -84),
                           INT8_C( -84), INT8_C(  18), INT8_C(-127), INT8_C( 127),
                           INT8_C( -16), INT8_C( -48), INT8_C( -39), INT8_C( -84),
                           INT8_C(  27), INT8_C( -84), INT8_C( -58), INT8_C( -56),
                           INT8_C( -84), INT8_C(  17), INT8_C( -84), INT8_C( -46)) },
    { simde_mm512_set_epi8(INT8_C(  38), INT8_C( -12), INT8_C( -37), INT8_C(  58),
                           INT8_C(  89), INT8_C(-127), INT8_C( -11), INT8_C(  26),
                           INT8_C( -29), INT8_C(-122), INT8_C(  86), INT8_C(  69),
                           INT8_C(  63), INT8_C(  74), INT8_C(  90), INT8_C(  88),
                           INT8_C( -75), INT8_C( -43), INT8_C(  36), INT8_C(  61),
                           INT8_C( -19), INT8_C(  27), INT8_C(-123), INT8_C(  78),
                           INT8_C(  67), INT8_C(  58), INT8_C( -32), INT8_C(  42),
                           INT8_C(  25), INT8_C( -26), INT8_C( 122), INT8_C(-100),
                           INT8_C(-107), INT8_C( -53), INT8_C(-114), INT8_C(  63),
                           INT8_C(-100), INT8_C(  53), INT8_C( -32), INT8_C( -39),
                           INT8_C( -75), INT8_C(-119), INT8_C( -67), INT8_C(  96),
                           INT8_C(  -6), INT8_C( -22), INT8_C( -12), INT8_C(  19),
                           INT8_C( -51), INT8_C(  42), INT8_C(  39), INT8_C(-124),
                           INT8_C(  38), INT8_C( -95), INT8_C(-119), INT8_C(  -9),
                           INT8_C(  94), INT8_C( -51), INT8_C(   1), INT8_C( -64),
                           INT8_C( -67), INT8_C(-127), INT8_C( -33), INT8_C(  75)),
      UINT64_C(          3579095368),
      simde_mm_set_epi8(INT8_C( -71), INT8_C(-112), INT8_C(-122), INT8_C( -13),
                        INT8_C(-109), INT8_C(  21), INT8_C(  27), INT8_C(-109),
                        INT8_C(  55), INT8_C(   9), INT8_C( 117), INT8_C( -28),
                        INT8_C( -58), INT8_C(  -1), INT8_C(   3), INT8_C( -34)),
      simde_mm512_set_epi8(INT8_C(  38), INT8_C( -12), INT8_C( -37), INT8_C(  58),
                           INT8_C(  89), INT8_C(-127), INT8_C( -11), INT8_C(  26),
                           INT8_C( -29), INT8_C(-122), INT8_C(  86), INT8_C(  69),
                           INT8_C(  63), INT8_C(  74), INT8_C(  90), INT8_C(  88),
                           INT8_C( -75), INT8_C( -43), INT8_C(  36), INT8_C(  61),
                           INT8_C( -19), INT8_C(  27), INT8_C(-123), INT8_C(  78),
                           INT8_C(  67), INT8_C(  58), INT8_C( -32), INT8_C(  42),
                           INT8_C(  25), INT8_C( -26), INT8_C( 122), INT8_C(-100),
                           INT8_C( -34), INT8_C( -34), INT8_C(-114), INT8_C( -34),
                           INT8_C(-100), INT8_C( -34), INT8_C( -32), INT8_C( -34),
                           INT8_C( -75), INT8_C( -34), INT8_C( -67), INT8_C( -34),
                           INT8_C(  -6), INT8_C( -34), INT8_C( -12), INT8_C(  19),
                           INT8_C( -34), INT8_C(  42), INT8_C( -34), INT8_C(-124),
                           INT8_C( -34), INT8_C( -95), INT8_C(-119), INT8_C( -34),
                           INT8_C(  94), INT8_C( -34), INT8_C(   1), INT8_C( -64),
                           INT8_C( -34), INT8_C(-127), INT8_C( -33), INT8_C(  75)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_broadcastb_epi8(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_broadcastb_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask64 k;
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { UINT64_C(          2081702095),
      simde_mm_set_epi8(INT8_C( 126), INT8_C(  -6), INT8_C(  16), INT8_C( 102),
                        INT8_C( -47), INT8_C(-116), INT8_C(  -4), INT8_C(  33),
                        INT8_C( -25), INT8_C(-108), INT8_C(-115), INT8_C(-104),
                        INT8_C( -39), INT8_C(  49), INT8_C(  72), INT8_C(  44)),
      simde_mm512_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(  44), INT8_C(  44), INT8_C(  44),
                           INT8_C(  44), INT8_C(  44), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(  44),
                           INT8_C(   0), INT8_C(  44), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(  44), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(  44), INT8_C(  44), INT8_C(   0), INT8_C(   0),
                           INT8_C(  44), INT8_C(  44), INT8_C(  44), INT8_C(  44)) },
    { UINT64_C(          4229458596),
      simde_mm_set_epi8(INT8_C(-106), INT8_C(-123), INT8_C( 120), INT8_C(  43),
                        INT8_C( -31), INT8_C(   4), INT8_C(  10), INT8_C(  96),
                        INT8_C( -40), INT8_C(  23), INT8_C(  31), INT8_C(  73),
                        INT8_C( -51), INT8_C(  91), INT8_C(  68), INT8_C( -23)),
      simde_mm512_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C( -23), INT8_C( -23), INT8_C( -23), INT8_C( -23),
                           INT8_C( -23), INT8_C( -23), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C( -23),
                           INT8_C( -23), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C( -23), INT8_C( -23), INT8_C(   0),
                           INT8_C( -23), INT8_C(   0), INT8_C( -23), INT8_C(   0),
                           INT8_C( -23), INT8_C(   0), INT8_C( -23), INT8_C(   0),
                           INT8_C(   0), INT8_C( -23), INT8_C(   0), INT8_C(   0)) },
    { UINT64_C(          3399230491),
      simde_mm_set_epi8(INT8_C( -40), INT8_C( -29), INT8_C(  78), INT8_C(  94),
                        INT8_C( -79), INT8_C(  10), INT8_C(-103), INT8_C(-109),
                        INT8_C(  65), INT8_C( -68), INT8_C( 102), INT8_C(-122),
                        INT8_C(  40), INT8_C(  19), INT8_C(-111), INT8_C(   8)),
      simde_mm512_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   8), INT8_C(   8), INT8_C(   0), INT8_C(   0),
                           INT8_C(   8), INT8_C(   0), INT8_C(   8), INT8_C(   0),
                           INT8_C(   8), INT8_C(   0), INT8_C(   0), INT8_C(   8),
                           INT8_C(   8), INT8_C(   8), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   8), INT8_C(   0),
                           INT8_C(   0), INT8_C(   8), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   8),
                           INT8_C(   8), INT8_C(   0), INT8_C(   8), INT8_C(   8)) },
    { UINT64_C(          2871199873),
      simde_mm_set_epi8(INT8_C(  27), INT8_C(  40), INT8_C(  10), INT8_C(-105),
                        INT8_C(  76), INT8_C(-101), INT8_C(  87), INT8_C( 112),
                        INT8_C( -78), INT8_C( -60), INT8_C(-122), INT8_C( -66),
                        INT8_C(   8), INT8_C( -42), INT8_C(  44), INT8_C(  45)),
      simde_mm512_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(  45), INT8_C(   0), INT8_C(  45), INT8_C(   0),
                           INT8_C(  45), INT8_C(   0), INT8_C(  45), INT8_C(  45),
                           INT8_C(   0), INT8_C(   0), INT8_C(  45), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(  45), INT8_C(  45),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(  45), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(  45), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(  45)) },
    { UINT64_C(          1884717912),
      simde_mm_set_epi8(INT8_C(  48), INT8_C(  40), INT8_C(-108), INT8_C( -12),
                        INT8_C( -11), INT8_C( -71), INT8_C(-114), INT8_C( -36),
                        INT8_C( -92), INT8_C( 101), INT8_C(  30), INT8_C(  10),
                        INT8_C(  43), INT8_C(-116), INT8_C( -45), INT8_C(-104)),
      simde_mm512_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(-104), INT8_C(-104), INT8_C(-104),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(-104), INT8_C(   0), INT8_C(-104),
                           INT8_C(   0), INT8_C(-104), INT8_C(-104), INT8_C(   0),
                           INT8_C(-104), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(-104), INT8_C(-104),
                           INT8_C(   0), INT8_C(-104), INT8_C(   0), INT8_C(-104),
                           INT8_C(-104), INT8_C(   0), INT8_C(   0), INT8_C(   0)) },
    { UINT64_C(           867744017),
      simde_mm_set_epi8(INT8_C( 104), INT8_C( 101), INT8_C(  53), INT8_C(-121),
                        INT8_C( 102), INT8_C(-115), INT8_C(  90), INT8_C(  31),
                        INT8_C(  11), INT8_C(  68), INT8_C(  48), INT8_C(   4),
                        INT8_C(  55), INT8_C( -83), INT8_C(  75), INT8_C( -60)),
      simde_mm512_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C( -60), INT8_C( -60),
                           INT8_C(   0), INT8_C(   0), INT8_C( -60), INT8_C( -60),
                           INT8_C( -60), INT8_C(   0), INT8_C( -60), INT8_C( -60),
                           INT8_C( -60), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C( -60), INT8_C(   0), INT8_C( -60), INT8_C( -60),
                           INT8_C( -60), INT8_C(   0), INT8_C(   0), INT8_C( -60),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C( -60),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C( -60)) },
    { UINT64_C(          2018049220),
      simde_mm_set_epi8(INT8_C(  52), INT8_C( -63), INT8_C( -88), INT8_C( -55),
                        INT8_C(  90), INT8_C( -15), INT8_C( -11), INT8_C( -21),
                        INT8_C( 100), INT8_C( -84), INT8_C( -92), INT8_C( -78),
                        INT8_C(  27), INT8_C(  91), INT8_C(  46), INT8_C(-117)),
      simde_mm512_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(-117), INT8_C(-117), INT8_C(-117),
                           INT8_C(-117), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(-117), INT8_C(   0), INT8_C(   0),
                           INT8_C(-117), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(-117), INT8_C(-117), INT8_C(-117), INT8_C(-117),
                           INT8_C(-117), INT8_C(-117), INT8_C(   0), INT8_C(   0),
                           INT8_C(-117), INT8_C(-117), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(-117), INT8_C(   0), INT8_C(   0)) },
    { UINT64_C(          1858505628),
      simde_mm_set_epi8(INT8_C( 103), INT8_C( -85), INT8_C( -29), INT8_C(  86),
                        INT8_C(  71), INT8_C(  28), INT8_C( -23), INT8_C(  28),
                        INT8_C( -53), INT8_C( -82), INT8_C(  58), INT8_C( -12),
                        INT8_C(  63), INT8_C(  39), INT8_C( -32), INT8_C( -94)),
      simde_mm512_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C( -94), INT8_C( -94), INT8_C(   0),
                           INT8_C( -94), INT8_C( -94), INT8_C( -94), INT8_C(   0),
                           INT8_C( -94), INT8_C( -94), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C( -94), INT8_C( -94), INT8_C(   0),
                           INT8_C( -94), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C( -94), INT8_C(   0), INT8_C( -94), INT8_C( -94),
                           INT8_C( -94), INT8_C(   0), INT8_C(   0), INT8_C( -94),
                           INT8_C( -94), INT8_C( -94), INT8_C(   0), INT8_C(   0)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_broadcastb_epi8(test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_broadcastw_epi16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm_set_epi16(INT16_C( -4264), INT16_C(-31095), INT16_C( 28503), INT16_C(-19429),
                         INT16_C(-23560), INT16_C( -4328), INT16_C( 17780), INT16_C(-19836)),
      simde_mm512_set_epi16(INT16_C(-19836), INT16_C(-19836), INT16_C(-19836), INT16_C(-19836),
                            INT16_C(-19836), INT16_C(-19836), INT16_C(-19836), INT16_C(-19836),
                            INT16_C(-19836), INT16_C(-19836), INT16_C(-19836), INT16_C(-19836),
                            INT16_C(-19836), INT16_C(-19836), INT16_C(-19836), INT16_C(-19836),
                            INT16_C(-19836), INT16_C(-19836), INT16_C(-19836), INT16_C(-19836),
                            INT16_C(-19836), INT16_C(-19836), INT16_C(-19836), INT16_C(-19836),
                            INT16_C(-19836), INT16_C(-19836), INT16_C(-19836), INT16_C(-19836),
                            INT16_C(-19836), INT16_C(-19836), INT16_C(-19836), INT16_C(-19836)) },
    { simde_mm_set_epi16(INT16_C( -1138), INT16_C(-21762), INT16_C(  8538), INT16_C(-12772),
                         INT16_C(  3852), INT16_C(  8246), INT16_C( -3641), INT16_C(  9422)),
      simde_mm512_set_epi16(INT16_C(  9422), INT16_C(  9422), INT16_C(  9422), INT16_C(  9422),
                            INT16_C(  9422), INT16_C(  9422), INT16_C(  9422), INT16_C(  9422),
                            INT16_C(  9422), INT16_C(  9422), INT16_C(  9422), INT16_C(  9422),
                            INT16_C(  9422), INT16_C(  9422), INT16_C(  9422), INT16_C(  9422),
                            INT16_C(  9422), INT16_C(  9422), INT16_C(  9422), INT16_C(  9422),
                            INT16_C(  9422), INT16_C(  9422), INT16_C(  9422), INT16_C(  9422),
                            INT16_C(  9422), INT16_C(  9422), INT16_C(  9422), INT16_C(  9422),
                            INT16_C(  9422), INT16_C(  9422), INT16_C(  9422), INT16_C(  9422)) },
    { simde_mm_set_epi16(INT16_C(-12364), INT16_C(-15754), INT16_C(  -793), INT16_C(-14722),
                         INT16_C(-29314), INT16_C(-26497), INT16_C(  3881), INT16_C( 17439)),
      simde_mm512_set_epi16(INT16_C( 17439), INT16_C( 17439), INT16_C( 17439), INT16_C( 17439),
                            INT16_C( 17439), INT16_C( 17439), INT16_C( 17439), INT16_C( 17439),
                            INT16_C( 17439), INT16_C( 17439), INT16_C( 17439), INT16_C( 17439),
                            INT16_C( 17439), INT16_C( 17439), INT16_C( 17439), INT16_C( 17439),
                            INT16_C( 17439), INT16_C( 17439), INT16_C( 17439), INT16_C( 17439),
                            INT16_C( 17439), INT16_C( 17439), INT16_C( 17439), INT16_C( 17439),
                            INT16_C( 17439), INT16_C( 17439), INT16_C( 17439), INT16_C( 17439),
                            INT16_C( 17439), INT16_C( 17439), INT16_C( 17439), INT16_C( 17439)) },
    { simde_mm_set_epi16(INT16_C( -7447), INT16_C(-10523), INT16_C(-25861), INT16_C(-22174),
                         INT16_C(  8521), INT16_C( 32120), INT16_C(-17861), INT16_C( 31790)),
      simde_mm512_set_epi16(INT16_C( 31790), INT16_C( 31790), INT16_C( 31790), INT16_C( 31790),
                            INT16_C( 31790), INT16_C( 31790), INT16_C( 31790), INT16_C( 31790),
                            INT16_C( 31790), INT16_C( 31790), INT16_C( 31790), INT16_C( 31790),
                            INT16_C( 31790), INT16_C( 31790), INT16_C( 31790), INT16_C( 31790),
                            INT16_C( 31790), INT16_C( 31790), INT16_C( 31790), INT16_C( 31790),
                            INT16_C( 31790), INT16_C( 31790), INT16_C( 31790), INT16_C( 31790),
                            INT16_C( 31790), INT16_C( 31790), INT16_C( 31790), INT16_C( 31790),
                            INT16_C( 31790), INT16_C( 31790), INT16_C( 31790), INT16_C( 31790)) },
    { simde_mm_set_epi16(INT16_C( -4580), INT16_C( -4681), INT16_C( -4797), INT16_C( 20435),
                         INT16_C(-31664), INT16_C(-25722), INT16_C(-13794), INT16_C( -4041)),
      simde_mm512_set_epi16(INT16_C( -4041), INT16_C( -4041), INT16_C( -4041), INT16_C( -4041),
                            INT16_C( -4041), INT16_C( -4041), INT16_C( -4041), INT16_C( -4041),
                            INT16_C( -4041), INT16_C( -4041), INT16_C( -4041), INT16_C( -4041),
                            INT16_C( -4041), INT16_C( -4041), INT16_C( -4041), INT16_C( -4041),
                            INT16_C( -4041), INT16_C( -4041), INT16_C( -4041), INT16_C( -4041),
                            INT16_C( -4041), INT16_C( -4041), INT16_C( -4041), INT16_C( -4041),
                            INT16_C( -4041), INT16_C( -4041), INT16_C( -4041), INT16_C( -4041),
                            INT16_C( -4041), INT16_C( -4041), INT16_C( -4041), INT16_C( -4041)) },
    { simde_mm_set_epi16(INT16_C(  1787), INT16_C(  9631), INT16_C(  4347), INT16_C( -4594),
                         INT16_C(-30523), INT16_C(-10849), INT16_C(-17993), INT16_C(-18482)),
      simde_mm512_set_epi16(INT16_C(-18482), INT16_C(-18482), INT16_C(-18482), INT16_C(-18482),
                            INT16_C(-18482), INT16_C(-18482), INT16_C(-18482), INT16_C(-18482),
                            INT16_C(-18482), INT16_C(-18482), INT16_C(-18482), INT16_C(-18482),
                            INT16_C(-18482), INT16_C(-18482), INT16_C(-18482), INT16_C(-18482),
                            INT16_C(-18482), INT16_C(-18482), INT16_C(-18482), INT16_C(-18482),
                            INT16_C(-18482), INT16_C(-18482), INT16_C(-18482), INT16_C(-18482),
                            INT16_C(-18482), INT16_C(-18482), INT16_C(-18482), INT16_C(-18482),
                            INT16_C(-18482), INT16_C(-18482), INT16_C(-18482), INT16_C(-18482)) },
    { simde_mm_set_epi16(INT16_C( 30524), INT16_C( 16358), INT16_C( 12856), INT16_C( 10489),
                         INT16_C( 17653), INT16_C( -5197), INT16_C( 14483), INT16_C(-30060)),
      simde_mm512_set_epi16(INT16_C(-30060), INT16_C(-30060), INT16_C(-30060), INT16_C(-30060),
                            INT16_C(-30060), INT16_C(-30060), INT16_C(-30060), INT16_C(-30060),
                            INT16_C(-30060), INT16_C(-30060), INT16_C(-30060), INT16_C(-30060),
                            INT16_C(-30060), INT16_C(-30060), INT16_C(-30060), INT16_C(-30060),
                            INT16_C(-30060), INT16_C(-30060), INT16_C(-30060), INT16_C(-30060),
                            INT16_C(-30060), INT16_C(-30060), INT16_C(-30060), INT16_C(-30060),
                            INT16_C(-30060), INT16_C(-30060), INT16_C(-30060), INT16_C(-30060),
                            INT16_C(-30060), INT16_C(-30060), INT16_C(-30060), INT16_C(-30060)) },
    { simde_mm_set_epi16(INT16_C(-28607), INT16_C(  6822), INT16_C(-19640), INT16_C(   516),
                         INT16_C(-13138), INT16_C( -4418), INT16_C(-29962), INT16_C( 13528)),
      simde_mm512_set_epi16(INT16_C( 13528), INT16_C( 13528), INT16_C( 13528), INT16_C( 13528),
                            INT16_C( 13528), INT16_C( 13528), INT16_C( 13528), INT16_C( 13528),
                            INT16_C( 13528), INT16_C( 13528), INT16_C( 13528), INT16_C( 13528),
                            INT16_C( 13528), INT16_C( 13528), INT16_C( 13528), INT16_C( 13528),
                            INT16_C( 13528), INT16_C( 13528), INT16_C( 13528), INT16_C( 13528),
                            INT16_C( 13528), INT16_C( 13528), INT16_C( 13528), INT16_C( 13528),
                            INT16_C( 13528), INT16_C( 13528), INT16_C( 13528), INT16_C( 13528),
                            INT16_C( 13528), INT16_C( 13528), INT16_C( 13528), INT16_C( 13528)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_broadcastw_epi16(test_vec[i].a);
    simde_assert_m512i_i16(r, ==, test_vec[i].r);
  }

  return 0;
}

SIMDE_TEST_FUNC_LIST_BEGIN

  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_broadcast_f32x2)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_mask_broadcast_f32x2)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_maskz_broadcast_f32x2)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_broadcast_f32x2)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_broadcast_f32x2)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_broadcast_f32x2)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_broadcast_f32x8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_broadcast_f32x8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_broadcast_f32x8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_broadcast_f64x2)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_broadcast_f64x2)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_broadcast_f64x2)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_broadcast_f32x4)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_mask_broadcast_f32x4)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_maskz_broadcast_f32x4)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_broadcast_f64x2)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_mask_broadcast_f64x2)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_maskz_broadcast_f64x2)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_broadcast_f32x4)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_broadcast_f32x4)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_broadcast_f32x4)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_broadcast_f64x4)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_broadcast_f64x4)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_broadcast_f64x4)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_broadcast_i32x4)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_broadcast_i32x4)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_broadcast_i32x4)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_broadcast_i64x4)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_broadcast_i64x4)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_broadcast_i64x4)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_broadcastd_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_broadcastd_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_broadcastd_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_broadcastq_epi64)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_broadcastq_epi64)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_broadcastq_epi64)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_broadcastss_ps)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_broadcastss_ps)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_broadcastss_ps)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_broadcastsd_pd)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_broadcastsd_pd)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_broadcastsd_pd)

  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_broadcastb_epi8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_broadcastb_epi8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_broadcastb_epi8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_broadcastw_epi16)
SIMDE_TEST_FUNC_LIST_END

#include <test/x86/avx512/test-avx512-footer.h>
