# Sorune Libraries
# Copyright (C) 2004-2005 Darren Smith
# All Rights Reserved.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#
# e-mail: sorune2004@yahoo.com

use strict;
use File::Path;
use File::Basename;
use FileHandle;

# Create the Neuros Audio Menu
sub createNAM($$$$$$$)
{
    my ($cfgRef,$musicDbRef,$playlistDbRef,$recordDbRef,$neurosDbHome,
        $fwVersion,$top) = @_;
    my %locations = ();
    my ($artalb,$genart,$genalb) = (0,0,0);

    # setup the Neuros Audio Menu
    my ($neurosAudio,$songs,$playlists,$artists,$albums,$genres,$years,
        $recordings) = ("Neuros Audio","Songs","Playlists","Artists","Albums",
        "Genres","Years","Recordings");
    if (defined $cfgRef->{'nam'}{'audio'}) {
        $neurosAudio = $cfgRef->{'nam'}{'audio'};
    }
    if (defined $cfgRef->{'nam'}{'songs'}) {
        $songs = $cfgRef->{'nam'}{'songs'};
    }
    if (defined $cfgRef->{'nam'}{'playlists'}) {
        $playlists = $cfgRef->{'nam'}{'playlists'};
    }
    if (defined $cfgRef->{'nam'}{'artists'}) {
        $artists = $cfgRef->{'nam'}{'artists'};
    }
    if (defined $cfgRef->{'nam'}{'albums'}) {
        $albums = $cfgRef->{'nam'}{'albums'};
    }
    if (defined $cfgRef->{'nam'}{'genres'}) {
        $genres = $cfgRef->{'nam'}{'genres'};
    }
    if (defined $cfgRef->{'nam'}{'years'}) {
        $years = $cfgRef->{'nam'}{'years'};
    }
    if (defined $cfgRef->{'nam'}{'recordings'}) {
        $recordings = $cfgRef->{'nam'}{'recordings'};
    }
    if (defined $cfgRef->{'nam'}{'artistalbum'} and
        $cfgRef->{'nam'}{'artistalbum'} eq "1") {
        $artalb = 1;
    }
    if (defined $cfgRef->{'nam'}{'genreartist'} and
        $cfgRef->{'nam'}{'genreartist'} eq "1") {
        $genart = 1;
    }
    if (defined $cfgRef->{'nam'}{'genrealbum'} and
        $cfgRef->{'nam'}{'genrealbum'} eq "1") {
        $genalb = 1;
    }

    # create the directory
    eval { mkpath $neurosDbHome, 0, 0700; };
    if ($@) {
        message('ERR',"Could not create directory $neurosDbHome: $!\n");
        return;
    }

    # Create the mdbs
    my ($artistsHL,$albumsHL) = (0,0);
    
    # Check for open source firmware version (all open source fw is > 2.14,
    # setting to 2.23 as original open source release)
    if ($fwVersion !~ /^2\.\d\d$/) { $fwVersion = '2.23'; }

    if ($fwVersion >= 2.14) {
        if ($genart) {
            $artistsHL = createMultiMDB("$neurosDbHome/artist.mdb",$musicDbRef,
                $cfgRef,\%locations,$recordDbRef,"Artist","audio.mdb",
                "artist",$top,$genres,"genreartist.mdb");
        } else {
            $artistsHL = createChildMDB("$neurosDbHome/artist.mdb",$musicDbRef,
                $cfgRef,\%locations,$recordDbRef,"Artist","audio.mdb",
                $artists,"","artist",$top);
        }
        if ($artalb and $genalb) {
            $albumsHL = createMultiMDB("$neurosDbHome/albums.mdb",$musicDbRef,
                $cfgRef,\%locations,$recordDbRef,"Albums","audio.mdb",
                "album",$top,$artists,"artistalbum.mdb",$genres,
                "genrealbum.mdb");
        } elsif ($artalb) {
            $albumsHL = createMultiMDB("$neurosDbHome/albums.mdb",$musicDbRef,
                $cfgRef,\%locations,$recordDbRef,"Albums","audio.mdb",
                "album",$top,$artists,"artistalbum.mdb");
        } elsif ($genalb) {
            $albumsHL = createMultiMDB("$neurosDbHome/albums.mdb",$musicDbRef,
                $cfgRef,\%locations,$recordDbRef,"Albums","audio.mdb",
                "album",$top,$genres,"genrealbum.mdb");
        } else {
            $albumsHL = createChildMDB("$neurosDbHome/albums.mdb",$musicDbRef,
                $cfgRef,\%locations,$recordDbRef,"Albums","audio.mdb",
                $albums,"","album",$top);
        }
    } else {
        $artistsHL = createChildMDB("$neurosDbHome/artist.mdb",$musicDbRef,
            $cfgRef,\%locations,$recordDbRef,"Artist","audio.mdb",
            $artists,"","artist",$top);
        $albumsHL = createChildMDB("$neurosDbHome/albums.mdb",$musicDbRef,
            $cfgRef,\%locations,$recordDbRef,"Albums","audio.mdb",
            $albums,"","album",$top);
    }

    my $genresHL = createChildMDB("$neurosDbHome/genre.mdb",$musicDbRef,
        $cfgRef,\%locations,$recordDbRef,"Genre","audio.mdb",$genres,"",
        "genre",$top);
    my $yearsHL = createChildMDB("$neurosDbHome/year.mdb",$musicDbRef,
        $cfgRef,\%locations,$recordDbRef,"Year","audio.mdb",$years,"",
        "date",$top);
    my $playlistsHL = createChildMDB("$neurosDbHome/playlist.mdb",
        $playlistDbRef,$cfgRef,\%locations,$recordDbRef,"Playlists",
        "audio.mdb",$playlists,"","playlist",$top);
    my $recordingsHL = createChildMDB("$neurosDbHome/recordings.mdb",
        $musicDbRef,$cfgRef,\%locations,$recordDbRef,"Recordings",
        "audio.mdb",$recordings,"","type",$top);
    my $audioHL = createAudioMDB("$neurosDbHome/audio.mdb",$musicDbRef,
        $cfgRef,\%locations,$recordDbRef,$neurosAudio,$songs,$playlists,
        $artists,$albums,$genres,$years,$recordings,$top);


    # Create the standard pais
    createPAI("$neurosDbHome/artist.pai",$cfgRef,$musicDbRef,\%locations,
        "artist","title",$top);
    createPAI("$neurosDbHome/albums.pai",$cfgRef,$musicDbRef,\%locations,
        "album","tracknumber",$top);
    createPAI("$neurosDbHome/genre.pai",$cfgRef,$musicDbRef,\%locations,
        "genre","title",$top);
    createPAI("$neurosDbHome/year.pai",$cfgRef,$musicDbRef,\%locations,
        "date","title",$top);
    createPAIPlaylist("$neurosDbHome/playlist.pai",$musicDbRef,$playlistDbRef,
        \%locations,$top);
    createPAI("$neurosDbHome/recordings.pai",$cfgRef,$recordDbRef,\%locations,
        "type","title",$top);

    # Create the standard sais
    createSAI("$neurosDbHome/artist.sai",$musicDbRef,\%locations,
        "artist",$artistsHL,$top);
    createSAI("$neurosDbHome/albums.sai",$musicDbRef,\%locations,
        "album",$albumsHL,$top);
    createSAI("$neurosDbHome/genre.sai",$musicDbRef,\%locations,
        "genre",$genresHL,$top);
    createSAI("$neurosDbHome/year.sai",$musicDbRef,\%locations,
        "date",$yearsHL,$top);
    createSAIPlaylist("$neurosDbHome/playlist.sai",$playlistDbRef,
        \%locations,$playlistsHL,$top);
    createSAI("$neurosDbHome/recordings.sai",$recordDbRef,\%locations,
        "type",$recordingsHL,$top);
    createSAI("$neurosDbHome/audio.sai",$musicDbRef,\%locations,
        "title",$audioHL,$top);

    # Create the submenus for 2.14+
    if ($fwVersion >= 2.14) {
        if ($genart) {
            my $genresArtistsHL = createChildMDB(
                "$neurosDbHome/genreartist.mdb",$musicDbRef,$cfgRef,
                \%locations,$recordDbRef,"GenreArtist","artist.mdb",
                $genres,"","genre",$top);
            createPAI("$neurosDbHome/genreartist.pai",$cfgRef,$musicDbRef,
                \%locations,"genre","artist",$top);
            createSAI("$neurosDbHome/genreartist.sai",$musicDbRef,\%locations,
                "genre",$genresArtistsHL,$top);
        }
        if ($artalb) {
            my $artistsAlbumsHL = createChildMDB(
                "$neurosDbHome/artistalbum.mdb",$musicDbRef,$cfgRef,
                \%locations,$recordDbRef,"ArtistAlbum","albums.mdb",
                $artists,"","artist",$top);
            createPAI("$neurosDbHome/artistalbum.pai",$cfgRef,$musicDbRef,
                \%locations,"artist","album",$top);
            createSAI("$neurosDbHome/artistalbum.sai",$musicDbRef,\%locations,
                "artist",$artistsAlbumsHL,$top);
        }
        if ($genalb) {
            my $genresAlbumsHL = createChildMDB(
                "$neurosDbHome/genrealbum.mdb",$musicDbRef,$cfgRef,
                \%locations,$recordDbRef,"GenreAlbum","albums.mdb",
                $genres,"","genre",$top);
            createPAI("$neurosDbHome/genrealbum.pai",$cfgRef,$musicDbRef,
                \%locations,"genre","album",$top);
            createSAI("$neurosDbHome/genrealbum.sai",$musicDbRef,\%locations,
                "genre",$genresAlbumsHL,$top);
        }
    }

    # Create misc databases
    createMiscDbs(dirname($neurosDbHome),$top);
}

sub createMultiMDB($$$$$$$$$)
{
    my ($file,$dbRef,$cfgRef,$locationsRef,$recordDbRef,$xref,$xrefFile,$dbKey,$top) =
        @_;
    my $fh = new FileHandle;
    my $buf = "";

    if (-e $file) { unlink $file; }

    splice @_, 0, 9;
    my $count = scalar @_ / 2 + 1;
    word(\$buf,0);       # Length of header (filled in later)
    word(\$buf,4);       # Bit 0 child/root, Bit 1 non-removable/removable
    word(\$buf,0);       # Bit 0 modified/non-modified
    word(\$buf,$count);  # Number of keys per record
    word(\$buf,1);       # Number of fields per record
    dword(\$buf,0);      # Pointer to record start (filled in later)
    dword(\$buf,0);      # Pointer to XIM start (filled in later)
    dword(\$buf,0);      # Reserved
    dword(\$buf,0);      # Reserved
    dword(\$buf,0);      # Reserved
    word(\$buf,0);       # Database ID

    createMenu(\$buf,
        $xref => $xrefFile,
        'All'    => '',
        @_,
    );
    $buf .= "WOID";

    my $headerLength = int(length($buf)/2);
    wordOverwrite(\$buf,0,$headerLength);
    wordOverwrite(\$buf,12,$headerLength);

    word(\$buf,0x8000);
    word(\$buf,0x0025);
    childDbRecord(\$buf,$dbKey,$dbRef,$cfgRef,$locationsRef,$recordDbRef);

    if (writeBuf($file,\$buf,$top) == -1) { return -1; }
    return $headerLength;
}

sub neurosName($) {
    if ($^O ne 'MSWin32') {
        my $string = encode('ucs-2', decodeFilename(shift));
        $string =~ s/.(.)/$1/gs;
        return $string;
    }
    return $_[0];
}

sub createAudioMDB($$$$$$$$$$$$$$)
{
    my ($file,$musicDbRef,$cfgRef,$locationsRef,$recordDbRef,$neurosAudio,
        $songs,$playlists,$artists,$albums,$genres,$years,$recordings,
        $top) = @_;
    my $fh = new FileHandle;
    my $buf = "";

    if (-e $file) { unlink $file; }

    word(\$buf,0);     # Length of header (filled in later)
    word(\$buf,1);     # Bit 0 child/root, Bit 1 non-removable/removable
    word(\$buf,0);     # Bit 0 modified/non-modified
    if ($cfgRef->{'general'}{'development'} >= 2) {
        word(\$buf,7);     # Number of keys
    } else {
        word(\$buf,6);     # Number of keys
    }
    word(\$buf,9);     # Number of fields per record
    dword(\$buf,0);    # Pointer to record start (filled in later)
    dword(\$buf,0);    # Pointer to XIM start (filled in later)
    dword(\$buf,0);    # Reserved
    dword(\$buf,0);    # Reserved
    dword(\$buf,0);    # Reserved
    word(\$buf,0);     # Database ID


    my @menuItems = ($neurosAudio => "");
    if ($cfgRef->{'general'}{'development'} >= 1) {
        foreach my $item (split /\s*,\s*/, $cfgRef->{'nam'}{'menuorder'}) {
            if ($item eq "Songs") {
                push @menuItems, ($songs => "");
            } elsif ($item eq "Playlists") {
                push @menuItems, ($playlists => "playlist.mdb");
            } elsif ($item eq "Artists") {
                push @menuItems, ($artists => "artist.mdb");
            } elsif ($item eq "Albums") {
                push @menuItems, ($albums => "albums.mdb");
            } elsif ($item eq "Genres") {
                push @menuItems, ($genres => "genre.mdb");
            } elsif ($item eq "Years") {
                if ($cfgRef->{'general'}{'development'} >= 2) {
                    push @menuItems, ($years => "year.mdb");
                }
            } elsif ($item eq "Recordings") {
                push @menuItems, ($recordings => "recordings.mdb");
            }
        }
    } else {
        push @menuItems, (
            $songs => "",
            $playlists => "playlist.mdb",
            $artists => "artist.mdb",
            $albums => "albums.mdb",
            $genres => "genre.mdb",
            $recordings => "recordings.mdb"
        );
    }
    createMenu(\$buf, @menuItems);

    my $ximStart = int(length($buf)/2);
    wordOverwrite(\$buf,16,$ximStart);

    # XIM
    dword(\$buf,0x00600008); # HEADER (XIM LENGTH, CMD COUNT)
    dword(\$buf,0x00200000); dword(\$buf,0x00000032); # CMD1, TEXT OFFSET1
    dword(\$buf,0x00000000);
    dword(\$buf,0x00240000); dword(\$buf,0x00000036);
    dword(\$buf,0x00000000);
    dword(\$buf,0x00230000); dword(\$buf,0x0000003A);
    dword(\$buf,0x00000000);
    dword(\$buf,0x00030000); dword(\$buf,0x00000043);
    dword(\$buf,0x00000000);
    dword(\$buf,0x00040000); dword(\$buf,0x00000049);
    dword(\$buf,0x00000000);
    dword(\$buf,0x00210000); dword(\$buf,0x0000004E);
    dword(\$buf,0x00000000);
    dword(\$buf,0x80020000); dword(\$buf,0x00000053);
    dword(\$buf,0x00000000);
    dword(\$buf,0x3FFF0000); dword(\$buf,0x0000005C);
    dword(\$buf,0x00000000);
    display(\$buf,"Play");
    display(\$buf,"Info");
    display(\$buf,"Add To My Mix");
    display(\$buf,"Shuffle");
    display(\$buf,"Repeat");
    display(\$buf,"Delete");
    display(\$buf,"Delete on Sync");
    display(\$buf,"Exit");

    $buf .= "WOID";

    my $headerLength = int(length($buf)/2);
    wordOverwrite(\$buf,0,$headerLength);
    wordOverwrite(\$buf,12,$headerLength);

    word(\$buf,0x8000);
    word(\$buf,0x0025);

    foreach my $key (sort 
        {lc($musicDbRef->{$a}{'title'}) cmp lc($musicDbRef->{$b}{'title'})}
        keys %$musicDbRef) {
        my $location = int(length($buf)/2);
        $locationsRef->{'title-loc'}{$key} = $location;
        $locationsRef->{'tracknumber-loc'}{$key} = $location;
        word(\$buf,0x8000);
        if ($cfgRef->{'general'}{'development'} >= 3) {
            string(\$buf, sprintf("[0x%x] %s",$location,
                $musicDbRef->{$key}{'title'}));
        } else {
            string(\$buf,$musicDbRef->{$key}{'title'});
        }
        word(\$buf,0x23); dword(\$buf,0x2E); word(\$buf,0x23);
        if ($musicDbRef->{$key}{'artist'} eq "Various" and
            defined $musicDbRef->{$key}{'artist2'} and
            $musicDbRef->{$key}{'artist2'} ne "Unknown") {
            dword(\$buf,
                $locationsRef->{'artist-loc'}{
                $musicDbRef->{$key}{'artist2'}});
        } else {
            dword(\$buf,
                $locationsRef->{'artist-loc'}{
                $musicDbRef->{$key}{'artist'}});
        }
        word(\$buf,0x23);
        dword(\$buf,
            $locationsRef->{'album-loc'}{$musicDbRef->{$key}{'album'}});
        word(\$buf,0x23);
        dword(\$buf,
            $locationsRef->{'genre-loc'}{$musicDbRef->{$key}{'genre'}});
        word(\$buf,0x23); dword(\$buf,0x2E); word(\$buf,0x23);
        dword(\$buf,$musicDbRef->{$key}{'length'});
        word(\$buf,0x23);
        dword(\$buf,$musicDbRef->{$key}{'size'}/1024);
        word(\$buf,0x23);
        string(\$buf,neurosName($key));
        dword(\$buf,0x25);
    }
    foreach my $key (sort keys %$recordDbRef) {
        $locationsRef->{'title-loc'}{$key} = int(length($buf)/2);
        word(\$buf,0x8000);
        string(\$buf,$recordDbRef->{$key}{'title'});
        word(\$buf,0x23); dword(\$buf,0x2E); word(\$buf,0x23);
        dword(\$buf,0);
        word(\$buf,0x23);
        dword(\$buf,0);
        word(\$buf,0x23);
        dword(\$buf,0);
        word(\$buf,0x23); dword(\$buf,0x2E); word(\$buf,0x23);
        dword(\$buf,$recordDbRef->{$key}{'length'});
        word(\$buf,0x23);
        dword(\$buf,$recordDbRef->{$key}{'size'}/1024);
        word(\$buf,0x23);
        string(\$buf,$key);
        dword(\$buf,0x25);
    }

    if (writeBuf($file,\$buf,$top) == -1) { return -1; }
    return $headerLength;
}

sub createChildMDB($$$$$$$$$$$)
{
    my ($file,$dbRef,$cfgRef,$locationsRef,$recordDbRef,
        $xref,$xrefFile,$title,$titleFile,$dbKey,$top) = @_;
    my $fh = new FileHandle;
    my $buf = "";

    if (-e $file) { unlink $file; }

    word(\$buf,0);     # Length of header (filled in later)
    word(\$buf,0);     # Bit 0 child/root, Bit 1 non-removable/removable
    word(\$buf,0);     # Bit 0 modified/non-modified
    word(\$buf,1);     # Number of keys per record
    word(\$buf,1);     # Number of fields per record
    dword(\$buf,0);    # Pointer to record start (filled in later)
    dword(\$buf,0);    # Pointer to XIM start (filled in later)
    dword(\$buf,0);    # Reserved
    dword(\$buf,0);    # Reserved
    dword(\$buf,0);    # Reserved
    word(\$buf,0);     # Database ID

    createMenu(\$buf,
        $xref  => $xrefFile,
        $title => $titleFile,
    );
    $buf .= "WOID";

    my $headerLength = int(length($buf)/2);
    wordOverwrite(\$buf,0,$headerLength);
    wordOverwrite(\$buf,12,$headerLength);

    word(\$buf,0x8000);
    word(\$buf,0x0025);
    if ($dbKey eq "playlist") {
        childDbPlaylistRecord(\$buf,$dbRef,$locationsRef);
    } else {
        childDbRecord(\$buf,$dbKey,$dbRef,$cfgRef,$locationsRef,$recordDbRef);
    }

    if (writeBuf($file,\$buf,$top) == -1) { return -1; }
    return $headerLength;
}

sub createMenu($@)
{
    my $bufRef = shift;
    my $entries = scalar(@_);
    my @offsets = ();

    for (my $i=0;$i<$entries;$i++) {
        push @offsets, length($$bufRef);
        dword($bufRef,0);
    }
    
    while (scalar(@_)) {
        my $name = shift;
        my $filename = shift;

        my $offset = shift @offsets;
        my $offset2 = int(length($$bufRef)/2);
        dwordOverwrite($bufRef,$offset,$offset2);
        display($bufRef,$name);

        $offset = shift @offsets;
        $offset2 = int(length($$bufRef)/2);
        if ($filename ne "") {
            dwordOverwrite($bufRef,$offset,$offset2);
            string($bufRef,$filename);
        }
    }
}

sub childDbPlaylistRecord($$$)
{
    my ($bufRef,$playlistDbRef,$locationsRef) = @_;

    foreach my $list (sort {lc($a) cmp lc($b)} keys %$playlistDbRef) {
        $locationsRef->{'playlist-loc'}{$list} = int(length($$bufRef)/2);
        word($bufRef,0x8000);
        string($bufRef,$list);
        word($bufRef,0x25);
    }
}

sub childDbRecord($$$$$$)
{
    my ($bufRef,$dbKey,$musicDbRef,$cfgRef,$locationsRef,$recordDbRef) = @_;
    my %location = ();
    my @uniqueKeys = ();
    my $found;

    if ($dbKey ne 'type') {
        my @keys = ();
        if ($dbKey eq 'date') {
            @keys = sort { lc($musicDbRef->{$b}{$dbKey}) cmp
               lc($musicDbRef->{$a}{$dbKey}) } keys %$musicDbRef;
        } else {
            @keys = sort { lc($musicDbRef->{$a}{$dbKey}) cmp
               lc($musicDbRef->{$b}{$dbKey}) } keys %$musicDbRef;
        }

        foreach my $key (@keys) {
            $found = 0;
            foreach my $uniqueKey (@uniqueKeys) {
                if ($musicDbRef->{$key}{$dbKey} eq $uniqueKey) {
                    $found = 1;
                    last;
                }
            }
            if (!$found) {
                push @uniqueKeys, $musicDbRef->{$key}{$dbKey};
                my $location = int(length($$bufRef)/2);
                $locationsRef->{"$dbKey-loc"}{$musicDbRef->{$key}{$dbKey}} =
                    $location;
                word($bufRef,0x8000);
                if ($cfgRef->{'general'}{'development'} >= 3) {
                    string($bufRef, sprintf("[0x%x] %s",$location,
                        $musicDbRef->{$key}{$dbKey}));
                } else {
                    string($bufRef,$musicDbRef->{$key}{$dbKey});
                }
                word($bufRef,0x25);
            }
        }
        # Handle special various artist case
        if ($dbKey eq 'artist') {
            foreach my $key (sort { $musicDbRef->{$a}{'title'} cmp
                $musicDbRef->{$b}{'title'} } keys %$musicDbRef) {
                if ($musicDbRef->{$key}{'artist'} eq "Various") {
                    $found = 0;
                    foreach my $uniqueKey (@uniqueKeys) {
                        if (defined $musicDbRef->{$key}{'artist2'} and
                            $musicDbRef->{$key}{'artist2'} eq $uniqueKey) {
                            $found = 1;
                            last;
                        }
                    }
                    if (!$found and defined $musicDbRef->{$key}{'artist2'}) {
                        push @uniqueKeys, $musicDbRef->{$key}{'artist2'};
                        $locationsRef->{"artist-loc"}{
                            $musicDbRef->{$key}{'artist2'}} =
                            int(length($$bufRef)/2);
                        word($bufRef,0x8000);
                        string($bufRef,$musicDbRef->{$key}{'artist2'});
                        word($bufRef,0x25);
                    }
                }
            }
        }
    } elsif (scalar keys %$recordDbRef) {
        $locationsRef->{'type-loc'}{'Microphone'} = 0;
        $locationsRef->{'type-loc'}{'Line'} = 0;
        $locationsRef->{'type-loc'}{'FM Radio'} = 0;
        foreach my $key (sort keys %$recordDbRef) {
            if ($recordDbRef->{$key}{$dbKey} eq "Microphone" and
                $locationsRef->{'type-loc'}{'Microphone'} == 0) {
                $locationsRef->{'type-loc'}{'Microphone'} = 
                    int(length($$bufRef))/2;
                word($bufRef,0x8000);
                string($bufRef,"Microphone");
                word($bufRef,0x25);
            } elsif ($recordDbRef->{$key}{$dbKey} eq "Line" and
                $locationsRef->{'type-loc'}{'Line'} == 0) {
                $locationsRef->{'type-loc'}{'Line'} = 
                    int(length($$bufRef))/2;
                word($bufRef,0x8000);
                string($bufRef,"Line");
                word($bufRef,0x25);
            } elsif ($recordDbRef->{$key}{$dbKey} eq "FM Radio" and
                $locationsRef->{'type-loc'}{'FM Radio'} == 0) {
                $locationsRef->{'type-loc'}{'FM Radio'} = 
                    int(length($$bufRef))/2;
                word($bufRef,0x8000);
                string($bufRef,"FM Radio");
                word($bufRef,0x25);
            }
        }
    }
}

sub createPAI($$$$$$$)
{
    my ($file,$cfgRef,$musicDbRef,$locationsRef,$dbKey,$pdbKey,$top) = @_;
    my $fh = new FileHandle;
    my $buf = "";

    if (-e $file) { unlink $file; }

    dword(\$buf,0x01162002);                   # Signature
    dword(\$buf,0);                            # Reserved
    dword(\$buf,0);                            # Reserved
    dword(\$buf,0);                            # Reserved

    foreach my $uniqueKey (sort { lc($a) cmp lc($b) }
        getUniqueNames($musicDbRef,$dbKey)) {
        my @files = ();
        if ($pdbKey eq 'album' or $pdbKey eq 'artist') {
            @files = getUniqueFileData($musicDbRef,$dbKey,$uniqueKey,
                $pdbKey);
        } else {
            @files = getFileData($musicDbRef,$dbKey,$uniqueKey);
            if ($dbKey eq 'artist') {
                push @files, getFileData($musicDbRef,'artist2',$uniqueKey);
            }
        }
        my $fileCount = scalar @files;
        my $noPadSize = ($fileCount * 2) + 8;
        my $padSize = 0;
        if ($noPadSize % 32) { $padSize = 32 - ($noPadSize % 32); }

        word(\$buf,$noPadSize+$padSize);     # Size of module in words
        if ($fileCount) {
            word(\$buf,0);                   # State: 1 if empty, otw 0
        } else {
            word(\$buf,1);
        }
        word(\$buf,$fileCount);              # Number of record entries
        word(\$buf,0);                       # Reserved
        dword(\$buf,0);                      # Reserved

        my $paiLocation = int(length($buf)/2);
        my @keys = ();
        if ($pdbKey eq 'tracknumber') {
            @keys = sort { $musicDbRef->{$a}{'tracknumber'} <=>
                $musicDbRef->{$b}{'tracknumber'} } @files;
        } elsif ($pdbKey eq 'album') {
            if ($uniqueKey =~ /^various/i) {
                @keys = sort { lc($musicDbRef->{$a}{'album'}) cmp
                      lc($musicDbRef->{$b}{'album'}) } @files;
            } else {
                @keys = sort { lc($musicDbRef->{$a}{'album'}) cmp
                      lc($musicDbRef->{$b}{'album'}) } @files;
                if ($cfgRef->{'nam'}{'aasort'} eq '1') {
                    @keys = sort { lc($musicDbRef->{$a}{'date'}) cmp 
                          lc($musicDbRef->{$b}{'date'}) } @keys;
                } elsif ($cfgRef->{'nam'}{'aasort'} eq '2') {
                    @keys = sort { lc($musicDbRef->{$b}{'date'}) cmp 
                          lc($musicDbRef->{$a}{'date'}) } @keys;
                }
            }
        } else {
            @keys = sort { lc($musicDbRef->{$a}{$pdbKey}) cmp
                  lc($musicDbRef->{$b}{$pdbKey}) } @files;
        }
        foreach my $key (@keys) {
            $locationsRef->{"$dbKey-pai-loc"}{
                $musicDbRef->{$key}{$dbKey}} = $paiLocation;
            if ($pdbKey eq 'tracknumber' or $pdbKey eq 'title') {
                dword(\$buf,$locationsRef->{"$pdbKey-loc"}{$key});
            } else {
                dword(\$buf,$locationsRef->{"$pdbKey-loc"}{
                    $musicDbRef->{$key}{$pdbKey}});
            }
        }

        # pad module out to multiple of 32 words
        while ($padSize) {
            dword(\$buf,0);
            $padSize -= 2;
        }
        dword(\$buf,0);                      # Marks end of module
    }

    if (writeBuf($file,\$buf,$top) == -1) { return -1; }
    return 0;
}

sub createPAIPlaylist($$$$$)
{
    my ($file,$musicDbRef,$playlistDbRef,$locationsRef,$top) = @_;
    my $fh = new FileHandle;
    my $buf = "";

    if (-e $file) { unlink $file; }

    dword(\$buf,0x01162002);                   # Signature
    dword(\$buf,0);                            # Reserved
    dword(\$buf,0);                            # Reserved
    dword(\$buf,0);                            # Reserved

    foreach my $list (sort {lc($a) cmp lc($b)} keys %$playlistDbRef) {
        my @files = @{$playlistDbRef->{$list}};
        my $fileCount = scalar(@files);
        my $noPadSize = ($fileCount * 2) + 8;
        my $padSize = 0;
        if ($noPadSize % 32) { $padSize = 32 - ($noPadSize % 32); }

        word(\$buf,$noPadSize+$padSize);     # Size of module in words
        if ($fileCount) {
            word(\$buf,0);                   # State: 1 if empty, otw 0
        } else {
            word(\$buf,1);
        }
        word(\$buf,$fileCount);              # Number of record entries
        word(\$buf,0);                       # Reserved
        dword(\$buf,0);                      # Reserved
        
        my $paiLocation = int(length($buf)/2);
        foreach my $file (@files) {
            dword(\$buf,$locationsRef->{'title-loc'}{$file});
        }
        $locationsRef->{'playlist-pai-loc'}{$list} = $paiLocation;

        # pad module out to multiple of 32 words
        while ($padSize) {
            dword(\$buf,0);
            $padSize -= 2;
        }
        dword(\$buf,0);                      # Marks end of module
    }

    if (writeBuf($file,\$buf,$top) == -1) { return -1; }
    return 0;
}

sub createSAI($$$$$$)
{
    my ($file,$dbRef,$locationsRef,$dbKey,$headerLength,$top) = @_;
    my @uniqueKeys;
    my $fh = new FileHandle;
    my ($buf,$count,$found) = ("",0,0);

    if (-e $file) { unlink $file; }

    if ($dbKey eq "title") {
        $count = scalar(keys(%$dbRef)) + 1;
    } else {
        $count = scalar(getUniqueNames($dbRef,$dbKey)) + 1;
    }

    dword(\$buf,0x05181971);          # Signature
    dword(\$buf,0);                   # Reserved
    word(\$buf,$count);               # Number of entries
    word(\$buf,0);                    # Reserved
    dword(\$buf,0);                   # Reserved
    dword(\$buf,0);                   # Reserved
    dword(\$buf,0);                   # Reserved
    dword(\$buf,$headerLength);       # MDB pointer for record 1
    dword(\$buf,0);                   # PAI pointer for record 1 ???

    my @keys = ();
    
    if ($dbKey eq 'date') {
        @keys = sort { lc($dbRef->{$b}{$dbKey}) cmp
            lc($dbRef->{$a}{$dbKey}) } keys %$dbRef;
    } else {
        @keys = sort { lc($dbRef->{$a}{$dbKey}) cmp
            lc($dbRef->{$b}{$dbKey}) } keys %$dbRef;
    }

    if ($dbKey ne "title") {
        foreach my $key (@keys) {
            $found = 0;
            foreach my $uniqueKey (@uniqueKeys) {
                if ($dbRef->{$key}{$dbKey} eq $uniqueKey) {
                    $found = 1;
                    last;
                }
            }
            if (!$found) {
                push @uniqueKeys, $dbRef->{$key}{$dbKey};
                dword(\$buf,$locationsRef->{"$dbKey-loc"}{
                    $dbRef->{$key}{$dbKey}});
                dword(\$buf,$locationsRef->{"$dbKey-pai-loc"}{
                    $dbRef->{$key}{$dbKey}});
            }
        }
    } else {
        foreach my $key (@keys) {
            dword(\$buf,$locationsRef->{"title-loc"}{$key});
            dword(\$buf,0);
        }
    }

    dword(\$buf,0);                   # Reserved
    dword(\$buf,0);                   # Reserved
    if (writeBuf($file,\$buf,$top) == -1) { return -1; }
    return 0;
}

sub createSAIPlaylist($$$$$)
{
    my ($file,$playlistDbRef,$locationsRef,$headerLength,$top) = @_;
    my $fh = new FileHandle;
    my ($buf,$count) = ("",0);

    if (-e $file) { unlink $file; }

    $count = scalar(keys %$playlistDbRef) + 1;

    dword(\$buf,0x05181971);          # Signature
    dword(\$buf,0);                   # Reserved
    word(\$buf,$count);               # Number of entries
    word(\$buf,0);                    # Reserved
    dword(\$buf,0);                   # Reserved
    dword(\$buf,0);                   # Reserved
    dword(\$buf,0);                   # Reserved
    dword(\$buf,$headerLength);       # MDB pointer for record 1
    dword(\$buf,0);                   # PAI pointer for record 1
   
    foreach my $list (sort {lc($a) cmp lc($b)} keys %$playlistDbRef) {
        dword(\$buf,$locationsRef->{'playlist-loc'}{$list});
        dword(\$buf,$locationsRef->{'playlist-pai-loc'}{$list});
    }

    dword(\$buf,0);                   # Reserved
    dword(\$buf,0);                   # Reserved
    if (writeBuf($file,\$buf,$top) == -1) { return -1; }
    return 0;
}

sub createMiscDbs($$)
{
    my ($home,$top) = @_;
    my $buf;

    my $file = "$home/failedhisi/failedhisi.mdb";
    if (!-e $file) {
        $buf = "";
        $buf .= pack("H*","004a00010000000100030000004a0000");
        $buf .= pack("H*","00240000000000000000000000000000");
        $buf .= pack("H*","00000018000000000000001c00000000");
        $buf .= pack("H*","00034869536900000007556e73756363");
        $buf .= pack("H*","65737366756c00000024000300200000");
        $buf .= pack("H*","00000014000000000025000000000018");
        $buf .= pack("H*","000000003fff00000000002000000000");
        $buf .= pack("H*","0003506c61790000000744656c657465");
        $buf .= pack("H*","20436c69700000000003457869740000");
        $buf .= pack("H*","574f4944800000250d0a");
        if (writeBuf($file,\$buf,$top) == -1) { return -1; }
    }

    $file = "$home/failedhisi/failedhisi.sai";
    if (!-e $file) {
        $buf = "";
        $buf .= pack("H*","05181971000000000001000000000000");
        $buf .= pack("H*","00000000000000000000004a00000000");
        $buf .= pack("H*","00000000000000000d0a");
        if (writeBuf($file,\$buf,$top) == -1) { return -1; }
    }

    $file = "$home/idedhisi/idedhisi.mdb";
    if (!-e $file) {
        $buf = "";
        $buf .= pack("H*","00530001000000010009000000530000");
        $buf .= pack("H*","00230000000000000000000000000000");
        $buf .= pack("H*","00000018000000000000001c00000000");
        $buf .= pack("H*","000348695369000000064964656e7469");
        $buf .= pack("H*","666965640000002e0004002400000000");
        $buf .= pack("H*","001a00000000002000000000001e0000");
        $buf .= pack("H*","00000025000000000022000000003fff");
        $buf .= pack("H*","00000000002a000000000003496e666f");
        $buf .= pack("H*","00000003506c61790000000744656c65");
        $buf .= pack("H*","746520436c6970000000000345786974");
        $buf .= pack("H*","0000574f4944800000250d0a");
        if (writeBuf($file,\$buf,$top) == -1) { return -1; }
    }

    $file = "$home/idedhisi/idedhisi.sai";
    if (!-e $file) {
        $buf = "";
        $buf .= pack("H*","05181971000000000001000000000000");
        $buf .= pack("H*","00000000000000000000005300000000");
        $buf .= pack("H*","00000000000000000d0a");
        if (writeBuf($file,\$buf,$top) == -1) { return -1; }
    }

    $file = "$home/unidedhisi/unidedhisi.mdb";
    if (!-e $file) {
        $buf = "";
        $buf .= pack("H*","004c00010000000100030000004c0000");
        $buf .= pack("H*","00260000000000000000000000000000");
        $buf .= pack("H*","00000018000000000000001c00000000");
        $buf .= pack("H*","00034869536900000009546f20426520");
        $buf .= pack("H*","4964656e746966696564000000240003");
        $buf .= pack("H*","00200000000000140000000000250000");
        $buf .= pack("H*","00000018000000003fff000000000020");
        $buf .= pack("H*","000000000003506c6179000000074465");
        $buf .= pack("H*","6c65746520436c697000000000034578");
        $buf .= pack("H*","69740000574f4944800000250d0a");
        if (writeBuf($file,\$buf,$top) == -1) { return -1; }
    }

    $file = "$home/unidedhisi/unidedhisi.sai";
    if (!-e $file) {
        $buf = "";
        $buf .= pack("H*","05181971000000000001000000000000");
        $buf .= pack("H*","00000000000000000000004c00000000");
        $buf .= pack("H*","00000000000000000d0a");
        if (writeBuf($file,\$buf,$top) == -1) { return -1; }
    }

    $file = "$home/pcaudio/albums.mdb";
    if (!-e $file) {
        $buf = "";
        $buf .= pack("H*","002a00000000000100010000002a0000");
        $buf .= pack("H*","00000000000000000000000000000000");
        $buf .= pack("H*","000000180000001d0000002300000000");
        $buf .= pack("H*","0004416c62756d730000617564696f2e");
        $buf .= pack("H*","6d64620000000004416c62756d730000");
        $buf .= pack("H*","574f4944800000250d0a");
        if (writeBuf($file,\$buf,$top) == -1) { return -1; }
    }

    $file = "$home/pcaudio/albums.pai";
    if (!-e $file) {
        $buf = "";
        $buf .= pack("H*","01162002000000000000000000000000");
        $buf .= pack("H*","0d0a");
        if (writeBuf($file,\$buf,$top) == -1) { return -1; }
    }

    $file = "$home/pcaudio/albums.sai";
    if (!-e $file) {
        $buf = "";
        $buf .= pack("H*","05181971000000000001000000000000");
        $buf .= pack("H*","00000000000000000000002a00000000");
        $buf .= pack("H*","00000000000000000d0a");
        if (writeBuf($file,\$buf,$top) == -1) { return -1; }
    }

    $file = "$home/pcaudio/artist.mdb";
    if (!-e $file) {
        $buf = "";
        $buf .= pack("H*","002b00000000000100010000002b0000");
        $buf .= pack("H*","00000000000000000000000000000000");
        $buf .= pack("H*","000000180000001d0000002300000000");
        $buf .= pack("H*","00044172746973740000617564696f2e");
        $buf .= pack("H*","6d646200000000054172746973747300");
        $buf .= pack("H*","0000574f4944800000250d0a");
        if (writeBuf($file,\$buf,$top) == -1) { return -1; }
    }

    $file = "$home/pcaudio/artist.pai";
    if (!-e $file) {
        $buf = "";
        $buf .= pack("H*","01162002000000000000000000000000");
        $buf .= pack("H*","0d0a");
        if (writeBuf($file,\$buf,$top) == -1) { return -1; }
    }

    $file = "$home/pcaudio/artist.sai";
    if (!-e $file) {
        $buf = "";
        $buf .= pack("H*","05181971000000000001000000000000");
        $buf .= pack("H*","00000000000000000000002b00000000");
        $buf .= pack("H*","00000000000000000d0a");
        if (writeBuf($file,\$buf,$top) == -1) { return -1; }
    }

    $file = "$home/pcaudio/genre.mdb";
    if (!-e $file) {
        $buf = "";
        $buf .= pack("H*","002a00000000000100010000002a0000");
        $buf .= pack("H*","00000000000000000000000000000000");
        $buf .= pack("H*","000000180000001d0000002300000000");
        $buf .= pack("H*","000447656e7265000000617564696f2e");
        $buf .= pack("H*","6d6462000000000447656e7265730000");
        $buf .= pack("H*","574f4944800000250d0a");
        if (writeBuf($file,\$buf,$top) == -1) { return -1; }
    }

    $file = "$home/pcaudio/genre.pai";
    if (!-e $file) {
        $buf = "";
        $buf .= pack("H*","01162002000000000000000000000000");
        $buf .= pack("H*","0d0a");
        if (writeBuf($file,\$buf,$top) == -1) { return -1; }
    }

    $file = "$home/pcaudio/genre.sai";
    if (!-e $file) {
        $buf = "";
        $buf .= pack("H*","05181971000000000001000000000000");
        $buf .= pack("H*","00000000000000000000002a00000000");
        $buf .= pack("H*","00000000000000000d0a");
        if (writeBuf($file,\$buf,$top) == -1) { return -1; }
    }

    $file = "$home/pcaudio/pcaudio.mdb";
    if (!-e $file) {
        $buf = "";
        $buf .= pack("H*","00b50001000000060009000000b50000");
        $buf .= pack("H*","00770000000000000000000000000000");
        $buf .= pack("H*","0000002c000000000000003300000000");
        $buf .= pack("H*","000000380000003f000000460000004c");
        $buf .= pack("H*","00000052000000570000005d00000062");
        $buf .= pack("H*","000000680000006f00065043204c6962");
        $buf .= pack("H*","7261727900000004536f6e6773000000");
        $buf .= pack("H*","0006506c61796c69737473000000706c");
        $buf .= pack("H*","61796c6973742e6d6462000000054172");
        $buf .= pack("H*","74697374730000006172746973742e6d");
        $buf .= pack("H*","646200000004416c62756d730000616c");
        $buf .= pack("H*","62756d732e6d64620000000447656e72");
        $buf .= pack("H*","6573000067656e72652e6d6462000000");
        $buf .= pack("H*","00065265636f7264696e677300007265");
        $buf .= pack("H*","636f7264696e67732e6d64620000003c");
        $buf .= pack("H*","00050024000000000020000000008000");
        $buf .= pack("H*","00000000002400000000002300000000");
        $buf .= pack("H*","002a0000000000210000000000330000");
        $buf .= pack("H*","00003fff000000000038000000000003");
        $buf .= pack("H*","496e666f000000054765742046696c65");
        $buf .= pack("H*","0000000841646420546f204d79204d69");
        $buf .= pack("H*","78000000000444656c65746500000003");
        $buf .= pack("H*","457869740000574f4944800000250d0a");
        if (writeBuf($file,\$buf,$top) == -1) { return -1; }
    }

    $file = "$home/pcaudio/pcaudio.sai";
    if (!-e $file) {
        $buf = "";
        $buf .= pack("H*","05181971000000000001000000000000");
        $buf .= pack("H*","0000000000000000000000b500000000");
        $buf .= pack("H*","00000000000000000d0a");
        if (writeBuf($file,\$buf,$top) == -1) { return -1; }
    }

    $file = "$home/pcaudio/playlist.mdb";
    if (!-e $file) {
        $buf = "";
        $buf .= pack("H*","002e00000000000100010000002e0000");
        $buf .= pack("H*","00000000000000000000000000000000");
        $buf .= pack("H*","000000180000001f0000002500000000");
        $buf .= pack("H*","0006506c61796c697374730000006175");
        $buf .= pack("H*","64696f2e6d64620000000006506c6179");
        $buf .= pack("H*","6c69737473000000574f494480000025");
        $buf .= pack("H*","8000214d79204d697800000000250d0a");
        if (writeBuf($file,\$buf,$top) == -1) { return -1; }
    }

    $file = "$home/pcaudio/playlist.pai";
    if (!-e $file) {
        $buf = "";
        $buf .= pack("H*","01162002000000000000000000000000");
        $buf .= pack("H*","00200000000000000000000000000000");
        $buf .= pack("H*","00000000000000000000000000000000");
        $buf .= pack("H*","00000000000000000000000000000000");
        $buf .= pack("H*","00000000000000000000000000000000");
        $buf .= pack("H*","0d0a");
        if (writeBuf($file,\$buf,$top) == -1) { return -1; }
    }

    $file = "$home/pcaudio/playlist.sai";
    if (!-e $file) {
        $buf = "";
        $buf .= pack("H*","05181971000000000002000000000000");
        $buf .= pack("H*","00000000000000000000002e00000000");
        $buf .= pack("H*","000000300000000e0000000000000000");
        $buf .= pack("H*","0d0a");
        if (writeBuf($file,\$buf,$top) == -1) { return -1; }
    }
    
    $file = "$home/pcaudio/recordings.mdb";
    if (!-e $file) {
        $buf = "";
        $buf .= pack("H*","002e00000000000100010000002e0000");
        $buf .= pack("H*","00000000000000000000000000000000");
        $buf .= pack("H*","000000180000001f0000002500000000");
        $buf .= pack("H*","00065265636f7264696e677300006175");
        $buf .= pack("H*","64696f2e6d646200000000065265636f");
        $buf .= pack("H*","7264696e67730000574f494480000025");
        $buf .= pack("H*","0d0a");
        if (writeBuf($file,\$buf,$top) == -1) { return -1; }
    }
    
    $file = "$home/pcaudio/recordings.pai";
    if (!-e $file) {
        $buf = "";
        $buf .= pack("H*","01162002000000000000000000000000");
        $buf .= pack("H*","0d0a");
        if (writeBuf($file,\$buf,$top) == -1) { return -1; }
    }

    $file = "$home/pcaudio/recordings.sai";
    if (!-e $file) {
        $buf = "";
        $buf .= pack("H*","05181971000000000001000000000000");
        $buf .= pack("H*","00000000000000000000002e00000000");
        $buf .= pack("H*","00000000000000000d0a");
        if (writeBuf($file,\$buf,$top) == -1) { return -1; }
    }
    return 0;
}

sub string($$)
{
    my ($bufRef,$string) = @_;

    $$bufRef .= pack("A*",$string);
    while (length($$bufRef) % 2) {
        byte($bufRef,0);
    }
    word($bufRef,0);
}

sub display($$)
{
    my ($bufRef,$string) = @_;
    my $length = length($string);
    my $offset = length($$bufRef);

    word($bufRef,0);
    string($bufRef,$string);
    my $offset2 = length($$bufRef);
    wordOverwrite($bufRef,$offset,($offset2 - $offset) / 2 - 1);
}

sub byte($$)
{
    my ($bufRef,$value) = @_;
    $$bufRef .= pack("C",$value);
}

sub word($$)
{
    my ($bufRef,$value) = @_;
    $$bufRef .= pack("n",$value);
}

sub dword($$)
{
    my ($bufRef,$value) = @_;
    $$bufRef .= pack("N",$value);
}

sub wordOverwrite($$$)
{
    my ($bufRef,$offset,$value) = @_;
    substr($$bufRef,$offset,2) = pack("n",$value);
}

sub dwordOverwrite($$$)
{
    my ($bufRef,$offset,$value) = @_;
    substr($$bufRef,$offset,4) = pack("N",$value);
}

sub writeBuf($$$)
{
    my ($file,$bufRef,$top) = @_;
    my $fh = new FileHandle;
    my $dir = dirname($file);

    if (defined $top) { $top->update; }

    eval { mkpath $dir, 0, 0700; };
    if ($@) {
        message('ERR',"Could not create directory $dir: $!\n");
        return -1;
    }

    if ($fh->open(">$file")) {
        binmode $fh;
        my $length = length $$bufRef;
        my $chunksize = 1024 * 64;
        my $offset = 0;

        while ($length) {
            my $wlen = syswrite($fh, $$bufRef, $chunksize, $offset);
            if (!defined $wlen) {
                message('ERR', "Error writing $file!\n");
                $fh->close;
                unlink $file;
                return -1;
            }
            $offset += $wlen;
            $length -= $wlen;
        }
        $fh->close;
        return 0;
    } else {
        message('ERR',"Could not open $file: $!\n");
        return -1;
    }
}

1
