/**
 *
 * This file is part of Tulip (www.tulip-software.org)
 *
 * Authors: David Auber and the Tulip development Team
 * from LaBRI, University of Bordeaux 1 and Inria Bordeaux - Sud Ouest
 *
 * Tulip is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 *
 * Tulip is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 */

#ifndef SOMMAP_H_
#define SOMMAP_H_

#include <tulip/GraphDecorator.h>
#include <map>
#include "DynamicVector.h"

/**
 * @brief Self Organizing Map representation.
 * Self Organizing Map representation. Use the Tulip graph library to represent a SOM. Store the weight of each node in a map and can transmit them to Graph Property.
 * Grid is generated by the Grid import algorithm from Tulip. Give to user function to handle the graph like a traditional grid (with x and y position).
 */
namespace tlp {

class SOMMap: public tlp::GraphDecorator {
public:
  /**
   * The node connectivity.
   */
  enum SOMMapConnectivity {
    four = 4, six = 6, eight = 8
  };

  /**
   * Build a SOM from an already existing graph.
   * @param root The graph to store SOM grid into.
   * @param width The width of the grid.
   * @param height The height of the grid.
   * @param connectivity The connectivity of each node of the grid (the neighborhood)
   * @param oppositeConnected Connect opposite node of the map to build a torus.
   * @return
   */
  SOMMap(Graph* root, unsigned int width, unsigned int height,
         SOMMapConnectivity connectivity = four, bool oppositeConnected = false);
  /**
   * Build a SOM in a new Graph.
   * @param width The width of the grid.
   * @param height The height of the grid.
   * @param connectivity The connectivity of each node of the grid (the neighborhood)
   * @param oppositeConnected Connect opposite node of the map to build a torus.
   * @return
   */
  SOMMap(unsigned int width, unsigned int height,
         SOMMapConnectivity connectivity = four, bool oppositeConnected = false);
  virtual ~SOMMap();

  /**
   * Get the weight linked to the node n.
   * @param n The node.
   * @return The reference on the weight linked to the node.
   */
  DynamicVector<double>& getWeight(tlp::node n);
  /**
   * Get the weight linked to the node n.
   * @param n The node.
   * @return The constant reference on the weight linked to the node.
   */
  const DynamicVector<double> getWeight(const tlp::node& n) const;

  /**
   * Set the weight for the node n.
   * @param n The node.
   * @param weight The weight.
   */
  void setWeight(tlp::node n, const DynamicVector<double>& weight);

  /**
   * Copy each node weight value in the given graph properties. If properties doesn't exist create them.
   * @param propertiesToListen The vector with the name of the properties to store weight into.
   */
  void registerModification(const std::vector<std::string>&propertiesToListen);

  tlp::node getNodeAt(unsigned int pos);
  /**
   * Get the node at the x and y coordinates.
   * @param x
   * @param y
   * @return The node or an invalid node if coordinates are invalid.
   */
  tlp::node getNodeAt(unsigned int x, unsigned int y);
  /**
   * Return the position for the node n. If the node is invalid or isn't an element of the graph return false.
   * @param
   * @param x
   * @param y
   * @return
   */
  bool getPosForNode(tlp::node, unsigned int &x, unsigned int &y);

  inline unsigned int getWidth() const {
    return width;
  }
  inline unsigned int getHeight() const {
    return height;
  }

  inline SOMMapConnectivity getConnectivity() {
    return connectivity;
  }
  inline bool getOppositeConnected() {
    return oppositeConnected;
  }
protected:

  void initMap();

  unsigned int width;
  unsigned int height;

  std::map<tlp::node, DynamicVector<double> > nodeToNodeVec;

  SOMMapConnectivity connectivity;
  bool oppositeConnected;

  bool graphCreated;
};
}
#endif /* SOMMAP_H_ */
