# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
#
# Copyright 2014 Canonical Ltd.
# Author: Omer Akram <omer.akram@canonical.com>
#         Brendan Donegan <brendan.donegan@canonical.com>
#
# This file is part of ubuntu-experience-tests.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>

"""Helper functions to make writing Inter-app integration tests easier."""

import dbus
import re
import subprocess
import time

from autopilot.introspection import get_proxy_object_for_existing_process
from ubuntuuitoolkit import emulators as toolkit_emulators
from unity8.shell.emulators import UnityEmulatorBase

DBUS_IFACE_ADD_BOOK = 'com.canonical.pim.AddressBook'
DBUS_IFACE_ADD_BOOKVIEW = 'com.canonical.pim.AddressBookView'


class ContactsDbusService(object):
    def __init__(self):
        self.bus = None
        self.addr = None
        self.addr_iface = None
        self.bus = dbus.SessionBus()
        self.addr = self.bus.get_object(
            'com.canonical.pim', '/com/canonical/pim/AddressBook')
        self.addr_iface = dbus.Interface(
            self.addr, dbus_interface=DBUS_IFACE_ADD_BOOK)

    def query_contacts(self, fields='', query='', sources=[]):
        view_path = self.addr_iface.query(fields, query, [])
        view = self.bus.get_object(
            'com.canonical.pim', view_path)
        view_iface = dbus.Interface(
            view, dbus_interface=DBUS_IFACE_ADD_BOOKVIEW)
        contacts = view_iface.contactsDetails([], 0, -1)
        view.close()
        return contacts

    def update_contact(self, vcard):
        return self.addr_iface.updateContacts([vcard])

    def create_contact(self, vcard):
        return self.addr_iface.createContact(vcard, "")

    def delete_contact(self, ids):
        return self.addr_iface.removeContacts(ids)

    def get_vcard_uid(self, vcard):
        """
        Return the UID of the provided vcard
        :param vcard: vcard formatted data
        :return: uid value from vcard, else None
        """
        uid_pattern = re.compile(r'''
        ^          # start of string
        .*         # any characters
        UID:(.*?)  # UID value, saved to group, non-greedy
        \r\n       # new line
        .*         # anything else
        $          # end of string
        ''', re.VERBOSE | re.IGNORECASE | re.DOTALL)
        uid = None
        match = uid_pattern.match(vcard)
        if match:
            uid = match.group(1)
        return uid

    def delete_all_contacts(self):
        """
        Delete all contacts using the dbus service
        """
        uids = []
        contacts = self.query_contacts()
        for contact in contacts:
            uid = self.get_vcard_uid(contact)
            if uid:
                uids.append(uid)
        self.delete_contact(uids)


def get_proxy_object_by_process_name(proc_name):
    """Helper function to return proxy object of a process

    :param proc_name: Name of the running application process with testability
        driver loaded.
    :returns: Proxy object for the given process using emulator base from
        ubuntuuittoolkit and in case proc_name is unity8 use UnityEmulatorBase.
    :raises: RuntimeError if process is not found to be running after
        10 seconds.

    """
    return get_proxy_object_for_existing_process(
        _get_pid_by_name(proc_name),
        emulator_base=_get_emulator_base(proc_name)
    )


def _get_pid_by_name(proc_name):
        for i in range(10):
            try:
                return int(
                    subprocess.check_output(['pidof', proc_name]).strip())
            except subprocess.CalledProcessError:
                # application not started yet, check in a second
                time.sleep(1)
        else:
            raise RuntimeError(
                'Could not find autopilot interface for {} after 10'
                'seconds'.format(proc_name)
            )


def _get_emulator_base(proc_name):
    if proc_name == 'unity8':
        return UnityEmulatorBase
    else:
        return toolkit_emulators.UbuntuUIToolkitEmulatorBase
