/*----------------------------------------------------------------------
  File    : tract.h
  Contents: item and transaction management
  Author  : Christian Borgelt
  History : 2001.11.18 file created from file apriori.c
            2001.12.28 first version completed
            2001.01.02 ta_sort mapped to v_intsort
            2002.02.19 transaction tree functions added
            2003.07.17 functions is_filter, ta_filter, tas_filter added
            2003.08.21 parameter 'heap' added to tas_sort, tat_create
            2003.09.12 function tas_total added
            2003.09.20 empty transactions in input made possible
            2004.11.20 function tat_mark added
            2004.12.11 access functions for extended frequency added
            2004.12.15 function nim_trunc added
            2006.11.26 structures ISFMTR and ISEVAL added
            2007.02.13 adapted to modified tabscan module
----------------------------------------------------------------------*/
#ifndef __TRACT__
#define __TRACT__
#ifndef NIMAPFN
#define NIMAPFN
#endif
#include "vecops.h"
#include "symtab.h"
#include "tabscan.h"
#include "apriori_entries.h"

#ifdef __cplusplus
extern "C" {
#endif

/*----------------------------------------------------------------------
  Preprocessor Definitions
----------------------------------------------------------------------*/
/* --- item appearance flags --- */
#define APP_NONE    0x00        /* item should be ignored */
#define APP_BODY    0x01        /* item may appear in rule body */
#define APP_HEAD    0x02        /* item may appear in rule head */
#define APP_BOTH    (APP_HEAD|APP_BODY)

/* --- error codes --- */
#define E_NONE        0         /* no error */
#define E_NOMEM     (-1)        /* not enough memory */
#define E_FOPEN     (-2)        /* cannot open file */
#define E_FREAD     (-3)        /* read error on file */
#define E_FWRITE    (-4)        /* write error on file */

#define E_ITEMEXP  (-16)        /* item expected */
#define E_DUPITEM  (-17)        /* duplicate item */
#define E_APPEXP   (-18)        /* appearance indicator expected */
#define E_UNKAPP   (-19)        /* unknown appearance indicator */
#define E_FLDCNT   (-20)        /* too many fields */

/*----------------------------------------------------------------------
  Type Definitions
----------------------------------------------------------------------*/
typedef struct {                /* --- an item --- */
  INT64     id;                   /* item identifier */
  INT64     frq;                  /* frequency in transactions */
  INT64     xfq;                  /* extended frequency (t.a. sizes) */
  INT64     app;                  /* appearance indicator */
} ITEM;                         /* (item) */

typedef struct {                /* --- an itemset --- */
  TABSCAN *tscan;               /* table scanner */
  char    chars[4];             /* special characters */
  NIMAP   *nimap;               /* name/identifier map */
  INT64     app;                  /* default appearance indicator */
  INT64     vsz;                  /* size of transaction buffer */
  INT64     cnt;                  /* number of items in transaction */
  INT64     *items;               /* items in transaction */
} ITEMSET;                      /* (item set) */

typedef struct {                /* --- a transaction --- */
  INT64     cnt;                  /* number of items */
  INT64     items[1];             /* item identifier vector */
} TRACT;                        /* (transaction) */

typedef struct {                /* --- a transaction set --- */
  ITEMSET *itemset;             /* underlying item set */
  INT64     max;                  /* maximum number of items per t.a. */
  INT64     vsz;                  /* size of transaction vector */
  INT64     cnt;                  /* number of transactions */
  INT64     total;                /* total number of items */
  TRACT   **tracts;             /* transaction vector */
} TASET;                        /* (transaction set) */

typedef struct _tatree {        /* --- a transaction tree (node) --- */
  INT64     cnt;                  /* number of transactions */
  INT64     max;                  /* size of largest transaction */
  INT64     size;                 /* node size (number of children) */
  INT64     items[1];             /* next items in rep. transactions */
} TATREE;                       /* (transaction tree) */

typedef struct {                /* --- an item set evaluator --- */
  double  logta;                /* logarithm of num. of transactions */
  double  *logfs;               /* logarithms of item frequencies */
  double  lsums[1];             /* sums of logarithms for prefixes */
} ISEVAL;                       /* (item set evaluator) */

typedef struct {                /* --- item set formatter --- */
  INT64        cnt;               /* number of formatted item names */
  INT64        len;               /* length of description in buffer */
  INT64        *offs;             /* prefix lengths in output buffer */
  char       *buf;              /* output buffer */
  const char *names[1];         /* formatted item names */
} ISFMTR;                       /* (item set formatter) */

/*----------------------------------------------------------------------
  Item Set Functions
----------------------------------------------------------------------*/
extern ITEMSET*    is_create  (void);
extern void        is_delete  (ITEMSET *iset);
extern TABSCAN*    is_tabscan (ITEMSET *iset);
extern void        is_chars   (ITEMSET *iset, const char *blanks,
                                              const char *fldseps,
                                              const char *recseps,
                                              const char *cominds);

extern INT64         is_cnt     (ITEMSET *iset);
extern INT64         is_item    (ITEMSET *iset, const char *name);
extern const char* is_name    (ITEMSET *iset, INT64 item);
extern INT64         is_getfrq  (ITEMSET *iset, INT64 item);
extern INT64         is_setfrq  (ITEMSET *iset, INT64 item, INT64 frq);
extern INT64         is_addfrq  (ITEMSET *iset, INT64 item, INT64 frq);
extern INT64         is_getxfq  (ITEMSET *iset, INT64 item);
extern INT64         is_setxfq  (ITEMSET *iset, INT64 item, INT64 frq);
extern INT64         is_getapp  (ITEMSET *iset, INT64 item);
extern INT64         is_setapp  (ITEMSET *iset, INT64 item, INT64 app);

extern INT64         is_readapp (ITEMSET *iset, FILE *file);
extern INT64         is_read    (ITEMSET *iset, FILE *file);

extern INT64         is_recode  (ITEMSET *iset, INT64 minfrq, INT64 dir,
                               INT64 *map);
extern void        is_trunc   (ITEMSET *iset, INT64 cnt);
extern INT64         is_filter  (ITEMSET *iset, const char *marks);
extern INT64         is_tsize   (ITEMSET *iset);
extern INT64*        is_tract   (ITEMSET *iset);

/*----------------------------------------------------------------------
  Transaction Functions
----------------------------------------------------------------------*/
extern void        ta_sort    (INT64 *items, INT64 n);
extern INT64         ta_unique  (INT64 *items, INT64 n);
extern INT64         ta_filter  (INT64 *items, INT64 n, const char *marks);

/*----------------------------------------------------------------------
  Transaction Set Functions
----------------------------------------------------------------------*/
extern TASET*      tas_create  (ITEMSET *itemset);
extern void        tas_delete  (TASET *taset, INT64 delis);
extern ITEMSET*    tas_itemset (TASET *taset);

extern INT64         tas_cnt     (TASET *taset);
extern INT64         tas_add     (TASET *taset, const INT64 *items, INT64 n);
extern INT64*        tas_tract   (TASET *taset, INT64 index);
extern INT64         tas_tsize   (TASET *taset, INT64 index);
extern INT64         tas_total   (TASET *taset);

extern void        tas_recode  (TASET *taset, INT64 *map, INT64 cnt);
extern INT64         tas_filter  (TASET *taset, const char *marks);
extern void        tas_shuffle (TASET *taset, double randfn(void));
extern void        tas_sort    (TASET *taset, INT64 heap);
extern INT64         tas_occur   (TASET *taset, const INT64 *items, INT64 n);

#ifndef NDEBUG
extern void        tas_show    (TASET *taset);
#endif

/*----------------------------------------------------------------------
  Transaction Tree Functions
----------------------------------------------------------------------*/
extern TATREE*     tat_create  (TASET *taset, INT64 heap);
extern void        tat_delete  (TATREE *tat);
extern INT64         tat_cnt     (TATREE *tat);
extern INT64         tat_max     (TATREE *tat);
extern INT64         tat_size    (TATREE *tat);
extern INT64*        tat_items   (TATREE *tat);
extern INT64         tat_item    (TATREE *tat, INT64 index);
extern TATREE*     tat_child   (TATREE *tat, INT64 index);
extern void        tat_mark    (TATREE *tat);

#ifndef NDEBUG
extern void        tat_show    (TATREE *tat);
#endif

/*----------------------------------------------------------------------
  Item Set Evaluation Functions
----------------------------------------------------------------------*/
extern ISEVAL*     ise_create (ITEMSET *iset, INT64 tacnt);
extern void        ise_delete (ISEVAL *eval);
extern double      ise_eval   (ISEVAL *eval, INT64 *ids, INT64 cnt, INT64 pre,
                               INT64 supp);

/*----------------------------------------------------------------------
  Item Set Formatting Functions
----------------------------------------------------------------------*/
extern ISFMTR*     isf_create (ITEMSET *iset, INT64 scan);
extern void        isf_delete (ISFMTR *fmt);
extern const char* isf_format (ISFMTR *fmt, INT64 *ids, INT64 cnt, INT64 pre);
extern INT64         isf_length (ISFMTR *fmt);
extern void        isf_print  (ISFMTR *fmt, FILE *out);

/*----------------------------------------------------------------------
  Preprocessor Definitions
----------------------------------------------------------------------*/
#define is_tabscan(s)     ((s)->tscan)

#define is_cnt(s)         nim_cnt((s)->nimap)
#define is_name(s,i)      nim_name(nim_byid((s)->nimap, i))
#define is_getfrq(s,i)    (((ITEM*)nim_byid((s)->nimap, i))->frq)
#define is_setfrq(s,i,f)  (((ITEM*)nim_byid((s)->nimap, i))->frq  = (f))
#define is_addfrq(s,i,f)  (((ITEM*)nim_byid((s)->nimap, i))->frq += (f))
#define is_getxfq(s,i)    (((ITEM*)nim_byid((s)->nimap, i))->xfq)
#define is_setxfq(s,i,f)  (((ITEM*)nim_byid((s)->nimap, i))->xfq  = (f))
#define is_getapp(s,i)    (((ITEM*)nim_byid((s)->nimap, i))->app)
#define is_setapp(s,i,a)  (((ITEM*)nim_byid((s)->nimap, i))->app  = (a))

#define is_trunc(s,n)     nim_trunc((s)->nimap, n)

#define is_tsize(s)       ((s)->cnt)
#define is_tract(s)       ((s)->items)

/*--------------------------------------------------------------------*/
#define ta_sort(v,n)      v_intsort(v,n)

/*--------------------------------------------------------------------*/
#define tas_itemset(s)    ((s)->itemset)
#define tas_cnt(s)        ((s)->cnt)
#define tas_max(s)        ((s)->max)

#define tas_tract(s,i)    ((s)->tracts[i]->items)
#define tas_tsize(s,i)    ((s)->tracts[i]->cnt)
#define tas_total(s)      ((s)->total)

#define tas_shuffle(s,f)  v_shuffle((s)->tracts, (s)->cnt, f)

/*--------------------------------------------------------------------*/
#define tat_cnt(t)        ((t)->cnt)
#define tat_max(t)        ((t)->max)
#define tat_size(t)       ((t)->size)
#define tat_item(t,i)     ((t)->items[i])
#define tat_items(t)      ((t)->items)
#ifndef ARCH64
#define tat_child(t,i)    (((TATREE**)((t)->items +(t)->size))[i])
#endif

/*--------------------------------------------------------------------*/
#define ise_delete(e)     free(e)

/*--------------------------------------------------------------------*/
#define isf_length(f)     ((f)->len)
#define isf_print(f,o)    fwrite((f)->buf, sizeof(char), (f)->len, o)

#ifdef __cplusplus
}
#endif


#endif
