/*
 imobj.c implementation of interfaces to basic objects used by predicting functions.
*/

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <memory.h>

#include <stdarg.h>
#include <ctype.h>

#include "IMObjects.h"

#define IMINER_KEYWORD_VERSION "$imcsrcVersion"
#define IMINER_KEYWORD_CLASS "$class"
#define IMINER_KEYWORD_MODE "$mode"
#define IMINER_KEYWORD_LENGTH "$length"
#define IMINER_KEYWORD_NROWS "$nRows"
#define IMINER_KEYWORD_NCOLUMNS "$nColumns"
#define IMINER_KEYWORD_DATA "$data"

#define LINE_LEN 255


/********************************* Object *********************************************/

/* return 1 if empty */
long IMCSRC_STDCALL IMinerObject_isEmpty(const IMinerObject* pVector)
{
	if(pVector == NULL || pVector->m_nLen < 1L)
		return 1L;
	return 0L;
}

long IMCSRC_STDCALL IMinerObject_isValid(const IMinerObject* pObject /* in: the object */)
{
	long nStatus=1L;
	long nMode;

	/* a null or length < 0 is invalid */
	if(pObject == NULL || pObject->m_nLen < 0L
	/* zero length is valid but a non-zero length must have non-null data pointer IMINER_VOID_PTR(pObject)*/
	|| (pObject->m_nLen > 0L && IMINER_VOID_PTR(pObject) == NULL))
		return 0L;

	nMode = IMINER_OBJECT_MODE(pObject);

	if (IMINER_IS_VECTOR_MODE(nMode))
		nStatus = IMinerVector_isValid(pObject);
	else /* the list family? */
	{
		switch(nMode)
		{
		case IMINER_MODE_FACTOR :
			nStatus = IMinerFactor_isValid(pObject);
			break;
		case IMINER_MODE_DOUBLEMATRIX :
			nStatus = IMinerDoubleMatrix_isValid(pObject);
			break;
		case IMINER_MODE_MODELMATRIX :
			nStatus = IMinerModelMatrix_isValid(pObject);
			break;
		case IMINER_MODE_DATASET :
			nStatus = IMinerDataSet_isValid(pObject);
			break;
		case IMINER_MODE_LIST :
		default:
			nStatus = IMinerList_isValid(pObject);
			break;
		}
	}
	return nStatus;
}

/* alloc memory for a generic object */
long IMCSRC_STDCALL IMinerObject_alloc(IMinerObject* pObject,  /* out: data object */
 long nLen, /* in: top-level length */
 long nMode /* in: mode of the data */
)
{
	long nStatus=IMINER_SUCCESS;
	long i;
	IMinerObject* pElement;

	/* pObject must be valid. We do not alloc mem for header of the input argument pOpbject */
	if(pObject==NULL)
		return IMINER_BAD_INPUT_ARGS;

	pObject->m_uData.m_pVoid = NULL;
	if(nLen <= 0L)
	{
		memset(pObject, 0, sizeof(IMinerObject));
		pObject->m_nLen = 0;
		pObject->m_nMode = nMode;
	}

	/* alloc memory according to mode */
	if(IMINER_IS_VECTOR_MODE(nMode))
	{
		if(nLen > 0L)
		{
			switch(nMode)
			{
				case IMINER_MODE_BOOLEAN:
				case IMINER_MODE_LONG:
					pObject->m_uData.m_pLong = (long*) calloc(nLen, sizeof(long));
					break;
				case IMINER_MODE_FLOAT:
					pObject->m_uData.m_pFloat = (float*) calloc(nLen, sizeof(float));
					break;
				case IMINER_MODE_DOUBLE:
					pObject->m_uData.m_pDouble = (double*) calloc(nLen, sizeof(double));
					break;
				case IMINER_MODE_STRING:
				{
					char** ppChar = (char**) calloc(nLen, sizeof(char*));
					pObject->m_uData.m_ppString = ppChar;
					/* alloc strings with fixed length and assign to each char* */
					ppChar[0] = (char*) calloc(IMINER_MAX_STRING_LEN_PLUS_1*nLen, sizeof(char));
					for(i=1L; i< nLen; ++i)
						ppChar[i] = ppChar[i-1L] + IMINER_MAX_STRING_LEN_PLUS_1;
					break;
				}
				default:
					nStatus = IMINER_BAD_INPUT_ARGS;
					break;
			}
		}
		strcpy(pObject->m_szClass, IMINER_CLASS_VECTOR);
	}
	else /* the list family? */
	{
		if(nLen > 0L)
		{
			/* alloc the header : nLen+1L to leave room for element names*/
			IMINER_LIST_PTR(pObject) = (IMinerObject**) malloc(nLen*sizeof(IMinerObject*));
			if(!IMINER_LIST_PTR(pObject))
				return IMINER_OUT_OF_MEMORY;

			/* alloc an array of header pointers : nLen+1L to leave room for element names*/
			pElement = (IMinerObject*) calloc(nLen, sizeof(IMinerObject));
			if(!pElement)
				return IMINER_OUT_OF_MEMORY;

			for(i=0L; i<nLen; ++i)
				IMINER_LIST_PTR(pObject)[i] = &pElement[i];
		}
		/* set class name */
		switch(nMode)
		{
		case IMINER_MODE_FACTOR :
			strcpy(pObject->m_szClass, IMINER_CLASS_FACTOR);
			break;
		case IMINER_MODE_DOUBLEMATRIX :
			strcpy(pObject->m_szClass, IMINER_CLASS_MATRIX);
			break;
		case IMINER_MODE_MODELMATRIX :
			strcpy(pObject->m_szClass, IMINER_CLASS_MODELMATRIX);
			break;
		case IMINER_MODE_DATASET :
				strcpy(pObject->m_szClass, IMINER_CLASS_DATASET);
			break;
		case IMINER_MODE_LIST :
		default:
			strcpy(pObject->m_szClass, IMINER_CLASS_LIST);
			break;
		}
	}

	/* set length */
	pObject->m_nLen = nLen;

	/* set the class ID or mode*/
	pObject->m_nMode  = nMode;

	return nStatus;
}

/* create a new object and copy content from the source object */
long IMCSRC_STDCALL IMinerObject_clone(IMinerObject* pObjDest,   /* in/out: data object to copy to */
 const IMinerObject* pObjSource  /* in: data object to copy from */
)
{
	long nStatus=IMINER_SUCCESS;
	long nLen, nMode;
	IMinerObject* pElement = NULL;
	IMinerObject* pNewElement=NULL;

	if(!IMinerObject_isValid(pObjSource) || pObjSource->m_nLen < 0L)
		return IMINER_BAD_INPUT_ARGS;
	if(pObjSource->m_nLen == 0L)
	{
		*pObjDest = *pObjSource;
		strcpy(pObjDest->m_szClass, pObjDest->m_szClass);
		return nStatus;
	}

	nLen  = IMINER_OBJECT_LENGTH(pObjSource);
	nMode = IMINER_OBJECT_MODE(pObjSource);
	if(IMINER_IS_VECTOR_MODE(nMode))
		nStatus = IMinerVector_clone(pObjDest, pObjSource);
	else /* the list family? */
		nStatus = IMinerList_clone(pObjDest, pObjSource);
	return nStatus;
}

/* destroy and free memory */
long IMCSRC_STDCALL IMinerObject_destroy(IMinerObject* pObject  /* in/out: data object */)
{
	long nStatus = IMINER_SUCCESS;
	long nMode;

	nMode = IMINER_OBJECT_MODE(pObject);
	if(IMINER_IS_VECTOR_MODE(nMode))
		nStatus = IMinerVector_destroy(pObject);
	else /* the list family? */
	{
		IMINER_OBJECT_MODE(pObject) = IMINER_MODE_LIST;
		nStatus = IMinerList_destroy(pObject); /* recursive */
	}
	return nStatus;
}

/* get mode of the vector data */
long IMCSRC_STDCALL IMinerObject_getMode(long* pnMode, /*out: mode of the data */
 const IMinerObject* pObject  /* in: data object */
)
{
	*pnMode = IMINER_MODE_UNKNOWN;
	if(pObject->m_nLen < 1L || pObject->m_uData.m_pVoid == NULL)
		return IMINER_BAD_INPUT_ARGS;

	*pnMode = pObject->m_nMode;
	return IMINER_SUCCESS;
}

/* write to an opened file */
long IMCSRC_STDCALL IMinerObject_writeToStream(FILE* pFile, /* in: output file pointer */
 const IMinerObject* pObject /* in: data object */
)
{
	long nStatus=IMINER_SUCCESS;
	long nMode;

	if(pObject == NULL || !IMinerObject_isValid(pObject) )
		return IMINER_BAD_INPUT_ARGS;

	nMode = IMINER_OBJECT_MODE(pObject);
	if(IMINER_IS_VECTOR_MODE(nMode))
		nStatus = IMinerVector_writeToStream(pFile, pObject);
	else /* the list family? */
		nStatus = IMinerList_writeToStream(pFile, pObject);

	return nStatus;
}

/* read from a stream */
long IMCSRC_STDCALL IMinerObject_createFromStream(IMinerObject* pObject, /* out: data object */
 FILE* pFile /* in: input file pointer */
)
{
	long nStatus=IMINER_SUCCESS;
	long nMode;

	/* read header */
	nStatus = IMinerObjectHeader_readFromStream(&nMode, pFile);
	if(nStatus!=IMINER_SUCCESS)
		return nStatus;
	/* set the class ID or mode */
	pObject->m_nMode  = nMode;

	if(IMINER_IS_VECTOR_MODE(nMode))
		nStatus = IMinerVector_createFromStream(pObject, pFile);
	else /* the list family? */
		nStatus = IMinerList_createFromStream(pObject, pFile);

	return nStatus;
}

/* open a file, write content and close the file */
long IMCSRC_STDCALL IMinerObject_writeToFile(const char* pszFile, /* output file name */
 const IMinerObject* pObject /* in: data object */
)
{
	long nStatus;
	FILE* pFile;

	/* open and write version number to the file */
	nStatus = IMinerDataFile_openForWrite(&pFile, pszFile);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;

	nStatus = IMinerObject_writeToStream(pFile, pObject);
	fclose(pFile);
	return nStatus;
}

/* open, read from a file, and close it */
long IMCSRC_STDCALL IMinerObject_createFromFile(IMinerObject* pObject, /* out: data object */
 const char* pszFile                      /* in: file name */
)
{
	long nStatus=IMINER_SUCCESS;
	FILE* pFile;

	/* open and read version number from the file */
	nStatus = IMinerDataFile_openForRead(&pFile, pszFile);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;

	nStatus = IMinerObject_createFromStream(pObject, pFile);
	fclose(pFile);
	return nStatus;
}

/********************************* Vector *********************************************/

/* return 1L if valid */
long IMCSRC_STDCALL IMinerVector_isValid(const IMinerObject* pVector)
{
	if(pVector == NULL
	|| pVector->m_nLen < 0L                                       /* nLen == 0L is a valid empty Atomic*/
	|| (pVector->m_nLen > 0L && IMINER_VOID_PTR(pVector) == NULL) /* len > 0 must have valid Data pointer */
	|| !IMINER_IS_VECTOR_MODE(IMINER_OBJECT_MODE(pVector)) )
		return 0L;
	else
		return 1L;
}

/* alloc memory for type IMinerVector. */
long IMCSRC_STDCALL IMinerVector_create(IMinerObject* pVector,  /* out: data object */
 long nLen,          /* in: length */
 long nMode,         /* in: mode of the data */
 const void* pvValue /* in: NULL or pointer to content of data to be copied */
)
{
	long nStatus=IMINER_SUCCESS;
	long i;

	if(pVector==NULL || nLen < 0L )
		return IMINER_BAD_INPUT_ARGS;

	/* set length */
	pVector->m_nLen = nLen;

	if(nLen > 0L)
	{
		nStatus = IMinerObject_alloc(pVector, nLen, nMode);
		/* have faith that the caller passes in the approriate type */
		if(nStatus && pvValue != NULL)
		{
			switch(nMode)
			{
				case IMINER_MODE_BOOLEAN:
				case IMINER_MODE_LONG:
					for(i=0L; i<nLen; ++i)
						pVector->m_uData.m_pLong[i] = ((long*)pvValue)[i];
					break;
				case IMINER_MODE_FLOAT:
					for(i=0L; i<nLen; ++i)
						pVector->m_uData.m_pFloat[i] = ((float*)pvValue)[i];
					break;
				case IMINER_MODE_DOUBLE:
					for(i=0L; i<nLen; ++i)
						pVector->m_uData.m_pDouble[i] = ((double*)pvValue)[i];
					break;
				case IMINER_MODE_STRING:
				{
					char** pszNames = (char**)pvValue;
					for(i=0L; i<nLen; ++i)
					{
						if(pszNames[i] && *pszNames[i])
							strcpy(pVector->m_uData.m_ppString[i], pszNames[i]);
					}
					break;
				}
				default:
					nStatus = IMINER_BAD_INPUT_ARGS;
					break;
			}
		}
	}
	else /* nLen <= 0 */
	{
		pVector->m_uData.m_pVoid = NULL;
	}
	/* set the class ID or mode */
	pVector->m_nMode = nMode;
	return IMINER_SUCCESS;
}

/* return 1 if true */
long IMCSRC_STDCALL IMinerVector_isBoolean(const IMinerObject* pVector)
{
	return IMinerVector_isValid(pVector) && pVector->m_nMode==IMINER_MODE_BOOLEAN;
}
/* return 1 if true */
long IMCSRC_STDCALL IMinerVector_isLong(const IMinerObject* pVector)
{
	return IMinerVector_isValid(pVector) && pVector->m_nMode==IMINER_MODE_LONG;
}
/* return 1 if true */
long IMCSRC_STDCALL IMinerVector_isFloat(const IMinerObject* pVector)
{
	return IMinerVector_isValid(pVector) && pVector->m_nMode==IMINER_MODE_FLOAT;
}
/* return 1 if true */
long IMCSRC_STDCALL IMinerVector_isDouble(const IMinerObject* pVector)
{
	return IMinerVector_isValid(pVector) && pVector->m_nMode==IMINER_MODE_DOUBLE;
}
/* return 1 if true */
long IMCSRC_STDCALL IMinerVector_isString(const IMinerObject* pVector)
{
	return IMinerVector_isValid(pVector) && pVector->m_nMode==IMINER_MODE_STRING;
}

/* create a new object and copy content from the source object */
long IMCSRC_STDCALL IMinerVector_clone(IMinerObject* pVector,   /* in/out: data object to copy to */
 const IMinerObject* pvSource  /* in: data object to copy from */
)
{
	if(!IMinerVector_isValid(pvSource))
		return IMINER_BAD_INPUT_ARGS;

	return IMinerVector_create(pVector,
		IMINER_OBJECT_LENGTH(pvSource), IMINER_OBJECT_MODE(pvSource), IMINER_VOID_PTR(pvSource));
}

/* destroy and free memory */
long IMCSRC_STDCALL IMinerVector_destroy(IMinerObject* pVector  /* in/out: data object */)
{
	if(!pVector || pVector->m_nLen < 0L)
		return IMINER_BAD_INPUT_ARGS;
	if(pVector->m_nLen==0L)
		return IMINER_SUCCESS; /* do nothing: an empty vector */
	if(pVector->m_uData.m_pVoid == NULL)
		return IMINER_BAD_INPUT_ARGS;

	switch(pVector->m_nMode)
	{
		case IMINER_MODE_STRING:
		{
			if(pVector->m_uData.m_ppString && pVector->m_uData.m_ppString[0L])
				free(pVector->m_uData.m_ppString[0]); /* free the strings */
			pVector->m_uData.m_ppString[0L] = NULL;
		}
		case IMINER_MODE_BOOLEAN:
		case IMINER_MODE_LONG:
		case IMINER_MODE_FLOAT:
		case IMINER_MODE_DOUBLE:
		  free(pVector->m_uData.m_pVoid);
		  break;
		default:
		  return IMINER_BAD_INPUT_ARGS;
	}
	pVector->m_nLen = 0L;
	return IMINER_SUCCESS;
}

/* add an element to the end of IMinerVector.
*  Warning: the void* pointer must be a correct type
*/
IMCSRC_DLLAPI(long) IMinerVector_add(IMinerObject* pVector,  /* out: data object */
 const void* pElement /* in: NULL or pointer to content of data to be copied. Note that char** is expected for character */
)
{
	long nStatus=IMINER_SUCCESS;
	long i, nLen, nMode;

	if(!pVector || pVector->m_nLen < 0L || !IMINER_IS_VECTOR_MODE(pVector->m_nMode))
		return IMINER_BAD_INPUT_ARGS;

	nMode = IMINER_OBJECT_MODE(pVector);
	nLen  = IMINER_OBJECT_LENGTH(pVector);

	if(nLen == 0L)
	{
		nStatus = IMinerVector_create(pVector, 1L, nMode, pElement);
	}
	else
	{
		/* save pointer to old memory */
		void* pvoidOldValue = pVector->m_uData.m_pVoid;
		/* allocate new memory*/
		nStatus = IMinerObject_alloc(pVector, nLen+1L, nMode);
		if(nStatus != IMINER_SUCCESS)
			return nStatus;
		switch(nMode)
		{
			case IMINER_MODE_BOOLEAN:
			case IMINER_MODE_LONG:
			{
				long* pnOldValue = (long*) pvoidOldValue;
				for(i=0L; i<nLen; ++i)
					pVector->m_uData.m_pLong[i] = pnOldValue[i];
				pVector->m_uData.m_pLong[i] = *((long*) pElement);
				break;
			}
			case IMINER_MODE_FLOAT:
			{
				float* pfOldValue = (float*)pvoidOldValue;
				for(i=0L; i<nLen; ++i)
					pVector->m_uData.m_pFloat[i] = pfOldValue[i];
				pVector->m_uData.m_pFloat[i] = *((float*) pElement);
				free(pfOldValue);
				break;
			}
			case IMINER_MODE_DOUBLE:
			{
				double* pdOldValue = (double*)pvoidOldValue;
				for(i=0L; i<nLen; ++i)
					pVector->m_uData.m_pDouble[i] = pdOldValue[i];
				pVector->m_uData.m_pDouble[i] = *((double*) pElement);
				free(pdOldValue);
				break;
			}
			case IMINER_MODE_STRING:
			{
				char** ppszElement = (char**) pElement;
				char** ppszOldValue = (char**) pvoidOldValue;
				nStatus = IMinerObject_alloc(pVector, nLen+1L, nMode);
				if(nStatus != IMINER_SUCCESS)
					break;
				/* copy old content */
				for(i=0L; i<nLen; ++i)
					strcpy(pVector->m_uData.m_ppString[i], ppszOldValue[i]);
				/* add the new element */
				if(ppszElement[0] && *ppszElement[0])
					strcpy(pVector->m_uData.m_ppString[i], ppszElement[0]);
				/* destroy previously allocated memory */
				free(ppszOldValue[0]);
				break;
			}
			default:
				nStatus = IMINER_BAD_INPUT_ARGS;
				break;
		}
		if(nStatus == IMINER_SUCCESS)
			free(pvoidOldValue);
	}
	return nStatus;
}

/* add an element to the end of IMinerVector. */
IMCSRC_DLLAPI(long) IMinerVector_addLong(IMinerObject* pVector,  /* in/out: data object */
 const long lElement /* in: content to be added */
)
{
	if(IMinerVector_isLong(pVector)
	|| IMinerVector_isBoolean(pVector))
		return IMinerVector_add(pVector, (void*)&lElement);
	else
		return IMINER_FAIL;
}

/* add an element to the end of IMinerVector. */
IMCSRC_DLLAPI(long) IMinerVector_addDouble(IMinerObject* pVector,  /* in/out: data object */
 const double dElement /* in: content to be added */
)
{
	if(IMinerVector_isDouble(pVector))
		return IMinerVector_add(pVector, (void*)&dElement);
	else
		return IMINER_FAIL;
}

/* add an element to the end of IMinerVector. */
IMCSRC_DLLAPI(long) IMinerVector_addString(IMinerObject* pVector,  /* in/out: data object */
 const char* pszElement /* in: content to be added */
)
{
	if(IMinerVector_isString(pVector))
		return IMinerVector_add(pVector, (void*)&pszElement);
	else
		return IMINER_FAIL;
}

/* get element at */
long IMCSRC_STDCALL  IMinerVector_getAt(void* pElement, /* out: a pointer to content of data to be copied.  It must be cast-able to type of pVector*/
 const IMinerObject* pVector,  /* in: data object */
 long i                  /* in: zero-index */
)
{
	if(!IMinerVector_isValid(pVector) || i< 0L || i >= IMINER_OBJECT_LENGTH(pVector))
		return IMINER_BAD_INPUT_ARGS;

	/* have faith that the caller passes in the approriate type */
	switch(IMINER_OBJECT_MODE(pVector))
	{
		case IMINER_MODE_BOOLEAN:
		case IMINER_MODE_LONG:
			*((long*)pElement) = pVector->m_uData.m_pLong[i];
			break;
		case IMINER_MODE_FLOAT:
			*((float*)pElement) =	pVector->m_uData.m_pFloat[i];
			break;
		case IMINER_MODE_DOUBLE:
			*((double*)pElement) = pVector->m_uData.m_pDouble[i];
			break;
		case IMINER_MODE_STRING:
		{
			strcpy(((char**)pElement)[0], pVector->m_uData.m_ppString[i]);
			break;
		}
		default:
			return IMINER_BAD_INPUT_ARGS;
	}
	return IMINER_SUCCESS;
}

long IMCSRC_STDCALL  IMinerVector_getLongAt(long* plElement, /* out: pointer to address of result */
 const IMinerObject* pVector,  /* in: data object */
 long i                        /* in: zero-index */
)
{
	if(IMinerVector_isLong(pVector)
	|| IMinerVector_isBoolean(pVector))
		return IMinerVector_getAt((void*)plElement, pVector, i);
	else
		return IMINER_FAIL;
}

long IMCSRC_STDCALL  IMinerVector_getDoubleAt(double* pdElement, /* out: pointer to address of result */
 const IMinerObject* pVector,  /* in: data object */
 long i                        /* in: zero-index */
)
{
	if(IMinerVector_isDouble(pVector))
		return IMinerVector_getAt((void*)pdElement, pVector, i);
	else
		return IMINER_FAIL;
}

long IMCSRC_STDCALL  IMinerVector_getStringAt(char* pszBuff, /* out: pointer to buffer with length nBufLen */
 size_t nBuffLen,               /* in: length of buffer to recieve content */
 const IMinerObject* pVector,  /* in: data object */
 long i                        /* in: zero-index */
)
{
	if(IMinerVector_isString(pVector)
	&& i < IMINER_OBJECT_LENGTH(pVector)
	&& strlen(pVector->m_uData.m_ppString[i]) < nBuffLen)
	{
		return IMinerVector_getAt((void*) &pszBuff, pVector, i);
	}
	else
		return IMINER_FAIL;
}

/* set element at */

long IMCSRC_STDCALL  IMinerVector_setAt(IMinerObject* pVector,  /* out: data object */
 long i,              /* in: zero-index */
 const void* pElement /* in: a pointer to content of data to be copied.  It must be cast-able to type of pVector*/
)
{
	if(!IMinerVector_isValid(pVector) || i<0L || i > IMINER_OBJECT_LENGTH(pVector))
		return IMINER_BAD_INPUT_ARGS;
	if(i == IMINER_OBJECT_LENGTH(pVector))
		return IMinerVector_add(pVector, pElement);

	/* have faith that the caller passes in the approriate type */
	if(pElement != NULL)
	{
		switch(IMINER_OBJECT_MODE(pVector))
		{
			case IMINER_MODE_BOOLEAN:
			case IMINER_MODE_LONG:
				pVector->m_uData.m_pLong[i] = *((long*)pElement);
				break;
			case IMINER_MODE_FLOAT:
				pVector->m_uData.m_pFloat[i] = *((float*)pElement);
				break;
			case IMINER_MODE_DOUBLE:
				pVector->m_uData.m_pDouble[i] = *((double*)pElement);
				break;
			case IMINER_MODE_STRING:
			{
				char** ppszNames = (char**)pElement;
				if(pVector->m_uData.m_ppString[i] && *pVector->m_uData.m_ppString[i])
					free(pVector->m_uData.m_ppString[i]);
				pVector->m_uData.m_ppString[i] = NULL;

				if(ppszNames[0] && *ppszNames[0])
					strcpy(pVector->m_uData.m_ppString[i], ppszNames[0]);
				break;
			}
			default:
				return IMINER_BAD_INPUT_ARGS;
		}
	}
	return IMINER_SUCCESS;
}

long IMCSRC_STDCALL  IMinerVector_setLongAt(IMinerObject* pVector,  /* out: data object */
 long i,       /* in: zero-index */
 long lElement /* in: value to set*/
)
{
	if(IMinerVector_isLong(pVector)
	|| IMinerVector_isBoolean(pVector))
		return IMinerVector_setAt(pVector, i, (void*)&lElement);
	else
		return IMINER_FAIL;
}

long IMCSRC_STDCALL  IMinerVector_setDoubleAt(IMinerObject* pVector,  /* out: data object */
 long i,       /* in: zero-index */
 double dElement /* in: value to set*/
)
{
	if(IMinerVector_isDouble(pVector))
		return IMinerVector_setAt(pVector, i, (void*)&dElement);
	else
		return IMINER_FAIL;
}

long IMCSRC_STDCALL  IMinerVector_setStringAt(IMinerObject* pVector,  /* out: data object */
 long i,       /* in: zero-index */
 const char* pszElement /* in: value to set*/
)
{
	if(IMinerVector_isString(pVector))
		return IMinerVector_setAt(pVector, i, (void*)&pszElement);
	else
		return IMINER_FAIL;
}

/* get copy of data as string array: caller must free the pointes */
long IMCSRC_STDCALL IMinerVector_getAsStringAt(char* pszBuff, /* out: pointer to buffer for reciving the data*/
 size_t nBuffLen,             /* in: length of buffer */
 const IMinerObject* pVector, /* in: the vector object */
 long i                        /* in: zero-index */
)
{
	long nStatus=IMINER_SUCCESS;
	long nLen, nMode;

	if(IMinerObject_isEmpty(pVector) || !IMinerVector_isValid(pVector) || !pszBuff
	|| i < IMINER_OBJECT_LENGTH(pVector)
	|| strlen(pVector->m_uData.m_ppString[i]) < nBuffLen)
		return IMINER_BAD_INPUT_ARGS;

	nMode = IMINER_OBJECT_MODE(pVector);
	nLen = IMINER_OBJECT_LENGTH(pVector);

	switch(nMode)
	{
		case IMINER_MODE_BOOLEAN:
			IMINER_LONG_PTR(pVector)[i] ? sprintf(pszBuff, "TRUE") : sprintf(pszBuff, "FALSE");
			break;
		case IMINER_MODE_LONG:
			sprintf(pszBuff, "%d", pVector->m_uData.m_pFloat[i]);
			break;
		case IMINER_MODE_FLOAT:
			sprintf(pszBuff, "%g", pVector->m_uData.m_pFloat[i]);
			break;
		case IMINER_MODE_DOUBLE:
			sprintf(pszBuff, "%g", pVector->m_uData.m_pDouble[i]);
			break;
		case IMINER_MODE_STRING:
			strcpy(pszBuff, pVector->m_uData.m_ppString[i]);
			break;
		default:
			nStatus = IMINER_BAD_INPUT_ARGS;
	}
	return nStatus;
}

/* print one elem to stdout */
long IMCSRC_STDCALL IMinerVector_printElement(const IMinerObject* pVector,  /* in: data object */
 long i /* zero-based index */
)
{
	if(!IMinerVector_isValid(pVector) || IMinerObject_isEmpty(pVector))
	{
		IMiner_error("bad vector pointer argument of IMinerVector_printElement\n");
		return IMINER_BAD_INPUT_ARGS;
	}

	switch(pVector->m_nMode)
	{
	case IMINER_MODE_BOOLEAN :
		IMINER_LONG_PTR(pVector)[i] ? printf("TRUE") : printf("FALSE");
		break;
	case IMINER_MODE_LONG :
		printf("%d", IMINER_LONG_PTR(pVector)[i] );
		break;
	case IMINER_MODE_FLOAT :
		printf("%g", IMINER_FLOAT_PTR(pVector)[i] );
		break;
	case IMINER_MODE_DOUBLE :
		printf("%g", IMINER_DOUBLE_PTR(pVector)[i] );
		break;
	case IMINER_MODE_STRING :
		printf("%s", IMINER_STRING_PTR(pVector)[i] );
		break;
	default:
		printf("NULL");
		break;
	}
	return IMINER_SUCCESS;
}

/* print one elem to stdout */
long IMCSRC_STDCALL IMinerVector_print(const IMinerObject* pVector  /* in: data object */)
{
	long i;
	if(!IMinerVector_isValid(pVector))
	{
		IMiner_error("bad vector pointer argument of IMinerVector_printElement\n");
		return IMINER_BAD_INPUT_ARGS;
	}
	else if(IMinerObject_isEmpty(pVector))
		return IMINER_SUCCESS;

	for(i=0L; i< pVector->m_nLen-1L; ++i)
	{
		IMinerVector_printElement(pVector, i);
		printf(", ");
	}
	IMinerVector_printElement(pVector, i);
	printf("\n");
	return IMINER_SUCCESS;
}

/* write to an opened file */
long IMCSRC_STDCALL IMinerVector_writeToStream(FILE* pFile, /* in: output file pointer */
 const IMinerObject* pVector /* in: data object */
)
{
	long i;
	if(!IMinerVector_isValid(pVector))
		return IMINER_BAD_INPUT_ARGS;

	/* write header info */
	IMinerObjectHeader_writeToStream(pFile, NULL, IMINER_OBJECT_MODE(pVector));

	/* write length */
	fprintf(pFile, "%s\n", IMINER_KEYWORD_LENGTH);
	fprintf(pFile, "%d\n", pVector->m_nLen);

	if(pVector->m_nLen < 1L)
		return IMINER_SUCCESS;

	/* write the content */
	fprintf(pFile, "%s\n", IMINER_KEYWORD_DATA);
	switch(pVector->m_nMode)
	{
		case IMINER_MODE_BOOLEAN:
			break;
		case IMINER_MODE_LONG:
			for(i=0L; i< pVector->m_nLen; ++i)
				fprintf(pFile, " %d", pVector->m_uData.m_pLong[i]);
			break;
		case IMINER_MODE_FLOAT:
			for(i=0L; i< pVector->m_nLen; ++i)
				fprintf(pFile, " %16.7e", pVector->m_uData.m_pFloat[i]);
			break;
		case IMINER_MODE_DOUBLE:
			for(i=0L; i< pVector->m_nLen; ++i)
				fprintf(pFile, " %.17g", pVector->m_uData.m_pDouble[i]);
			break;
		case IMINER_MODE_STRING:
		{
			for(i=0L; i< pVector->m_nLen; ++i)
				fprintf(pFile, " %s", pVector->m_uData.m_ppString[i]);
			break;
		}
		default:
			return IMINER_BAD_INPUT_ARGS;
	}
	fprintf(pFile, " \n");
	return IMINER_SUCCESS;
}


/* read from a stream
*  pVector->m_nMode must be properly set for this function to succeed.
*/
long IMCSRC_STDCALL IMinerVector_createFromStream(IMinerObject* pVector, /* out: data object */
 FILE* pFile /* in: input file pointer */
)
{
	long nStatus=IMINER_SUCCESS;
	long i, nLen;
	int len;
	char szLine[LINE_LEN];

	if(!pVector || !IMINER_IS_VECTOR_MODE(pVector->m_nMode))
		return IMINER_BAD_INPUT_ARGS;

	/* do not read mode, it should be read before calling this function. */

	/* read length */
	if( fgets( szLine, LINE_LEN, pFile ) == NULL)
		return IMINER_BAD_INPUT_ARGS;
	fscanf(pFile, "%d\n", &len);
	nLen = len;
	if(nLen < 1L)
	{
		pVector->m_nLen = 0L;
		pVector->m_uData.m_pVoid = NULL;
		return nStatus;
	}
	/* allocate memory for data */
	nStatus= IMinerVector_create(pVector, nLen, pVector->m_nMode, NULL);
	if(nStatus!=IMINER_SUCCESS)
		return nStatus;
	/* read data */
	if( fgets( szLine, LINE_LEN, pFile ) == NULL)
		return IMINER_BAD_INPUT_ARGS;
	switch(pVector->m_nMode)
	{
		case IMINER_MODE_BOOLEAN:
			break;
		case IMINER_MODE_LONG:
			for(i=0L; i< pVector->m_nLen; ++i)
				fscanf(pFile, " %d", &pVector->m_uData.m_pLong[i]);
			break;
		case IMINER_MODE_FLOAT:
			for(i=0L; i< pVector->m_nLen; ++i)
				fscanf(pFile, "%e", &pVector->m_uData.m_pFloat[i]);
			break;
		case IMINER_MODE_DOUBLE:
			for(i=0L; i< pVector->m_nLen; ++i)
			{
				double d;
				fscanf(pFile, "%lf", &d);
				pVector->m_uData.m_pDouble[i] = d;
			}
			break;
		case IMINER_MODE_STRING:
		{
			for(i=0L; i< pVector->m_nLen; ++i)
				fscanf(pFile, " %s", pVector->m_uData.m_ppString[i]);
			break;
		}
		default:
			return IMINER_BAD_INPUT_ARGS;
	}
	/* read the last "\n" */
	if( fgets( szLine, LINE_LEN, pFile ) == NULL)
		return IMINER_BAD_INPUT_ARGS;

	return IMINER_SUCCESS;
}

/********************************* List *********************************************/

/* return 0L if not valid */
long IMCSRC_STDCALL IMinerList_isValid(const IMinerObject* pList)
{
	long i;
	IMinerObject** ppElement;
	/* any list-liked structure : list, factor, matrix, dataset*/
	if(pList==NULL || IMINER_IS_VECTOR_MODE(pList->m_nMode) || (pList->m_nLen>0L && !IMINER_LIST_PTR(pList)))
		return 0L;
	/* easy on 0 length */
	if(pList->m_nLen == 0L)
		return 1L;

	ppElement = IMINER_LIST_PTR(pList);
	for(i=0L; i< pList->m_nLen+1L; ++i) /* nLen of elements + 1 of element names */
	{
		if(!IMinerObject_isValid(ppElement[i]))
			return 0L;
	}
	/* element names : the last one must be a string vector */
	if(IMINER_OBJECT_MODE(ppElement[pList->m_nLen]) != IMINER_MODE_STRING)
		return 0L;
	return 1L;
}

/* alloc memory for object of list-liked structure */
long IMCSRC_STDCALL IMinerList_create(IMinerObject* pList,  /* out: data object */
 long nLen, /* len of list to be created */
 long nMode /* class ID of any list-liked structure*/
)
{
	long nStatus=IMINER_SUCCESS;
	IMinerObject** ppElement;

	if(pList==NULL
	|| IMINER_IS_VECTOR_MODE(nMode)) /* cannot be a vector mode */
		return IMINER_BAD_INPUT_ARGS;

	if(nLen < 0L)
	{
		IMINER_LIST_PTR(pList) = NULL;
		return IMINER_BAD_INPUT_ARGS;
	}
	else if(nLen == 0L)
	{
		return IMinerObject_alloc(pList, 0L, nMode);
	}
	/* alloc the header: nLen+1 to leave room for element names */
	nStatus = IMinerObject_alloc(pList, nLen+1L, nMode);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;

//	IMINER_LIST_PTR(pList) = (IMinerObject**) malloc((nLen+1L)*sizeof(IMinerObject*));
//	if(!IMINER_LIST_PTR(pList))
//		return IMINER_OUT_OF_MEMORY;

	/* alloc an array of header pointers: nLen+1 to leave room for element names */
//	pElement = (IMinerObject*) calloc(nLen+1L, sizeof(IMinerObject));

//	if(!pElement)
//		return IMINER_OUT_OF_MEMORY;

	ppElement = IMINER_LIST_PTR(pList);
//	for(i=0L; i<nLen+1L; ++i) /* nLen of elements + 1 of element names */
//		ppElement[i] = &pElement[i];
	/* element names : the last one must be a string vector */
	IMINER_OBJECT_MODE(ppElement[nLen]) = IMINER_MODE_STRING; /* string vector for names at at the end */
	/* set length */
	pList->m_nLen= nLen;
	/* set the class ID or mode */
	pList->m_nMode = nMode;
	return nStatus;
}

/* create a new object and copy content from the source object */
long IMCSRC_STDCALL IMinerList_clone(IMinerObject* pList,   /* in/out: data object to copy to */
 const IMinerObject* pSource  /* in: data object to copy from */
)
{
	long nStatus=IMINER_SUCCESS;
	long i;
	IMinerObject** ppSourceElement = NULL;
	IMinerObject** ppDistElement = NULL;

	if(!pSource || !IMinerList_isValid(pSource)) /* not a vector is a list-liked */
		return IMINER_BAD_INPUT_ARGS;
	else if(pSource->m_nLen == 0L)
	{
		memset(pList, 0, sizeof(IMinerObject));
		pList->m_nMode = pSource->m_nMode;
		pList->m_nLen = pSource->m_nLen;
		if(pSource->m_szClass && pSource->m_szClass)
			strcpy(pList->m_szClass, pSource->m_szClass);
	}
	else
	{
		/* alloc the top-level mem */
		nStatus = IMinerList_create(pList, pSource->m_nLen, pSource->m_nMode);
		if(nStatus != IMINER_SUCCESS)
			return nStatus;

		ppSourceElement = IMINER_LIST_PTR(pSource);
		ppDistElement = IMINER_LIST_PTR(pList);
		for(i=0L; i<pSource->m_nLen+1L; ++i) /* nLen of elements + 1 of element names */
		{
			nStatus = IMinerObject_clone(ppDistElement[i], ppSourceElement[i]);
			if(nStatus != IMINER_SUCCESS)
				return nStatus;
		}
	}
	return nStatus;
}

/* set names */
long IMCSRC_STDCALL IMinerList_setNamesFromStrings(IMinerObject* pList,   /* in/out: data object */
 const char** ppszNames  /* in: names of the list elements.  length must be the same as pList->m_nLen */
)
{
	long nStatus=IMINER_SUCCESS;
	IMinerObject* pvsNames;

	if(!IMinerList_isValid(pList) || !ppszNames || !ppszNames[0])
		return IMINER_BAD_INPUT_ARGS;

	pvsNames = IMINER_LIST_NAMES_PTR(pList);
	IMinerVector_destroy(pvsNames);
	/* assume length of ppszNames is correct */
	nStatus = IMinerVector_create(pvsNames, pList->m_nLen, IMINER_MODE_STRING, ppszNames);
	return nStatus;
}

/* set names */
long IMCSRC_STDCALL IMinerList_setNames(IMinerObject* pList,   /* in/out: data object */
 const IMinerObject* pvszNames  /* in: names of the list elements.  length must be the same as pList->m_nLen */
)
{
	long nStatus=IMINER_SUCCESS;
	IMinerObject* pvszListNames;

	if(!IMinerList_isValid(pList) || !IMinerVector_isString(pvszNames) || pList->m_nLen!=pvszNames->m_nLen)
		return IMINER_BAD_INPUT_ARGS;

	if(IMinerObject_isEmpty(pList))
		return IMINER_FAIL;

	pvszListNames = IMINER_LIST_NAMES_PTR(pList);
	nStatus = IMinerVector_destroy(pvszListNames);
	nStatus = IMinerVector_clone(pvszListNames, pvszNames);
	return nStatus;
}

/* get a copy of names as a string vector */
long IMCSRC_STDCALL IMinerList_getNames( IMinerObject* pvszNames,  /* out: names of the list elements.  length must be the same as pList->m_nLen */
 const IMinerObject* pList   /* in: the list object */
)
{
	long nStatus=IMINER_SUCCESS;
	IMinerObject* pvszListNames;

	if(!IMinerList_isValid(pList) || !pvszNames)
		return IMINER_BAD_INPUT_ARGS;

	if(IMinerObject_isEmpty(pList))
		return IMINER_FAIL;

	pvszListNames = IMINER_LIST_NAMES_PTR(pList);
	nStatus = IMinerVector_clone(pvszNames, pvszListNames);
	return nStatus;
}

/* set name at */
long IMCSRC_STDCALL IMinerList_setNameAt(IMinerObject* pList,   /* in/out: data object */
 long i,              /* in: zero-index */
 const char* pszName  /* in: name */
)
{
	long nStatus=IMINER_SUCCESS;
	IMinerObject* pvszNames;

	if(!IMinerList_isValid(pList) || !pszName || !pszName[0] || i<0L || i>pList->m_nLen)
		return IMINER_BAD_INPUT_ARGS;

	pvszNames = IMINER_LIST_NAMES_PTR(pList);
	nStatus = IMinerVector_setAt(pvszNames, i, (const void*) pszName);
	return nStatus;
}

/* append an element to the list */
long IMCSRC_STDCALL IMinerList_add(IMinerObject* pList,  /* out: data object */
 const IMinerObject* pElement,     /* in: the element object */
 const char* pszName /* in: element name */
)
{
	long nStatus=IMINER_SUCCESS;
	IMinerObject* pNewElement;
	long i, nOldLen;

	if(pList==NULL || pElement == NULL)
		return IMINER_BAD_INPUT_ARGS;

	nOldLen = IMINER_OBJECT_LENGTH(pList);

	if(nOldLen< 1L) /* empty element ? */
	{
		nOldLen = 0L;
		nStatus = IMinerList_create(pList, 1L, IMINER_MODE_LIST);
	}
	else if(!IMinerList_isValid(pList))
		return IMINER_BAD_INPUT_ARGS;
	else
	{
		IMinerObject** ppElement;
		IMinerObject** ppOldElement = IMINER_LIST_PTR(pList);
		ppElement = (IMinerObject**) malloc((nOldLen+2L)*sizeof(IMinerObject*)); /* add 2 because one of them is always elem names */
		for(i=0L; i< nOldLen; ++i)
			ppElement[i] = ppOldElement[i];
		IMINER_LIST_PTR(pList) = ppElement; /* set to the new array */
		/* get the element names which is assumed to be at the end pass m_nLen */
		ppElement[nOldLen+1] = ppOldElement[nOldLen];
		free(ppOldElement);
	}

	/* clone before set as an element to this list */
	pNewElement = (IMinerObject*) calloc(1L, sizeof(IMinerObject));
	nStatus = IMinerObject_clone(pNewElement, pElement);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;
	IMINER_LIST_PTR(pList)[nOldLen] = pNewElement;
	pList->m_nLen= nOldLen+1L;

	/* add the name */
	nStatus = IMinerVector_add(IMINER_LIST_NAMES_PTR(pList), (const void*) &pszName);
	return nStatus;
}

/* get a copy of the element at : caller is responsible for destroying the output object*/
long IMCSRC_STDCALL  IMinerList_getAt(IMinerObject* pElement, /* out: a pointer to content of data to be copied.  It must be cast-able to type of pVector*/
 const IMinerObject* pList, /* in: data object */
 long i                     /* in: zero-index */
)
{
	if(IMinerObject_isEmpty(pList) || !IMinerList_isValid(pList) || !pElement)
		return IMINER_BAD_INPUT_ARGS;
	return IMinerObject_clone(pElement, IMINER_LIST_PTR(pList)[i]);
}

/* set element at */
long IMCSRC_STDCALL IMinerList_setAt(IMinerObject* pList,  /* out: data object */
 long i,              /* in: zero-index */
 const IMinerObject* pElement /* in: a pointer to content of data to be copied.  It must be cast-able to type of pVector*/
)
{
	long nStatus=IMINER_SUCCESS;

	if(!IMinerList_isValid(pList) || i<0L || i > IMINER_OBJECT_LENGTH(pList) || !pElement)
		return IMINER_BAD_INPUT_ARGS;

	if(i == pList->m_nLen)
		return IMinerList_add(pList, pElement, "");

	/* destroy the old object */
	nStatus = IMinerObject_destroy(IMINER_LIST_PTR(pList)[i]);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;

	/* clone and set it */
	nStatus = IMinerObject_clone(IMINER_LIST_PTR(pList)[i], pElement);
	return nStatus;
}


#if 0
/* TODO */
/* insert element at */
long IMCSRC_STDCALL IMinerList_insertAt(IMinerObject* pList,  /* out: data object */
 long i,              /* in: zero-index */
 const char* pszName, /* in: name of this element */
 const IMinerObject* pElement /* in: a pointer to content of data to be copied.  It must be cast-able to type of pVector*/
{
	long nStatus=IMINER_SUCCESS;
	IMinerObject* pNewElement;
	IMinerObject** ppElement;
	IMinerObject** ppOldElement;
	long ii, nOldLen;

	if(!IMinerList_isValid(pList) || i<0L || i > IMINER_OBJECT_LENGTH(pList) || !pElement)
		return IMINER_BAD_INPUT_ARGS;

	nOldLen = IMINER_OBJECT_LENGTH(pList);
	if(i == nOldLen)
		return IMinerList_add(pList, pElement);

	/* clone before set as an element to this list */
	pNewElement = (IMinerObject*) calloc(1L, sizeof(IMinerObject));
	nStatus = IMinerObject_clone(pNewElement, pElement);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;
	/* alloc new headers */
	ppOldElement = IMINER_LIST_PTR(pList);
	ppElement = (IMinerObject**) malloc((nOldLen+2L)*sizeof(IMinerObject*)); /* add 2 because one of them is always elem names */
	/* put back the element with index less than i */
	for(ii=0L; ii< i; ++ii)
		ppElement[ii] = ppOldElement[ii];
	/* insert it */
	ppElement[i] = pNewElement;
	/* put back the element with index greater than equal to i */
	for(ii=i; ii< nOldLen; ++ii)
		ppElement[ii+1L] = ppOldElement[ii];
	/* get the element names which is assumed to be at the end pass m_nLen */
	ppElement[nOldLen+1L] = ppOldElement[nOldLen]; /* same as IMINER_LIST_NAMES_PTR(pList) = ppOldElement[nOldLen]; */
	/* set to the new array */
	IMINER_LIST_PTR(pList) = ppElement;
	free(ppOldElement);

	pList->m_nLen= nOldLen+1L;

	/* insert the name */
	nStatus = IMinerVector_insertAt(IMINER_LIST_NAMES_PTR(pList), i, (const void*) &pszName);
	return nStatus;
}
#endif

/* destroy and free memory */
long IMCSRC_STDCALL IMinerList_destroy(IMinerObject* pList  /* in/out: data object */)
{
	long nStatus=IMINER_SUCCESS;
	long i;
	IMinerObject** ppElement = NULL;

	if(!IMinerList_isValid(pList))
		return IMINER_BAD_INPUT_ARGS;
	if(pList->m_nLen == 0L)
		return nStatus;

	ppElement = IMINER_LIST_PTR(pList);
	for(i=0L; i< pList->m_nLen+1L; ++i) /* nLen of elem + 1 of elem names */
	{
		if(!ppElement[i])
			return IMINER_BAD_INPUT_ARGS;
		nStatus = IMinerObject_destroy(ppElement[i]);
		if(nStatus != IMINER_SUCCESS)
			return nStatus;
	}
	free(IMINER_LIST_PTR(pList)[0]);
	free(IMINER_LIST_PTR(pList));
	IMINER_LIST_PTR(pList) = NULL;

	pList->m_nLen = 0L;

	return nStatus;
}

/* print one elem to stdout */
long IMCSRC_STDCALL IMinerList_printElement(const IMinerObject* pList,  /* in: data object */
 long i /* zero-based index */
)
{
	long nStatus=IMINER_SUCCESS;

	if(!IMinerList_isValid(pList) || pList->m_nLen == 0L)
	{
		IMiner_error("bad List pointer argument of IMinerList_printElement\n");
		return IMINER_BAD_INPUT_ARGS;
	}

	/* print element name */
	printf("$");
	nStatus = IMinerVector_printElement(IMINER_LIST_NAMES_PTR(pList), i);
	printf("\n");

	if(!IMINER_LIST_PTR(pList)[i])
		return IMINER_BAD_INPUT_ARGS;

	switch(IMINER_OBJECT_MODE(IMINER_LIST_PTR(pList)[i]))
	{
		case IMINER_MODE_BOOLEAN :
		case IMINER_MODE_LONG :
		case IMINER_MODE_FLOAT :
		case IMINER_MODE_DOUBLE :
		case IMINER_MODE_STRING  :
			nStatus = IMinerVector_print((IMinerObject*) IMINER_LIST_PTR(pList)[i]);
			break;
		case IMINER_MODE_FACTOR :
			nStatus = IMinerFactor_print((IMinerObject*) IMINER_LIST_PTR(pList)[i]);
			break;
		case IMINER_MODE_DOUBLEMATRIX :
			nStatus = IMinerDoubleMatrix_print((IMinerDoubleMatrix*) IMINER_LIST_PTR(pList)[i]);
			break;
		case IMINER_MODE_MODELMATRIX :
			nStatus = IMinerModelMatrix_print((IMinerObject*) IMINER_LIST_PTR(pList)[i]);
			break;
		case IMINER_MODE_DATASET :
			nStatus = IMinerDataSet_print((IMinerObject*) IMINER_LIST_PTR(pList)[i]);
			break;
		default:
			nStatus = IMINER_FAIL;
			break;
	}
	return nStatus;
}

/* print the whole object to stdout */
long IMCSRC_STDCALL IMinerList_print(const IMinerObject* pList  /* in: data object */)
{
	long i;
	if(!IMinerList_isValid(pList))
	{
		IMiner_error("bad List pointer argument of IMinerList_printElement\n");
		return IMINER_BAD_INPUT_ARGS;
	}
	for(i=0L; i< pList->m_nLen; ++i)
	{
		IMinerList_printElement(pList, i);
		printf("\n");
	}
	return IMINER_SUCCESS;
}

/* write to an opened file */
long IMCSRC_STDCALL IMinerList_writeToStream(FILE* pFile, /* in: output file pointer */
 const IMinerObject* pList /* in: data object */
)
{
	long nStatus=IMINER_SUCCESS;
	long i;
	IMinerObject** ppElement=NULL;

	if(!IMinerList_isValid(pList))
		return IMINER_BAD_INPUT_ARGS;

	/* write header info */
	IMinerObjectHeader_writeToStream(pFile, NULL, IMINER_OBJECT_MODE(pList));

	/* write length */
	fprintf(pFile, "%s\n", IMINER_KEYWORD_LENGTH);
	fprintf(pFile, "%d\n", pList->m_nLen);

	/* write the data */
	fprintf(pFile, "%s\n", IMINER_KEYWORD_DATA);

	/* loop through all elements */
	ppElement = IMINER_LIST_PTR(pList);
	for(i=0L; i< pList->m_nLen+1L; ++i) /* nLen of elem + 1 of elem names */
	{
		if(!ppElement[i])
			return IMINER_BAD_INPUT_ARGS;
		nStatus = IMinerObject_writeToStream(pFile, ppElement[i]);
		if(nStatus != IMINER_SUCCESS)
			return nStatus;
	}

	return nStatus;
}
/* read from a stream */
long IMCSRC_STDCALL IMinerList_createFromStream(IMinerObject* pList, /* out: data object */
 FILE* pFile /* in: input file pointer */
)
{
	long nStatus=IMINER_SUCCESS;
	long i, nLen;
	int len;
	char szLine[LINE_LEN];
	IMinerObject** ppElement=NULL;

	if(pList==NULL || IMINER_IS_VECTOR(pList)) /* not a vector mode is a list-liked object */
		return IMINER_BAD_INPUT_ARGS;

	/* do not read mode, it should be read before calling this function. */

	/* read length */
	if( fgets( szLine, LINE_LEN, pFile ) == NULL)
		return IMINER_BAD_INPUT_ARGS;
	fscanf(pFile, "%d\n", &len);
	nLen = len;
	if(nLen < 1L)
	{
		pList->m_nLen = 0L;
		IMINER_LIST_PTR(pList) = NULL;
		return nStatus;
	}
	/* alloc memory for a list object */
	nStatus = IMinerList_create(pList, nLen, IMINER_OBJECT_MODE(pList)); /* a list-liked object */
	if(nStatus != IMINER_SUCCESS)
		return nStatus;
	/* read the data */
	if( fgets( szLine, LINE_LEN, pFile ) == NULL)
		return IMINER_BAD_INPUT_ARGS;

	ppElement = IMINER_LIST_PTR(pList);
	for(i=0L; i< nLen+1L; ++i) /* nLen of elements and 1 for element names */
	{
		nStatus = IMinerObject_createFromStream(ppElement[i], pFile);
		if(nStatus != IMINER_SUCCESS)
			return nStatus;
	}
	pList->m_nLen = nLen; /* make sure the length does not include element names */
	return nStatus;

}

/********************************* Factor : List *********************************************/

/* return 0L if not valid */
long IMCSRC_STDCALL IMinerFactor_isValid(const IMinerObject* pFactor)
{
	if(pFactor->m_nMode != IMINER_MODE_FACTOR || IMINER_OBJECT_LENGTH(pFactor) != 2L)
		return 0L;
	/* A factor object must have 2 vector elements */
	if(!IMinerVector_isValid(IMINER_LIST_PTR(pFactor)[IMINER_FACTOR_DATA_MEMBER])
	|| !IMinerVector_isValid(IMINER_LIST_PTR(pFactor)[IMINER_FACTOR_LEVEL_MEMBER]))
		return 0L;
	/* the first vector must be integer vector */
	if(IMINER_OBJECT_MODE(IMINER_LIST_PTR(pFactor)[IMINER_FACTOR_DATA_MEMBER]) != IMINER_MODE_LONG)
		return 0L;
	/* the 2nd vector must be char vector */
	if(IMINER_OBJECT_MODE(IMINER_LIST_PTR(pFactor)[IMINER_FACTOR_LEVEL_MEMBER]) != IMINER_MODE_STRING)
		return 0L;
	return 1L;
}

/* alloc memory for object of class factor */
long IMCSRC_STDCALL IMinerFactor_create(IMinerObject* pFactor,  /* out: data object */
 long nLen,            /* in: length of pnValue*/
 const long*  pnValue, /* in: value of length nLen */
 long nLevels,         /* in: length of ppcLevel */
 const char** ppcLevel /* in: level */
)
{
	long nStatus=IMINER_SUCCESS;
	char* ppszMemberNames[] = {"data", "levels"};

	if(pFactor==NULL || nLen < 0L || nLevels <0L)
		return IMINER_BAD_INPUT_ARGS;
	/* a factor always have 2 elements as list : the 3th element is for member names as a string vector */
	nStatus = IMinerList_create(pFactor, 2L, IMINER_MODE_FACTOR);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;

	/*member 0 : data */
	nStatus = IMinerVector_create(IMINER_FACTOR_DATA_PTR(pFactor), nLen, IMINER_MODE_LONG, (void*)pnValue);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;
	/*member 1 : levels */
	nStatus = IMinerVector_create(IMINER_FACTOR_LEVELS_PTR(pFactor), nLevels, IMINER_MODE_STRING, (void*)ppcLevel);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;

	/*member 2 : member names */
	nStatus = IMinerList_setNamesFromStrings(pFactor, (const char**)ppszMemberNames);

	/* set the class ID */
	pFactor->m_nMode = IMINER_MODE_FACTOR;
	return nStatus;
}

/* create a new object and copy content from the char array */
long IMCSRC_STDCALL IMinerFactor_createFromStrings(IMinerObject* pFactor,  /* out: data object */
 long nLen,       /* in: length of ppszValue*/
 const char** ppszValue /* in: data */
)
{
	long nStatus=IMINER_SUCCESS;
	long* pnLevel;
	char** ppszLevel;
	long i, nLevels;

#ifdef USE_HASH_LOOKUP
	IMinerHashTable *level_hash = IMinerHashTable_createHash();
	IMinerHashTable_put(level_hash, ppszValue[0], 0);
#endif /*USE_HASH_LOOKUP*/

	if(pFactor==NULL || nLen < 1L || ppszValue == NULL)
		return IMINER_BAD_INPUT_ARGS;

	/* create an empty factor object. */
	nStatus = IMinerFactor_create(pFactor, 0L, NULL, 0L, NULL);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;
	/* member 0: data */
	nStatus = IMinerVector_create(IMINER_FACTOR_DATA_PTR(pFactor), nLen, IMINER_MODE_LONG, NULL);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;

	/* how many levels? */
	pnLevel = (long*) calloc(nLen, sizeof(long));
	nLevels = 1L;
	pnLevel[0L] = 0L;

	IMINER_FACTOR_DATA_VALUE(pFactor, 0L) = 0L ;

	for(i=1L; i< nLen; ++i)
	{
		int iFound = -1L;

#ifdef USE_HASH_LOOKUP
		iFound = IMinerHashTable_get(level_hash, ppszValue[i]);
		if (iFound == -1) {
			iFound = nLevels;
			pnLevel[nLevels] = i;
			++nLevels;

			IMinerHashTable_put(level_hash, ppszValue[i], iFound);
		}

#else
		int nUnique = 0L;

		do
		{
			if(strcmp(ppszValue[pnLevel[nUnique]], ppszValue[i]) == 0) /* already exist? */
				iFound = nUnique;
		}
		while(iFound < 0L && ++nUnique < nLevels);

		if(iFound < 0L)
		{/* new levels */
			iFound = nLevels;
			pnLevel[nLevels] = i;
			++nLevels;
		}
#endif /*USE_HASH_LOOKUP*/

		IMINER_FACTOR_DATA_VALUE(pFactor, i) = iFound; /* zero-index */
	}

	ppszLevel = (char**) malloc(nLen*sizeof(char*));
	for(i=0L; i< nLevels; ++i)
		ppszLevel[i] = (char*) ppszValue[pnLevel[i]];

	/* create content for the levels */
	nStatus = IMinerVector_create(IMINER_FACTOR_LEVELS_PTR(pFactor), nLevels, IMINER_MODE_STRING, (void*)ppszLevel);

#ifdef USE_HASH_LOOKUP
	IMinerHashTable_freeHash(level_hash);
#endif /*USE_HASH_LOOKUP*/

	if(pnLevel)
		free(pnLevel);
	if(ppszLevel)
		free(ppszLevel);
	return nStatus;
}

/* print one elem to stdout */
long IMCSRC_STDCALL IMinerFactor_printElement(const IMinerObject* pFactor,  /* in: data object */
 long i /* zero-based index */
)
{
	long nLevel;

	if(!IMinerFactor_isValid(pFactor))
	{
		IMiner_error("bad Factor pointer argument of IMinerFactor_printElement\n");
		return IMINER_BAD_INPUT_ARGS;
	}
	nLevel = IMINER_LONG_VALUE(IMINER_FACTOR_DATA_PTR(pFactor), i);
	return IMinerVector_printElement(IMINER_FACTOR_LEVELS_PTR(pFactor), nLevel);
}

/* print one elem to stdout */
long IMCSRC_STDCALL IMinerFactor_print(const IMinerObject* pFactor  /* in: data object */)
{
	long i, nLen, iLevel;
	if(!IMinerFactor_isValid(pFactor))
	{
		IMiner_error("bad Factor pointer argument of IMinerFactor_printElement\n");
		return IMINER_BAD_INPUT_ARGS;
	}
	nLen = IMINER_OBJECT_LENGTH(IMINER_FACTOR_DATA_PTR(pFactor));
	for(i=0L; i< nLen-1L; ++i)
	{
		iLevel = IMINER_LONG_VALUE(IMINER_FACTOR_DATA_PTR(pFactor), i);
		IMinerVector_printElement(IMINER_FACTOR_LEVELS_PTR(pFactor), iLevel);
		printf(", ");
	}
	iLevel = IMINER_LONG_VALUE(IMINER_FACTOR_DATA_PTR(pFactor), i);
	IMinerVector_printElement(IMINER_FACTOR_LEVELS_PTR(pFactor), iLevel);
	printf("\n");
	return IMINER_SUCCESS;
}

/********************************* DoubleMatrix : List ******************************************/

/* return 1 if valid */
long IMCSRC_STDCALL IMinerDoubleMatrix_isValid(const IMinerObject* pdMatrix)
{
	long nLen, nRows, nColumns;
	IMinerObject* pElement;

	/* ignor mode of this class so that its "derived class" can use this function as a method. */
	if(pdMatrix == NULL
	|| !IMinerList_isValid(pdMatrix) /* kind of list ? */
	|| IMINER_OBJECT_LENGTH(pdMatrix) < 4L) /* length at least 4?: data, dim, row names, col names */
		return 0L;
	/* member 0: data */
	pElement = IMINER_MATRIX_DATA_PTR(pdMatrix);
	if(!IMinerVector_isDouble(pElement))
		return 0L;
	/* member 1: dim */
	pElement = IMINER_MATRIX_DIM_PTR(pdMatrix);
	if(!IMinerVector_isLong(pElement) && pElement->m_nLen == 2L)
		return 0L;
	/* member 2: row name */
	pElement = IMINER_MATRIX_ROWNAME_PTR(pdMatrix);
	if(!IMinerVector_isString(pElement))
		return 0L;
	/* member 3: column name */
	pElement = IMINER_MATRIX_COLUMNNAME_PTR(pdMatrix);
	if(!IMinerVector_isString(pElement))
		return 0L;
	/* special member : member names mut also be valid and has the same length as this object*/
	pElement = IMINER_LIST_NAMES_PTR(pdMatrix);
	if(!IMinerVector_isString(pElement) && pElement->m_nLen == pdMatrix->m_nLen)
		return 0L;

	/* total length of the data */
	nLen     = IMINER_OBJECT_LENGTH(IMINER_MATRIX_DATA_PTR(pdMatrix));
	/* number of rows and columns */
	nRows    = IMINER_MATRIX_NROWS(pdMatrix);
	nColumns = IMINER_MATRIX_NCOLUMNS(pdMatrix);
	/* total length must be the same as nRow * nColumns */
	if(nLen != (nRows*nColumns))
		return 0L;
	return 1L;
}

/* alloc memory and init content for a DoubleMatrix object */
long IMCSRC_STDCALL IMinerDoubleMatrix_create(IMinerObject* pdMatrix,  /* out: data object */
 long nRows,                 /* in:  nunber of rows */
 long nColumns,              /* in:  nunber of columns */
 const double* pdValue,      /* in: NULL or content to be set, column-dominent */
 const char** ppszRowName,   /* in: row names if not NULL */
 const char** ppszColumnName /* in: column names if not NULL */
)
{
	long nStatus = IMINER_SUCCESS;
	long pnDim[2];
	char* ppszMemberNames[] = {"data", "dim", "rowNames", "columnNames"};
	if(pdMatrix == NULL || nRows < 1L  || nColumns < 1L)
		return IMINER_BAD_INPUT_ARGS;

	/* a matrix always have 4 elements as list : the 5th element is for member names as a string vector */
	nStatus = IMinerList_create(pdMatrix, 4L, IMINER_MODE_DOUBLEMATRIX);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;

	/* member 0: data */
	nStatus = IMinerVector_create(IMINER_MATRIX_DATA_PTR(pdMatrix), nRows*nColumns, IMINER_MODE_DOUBLE, pdValue);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;
	/* member 1: dim */
	pnDim[0] = nRows;
	pnDim[1] = nColumns;
	nStatus = IMinerVector_create(IMINER_MATRIX_DIM_PTR(pdMatrix), 2L, IMINER_MODE_LONG, pnDim);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;
	/* member 2: row names */
	if(ppszRowName && ppszRowName[0])
		nStatus = IMinerVector_create(IMINER_MATRIX_ROWNAME_PTR(pdMatrix), nRows, IMINER_MODE_STRING, ppszRowName); /* risky */
	else
		nStatus = IMinerVector_create(IMINER_MATRIX_ROWNAME_PTR(pdMatrix), 0L, IMINER_MODE_STRING, NULL); /* risky */
	if(nStatus != IMINER_SUCCESS)
		return nStatus;
	/* member 3: column names */
	if(ppszColumnName && ppszColumnName[0])
		nStatus = IMinerVector_create(IMINER_MATRIX_COLUMNNAME_PTR(pdMatrix), nColumns, IMINER_MODE_STRING, ppszColumnName);
	else
		nStatus = IMinerVector_create(IMINER_MATRIX_COLUMNNAME_PTR(pdMatrix), 0L, IMINER_MODE_STRING, NULL);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;
	/*member 4 : member names */
	nStatus = IMinerList_setNamesFromStrings(pdMatrix, (const char**)ppszMemberNames);

	/* set the class ID */
	pdMatrix->m_nMode = IMINER_MODE_DOUBLEMATRIX;
	return nStatus;
}

/* alloc memory for the object and copy content from the source*/
long IMCSRC_STDCALL IMinerDoubleMatrix_createFromDataSet(IMinerObject* pdMatrix,  /* out: data object */
 const IMinerObject* pDataSet /* in: source */
)
{
	/* treatment contrasts only */
	long nStatus = IMINER_SUCCESS;
	long i,j, jj, jMatrix, nLevels, nRows, nCols, nMatrixCols;
	IMinerObject* pFactor;
	IMinerObject* pVector;
	char szLine[LINE_LEN];

	if(pdMatrix == NULL || !IMinerDataSet_isValid(pDataSet))
		return IMINER_BAD_INPUT_ARGS;

	nRows = IMINER_DATASET_NROWS(pDataSet);
	nCols = IMINER_DATASET_NCOLUMNS(pDataSet);
	nLevels = 0L;
	nMatrixCols=0L;

	/* get total levels from all factor columns*/
	for(j=0L; j< nCols; ++j)
	{
		if(IMINER_DATASET_IS_FACTOR(pDataSet, j))
		{
			pFactor = IMINER_DATASET_FACTOR_PTR(pDataSet, j);
			nLevels = IMINER_FACTOR_NLEVELS(pFactor);
			nMatrixCols += nLevels;
		}
		else
			++nMatrixCols;
	}

	/* create the matrix with initial value of zero elements */
	nStatus = IMinerDoubleMatrix_create(pdMatrix, nRows, nMatrixCols, NULL, NULL, NULL);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;

	/* create column names */
//YB:test	IMinerVector_create(&pdMatrix->m_dMatrix.m_vchColumnName, nMatrixCols, IMINER_MODE_STRING, NULL);

	/* fill in the data */
	jMatrix = 0L;
	for(j=0L; j< nCols; ++j)
	{
		if(IMINER_DATASET_IS_FACTOR(pDataSet, j))
		{
			pFactor = IMINER_DATASET_FACTOR_PTR(pDataSet, j);

			/* turn on the elem of column that matches with the level */
			for(i=0L; i< nRows; ++i)
			{
				jj = j+IMINER_FACTOR_DATA_VALUE(pFactor, i); /* zero-index */
				IMINER_Aij(pdMatrix, i, jj) = 1.0;
			}

			/* set column names */
			for(jj=0; jj< IMINER_OBJECT_LENGTH(IMINER_FACTOR_LEVELS_PTR(pFactor)); ++jj)
			{
				char* pszLine = szLine;
				sprintf(szLine, "%s(%s)", IMINER_STRING_VALUE(IMINER_DATASET_COLUMNNAME_PTR(pDataSet), j), IMINER_STRING_VALUE(IMINER_FACTOR_LEVELS_PTR(pFactor), jj));
				IMinerVector_setAt(IMINER_MATRIX_COLUMNNAME_PTR(pdMatrix), jMatrix, (const void*) &pszLine);
				++jMatrix;
			}
		}
		else if(IMINER_DATASET_IS_VECTOR(pDataSet, j))
		{
			pVector = IMINER_DATASET_VECTOR_PTR(pDataSet, j);
			/* cast the content from any to double. */
			switch(IMINER_OBJECT_MODE(pVector))
			{
				case IMINER_MODE_BOOLEAN:
				case IMINER_MODE_LONG:
					for(i=0L; i<nRows; ++i)
						IMINER_Aij(pdMatrix, i, jMatrix) = (double) pVector->m_uData.m_pLong[i];
					break;
				case IMINER_MODE_FLOAT:
					for(i=0L; i<nRows; ++i)
						IMINER_Aij(pdMatrix, i, jMatrix) = (double) pVector->m_uData.m_pFloat[i];
					break;
				case IMINER_MODE_DOUBLE:
					for(i=0L; i<nRows; ++i)
						IMINER_Aij(pdMatrix, i, jMatrix) = (double) pVector->m_uData.m_pDouble[i];
					break;
				default:
					nStatus=IMINER_BAD_INPUT_ARGS;
			}

			/* set column name */
			IMinerVector_setAt(IMINER_MATRIX_COLUMNNAME_PTR(pdMatrix), jMatrix, &IMINER_STRING_VALUE(IMINER_DATASET_COLUMNNAME_PTR(pDataSet), j));
			++jMatrix;
		}
	}
	return nStatus;
}

/* get column names as a new IMinerVector.  Make sure you call IMinerVector_destroy() to avoid memory leak*/
long IMCSRC_STDCALL IMinerDoubleMatrix_getColumnNames(IMinerObject* pvszColumnName,   /* out: string vector of the column */
 const IMinerDoubleMatrix* pdMatrix /* in: data object */
)
{
	long nStatus = IMINER_SUCCESS;
	if(!IMinerDoubleMatrix_isValid(pdMatrix) || !pvszColumnName || !IMINER_IS_VECTOR(pvszColumnName) )
		return IMINER_BAD_INPUT_ARGS;
	nStatus = IMinerVector_clone(pvszColumnName, IMINER_MATRIX_COLUMNNAME_PTR(pdMatrix));
	return nStatus;
}
/* set column names*/
long IMCSRC_STDCALL IMinerDoubleMatrix_setColumnNames(IMinerDoubleMatrix* pdMatrix, /* out: data object */
 const IMinerObject* pvszColumnName   /* in: string vector of the column */
)
{
	long nStatus = IMINER_SUCCESS;
	if(!IMinerDoubleMatrix_isValid(pdMatrix) || !IMinerVector_isValid(pvszColumnName) )
		return IMINER_BAD_INPUT_ARGS;
	nStatus = IMinerVector_destroy(IMINER_MATRIX_COLUMNNAME_PTR(pdMatrix));
	if(nStatus != IMINER_SUCCESS)
		return nStatus;
	nStatus = IMinerVector_clone(IMINER_MATRIX_COLUMNNAME_PTR(pdMatrix), pvszColumnName);
	return nStatus;
}

/* set column names*/
long IMCSRC_STDCALL IMinerDoubleMatrix_setColumnNamesFromStrings(IMinerDoubleMatrix* pdMatrix, /* out: data object */
 const char** ppszColumnName   /* in: string vector of the column */
)
{
	long nStatus = IMINER_SUCCESS;
	if(!IMinerDoubleMatrix_isValid(pdMatrix) || !ppszColumnName || !ppszColumnName[0])
		return IMINER_BAD_INPUT_ARGS;
	nStatus = IMinerVector_destroy(IMINER_MATRIX_COLUMNNAME_PTR(pdMatrix));
	if(nStatus != IMINER_SUCCESS)
		return nStatus;
	/* risky : assume length of ppszColumnName to be the same as ncolumns */
	nStatus = IMinerVector_create(IMINER_MATRIX_COLUMNNAME_PTR(pdMatrix), IMINER_MATRIX_NCOLUMNS(pdMatrix), IMINER_MODE_STRING, ppszColumnName);
	return nStatus;
}

/* get row names as a new IMinerVector object.  Make sure you call IMinerVector_destroy() to avoid memory leak */
long IMCSRC_STDCALL IMinerDoubleMatrix_getRowNames(IMinerObject* pvszRowName,  /* out: names */
 const IMinerDoubleMatrix* pdMatrix /* in: data object */
)
{
	long nStatus = IMINER_SUCCESS;
	if(!IMinerDoubleMatrix_isValid(pdMatrix) || !pvszRowName || !IMINER_IS_VECTOR(pvszRowName) )
		return IMINER_BAD_INPUT_ARGS;
	nStatus = IMinerVector_clone(pvszRowName, IMINER_MATRIX_ROWNAME_PTR(pdMatrix));
	return nStatus;
}

/* set row names*/
long IMCSRC_STDCALL IMinerDoubleMatrix_setRowNames(IMinerDoubleMatrix* pdMatrix, /* out: data object */
 const IMinerObject* pvszRowName   /* in: string vector of the column */
)
{
	long nStatus = IMINER_SUCCESS;
	if(!IMinerDoubleMatrix_isValid(pdMatrix) || !IMinerVector_isValid(pvszRowName) )
		return IMINER_BAD_INPUT_ARGS;
	nStatus = IMinerVector_destroy(IMINER_MATRIX_ROWNAME_PTR(pdMatrix));
	if(nStatus != IMINER_SUCCESS)
		return nStatus;
	nStatus = IMinerVector_clone(IMINER_MATRIX_ROWNAME_PTR(pdMatrix), pvszRowName);
	return nStatus;
}

long IMCSRC_STDCALL IMinerDoubleMatrix_setRowNamesFromStrings(IMinerDoubleMatrix* pdMatrix, /* out: data object */
 const char** ppszRowName   /* in: string vector of the column */
)
{
	long nStatus = IMINER_SUCCESS;
	if(!IMinerDoubleMatrix_isValid(pdMatrix) || !ppszRowName || !ppszRowName[0] )
		return IMINER_BAD_INPUT_ARGS;
	nStatus = IMinerVector_destroy(IMINER_MATRIX_ROWNAME_PTR(pdMatrix));
	if(nStatus != IMINER_SUCCESS)
		return nStatus;
	/* risky : assume length of ppszColumnName to be the same as ncolumns */
	nStatus = IMinerVector_create(IMINER_MATRIX_ROWNAME_PTR(pdMatrix), IMINER_MATRIX_NCOLUMNS(pdMatrix), IMINER_MODE_STRING, ppszRowName);
	return nStatus;
}

/* get number of rows */
long IMCSRC_STDCALL IMinerDoubleMatrix_getNRows(long* pnNumberOfRows, /* out: number of rows */
																const IMinerDoubleMatrix* pdMatrix /* in: data object */
)
{
	if(!IMinerDoubleMatrix_isValid(pdMatrix))
		return IMINER_BAD_INPUT_ARGS;

	*pnNumberOfRows= IMINER_MATRIX_NROWS(pdMatrix);
	return IMINER_SUCCESS;
}

/* get number of columns */
long IMCSRC_STDCALL IMinerDoubleMatrix_getNColumns(long* pnNumberOfColumns, /* out: number of columns */
																const IMinerDoubleMatrix* pdMatrix /* in: data object */
)
{
	if(!IMinerDoubleMatrix_isValid(pdMatrix))
		return IMINER_BAD_INPUT_ARGS;

	*pnNumberOfColumns= IMINER_MATRIX_NCOLUMNS(pdMatrix);
	return IMINER_SUCCESS;
}

/* print to standard out */
long IMCSRC_STDCALL IMinerDoubleMatrix_print(const IMinerDoubleMatrix* pdMatrix )
{
	long i, j, nRows=0L, nColumns=0L;
	char szdFormat[32];
	char szMsg[155];
	IMinerObject* pVector;

	/* printf("Begin IMinerDoubleMatrix_print ...\n"); */
	if(!IMinerDoubleMatrix_isValid(pdMatrix))
	{
		IMiner_error("Invalid pdMatrix\n");
		return IMINER_BAD_INPUT_ARGS;
	}
	/* print to stdout */
	sprintf(szdFormat, "%d", IMINER_MATRIX_NROWS(pdMatrix));
	sprintf(szdFormat, "[%c%dd,] ", '%', strlen(szdFormat));

	/* print the column names */
	pVector = IMINER_MATRIX_COLUMNNAME_PTR(pdMatrix);
	if(IMinerVector_isValid(pVector) && IMINER_OBJECT_LENGTH(pVector) > 0L)
	{
		nColumns = IMINER_OBJECT_LENGTH(pVector);
		sprintf(szMsg, szdFormat, 1);
		for(i=0L; i < (long)strlen(szMsg); ++i)
			printf(" ");
		for(j=0L; j< nColumns-1L; ++j)
			printf("%s, ", IMINER_STRING_VALUE(pVector, j));
		printf("%s\n", IMINER_STRING_VALUE(pVector, j));
	}

	/* print the row names */
	pVector = IMINER_MATRIX_ROWNAME_PTR(pdMatrix);
	nRows = IMINER_MATRIX_NROWS(pdMatrix);
	nColumns = IMINER_MATRIX_NCOLUMNS(pdMatrix);
	for(i=0L; i<nRows; ++i)
	{
		printf(szdFormat,i+1);
		for(j=0L; j< nColumns-1L; ++j)
		{
			/* TODO: print row names */
			/* if(IMinerVector_isValid(pVector) && IMINER_OBJECT_LENGTH(pVector) > 0L) */

				printf("%g, ", IMINER_Aij(pdMatrix, i, j) );
		}
		printf("%g\n", IMINER_Aij(pdMatrix, i, nColumns-1L) );
	}

	return IMINER_SUCCESS;
}

/********************************* ModelMatrix : DoubleMatrix : List *************************************/

long IMCSRC_STDCALL IMinerModelMatrix_isValid(const IMinerObject* pMMatrix)
{
	IMinerObject* plistLevel;
	long i, nLen, nColumns, nTotalLevels, nLevels;

	/* check with "base class" first */
	if(!IMinerDoubleMatrix_isValid(pMMatrix))
		return 0L;
	/* it has one element added to double matrix */
	if( IMINER_OBJECT_LENGTH(pMMatrix) < 5L)
		return 0L;

	plistLevel = IMINER_MODELMATRIX_LEVEL_PTR(pMMatrix);
	if(!IMinerList_isValid(plistLevel))
		return 0L;

	nLen = IMINER_OBJECT_LENGTH(plistLevel);
	nTotalLevels = 0L;
	for(i=0; i<nLen; ++i)
	{
		nLevels = IMINER_OBJECT_LENGTH(IMINER_LIST_PTR(plistLevel)[i]);
		if(nLevels==0L) /* a vector column */
			nLevels = 1L; /* one column for a vector */
		nTotalLevels += nLevels;
	}

	nColumns = IMINER_MATRIX_NCOLUMNS(pMMatrix);
	if( nTotalLevels!= nColumns)
		return 0L;
	return 1L;
}

/* alloc memory and init content for a ModelMatrix object */
long IMCSRC_STDCALL IMinerModelMatrix_create(IMinerObject* pMMatrix,  /* out: data object */
 long nRows,                 /* in:  nunber of rows */
 long nColumns,              /* in:  nunber of columns */
 const double* pdValue,      /* in: NULL or content to be set, column-dominent */
 const char** ppszRowName,   /* in: row names if not NULL */
 const char** ppszColumnName /* in: column names if not NULL */
)
{
	long nStatus = IMINER_SUCCESS;
	IMinerObject listLevel;

	if(pMMatrix == NULL || nRows < 1L  || nColumns < 1L)
		return IMINER_BAD_INPUT_ARGS;

	/* "base class" : double matrix */
	nStatus = IMinerDoubleMatrix_create(pMMatrix, nRows, nColumns, pdValue, ppszRowName, ppszColumnName);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;

	/* level info from DataSet as a list object: an additional member to its base class*/
	nStatus = IMinerList_create(&listLevel, nColumns, IMINER_MODE_LIST);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;
	nStatus = IMinerList_add(pMMatrix, (const IMinerObject*)&listLevel, "level");

	/* set the class ID */
	pMMatrix->m_nMode = IMINER_MODE_MODELMATRIX;
	return nStatus;
}

/* alloc memory for the object and copy content from the source*/
long IMCSRC_STDCALL IMinerModelMatrix_createFromDataSet(IMinerObject* pMMatrix,  /* out: data object */
 const IMinerObject* pDataSet /* in: source */
)
{
	/* treatment contrasts only */
	long nStatus = IMINER_SUCCESS;
	long j, nRows, nCols;
	IMinerObject* pFactor;
	IMinerVector vszLevel;
	IMinerObject listLevel;

	nRows = IMINER_DATASET_NROWS(pDataSet);
	nCols = IMINER_DATASET_NCOLUMNS(pDataSet);

	/* "base class method": create the matrix with initial value of zero elements */
	nStatus = IMinerDoubleMatrix_createFromDataSet(pMMatrix, pDataSet);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;
	/* an additional member to its base class : level info from DataSet as a list object */
	nStatus = IMinerList_create(&listLevel, 0L, IMINER_MODE_LIST);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;
	nStatus = IMinerList_add(pMMatrix, (const IMinerObject*)&listLevel, "level");
	if(nStatus != IMINER_SUCCESS)
		return nStatus;

	/* set factor levels */
	nStatus = IMinerVector_create(&vszLevel, 0L, IMINER_MODE_STRING, NULL);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;
	for(j=0L; j< nCols; ++j)
	{
		if(IMINER_DATASET_IS_FACTOR(pDataSet, j))
		{
			pFactor = IMINER_DATASET_FACTOR_PTR(pDataSet, j);
			nStatus = IMinerList_add(IMINER_MODELMATRIX_LEVEL_PTR(pMMatrix), IMINER_FACTOR_LEVELS_PTR(pFactor), IMINER_STRING_VALUE(IMINER_DATASET_COLUMNNAME_PTR(pDataSet), j));
			if(nStatus != IMINER_SUCCESS)
				return nStatus;
		}
		else if(IMINER_DATASET_IS_VECTOR(pDataSet, j))
		{
			nStatus = IMinerList_add(IMINER_MODELMATRIX_LEVEL_PTR(pMMatrix), &vszLevel, IMINER_STRING_VALUE(IMINER_DATASET_COLUMNNAME_PTR(pDataSet), j));
			if(nStatus != IMINER_SUCCESS)
				return nStatus;
		}
	}

	nStatus = IMinerVector_destroy(&vszLevel);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;

	/* set the class ID */
	pMMatrix->m_nMode = IMINER_MODE_MODELMATRIX;

	return nStatus;
}

/* print to standard out */
long IMCSRC_STDCALL IMinerModelMatrix_print(const IMinerObject* pMMatrix )
{
	long nStatus=IMINER_SUCCESS;

	/* printf("Begin IMinerModelMatrix_print ...\n"); */
	if(!IMinerModelMatrix_isValid(pMMatrix))
	{
		IMiner_error("Invalid pdMatrix\n");
		return IMINER_BAD_INPUT_ARGS;
	}

	nStatus = IMinerDoubleMatrix_print(pMMatrix);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;

	printf("\nFactor levels:\n");
	nStatus = IMinerList_print(IMINER_MODELMATRIX_LEVEL_PTR(pMMatrix));

	/* printf("End IMinerModelMatrix_print\n"); */
	return IMINER_SUCCESS;
}

/********************************* DataSet : List ******************************************/

/* returns 1L if this is a valid object */
long IMCSRC_STDCALL IMinerDataSet_isValid(const IMinerObject* pDataSet)
{
	/* purposely ignor mode of this class so that its "derived class" can use this function as a method. */
	IMinerObject* pElement;

	if(pDataSet == NULL
	|| !IMinerList_isValid(pDataSet) /* kind of list ? */
	|| IMINER_OBJECT_LENGTH(pDataSet) < 4L) /* length at least 4?: data, dim, row names, col names */
		return 0L;
	/* member 0: data */
	pElement = IMINER_DATASET_DATA_PTR(pDataSet);
	if(!IMinerList_isValid(pElement))
		return 0L;
	/* member 1: dim */
	pElement = IMINER_DATASET_DIM_PTR(pDataSet);
	if(!IMinerVector_isLong(pElement) && pElement->m_nLen == 2L)
		return 0L;
	/* member 2: row name */
	pElement = IMINER_DATASET_ROWNAME_PTR(pDataSet);
	if(!IMinerVector_isString(pElement))
		return 0L;
	/* member 3: column name */
	pElement = IMINER_DATASET_COLUMNNAME_PTR(pDataSet);
	if(!IMinerVector_isString(pElement))
		return 0L;
	/* special member : member names mut also be valid and has the same length as this object*/
	pElement = IMINER_LIST_NAMES_PTR(pDataSet);
	if(!IMinerVector_isString(pElement) && pElement->m_nLen == pDataSet->m_nLen)
		return 0L;

	return 1L;
}

long IMCSRC_STDCALL IMinerColumn_isValid(const IMinerObject* pColumn)
{
	if(IMINER_IS_VECTOR(pColumn))
		return IMinerVector_isValid(pColumn);
	else if(IMINER_IS_FACTOR(pColumn))
		return IMinerFactor_isValid(pColumn);
	return 0L; /* not a vector nor factor return FALSE */
}

/* alloc memory for object of class DataSet. */
long IMCSRC_STDCALL IMinerDataSet_create(IMinerObject* pDataSet,  /* out: data object */
 long nRows,                /* in:  number of rows */
 long nColumns,             /* in:  number of cols */
 const long* pnColumnsModes /* in:  types of columns desired */
)
{
	long nStatus = IMINER_SUCCESS;
	long j;
	long pnDim[2];
	char* ppszMemberNames[] = {"data", "dim", "rowNames", "columnNames"};
	IMinerObject* plistColumn=NULL;

	if(pDataSet==NULL || nRows < 1L || nColumns< 1L)
		return IMINER_BAD_INPUT_ARGS;

	/* DataSet object always has 4 members: the 5th one is for member names */
	nStatus = IMinerList_create(pDataSet, 4, IMINER_MODE_DATASET);

	/* member 0: data */
	plistColumn = IMINER_DATASET_DATA_PTR(pDataSet);
	nStatus = IMinerList_create(plistColumn, nColumns, IMINER_MODE_LIST);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;
	if(pnColumnsModes != NULL) /* known column modes ? */
	{
		for(j=0; j<nColumns; ++j)
		{
			long nMode = pnColumnsModes[j];
			IMinerObject* pvColumn = IMINER_LIST_PTR(plistColumn)[j];

			IMINER_OBJECT_MODE(pvColumn) = nMode;
			if(IMINER_IS_VECTOR_MODE(nMode))
				nStatus = IMinerVector_create(pvColumn, nRows, nMode, NULL);
			else if(IMINER_IS_FACTOR_MODE(nMode))
				nStatus = IMinerFactor_create(pvColumn, nRows, NULL, 0L, NULL);
			else
				nStatus = IMINER_FAIL;

			if(nStatus != IMINER_SUCCESS)
				return nStatus;
		}
	}
	/* member 1: dim */
	pnDim[0] = nRows;
	pnDim[1] = nColumns;
	nStatus = IMinerVector_create(IMINER_DATASET_DIM_PTR(pDataSet), 2L, IMINER_MODE_LONG, pnDim);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;
	/* member 2: row names */
	nStatus = IMinerVector_create(IMINER_DATASET_ROWNAME_PTR(pDataSet), 0L, IMINER_MODE_STRING, NULL);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;
	/* member 3: column names */
	nStatus = IMinerVector_create(IMINER_DATASET_COLUMNNAME_PTR(pDataSet), 0L, IMINER_MODE_STRING, NULL);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;
	/*member 4 : member names */
	nStatus = IMinerList_setNamesFromStrings(pDataSet, (const char**)ppszMemberNames);

	/* set the class ID */
	pDataSet->m_nMode = IMINER_MODE_DATASET;
	return IMINER_SUCCESS;
}

/* get modes of the data columns */
long IMCSRC_STDCALL IMinerDataSet_getColumnMode(long* pnMode, /* out: mode(type) of the data */
 const IMinerObject* pDataSet, /* in: data object */
 long j /* in: j-th column */
)
{
	long nStatus = IMINER_SUCCESS;
	IMinerObject* pColumn= NULL;
	if(!IMinerDataSet_isValid(pDataSet) || IMINER_DATASET_NROWS(pDataSet) < 1L || IMINER_DATASET_NCOLUMNS(pDataSet)< 1L )
		return IMINER_BAD_INPUT_ARGS;
	if(j<0L || j >= IMINER_DATASET_NCOLUMNS(pDataSet))
		return IMINER_BAD_INPUT_ARGS;

	pColumn = IMINER_DATASET_COLUMN_PTR(pDataSet, j);
	if(IMinerVector_isValid(pColumn) || IMinerFactor_isValid(pColumn))
		*pnMode = IMINER_OBJECT_MODE(pColumn);
	else
		nStatus = IMINER_BAD_INPUT_ARGS;
	return nStatus;
}

/* set column of IMnerDATASET with data. */
long IMCSRC_STDCALL IMinerDataSet_setColumnAt(IMinerObject* pDataSet, /* in/out: data object */
 long j,      /* in:  jth column to set the data */
 const IMinerObject* pcolSource   /* in: the column, length of its length must be long enough */
)
{
	long nStatus = IMINER_SUCCESS;
	IMinerObject* pColumn;

	if(!IMinerDataSet_isValid(pDataSet) || IMINER_DATASET_NCOLUMNS(pDataSet) <= j || !IMinerColumn_isValid(pcolSource))
		return IMINER_BAD_INPUT_ARGS;

	if(IMINER_IS_VECTOR(pcolSource) || IMINER_IS_FACTOR(pcolSource))
	{
		pColumn = IMINER_DATASET_COLUMN_PTR(pDataSet, j);
		if(IMinerObject_isValid(pColumn))
			IMinerObject_destroy(pColumn);
		nStatus = IMinerObject_clone(pColumn, pcolSource);
	}
	else
		return IMINER_FAIL;
	return nStatus;
}

/* get column names as a new IMinerVector.  Make sure you call IMinerVector_destroy() to avoid memory leak*/
long IMCSRC_STDCALL IMinerDataSet_getColumnNames(IMinerObject* pvszColumnName,   /* out: string vector of the column */
 const IMinerObject* pDataSet /* in: data object */
)
{
	long nStatus = IMINER_SUCCESS;
	if(!IMinerDataSet_isValid(pDataSet) || !IMinerVector_isValid(pvszColumnName) )
		return IMINER_BAD_INPUT_ARGS;
	nStatus = IMinerVector_create(pvszColumnName, IMINER_DATASET_NCOLUMNS(pDataSet), IMINER_MODE_STRING, (const char**) IMINER_STRING_PTR(IMINER_DATASET_COLUMNNAME_PTR(pDataSet)));
	return nStatus;
}
/* set column names*/
long IMCSRC_STDCALL IMinerDataSet_setColumnNames(IMinerObject* pDataSet, /* out: data object */
 const IMinerObject* pvszColumnName   /* in: string vector of the column */
)
{
	long nStatus = IMINER_SUCCESS;
	if(!IMinerDataSet_isValid(pDataSet) || !IMinerVector_isValid(pvszColumnName) )
		return IMINER_BAD_INPUT_ARGS;
	nStatus = IMinerVector_destroy(IMINER_DATASET_COLUMNNAME_PTR(pDataSet));
	if(nStatus != IMINER_SUCCESS)
		return nStatus;
	nStatus = IMinerVector_create(IMINER_DATASET_COLUMNNAME_PTR(pDataSet), IMINER_DATASET_NCOLUMNS(pDataSet), IMINER_MODE_STRING, (const char**) IMINER_STRING_PTR(pvszColumnName));
	return nStatus;
}

/* set column names*/
long IMCSRC_STDCALL IMinerDataSet_setColumnNamesFromStrings(IMinerObject* pDataSet, /* out: data object */
 const char** ppszColumnName   /* in: string vector of the column */
)
{
	long nStatus = IMINER_SUCCESS;
	if(!IMinerDataSet_isValid(pDataSet) || !ppszColumnName )
		return IMINER_BAD_INPUT_ARGS;
	nStatus = IMinerVector_destroy(IMINER_DATASET_COLUMNNAME_PTR(pDataSet));
	if(nStatus != IMINER_SUCCESS)
		return nStatus;
	nStatus = IMinerVector_create(IMINER_DATASET_COLUMNNAME_PTR(pDataSet), IMINER_DATASET_NCOLUMNS(pDataSet), IMINER_MODE_STRING, ppszColumnName);
	return nStatus;
}

/* get row names */
long IMCSRC_STDCALL IMinerDataSet_getRowNames(IMinerObject* pvszRowName,  /* out: names */
 const IMinerObject* pDataSet /* in: data object */
)
{
	long nStatus = IMINER_SUCCESS;
	if(!IMinerDataSet_isValid(pDataSet) || !IMinerVector_isValid(pvszRowName) )
		return IMINER_BAD_INPUT_ARGS;
	nStatus = IMinerVector_create(pvszRowName, IMINER_DATASET_NROWS(pDataSet), IMINER_MODE_STRING, (const char**) IMINER_STRING_PTR(IMINER_DATASET_ROWNAME_PTR(pDataSet)));
	return nStatus;
}

/* set row names*/
long IMCSRC_STDCALL IMinerDataSet_setRowNames(IMinerObject* pDataSet, /* out: data object */
 const IMinerObject* pvszRowName   /* in: string vector of the column */
)
{
	long nStatus = IMINER_SUCCESS;
	if(!IMinerDataSet_isValid(pDataSet) || !IMinerVector_isValid(pvszRowName) )
		return IMINER_BAD_INPUT_ARGS;
	nStatus = IMinerVector_destroy(IMINER_DATASET_ROWNAME_PTR(pDataSet));
	if(nStatus != IMINER_SUCCESS)
		return nStatus;
	nStatus = IMinerVector_create(IMINER_DATASET_COLUMNNAME_PTR(pDataSet), IMINER_DATASET_NROWS(pDataSet), IMINER_MODE_STRING, (const char**) IMINER_STRING_PTR(pvszRowName));
	return nStatus;
}

long IMCSRC_STDCALL IMinerDataSet_setRowNamesFromStrings(IMinerObject* pDataSet, /* out: data object */
 const char** ppszRowName   /* in: string vector of the column */
)
{
	long nStatus = IMINER_SUCCESS;
	if(!IMinerDataSet_isValid(pDataSet) || !ppszRowName )
		return IMINER_BAD_INPUT_ARGS;
	nStatus = IMinerVector_destroy(IMINER_DATASET_ROWNAME_PTR(pDataSet));
	if(nStatus != IMINER_SUCCESS)
		return nStatus;
	nStatus = IMinerVector_create(IMINER_DATASET_ROWNAME_PTR(pDataSet), IMINER_DATASET_NROWS(pDataSet), IMINER_MODE_STRING, ppszRowName);
	return nStatus;
}

/* print to standard out */
long IMCSRC_STDCALL IMinerDataSet_print(const IMinerObject* pDataSet	)
{
	long i, j, nRows, nColumns;
	char szdFormat[32];
	char szMsg[128];
	/* printf("Begin IMinerDataSet_print ...\n"); */
	if(!IMinerDataSet_isValid(pDataSet))
	{
		IMiner_error("Invalid DataSet object\n");
		return IMINER_BAD_INPUT_ARGS;
	}

	nRows = IMINER_DATASET_NROWS(pDataSet);
	nColumns = IMINER_DATASET_NCOLUMNS(pDataSet);
	/* print to stdout */

	sprintf(szdFormat, "%d", IMINER_DATASET_NROWS(pDataSet));
	sprintf(szdFormat, "[%c%dd,] ", '%', strlen(szdFormat));

	/* print the column names */
	if(IMINER_OBJECT_LENGTH(IMINER_DATASET_COLUMNNAME_PTR(pDataSet)))
	{
		sprintf(szMsg, szdFormat, 1);
		for(i=0L; i < (long)strlen(szMsg); ++i)
			printf(" ");
		for(j=0L; j< IMINER_DATASET_NCOLUMNS(pDataSet)-1L; ++j)
			printf("%s, ", IMINER_STRING_VALUE(IMINER_DATASET_COLUMNNAME_PTR(pDataSet), j));
		printf("%s\n", IMINER_STRING_VALUE(IMINER_DATASET_COLUMNNAME_PTR(pDataSet), j));
	}

	for(i=0L; i< nRows; ++i)
	{
		printf(szdFormat,i+1);
		for(j=0L; j< nColumns; ++j)
		{
			/* TODO: print row names */
			if(IMINER_DATASET_IS_FACTOR(pDataSet, j))
			{
				IMinerFactor_printElement(IMINER_DATASET_FACTOR_PTR(pDataSet, j), i);
			}
			else if(IMINER_DATASET_IS_VECTOR(pDataSet, j))
			{
				IMinerVector_printElement(IMINER_DATASET_VECTOR_PTR(pDataSet, j), i);
			}
			else
				printf("unable to print cell: ");

			if(	j< IMINER_DATASET_NCOLUMNS(pDataSet)-1L)
				printf(", ");
		}
		printf("\n");
	}
	printf("\n");
	/* printf("End IMinerDataSet_print\n"); */
	return IMINER_SUCCESS;
}


/* open, create from from a formatted text file and close the file.
*   Delimiter is ','.
*/
long IMCSRC_STDCALL IMinerDataSet_createFromTextFile(IMinerObject* pDataSet, /* out: data object */
 const char* pszFile                      /* in: file name */
)
{
	long nStatus = IMINER_SUCCESS;
	FILE* pFile;
	long nRows, nColumns, nLen, i, j, jLine, nLenOfToken;
	long pnColumnMode[MAX_LINE];
	char szLine[MAX_LINE]; /* limited line for reading */
	char szToken[LINE_LEN];
	char** ppszColumnName;
	IMinerObject* pVector=NULL;

	if(!pszFile|| !*pszFile)
		return IMINER_BAD_INPUT_ARGS;

	pFile	= fopen(pszFile, "r");

	/* first line is assumed to be the column names */
	if( fgets( szLine, MAX_LINE, pFile ) == NULL)
		return IMINER_BAD_INPUT_ARGS;
	if((strlen(szLine) == (MAX_LINE-1)) && szLine[MAX_LINE-1] !='\n')
		return IMINER_BAD_INPUT_ARGS; /* MAX_LINE is not long enough */

	/* first pass: determine number of columns and column types */
	nColumns = 0L;
	if( fgets( szLine, MAX_LINE, pFile ) == NULL)
		return IMINER_BAD_INPUT_ARGS;
	nLen = strlen(szLine);
	if((nLen == (MAX_LINE-1)) && szLine[MAX_LINE-1] !='\n')
		return IMINER_BAD_INPUT_ARGS; //MAX_LINE is not long enough

	/* assume dense data points (i.e. no NA as empty cell) */
	pnColumnMode[0] = ('0' <= szLine[0] && szLine[0] <= '9')? IMINER_MODE_DOUBLE : IMINER_MODE_STRING;/* ASCII for now */
	for(j=0L; j<nLen; ++j)
	{
		if(szLine[j] == ',' || szLine[j]=='\n')
		{
			++nColumns;
			if(j+1L < nLen-1L)
				pnColumnMode[nColumns] = ('0' <= szLine[j+1L] && szLine[j+1L] <= '9')? IMINER_MODE_DOUBLE : IMINER_MODE_STRING;/* ASCII for now */
		}
	}

	/* still First pass: determine number of rows */
	nRows = 1L; /* row 1 was read above to determine columns */
	while( fgets( szLine, MAX_LINE, pFile ) != NULL)
	{
		if((strlen(szLine) == (MAX_LINE-1)) && szLine[MAX_LINE-1] !='\n')
			return IMINER_BAD_INPUT_ARGS; /* MAX_LINE is not long enough */
		++nRows;
	}

	/* second pass: read into vectors */

	nStatus = IMinerDataSet_create(pDataSet, nRows, nColumns, pnColumnMode);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;

	fseek(pFile, 0L, SEEK_SET);
	if( fgets( szLine, MAX_LINE, pFile ) == NULL) //column names
	{
		IMinerObject_destroy(pDataSet);
		return IMINER_BAD_INPUT_ARGS;
	}

	/* put in column names */
	i=0L;
	nLen = strlen(szLine);
	while(i< nLen && szLine[i] == ' ') /* skip pass all spaces */
		++i;

	ppszColumnName = (char**)malloc(nColumns*sizeof(char*));
	for(j=0L; j<nColumns; ++j)
	{
		ppszColumnName[j] = &szLine[i];
		while(i< nLen && szLine[i] != ',' && szLine[i] != '\n') /* skip pass ',' */
			++i;
		szLine[i++] = 0x0; /* replace ',' by the end of string */
		if(j < nColumns-1L)
		{
			while(i< nLen && szLine[i] == ' ') /* skip pass all spaces */
				++i;
			if(i>=nLen)
			{
				IMinerObject_destroy(pDataSet);
				free(ppszColumnName);
				return IMINER_BAD_INPUT_ARGS;
			}
		}
	}

	nStatus = IMinerDataSet_setColumnNamesFromStrings(pDataSet, (const char**) ppszColumnName);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;
	free(ppszColumnName);

	/* put in data */
	for(i=0L; i<nRows; ++i)
	{
		if( fgets( szLine, MAX_LINE, pFile ) == NULL)
		{
			IMinerObject_destroy(pDataSet);
			return IMINER_BAD_INPUT_ARGS;
		}

		nLen = strlen(szLine);
		jLine=0L;
		for(j=0L; j<nColumns; ++j)
		{
			nLenOfToken = 0L;
			while(jLine<nLen && szLine[jLine]!=',' && szLine[jLine] != '\n')
			{
				++jLine;
				++nLenOfToken;
			}
			strncpy(szToken, &szLine[jLine-nLenOfToken], nLenOfToken);
			++jLine; /* skip pass the deliminter */
			szToken[nLenOfToken] = 0x0;
			pVector = IMINER_DATASET_VECTOR_PTR(pDataSet, j);
			switch(IMINER_OBJECT_MODE(pVector))
			{
				case IMINER_MODE_LONG:
					IMINER_LONG_PTR(pVector)[i] = atoi(szToken);
					break;
				case IMINER_MODE_DOUBLE:
					IMINER_DOUBLE_PTR(pVector)[i] = atof(szToken);
					break;
				case IMINER_MODE_STRING:
					strcpy(IMINER_STRING_PTR(pVector)[i], szToken);
					break;
				case IMINER_MODE_BOOLEAN:
				case IMINER_MODE_FLOAT:
				default:
					IMinerObject_destroy(pDataSet);
					return IMINER_BAD_INPUT_ARGS;
					break;
			}
		}
	}
	fclose(pFile);

	/* change all STRING to factor */
	for(j=0L; j<nColumns; ++j)
	{
		pVector = IMINER_DATASET_VECTOR_PTR(pDataSet, j);
		if(IMINER_OBJECT_MODE(pVector) == IMINER_MODE_STRING)
		{
			IMinerFactor factor;
			IMinerFactor_createFromStrings(&factor, IMINER_OBJECT_LENGTH(pVector), (const char**)IMINER_STRING_PTR(pVector));
			IMinerDataSet_setColumnAt(pDataSet, j, &factor);
		}
	}

	return IMINER_SUCCESS;
}

#ifdef USE_HASH_LOOKUP
/*  open, create from from a formatted text file with the aid of a
 *  valid model.
 *  Delimiter is ','.
 */
long IMCSRC_STDCALL IMinerDataSet_createWithModel(
							IMinerObject* pModel, /* out: data object */
							IMinerObject* pDataSet, /* out: data object */
							const char* dataFile     /* in: file name */
							)
{
	long nStatus=IMINER_SUCCESS, nRows, numInputCols, numModelCols, nLen, i, j;
	long *outputColumnNum=0, *inputColumnMode=0, *modelColumnMode=0;

	char **ppszColumnName=0;
	char szLine[MAX_LINE];

	IMinerHashTable *modelColumns=0, **columnLevels=0;
	IMinerObject* pVector=NULL;

	FILE* pFile;

	if(!dataFile|| !*dataFile)
		return IMINER_BAD_INPUT_ARGS;

	pFile	= fopen(dataFile, "r");
	if(!pFile) return IMINER_BAD_INPUT_ARGS;

	/* get desired dimensions */
	numModelCols = IMINER_MD_NUM_COLUMNS(pModel);
	modelColumnMode = (long*)malloc(numModelCols * sizeof(long));
	modelColumns = IMinerHashTable_createHash();
	for (i=0; i<numModelCols; i++)
	{
		char *colName = IMINER_MD_GET_COLUMN_NAME(pModel, i);
		/* create fast accessible list of column index accessed by column name */
		IMinerHashTable_put(modelColumns, colName, i);
	}

	/* first line is assumed to be the column names */
	if( nStatus==IMINER_SUCCESS && fgets( szLine, MAX_LINE, pFile ) == NULL)
		nStatus = IMINER_BAD_INPUT_ARGS;
	if(nStatus==IMINER_SUCCESS && (strlen(szLine) == (MAX_LINE-1)) && szLine[MAX_LINE-1] !='\n')
		nStatus =  IMINER_BAD_INPUT_ARGS; /* MAX_LINE is not long enough */

	/* first pass: determine number of columns */
	numInputCols = 0L;
	if( nStatus==IMINER_SUCCESS && fgets( szLine, MAX_LINE, pFile ) == NULL)
		nStatus = IMINER_BAD_INPUT_ARGS;
	if (nStatus==IMINER_SUCCESS)
	{
		nLen = strlen(szLine);
		if((nLen == (MAX_LINE-1)) && szLine[MAX_LINE-1] !='\n')
			nStatus = IMINER_BAD_INPUT_ARGS; //MAX_LINE is not long enough
	}

	if (nStatus==IMINER_SUCCESS)
	{
		/* remember, we only care about columns that are found in the model metadata */
		/* because of that, we don't need to worry about the data type, since we*/
		/* have it in the meta data.  default all columns to unknown type for now. */
		/* later, we'll specify the type of the columns we know and ditch the rest */
		for(j=0L; j<nLen; ++j)
		{
			if(szLine[j] == ',' || szLine[j]=='\n')
			{
				numInputCols++;
			}
		}
		inputColumnMode = (long*)malloc(numInputCols * sizeof(long));

		/* still First pass: determine number of rows */
		nRows = 1L;
		while( nStatus==IMINER_SUCCESS && fgets( szLine, MAX_LINE, pFile ) != NULL)
		{
			if((strlen(szLine) == (MAX_LINE-1)) && szLine[MAX_LINE-1] !='\n')
				nStatus = IMINER_BAD_INPUT_ARGS; /* MAX_LINE is not long enough */
			++nRows;
		}

		/* second pass: reset the file location */
		fseek(pFile, 0L, SEEK_SET);
		if( fgets( szLine, MAX_LINE, pFile ) == NULL)
			nStatus = IMINER_BAD_INPUT_ARGS;

		/* get column names in input file and denote which columns are found in the */
		/* meta data by specifying their type */
		nLen = strlen(szLine);
		outputColumnNum = (long*)malloc(numInputCols * sizeof(long));
		for(i=0L, j=0L; j<numInputCols; ++j)
		{
			char *colName=0;
			int index = -1;
			/* skip pass all spaces */
			while(i<nLen && szLine[i] == ' ') ++i;
			/* denote start of column name */
			colName = &szLine[i];
			/* skip passed ',' */
			while(i< nLen && szLine[i] != ',' && szLine[i] != '\n') ++i;
			/* makr end of name by replacing ',' with \0 */
			szLine[i++] = 0x0;

			index = IMinerHashTable_get(modelColumns, colName);
			outputColumnNum[j] = index;
			if (index==-1 && colName[0] == '\"' && szLine[i-2] == '\"') {
				szLine[i-2] = 0x0;
				index = IMinerHashTable_get(modelColumns, &colName[1]);
				outputColumnNum[j] = index;
			}
			if (index != -1)
			{
				/* this is a column found in the meta data and it is a factor */
				/* iff there are levels.  otherwise, it is a double */
				modelColumnMode[index] =
					(IMINER_MD_COLUMN_LEVEL_COUNT(pModel, index)) ? IMINER_MODE_FACTOR : IMINER_MODE_DOUBLE;
				inputColumnMode[j] = modelColumnMode[index];
			} else
			{
				/* watch for column names that are defined inside quotes */
				inputColumnMode[j] = IMINER_MODE_UNKNOWN;
			}
		}
	}

	/* we are now done with the model column name hash */
	IMinerHashTable_freeHash(modelColumns);

	if (nStatus==IMINER_SUCCESS)
	{
		/* create dataset based on model metadata info */
		nStatus = IMinerDataSet_create(pDataSet, nRows, numModelCols, modelColumnMode);

		/* set dataset column names */
		ppszColumnName = (char**)malloc(numModelCols*sizeof(char*));
		for (i=0; i<numModelCols; i++) ppszColumnName[i] = IMINER_MD_GET_COLUMN_NAME(pModel, i);
		nStatus = IMinerDataSet_setColumnNamesFromStrings(pDataSet, (const char**) ppszColumnName);
		free(ppszColumnName);

		/* get level info */
		columnLevels = (IMinerHashTable**)malloc(numModelCols * sizeof(IMinerHashTable*));
		for (i=0; i<numModelCols; i++) {
			/* create fast accessible list of column level index accessed by column level name */
			int numLevs = IMINER_MD_COLUMN_LEVEL_COUNT(pModel, i);
			IMinerObject *tmp=0;
			columnLevels[i] = 0;

			if (numLevs) {
				/* create hash for values versus index */
				columnLevels[i] = IMinerHashTable_createHash();
				/* copy levels from metadata */
				tmp = IMINER_DATASET_FACTOR_PTR(pDataSet, i);
				tmp = IMINER_FACTOR_LEVELS_PTR(tmp);
				tmp = IMINER_MD_COLUMN_LEVEL_NAMES(pModel, i);
				IMINER_FACTOR_LEVELS_PTR(IMINER_DATASET_FACTOR_PTR(pDataSet, i)) =
					IMINER_MD_COLUMN_LEVEL_NAMES(pModel, i);
			}
			for (j=0; j<numLevs; j++) {
				char *modelLevelName = IMINER_STRING_VALUE(IMINER_MD_COLUMN_LEVEL_NAMES(pModel, i), j);
				/* create mapping from level name to  level index */
				IMinerHashTable_put(columnLevels[i], modelLevelName, j);
			}
		}


		/* second pass: read into vectors */
		/* put in data */
		for(i=0L; nStatus==IMINER_SUCCESS && i<nRows; ++i)
		{
			char szToken[LINE_LEN];
			int jLine=0L;

			if( fgets( szLine, MAX_LINE, pFile ) == NULL)
			{
				nStatus = IMINER_BAD_INPUT_ARGS;
				break;
			}

			nLen = strlen(szLine);
			for(j=0L; nStatus==IMINER_SUCCESS && j<numInputCols; ++j)
			{
				int nLenOfToken = 0L;
				while(jLine<nLen && szLine[jLine]!=',' && szLine[jLine] != '\n')
				{
					++jLine;
					++nLenOfToken;
				}
				strncpy(szToken, &szLine[jLine-nLenOfToken], nLenOfToken);
				++jLine; /* skip pass the deliminter */
				szToken[nLenOfToken] = 0x0;

				/* do we care about this column ? */
				if (inputColumnMode[j] != IMINER_MODE_UNKNOWN)
				{
					int levelNum = -1;

					pVector = IMINER_DATASET_VECTOR_PTR(pDataSet, outputColumnNum[j]);

					switch(IMINER_OBJECT_MODE(pVector))
					{
						case IMINER_MODE_DOUBLE:
							IMINER_DOUBLE_PTR(pVector)[i] = atof(szToken);
							break;
						case IMINER_MODE_FACTOR:
							levelNum = IMinerHashTable_get(columnLevels[outputColumnNum[j]], szToken);
							if (levelNum == -1) {
								nStatus = IMINER_BAD_INPUT_ARGS;
								break;
							}
							IMINER_FACTOR_DATA_VALUE(pVector, i) = levelNum;

							break;
						default:
							nStatus = IMINER_BAD_INPUT_ARGS;
							break;
					}
				}
			}
		}
	}

	if (nStatus != IMINER_SUCCESS) IMinerObject_destroy(pDataSet);
	if (pFile) fclose(pFile);

	for (i=0; columnLevels && i<numModelCols; i++) {
		if (columnLevels[i]) IMinerHashTable_freeHash(columnLevels[i]);
	}
	if (columnLevels) free(columnLevels);

	if (outputColumnNum) free(outputColumnNum);
	if (inputColumnMode) free(inputColumnMode);
	if (modelColumnMode) free(modelColumnMode);

	return nStatus;
}
#endif /*USE_HASH_LOOKUP*/

double IMCSRC_STDCALL IMinerDataSet_getNonStringValue(IMinerDataSet *ds, int col, int row) {
	IMinerObject *column = IMINER_DATASET_COLUMN_PTR(ds, col);
	if (IMINER_IS_FACTOR(column)) {
		return IMINER_FACTOR_DATA_VALUE(column, row);
	}
	if (IMINER_OBJECT_MODE(column)==IMINER_MODE_DOUBLE) return IMINER_DOUBLE_VALUE(column, row);
	if (IMINER_OBJECT_MODE(column)==IMINER_MODE_INT) return IMINER_INT_VALUE(column, row);
	if (IMINER_OBJECT_MODE(column)==IMINER_MODE_FLOAT) return IMINER_FLOAT_VALUE(column, row);

	return 0;
}

long IMCSRC_STDCALL IMinerDataSet_createFromInput(
 IMinerObject *rectDataInput,	/* out: dataset object */
 long numberOfRows,				/* in:  number of rows */
 long numberOfColumns,			/* in:  number of columns */
 long *columnTypes,				/* in:  column types; must either be
										IMINER_MODE_DOUBLE or IMINER_MODE_STRING */
 char **columnNames,			/* in:  column names */
 void **data					/* in:  data; each column must be either
										(double*) or (char**) */
)
{
	long column=0, nStatus=IMINER_SUCCESS;
	IMinerObject columnData;

	/*************************/
	/* CREATE DATASET OBJECT */
	/*************************/
	nStatus = IMinerDataSet_create(rectDataInput, numberOfRows, numberOfColumns, columnTypes);
	if(nStatus != IMINER_SUCCESS) return nStatus;

	/************************/
	/* SET THE COLUMN NAMES */
	/************************/
	nStatus = IMinerDataSet_setColumnNamesFromStrings(rectDataInput, (const char**)columnNames);
	if(nStatus != IMINER_SUCCESS) return nStatus;

	for (column=0; column<numberOfColumns; column++) {
		if (columnTypes[column] == IMINER_MODE_STRING) {
			nStatus = IMinerFactor_createFromStrings(&columnData, numberOfRows, (const char**)data[column]);
			if(nStatus != IMINER_SUCCESS) return nStatus;
		} else {
			nStatus = IMinerVector_create(&columnData, numberOfRows, IMINER_MODE_DOUBLE, data[column]);
			if(nStatus != IMINER_SUCCESS) return nStatus;
		}
		nStatus = IMinerDataSet_setColumnAt(rectDataInput, column, &columnData);
		if(nStatus != IMINER_SUCCESS) return nStatus;

		nStatus = IMinerObject_destroy(&columnData);
		if(nStatus != IMINER_SUCCESS) return nStatus;
	}

	return nStatus;
}



/********************************* MetaData *********************************************/
long IMCSRC_STDCALL IMinerMetaData_createFromDataSet(
 IMinerObject *createMD,		/* out: metadata */
 const IMinerDataSet * pInput			/* in: input dataset */
)
{
	long numCols=0, numRows=0, *numLevels, i, j, maxNumLevels=0, nlevs, retval;
	char **columnNames, ***columnLevelsNames, **levels, *emptyString="";

	if (!IMinerDataSet_isValid(pInput)) return 0L;

	columnNames = (char**)IMINER_STRING_PTR(IMINER_DATASET_COLUMNNAME_PTR(pInput));
	numCols = IMINER_DATASET_NCOLUMNS(pInput);
	numRows = IMINER_DATASET_NROWS(pInput);

	numLevels = (long*)malloc(numCols*sizeof(long));
	columnLevelsNames = (char***)malloc(numCols*sizeof(char**));

	for (i=0; i<numCols; i++) {
		if (IMINER_DATASET_IS_FACTOR(pInput, i)) {
			nlevs = IMINER_FACTOR_NLEVELS(IMINER_DATASET_FACTOR_PTR(pInput, i));
			if (nlevs>maxNumLevels) maxNumLevels=nlevs;
		}
	}

	for (i=0; i<numCols; i++) {
		columnLevelsNames[i] = (char**)malloc(maxNumLevels*sizeof(char*));
		nlevs = 0;

		if (IMINER_DATASET_IS_FACTOR(pInput, i)) {
			levels = (char**)IMINER_STRING_PTR(IMINER_FACTOR_LEVELS_PTR(IMINER_DATASET_FACTOR_PTR(pInput, i)));
			nlevs = IMINER_FACTOR_NLEVELS(IMINER_DATASET_FACTOR_PTR(pInput, i));
		}
		for (j=0; j<maxNumLevels; j++) {
			if (j>=nlevs) {
				columnLevelsNames[i][j] = emptyString;
			} else {
				columnLevelsNames[i][j] = levels[j];
			}
		}
		numLevels[i] = nlevs;
	}


	retval = IMinerMetaData_createInputDescription(createMD, numCols, columnNames, numLevels, maxNumLevels, columnLevelsNames);

	for (i=0; i<numCols; i++) free(columnLevelsNames[i]);
	free(columnLevelsNames);
	free(numLevels);

	return retval;
}

long IMCSRC_STDCALL IMinerMetaData_createInputDescription(
 IMinerObject* metaData,	    /* out: data object */
 long nColumns,					/* in:  number of columns */
 char **colNames,				/* in:  column names of length (nColumns) */
 long *numLevels,				/* in:  number of levels of length (nColumns) */
 long nMaxLevels,				/* in:  max number of levels */
 char ***levelNames				/* in:  level names of length (nColumns*nMaxLevels) */
)
{
	long i=0, nStatus, *splitCols = (long*)malloc(nColumns*sizeof(long));
	for (i=0; i<nColumns; i++) splitCols[i] = 0;

	nStatus = IMinerMetaData_create(metaData, nColumns, colNames, numLevels, splitCols, nMaxLevels, levelNames);

	free(splitCols);
	return nStatus;
}

long IMCSRC_STDCALL IMinerMetaData_create(
 IMinerObject* metaData,	    /* out: data object */
 long nColumns,					/* in:  number of columns */
 char **colNames,				/* in:  column names of length (nColumns) */
 long *numLevels,				/* in:  number of levels of length (nColumns) */
 long *splitCategories,			/* in:  split categorical column into multiple double columns of 0/1 */
 long nMaxLevels,				/* in:  max number of levels */
 char ***levelNames				/* in:  level names of length (nColumns*nMaxLevels) */
)
{
	long retval = IMinerMetaData_createTerms(metaData, nColumns, 0, colNames, numLevels, NULL, NULL, splitCategories, nMaxLevels, levelNames);
	return retval;
}

long IMCSRC_STDCALL IMinerMetaData_createTerms(
 IMinerObject* metaData,	    /* out: data object */
 long nColumns,					/* in:  number of columns */
 long nTerms,					/* in:  number of terms */
 char **colNames,				/* in:  column names of length (nColumns) */
 long *numLevels,				/* in:  number of levels of length (nColumns) */
 char **termNames,				/* in:  names of the terms of length (nTerms) */
 long **colUse,					/* in:  matrix of column use per term of length (nColumns * nTerms) */
 long *splitCategories,			/* in:  split categorical column into multiple double columns of 0/1 */
 long nMaxLevels,				/* in:  max number of levels */
 char ***levelNames				/* in:  level names of length (nColumns*nMaxLevels) */
)
{
	char *ppszMemberNames[]={"columnInfo", "termInfo"}, **colDescrNames, **termDescrNames;
	long nStatus=IMINER_SUCCESS, col, i, j, val;
	IMinerObject *termVector;

	if(!metaData || nColumns<1L) return IMINER_BAD_INPUT_ARGS;

	/* a ModelMetaData always has 2 elements...a column description and a term description */
	nStatus = IMinerList_create(metaData, 2, IMINER_MODE_LIST);
	if(nStatus != IMINER_SUCCESS) return nStatus;


	/* a column description section always have nColumns lists as list */
	nStatus = IMinerList_create(IMINER_MD_PTR(metaData, IMINER_MD_COLUMN_DEF_NUM),
								nColumns, IMINER_MODE_LIST);
	if(nStatus != IMINER_SUCCESS) return nStatus;
	colDescrNames = (char**)malloc(nColumns*sizeof(char*));
	/* member: nColumns */
	for (col=0; col<nColumns; col++) {
		/* a column always have IMINER_MD_SPLIT_CATEGORY_NUM+1 Vectors in it as a list*/
		nStatus = IMinerList_create(IMINER_MD_COLUMN_PTR(metaData, col),
									IMINER_MD_SPLIT_CATEGORY_NUM+1,
									IMINER_MODE_LIST);
		if(nStatus != IMINER_SUCCESS) return nStatus;

		colDescrNames[col] = (char*)colNames[col];

		nStatus = IMinerVector_create(IMINER_MD_COLUMN_LEVEL_NAMES(metaData, col),
									  numLevels[col], IMINER_MODE_STRING, levelNames[col]);
		if(nStatus != IMINER_SUCCESS) return nStatus;
		nStatus = IMinerVector_create(IMINER_MD_COLUMN_SPLIT_CATEGORIES(metaData, col),
									  1, IMINER_MODE_LONG, &splitCategories[col]);
		if(nStatus != IMINER_SUCCESS) return nStatus;
	}
	/*member 4 (special) : member names */
	nStatus = IMinerList_setNamesFromStrings(IMINER_MD_PTR(metaData, IMINER_MD_COLUMN_DEF_NUM),
											 (const char**)colDescrNames);
	if(nStatus != IMINER_SUCCESS) return nStatus;
	free(colDescrNames);


	/* a term description section always has nTerms lists as list */
	nStatus = IMinerList_create(IMINER_MD_PTR(metaData, IMINER_MD_TERM_DEF_NUM),
								nTerms, IMINER_MODE_LIST);
	if(nStatus != IMINER_SUCCESS) return nStatus;
	termDescrNames = (char**)malloc(nTerms*sizeof(char*));
	/* member: nColumns */
	for (col=0; col<nTerms; col++) {
		/* a term is always a single vector of Strings */
		termVector = IMINER_MD_TERM_PTR(metaData, col);
		nStatus = IMinerVector_create(termVector, 0, IMINER_MODE_STRING, NULL);
		if(nStatus != IMINER_SUCCESS) return nStatus;

		termDescrNames[col] = (char*)termNames[col];

		for (i=0; i<nColumns; i++) {
			val = colUse[col][i];
			for (j=0; j<val; j++) {
				nStatus = IMinerVector_addString(termVector, (const char*)colNames[i]);
				if(nStatus != IMINER_SUCCESS) return nStatus;
			}
		}
	}
	if (nTerms>0) {
		/*member 4 (special) : member names */
		nStatus = IMinerList_setNamesFromStrings(IMINER_MD_PTR(metaData, IMINER_MD_TERM_DEF_NUM),
												 (const char**)termDescrNames);
		if(nStatus != IMINER_SUCCESS) return nStatus;
	}
	free(termDescrNames);

	nStatus = IMinerList_setNamesFromStrings(metaData, (const char**)ppszMemberNames);
	/* set the class ID */
	metaData->m_nMode = IMINER_MODE_MODEL;
	return nStatus;
}

long IMCSRC_STDCALL IMinerMetaData_isValid(const IMinerObject* pModel)
{
	long nStatus=IMINER_SUCCESS, nColumns, i;
	IMinerObject *pElement;

	if (!pModel || !IMinerList_isValid(pModel)) return 0L;

	/* test columns */
	nColumns = IMINER_OBJECT_LENGTH(IMINER_MD_PTR(pModel, IMINER_MD_COLUMN_DEF_NUM));
	for (i=0; i<nColumns; i++) {
		if(!IMinerVector_isValid(IMINER_MD_COLUMN_LEVEL_NAMES(pModel, i)) ||
		   !IMinerVector_isValid(IMINER_MD_COLUMN_SPLIT_CATEGORIES(pModel, i))) {
			return 0L;
		}
	}
	pElement = IMINER_LIST_NAMES_PTR(IMINER_MD_PTR(pModel, IMINER_MD_COLUMN_DEF_NUM));
	if(!IMinerVector_isString(pElement) && pElement->m_nLen == pModel->m_nLen) return 0L;

	/* test terms */
	nColumns = IMINER_OBJECT_LENGTH(IMINER_MD_PTR(pModel, IMINER_MD_TERM_DEF_NUM));
	for (i=0; i<nColumns; i++) {
		if(!IMinerVector_isValid(IMINER_MD_TERM_PTR(pModel, i))) {
			return 0L;
		}
	}
	if (nColumns > 0) {
		pElement = IMINER_LIST_NAMES_PTR(IMINER_MD_PTR(pModel, IMINER_MD_TERM_DEF_NUM));
		if(!IMinerVector_isString(pElement) && pElement->m_nLen == pModel->m_nLen) return 0L;
	}

	/* special member : member names mut also be valid and has the same length as this object*/
	pElement = IMINER_LIST_NAMES_PTR(pModel);
	if(!IMinerVector_isString(pElement) && pElement->m_nLen == pModel->m_nLen)
		return 0L;

	return 1L;
}

/* free memory */
long IMCSRC_STDCALL IMinerMetaData_destroy(IMinerObject* pModel )
{
	return IMinerObject_destroy(pModel);
}


/* write to stdout */
long IMCSRC_STDCALL IMinerMetaData_print(const IMinerObject* pModel)
{
	long nStatus=IMINER_SUCCESS, nColumns, i;

	if(!IMinerMetaData_isValid(pModel))
	{
		IMiner_error("%s(%d) : ", __FILE__, __LINE__);
		IMiner_error("Invalid meta data\n");
		return IMINER_BAD_INPUT_ARGS;
	}


	nColumns = IMINER_OBJECT_LENGTH(IMINER_MD_PTR(pModel, IMINER_MD_COLUMN_DEF_NUM));
	for (i=0; i<nColumns; i++) {
		printf("Column (%s):\n", IMINER_STRING_VALUE(IMINER_LIST_NAMES_PTR(IMINER_MD_PTR(pModel, IMINER_MD_COLUMN_DEF_NUM)), i));
		printf("Level Names:\n");
		if (IMinerVector_print(IMINER_MD_COLUMN_LEVEL_NAMES(pModel, i)) != IMINER_SUCCESS) {
			return IMINER_BAD_INPUT_ARGS;
		}
		printf("Split Categories:\n");
		if (IMinerVector_print(IMINER_MD_COLUMN_SPLIT_CATEGORIES(pModel, i)) != IMINER_SUCCESS) {
			return IMINER_BAD_INPUT_ARGS;
		}
	}

	nColumns = IMINER_OBJECT_LENGTH(IMINER_MD_PTR(pModel, IMINER_MD_TERM_DEF_NUM));
	for (i=0; i<nColumns; i++) {
		printf("Term (%s):\n", IMINER_STRING_VALUE(IMINER_LIST_NAMES_PTR(IMINER_MD_PTR(pModel, IMINER_MD_TERM_DEF_NUM)), i));
		printf("Column Names:\n");
		if (IMinerVector_print(IMINER_MD_TERM_PTR(pModel, i)) != IMINER_SUCCESS) {
			return IMINER_BAD_INPUT_ARGS;
		}
	}

	return IMINER_SUCCESS;
}

long find_level(char *levToFind, IMinerVector *levsToSearch, long numLevels) {
	long lev=0;
	for (lev=0; lev<numLevels; lev++) {
		if (!strcmp(levToFind, IMINER_STRING_VALUE(levsToSearch, lev))) return lev;
	}

	return -1;
}


long IMCSRC_STDCALL IMinerMetaData_InputConvert(
 IMinerObject* pConvertedInput, /* out: data object */
 const IMinerObject* modelMetaData,	/* in:  meta data stored in model */
 const IMinerObject* pInput,			/* in:  input data object */
 IMinerObject* inputMD	/* in:  input data meta data */
)
{
	long nTerms, nModelColumns, i, j, k, term, out, outputColNum, numColsUsed, inputColNum, modelLevNum, used, inputLevel, inputModelLevel, modelColNum;
	long *nColsUsed, *nColsOutput, **inputColumnNums, **nLevels, **curLevels, *longColumnData;
	double *columnData;

	long nRows, nOutputColumns, *pnColumnModes, nInputColumns, col, tCol, lev, tLev;
	long row, val, numLevs, tNumLevs, split, nMode, nStatus;
	long nUnexpandedModelColumns, *outputToInputColumns, *outputToInputLevels;
	long *inputToModelColumns, **inputToModelLevels, maxNumInputLevs;
	long *modelToInputColumns, **modelToInputLevels, maxNumModelLevs;
	long *inputToExplodedColumns;
	char *modelColName, *modelLevelName, **levNamesPtr;
	IMinerObject *iCol, *oCol, *levelNames, *inputMetaData;

	if (!IMinerDataSet_isValid(pInput)) return 0;
	if (inputMD == NULL) {
		/* create from input data */
		IMinerObject createMD;
		IMinerMetaData_createFromDataSet(&createMD, pInput);
		inputMetaData = &createMD;
	} else {
		inputMetaData = inputMD;
	}

	nRows = IMINER_DATASET_NROWS(pInput);
	nInputColumns = IMINER_DATASET_NCOLUMNS(pInput);
	if (nInputColumns!=IMINER_MD_NUM_COLUMNS(inputMetaData)) return 0;

	nUnexpandedModelColumns = IMINER_MD_NUM_COLUMNS(modelMetaData);

	/* get number of output columns, max num of levels and column modes from model */
	for (col=0, nOutputColumns=0; col<nUnexpandedModelColumns; col++) {
		numLevs = IMINER_MD_COLUMN_LEVEL_COUNT(modelMetaData, col);

		if (col==0) maxNumModelLevs=numLevs;
		else if (maxNumModelLevs<numLevs) maxNumModelLevs=numLevs;
	}

	/* get max num levels from input as well */
	for (col=0; col<nInputColumns; col++) {
		numLevs = IMINER_MD_COLUMN_LEVEL_COUNT(inputMetaData, col);
		if (col==0) maxNumInputLevs=numLevs;
		else if (maxNumInputLevs<numLevs) maxNumInputLevs=numLevs;
	}
	/* set up mapping arrays */
	modelToInputColumns = (long*)malloc(nUnexpandedModelColumns*sizeof(long));
	modelToInputLevels = (long**)malloc(nUnexpandedModelColumns*sizeof(long*));
	inputToModelColumns = (long*)malloc(nInputColumns*sizeof(long));
	inputToExplodedColumns = (long*)malloc(nInputColumns*sizeof(long));
	inputToModelLevels = (long**)malloc(nInputColumns*sizeof(long*));
	/* set all input to model as invalid (-1)...will set properly if found in next loop */
	for (col=0; col<nInputColumns; col++) {
		inputToModelColumns[col] = -1;
		inputToExplodedColumns[col] = -1;
		numLevs = IMINER_MD_COLUMN_LEVEL_COUNT(inputMetaData, col);
		inputToModelLevels[col] = (long*)malloc(maxNumInputLevs*sizeof(long));
		for (lev=0; lev<maxNumInputLevs; lev++) inputToModelLevels[col][lev] = -1;
	}
	/* set up column and level mapping here in both directions (input->model && model->input) */
	for (col=0; col<nUnexpandedModelColumns; col++) {
		modelToInputColumns[col] = -1;

		modelColName = IMINER_MD_GET_COLUMN_NAME(modelMetaData, col);
		for (tCol=0; tCol<nInputColumns; tCol++) {
			if (!strcmp(modelColName, IMINER_MD_GET_COLUMN_NAME(inputMetaData, tCol))) {
				modelToInputColumns[col] = tCol;
				inputToModelColumns[tCol] = col;
				inputToExplodedColumns[tCol] = col;
				break;
			}
		}
		if (modelToInputColumns[col]==-1) {
			IMiner_error("Column required by model not found: %s\n", modelColName);
			return IMINER_BAD_INPUT_ARGS;
		}

		numLevs = IMINER_MD_COLUMN_LEVEL_COUNT(modelMetaData, col);
		modelToInputLevels[col] = (long*)malloc(maxNumModelLevs*sizeof(long));
		for (lev=0; lev<maxNumModelLevs; lev++) {
			modelToInputLevels[col][lev] = -1;
			if (lev>=numLevs) continue;

			modelLevelName = IMINER_STRING_VALUE(IMINER_MD_COLUMN_LEVEL_NAMES(modelMetaData, col), lev);
			tCol = modelToInputColumns[col];
			if (lev<numLevs && tCol!=-1) {
				tNumLevs = IMINER_MD_COLUMN_LEVEL_COUNT(inputMetaData, tCol);
				for (tLev=0; tLev<tNumLevs; tLev++) {
					if (!strcmp(modelLevelName, IMINER_STRING_VALUE(IMINER_MD_COLUMN_LEVEL_NAMES(inputMetaData, tCol), tLev))) {
						modelToInputLevels[col][lev] = tLev;
						inputToModelLevels[tCol][tLev] = lev;
						break;
					}
				}
			}
		}
	}

	/* get term info from model meta data */
	nOutputColumns = 0;
	nTerms = IMINER_MD_NUM_TERMS(modelMetaData);
	nModelColumns = IMINER_MD_NUM_COLUMNS(modelMetaData);
	nColsUsed = (long*)malloc(nTerms*sizeof(long));
	nColsOutput = (long*)malloc(nTerms*sizeof(long));
	inputColumnNums = (long**)malloc(nTerms*sizeof(long*));
	nLevels = (long**)malloc(nTerms*sizeof(long*));
	curLevels = (long**)malloc(nTerms*sizeof(long*));
	/* step through each term */
	for (i=0; i<nTerms; i++) {
		nColsOutput[i] = 1;
		nColsUsed[i] = IMINER_MD_NUM_TERM_NAMES(modelMetaData, i);
		inputColumnNums[i] = (long*)malloc(nColsUsed[i]*sizeof(long));
		nLevels[i] = (long*)malloc(nColsUsed[i]*sizeof(long));
		curLevels[i] = (long*)malloc(nColsUsed[i]*sizeof(long));
		/* look at each column used for this term */
		for (j=0; j<nColsUsed[i]; j++) {
			nLevels[i][j] = curLevels[i][j] = 0;
			inputColumnNums[i][j] = -1;

			/* which column in the input does this column in the model represent */
			modelColName = IMINER_MD_GET_TERM_NAME(modelMetaData, i, j);
			for (k=0; k<nInputColumns; k++) {
				if (!strcmp(modelColName, IMINER_MD_GET_COLUMN_NAME(inputMetaData, k))) {
					inputColumnNums[i][j] = k;
					nLevels[i][j] = IMINER_MD_COLUMN_LEVEL_COUNT(modelMetaData, inputToModelColumns[k]);
					break;
				}
			}

			/* if this is categorical, keep a running multiplicitive total */
			if (nLevels[i][j]>0) {
				nColsOutput[i] *= nLevels[i][j];
			}
		}

		if (nColsUsed[i] == 1) {
			/* if only one column is used for this, check if split categories */
			col = inputToModelColumns[inputColumnNums[i][0]];
			numLevs = IMINER_MD_COLUMN_LEVEL_COUNT(modelMetaData, col);
			split = IMINER_LONG_VALUE(IMINER_MD_COLUMN_SPLIT_CATEGORIES(modelMetaData, col), 0);
			nColsOutput[i] = ((split>=IMINER_MD_INDICATOR) ? numLevs : 1);
		}

		/* increment number of output columns */
		nOutputColumns += nColsOutput[i];

		/* adjust subsequent stored column info for exploded factor column */
		for (j=(inputColumnNums[i][0]+1); nColsOutput[i]>1 && j<nInputColumns; j++) {
			/* now that we have exploded a previous column, all subsequent columns
			   will be mapped to a different column */
			if (inputToExplodedColumns[j] != -1) {
				inputToExplodedColumns[j] += (nColsOutput[i]-1);
			}
		}
	}

	/* build mapping of desired output to the input data */
	outputToInputColumns = (long*)malloc(nOutputColumns*sizeof(long));
	outputToInputLevels = (long*)malloc(nOutputColumns*sizeof(long));
	for (col=0, tCol=0; col<nUnexpandedModelColumns; col++) {
		numLevs = IMINER_MD_COLUMN_LEVEL_COUNT(modelMetaData, col);
		split = IMINER_MD_COLUMN_NUM_SPLIT_CATEGORIES(modelMetaData, col);
		if (split>=IMINER_MD_INDICATOR) {
			for (lev=0; lev<numLevs; lev++) {
				outputToInputColumns[tCol] = modelToInputColumns[col];
				outputToInputLevels[tCol++] = modelToInputLevels[col][lev];
			}
		} else {
			outputToInputColumns[tCol] = modelToInputColumns[col];
			outputToInputLevels[tCol++] = -1;
		}
	}

	/* create an output data set of proper number of columns */
	pnColumnModes = (long*)malloc(nOutputColumns*sizeof(long));
	for (col=0; col<nOutputColumns; col++) pnColumnModes[col] = IMINER_MODE_DOUBLE;
	nStatus = IMinerDataSet_create(pConvertedInput, nRows, nOutputColumns, pnColumnModes);
	if (nStatus != IMINER_SUCCESS) return nStatus;

	/* create new data set */
	columnData = (double*)malloc(nRows*sizeof(double));
	longColumnData = (long*)malloc(nRows*sizeof(long));


	outputColNum = 0;
	for (term=0; term<nTerms; term++) {
		// cycle through the terms
		numColsUsed = nColsUsed[term];
		for (out=0; out<nColsOutput[term]; out++) {
			// cycle through all the output columns

			// reset column data to 1.0
			for (row=0; row<nRows; row++) columnData[row] = 1.0;

			for (used=0; used<nColsUsed[term]; used++) {
				// cycle through all the inputs for each output columns
				inputColNum = inputColumnNums[term][used];

				iCol = IMINER_DATASET_COLUMN_PTR(pInput, inputColNum);
				oCol = IMINER_DATASET_COLUMN_PTR(pConvertedInput, outputColNum);
				if(IMinerDataSet_getColumnMode(&nMode, pInput, inputColNum)!=IMINER_SUCCESS) {
					return IMINER_BAD_INPUT_ARGS;
				}

				if (nMode==IMINER_MODE_FACTOR) {
					// desired level number according to the model
					modelLevNum = curLevels[term][used];

					for (row=0; row<nRows; row++) {
						// this rows level number according to the input
						inputLevel = IMINER_FACTOR_DATA_VALUE(iCol, row);
						// convert input level number to model level number
						inputModelLevel = inputToModelLevels[inputColNum][inputLevel];
						if (outputToInputLevels[inputToExplodedColumns[inputColNum]]==-1) {
							// if categorical column is NOT being split, output model level number
							columnData[row] *= inputModelLevel;
						} else {
							// if categorical column IS begin split, output if this is the desired level
							val = (inputModelLevel==modelLevNum) ? 1 : 0;
							columnData[row] *= val;
						}
					}
				} else if (nMode==IMINER_MODE_DOUBLE) {
					// desired level number according to the model
					modelLevNum = curLevels[term][used];

					for (row=0; row<nRows; row++) {
						val = (long)(IMINER_DOUBLE_VALUE(iCol, row));
						if (outputToInputLevels[inputToExplodedColumns[inputColNum]]>-1) {
							// FACTOR doubles need to be converted to model level numbers
							inputModelLevel = inputToModelLevels[inputColNum][val];
							val = (inputModelLevel==modelLevNum) ? 1: 0;
							columnData[row] = val;
						} else {
							// NON-FACTOR doubles just get multiplied out
							columnData[row] *= IMINER_DOUBLE_VALUE(iCol, row);
						}
					}
				} else {
					return IMINER_BAD_INPUT_ARGS;
				}

			}


			// output to modified dataset
			if (nColsUsed[term]==1 && nColsOutput[term]==1 && nMode==IMINER_MODE_FACTOR) {
				// output factor column if only one output per column and mode is factor
				modelColNum = inputToModelColumns[inputColNum];
				if (IMINER_MD_COLUMN_NUM_SPLIT_CATEGORIES(modelMetaData, modelColNum)!=IMINER_MD_NO_SPLIT) {
					// if mode is double or this is an interaction or categorical has been split
					IMinerObject columnVector;
					IMinerVector_create(&columnVector, nRows, IMINER_MODE_DOUBLE, columnData);
					IMinerDataSet_setColumnAt(pConvertedInput, outputColNum, &columnVector);
					IMinerObject_destroy(&columnVector);
				} else {
					IMinerObject factor;

					levelNames = IMINER_MD_COLUMN_LEVEL_NAMES(modelMetaData, modelColNum);
					numLevs = IMINER_MD_COLUMN_LEVEL_COUNT(modelMetaData, modelColNum);
					levNamesPtr = IMINER_STRING_PTR(levelNames);
					for (row =0; row<nRows; row++) longColumnData[row] = (long)columnData[row];
					IMinerFactor_create(&factor, nRows, longColumnData, numLevs, (const char**)levNamesPtr);
					IMinerDataSet_setColumnAt(pConvertedInput, outputColNum, &factor);
					IMinerObject_destroy(&factor);
				}
			} else {
				// if mode is double or this is an interaction or categorical has been split
				IMinerObject columnVector;
				IMinerVector_create(&columnVector, nRows, IMINER_MODE_DOUBLE, columnData);
				IMinerDataSet_setColumnAt(pConvertedInput, outputColNum, &columnVector);
				IMinerObject_destroy(&columnVector);
			}

			// increment levels as needed
			for (i=nColsUsed[term]-1; i>=0; i--) {
				if (curLevels[term][i]<nLevels[term][i]-1) {
					curLevels[term][i] += 1;
					break;
				} else {
					curLevels[term][i] = 0;
				}
			}

			outputColNum++;
		}
	}


	if (inputMD == NULL) {
		IMinerMetaData_destroy(inputMetaData);
	}

	for (i=0; i<nUnexpandedModelColumns; i++) {
		free(modelToInputLevels[i]);
	}
	for (i=0; i<nInputColumns; i++) {
		free(inputToModelLevels[i]);
	}
	for (i=0; i<nTerms; i++) {
		free(inputColumnNums[i]);
		free(nLevels[i]);
		free(curLevels[i]);
	}

	free(longColumnData);
	free(columnData);

	free(outputToInputColumns);
	free(outputToInputLevels);
	free(modelToInputColumns);
	free(modelToInputLevels);
	free(inputToModelColumns);
	free(inputToExplodedColumns);
	free(inputToModelLevels);
	free(pnColumnModes);

	free(inputColumnNums);
	free(nLevels);
	free(curLevels);
	free(nColsUsed);
	free(nColsOutput);

	return 1;
}


/********************************* Misc *********************************************/

/* open IMCSRC a data file for write*/
long IMCSRC_STDCALL IMinerDataFile_openForWrite(FILE** ppFile, /* out: output file pointer */
 const char* pszFile    /* in: file name */
)
{
	long nStatus = IMINER_SUCCESS;
	*ppFile = fopen(pszFile, "w");
	if(!(*ppFile))
		return IMINER_BAD_INPUT_ARGS;

	/* Version number must be at the top of the file */
	nStatus = IMinerVersion_writeToStream(*ppFile);
	if(nStatus != IMINER_SUCCESS)
	{
		fclose(*ppFile);
		*ppFile = NULL;
	}
	return nStatus;
}

/* open IMCSRC a data file for read*/
long IMCSRC_STDCALL IMinerDataFile_openForRead(FILE** ppFile, /* out: output file pointer */
 const char* pszFile    /* in: file name */
)
{
	long nStatus = IMINER_SUCCESS;
	*ppFile = fopen(pszFile, "r");
	if(!(*ppFile))
		return IMINER_BAD_INPUT_ARGS;

	/* Version number must be at the top of the file.*/
	nStatus = IMinerVersion_readFromStreamAndValidate(*ppFile);
	if(nStatus != IMINER_SUCCESS)
	{
		fclose(*ppFile);
		*ppFile = NULL;
	}
	return nStatus;
}

/*  write to an opened file. Version number must be at the top of the file. */
long IMCSRC_STDCALL IMinerVersion_writeToStream(FILE* pFile /* in: output file pointer */)
{
	if(!pFile)
		return IMINER_BAD_INPUT_ARGS;
	/* write the comment on the first line */
	fprintf(pFile, "# IMCSRC data's dump\n");
	/* write the version number */
	fprintf(pFile, "%s\n", IMINER_KEYWORD_VERSION);
	fprintf(pFile, "%6.2f\n", IMINER_IMCSRC_VERSION);
	return IMINER_SUCCESS;
}

/* read from an opened file and validate it.  Version number must be at the top of the file */
long IMCSRC_STDCALL IMinerVersion_readFromStreamAndValidate(FILE* pFile/* in: input file pointer */)
{
	char szLine[LINE_LEN];
	float fVersion;
	if(!pFile)
		return IMINER_BAD_INPUT_ARGS;
	/* read the comment on the first line */
	if( fgets( szLine, LINE_LEN, pFile ) == NULL)
		return IMINER_BAD_INPUT_ARGS;
	/* read version from the top of the file*/
	if( fgets( szLine, LINE_LEN, pFile ) == NULL)
		return IMINER_BAD_INPUT_ARGS;
	if( fgets( szLine, LINE_LEN, pFile ) == NULL)
		return IMINER_BAD_INPUT_ARGS;
	sscanf(szLine, "%f", &fVersion);

	/* version of file must match with this code*/
	if(fVersion == (float)IMINER_IMCSRC_VERSION)
		return IMINER_SUCCESS;
	else
		return IMINER_BAD_INPUT_ARGS;
}

/* write header info to an opened file. */
long IMCSRC_STDCALL IMinerObjectHeader_writeToStream(FILE* pFile, /* in/out: input file pointer */
 const char* pszClass, /* in: object type if not NULL.  If NULL, use result from IMiner_getClass()*/
 long nMode            /* in: mode of an IMiner object which is a unique type (ID) */
)
{
	char szClassKeyWord[] = IMINER_KEYWORD_CLASS;
	char szModeKeyWord[] = IMINER_KEYWORD_MODE;
	char szClass[LINE_LEN];

	/* write class keyword*/
	fprintf(pFile, "\n%s\n", szClassKeyWord);

	/* write the class name*/
	if(pszClass == NULL)
	{
		IMiner_getClassFromMode(szClass, LINE_LEN, nMode);
		fprintf(pFile, "%s\n", szClass);
	}
	else
		fprintf(pFile, "%s\n", pszClass);

	/* write mode */
	fprintf(pFile, "%s\n", szModeKeyWord);
	fprintf(pFile, "%d\n", nMode);

	return IMINER_SUCCESS;
}

/* read header info from an opened file and validate it. */
long IMCSRC_STDCALL IMinerObjectHeader_readFromStream(long* pnMode, /* out: mode of an IMiner object which is a unique type (ID) */
 FILE* pFile /* in: input file pointer */
)
{
	char szLine[LINE_LEN];
	char szClassKeyWord[] = IMINER_KEYWORD_CLASS;
	char szModeKeyWord[] = IMINER_KEYWORD_MODE;
	char szClass[LINE_LEN];
	char szClassFromStream[LINE_LEN];

	/* skip pass szClassKeyWord*/
	while(strncmp(szLine, szClassKeyWord, strlen(szClassKeyWord)) != 0)
	{
		if( fgets( szLine, LINE_LEN, pFile ) == NULL)
			return IMINER_BAD_INPUT_ARGS;
	}
	/* read class name*/
	if( fgets( szClassFromStream, LINE_LEN, pFile ) == NULL)
		return IMINER_BAD_INPUT_ARGS;
	/* skip pass szModeKeyWord */
	while(strncmp(szLine, szModeKeyWord, strlen(szModeKeyWord)) != 0)
	{
		if( fgets( szLine, LINE_LEN, pFile ) == NULL)
			return IMINER_BAD_INPUT_ARGS;
	}
	/* read the mode */
	fscanf(pFile, "%d\n", pnMode);

	/* validate */
	IMiner_getClassFromMode(szClass, LINE_LEN, *pnMode);
	if(strncmp(szClass, "unknown", 7) == 0)
		return IMINER_SUCCESS; /* ok if we don't know this type */

	if(strncmp(szClassFromStream, szClass, strlen(szClass))!=0)
		return IMINER_BAD_INPUT_ARGS;

	return IMINER_SUCCESS;
}

/* return object type as char* given a mode */
long IMCSRC_STDCALL IMiner_getClassFromMode(char* pszBuf, /*in/out: object type */
 size_t nBufLen, /* in: length of the buffer, pszBuf, must be long enough.  If not result will be trancated */
 long nMode      /* in: mode of an IMiner object which is a unique type (ID) */
)
{
	switch(nMode)
	{
		case IMINER_MODE_NULL :
			strncpy(pszBuf, "null", nBufLen);
			break;
		case IMINER_MODE_BOOLEAN :
		case IMINER_MODE_LONG :
		case IMINER_MODE_FLOAT :
		case IMINER_MODE_DOUBLE :
		case IMINER_MODE_STRING   :
		case IMINER_MODE_LIST   :
			strncpy(pszBuf, IMINER_CLASS_VECTOR, nBufLen);
			break;
		case IMINER_MODE_FACTOR :
			strncpy(pszBuf, IMINER_CLASS_FACTOR, nBufLen);
			break;
		case IMINER_MODE_DOUBLEMATRIX :
			strncpy(pszBuf, IMINER_CLASS_MATRIX, nBufLen);
			break;
		case IMINER_MODE_MODELMATRIX :
			strncpy(pszBuf, IMINER_CLASS_MODELMATRIX, nBufLen);
			break;
		case IMINER_MODE_DATASET :
			strncpy(pszBuf, IMINER_CLASS_DATASET, nBufLen);
			break;
		default:
			strncpy(pszBuf, "unknown", nBufLen);
			break;
	}
	return strlen(pszBuf);
}

/* some simple test */
void IMinerTest(void)
{
	long nStatus;
	IMinerObject dataSet;
	IMinerObject mmatrix;

	nStatus = IMinerDataSet_createFromTextFile((IMinerObject*)&dataSet, "d:\\sprime\\windows\\dist\\sprime\\examples\\fuel.txt");
	nStatus = IMinerDataSet_print(&dataSet);

	nStatus = IMinerModelMatrix_createFromDataSet(&mmatrix, &dataSet);
	nStatus = IMinerModelMatrix_print(&mmatrix);

	nStatus = IMinerObject_destroy(&dataSet);
	nStatus = IMinerObject_destroy(&mmatrix);
}

/*
 print to stderr and dealing with ellipse ...
*/
long IMCSRC_STDCALL IMiner_error(const char* pszFormat, ...)
{
	va_list args;
	va_start(args, pszFormat);
	vfprintf(stderr, pszFormat, args);
	return IMINER_SUCCESS;
}

#ifdef USE_HASH_LOOKUP
static int HASH_CAPACITIES = 769;
/*
  Create & return a new hash table.

  capacity Number of entries this hash table will have.  This
           should be a prime number, and ideally a prime number just
           over a power of two, for  best performance and distribution.
*/
struct IMinerHashTable *IMinerHashTable_createHash()
{
	struct IMinerHashTable *r = (IMinerHashTable*)malloc(sizeof(struct IMinerHashTable));

	if (r == NULL) {
		printf(("Not enough memory for hash table."));
		return NULL;
	}

	r->used = 0;
	r->capacity = HASH_CAPACITIES;
	r->objects = (IMinerHashEntry**)calloc(HASH_CAPACITIES, sizeof(IMinerHashEntry*));

	if (r->objects == NULL) {
		printf("Not enough memory for %d hash table capacity.", HASH_CAPACITIES);
		free(r);
		return NULL;
	}

	return r;
}

/* Destroys a hash table. */
void IMinerHashTable_freeHash(struct IMinerHashTable *ht)
{
	unsigned int i;

	if (ht == NULL) return;

	for (i = 0; i < ht->capacity; i++) {
		if (ht->objects[i] != NULL) {
			struct IMinerHashEntry *e = ht->objects[i];
			while (e) {
				struct IMinerHashEntry *n = e->collision;
				free(e->key);
				free(e);
				e = n;
			}
		}
	}

	free(ht->objects);
	free(ht);
}

/**
 * Hash a stringinto a 32bit value.  The hash algorithm used is
 *
 * djb2 - this algorithm (k=33) was first reported by dan bernstein
 * many years ago in comp.lang.c. another version of this algorithm
 * (now favored by bernstein) uses xor: hash(i) = hash(i - 1) * 33 ^ str[i]
 * the magic of number 33 (why it works better than many other constants,
 * prime or not) has never been adequately explained.
 *
 *
 * Fowler Noll Vo - a very fast and simple hash, ideal for short strings.
 * See http://en.wikipedia.org/wiki/Fowler_Noll_Vo_hash for more details.
 */
unsigned int IMinerHashTable_hashString(const char *key)
{
#define BERSTEIN_ALG
#ifdef BERSTEIN_ALG
	unsigned long hash = 5381;
	int c;

	while (c = *key++)
		hash = ((hash << 5) + hash) + c; /* hash * 33 + c */

	return hash;

#else

	unsigned int z = 0x013;

	if (key == NULL) return 0;

	while (*key) {
		z *= 0x013;
		z ^= *key++;
	}

	return z;
#endif /*BERSTEIN_ALG*/
}

/**
 * Adds a key/value pair to a hash table.  If the key you're adding is already
 * in the hash table, it does not replace it, but it does take precedent over
 * it.
 */
unsigned int IMinerHashTable_put(struct IMinerHashTable *ht, const char *key, int value)
{
	unsigned int h, c;
	struct IMinerHashEntry *e;

	if (ht == NULL || key == NULL) return 0;

	e = (IMinerHashEntry*)malloc(sizeof(struct IMinerHashEntry));
	if (e == NULL) {
		printf("Not enough memory for hash entry.");
		return -1;
	}

	h = IMinerHashTable_hashString(key);
	c = h % ht->capacity;

	e->value = value;
	e->length = strlen(key);
	e->key = (char*)malloc(e->length + 1);
	if (e->key == NULL) {
		printf("Not enough memory for string duplication.");
		free(e);
		return -1;
	}
	memcpy(e->key, key, e->length + 1);

	/* look for collisions...if an object already exists at
		this hash address, link it up so it isn't lost */
	e->collision = ht->objects[c];
	ht->objects[c] = e;
	/* increment the used count */
	ht->used += 1;

	return e->value;
}

 /**
  * Looks up a the value associated with with a key from a specific hash table.
  */

 unsigned int IMinerHashTable_get(struct IMinerHashTable *ht, const char *key)
 {
         unsigned int h, c;
         struct IMinerHashEntry *e;

         if (ht == NULL || key == NULL) return -1;

         h = IMinerHashTable_hashString(key);
         c = h % ht->capacity;

         for (e = ht->objects[c]; e; e = e->collision)
			 if (!strcmp(key, e->key)) return e->value;

         return -1;
 }


#endif /*USE_HASH_LOOKUP*/
