<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform">
	<!-- XSL style sheet for creating IMML from PMML -->
	<xsl:output method="xml" standalone="yes" indent="yes"/>
	<xsl:template match="PMML">
		<xsl:element name="IMML">
			<xsl:attribute name="version"><xsl:text>3.0</xsl:text></xsl:attribute>
			<!-- Header information -->
			<xsl:apply-templates select="Header"/>
			<!-- Description of the data -->
			<xsl:apply-templates select="DataDictionary"/>

			<!-- Models -->
			<xsl:apply-templates select="GeneralRegressionModel"/>
			<xsl:apply-templates select="NeuralNetwork"/>
			<xsl:apply-templates select="ClusteringModel"/>
			<xsl:apply-templates select="NaiveBayesModel"/>
			<!-- Handle single or ensemble trees appropriately -->
			<xsl:if test="count(TreeModel) > 0">
					<xsl:element name="TreeList">
						<xsl:attribute name="modelType"><xsl:value-of select="TreeModel[1]/@functionName"/></xsl:attribute>
						<xsl:apply-templates select="TreeModel[1]/MiningSchema"/>
						<xsl:apply-templates select="TreeModel"/>
					</xsl:element>
			</xsl:if>

			<!-- Models not in standard PMML.  Extensions in our generated PMML.  -->
			<xsl:apply-templates select="Extension"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Extension">
			<!-- Models not in standard PMML.  Extensions in our generated PMML.  -->
			<xsl:apply-templates select="X-IMML-TreeModel-TreeList"/>
			<xsl:apply-templates select="X-IMML-SurvivalModel"/>
			<xsl:apply-templates select="X-IMML-PrincipalComponentsModel"/>
	</xsl:template>
	<xsl:template match="Header">
		<!-- Get info from the IMML Header element for Application and Timestamp -->
		<xsl:element name="Header">
			<xsl:attribute name="copyright"><xsl:value-of select="@copyright"/></xsl:attribute>
			<xsl:if test="string-length(normalize-space(@description)) > 0">
				<xsl:attribute name="name"><xsl:value-of select="@description"/></xsl:attribute>
			</xsl:if>	
			<xsl:apply-templates select="Application"/>
			<xsl:apply-templates select="Timestamp"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Application">
		<xsl:element name="Application">
			<xsl:attribute name="name"><xsl:value-of select="@name"/></xsl:attribute>
			<xsl:attribute name="version"><xsl:value-of select="@version"/></xsl:attribute>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Timestamp">
		<xsl:element name="Timestamp">
			<xsl:value-of select="./text()"/>
		</xsl:element>
	</xsl:template>
	<!-- DataDictionary -->
	<xsl:template match="DataDictionary">
		<xsl:element name="XTMetaData">
			<xsl:element name="ColumnDefinitions">
				<xsl:attribute name="fieldCount"><xsl:value-of select="@numberOfFields"/></xsl:attribute>
				<xsl:apply-templates select="DataField"/>
			</xsl:element>
		</xsl:element>
	</xsl:template>
	<xsl:template match="DataField">
		<xsl:element name="ColumnInfo">
			<xsl:attribute name="name"><xsl:value-of select="@name"/></xsl:attribute>
			<xsl:attribute name="type"><xsl:value-of select="@optype"/></xsl:attribute>			
			
			<!-- Reconstruct role attribute values -->
			<!-- This logic currently assumes there's a single model in the file -->

			<xsl:variable name="curName"><xsl:value-of select="@name"/></xsl:variable>
			<xsl:variable name="curUsageType"><xsl:value-of select="//MiningField[@name=$curName][1]/@usageType"/></xsl:variable>
			<xsl:choose>
				<!-- Use the usageType if it's present, otherwise inspect other elements in a model-specific manner -->
				<xsl:when test="curUsageType = 'predicted'">
					<xsl:attribute name="role">dependent</xsl:attribute>					
				</xsl:when>
				<xsl:when test="curUsageType  = 'active'">
					<xsl:attribute name="role">independent</xsl:attribute>					
				</xsl:when>				
				<xsl:when test="curUsageType = 'supplementary'">
					<xsl:attribute name="role">information</xsl:attribute>					
				</xsl:when>
				<xsl:when test="count(//X-IMML-SurvivalModel)>0">
					<!-- SurvivalModel -->
					<xsl:choose>
						<xsl:when test="count(//X-IMML-SurvivalModel-SurvResponse[@event=$curName]) >0 or count(//X-IMML-SurvivalModel-SurvResponse[@stop=$curName]) >0 or count(//X-IMML-SurvivalModel-SurvResponse[@start=$curName]) >0">																<xsl:attribute name="role">dependent</xsl:attribute>
						</xsl:when>
						<xsl:when test="count(//Predictor[@name=$curName]) > 0">
								<xsl:attribute name="role">independent</xsl:attribute>							
						</xsl:when>
						<xsl:when test="count(//X-IMML-SurvivalModel-Weights[@value=$curName]) > 0">
								<xsl:attribute name="role">weights</xsl:attribute>							
						</xsl:when>
						<xsl:when test="count(//X-IMML-SurvivalModel-Strata[@value=$curName]) > 0">
								<xsl:attribute name="role">strata</xsl:attribute>							
						</xsl:when>
						<xsl:otherwise>
							<xsl:attribute name="role">information</xsl:attribute>							
						</xsl:otherwise>
					</xsl:choose>
				</xsl:when>
				<xsl:when test="count(//GeneralRegressionModel) > 0">
					<!-- Linear/Logistic Regression -->
					<xsl:choose>
						<xsl:when test="count(//GeneralRegressionModel[@targetVariableName=$curName])>0">
							<xsl:attribute name="role">dependent</xsl:attribute>					
						</xsl:when>
						<xsl:when test="count(//Predictor[@name=$curName]) > 0">
								<xsl:attribute name="role">independent</xsl:attribute>							
						</xsl:when>
						<xsl:otherwise>
								<xsl:attribute name="role">information</xsl:attribute>						
						</xsl:otherwise>
					</xsl:choose>
				</xsl:when>
				<xsl:when test="count(//NeuralNetwork) > 0">
					<!-- Neural Networks -->
					<xsl:choose>
						<xsl:when test="count(//NeuralOutput[descendant::node()/@field=$curName])>0">
							<xsl:attribute name="role">dependent</xsl:attribute>					
						</xsl:when>
						<xsl:when test="count(//NeuralInput[descendant::node()/@field=$curName])>0">
								<xsl:attribute name="role">independent</xsl:attribute>							
						</xsl:when>
						<xsl:otherwise>
								<xsl:attribute name="role">information</xsl:attribute>						
						</xsl:otherwise>
					</xsl:choose>
				</xsl:when>			
				<xsl:when test="count(//NaiveBayesModel) > 0">
					<!-- Naive Bayes -->
					<xsl:choose>
						<xsl:when test="count(//BayesOutput[@fieldName=$curName])>0">
							<xsl:attribute name="role">dependent</xsl:attribute>					
						</xsl:when>
						<xsl:when test="count(//BayesInput[@fieldName=$curName])>0">
								<xsl:attribute name="role">independent</xsl:attribute>							
						</xsl:when>
						<xsl:otherwise>
								<xsl:attribute name="role">information</xsl:attribute>						
						</xsl:otherwise>
					</xsl:choose>
				</xsl:when>	
				<xsl:when test="count(//TreeModel) > 0">
					<!-- Tree -->
					<xsl:choose>
						<xsl:when test="count(//SimplePredicate[@field=$curName])>0">
							<xsl:attribute name="role">independent</xsl:attribute>					
						</xsl:when>
						<xsl:when test="count(//MiningField[@name=$curName][@usageType='predicted'])>0">
								<xsl:attribute name="role">dependent</xsl:attribute>							
						</xsl:when>
						<xsl:when test="count(//MiningField[@name=$curName])>0">
								<xsl:attribute name="role">independent</xsl:attribute>							
						</xsl:when>
						<xsl:otherwise>
								<xsl:attribute name="role">information</xsl:attribute>						
						</xsl:otherwise>
					</xsl:choose>
				</xsl:when>					
				<xsl:when test="count(//MiningSchema[MiningField/@name=$curName]) > 0"> 
					<!-- Cluster, Princomp, possibly others -->
					<xsl:attribute name="role">independent</xsl:attribute>
				</xsl:when>
				<xsl:otherwise>
					<!-- Call it information if no other role found -->
					<xsl:attribute name="role">information</xsl:attribute>
				</xsl:otherwise>
			</xsl:choose>
			<xsl:apply-templates select="Value"/>			
		</xsl:element>
	</xsl:template>
	<xsl:template match="Value">
		<xsl:element name="Level">
			<xsl:attribute name="value"><xsl:value-of select="@value"/></xsl:attribute>
		</xsl:element>
	</xsl:template>
	<!-- MiningSchema -->
	<xsl:template match="MiningSchema">
		<xsl:element name="MiningSchema">
			<xsl:apply-templates select="MiningField"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="MiningField">
		<xsl:element name="MiningField">
			<xsl:attribute name="name"><xsl:value-of select="@name"/></xsl:attribute>
			<xsl:if test="@usageType != 'active'">
				<xsl:attribute name="usageType"><xsl:value-of select="@usageType"/></xsl:attribute>
			</xsl:if>
		</xsl:element>
	</xsl:template>
	<!-- TreeModel -->
	<!-- Multiple trees are in an X-IMML-TreeModel-TreeList.  Single tree is just in a TreeModel. -->
	<xsl:template match="X-IMML-TreeModel-TreeList">
		<xsl:element name="TreeList">
			<xsl:attribute name="modelType"><xsl:value-of select="TreeModel[1]/@functionName"/></xsl:attribute>
			<xsl:apply-templates select="TreeModel[1]/MiningSchema"/>
			<xsl:apply-templates select="TreeModel"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="TreeModel">
		<xsl:element name="TreeModel">
			<xsl:attribute name="name"><xsl:value-of select="@modelName"/></xsl:attribute>
			<xsl:attribute name="splitCharacteristic"><xsl:value-of select="@splitCharacteristic"/></xsl:attribute>
			<xsl:attribute name="criterion"><xsl:value-of select="Extension/X-IMML-XTProps/X-IMML-Property[@name='criterion']/@value"/></xsl:attribute>
			<xsl:apply-templates select="Node"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Node">
		<xsl:element name="Node">
			<!-- TODO: Store these nonstandard attributes in an extension -->
			<xsl:attribute name="id"><xsl:value-of select="Extension/X-IMML-XTProps/X-IMML-Property[@name='id']/@value"/></xsl:attribute>

			<xsl:attribute name="score"><xsl:value-of select="@score"/></xsl:attribute>
			<xsl:attribute name="recordCount"><xsl:value-of select="@recordCount"/></xsl:attribute>

			<xsl:attribute name="group"><xsl:value-of select="Extension/X-IMML-XTProps/X-IMML-Property[@name='group']/@value"/></xsl:attribute>
			<xsl:attribute name="deviance"><xsl:value-of select="Extension/X-IMML-XTProps/X-IMML-Property[@name='deviance']/@value"/></xsl:attribute>
			<xsl:attribute name="entropy"><xsl:value-of select="Extension/X-IMML-XTProps/X-IMML-Property[@name='entropy']/@value"/></xsl:attribute>
			<xsl:attribute name="gini"><xsl:value-of select="Extension/X-IMML-XTProps/X-IMML-Property[@name='gini']/@value"/></xsl:attribute>
			<xsl:attribute name="risk"><xsl:value-of select="Extension/X-IMML-XTProps/X-IMML-Property[@name='risk']/@value"/></xsl:attribute>
			
			<xsl:variable name="yp"><xsl:value-of select="Extension/X-IMML-XTProps/X-IMML-Property[@name='yprob']/@value"/></xsl:variable>
			<xsl:if test="string-length($yp) > 0">
				<xsl:attribute name="yprob"><xsl:value-of select="$yp"/></xsl:attribute>
			</xsl:if>
			<xsl:apply-templates select="SimplePredicate"/>
			<xsl:apply-templates select="CompoundPredicate"/>
			<xsl:apply-templates select="True"/>
			<xsl:apply-templates select="False"/>
			<xsl:apply-templates select="Node"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="SimplePredicate">
		<xsl:element name="SimplePredicate">
			<xsl:attribute name="field"><xsl:value-of select="@field"/></xsl:attribute>
			<xsl:attribute name="operator"><xsl:value-of select="@operator"/></xsl:attribute>
			<xsl:attribute name="value"><xsl:value-of select="@value"/></xsl:attribute>
			
			<xsl:variable name="imp"><xsl:value-of select="../Extension/X-IMML-XTProps/X-IMML-Property[@name='improvement']/@value"/></xsl:variable>
			<xsl:if test="string-length($imp) > 0">
				<xsl:attribute name="improvement"><xsl:value-of select="$imp"/></xsl:attribute>
			</xsl:if>					
		</xsl:element>
	</xsl:template>
	<xsl:template match="CompoundPredicate">
		<xsl:element name="CompoundPredicate">
			<xsl:attribute name="booleanOperator"><xsl:value-of select="@booleanOperator"/></xsl:attribute>
				<xsl:variable name="imp"><xsl:value-of select="../Extension/X-IMML-XTProps/X-IMML-Property[@name='improvement']/@value"/></xsl:variable>
			<xsl:if test="string-length($imp) > 0">
				<xsl:attribute name="improvement"><xsl:value-of select="$imp"/></xsl:attribute>
			</xsl:if>		
			
			<xsl:apply-templates select="SimplePredicate"/>
			<xsl:apply-templates select="CompoundPredicate"/>
			<xsl:apply-templates select="True"/>
			<xsl:apply-templates select="False"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="True">
		<xsl:element name="True"/>
	</xsl:template>
	<xsl:template match="False">
		<xsl:element name="False"/>
	</xsl:template>
	<!-- NeuralNetwork -->
	<xsl:template match="NeuralNetwork">
		<xsl:element name="NeuralNetwork">
			<xsl:attribute name="modelName"><xsl:value-of select="@modelName"/></xsl:attribute>
			<xsl:attribute name="activationFunction"><xsl:value-of select="@activationFunction"/></xsl:attribute>
			<xsl:attribute name="targetVariableName"><xsl:value-of select="./MiningSchema/MiningField[@usageType='predicted']/@name"/></xsl:attribute>
			<xsl:attribute name="modelType"><xsl:value-of select="@functionName"/></xsl:attribute>
			<xsl:apply-templates select="MiningSchema"/>
			<xsl:apply-templates select="NeuralInputs"/>
			<xsl:apply-templates select="NeuralLayer"/>
			<xsl:apply-templates select="NeuralOutputs"/>
			<xsl:for-each select="Extension">
				<xsl:apply-templates select="X-IMML-NeuralNetwork-NeuralParameters"/>
				<xsl:apply-templates select="X-IMML-DevMatrix"/>
				<xsl:apply-templates select="X-IMML-NeuralNetwork-ErrorWeights"/>
			</xsl:for-each>
		</xsl:element>
	</xsl:template>
	<xsl:template match="NeuralInputs">
		<xsl:element name="NeuralInputs">
			<xsl:apply-templates select="NeuralInput"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="NeuralInput">
		<xsl:element name="NeuralInput">
			<xsl:attribute name="id"><xsl:value-of select="@id"/></xsl:attribute>
			<xsl:for-each select="DerivedField">
				<!-- Add other DerivedField types if they are implemented -->
				<xsl:apply-templates select="NormContinuous"/>
				<xsl:apply-templates select="NormDiscrete"/>			
			</xsl:for-each>
		</xsl:element>
	</xsl:template>
	<xsl:template match="NormContinuous">
			<xsl:element name="NormContinuous">
				<xsl:attribute name="field"><xsl:value-of select="@field"/></xsl:attribute>
				<xsl:apply-templates select="LinearNorm"/>
			</xsl:element>
	</xsl:template>
	<xsl:template match="LinearNorm">
		<xsl:element name="LinearNorm">
			<xsl:attribute name="orig"><xsl:value-of select="@orig"/></xsl:attribute>
			<xsl:attribute name="norm"><xsl:value-of select="@norm"/></xsl:attribute>
		</xsl:element>
	</xsl:template>
	<xsl:template match="NormDiscrete">
			<xsl:element name="NormDiscrete">
				<xsl:attribute name="field"><xsl:value-of select="@field"/></xsl:attribute>
				<xsl:attribute name="value"><xsl:value-of select="@value"/></xsl:attribute>
			</xsl:element>
	</xsl:template>
	<xsl:template match="NeuralLayer">
		<xsl:element name="NeuralLayer">
			<xsl:apply-templates select="Neuron"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Neuron">
		<xsl:element name="Neuron">
			<xsl:attribute name="id"><xsl:value-of select="@id"/></xsl:attribute>
			<xsl:apply-templates select="Con"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Con">
		<xsl:element name="Con">
			<xsl:attribute name="from"><xsl:value-of select="@from"/></xsl:attribute>
			<xsl:attribute name="weight"><xsl:value-of select="@weight"/></xsl:attribute>
			
			<xsl:attribute name="learnRate"><xsl:value-of select="Extension/X-IMML-XTProps/X-IMML-Property[@name='learnRate']/@value"/></xsl:attribute>
			<xsl:attribute name="momentum"><xsl:value-of select="Extension/X-IMML-XTProps/X-IMML-Property[@name='momentum']/@value"/></xsl:attribute>
			<xsl:attribute name="deltaWeight"><xsl:value-of select="Extension/X-IMML-XTProps/X-IMML-Property[@name='deltaWeight']/@value"/></xsl:attribute>
			<xsl:attribute name="weightDelta"><xsl:value-of select="Extension/X-IMML-XTProps/X-IMML-Property[@name='weightDelta']/@value"/></xsl:attribute>
		</xsl:element>
	</xsl:template>
	<xsl:template match="NeuralOutputs">
		<xsl:element name="NeuralOutputs">
			<xsl:apply-templates select="NeuralOutput"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="NeuralOutput">
		<xsl:element name="NeuralOutput">
			<xsl:attribute name="id"><xsl:value-of select="@outputNeuron"/></xsl:attribute>
			<xsl:for-each select="DerivedField">
				<!-- Add other DerivedField types if they are implemented -->
				<xsl:apply-templates select="NormContinuous"/>
				<xsl:apply-templates select="NormDiscrete"/>			
			</xsl:for-each>
		</xsl:element>
	</xsl:template>
	
	<xsl:template match="X-IMML-NeuralNetwork-NeuralParameters">
		<xsl:element name="NeuralParameters">
			<xsl:copy-of select="@*"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-DevMatrix">
		<xsl:element name="DevMatrix">
			<xsl:copy-of select="@*"/>
			<xsl:apply-templates select="X-IMML-DevCell"/>
		</xsl:element>	
	</xsl:template>
	<xsl:template match="X-IMML-DevCell">
		<xsl:element name="DevCell">
			<xsl:copy-of select="@*"/>			
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-NeuralNetwork-ErrorWeights">
		<xsl:element name="ErrorWeights">
			<xsl:apply-templates select="X-IMML-XTProps"/>			
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-XTProps">
		<xsl:element name="XTProps">
			<xsl:apply-templates select="X-IMML-Property"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-Property">
		<xsl:element name="Property">
			<xsl:copy-of select="@name"/>
			<xsl:copy-of select="@value"/>
			<xsl:if test="@complex = 'true'">
				<xsl:copy-of select="@complex"/>
				<xsl:apply-templates select="X-IMML-Complex"/>
			</xsl:if>	
			<xsl:apply-templates select="X-IMML-Property"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-Complex">
		<xsl:element name="Complex">
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
		
	<!-- ClusteringModel -->
	<xsl:template match="ClusteringModel">
		<xsl:element name="ClusteringModel">
			<xsl:attribute name="modelName"><xsl:value-of select="@modelName"/></xsl:attribute>
			<xsl:attribute name="numberOfClusters"><xsl:value-of select="@numberOfClusters"/></xsl:attribute>
			<xsl:attribute name="numberOfClusteringFields"><xsl:value-of select="count(ClusteringField)"/></xsl:attribute>
			<xsl:apply-templates select="MiningSchema"/>
			<xsl:apply-templates select="ClusteringField"/>
			<xsl:apply-templates select="Cluster"/>
			<!-- Look at the ClusteringField similarityScale to sort out the InputScaling -->
			<!-- We store the column stdDev and use it for scaling -->
			<xsl:element name="InputScaling">
				<xsl:for-each select="ClusteringField">
					<xsl:element name="InputScalingField">
						<xsl:attribute name="data"><xsl:value-of select="@similarityScale"/></xsl:attribute>
					</xsl:element>
				</xsl:for-each>
			</xsl:element>
			
			<!-- Get the Total element info from an extension -->
			<xsl:element name="Total">
				<xsl:for-each select="Extension/X-IMML-XTProps/X-IMML-Property[@name='Total']">
					<xsl:attribute name="size"><xsl:value-of select="X-IMML-Property[@name='size']/@value"/></xsl:attribute>
					<xsl:attribute name="sumSq"><xsl:value-of select="X-IMML-Property[@name='sumSq']/@value"/></xsl:attribute>
					<xsl:attribute name="sumSqOverSize"><xsl:value-of select="X-IMML-Property[@name='sumSqOverSize']/@value"/></xsl:attribute>
				</xsl:for-each>
			</xsl:element>
		</xsl:element>
	</xsl:template>
	<xsl:template match="ClusteringField">
		<xsl:variable name="index" select="position()"/>
		<xsl:element name="ClusteringField">
			<xsl:attribute name="field"><xsl:value-of select="@field"/></xsl:attribute>
			<xsl:attribute name="compareFunction">squaredEuclidean</xsl:attribute>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Cluster">
		<xsl:element name="Cluster">
			<xsl:attribute name="name"><xsl:value-of select="@name"/></xsl:attribute>
			<xsl:attribute name="number"><xsl:number value="position()" format="01"/></xsl:attribute>

			<xsl:attribute name="sizeInPercent"><xsl:value-of select="Extension/X-IMML-XTProps/X-IMML-Property[@name='sizeInPercent']/@value"/></xsl:attribute>	
			<xsl:attribute name="size"><xsl:value-of select="Extension/X-IMML-XTProps/X-IMML-Property[@name='size']/@value"/></xsl:attribute>		
			<xsl:attribute name="withinSS"><xsl:value-of select="Extension/X-IMML-XTProps/X-IMML-Property[@name='withinSS']/@value"/></xsl:attribute>
			<xsl:attribute name="withinSSOverSize"><xsl:value-of select="Extension/X-IMML-XTProps/X-IMML-Property[@name='withinSSOverSize']/@value"/></xsl:attribute>
			<xsl:element name="Center">
				<!-- Create CenterField elements in IMML for the numbers in the Array in PMML -->
				<xsl:call-template name="SplitClusterArray">
					<xsl:with-param name="string"><xsl:value-of select="normalize-space(Array/text())"/></xsl:with-param>
				</xsl:call-template>
			</xsl:element>
		</xsl:element>
	</xsl:template>

	<!-- Template to split the Array of Cluster centers into separate CenterField elements -->
	<xsl:template name="SplitClusterArray">
		<xsl:param name="string" select="''" />
	   	<xsl:param name="separator" select="' '" />

		<xsl:choose>
		      <xsl:when test="contains($string, $separator)">
				<!-- Create a CenterField element for the first number in the string -->
		      		<xsl:element name="CenterField">
		      			<xsl:attribute name="data"><xsl:value-of select="substring-before($string, $separator)"/></xsl:attribute>
		      		</xsl:element>
		      		<!-- Call the template on the rest of the string -->
				<xsl:call-template name="SplitClusterArray">
					<xsl:with-param name="string"><xsl:value-of select="substring-after($string, $separator)"/></xsl:with-param>
				</xsl:call-template>
	      		</xsl:when>
			<xsl:otherwise>
				<!-- Create a CenterField for the single number in the string -->
		      		<xsl:element name="CenterField">
		      			<xsl:attribute name="data"><xsl:value-of select="$string"/></xsl:attribute>
		      		</xsl:element>			
			</xsl:otherwise>
		</xsl:choose>
	</xsl:template>
	
	<!-- GeneralRegressionModel -->
	<xsl:template match="GeneralRegressionModel">
		<xsl:element name="GeneralRegressionModel">
			<xsl:attribute name="targetVariableName"><xsl:value-of select="@targetVariableName"/></xsl:attribute>
			<xsl:attribute name="modelType"><xsl:value-of select="@modelType"/></xsl:attribute>

			<xsl:apply-templates select="FactorList"/>
			<xsl:apply-templates select="CovariateList"/>
			<xsl:apply-templates select="ParameterList"/>
			<xsl:apply-templates select="PPMatrix"/>
			<xsl:apply-templates select="PCovMatrix"/>
			<xsl:apply-templates select="ParamMatrix"/>
			
			<xsl:apply-templates select="Extension/X-IMML-DevMatrix"/>
			<xsl:apply-templates select="Extension/X-IMML-GeneralRegressionModel-CoefCorr"/>
			<xsl:apply-templates select="Extension/X-IMML-GeneralRegressionModel-Importance"/>
			<xsl:apply-templates select="Extension/X-IMML-GeneralRegressionModel-RMatrix"/>
			<xsl:apply-templates select="Extension/X-IMML-GeneralRegressionModel-QTYMatrix"/>
			<xsl:apply-templates select="Extension/X-IMML-GeneralRegressionModel-ModelExtraInfo"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="ParameterList">
		<xsl:element name="ParameterList">
			<xsl:apply-templates select="Parameter"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Parameter">
		<xsl:element name="Parameter">
			<xsl:attribute name="name"><xsl:value-of select="@name"/></xsl:attribute>
		</xsl:element>
	</xsl:template>
	<xsl:template match="FactorList">
		<xsl:element name="FactorList">
			<xsl:apply-templates select="Predictor"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Predictor">
		<xsl:element name="Predictor">
			<xsl:attribute name="name"><xsl:value-of select="@name"/></xsl:attribute>
		</xsl:element>
	</xsl:template>
	<xsl:template match="CovariateList">
		<xsl:element name="CovariateList">
			<xsl:apply-templates select="Predictor"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="PPMatrix">
		<xsl:element name="PPMatrix">
			<xsl:apply-templates select="PPCell"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="PPCell">
		<xsl:element name="PPCell">
			<xsl:attribute name="value"><xsl:value-of select="@value"/></xsl:attribute>
			<xsl:attribute name="predictorName"><xsl:value-of select="@predictorName"/></xsl:attribute>
			<xsl:attribute name="variableName"><xsl:value-of select="@parameterName"/></xsl:attribute>
		</xsl:element>
	</xsl:template>
	<xsl:template match="PCovMatrix">
		<xsl:element name="PCovMatrix">
			<xsl:apply-templates select="PCovCell"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="PCovCell">
		<xsl:element name="PCovCell">
			<xsl:attribute name="pRow"><xsl:value-of select="@pRow"/></xsl:attribute>
			<xsl:attribute name="pCol"><xsl:value-of select="@pCol"/></xsl:attribute>
			<xsl:attribute name="tRow"><xsl:value-of select="@tRow"/></xsl:attribute>
			<xsl:attribute name="tCol"><xsl:value-of select="@tCol"/></xsl:attribute>
			<xsl:attribute name="value"><xsl:value-of select="@value"/></xsl:attribute>
			<xsl:attribute name="targetCategory"><xsl:value-of select="@targetCategory"/></xsl:attribute>
		</xsl:element>
	</xsl:template>
	<xsl:template match="ParamMatrix">
		<xsl:element name="ParamMatrix">
			<xsl:apply-templates select="PCell"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="PCell">
		<xsl:element name="PCell">
			<xsl:attribute name="targetCategory"><xsl:value-of select="@targetCategory"/></xsl:attribute>
			<xsl:attribute name="parameterName"><xsl:value-of select="@parameterName"/></xsl:attribute>
			<xsl:attribute name="beta"><xsl:value-of select="@beta"/></xsl:attribute>
			<xsl:attribute name="df"><xsl:value-of select="@df"/></xsl:attribute>
			
			<xsl:variable name="paramName"><xsl:value-of select="@parameterName"/></xsl:variable>
			<xsl:variable name="curProp"></xsl:variable>
			<xsl:attribute name="stdErr"><xsl:value-of select="../../Extension/X-IMML-GeneralRegressionModel-ParamMatrixExtraAttributes/X-IMML-XTProps/X-IMML-Property[@value=$paramName]/X-IMML-Property[@name='stdErr']/@value"/></xsl:attribute>		
			<xsl:attribute name="t"><xsl:value-of select="../../Extension/X-IMML-GeneralRegressionModel-ParamMatrixExtraAttributes/X-IMML-XTProps/X-IMML-Property[@value=$paramName]/X-IMML-Property[@name='t']/@value"/></xsl:attribute>
			<xsl:attribute name="Pr"><xsl:value-of select="../../Extension/X-IMML-GeneralRegressionModel-ParamMatrixExtraAttributes/X-IMML-XTProps/X-IMML-Property[@value=$paramName]/X-IMML-Property[@name='Pr']/@value"/></xsl:attribute>
	</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-GeneralRegressionModel-CoefCorr">
		<xsl:element name="CoefCorr">
			<xsl:copy-of select="@*"/>
			<xsl:apply-templates select="X-IMML-GeneralRegressionModel-CorrCell"/>
		</xsl:element>		
	</xsl:template>
	<xsl:template match="X-IMML-GeneralRegressionModel-CorrCell">
		<xsl:element name="CorrCell">
			<xsl:copy-of select="@*"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-GeneralRegressionModel-Importance">
		<xsl:element name="Importance">
			<xsl:copy-of select="@*"/>
			<xsl:apply-templates select="X-IMML-GeneralRegressionModel-Effect"/>
		</xsl:element>		
	</xsl:template>
	<xsl:template match="X-IMML-GeneralRegressionModel-Effect">
		<xsl:element name="Effect">
			<xsl:copy-of select="@*"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-GeneralRegressionModel-RMatrix">
		<xsl:element name="RMatrix">
			<xsl:copy-of select="@*"/>
			<xsl:apply-templates select="X-IMML-GeneralRegressionModel-Row"/>
		</xsl:element>		
	</xsl:template>
	<xsl:template match="X-IMML-GeneralRegressionModel-QTYMatrix">
		<xsl:element name="QTYMatrix">
			<xsl:copy-of select="@*"/>
			<xsl:apply-templates select="X-IMML-GeneralRegressionModel-Row"/>
		</xsl:element>		
	</xsl:template>
	<xsl:template match="X-IMML-GeneralRegressionModel-Row">
		<xsl:element name="Row">
			<xsl:copy-of select="@*"/>
			<xsl:apply-templates select="X-IMML-GeneralRegressionModel-Col"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-GeneralRegressionModel-Col">
		<xsl:element name="Col">
			<xsl:copy-of select="@*"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-GeneralRegressionModel-ModelExtraInfo">
		<xsl:element name="ModelExtraInfo">
			<xsl:apply-templates select="X-IMML-GeneralRegressionModel-Family"/>
			<xsl:apply-templates select="X-IMML-GeneralRegressionModel-Link"/>
			<xsl:apply-templates select="X-IMML-GeneralRegressionModel-Mean"/>
			<xsl:apply-templates select="X-IMML-GeneralRegressionModel-Var"/>
			<xsl:apply-templates select="X-IMML-GeneralRegressionModel-Iterations"/>
		</xsl:element>		
	</xsl:template>
	<xsl:template match="X-IMML-GeneralRegressionModel-Family">
		<xsl:element name="Iterations">
			<xsl:copy-of select="@*"/>
		</xsl:element>		
	</xsl:template>
	<xsl:template match="X-IMML-GeneralRegressionModel-Link">
		<xsl:element name="Iterations">
			<xsl:copy-of select="@*"/>
		</xsl:element>		
	</xsl:template>
	<xsl:template match="X-IMML-GeneralRegressionModel-Mean">
		<xsl:element name="Iterations">
			<xsl:copy-of select="@*"/>
		</xsl:element>		
	</xsl:template>
	<xsl:template match="X-IMML-GeneralRegressionModel-Var">
		<xsl:element name="Iterations">
			<xsl:copy-of select="@*"/>
		</xsl:element>		
	</xsl:template>
	<xsl:template match="X-IMML-GeneralRegressionModel-Iterations">
		<xsl:element name="Iterations">
			<xsl:copy-of select="@*"/>
		</xsl:element>		
	</xsl:template>
	<!-- NaiveBayes -->
	<xsl:template match="NaiveBayesModel">
		<xsl:element name="NaiveBayesModel">
			<xsl:element name="XTProps">
				<!-- First Property is overall totals for dependent column -->
				<xsl:element name="Property">
					<xsl:attribute name="name">dependentColumn</xsl:attribute>
					<xsl:attribute name="value">
						<xsl:value-of select="BayesOutput/@fieldName"/>
					</xsl:attribute>
					<xsl:for-each select="BayesOutput/TargetValueCounts/TargetValueCount">
						<xsl:element name="Property">
							<!-- TODO: May need to add something for missings -->
							<xsl:attribute name="name">
								<xsl:value-of select="@value"/>
							</xsl:attribute>
							<xsl:attribute name="value">
								<xsl:value-of select="@count"/>
							</xsl:attribute>											
						</xsl:element>					
					</xsl:for-each>
					<xsl:element name="Property">
						<xsl:attribute name="name">total</xsl:attribute>
						<xsl:attribute name="value">
							<xsl:value-of select="sum(BayesOutput/TargetValueCounts/TargetValueCount/@count)"/>
						</xsl:attribute>							
					</xsl:element>
				</xsl:element>
				<!-- Second Property is counts for each dependent level within each predictor category -->
				<xsl:element name="Property">
					<xsl:attribute name="name">counts</xsl:attribute>
					<xsl:for-each select="BayesInputs/BayesInput">
						<xsl:element name="Property">
							<xsl:attribute name="name">
								<xsl:value-of select="@fieldName"/>
							</xsl:attribute>
							<xsl:for-each select="PairCounts">
								<xsl:element name="Property">
									<xsl:attribute name="name">
										<xsl:value-of select="@value"/>
									</xsl:attribute>
									<xsl:for-each select="TargetValueCounts/TargetValueCount">
										<xsl:element name="Property">
											<xsl:attribute name="name">
												<xsl:value-of select="@value"/>
											</xsl:attribute>
											<xsl:attribute name="value">
												<xsl:value-of select="@count"/>
											</xsl:attribute>
										</xsl:element>
									</xsl:for-each>						
								</xsl:element>
							</xsl:for-each>	
						</xsl:element>
					</xsl:for-each>				
				</xsl:element>
			</xsl:element>
		</xsl:element>
	</xsl:template>
	<!-- PrincipalComponents -->
	<xsl:template match="X-IMML-PrincipalComponentsModel">
		<xsl:element name="PrincipalComponentsModel">
			<xsl:copy-of select="@*"/>
			<xsl:apply-templates select="MiningSchema"/>
			<xsl:apply-templates select="X-IMML-PrincipalComponentsModel-Parameters"/>
			<xsl:apply-templates select="X-IMML-PrincipalComponentsModel-TermIndicies"/>
			<xsl:apply-templates select="X-IMML-PrincipalComponentsModel-PrinComp"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-PrincipalComponentsModel-Parameters">
		<xsl:element name="Parameters">
			<xsl:copy-of select="@*"/>
			<xsl:apply-templates select="X-IMML-PrincipalComponentsModel-Param"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-PrincipalComponentsModel-TermIndicies">
		<xsl:element name="TermIndicies">
			<xsl:copy-of select="@*"/>
			<xsl:apply-templates select="X-IMML-PrincipalComponentsModel-Index"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-PrincipalComponentsModel-PrinComp">
		<xsl:element name="PrinComp">
			<xsl:copy-of select="@*"/>
			<xsl:apply-templates select="X-IMML-PrincipalComponentsModel-Component"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-PrincipalComponentsModel-Param">
		<xsl:element name="Param">
			<xsl:copy-of select="@*"/>
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-PrincipalComponentsModel-Index">
		<xsl:element name="Index">
			<xsl:copy-of select="@*"/>
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-PrincipalComponentsModel-Component">
		<xsl:element name="Component">
			<xsl:copy-of select="@*"/>
			<xsl:apply-templates select="X-IMML-PrincipalComponentsModel-Loading"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-PrincipalComponentsModel-Loading">
		<xsl:element name="Loading">
			<xsl:copy-of select="@*"/>
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
	<!-- SurvivalModel -->
	<xsl:template match="X-IMML-SurvivalModel">
		<xsl:element name="SurvivalModel">
			<xsl:attribute name="type"><xsl:value-of select="@type"/></xsl:attribute>
			<xsl:apply-templates select="*"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-SurvivalModel-SurvResponse">
		<xsl:element name="SurvResponse">
			<xsl:copy-of select="@*"/>
			<xsl:apply-templates select="*"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-SurvivalModel-EventCodes">
		<xsl:element name="EventCodes">
			<xsl:copy-of select="@*"/>
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-SurvivalModel-LinearModel">
		<xsl:element name="LinearModel">
			<xsl:copy-of select="@*"/>
			<xsl:apply-templates select="*"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-SurvivalModel-Term">
		<xsl:element name="Term">
			<xsl:copy-of select="@*"/>
			<xsl:apply-templates select="*"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-SurvivalModel-TermVar">
		<xsl:element name="TermVar">
			<xsl:copy-of select="@*"/>
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-SurvivalModel-Intercept">
		<xsl:element name="Intercept">
			<xsl:copy-of select="@*"/>
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-SurvivalModel-Weights">
		<xsl:element name="Weights">
			<xsl:copy-of select="@*"/>
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-SurvivalModel-Strata">
		<xsl:element name="Strata">
			<xsl:copy-of select="@*"/>
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-SurvivalModel-Coefficients">
		<xsl:element name="Coefficients">
			<xsl:copy-of select="@*"/>
			<xsl:apply-templates select="*"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-SurvivalModel-Coef">
		<xsl:element name="Coef">
			<xsl:copy-of select="@*"/>
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-SurvivalModel-Hessian">
		<xsl:element name="Hessian">
			<xsl:copy-of select="@*"/>
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-SurvivalModel-SurvModelStats">
		<xsl:element name="SurvModelStats">
			<xsl:copy-of select="@*"/>
			<xsl:apply-templates select="*"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-SurvivalModel-LikelihoodRatio">
		<xsl:element name="LikelihoodRatio">
			<xsl:copy-of select="@*"/>
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-SurvivalModel-BaselineSurvival">
		<xsl:element name="BaselineSurvival">
			<xsl:copy-of select="@*"/>
			<xsl:apply-templates select="*"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-SurvivalModel-Time">
		<xsl:element name="Time">
			<xsl:copy-of select="@*"/>
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-SurvivalModel-Failures">
		<xsl:element name="Failures">
			<xsl:copy-of select="@*"/>
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-SurvivalModel-RiskSet">
		<xsl:element name="RiskSet">
			<xsl:copy-of select="@*"/>
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-SurvivalModel-Hazard">
		<xsl:element name="Hazard">
			<xsl:copy-of select="@*"/>
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="X-IMML-SurvivalModel-VarHazard">
		<xsl:element name="VarHazard">
			<xsl:copy-of select="@*"/>
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
</xsl:stylesheet>
