## These functions are for use with GraphServlet, which
##   calls S-PLUS to generate graphlets on demand.
## This version is designed for use with Analytic Server 2.0.
##
## Author:  Gary Nelson, gnelson@insightful.com
## Version: 18 April 2001

################################################################
## GraphletCreate
################################################################
##
## The GraphletCreate function opens a Graphlet file with the
##   given filename, calls the function specified by the cmd
##   parameter to draw the graph, and closes the file.  All
##   other parameters are passed to the cmd function.
##
GraphletCreate <- function(cmd, ...)
{
	## Get a name for a temporary file.
	chFilename <- tempfile()

	## Open a Graphlet file.
	java.graph(file = chFilename, format = "SPJ")

	## Evaluate the function cmd.
	## Save the return value to return from this function.
	chTitle <- eval(substitute(cmd))(...)

	## Close the file.
	dev.off()

	## Return a vector of two strings, the Graphlet title
	##   and the filename.
	chReturn <- character(2)
	chReturn[1] <- ifelse(is.character(chTitle), chTitle[1], "")
	chReturn[2] <- chFilename

	## Return a title for the Web page.
	return(chReturn)
}

################################################################
## MakeAction
################################################################
##
## The MakeAction function simplifies using live Graphlets.
## This version is for servlets.  Live Graphlets with StatServer
##   probably require a slightly different version.
## The function takes as input the name and arguments for an
##   S-PLUS command and outputs the XML string a graphlet needs
##   to call that function in response to a mouse click.
## The function is vectorized; if any of the arguments are
##   vectors, the output is a vector of XML strings.
## Thanks to Tim Hesterberg, who figured out a good way to
##   write this function.
##
## Usage example:
## Instead of constructing your own string:
##		actions = paste('<link ',
##			'href="GraphServlet?cmd=GtHistogram',
##				'&nMonth=', nMonth,
##				'&chState=\'', chState, '\'',
##				'&nEntity=', nEntity,
##				'&nSubm=', nSubm,
##				'&nOverlayWireCenter=', nWireCentHighlight,
##				'" ',
##			'target="_top"/>', sep = "")
##
## call MakeAction:
##		actions = MakeAction(GtHistogram,
##			nMonth = nMonth,
##			chState = chState,
##			nEntity = nEntity,
##			nSubm = nSubm,
##			nOverlayWireCenter = nWireCentHighlight)
##
MakeAction <- function(cmd, ...)
{
	## Create the parameters portion of the action string.
	if (nargs() - !missing(cmd)) {
		## We have some arguments to pass.
		## Make a data frame, one column for the values of
		##   each named argument.
		dfArgVals <- data.frame(...)

		## In front of each argument value paste the argument
		##   name (with "=" as a separator).
		lstNamedArgs <- lapply(names(dfArgVals),
			function(n, df)
			{
				if (is.character(df[[n]])) {
					paste(n, "='", df[[n]], "'", sep = "")
				}
				else if (is.factor(df[[n]])) {
					paste(n, "='", as.character(df[[n]]), "'",
						sep = "")
				}
				else {
					paste(n, df[[n]], sep = "=")
				}
			},
			df = dfArgVals)

		## Use lstNamedArgs as an argument list to the paste function.
		## First add another argument, the separator "&".
		lstNamedArgs[["sep"]] <- "&"
		chArgStrings <- do.call("paste", args = lstNamedArgs)

		## Add an initial ampersand.
		chArgStrings <- paste("&", chArgStrings, sep = "")
	}
	else {
		chArgStrings <- ""
	}

	## Construct the final strings.
	paste(
		## initial text, always the same
		"<link href=\"GraphServlet?",
		## the S-PLUS function name
		"cmd=", substitute(cmd),
		## arguments to the S-PLUS function
		chArgStrings,
		## final text, always the same
		"\" target=\"_top\"/>",
		sep="")
}
