map.counties <- function(state="Florida",vartoplot="Pop1990",county="palm beach"){


###########################################################################
###
### map.counties: Produces a colormap graphlet for counties from a single State.
###              
###             
### Arguments:
###
### state:     Name of the state to plot.
###
### vartoplot: Column name from the counties dataframe. This is the variable
###            that will be plotted in the colorpmap. Note. The dataset for
###            the colormap is currently hardwired to the counties dataset
###            which is included in data.sdd 
###
### county:    Not used ?
###
###########################################################################
 
  ## Strip out any "%20" string components that may be present in vartoplot.
  ## These originate from white spaces in labels sent to java.identify()
  state <- token(source=state)
  

  ##  Subset counties/cities dataframes down to counties/cities for the selected State
  counties.idx <- as.character(counties$State.name) == state
  cities.idx <- as.character(cities$State.name) == state
  counties.in.s <- counties[counties.idx,]
  cities.in.s <- cities[cities.idx,]

  ## Set the java color scheme to trellis colors
  java.set.colorscheme(java.colorscheme.trellis)
  
  ## Get the names of the countiess from mapping the "county" map for the selected state.
  name.polys <- map("county", regions=state, namesonly=T,plot=F)

  ## Coding required to strip out the county names from name.polys
  idx <- regexpr(pat=":",name.polys) == -1
  name.polys[idx] <- paste(name.polys[idx],":x",sep="")
  tmp <- unpaste(name.polys, sep=":")
  tmp <- unpaste(tmp[[1]], sep=",")
  
  ## Match the county names from the map with the row names in subset county df.
  poly.county.id <- match(tmp[[2]],as.character(counties.in.s$Name))

  ## Grab the value to be plotted in the colormap
  dat <- counties.in.s[,vartoplot]

  ## Fix colors
  dat.scaled <- (dat-min(dat, na.rm=T))/diff(range(dat, na.rm=T))
  cols <- trellis.par.get("regions")$col
  ncols <- length(cols)
  thecols <- cols[round(1+dat.scaled*(ncols-1))]
  thecols[is.na(thecols)] <- 0
  cols.to.plot <- thecols[poly.county.id]
  
## Produce the colormap
  map("county", regions=state, fill=T, resolution=0,color=cols.to.plot)

  ## Allow further plotting to overlay the initial map. 
  par(new=T)
  ## Replot the county borders
  map("county", regions=state, resolution=0)
  
  ## Add a title to the plot
  title(main=paste(vartoplot, " by county for ",state, " state")) 
  
  ## Add a legend to the plot
  image.legend(dat,x=par()$usr[1],y=par()$usr[3], size=c(1,0.25),hor=T)
   
  ## Add a key to the plot
  key(x=par()$usr[2],y=par()$usr[3],corner=c(1,0),
      points=list(pch=c(0,16), col=c(8,1)),text=list(c("Counties","Cities")),
      border=1)

  ## Plot points for county centroids
  points(counties.in.s$X, counties.in.s$Y, pch=0,col=8)

  ## Plot points for cities within the state
  points(cities.in.s$X, cities.in.s$Y, pch=16,col=1)

  ## Set up possible names for change of colormap / variable for histograms 
  pos.names <- c("Pop1990", "Pop1999", "Pct.0.4","Pct.65up","Pct.F" )

  ## Remove the "current"  vartoplot variable from the above list
  change.names.idx <- !is.element(pos.names,vartoplot)
  menu.options <- pos.names[change.names.idx]

  menu <- list()


  ## Set up the action areas and commands to execute following selevtion
  ## There are two components built for all states. (Color map changes if 
  ## a county centroid is selected. Histogram plots if a City is selected.
  ## For Florida, ther eis the addition of the Voting options if a county
  ## centroid is selected.
  
  ## County Centroid Actions
  for (name in menu.options){
    menu[[name]]<- paste('<menuitem label="',name,'">',
			 MakeAction(map.counties,state=state, 
				    county=as.character(counties.in.s$Name), vartoplot=name), 
			 '</menuitem>')}

  menupaste <- apply(data.frame(menu),1,paste, collapse="")


  ## If Florida is the chosen State.. allow access to Florida vote analytics.
  if(state == "Florida"){

    FloridaAction <- paste(
			   '<menuitem label="State-Wide Bush-Buchanan Regression">',
			   MakeAction(florida.votes.scatter,county=as.character(counties.in.s$Name)),
			   '</menuitem>',
			   '<menuitem label="Barplot of Votes in County">',
			   MakeAction(florida.barplot,county=as.character(counties.in.s$Name)),
			   '</menuitem>')

    menupaste <- paste(menupaste,FloridaAction)
  }

  menupaste <- paste("<menu>",menupaste,"</menu>")

  java.identify(counties.in.s$X, counties.in.s$Y,
		labels=paste("County:", as.character(counties.in.s$Name)), actions = menupaste)


  ## Set up City actions
  java.identify(cities.in.s$X, cities.in.s$Y, labels=paste("City:", cities.in.s$City.name),
		actions = paste(
		  '<menu title="', 'Title', '">',
		  '<menuitem label="Histograms">',
		  MakeAction(hist.cities,state=state,city=as.character(cities.in.s$City.name)),
		  '</menuitem>',
		  '</menu>'))
}

