/*
	Connect/C++ : Copyright (c) 2001, 2006 Insightful Corp.
	All rights reserved.
	Version 6.0: 2001

	spcall.cxx: implementation of the CSPcall class.
*/
#include "S_y_tab.h"
#include "sconnect.h"
#include "spalcfrm.h"
#include "string.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////
CSPcall::CSPcall()
: CSPlanguage()
{
}
//Copy constructor 
CSPcall::CSPcall(const CSPcall& sObject)
: CSPlanguage()
{
	Attach(&sObject, sObject.GetTryToFreeOnDetach());
}
//Construct from a base class object
CSPcall::CSPcall(const CSPobject& sObject)
: CSPlanguage()
{	
	Attach(&sObject, sObject.GetTryToFreeOnDetach());
}
//Construct from a valid S object
CSPcall::CSPcall(s_object* ps_object, BOOL bTryToFreeOnDetach)
: CSPlanguage()
{
	Attach(ps_object, bTryToFreeOnDetach);
}
//Assigment from the same class
CSPcall& CSPcall::operator=(const CSPcall& sObject)
{
	Attach(&sObject, TRUE);
	return *this;
}
//Assigment from the base class
CSPcall& CSPcall::operator=(const CSPobject& sObject)
{
	Attach(&sObject, TRUE);
	return *this;
}
//Assigment from the S object
CSPcall& CSPcall::operator=(s_object* ps_object)
{
	Attach(ps_object, FALSE);
	return *this;
}
//The destructor
CSPcall::~CSPcall()
{
}

CSPcall::CSPcall(const char* pszSFunctionName)
:CSPlanguage()
{
	Create(pszSFunctionName);
}
CSPcall::CSPcall(const char* pszSFunctionName, s_object* ps_arg)
:CSPlanguage()
{
	Create(pszSFunctionName, ps_arg);
}
//Use the first elem of char. vector as function name.
CSPcall::CSPcall(const CSPcharacter& scharFunctionNane)
: CSPlanguage()
{
	Create(scharFunctionNane[0]);
}
//Construct a call with a whole funtion definition.
CSPcall::CSPcall(const CSPfunction& sFunction)
: CSPlanguage()
{
	Create(sFunction);
}
CSPcall::CSPcall(const CSPfunction& sFunction, s_object* ps_arg)
:CSPlanguage()
{
	Create(sFunction, ps_arg);
}
//Construct a call with a list of arguments.
CSPcall::CSPcall(const char* pszSFunctionName, const CSPlist& slistArgs)
: CSPlanguage()
{
	Create(pszSFunctionName, slistArgs);
}
//Construct a call with a list of arguments.
CSPcall::CSPcall(const char* pszSFunctionName, const CSPlistNamed& slistNamedArgs)
: CSPlanguage()
{
	CSPlist slistArgs(slistNamedArgs);
	Create(pszSFunctionName, slistArgs);
}

//Construct a call with a list of arguments.
CSPcall::CSPcall(const char* pszSFunctionName, const CSPobject& sArg)
: CSPlanguage()
{
	Create(pszSFunctionName, &sArg);
}

//Construct a call with a whole funtion definition and a list of arguments.
CSPcall::CSPcall(const CSPfunction& sFunction, const CSPlist& slistArgs)
: CSPlanguage()
{
	Create(sFunction, slistArgs);
}
//Construct a call with a whole funtion definition and a list of arguments.
CSPcall::CSPcall(const CSPfunction& sFunction, const CSPlistNamed& slistNamedArgs)
: CSPlanguage()
{
	CSPlist slistArgs(slistNamedArgs);
	Create(sFunction, slistArgs);
}

//Create a call with a funtion name and a single arg.
CSPcall::CSPcall(const CSPfunction& sFunction, const CSPobject& sArg)
: CSPlanguage()
{
	Create(sFunction, &sArg);
}

void CSPcall::Attach(s_object *ps_object, BOOL bTryToFreeOnDetach)
{
	if(ps_object == NULL)
		return;
	s_object* ps_call = NULL;

	CSPevaluator sEvaluator;
	switch(ps_object->mode)
	{
	case S_MODE_FUN_CALL:
		ps_call = ps_object;
		break;
	case S_MODE_CHAR:
		ps_call = ::alcf(CHARACTER_VALUE(ps_object), NULL_ENTRY, S_evaluator);
		break;
	case S_MODE_FUN_DEF:
	{
		ps_call = sEvaluator.alcvec(S_MODE_FUN_CALL, 1L); 
		SET_ELEMENT(ps_call, 0L, sEvaluator.CopyForWrite(ps_object)); //Just to be safe: don't share this small block of mem.
		break;
	}
	default: //Assume class 'call'
		BaseCoerceAttach(ps_object, MAKE_CLASS("call"), bTryToFreeOnDetach);
		return;
		break;
	}
	CSPlanguage::Attach(sEvaluator.CloneIfNeeded(ps_call), bTryToFreeOnDetach); 
}

//Create a function call for the object of class char, function or call
BOOL CSPcall::Create(const s_object* ps_object)
{
	if(ps_object == NULL)
		return FALSE;
	s_object* ps_fun = const_cast<s_object*>(ps_object);

	CSPevaluator sEvaluator;
	switch(ps_fun->mode)
	{
	case S_MODE_CHAR:
		Attach(::alcf(CHARACTER_VALUE(ps_fun), NULL_ENTRY, S_evaluator));
		break;
	case S_MODE_FUN_DEF:
	{
		s_object* ps_call = ::alcvec(S_MODE_FUN_CALL, 1L, S_evaluator); 
		SET_ELEMENT(ps_call, 0L, ps_fun);
		Attach(ps_call);
		break;
	}
	default: //Assume class 'call'
		CSPlanguage::BaseCoerceAttach(ps_fun, "call");
	}
	return TRUE;
}

BOOL CSPcall::Create(const char* pszSFunctionName)
{
		//Set allocated frame to be SPL_WORKING_FRAME and save previous allocated frame
		CSPallocFrame sAllocFrame;
		//Construct an S object of class "call"
		Attach( ::alcf((char*) pszSFunctionName, NULL_ENTRY, S_evaluator));
		return TRUE;
}
//Create a call with a whole funtion definition.
BOOL CSPcall::Create(const CSPfunction& sFunction)
{
	if(!sFunction.IsValid())
		return FALSE;
	//Set allocated frame to be SPL_WORKING_FRAME if the evaluator is close
	CSPallocFrame sAllocFrame;
	s_object* ps_call = sAllocFrame.alcvec(S_MODE_FUN_CALL, 1L); 
	SET_ELEMENT(ps_call, 0L, &sFunction);
	Attach(ps_call);
	return TRUE;
}

//Create a call with a whole funtion definition and a list of arguments.
BOOL CSPcall::Create(const CSPfunction& sFunction, const CSPlist& slistArgs)
{
	if(!sFunction.IsValid())
		return FALSE;
	//Set allocated frame to be SPL_WORKING_FRAME if the evaluator is close
	CSPallocFrame sAllocFrame;
	s_object* ps_call = sAllocFrame.alcvec(S_MODE_FUN_CALL, slistArgs.GetLength()+1L); 
	SET_ELEMENT(ps_call, 0L, &sFunction);
	for(long n=0; n < slistArgs.GetLength(FALSE); ++n)
		SET_ELEMENT(const_cast<s_object*>(ps_call), n+1, const_cast<s_object*>(slistArgs.GetAt(n))); 
	Attach(ps_call);
	return TRUE;
}

//Create a call with a funtion name and a list of arguments.
BOOL CSPcall::Create(const char* pszSFunctionName, const CSPlist& slistArgs)
{
	if(!pszSFunctionName || !*pszSFunctionName)
		return FALSE;
	//Set allocated frame to be SPL_WORKING_FRAME and save previous allocated frame
	CSPallocFrame sAllocFrame;
	s_object* ps_call = sAllocFrame.alcvec(S_MODE_FUN_CALL, slistArgs.GetLength()+1L); 	
	SET_ELEMENT(ps_call, 0L, ::alc_name(pszSFunctionName, S_evaluator));
	for(long n=0; n < slistArgs.GetLength(FALSE); ++n)
		SET_ELEMENT(const_cast<s_object*>(ps_call), n+1, const_cast<s_object*>(slistArgs.GetAt(n))); 
	Attach(ps_call);
	return TRUE;
}

BOOL CSPcall::Create(const char* pszSFunctionName, s_object* ps_arg)
{
		//Set allocated frame to be SPL_WORKING_FRAME and save previous allocated frame
		CSPallocFrame sAllocFrame;
		//Construct an S object of class "call"
		Attach( ::alcf((char*) pszSFunctionName, ps_arg, S_evaluator));
		return TRUE;
}
//Create a call with a funtion name and a list of arguments.
BOOL CSPcall::Create(const char* pszSFunctionName, const CSPobject& sArg)
{
	return Create(pszSFunctionName, &sArg);
}
//Create a call with a whole funtion definition and a list of arguments.
BOOL CSPcall::Create(const CSPfunction& sFunction, const CSPobject& sArg)
{
	return Create(sFunction, &sArg);
}
//Create a call with a whole funtion definition and a list of arguments.
BOOL CSPcall::Create(const CSPfunction& sFunction, s_object* ps_arg)
{
	if(!sFunction.IsValid())
		return FALSE;
	//Set allocated frame to be SPL_WORKING_FRAME if the evaluator is close
	CSPallocFrame sAllocFrame;
	s_object* ps_call = sAllocFrame.alcvec(S_MODE_FUN_CALL, 2L); 
	SET_ELEMENT(ps_call, 0L, &sFunction);
	SET_ELEMENT(const_cast<s_object*>(ps_call), 1L, ps_arg); 
	Attach(ps_call);
	return TRUE;
}

//SetArg() set an argument in the given position
void CSPcall::SetArg(s_object* ps_arg, long lPosition, const char* pszName)
{
	//Validating
	Validate(); 
	if((lPosition < 1) || (lPosition > 128))
		SCONNECT_ThrowException("Invalid argument index %d", lPosition);

	if ( !ps_arg )
		SCONNECT_ThrowException("Invalid argument. S-PLUS object is invalid");

	//Set allocated frame to be SPL_WORKING_FRAME and save previous allocated frame
	CSPevaluator sEvaluator;
	
	ps_arg = sEvaluator.CreateNewHeader(ps_arg);
	ps_arg->name=EMPTY_STRING; //Clear its name

	s_object* ps_call = sEvaluator.CopyForWrite(GetPtr());				
	if(ps_call->length < lPosition+1L)
		SET_LENGTH(ps_call, lPosition+1L);

	SET_ELEMENT(ps_call, lPosition, ps_arg);
	if(pszName && *pszName)
		LIST_POINTER(ps_call)[lPosition]->name = (char *)::make_name(const_cast<char*>(pszName), S_evaluator)->text;

	ReAttachAndAssign(ps_call);
}

void CSPcall::SetArg(const char* pszArgument, long lPosition, const char* pszName)
{
	if ( !pszArgument )
		SCONNECT_ThrowException("Invalid character string");
	CSPallocFrame sAllocFrame;
	s_object* ps_arg = sAllocFrame.alcvec(S_MODE_CHAR, 1L);
	//Very direct assign to object with ref count=0.  It should be ok as long as ps_arg is not shared.
	CHARACTER_POINTER(ps_arg)[0] = (char*) pszArgument;  
	SetArg(ps_arg, lPosition, pszName);
}

void CSPcall::SetArg(double dArgument, long lPosition, const char* pszName)
{
	CSPallocFrame sAllocFrame;
	s_object* ps_arg = sAllocFrame.alcvec(S_MODE_DOUBLE, 1L);
	NUMERIC_POINTER(ps_arg)[0] = dArgument;
	SetArg(ps_arg, lPosition, pszName);
}

void CSPcall::SetArg(long lArgument, long lPosition, const char* pszName)
{
	CSPallocFrame sAllocFrame;
	s_object* ps_arg = sAllocFrame.alcvec(S_MODE_INT, 1L);
	INTEGER_POINTER(ps_arg)[0] = lArgument;
	SetArg(ps_arg, lPosition, pszName);
}

void CSPcall::SetArgName(long lPosition, const char* pszName)
{
	Validate();
	if ((lPosition < 1) || (GetLength() < lPosition+1))
		SCONNECT_ThrowException("Invalid argument position %d", lPosition);
	if ( !pszName || !*pszName )
		SCONNECT_ThrowException("Invalid name");
	//TODO: copy before write, but this should be no worst than parts of the engine.
	(*this)->value.tree[lPosition]->name = (char *)make_name(pszName, S_evaluator)->text;
}

//SetArgDirect() set an argument in the given position directly
void CSPcall::SetArgDirect(s_object* ps_arg, long lPosition)
{
	s_object* ps_call = GetPtr();
	LIST_POINTER(ps_call)[lPosition] = ps_arg;
}

CSPobject CSPcall::GetArg(long lPosition, BOOL bValidate) const
{
	if(bValidate)
	{
		if(!IsValid())
			throw SCONNECT_INVALID_SOBJECT;
		else if(lPosition <1 || lPosition > this->GetLength()-1)
			SCONNECT_ThrowException("lPosition is out of range: %d", lPosition);
	}
	return CSPobject((*this)->value.tree[lPosition]);
}

void CSPcall::SetArgs(const CSPlist& slistArgs)
{
	if(!IsValid())
		SCONNECT_ThrowException("Invalide object");
	if(!slistArgs.IsValid()|| slistArgs.GetLength()< 1L)
		return;
	try
	{
		CSPevaluator sEvaluator;
		long lLen = slistArgs.GetLength()+1L;
		
		s_object* ps_call = sEvaluator.CopyForWrite(GetPtr());				
		if(GetLength()< lLen)
			SET_LENGTH(ps_call, lLen);							
		for(long n=0; n < lLen-1L; ++n)
			LIST_POINTER(ps_call)[n+1] = sEvaluator.CreateNewHeader(slistArgs.GetAt(n));

		ReAttachAndAssign(ps_call);
	}
	catch(...)
	{
		SCONNECT_ThrowException("Failed in CSPcall::SetArgs(const CSPlist& slistArgs): unable to set args");
	}
}

void CSPcall::SetArgs(const CSPlistNamed& slistNamedArgs)
{
	CSPlist slistArgs(slistNamedArgs);
	SetArgs(slistArgs);
}

void CSPcall::AddArgs(const CSPlist& slistArgs)
{
	if(!IsValid())
		SCONNECT_ThrowException("Invalide object");
	if(!slistArgs.IsValid()|| slistArgs.GetLength()< 1L)
		return;
	try
	{
		CSPevaluator sEvaluator;
		s_object* ps_call = sEvaluator.CopyForWrite(GetPtr());				
		long len = GetLength(FALSE);
		SET_LENGTH(ps_call, slistArgs.GetLength(FALSE)+len);
		for(long n=0; n < slistArgs.GetLength(); ++n)
			LIST_POINTER(ps_call)[len++] = sEvaluator.CreateNewHeader(slistArgs.GetAt(n));
		ReAttachAndAssign(ps_call);
	}
	catch(...)
	{
		SCONNECT_ThrowException("Failed in CSPcall::SetArgs(const CSPlist& slistArgs): unable to set args");
	}
}
void CSPcall::AddArgs(const CSPlistNamed& snamedArgs)
{
	CSPlist slistArgs(snamedArgs);
	AddArgs(slistArgs);
}

//Eval in the current evaluation frame
CSPobject CSPcall::Eval(void) const
{
	if(!IsValid())
		SCONNECT_ThrowException("object of class CSPcall is not valid"); 

	s_object* ps_value=NULL;
	try
	{ 
		CSPevaluator sEvaluator; //If frame 1 does not exist, open it.
		ps_value= sEvaluator.eval(GetPtr());
		ps_value = sEvaluator.CloneIfNeeded(ps_value);
	}
	catch(...)
	{
		SCONNECT_ThrowException("Failed to evaluate call");
	}
	return CSPobject(ps_value, TRUE);
}

//Evaluate the S-funciton call with its 1st, 2nd, ... arguments specified by the input parameter CSPlist
CSPobject CSPcall::Eval(const CSPlist& slistArgs) const
{
	if(!IsValid())
		SCONNECT_ThrowException("Invalid object of class CSPcall"); 
	if(!slistArgs.IsValid())
		SCONNECT_ThrowException("Invalid input list object"); 

	s_object* ps_value = NULL;
	try
	{
		CSPevaluator sEvaluator;
		s_object* ps_call = sEvaluator.alcvec(S_MODE_FUN_CALL, slistArgs.GetLength()+1L);
		s_object** els    = ps_call->value.tree;
		//Start with function definition
		els[0]= sEvaluator.CreateNewHeader((*this)->value.tree[0]);
		long n;
		for(n=0; n < slistArgs.GetLength(); n++)		
			els[n+1] = sEvaluator.CreateNewHeader(slistArgs.GetAt(n));

		//Evaluate the call object and return result in current or perm. storage frame.
		ps_value = sEvaluator.eval(ps_call);
		ps_value = sEvaluator.CloneIfNeeded(ps_value);

		//free headers created in this function.  Don't free els[0] which the name of object call, it may be shared
		for( n=0; n < GetLength(); n++)		
			sEvaluator.FreeHeader(els[n]);
		//free ps_call but not its children
		sEvaluator.TryToFree(ps_call, FALSE);
	}
	catch(...)
	{
		SCONNECT_ThrowException("Failed to evaluate call");
	}
	return CSPobject(ps_value, TRUE);
}

//Evaluate the S-function call (method) with its 1st, 2nd, ... arguments specified by the input parameters
//Argument names are ignor. 
CSPobject CSPcall::Eval(s_object*  ps_1, s_object*  ps_2, s_object*  ps_3, s_object*  ps_4, s_object*  ps_5,
                        s_object*  ps_6, s_object*  ps_7, s_object*  ps_8, s_object*  ps_9, s_object* ps_10,
                        s_object* ps_11, s_object* ps_12, s_object* ps_13, s_object* ps_14, s_object* ps_15,
                        s_object* ps_16, s_object* ps_17, s_object* ps_18, s_object* ps_19, s_object* ps_20,
                        s_object* ps_21, s_object* ps_22, s_object* ps_23, s_object* ps_24, s_object* ps_25,
                        s_object* ps_26, s_object* ps_27, s_object* ps_28, s_object* ps_29, s_object* ps_30,
												s_object* ps_31) const
{
	Validate();
	const long kMaxArgs=32L;
	s_object* pps_args[]={ps_1,   ps_2,   ps_3,   ps_4,   ps_5,
                ps_6,   ps_7,   ps_8,   ps_9,  ps_10,
                ps_11, ps_12,  ps_13,  ps_14,  ps_15,
                ps_16, ps_17,  ps_18,  ps_19,  ps_20,
                ps_21, ps_22,  ps_23,  ps_24,  ps_25,
                ps_26, ps_27,  ps_28,  ps_29,  ps_30,
								ps_31};	
	s_object* ps_value = NULL;
	try
	{
		CSPevaluator sEvaluator;
		s_object* ps_call = sEvaluator.alcvec(S_MODE_FUN_CALL, kMaxArgs); //alloc long enough space
		s_object** els    = ps_call->value.tree;
		int len=0;
		//Start with function definition
		els[len++]= sEvaluator.CreateNewHeader((*this)->value.tree[0]);
		for(int i=0; i< kMaxArgs-1; ++i)
		{
			if(pps_args[i]==NULL)
				break;
			els[len++] = sEvaluator.CreateNewHeader(pps_args[i]);
		}
		ps_call->length = len;
		//Strip-out name to avoid unexpected side effect
		//Use Eval() with CSPlist or CSPlistNamed for arg names are needed.
		for(long n=1; n<len; n++)
			els[n]->name = EMPTY_STRING; 

		//Evaluate the call object and return result in current or perm. storage frame.
		ps_value = sEvaluator.eval(ps_call);
		ps_value = sEvaluator.CloneIfNeeded(ps_value);

		//free headers created in this function.  Don't free els[0] which the name of object call, it may be shared
		while(--len >= 0 )
			sEvaluator.FreeHeader(els[len]);
		//free ps_call but not its children
		sEvaluator.TryToFree(ps_call, FALSE);

	}
	catch(...)
	{
		ps_value = NULL;
		SCONNECT_ThrowException("Failed to evaluate call");
	}
	return CSPobject(ps_value, TRUE);
}

