/*
	Connect/C++ : Copyright (c) 2001, 2006 Insightful Corp.
	All rights reserved.
	Version 6.0: 2001

	speval.cxx: implementation of the CSPevaluator class wrapping S-PLUS object of structure "s_evaluator".
*/

#include <string>

#ifdef __hppa
#include <rw/exception.h>
#endif

#include "speval.h"
#include "sconnect.h"
#include "S_y_tab.h"
#include "sqpe.h"

#ifdef __hppa
//namespace std{};
#endif

using namespace std;


extern "C" s_object* S_STDCALL S_c_is(s_object* from_v, s_object* to_v, s_object* maybe);
extern "C" s_object* S_STDCALL all_is_classes(s_class *cl);
extern "C" void S_STDCALL boot_objects(s_object *where_obj, int meta,  s_evaluator *S_evaluator);
extern "C" void S_STDCALL SetBootingFlag(s_boolean b);

#define WARN_LIST_VALID() (( (S_evaluator->_Warn_list != NULL) && ((S_evaluator->_Warn_list)->length > 0) ) ? TRUE:FALSE)

BOOL SPL_GetWarningList( s_object **warningList, BOOL bResetAfterGet )
{
	BOOL bSuccess = FALSE;
	if ( warningList != NULL )
		*warningList = NULL;

	if ( warningList != NULL && WARN_LIST_VALID() )
	{
		try
		{
			s_object *pNames = warn_format(S_evaluator->_Warn_list, NULL, S_evaluator);
			*warningList = AS_LIST(pNames);
			bSuccess = TRUE;
		}
		catch(...)
		{
			*warningList = NULL;
		}
	}
	if ( bResetAfterGet )
	{
		if ( WARN_LIST_VALID() )
			SET_LENGTH( S_evaluator->_Warn_list, 0 );
	}
	return bSuccess;
}

//////////////////////////////////////////////////////////////////////
//SPL_ParseEval: Parse and Evaluate an S expression
//This global function call does not notify client.
//To notify client use CSPobject::SyncEval()
//////////////////////////////////////////////////////////////////////

s_object* SPL_ParseEval_GetWarnings(const char* pszExpression, s_object **warningList)
{
	if ( warningList != NULL )
		*warningList = NULL;
	CSPobject sReturn;

	if ( !pszExpression || !*pszExpression )
		return NULL;

	char *pszPreviousName=NULL;
	try
	{
		CSPevaluator sEvaluator; //Open frame 1 and set error jump right-here if needed.
		s_object *obj, *parse_call;

		obj = alcvec(S_MODE_CHAR, 1L, S_evaluator);
		obj->value.Char[0] = c_s_cpy(pszExpression, S_evaluator);

		parse_call = alcf((char*)"parse",obj, S_evaluator);
		parse_call->value.tree[1]->name = (char*)make_name_p("text", -1, S_evaluator)->text;

		//Parse the 'text' to get an object of class 'expression'
		s_object*	ps_expr = ::eval(parse_call, S_evaluator);

		if(!IS_OBJ(ps_expr, expression)) //must result in "expression" class
				SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);
		long nLen = GET_LENGTH(ps_expr);

		//Save the mode of the last element of the expression
		s_object *ps_last_expr=ps_expr->value.tree[nLen-1];
		int nMode = Data_mode(ps_last_expr, S_evaluator);
		obj = alcf((char*)"eval",ps_expr,S_evaluator);

		//do_task() will call eval_close() which will destroy all non-permanent object.
		//we need to make it permanent first.
		//s_object* ps_frame = ::get_frame_pointer( CSPobject::m_lGUIFrame, S_evaluator);

//Commented out for now - may be more efficient in future
//		set_perm_ref(obj, obj->frame, S_evaluator);
//		set_perm_ref(parse_call, parse_call->frame, S_evaluator);

		set_local(blt_in_TRUE, make_name(".Auto.print", S_evaluator));

		SPL_GetWarningList( NULL, TRUE );

		sReturn = eval(obj, S_evaluator);

		// if an error was encountered and do_stop was executed,
		// stop_in_progress will not be reset.  it should be reset
		// before the next evaluation is executed.
		S_evaluator->_stop_in_progress = (s_boolean_enum)FALSE;

		int iShowLevel=DoLastValueAndShouldShow(S_GUI_EVAL, nMode, sReturn.GetPtr());

		//Flush changed object to engine so it will be available immediately
		flush_data(0, S_evaluator);

		if( WARN_LIST_VALID() )
			warn_message(S_evaluator);

		SPL_GetWarningList( warningList, TRUE );

		//If return object has a name, temporarily rename it to an empty string
		//to fix problem caused by making this name the parameter name to 'show'
		pszPreviousName = sReturn.GetPtr()->name;
		sReturn.GetPtr()->name = EMPTY_STRING;

		if (iShowLevel) {
			DoShouldShow(iShowLevel, ps_last_expr, NULL);
		}

		//Restore original object name if any after 'show' call
		sReturn.GetPtr()->name = pszPreviousName;

//Commented out for now - may be more efficient in future
//		try_to_free(parse_call, S_FALSE, Nframe, S_evaluator);
//		try_to_free(obj, S_FALSE, Nframe, S_evaluator);

		//Ensure valid object if the evaluator will be closed at the end of this scope.
		sReturn.Attach(sEvaluator.CloneIfNeeded(sReturn.GetPtr()));

	}
	catch(...)
	{
		//Restore original object name if any after 'show' call - maybe there
		//is a non-zero reference count on it?
		if (pszPreviousName != NULL && sReturn.IsValid())
			sReturn.GetPtr()->name = pszPreviousName;

		sReturn.Attach(NULL);
	}
	return sReturn.Detach();
}

s_object* SPL_ParseEval(const char* pszExpression)
{
	return SPL_ParseEval_GetWarnings(pszExpression, NULL);
}

static s_object* g_ps_ent_BinaryOp = NULL;
//Generic element-by-element binary operation
s_object* SPL_BinaryOp( s_object* ps_e1, s_object* ps_e2, long lBinOp)
{
	CSPevaluator sEvaluator;
	s_object* ps_ans=NULL;
	char szBinaryOp[] = "+";
	switch(lBinOp)
	{
		case S_BINOP_MULTIPLY:
			S_evaluator->_sys_index = S_BINOP_MULTIPLY; //MUL_OP=4
			szBinaryOp[0] = '*';
			break;
		case S_BINOP_PLUS:
			S_evaluator->_sys_index = S_BINOP_PLUS; //ADD_OP=5
			szBinaryOp[0] = '+';
			break;
		case S_BINOP_MINUS:
			S_evaluator->_sys_index = S_BINOP_MINUS; //SUB_OP=6
			szBinaryOp[0] = '-';
			break;
		case S_BINOP_DIVIDE:
			S_evaluator->_sys_index = S_BINOP_DIVIDE; //DIV_OP=7
			szBinaryOp[0] = '/';
			break;
		default:
			SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);
			break;
	}
	if(	g_ps_ent_BinaryOp == NULL)
	{
		s_object* ps_ent      = ::alcvec(S_MODE_FUN_CALL, 3L, S_evaluator);
		ps_ent->value.tree[1] = ::alc_name("e1", S_evaluator);
		ps_ent->value.tree[2] = ::alc_name("e2", S_evaluator);
		g_ps_ent_BinaryOp = SPL_Clone(ps_ent, TRUE);
	}
	g_ps_ent_BinaryOp->value.tree[0] = ::alc_name(szBinaryOp, S_evaluator);

	s_object*  ps_arglist = NEW_LIST(2);
	ps_arglist->value.tree[0] = ps_e1;
	ps_arglist->value.tree[1] = ps_e2;

	sEvaluator.IncrRef(ps_arglist);
	ps_ans = ::S_do_op(g_ps_ent_BinaryOp, ps_arglist, S_evaluator);
	sEvaluator.DecrRef(ps_arglist);

	//free ps_arglist but not its children
	sEvaluator.TryToFree(ps_arglist, S_FALSE);

	return sEvaluator.CloneIfNeeded(ps_ans);
}

//Clone an object (permanent or not)
s_object* SPL_Clone(s_object *ps_orig_object, BOOL bPermanent)
{
	s_object* ps_new_object = NULL;
	if ( !ps_orig_object )
		return ps_new_object;

	if ( bPermanent )
	{
		long lPrevFrame = ::set_alloc(PERM_FRAME,  S_evaluator);
		//Let ref. counting helps determine the life-time of this perm object created by SPL.
		ps_new_object = copy_data(ps_orig_object, NULL, S_evaluator);
		::set_perm_ref(ps_new_object, NULL, S_evaluator); //Too dangerous to ref. count any perm obj
		::set_alloc(lPrevFrame, S_evaluator); //Reset allocated frame
	}
	else
	{
		ps_new_object = copy_data(ps_orig_object, NULL, S_evaluator);
	}

	return ps_new_object;
}

//Routine to allocate a string in the same frame as s_object passed in
char * SPL_AllocateStringInFrame( s_object *ps_object, const char *pszString )
{
	char *pszAllocatedStringInFrame = EMPTY_STRING;
	if ( !pszString )
		return pszAllocatedStringInFrame;

	long len = strlen(pszString);
	if ( len <= 0 )
		return pszAllocatedStringInFrame;

	try
	{
		BOOL bPermanent = FALSE;
		long lCurrentFrame = Nframe;
		long lPrevFrame = lCurrentFrame;
		long lStorageFrame = ::get_storage_frame(ps_object, S_evaluator);
		if ( lStorageFrame < 0 )
			bPermanent = TRUE;

		if ( bPermanent )
		{
			lPrevFrame = ::set_alloc(PERM_FRAME,  S_evaluator);
		}
		else
		{
			if ( lStorageFrame < lCurrentFrame )
				lPrevFrame = ::set_alloc(lStorageFrame, S_evaluator);
		}

		pszAllocatedStringInFrame = (char*) S_alloc(len+ 1, sizeof(char), S_evaluator);

		if ( lPrevFrame != lCurrentFrame )
			::set_alloc(lPrevFrame, S_evaluator); //Reset allocated frame

		if ( pszAllocatedStringInFrame != NULL )
			strcpy(pszAllocatedStringInFrame, pszString);
		else
			pszAllocatedStringInFrame = EMPTY_STRING;
	}
	catch(...)
	{
	}
	return pszAllocatedStringInFrame;
}

//Used by CSPobject::BaseCreate() to eval expressions for initialization
//without the overhead of SPL_ParseEval_GetWarnings - eliminates output
//and warning list return.
s_object * SPL_DirectParseEval( const char *pszExpression, BOOL bTry )
{
	s_object *obj, *parse_call, *ps_object = NULL;
	if ( !pszExpression || !*pszExpression )
		return ps_object;
	obj = ::alcvec(S_MODE_CHAR, 1L, S_evaluator);
	if(bTry)
	{
		string strExpression("try(");
		strExpression += pszExpression;
		strExpression += ")";
		obj->value.Char[0] = c_s_cpy(strExpression.c_str(), S_evaluator);
	}
	else
		obj->value.Char[0] = c_s_cpy(pszExpression, S_evaluator);
	parse_call = ::alcf((char*)"parse",obj, S_evaluator);
	parse_call->value.tree[1]->name = (char*)make_name_p("text", -1, S_evaluator)->text;
	s_object* ps_call = alcf((char*)"eval", parse_call, S_evaluator);
	s_object* ps_return = ::eval(ps_call, S_evaluator);
	if(bTry && !SPL_NotThere(ps_return))
	{
		if(ps_return->Class && !strcmp(ps_return->Class->name, "Error"))
		{
			s_object* ps_char = LIST_POINTER(ps_return)[0];
			if(!SPL_NotThere(ps_char) && IS_CHARACTER(ps_char))
				 SCONNECT_ThrowException("%s",(char*) CHARACTER_VALUE(ps_char));
		}
	}

	return ps_return;
}

void SPL_NaSet(void* p, int mode)
{
	set_na3((char *)p,mode,To_NA,S_evaluator);
}

BOOL SPL_IsNa(void* p, int mode)
{
	return ( test_na((char *)p,mode, S_evaluator) != 0 );
}

//SPL_GetSearchPathPosition() returns database position (same id as S-function search())
//If non-permanent object, it returns S_NOT_A_DB.
long SPL_GetSearchPathPosition( s_object *ps_object )
{
	if ( !ps_object )
		return S_NOT_A_DB;
	long lFrame = get_storage_frame(ps_object, S_evaluator);
	if ( lFrame > 0 )
		return S_NOT_A_DB;
	if ( lFrame == 0 )
		return 0;
	return (long)s_database_position_from_storage_frame(lFrame, S_evaluator);
}

static inline s_object* CreateBraceCallWithColon(s_object* ps_vector, long lFrom, long lTo)
{
	//If you call me, big boy, you better have an evaluator open
	s_object* ps_Lbrace = alcvec(S_MODE_FUN_CALL, 3, S_evaluator);
	SET_ELEMENT(ps_Lbrace,0,alc_name("[", S_evaluator));

	s_object* ps_colon = alcvec(S_MODE_FUN_CALL, 3, S_evaluator);
	SET_ELEMENT(ps_colon,0,alc_name(":", S_evaluator));
	s_object* ps_from = NEW_INTEGER(1);
	INTEGER_POINTER(ps_from)[0] = lFrom;
	s_object* ps_to = NEW_INTEGER(1);
	INTEGER_POINTER(ps_to)[0] = lTo;

	SET_ELEMENT(ps_colon,1,ps_from);
	SET_ELEMENT(ps_colon,2,ps_to);
	SET_ELEMENT(ps_Lbrace,1,ps_vector);
	SET_ELEMENT(ps_Lbrace,2,ps_colon);

	return ps_Lbrace;
}


//utilty for extending the lengths of vectors
s_object* SPL_ExtendVector(s_object* ps_vector, long n, long index)
{
	if( !S_evaluator->_eval_open )
		SCONNECT_ThrowException(SCONNECT_EVALUATOR_CLOSED);

	s_object* ps_return = blt_in_NULL;
	if ( SPL_NotThere(ps_vector) )
		return ps_return;
	//bump up ref count;
	CSPobject spObject(ps_vector);
	//TODO: lists. SPL_IsAtomicVector excludes them
	if ( SPL_IsAtomicVector(ps_vector) && ps_vector->mode!=S_MODE_CHAR )
	{
		// unused: s_object* ps_orig = ps_vector;
		//if ref count greater than zero or perm. object, create a new header and copy the data.
		if ( multiple_ref_count(ps_vector, S_evaluator) )
		{
			ps_vector = s_header_for_object(ps_vector, NULL, S_evaluator);
			//force a copy
			int iType = ps_vector->Type;
			*ps_vector = *copy_lev(ps_vector, 0, NULL_ENTRY, S_evaluator);
			ps_vector->Type = iType;
		}

		long len = GET_LENGTH(ps_vector);
		long lPrev = len;
		if ( index > len )
		{
			n += index-len;
			index = len;
		}
		len += n;
		SET_LENGTH(ps_vector, len);
		if ( index >= 0 )
		{
			int mode = ps_vector->mode;
			size_t nBytes = mode_size(mode);
			size_t nOff = nBytes*n;
			char* pData = RAW_POINTER(ps_vector)+nBytes*(lPrev-1);
			for ( long i=lPrev; i>index; i--, pData-=nBytes )
			{
				memcpy(pData+nOff, pData, nBytes);
				na_set(pData, mode);
			}
		}
		ps_return = ps_vector;
	}
	else
	{
		long nFrame = ::set_alloc(Nframe, S_evaluator);
		//SET_ELEMENT was eradicating the name component of the object
		s_object* ps_vector1 = s_header_for_object(ps_vector, NULL, S_evaluator);
		try
		{
			//Construct an S object of class "call"
			s_object* ps_length = alcvec(S_MODE_FUN_CALL, 2, S_evaluator);
			SET_ELEMENT(ps_length,0,alc_name("length", S_evaluator));
			SET_ELEMENT(ps_length,1,ps_vector1);

			s_object* ps_frame = get_frame_pointer(Nframe, S_evaluator);
			incr_ref_count(ps_length, S_FALSE, ps_frame, S_evaluator);
			s_object* ps_len = Eval(ps_length, Nframe, S_evaluator);
			decr_ref_count(ps_length, S_FALSE, ps_frame, S_evaluator);

			if ( !SPL_NotThere(ps_len) )
			{
				long len = INTEGER_VALUE(ps_len);

				try_to_free(ps_length, S_FALSE, Nframe, S_evaluator);

				if ( index < 0 )
					index = len;

				if ( index >= len )
				{
					n += index-len;
					index = len;
					len += n;

					//v[1:len] will automagically extend the vector.
					s_object* ps_Lbrace = CreateBraceCallWithColon(ps_vector1, 1L, len);

					incr_ref_count(ps_Lbrace, S_FALSE, ps_frame, S_evaluator);
					ps_return = Eval(ps_Lbrace, Nframe, S_evaluator);
					decr_ref_count(ps_Lbrace, S_FALSE, ps_frame, S_evaluator);
				}
				else
				{
					s_copy_if_needed(ps_vector1, S_FALSE, S_evaluator);

					//v[a:b]<-v[c:d]
					s_object* ps_larrow = alcvec(S_MODE_LARROW, 2, S_evaluator);
					SET_CLASS(ps_larrow, MAKE_CLASS("<-"));

					static char* pszTempName = (char*)"???SPL_tempVariable???";
					s_object* ps_objName = alc_name(pszTempName, S_evaluator);
					s_name* ps_name = make_name(pszTempName, S_evaluator);

					::set_in_frame(Nframe, ps_vector1, ps_name, S_evaluator);

					s_object* ps_LbraceLHS = CreateBraceCallWithColon(ps_objName, index+1+n, len+n);
					s_object* ps_LbraceRHS = CreateBraceCallWithColon(ps_objName, index+1, len);

					SET_ELEMENT(ps_larrow,0,ps_LbraceLHS);
					SET_ELEMENT(ps_larrow,1,ps_LbraceRHS);

					incr_ref_count(ps_larrow, S_TRUE, ps_frame, S_evaluator);
					Eval(ps_larrow,Nframe,S_evaluator);

					s_object* ps_colon = LIST_POINTER(ps_LbraceLHS)[2];
					s_object* ps_from = LIST_POINTER(ps_colon)[1];
					s_object* ps_to = LIST_POINTER(ps_colon)[2];
					INTEGER_POINTER(ps_from)[0] = index+1;
					INTEGER_POINTER(ps_to)[0] = index+n;

					SET_ELEMENT(ps_larrow,0,ps_LbraceLHS);
					SET_ELEMENT(ps_larrow,1,alc_name("NA",S_evaluator));

					Eval(ps_larrow,Nframe,S_evaluator);

					//Take object out of frame incase we are looping through the columns
					//of a data frame.
					s_name_table* pFrmTbl = frame_table(Nframe, S_evaluator);

					long nn = Name_lookup(ps_name, pFrmTbl, NULL, S_evaluator);
					if ( nn > 0 )
					{
						s_object* ps_frame = FRAME_POINTER(Nframe);
						if( !SPL_NotThere(ps_frame) && nn <= ps_frame->length )
						{
							ps_return = ps_frame->value.tree[--nn];
							//Use arguement frame = NULL so object is not deleted if ref count==0
							decr_ref_count(ps_return, S_TRUE, NULL, S_evaluator);
							ps_frame->value.tree[nn] = unset_variable;
						}
					}
					decr_ref_count(ps_larrow, S_TRUE, ps_frame, S_evaluator);
				}
			}
		}
		catch(CSPexception& except)
		{
			except.Print();
		}
		catch(...)
		{
		}
		if ( nFrame != Nframe )
			::set_alloc(nFrame, S_evaluator);

		if ( !SPL_NotThere(ps_return) && ps_vector->name && *(ps_vector->name) )
			ps_return->name = (char*)make_name(ps_vector->name,S_evaluator)->text;
	}
	//decrement ref count;
	spObject.Detach();

	return ps_return;
}

//utilty for subsetting a vector
s_object* SPL_ShrinkVector(s_object* ps_vector, long n, long index)
{
	if( !S_evaluator->_eval_open )
		SCONNECT_ThrowException(SCONNECT_EVALUATOR_CLOSED);


	s_object* ps_return = blt_in_NULL;
	//bump up ref count;
	CSPobject spObject(ps_vector);
	if ( n < 0 )
		n = -n;

	//TODO: lists. SPL_IsAtomicVector excludes them
	if ( SPL_IsAtomicVector(ps_vector) )
	{
		long len = GET_LENGTH(ps_vector);
		if ( index < 0 )
			index = len-n;

		if ( index >= len )
			SCONNECT_ThrowException("Index %d extends beyond vector length %d", index, len);

		ps_vector = s_header_for_object(ps_vector, NULL, S_evaluator);
		s_copy_if_needed(ps_vector, S_FALSE, S_evaluator);

		if ( index+n > len )
		{
			len = index+1;
		}
		else
		{
			if ( index+n < len )
			{
				int mode = ps_vector->mode;
				int inc = mode_size(mode);
				long i = index;
				long iShift = inc*n;
				for ( char* pcData=RAW_POINTER(ps_vector)+index*inc; i<len-n; pcData+=inc, i++)
				{
					if ( mode == S_MODE_CHAR )
						*(char**)pcData = *(char**)(pcData+iShift);
					else
						memcpy(pcData, (pcData+iShift), inc);
				}
			}
			len -= n;
		}

		SET_LENGTH(ps_vector, len);
		ps_return = ps_vector;
	}
	else if ( !IS(ps_vector,s_vector_class) && !IS(ps_vector,MAKE_CLASS("factor")) )
	{
		SCONNECT_ThrowException("Invalid class %s", GET_CLASS_NAME(ps_vector));
	}
	else
	{
		long nFrame = ::set_alloc(Nframe, S_evaluator);
		//SET_ELEMENT was eradiating the name component of the object
		s_object* ps_vector1 = s_header_for_object(ps_vector, NULL, S_evaluator);
		try
		{
			//Construct an S object of class "call"
			s_object* ps_length = alcvec(S_MODE_FUN_CALL, 2, S_evaluator);
			SET_ELEMENT(ps_length,0,alc_name("length", S_evaluator));
			SET_ELEMENT(ps_length,1,ps_vector1);

			s_object* ps_frame = get_frame_pointer(Nframe, S_evaluator);
			incr_ref_count(ps_length, S_FALSE, ps_frame, S_evaluator);
			s_object* ps_len = Eval(ps_length, Nframe, S_evaluator);
			decr_ref_count(ps_length, S_FALSE, ps_frame, S_evaluator);

			if ( !SPL_NotThere(ps_len) )
			{
				long len = INTEGER_VALUE(ps_len);

				try_to_free(ps_length, S_FALSE, Nframe, S_evaluator);

				if ( index < 0 )
					index = len-n;

				if ( index >= len )
					SCONNECT_ThrowException("Index %d extends beyond vector length %d", index, len);

				if ( index + n > len )
				{
					len = index+1;
				}
				else
				{
					s_copy_if_needed(ps_vector1, S_FALSE, S_evaluator);

					//v[a:b]<-v[c:d]
					s_object* ps_larrow = alcvec(S_MODE_LARROW, 2, S_evaluator);
					SET_CLASS(ps_larrow, MAKE_CLASS("<-"));

					static char* pszTempName = (char*)"???SPL_tempVariable???";
					s_object* ps_objName = alc_name(pszTempName, S_evaluator);
					s_name* ps_name = make_name(pszTempName, S_evaluator);

					::set_in_frame(Nframe, ps_vector1, ps_name, S_evaluator);

					s_object* ps_LbraceLHS = CreateBraceCallWithColon(ps_objName, index+1, len-n);
					s_object* ps_LbraceRHS = CreateBraceCallWithColon(ps_objName, index+n+1, len);

					SET_ELEMENT(ps_larrow,0,ps_LbraceLHS);
					SET_ELEMENT(ps_larrow,1,ps_LbraceRHS);

					incr_ref_count(ps_larrow, S_TRUE, ps_frame, S_evaluator);
					Eval(ps_larrow,Nframe,S_evaluator);
					decr_ref_count(ps_larrow, S_TRUE, ps_frame, S_evaluator);

					//Take object out of frame incase we are looping through the columns
					//of a data frame.
					s_name_table* pFrmTbl = frame_table(Nframe, S_evaluator);

					long nn = Name_lookup(ps_name, pFrmTbl, NULL, S_evaluator);
					if ( nn <= 0 )
						SCONNECT_ThrowException("Vector subsetting failed");

					s_object* ps_frame = FRAME_POINTER(Nframe);
					if( SPL_NotThere(ps_frame) || nn > ps_frame->length )
						SCONNECT_ThrowException("Vector subsetting failed");

					ps_vector1 = ps_frame->value.tree[--nn];
					//Use arguement frame = NULL so object is not deleted if ref count==0
					decr_ref_count(ps_vector1, S_TRUE, NULL, S_evaluator);
					ps_frame->value.tree[nn] = unset_variable;

					len -= n;
				}
				//v[1:len] will automagically extend the vector.
				s_object* ps_Lbrace = CreateBraceCallWithColon(ps_vector1, 1L, len);

				incr_ref_count(ps_Lbrace, S_FALSE, ps_frame, S_evaluator);
				ps_return = Eval(ps_Lbrace, Nframe, S_evaluator);
				decr_ref_count(ps_Lbrace, S_FALSE, ps_frame, S_evaluator);
			}
		}
		catch(CSPexception& except)
		{
			except.Print();
		}
		catch(...)
		{
		}
		if ( nFrame != Nframe )
			::set_alloc(nFrame, S_evaluator);

		if ( !SPL_NotThere(ps_return) && ps_vector->name && *(ps_vector->name) )
			ps_return->name = (char*)make_name(ps_vector->name,S_evaluator)->text;
	}
	//decrement ref count;
	spObject.Detach();

	return ps_return;
}

long SPL_GetVectorLength(s_object* ps_object)
{
	if ( SPL_NotThere(ps_object) )
		return 0L;
	if ( SPL_IsAtomicVector(ps_object) )
		return GET_LENGTH(ps_object);
	if ( IS_OBJ(ps_object,list) )
		return GET_LENGTH(ps_object);

	//must use the evaluator
	if( !S_evaluator->_eval_open )
		SCONNECT_ThrowException(SCONNECT_EVALUATOR_CLOSED);

	long lPrevFrame = Nframe;
	long length = 0L;
	//SET_ELEMENT was eradiating the name component of the object
	ps_object = s_header_for_object(ps_object, NULL, S_evaluator);
	try
	{
		lPrevFrame = set_alloc(Nframe,  S_evaluator);

		//Construct an S object of class "call"
		s_object* ps_length = alcvec(S_MODE_FUN_CALL, 2, S_evaluator);
		SET_ELEMENT(ps_length,0,alc_name("length", S_evaluator));
		SET_ELEMENT(ps_length,1,ps_object);

		s_object* ps_frame = get_frame_pointer(Nframe, S_evaluator);
		incr_ref_count(ps_length, S_FALSE, ps_frame, S_evaluator);
		s_object* ps_len = Eval(ps_length, Nframe, S_evaluator);
		decr_ref_count(ps_length, S_FALSE, ps_frame, S_evaluator);

		if ( !SPL_NotThere(ps_len) )
		{
			length = INTEGER_VALUE(ps_len);
			try_to_free(ps_length, S_FALSE, Nframe, S_evaluator);
		}
	}
	catch(...)
	{
	}
	//Reset allocated frame
	if ( lPrevFrame != Nframe )
		set_alloc(lPrevFrame, S_evaluator);

	return length;
}

//Query whether a pszFrom extends from pszTo.
BOOL SPL_Extends(const char* pszClass1, const char* pszClass2)
{
	BOOL bExtends = FALSE;
	try
	{
		CSPevaluator sEvaluator;
		s_class* pcl_1 = MAKE_CLASS(pszClass1);
		ENSURE_CLASS(pcl_1);
		s_class* pcl_2 = MAKE_CLASS(pszClass2);
		ENSURE_CLASS(pcl_2);
		if ( pcl_1->index == 0)
			bExtends  = (pcl_1 == pcl_2);
		else
			bExtends = ( isClassIndex(pcl_1->index, pcl_2->index, S_FALSE, NULL, S_evaluator)==S_TRUE );
	}
	catch(...)
	{
	}
	return bExtends;
}

//Get all classes that are direct extensions of pszClass
s_object* SPL_ExtendsDirect(const char* pszClass)
{
	s_object* ps_return = blt_in_NULL;
	if ( !pszClass || !*pszClass )
		return ps_return;

	if( !S_evaluator->_eval_open )
		SCONNECT_ThrowException(SCONNECT_EVALUATOR_CLOSED);

	try
	{
		s_class* pcl = MAKE_CLASS(pszClass);
		s_object* ps_extends = all_is_classes(pcl);
		if ( !SPL_NotThere(ps_extends) )
		{
			if ( !VERSION3_CLASS(pcl) )
			{
				//first is always the original class name
				int n = 1;
				for ( int j=1; j<GET_LENGTH(ps_extends); j++ )
				{
					char* pszExtend = CHARACTER_POINTER(ps_extends)[j];
					if ( SPL_Extends(pszClass, pszExtend) )
					{
						if ( n++ != j )
						{
							CHARACTER_POINTER(ps_extends)[n] = CHARACTER_POINTER(ps_extends)[j];
							CHARACTER_POINTER(ps_extends)[j] = EMPTY_STRING;
						}
					}
				}

				SET_LENGTH(ps_extends,n);
			}
			ps_return = ps_extends;
		}
	}
	catch(...)
	{
	}

	return ps_return;
}

/*
db_purposes =  {REGULAR_DB, META_DB, HELP_DB, SEARCH_DB,
                 DATA_DB, RELOAD_DB, TEMP_DB, LOCK_DB, MAX_DB_PURPOSE};
*/

s_object* SPL_AttachedObjectFromPos(long lPos, s_db_purposes dbPurpose)
{
	if( !S_evaluator->_eval_open )
		SCONNECT_ThrowException(SCONNECT_EVALUATOR_CLOSED);

	if ( lPos <= 0 )
		return blt_in_NULL;

	long iSysIndex = S_evaluator->_sys_index;
	S_evaluator->_sys_index = 7;
	s_object* ps_attached = blt_in_NULL;
	try
	{
		s_object* ps_argList = NEW_LIST(2);
		LIST_POINTER(ps_argList)[0] = NEW_INTEGER(1);
		LIST_POINTER(ps_argList)[1] = NEW_INTEGER(1);
		INTEGER_POINTER(LIST_POINTER(ps_argList)[0])[0] = lPos;
		INTEGER_POINTER(LIST_POINTER(ps_argList)[1])[0] = dbPurpose;
		ps_attached = S_dictionary(blt_in_NULL, ps_argList, S_evaluator);
	}
	catch(...)
	{
	}
	S_evaluator->_sys_index = iSysIndex;

	return ps_attached;
}

/*
db_purposes =  {REGULAR_DB, META_DB, HELP_DB, SEARCH_DB,
                 DATA_DB, RELOAD_DB, TEMP_DB, LOCK_DB, MAX_DB_PURPOSE};
*/
s_object* SPL_AttachedObjectFromName(const char* pszName, s_db_purposes dbPurpose)
{
	if( !S_evaluator->_eval_open )
		SCONNECT_ThrowException(SCONNECT_EVALUATOR_CLOSED);

	if ( !pszName || !*pszName )
		return blt_in_NULL;

	long iSysIndex = S_evaluator->_sys_index;
	S_evaluator->_sys_index = 7;
	s_object* ps_attached = blt_in_NULL;
	try
	{
		s_object* ps_name;
		s_object* ps_argList = NEW_LIST(2);
		LIST_POINTER(ps_argList)[0] = ps_name = NEW_CHARACTER(1);
		LIST_POINTER(ps_argList)[1] = NEW_INTEGER(1);
		INTEGER_POINTER(LIST_POINTER(ps_argList)[1])[0] = dbPurpose;
		//assume all \ must be converted to /
		int nCh = strlen(pszName);

		char* pszName1 = (char*)pszName;
		pszName1 = (char*) S_alloc(nCh+1, sizeof(char), S_evaluator);
		for (int i=0; i<nCh; i++ )
		{
				pszName1[i] = pszName[i];
				if ( pszName[i] == '\\' )
					pszName1[i] = '/';
		}
		pszName1[nCh] = '\0';

		CHARACTER_POINTER(ps_name)[0] = pszName1;
		ps_attached = S_dictionary(blt_in_NULL, ps_argList, S_evaluator);
	}
	catch(...)
	{
	}
	S_evaluator->_sys_index = iSysIndex;
	return ps_attached;
}

//Retieve the search path used by the engine.
//dbPurpose = REGULAR_DB or META_DB to retieve the names of the
//regular or meta databases, respectively.
s_object* SPL_DatabaseSearch(s_db_purposes dbPurpose)
{
	if( !S_evaluator->_eval_open )
		SCONNECT_ThrowException(SCONNECT_EVALUATOR_CLOSED);

	long iSysIndex = S_evaluator->_sys_index;
	S_evaluator->_sys_index = 3;
	s_object* ps_search = blt_in_NULL;
	try
	{
		s_object* ps_argList = NEW_LIST(1);
		LIST_POINTER(ps_argList)[0] = NEW_INTEGER(1);
		INTEGER_POINTER(LIST_POINTER(ps_argList)[0])[0] = dbPurpose;
		//assume all backslashes must be converted to double backslashes
		ps_search = S_database(blt_in_NULL, ps_argList, S_evaluator);
	}
	catch(...)
	{
	}
	S_evaluator->_sys_index = iSysIndex;

	return ps_search;
}

//Retieve the directory search path used by the engine
s_object* SPL_SearchPaths(void)
{
	if( !S_evaluator->_eval_open )
		SCONNECT_ThrowException(SCONNECT_EVALUATOR_CLOSED);


	long iSysIndex = S_evaluator->_sys_index;
	s_object* ps_search = blt_in_NULL;
	try
	{
		ps_search = SPL_DatabaseSearch(REGULAR_DB);
		S_evaluator->_sys_index = 0;
		if ( !SPL_NotThere(ps_search) )
		{
			s_object* ps_arg1;
			s_object* ps_argList = NEW_LIST(2);
			LIST_POINTER(ps_argList)[0] = ps_arg1 = NEW_INTEGER(1);
			LIST_POINTER(ps_argList)[1] = NEW_INTEGER(1);
			INTEGER_POINTER(LIST_POINTER(ps_argList)[1])[0] = 0;
			for ( long i=0; i<GET_LENGTH(ps_search); i++ )
			{
				INTEGER_POINTER(ps_arg1)[0] = i+1;
				s_object* ps_value = S_dictionary(blt_in_NULL, ps_argList, S_evaluator);
				if ( !SPL_NotThere(ps_value) && IS_CHARACTER(ps_value) )
					CHARACTER_POINTER(ps_search)[i] = CHARACTER_VALUE(ps_value);
			}
		}
	}
	catch(...)
	{
	}
	S_evaluator->_sys_index = iSysIndex;
	return ps_search;
}

wchar_t SPL_BootResore(void)
{
	int iResult = 0;
	if ( SpValidEvaluator() && S_evaluator->_eval_open )
	{
		try
		{
			SetBootingFlag(S_TRUE);
			S_boot_restore(S_evaluator);
			iResult = 1;
		}
		catch(...)
		{
		}
		SetBootingFlag(S_FALSE);
		int bSynchronize = TRUE;
		if ( bSynchronize )
		{
			long lOldSys = S_evaluator->_sys_index;
			try
			{
				s_object* ps_arglist = NEW_LIST(2);
				s_object* ps_db = NEW_INTEGER(1);
				s_object* ps_meta = NEW_INTEGER(1);
				INTEGER_POINTER(ps_db)[0] = 1;
				INTEGER_POINTER(ps_meta)[0] = 0;
				LIST_POINTER(ps_arglist)[0] = ps_db;
				LIST_POINTER(ps_arglist)[1] = ps_meta;
				S_evaluator->_sys_index = 0;
				S_sync(blt_in_NULL, ps_arglist, S_evaluator);
			}
			catch(...)
			{
			}
			S_evaluator->_sys_index = lOldSys;
		}
	}
	return iResult;
}

s_object* SPL_GetFramePointer(long lFrame)
{
	return ::get_frame_pointer(lFrame, S_evaluator);
}

/////////////////////CSPevaluator ///////////////////////////

//Init the static mumber
long CSPevaluator::m_lInstanceCount = 0L;

//Copy constructor
void CSPevaluator::Init(void)
{
	m_ps_evaluator    = Sqpe_GetCurrentEvaluator(); //Currently the global evaluator
	m_lPushedCount    = 0;
	m_lPrevAllocFrame = GetCurrentAllocFrame();
	m_lPrevEvalFrame  = GetCurrentEvalFrame();
	++CSPevaluator::m_lInstanceCount;
}
//Copy constructor
CSPevaluator::CSPevaluator(const CSPevaluator& sEvaluator)
{
	Init();
}

//Assignment operators: useless, don't use it yet
CSPevaluator& CSPevaluator::operator = (const CSPevaluator& sEvaluator)
{
	Init();
	return *this;
}

//Destructor: close top-level-eval or a local frame if it was open or created by this object
CSPevaluator::~CSPevaluator()
{
	//Don't call Close(0) if longjmp is coming.
	if(GetPtr()->_Frames->length > 0)
	{
		//Close top-level evaluator if the constructor opened it and it is still open.
		Close(0);  //No error, close and copy objects to the default permanent storage frames if needed.
		SetCurrentAllocFrame(m_lPrevAllocFrame);
		SetCurrentEvalFrame(m_lPrevEvalFrame);
	}
	m_ps_evaluator    = NULL;
	m_lPushedCount=0;
	m_lPrevAllocFrame = 1L;
	m_lPrevEvalFrame  = 1L;
	--CSPevaluator::m_lInstanceCount;
}

//Validate the evaluator
BOOL CSPevaluator::IsValid(void) const
{
	return ((GetPtr() != NULL) && SqpeIsValidAddress(GetPtr(), 1));
}

//Validate the evaluator
void CSPevaluator::Validate(void) const
{
	if(!IsValid())
		SCONNECT_ThrowException("Invalid S_evaluator\n");
}

//Get/set the current eval. frame pointer
s_object* CSPevaluator::GetCurrentEvalFramePtr(void) const
{
	Validate();
	return GetPtr()->_Local_data;
}

//Get the current eval. frame
long CSPevaluator::GetCurrentEvalFrame(void) const
{
	Validate();
	return GetPtr()->_Nframe;
}

//Set the current eval. frame: dangereous!
long CSPevaluator::SetCurrentEvalFrame(long nFrame)
{
	Validate();

	long nOld = GetPtr()->_Nframe;
	GetPtr()->_Nframe = nFrame;
	return nOld;
}

//Get the current allocation frame
long CSPevaluator::GetCurrentAllocFrame(void) const
{
	Validate();
	return GetPtr()->_cur_frame;
}

//Set the current allocation frame
long CSPevaluator::SetCurrentAllocFrame(long lFrame)
{
	Validate();
	return ::set_alloc(lFrame, GetPtr());
}

int CSPevaluator::IsOpen(void) const
{
	Validate();
	return GetPtr()->_eval_open;
}

jmp_buf& CSPevaluator::GetErrorJmp() const
{
	Validate();
	return GetPtr()->_S_error_jmp;
}

//Push a frame (list object) to the evaluator stack
BOOL CSPevaluator::PushFrame(s_object* ps_frame )
{
	Validate();
	//Open a top-level eval frame if it is not already open
	if(!IsOpen())
	{
		::eval_init(GetPtr());
		m_lPushedCount = 0;
	}
	else if(ps_frame == NULL) //Want a new empty frame?
	{
		s_object* ps_frame = alcvec(S_MODE_LIST, 0L);
		::set_frame(::make_frame(ps_frame, GetPtr()), GetPtr());
		TryToFree(ps_frame, FALSE);
	}
	else if(IS_LIST(ps_frame))
		::set_frame(::make_frame(ps_frame, GetPtr()), GetPtr());
	else
		::SCONNECT_ThrowException("Bad input arg to CSPevaluator::PushFrame()");

	m_lPushFrames[m_lPushedCount++] = GetCurrentEvalFrame();
	return TRUE;
}

//Pop the most local frame off the stack
s_object* CSPevaluator::PopFrame(s_object* ps_object )
{
	//Bail-out: no eval. frame is open or longjmp is coming.
	if( GetPtr()->_Frames->length==0)
		return blt_in_NULL;

	Validate();
	s_object* ps_return=NULL;
	try
	{
		if(IsOpen())
		{
			if(GetCurrentEvalFrame() > 1L && (m_lPushedCount>0))
			{
				if(m_lPushFrames[m_lPushedCount-1]==GetCurrentEvalFrame())
				{
					long lParentFrame=1L;
					if(m_lPushedCount-2 >= 0)
						lParentFrame = m_lPushFrames[m_lPushedCount-2];
					else
						lParentFrame = GetPtr()->_parent_frame[GetCurrentEvalFrame()];
					if(lParentFrame < GetCurrentEvalFrame())
						CSPobject::UnnamedObjectMap_PopFrame( lParentFrame );
					if(	lParentFrame > 0)
						 ps_return = ::pop_frame(ps_object==NULL? S_void : ps_object, GetPtr());
					else //Longjmp or bad thing happens in the engine, don't pop frame to a non-positive parent.
						ps_return = ps_object==NULL? S_void : ::SPL_Clone(ps_object, TRUE);
				}
				m_lPushFrames[--m_lPushedCount] = 0L;
			}
			else
			{
				//Invalidate member m_ps_object for all unnamed local objects
				//because their frames will be closed by eval_close below.
				CSPobject::UnnamedObjectMap_InvalidateLocalObjects(FALSE);
				::eval_close(0, GetPtr());
				m_lPushedCount = 0;
				ps_return = S_void;
			}
		}
		else
			m_lPushedCount = 0;
	}
	catch(...)
	{
	}
	return ps_return;
}

//Close the evaluator opened by this object.
BOOL CSPevaluator::Close(const int nErrorCode)
{

	if(!IsValid())
		return FALSE;
	try
	{
		if(nErrorCode)
			eval_close(nErrorCode) ;
		else if (IsOpen())
		{
			while(m_lPushedCount>0)
					PopFrame();//m_lPushedCount should be decreased by PopFrame()
		}
	}
	catch(...)
	{
	}
	return TRUE;
}

//Close all evaluators
BOOL CSPevaluator::eval_close( int nError)
{
	try
	{
		//invalidate member m_ps_object for all unnamed local objects
		//because their frames will be closed by eval_close below.
		CSPobject::UnnamedObjectMap_InvalidateLocalObjects((BOOL)nError);

		::eval_close( nError, S_evaluator /* don't use the data member, it may be invalide */);
	}
	catch(...) //No exception should escape the destructor.
	{
	}
	return TRUE;
}

//Clone to perm frame if needed
s_object* CSPevaluator::CloneIfNeeded( s_object* ps_object) const
{
	Validate();

	s_object* ps_new_object = ps_object;
	if(WillCloseEvaluator()  //Need to copy object to the default perm. frame?
	&&(ps_object != NULL))
	{
		if( ps_object->mode==S_MODE_NULL)
			ps_new_object = blt_in_NULL;
		else if(ps_object->mode==S_MODE_MISSING)
			ps_new_object = blt_in_MISSING;
		else if(::get_storage_frame(ps_object, GetPtr()) > 0)
			ps_new_object = SPL_Clone(ps_object, TRUE);
	}
	return ps_new_object;
}

//Return TRUE if this object will close the top-level-evaluation frame (frame 1)
BOOL CSPevaluator::WillCloseEvaluator(void) const
{
	Validate();
	return(m_lPushedCount > 0);
}

//Parse and eval in the current eval frame.  If top-level eval is not open, open it first.
s_object* CSPevaluator::Eval(const char* pszExpression, BOOL bTry)
{
	Validate();
	s_object* ps_new_object = NULL;
	try
	{
		if(IsOpen())
			ps_new_object = SPL_DirectParseEval(pszExpression, bTry);
		else
		{ //If top-level is not open, open it, eval and close.
			Open();
			ps_new_object = SPL_DirectParseEval(pszExpression, bTry);
			ps_new_object = CloneIfNeeded(ps_new_object);
			Close(0);
		}
	}
	catch(CSPexception& e)
	{
		throw(e); //re-throw
	}
	catch(...)
	{
		SCONNECT_ThrowException("Failed to evaluate expression: %s", pszExpression);
	}
	return ps_new_object;
}

//Evaluate the object in the current frame. If top-level eval is not open, open it first.
s_object* CSPevaluator::Eval(s_object* ps_object)
{
	Validate();
	s_object* ps_new_object = NULL;
	try
	{
		if(IsOpen())
			ps_new_object = ::eval(ps_object, GetPtr());
		else
		{ //If top-level is not open, open it, eval and close.
			Open();
			ps_new_object = ::eval(ps_object, GetPtr());
			ps_new_object = CloneIfNeeded(ps_new_object);
			Close(0);
		}
	}
	catch(...)
	{
		SCONNECT_ThrowException("Failed to evaluate object");
	}
	return ps_new_object;
}

//Evaluate the object in a specified frame
s_object* CSPevaluator::Eval(s_object* ps_object, long lFrame)
{
	Validate();
	s_object* ps_new_object = NULL;
	try
	{
		if(IsOpen() && lFrame <= GetCurrentEvalFrame())
		{
			long lPrevFrame = SetCurrentEvalFrame(lFrame);
			ps_new_object = ::eval(ps_object, GetPtr());
			SetCurrentEvalFrame(lPrevFrame); //reset
		}
		else if(lFrame <= 1)
		{ //If top-level is not open, open it, eval and close.
			Open();
			ps_new_object = ::eval(ps_object, GetPtr());
			ps_new_object = CloneIfNeeded(ps_new_object);
			Close(0);
		}
	}
	catch(...)
	{
	}
	if(!ps_new_object || !SqpeIsValidAddress(ps_new_object, 1))
		SCONNECT_ThrowException("Failed to evaluate object in frame=%d", lFrame);

	return ps_new_object;
}

//////////////////////////////////////////////////////////////////////
//Set an element in current eval frame
s_object* CSPevaluator::SetInFrame(s_object* ps_object, const char* pszName)
{
	Validate();
	return ::set_in_frame(GetPtr()->_Nframe, ps_object, make_name(pszName, GetPtr()), GetPtr());
}
//Set an element in the target frame, nTargetFrame
s_object* CSPevaluator::SetInFrame(s_object* ps_object, const char* pszName, long nTargetFrame)
{
	Validate();
	return ::set_in_frame(nTargetFrame, ps_object, make_name(pszName, GetPtr()), GetPtr());
}

//Get an element in current eval frame
s_object* CSPevaluator::FindInFrame(const char* pszName) const
{
	Validate();
	return ::find_in_frame(make_name(pszName, GetPtr()), GetPtr()->_Nframe,  GetPtr());
}
//Get an element in the target frame, nTargetFrame
s_object* CSPevaluator::FindInFrame(const char* pszName, long nTargetFrame) const
{
	Validate();
	return ::find_in_frame(make_name(pszName, GetPtr()), nTargetFrame, GetPtr());
}

//Get the persistant object in the S databases: the first one in the search list will be returned.
s_object* CSPevaluator::Get(const char* pszName, BOOL bLocal, BOOL bGetData)
{
	Validate();
	s_object* ps_object = NULL;
	BOOL bOpen=FALSE;
	try
	{
		if(!IsOpen())
		{
			Open();
			bOpen = TRUE;
		}

		s_boolean bReadIt = (bGetData?S_TRUE:S_FALSE);
		s_name* ps_name = make_name(pszName, GetPtr());

		if ( bLocal )
		{
			//look in local data, then frame 1, followed a database search
			ps_object = ::find_data(ps_name, bReadIt, S_TRUE, GetPtr());
		}
		else
		{
			//look only in databases
			ps_object = ::db_table_lookup(ps_name, GetPtr()->_thread_data_index, GetPtr());
			if ( !ps_object )
			{
				s_boolean bKeep = bReadIt;
				ps_object = ::read_data(ps_name->text, bKeep, bReadIt, Nframe>0?Nframe:1,
							S_ANY_DATABASE, REGULAR_DB, NULL, GetPtr(), S_TRUE);
			}
		}
	}
	catch(...)
	{
		SCONNECT_ThrowException("Failed to get object %s", pszName);
	}

	if(bOpen)
	{
		CSPobject sReturn(CloneIfNeeded(ps_object));
		Close(0);
		ps_object = sReturn.Detach();
	}
	return ps_object;
}

//Get a method body which is an S object of class "{"
//Same as getFunction<- function(name, generic = TRUE, "must find" = TRUE, where), except for where arg
//Or s_c_find_fun(s_object *name_obj, s_object *generic, s_object *must_find)
s_object* CSPevaluator::selectMethod(s_object* ps_charName, s_object* ps_charSig, BOOL bCoerce) const
{
  s_object *ps_function=NULL;
	try
	{
		CSPcharacter schName(ps_charName);
		if(!schName.IsValid())
			return ps_function;

		CSPcharacter schSig(ps_charSig);
		if(!schSig.IsValid())
			schSig.Attach(alcvec(S_MODE_CHAR, 0L));

		CSPcall sCall("selectMethod");
		sCall.SetArg(schSig.GetPtr(), 1L, "sig");
		CSPlogical slogCoerce(&bCoerce, 1L);

		ps_function = sCall.Eval(schName.GetPtr(), schSig.GetPtr(), slogCoerce.GetPtr());
  }
	catch(...)
	{
		SCONNECT_ThrowException("Failed to get function definition for %s", CHARACTER_POINTER(ps_charName)[0]);
	}
  return(ps_function);
}

s_object* CSPevaluator::selectMethod(const char* pszName, const char* pszSig, BOOL bCoerce) const
{
  s_object* ps_function=NULL;
	s_object* ps_chSig=NULL;
	if(!pszName || !*pszName)
		return ps_function;
	try
	{
		CSPcharacter schName(const_cast<char**>(&pszName), 1L);
		if(pszSig && *pszSig)
		{
			string strSigExp("as.character(unpaste(");
			strSigExp += pszSig;
			strSigExp += ", sep=','))";
			CSPevaluator sEvaluator;
			CSPcharacter schSig(sEvaluator.eval(strSigExp.c_str()));
			if(schSig.IsValid())
				ps_chSig = schSig.Detach();
		}

		ps_function = selectMethod(schName.GetPtr(), ps_chSig, bCoerce);
  }
	catch(...)
	{
		SCONNECT_ThrowException("Failed to get function definition for %s", pszName);
	}
  return(ps_function);

}

//Get a function : same as the S function getFunction()
s_object* CSPevaluator::getFunction(s_object* ps_charName, BOOL bGeneric, BOOL bMustFind, long lWhere) const
{
  s_object *ps_function=NULL;
	try
	{
		CSPcharacter schName(ps_charName);
		if(!schName.IsValid())
			return ps_function;

		CSPlogical slogGenric(&bGeneric, 1L);
		CSPlogical slogMustFind(&bMustFind, 1L);

		if(lWhere < 0) //Default value -1
		{
			ps_function = ::s_c_find_fun(schName.GetPtr(), slogGenric.GetPtr(), slogMustFind.GetPtr());
		}
		else
		{
			CSPinteger sintWhere(&lWhere, 1L);
			CSPcall sCall("getFunction");
			ps_function = sCall.Eval(schName.GetPtr(), slogGenric.GetPtr(), slogMustFind.GetPtr(), sintWhere.GetPtr());
		}
  }
	catch(...)
	{
		SCONNECT_ThrowException("Failed to get function definition for %s", CHARACTER_POINTER(ps_charName)[0]);
	}
  return(ps_function);
}

//Get a function : same as the S function getFunction()
s_object* CSPevaluator::getFunction(const char* pszName, BOOL bGeneric, BOOL bMustFind, long lWhere) const
{
	if(!pszName || !*pszName)
		return NULL;
	CSPcharacter schName(const_cast<char**>(&pszName), 1L);
	return getFunction(schName.GetPtr(), bGeneric, bMustFind, lWhere);
}

//Comparing two objects: same as all.equal()
BOOL CSPevaluator::allEqual( s_object* ps_e1, s_object* ps_e2) const
{
	Validate();
	BOOL bRet = FALSE;
	try
	{
		//If frame 1 does not exist, open it.
		CSPevaluator sEvaluator;
    s_object *ps_call= alcvec(S_MODE_FUN_CALL, 3L);
    ps_call->value.tree[0] = ::alc_name("all.equal", GetPtr());
    ps_call->value.tree[1] = CreateNewHeader(ps_e1);
    ps_call->value.tree[2] = CreateNewHeader(ps_e2);
 		ps_call->value.tree[1]->name = EMPTY_STRING;
 		ps_call->value.tree[2]->name = EMPTY_STRING;

		IncrRef(ps_call, TRUE);

		s_object* ps_logical = ::eval(ps_call, GetPtr());

		if((ps_logical) && (ps_logical->mode == S_MODE_LGL))
			bRet = LOGICAL_DATA(ps_logical)[0];

		DecrRef(ps_call, TRUE, FALSE);
		FreeHeader(ps_call->value.tree[1]);
		FreeHeader(ps_call->value.tree[2]);
		//free ps_call but not its children
		TryToFree(ps_call, FALSE);

	}
	catch(...)
	{
		bRet = FALSE;
	}
	return bRet;
}

//Get the entry point of a C/Fortran routine: search based on the S search list
void* CSPevaluator::GetEntry(const char* pszRoutineName /* in: routine name */) const
{
	Validate();
	void* pRoutine = NULL;
	try
	{
		pRoutine = (void*)(::get_entry(const_cast<char*>(pszRoutineName), GetPtr()));
	}
	catch(...)
	{
	}
	return pRoutine;
}
//Get the entry point of a C/Fortran routine: search based on the S search list
void* CSPevaluator::GetEntry(s_object* ps_chRoutineName /* in: routine name as a scalar char obj*/) const
{
	Validate();
	void* pRoutine = NULL;
	try
	{
		CSPcharacter schRoutineName(ps_chRoutineName);
		pRoutine = (void*)(::get_entry(schRoutineName[0], GetPtr()));
	}
	catch(...)
	{
	}
	return pRoutine;
}

void CSPevaluator::TryToFree(s_object *ps_object, BOOL bRecursive) const
{
	Validate();
	::try_to_free(ps_object, static_cast<s_boolean>(bRecursive), GetCurrentEvalFrame(), GetPtr());
}

void CSPevaluator::IncrRef(s_object *ps_object, BOOL bRecursive) const
{
	Validate();
	s_object* ps_frame = (ps_object->frame!=NULL)?ps_object->frame: GetCurrentEvalFramePtr();
	::incr_ref_count(ps_object, static_cast<s_boolean>(bRecursive), ps_frame, GetPtr());
}

void CSPevaluator::DecrRef(s_object *ps_object, BOOL bRecursive, BOOL bTryToFree ) const
{
	s_object* ps_frame = NULL;
	if(bTryToFree)
	{
		if(ps_object->frame)
			ps_frame = ps_object->frame;
		else
			ps_frame = GetCurrentEvalFramePtr();
	}
	::decr_ref_count(ps_object, (s_boolean)bRecursive, ps_frame, GetPtr());
}

int CSPevaluator::GetRefCount(s_object* ps_object ) const
{
	return ::get_ref_count(ps_object, GetPtr());
}
//Set the ref. count, but only of the top-level block.  Use
//::SetPermRefCount to set ref. count to -1; incr_ref_count for most other cases.
void CSPevaluator::SetRefCount(s_object* ps_object, int nCount ) const
{
	::set_ref_count(ps_object, nCount, GetPtr());
}

void CSPevaluator::SetPermRefCount(s_object* ps_object, s_object* ps_frame ) const
{
	Validate();
	::set_perm_ref(ps_object, ps_frame,  GetPtr());
}

//Get a new header for the specified object: same as the global function s_header_for_object
s_object* CSPevaluator::CreateNewHeader(const s_object *ps_object) const
{
	s_object* ps_new_object = ::New_vector(GetPtr());
	*ps_new_object = *ps_object;
	return ps_new_object;
}
//Return TRUE if the object has ref. count <0 or > 1
BOOL CSPevaluator::IsSharing(const s_object *ps_object) const
{
	return ::multiple_ref_count(const_cast<s_object*>(ps_object), GetPtr());
}

//Copy the header and the first level of arena:
//wrapping the global function copy_lev(ps_object, 0, ...)
s_object* CSPevaluator::CopyForWrite(const s_object* ps_object) const
{
	return ::copy_lev(const_cast<s_object*>(ps_object), 0, NULL, GetPtr());
}

//Create a new vector
s_object* CSPevaluator::alcvec(long lMode, long len) const
{
	return ::alcvec( lMode, len, GetPtr());
}

//Coerce a vector
s_object* CSPevaluator::coevec(s_object* ps_vector, long lMode) const
{
	return (ps_vector!=NULL)? ::coevec(ps_vector, lMode, S_FALSE, CHECK_IT, GetPtr()) : NULL;

}

void CSPevaluator::Remove(const char* pszName, long lDataBase) const
{
	long lPosition = S_evaluator->_search_data_index[lDataBase-1];
	::do_rm(::make_name(pszName, GetPtr()), lPosition, S_FALSE, GetPtr());
}

void CSPevaluator::FreeHeader(s_object* ps_object) const
{
	if(ps_object != NULL)
		::free_header(ps_object, GetPtr());
}

//ReplaceElement of a list or recursive object: decr. ref count the old child and incr. the new one.
void CSPevaluator::ReplaceElement(s_object* ps_parent, long lZeroIndex, s_object* ps_child) const
{
	//Parent must be a list
	if(ps_parent->mode != S_MODE_LIST)
		SCONNECT_ThrowException("Internal error: list object is expected in CSPevaluator::ReplaceElement");
	DecrRef(ps_parent->value.tree[lZeroIndex]);
	IncrRef(ps_child);
	ps_parent->value.tree[lZeroIndex] = CreateNewHeader(ps_child);
}

s_object* CSPevaluator::As(s_object* ps_object, s_class* ps_class) const
{
	s_object* ps_value=NULL;
	if(SPL_NotThere(ps_object))
		ps_value = NULL;
	else if(IS(ps_object, ps_class))
		ps_value = ps_object;
	else if(GetRefCount(ps_object) == 0)
		ps_value = AS(ps_object, ps_class);
	else
		ps_value = AS(CreateNewHeader(ps_object), ps_class);
	return ps_value;
}

