/*
	Connect/C++ : Copyright (c) 2001, 2006 Insightful Corp.
	All rights reserved.
	Version 6.0: 2001

	spobject.h: interface for the CSPobject class.
*/
#if !defined(__SCONNECT_SPOBJECT_H_INCLUDED__)
#define __SCONNECT_SPOBJECT_H_INCLUDED__
#if _MSC_VER > 1000
#pragma once
#endif // _MSC_VER > 1000
#include "S.h"
#include "eval.h"
#include "spdefs.h"
#include "speval.h"
class CSPproxy;
class CSPengineConnect;

// CSPobject -the base class for all CSPxxx classes
class SCONNECT_LIB_EXTERN CSPobject 
{
//////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////
public:
	//Default constructor
	CSPobject();
	//Copy constructor, also copy appropriate data member
	CSPobject(const CSPobject& sObject); 
	//Construct from a valid S-expression
	explicit CSPobject(const char* pszExpression);
	//Construct from s_object* 
	CSPobject(s_object* ps_object, BOOL bTryToFreeOnDetach=FALSE);
	//Create an S object and attach via the virtual member function Attach() which expect to coerce to required class
	virtual BOOL Create(const char* pszExpression=NULL,	const char* pszName=NULL,	long lDataBase=1 );				
	//The destructor
	virtual ~CSPobject();
//////////////////////////////////////////////////////
// Attributes
//////////////////////////////////////////////////////
public:
	//Validating s_object*
	virtual BOOL IsValid(void) const;
	virtual BOOL IsValidAddress(void) const;
	virtual void Validate(void) const;
	//Casting operator to s_object*
	operator s_object*() const 
	{ 
	 //Set flag so the destructor won't try to free memory when ref. count drops to zero.
		const_cast<CSPobject*>(this)->SetTryToFreeOnDetach(FALSE); 
		return m_ps_object;
	}
	//Direct access to m_ps_object 
	s_object* GetPtr(void) const 
	{ return m_ps_object; }
	//Direct access to m_ps_object 
	s_object* operator& (void) const 
	{ return m_ps_object;	}
	//Direct access to union member of m_ps_object->value , e.g.: sObj->value.Double
	s_object* operator-> (void) const 
	{ return m_ps_object; }
	//Return S_MODE: double, integer, char, etc..
	virtual int GetMode(BOOL bValidate=TRUE) const;
	virtual int GetDataMode(BOOL bValidate=TRUE) const { return GetMode(bValidate); };
	virtual long GetLength(BOOL bValidate=TRUE) const;
	//Coerce the data to class given by pszClass	
	BOOL BaseCoerce(const char* pszClass);
	//Intends to be use by CSPproxy to match with CSParray::GetData()
	//One-based indexing
	virtual s_object* GetVector(long lIndex=1) const;
	//Accessing reference counting
	virtual void IncrRef(void) const;
#ifdef WIN32
	virtual void DecrRef(BOOL bTryToFree=FALSE) const; //try to free if bTryToFree is TRUE & ref. count drops to zero
#else	     // see comment in spobject.cxx.
	virtual void DecrRef(BOOL bTryToFree=FALSE);
#endif
	virtual int GetRefCount(void) const;
	BOOL IsSharing(BOOL bValidate=TRUE) const;
	virtual s_object* Detach(BOOL bTryToFree=FALSE); 
	virtual s_object* Release(BOOL bTryToFree=TRUE);
	virtual void ReAttachAndAssign(s_object *ps_object, long lDataBase=1);
	//Assignment operator
	const CSPobject& operator = (const CSPobject& sobject);
	//Get storage frame of the header
	long GetHeaderFrame(BOOL bValidate=TRUE) const;
	//Copy if needed: always returns a new header, copy the FIRST level of arena if needed.
	s_object* CopyForWrite(BOOL bValidate=TRUE); 
	//Create a new header for this object.
	s_object* CreateNewHeader(BOOL bValidate=TRUE) const; 
protected:
	//Coerce the data to class given by pszClass and attach
	BOOL BaseCoerceAttach(s_object* ps_object, const char* pszClass, BOOL bTryToFreeOnDetach=FALSE);
	BOOL BaseCoerceAttach(s_object* ps_object, s_class* ps_class, BOOL bTryToFreeOnDetach=FALSE);

//////////////////////////////////////////////////////
// Operations
//////////////////////////////////////////////////////
public:
	//IMPORTANT: The overwriten method of a derived class must call this base class method last.
	//Coerce, decr/incr ref. count and attach.
	virtual void Attach(s_object *ps_object, BOOL bTryToFreeOnDetach=FALSE); //=0 (almost) derive class must override!
	//Return the persistant object from the S databases: the first one in the search list will be returned.
	BOOL get(const char* pszName);
	//assign/reassign and commit it to disk 
	virtual void Assign(const char* pszName, long lDataBase=1); 
	virtual void AttachAndAssign(s_object *ps_object, const char* pszName=NULL, long lDataBase=1);
	//Commit permanent object to database after any modifications
	//NOTE: Assigns modified object to database 1 by default !
	virtual BOOL Commit( long lDataBase=1 );
	//Remove permanent object from database 
	virtual BOOL Remove( void );
	//Print to the S+stdout
	virtual CSPobject Print(BOOL bAsText=FALSE)const;	//eval the print method
	CSPobject print(void) const {return Print();}	
	//Clone this object to current allocation frame which usally is the same current evaluation frame
	virtual CSPobject Clone(BOOL bValidate=TRUE) const;
	//Clone this object to permance allocation frame which will persist through-out the current S+session
	virtual CSPobject ClonePermanent(BOOL bValidate=TRUE) const;
	//Allocate temporary string, char*, from the current evaluation frame: same as S_alloc().
	virtual char * AllocateStringInFrame( const char *pszString, BOOL bValidate = TRUE );
	//Return S class
	const s_class* GetClass(BOOL bValidate=TRUE) const;
	//Return the storage frame.
	long GetStorageFrame(BOOL bValidate=TRUE) const;
	//Return S_TRUE if the class of this object is pszClass, else S_FALSE or S_MAYBE
	s_boolean Is(const char* pszClass) const;	
	//Return TRUE if this object is a child of ps_parent
	BOOL IsChildOf(const s_object* ps_parent) const;
	//Return TRUE if this object is a parent of ps_child
	BOOL IsParentOf(const s_object* ps_child) const;
	long GetSearchPathPosition( BOOL bValidate = TRUE )const;
	//Deparse the call object and return text.
	CSPobject Deparse(BOOL bValidate=TRUE);
	//Return number of rows
	virtual long GetNRow(BOOL bValidate=TRUE) const;
	virtual long nrow(BOOL bValidate=TRUE) const {return GetNRow(bValidate);}
	//Return number of columns
	virtual long GetNCol(BOOL bValidate=TRUE) const;
	virtual long ncol(BOOL bValidate=TRUE)const {return GetNCol(bValidate);}

#ifdef WIN32
	#include "spobject_com.h"
#endif

public:
	//Object map support for use in S_notification to call virtual fn on notification (CRD)
	static CSPobject * S_STDCALL ObjectMap_Get( const char *pszObjectName );
	static BOOL S_STDCALL ObjectMap_Add( const char *pszObjectName, CSPobject *pcsObject );
	static BOOL S_STDCALL ObjectMap_Remove( const char *pszObjectName );
	static BOOL S_STDCALL ObjectMap_RemoveAll( void );
	//Return TRUE if one of objects is parent of ps_child
	static BOOL S_STDCALL ObjectMap_Exists(const s_object* ps_child);
	//Unnamed object set support for use in CSPengineConnect::CloseTopLevelEval() to
	//invalidate CSPobject member m_ps_object 
	static BOOL S_STDCALL UnnamedObjectMap_Add( CSPobject *pCSPobject );
	static BOOL S_STDCALL UnnamedObjectMap_Remove( CSPobject *pCSPobject );
	static BOOL S_STDCALL UnnamedObjectMap_RemoveAll( void );
	static BOOL S_STDCALL UnnamedObjectMap_InvalidateAll( void );
	static BOOL S_STDCALL UnnamedObjectMap_InvalidateLocalObjects(BOOL bError=FALSE);
	static BOOL UnnamedObjectMap_PopFrame( long lParentFrame );
	//Return TRUE if one of objects is parent of ps_child
	static BOOL S_STDCALL UnnamedObjectMap_Exists(const s_object* ps_child);
	//Single session engine connection support
	static CSPengineConnect * S_STDCALL GetEngineConnection( void );
	static void SetEngineConnection( CSPengineConnect *pEngineConnection );

	virtual BOOL EnableOnModify( BOOL bEnable );
	virtual BOOL EnableOnRemove( BOOL bEnable );
	virtual BOOL IsEnabledOnModify( void ) { return	m_bEnableOnModify; };
	virtual BOOL IsEnabledOnRemove( void ) { return	m_bEnableOnRemove; };
	//Override these in your inherited CSPobject classes for S_MSG_MODIFY and 
	//S_MSG_REMOVE messages from the engine
	//Last argument, ps_attached, is a S-PLUS attached object that uniquely   
	//identfies the database where the object resides
	virtual int OnModify( s_object *ps_objectOld, s_object *ps_objectNew, s_object* ps_attached );
	virtual int OnRemove( s_object *ps_objectOld, s_object* ps_attached );
	virtual int OnPreModify( s_object **pps_object );
	virtual long OnNotifyOrQuery(
		long lMsg,        /* message Id */
		long lArgs,       /* number of args */
		void** ppArgs    /* content depends on lMsg with length lArgs */
	);
  //If the arg is TRUE, try to free when ref. count drops to zero by this destructor.
	BOOL GetTryToFreeOnDetach(void) const
  {return m_bTryToFreeOnDetach;}
  void SetTryToFreeOnDetach(BOOL bTryToFree)
  {m_bTryToFreeOnDetach= bTryToFree;}
	//Access the object name, normally a persistent object.  This name is usally a file name in S database.
	const char* GetObjectName(void) const 
	{return (const char* ) m_pszObjectName;}
	void SetObjectName( const char *pszObjectName, BOOL bAddToMap=FALSE );
	void Init(void); //Init all data members.
	friend class CSPproxy;
protected:
	//Create an S object and coerce it to the specified class 
	BOOL BaseCreate(const char* pszClass, const char* pszExpression=NULL, const char* pszName=NULL, long lDataBase=1 );
	BOOL BaseCreate(s_class* ps_class, const char* pszExpression=NULL, const char* pszName=NULL, long lDataBase=1 );
//////////////////////////////////////////////////////
//Data Members and private member functions:
//////////////////////////////////////////////////////
private:
	s_object* m_ps_object;      //the S object pointer: must always be the first data member
	BOOL m_bTryToFreeOnDetach;  //If TRUE, m_ps_object can be free when its ref. count drops to zero.
	char *m_pszObjectName;      // Object name stored in this class for use in Attach() and Detach()
	BOOL m_bEnableOnModify;
	BOOL m_bEnableOnRemove;
	static CSPengineConnect *m_pEngineConnection;
#ifdef _DEBUG
	static long m_lTotalRefCount; //Total ref-counts on all CSPobject
#endif

}; // end class CSPobject

///////////////////////////// Gobal functions ////////////////////////

inline CSPobject  operator+(const CSPobject& sLhs, const CSPobject& sRhs)
{
	return CSPobject(SPL_BinaryOp(&sLhs, &sRhs, S_BINOP_PLUS), TRUE);
}
inline CSPobject  operator-(const CSPobject& sLhs, const CSPobject& sRhs)
{
	return CSPobject(SPL_BinaryOp(&sLhs, &sRhs, S_BINOP_MINUS), TRUE);
}
inline CSPobject  operator*(const CSPobject& sLhs, const CSPobject& sRhs)
{
	return CSPobject(SPL_BinaryOp(&sLhs, &sRhs, S_BINOP_MULTIPLY), TRUE);
}
inline CSPobject  operator/(const CSPobject& sLhs, const CSPobject& sRhs)
{
	return CSPobject(SPL_BinaryOp(&sLhs, &sRhs, S_BINOP_DIVIDE), TRUE);
}

inline CSPobject  operator+(double dLhs, const CSPobject& sRhs)
{
	CSPevaluator sEvaluator;
	s_object* ps_lhs = sEvaluator.alcvec(S_MODE_DOUBLE, 1L);
	NUMERIC_POINTER(ps_lhs)[0]= dLhs;
	CSPobject sLhs(ps_lhs, TRUE);
	return CSPobject(SPL_BinaryOp(&sLhs, &sRhs, S_BINOP_PLUS), TRUE);
}
inline CSPobject  operator-(double dLhs, const CSPobject& sRhs)
{
	CSPevaluator sEvaluator;
	s_object* ps_lhs = sEvaluator.alcvec(S_MODE_DOUBLE, 1L);
	NUMERIC_POINTER(ps_lhs)[0]= dLhs;
	CSPobject sLhs(ps_lhs, TRUE);
	return CSPobject(SPL_BinaryOp(&sLhs, &sRhs, S_BINOP_MINUS), TRUE);
}
inline CSPobject  operator*(double dLhs, const CSPobject& sRhs)
{
	CSPevaluator sEvaluator;
	s_object* ps_lhs = sEvaluator.alcvec(S_MODE_DOUBLE, 1L);
	NUMERIC_POINTER(ps_lhs)[0]= dLhs;
	CSPobject sLhs(ps_lhs, TRUE);
	return CSPobject(SPL_BinaryOp(&sLhs, &sRhs, S_BINOP_MULTIPLY), TRUE);
}
inline CSPobject  operator/(double dLhs, const CSPobject& sRhs)
{
	CSPevaluator sEvaluator;
	s_object* ps_lhs = sEvaluator.alcvec(S_MODE_DOUBLE, 1L);
	NUMERIC_POINTER(ps_lhs)[0]= dLhs;
	CSPobject sLhs(ps_lhs, TRUE);
	return CSPobject(SPL_BinaryOp(&sLhs, &sRhs, S_BINOP_DIVIDE), TRUE);
}

inline CSPobject  operator+( const CSPobject& sLhs, double dRhs)
{
	CSPevaluator sEvaluator;
	s_object* ps_rhs = sEvaluator.alcvec(S_MODE_DOUBLE, 1L);
	NUMERIC_POINTER(ps_rhs)[0]= dRhs;
	CSPobject sRhs(ps_rhs, TRUE);
	return CSPobject(SPL_BinaryOp(&sLhs, &sRhs, S_BINOP_PLUS), TRUE);
}
inline CSPobject  operator-( const CSPobject& sLhs, double dRhs)
{
	CSPevaluator sEvaluator;
	s_object* ps_rhs = sEvaluator.alcvec(S_MODE_DOUBLE, 1L);
	NUMERIC_POINTER(ps_rhs)[0]= dRhs;
	CSPobject sRhs(ps_rhs, TRUE);
	return CSPobject(SPL_BinaryOp(&sLhs, &sRhs, S_BINOP_MINUS), TRUE);
}
inline CSPobject  operator*( const CSPobject& sLhs, double dRhs)
{
	CSPevaluator sEvaluator;
	s_object* ps_rhs = sEvaluator.alcvec(S_MODE_DOUBLE, 1L);
	NUMERIC_POINTER(ps_rhs)[0]= dRhs;
	CSPobject sRhs(ps_rhs, TRUE);
	return CSPobject(SPL_BinaryOp(&sLhs, &sRhs, S_BINOP_MULTIPLY), TRUE);
}
inline CSPobject  operator/( const CSPobject& sLhs, double dRhs)
{
	CSPevaluator sEvaluator;
	s_object* ps_rhs = sEvaluator.alcvec(S_MODE_DOUBLE, 1L);
	NUMERIC_POINTER(ps_rhs)[0]= dRhs;
	CSPobject sRhs(ps_rhs, TRUE);
	return CSPobject(SPL_BinaryOp(&sLhs, &sRhs, S_BINOP_DIVIDE), TRUE);
}

#ifdef __cplusplus
extern "C" {
#endif
//Utility functions
SCONNECT_DLLAPI(char *) SPL_AllocateStringInFrame( s_object *ps_object, const char *pszString );
SCONNECT_DLLAPI(s_object*) S_SyncParseEval_GetWarnings(const char* pszExpression, s_object **warningList);
SCONNECT_DLLAPI(s_object*) S_SyncParseEval(const char* pszExpression);
//Return TRUE if this object is a parent of ps_child
SCONNECT_DLLAPI(BOOL) SPL_IsParentOf(const s_object* ps_parent, const s_object* ps_child);
// Notification from the engine via variable argument messaging
SCONNECT_DLLAPI(long) S_EngineMessage(long lMsg,/* message Id */long lArgs, /* number of args */ void** ppArgs /* content depends on lMsg with length lArgs */);    
// Access to the evaluator data
SCONNECT_DLLAPI(long) SpGetCurrentFrame(void);
SCONNECT_DLLAPI(BOOL) SpValidEvaluator(void);
SCONNECT_DLLAPI(BOOL) SPL_NotThere(const s_object* ps_object );
SCONNECT_DLLAPI(BOOL) SPL_IsAtomicVector(const s_object* ps_object );
SCONNECT_DLLAPI(s_object*) SPL_ProtectVector(s_object* ps_vector, BOOL bCopy);
SCONNECT_DLLAPI(s_object*) SPL_NewHeaderForInsertion(s_object* ps_vector);
SCONNECT_DLLAPI(s_object*) SPL_NewHeaderForInsertion2(s_object* ps_vector);

#ifdef __cplusplus
}
#endif

#endif // !defined(__SCONNECT_SPOBJECT_H_INCLUDED__)

