package Splus::SINSTALL;
use v5.8;
use strict ;

=head1 NAME

  Splus::SINSTALL - install S code files to a given package/.Data directory

=head1 SYNOPSIS

  The following is the plan.  Only the ones before the bar are implemented.

  use Splus::SINSTALL;
  $sh = Splus::SINSTALL->new(@ARGV) ; # args: DESTDIR file1.q file2.R ...
  $sh->make() ; # do the installation
  $sh->make_virgin() ; # remove all generated files, including destdir
  $sh->make_from_scratch(); # make_virgin then make
  $sh->add_source_files("foo.q"); # add source file to list
  $sh->delete_source_files("foo.q"); # remove source file from list
  $sh->get_source_files("foo.q"); # return list of *.q *.R files
  --- bar ---
  $sh->tidy_after_build() ; # do nothing.  Supplied to be like SHLIB.

=cut

use Splus::Vars;
use Splus::Utils;
use Splus::SplusUtils;
use Cwd qw(getcwd abs_path) ;

use Data::Dumper;

sub _Syntax {
    print "SINSTALL: install S code files\n";
    print "-h,--help          # print this help file\n";
    print "--destdir dirName  # install to directory dirName (don't include .Data)\n";
    print "--clean-first      # remove all installed Splus objects reinstalling functions\n";
    print "All .S, .s, .R, .q, .ssc, and .SSC files in current directory will be installed\n";
    print "You can use file1.S file2.S to install just a subset of them\n";
    @_ && die join(", ", @_) ;
}

sub new {
    # Act like old Unix QINSTALL : QINSTALL DESTDIR file1.q file2.R
    # which installs S objects defined in file to DESTDIR/.Data
    # We changed syntax to require '--destdir DESTDIR'.
    # $tmp = Splus::SINSTALL::new qw(--destdir DESTDIR --clean-first file1.q file2.R)
    # sets things up and $tmp->make() does the installation.
    my $class = shift ;
    my $self = {} ;
    my $ret = bless $self, $class ;
    $self->{srcs} = {};
    $self->{src_id} = 0 ;
    my $src_supplied = 0 ;

    while (@_) {
        $_ = shift @_ ;
        if (/^(-h|--help)/) {
            _Syntax() ;
            exit();
        } elsif (/^(-v|--verbose)/) {
            $self->{verbose} = 1 ;
        } elsif (/^--clean-first/) {
            $self->{clean_first} = 1 ;
        } elsif (/^--clean-after/) { # no clean_after action now.
            $self->{clean_after} = 1 ;
        } elsif (/^--no-clean-after/) {
            $self->{clean_after} = 0 ;
        } elsif (/^--destdir/) {
            $self->{destdir} = shift @_ or _Syntax "No directory name after --destdir" ;
            $self->_canonicalize_destdir() ;
        } else {
            $self->add_source_files($_) ;
            $src_supplied = 1 ;
        }
    }
    $self->{destdir} or _Syntax "No --destdir directory given" ;
    if (!$src_supplied) {
        $self->add_source_files(globlist(qw(*.q *.R *.ssc *.SSC *.S *.s))) ;
    }
    $ret ;
}

sub _mkdir
{
    # not for public use.
    # make a directory if it does not exist.  Die if problems arise.
    my $dir = shift ;
    if (! -d $dir ) {
       die "A non-directory file $dir exists and we need to make a directory by that name" if -e $dir ;
       mkdir $dir or die "Cannot make directory $dir ($!)" ;
    }
}

sub _prepare_destdir
{
    # make sure destdir has a .Data with right components in it.
    # Don't touch a good one, but fix up incomplete ones.
    # Die if destdir itself does not exist or if there are
    # any problems making .Data and its subdirectories.
    # Should have __Hhelp/  __Meta/  __Shelp/
    # and empty file ___nonfile (unix) or ___nonfi (windows or windows-compatible).
    # For now, we will leave out the nonfile.
    my $self = shift ;
    my $destdir = $self->{destdir} ;
    _Syntax "Destination directory $destdir does not exist or is not a directory" if ! -d $destdir ;
    _mkdir ("$destdir/.Data") ;
    _mkdir ("$destdir/.Data/__Meta") ;
    _mkdir ("$destdir/.Data/__Hhelp") ;
    _mkdir ("$destdir/.Data/__Shelp") ;
    -w "$destdir/.Data" or die abs_path("$destdir/.Data") . " is not writable" ;
    -w "$destdir/.Data/__Meta" or die abs_path("$destdir/.Data/__Meta") . " is not writable" ;
}

sub _clean_destdir_data
{
    # remove Splus objects from .Data (but not help files)
    # In particular, get rid of __nonfi* file (installFromSFiles()
    # can remove all objects, but not the __nonfi* file).
    # If is fine if .Data does not exist, but if it does
    # it must be writable.
    my $self = shift ;
    my $destdir = $self->{destdir} ;
    my $dot_data = "$destdir/.Data" ;
    my $meta = "$dot_data/__Meta" ;
    foreach my $dir ($meta, $dot_data) {
        if (-d "$dir") {
            -w "$dir" or die "Directory " . abs_path("$dir") . " is not writable" ;
        }
    }
    foreach my $dir ($meta, $dot_data) {
        if (-d "$dir") {
            $self->{verbose} and print "$0: removing files from " . abs_path("$dir") . "\n" ;
            opendir my $dirhandle, $dir or die "Cannot open $dir ($!)";
            while (my $filename = readdir $dirhandle) {
                next if $filename =~ /^(\.|\.\.|\.Audit)$/ ;
                my $fullname = "$dir/$filename" ;
                next if -d $fullname ;
                unlink $fullname or warn "Could not remove file $fullname, .Data may not be clean" ;
            }
            closedir $dirhandle ;
        }
    }
}

sub _canonicalize_destdir
{
    # not for end-user use
    # No arguments (except implicit $self).
    # Look at destdir and remove possible trailing /.Data
    # Also, change any backslashes to slashes.
    # This will only be used by perl and Splus, not by cmd.exe.
    my $self = shift ;
    $self->{destdir} =~ s^\\^/^g ;
    $self->{destdir} =~ s^/\.Data$^^i ;
    -d $self->{destdir} or _Syntax "--destdir $self->{destdir} does not name a directory" ;
}

sub add_source_files
{
    my $self = shift ;
    foreach my $arg (@_) {
        foreach my $source_file (glob($arg)) {
           -e $source_file or _Syntax "S code file $source_file does not exist" ;
           -d $source_file and _Syntax "Putative S code file $source_file is a directory (use --destdir dir)" ;
           ${$self->{srcs}}{"$source_file"}=$self->{src_id}++ ;
        }
    }
}
sub delete_source_files
{
    my $self = shift ;
    foreach my $source_file (@_) {
        delete ${$self->{srcs}}{"$source_file"} ;
    }
}

sub get_source_files
{
    my $self = shift ;
    # sort to order in which they were given
    my @sorted_srcs = sort { ${$self->{srcs}}{$a} <=> ${$self->{srcs}}{$b} } keys(%{$self->{srcs}}) ;
    @sorted_srcs ;
}

sub make
{
    # the DESCRIPTION file stuff depends on SINSTALL being run in a
    # subdirectory of a package (so ../DESCRIPTION relates to this code).
    # Same code is in DATAINSTALL.pm.
    my $self = shift ;
    my @srcs = get_source_files($self) ;
    @srcs = map { $_ =~ s^\\^/^g ; $_ } @srcs ; # To avoid Splus backslash issues on Windows (this is only used in S code)
    if ( scalar @srcs == 0) {
        print "$0 : No S code to install\n" ;
    } else {
        print "Installing S code to directory " . abs_path($self->{destdir}) . " from the ", scalar @srcs, " file(s) ", join(", ", @srcs), "\n" ;
        $self->_prepare_destdir() ;
        $self->_clean_destdir_data() if $self->{clean_first} ;
        my $args = "c('" . join("', '", @srcs) . "')" ;
        $args .= ", where='" . $self->{destdir} . "'" ;
        $args .= ", clean.first=TRUE" if $self->{clean_first} ;
        my $descriptionFile = "../DESCRIPTION" ;
        $args .= ", descriptionFile='$descriptionFile'" if -e $descriptionFile ;
        my $cmd = "installFromSFiles(" . $args . ");" ;
	$self->{verbose} and print "$0: Splus cmd=$cmd\n";
        my $had_dot_data = -d ".Data" ;
        my @out = Splus_run($cmd, "-vanilla -quitonerror");
        if (!$had_dot_data && -d ".Data") {
            # work around bug on Windows: Splus START -vanilla  makes .Data
            rm_empty_dot_data();
        }
        print "$0: Splus output:\n", join("\n\t", @out), "\n";
        # Splus_run_ex would return exit status, but Splus on Windows
        # always gives exit status of 0.  Hence we grep for string
        # that quitonerror produces.
        map { die("Error installing S code") if /^Quitting Splus session because of error$/ || /^Terminating S Session:/ } @out ;
    }
    1 ; # should return status indicator, or die if make failed
}

1;
