package Splus::SWINGUIINSTALL;
use v5.8;
use strict ;

=head1 NAME

  Splus::SWINGUIINSTALL - compile and install Splus Windows GUI code files in the

=head1 SYNOPSIS

  The following is the plan.  Only the ones before the bar are implemented.

  use Splus::SWINGUIINSTALL;
  $sh = Splus::SWINGUIINSTALL->new(@ARGV) ; # args: --destdir DESTDIR
  $sh->make() ; # do the installation
  --- bar ---

=head1 NOTES
  The expects that the Splus Windows GUI code is stored in the following directories.
  * create/ contains S code files with guiCreate() calls for "Property"
    and "FunctionInfo" GUI objects.  These calls create files in pkg/.Prefs.
  * support/ contains S code files defining functions used for
    callbacks, creating GUI objects on the fly, etc.  These assignments
    create things in pkg/.Data.

=cut

use Cwd qw(getcwd abs_path) ;
use File::Path; # for rmtree(), to remove directory recursively
use Splus::Vars;
use Splus::Utils;
use Splus::SplusUtils;
use Splus::SINSTALL;

Splus::Vars::error("SHOME");
my $SHOME=${Splus::Vars::SHOME} ;
my $ms = ${Splus::Vars::S_BUILD_SYS} =~ /^MS/i ;
if ($ms && $SHOME =~ / /) { $SHOME=dosify($SHOME); }

Splus::Vars::error("MAKE");
Splus::Vars::error("OSTYPE");
my $osname = ${Splus::Vars::OSTYPE} eq "windows" ? "windows" : $^O ; # "windows", "linux", "solaris", "aix", etc.

sub _Syntax {
    print "-h,--help             # print this help message\n";
    print "--destdir dirName" ;  # put .Prefs files into dirName/.Prefs
    @_ && die join(", ", @_) ;
}

sub new {
    # sets things up and $tmp->make() does the installation.
    my $class = shift ;
    my $self = {} ;
    my $ret = bless $self, $class ;

    while (@_) {
        $_ = shift @_ ;
        if (/^(-h|--help)/) {
            _Syntax() ;
            exit() ;
        } elsif (/^(-v|--verbose)/) {
            $self->{verbose} = 1 ;
        } elsif (/^--clean-first/) {
            $self->{clean_first} = 1 ;
        } elsif (/^--clean-after/) { # there is no clean_after action, but supply argument for consistency
            $self->{clean_after} = 1 ;
        } elsif (/^--no-clean-after/) {
            $self->{clean_after} = 0 ;
        } elsif (/^--destdir/) {
            $self->{destdir} = shift @_ or _Syntax("No directory name after --destdir");
            $self->_canonicalize_destdir() ;
        }
    }
    $self->{destdir} or _Syntax "No --destdir directory given" ;

    if ("$osname" eq "windows") {
        $self->{doNothing} = 0 ;
        $self->{guicreateDir} = "guicreate";
        $self->{RDir} = "R";
        $self->{htmlDir} = "html";
    } else {
        $self->{doNothing} = 1 ;
    }
    $ret ;
}

sub _canonicalize_destdir
{
    # not for end-user use
    # No arguments (except implicit $self).
    # Look at destdir and change any backslashes to slashes.
    # This will only be used by perl and Splus, not by cmd.exe.
    my $self = shift ;
    $self->{destdir} =~ s^\\^/^g ;
    -d $self->{destdir} or die "--destdir $self->{destdir} does not name a directory" ;
}

sub make_property_files
{
    my $self = shift ;
    my $cwd = getcwd() ;
    if ( ! -d $self->{guicreateDir}) {
        return ;
    }
    eval {
        chdir "$self->{guicreateDir}" or die "Cannot chdir $self->{guicreateDir} ($!)" ;
        File::Path::rmtree(".Prefs") if -d ".Prefs" ;
        File::Path::rmtree(".Data")  if -d ".Data" ;
        print "Processing guiCreate calls in swingui/$self->{guicreateDir}\n" ;
        my @splus_out = _Splus_run_gui($self, "dir(\".Prefs\", pattern=\"package\")\n.processGuiCreateCalls()\ndir(\".Prefs\", pattern=\"package\")");
        File::Path::rmtree(".Data")  if -d ".Data" ;
        my $n_err = 0 ; my $n_warn = 0 ;
        map { $n_warn += /^(There were [0-9]+ warnings |Warning messages:|Warning: )/ ; $n_err += /^Problem in/ ; } @splus_out ;
        print "   There were $n_err errors and $n_warn warnings in swingui/$self->{guicreateDir}\n";
        if ($n_err + $n_warn > 0) {
            print join("\n\t", @splus_out), "\n" ;
        }
        if ($n_err > 0) {
            die "There were errors processing the guiCreate() calls in swingui/$self->{guicreateDir}\n";
        }
        opendir my $d, ".Prefs" or die "Splus.exe did not create $cwd/.Prefs as expected" ;
        map { ! -d ".Prefs/$_" && ! /^package\./i && unlink(".Prefs/$_") } readdir $d ;
        closedir $d ;
    } ;
    die("Problem in SWINGUIINSTALL:make_property_files: $@") if $@ ;
    chdir "$cwd" or die "Cannot return to directory $cwd in SWINGUIINSTALL:make_property_files ($!)";
    File::Path::rmtree("$self->{destdir}/.Prefs") if -d "$self->{destdir}/.Prefs" ;
    mkdir "$self->{destdir}/.Prefs" or die "Cannot create directory $self->{destdir}/.Prefs ($!)" ;
    opendir my $d, "$self->{guicreateDir}/.Prefs" ;
    map { -f "$self->{guicreateDir}/.Prefs/$_" && Splus::SplusUtils::_copy_file("$self->{guicreateDir}/.Prefs/$_", "$self->{destdir}/.Prefs/$_", "asis") } readdir $d ;
    closedir $d ;
    File::Path::rmtree("$self->{guicreateDir}/.Prefs") ;
}

sub make_R_files {
    my $self = shift ;
    return if ! -d "R" ;
    my $exts = Splus::Utils::make_file_exts("code") ;
    my @R_files = list_files_with_exts("R", $exts) ;
    return if ! @R_files ;
    my @sinstall_args = () ;
    push @sinstall_args, "--destdir", "$self->{destdir}" ;
    my $sinstall = Splus::SINSTALL->new(@sinstall_args, @R_files);
    $sinstall->make();
}

sub make
{
    my $self = shift ;
    if ($self->{doNothing}) { return ; }
    my $cwd = getcwd() ;
    print "SWINGUIINSTALL::make: cwd=$cwd\n";
    make_property_files($self) ;
    make_R_files($self) ;
    1 ; # should return status indicator, or die if make failed
}

sub _Splus_run_gui {
    my $self = shift ;
    my $splus_command = shift ;

    my $splus_in = Splus_tempfile("in") ;
    open my $to_splus, ">", $splus_in or die "Cannot create Splus input file" ;
    print $to_splus "$splus_command\n";
    close $to_splus ;

    my $splus_out = Splus_tempfile("out") ;
    my $wd = Splus_cwd() ; $wd =~ s^/^\\^g ;

    # The following 'args' use Splus START
    # my @args = qw ( Splus.bat START --vanilla --quiet ) ;
    # push @args, "S_CWD=$wd" ;
    # push @args, "S_PROJ=$wd" ;
    # push @args, "--input", "$splus_in" ;
    # push @args, "--output", "$splus_out" ;

    # The following 'alt_args' use Splus.exe directly
    my @alt_args = qw(start " " /wait splus.exe S_CWD=".") ;
    push @alt_args, "S_PROJ=$wd" ;
    push @alt_args, "/MULTIPLEINSTANCES", "/NOSAVEREGPATHS", "/NOSTARTUPDLGS" ;
    # do we also want /NOSTARTUPDLGSATALL ?
    push @alt_args, "/NOSTATUSDLGS" ;
    push @alt_args, "/BATCH", "\"$splus_in\"", "\"$splus_out\"" ;

    # print "Want to invoke: ", join(" ", @alt_args), "\n" ;
    my @from_splus=() ;
    eval {
        my $status = system(@alt_args) ;
        if ($status == -1) {
            die("Could not run Splus.exe to create files in .Prefs") ;
        } elsif ($status != 0) {
            die("Could not create files in .Prefs using Splus.exe") ;
        }
        open my $from_splus, "<", $splus_out or die "Cannot open Splus output file $splus_out" ;
        while (<$from_splus>) {
            chomp ;
            # print "from_splus: $_\n";
            push @from_splus, $_ ;
	}
    } ;
    unlink($splus_in) ;
    unlink($splus_out) ;
    return @from_splus ;
}

1;
