package Splus::packup;
use v5.8;
use strict ;

=head1 NAME

  Splus::packup - make tar.gz or zip file out of package directory

=head1 SYNOPSIS

  The following is the plan.  Only the ones before the bar are implemented.

  use Splus::packup;
  $pu = Splus::packup->new(qw(--directory pkg_dir
                               --parent parent_directory
                               --name name
           # none of the above args are needed if your cwd is pkg_dir
                               --verbose
                               --type source|binary
                               --compress-with zip|tar.gz
                               ))
  $pu->make() ; # pack things up
=cut

use Splus::Vars;
use Splus::Utils;
use Splus::SplusUtils;
use Splus::src2bin;
use Splus::Dcf;
use File::Basename ;
use File::Copy;
use File::Path; # for rmtree(), to remove directory recursively
use Cwd qw(getcwd abs_path) ;

my $debug = 0 ;

Splus::Vars::error("SHOME");
Splus::Vars::error("S_TMPDIR");
my $SHOME=${Splus::Vars::SHOME} ;
my $S_TMPDIR=${Splus::Vars:S_TMPDIR} ;
my $ostype = ${Splus::Vars::OSTYPE} ; # "windows" or "unix"

sub syntax {
    print "-h,--help             # print this help message\n";
    print "--parent parentDir    # parent directory of package source\n";
    print "--name packageName    # parentDir/packageName contains the package itself\n";
    print "--directory directory # parentDir/packageName as one argument (default=.)\n";
    print "If neither --parent nor --name given, then parent is dirname cwd, name is basename cwd\n";
    print "-v,--verbose          # give more detailed progress reports\n";
    print "--type binary|source  # include compiled/installed code or just source\n";
    print "--no-clean-after      # do not remove unneeded files and directories after packing up package\n";
    print "--compress-with zip|tar.gz # compression program to use\n";
    # print "--binary-tag          # for debugging, to override default S_PLATFORM\n";
    # help calls this with no arguments, problems invoke it with an error string.
    @_ && die join(", ", @_) ;
}

sub new {
    my $class = shift ;
    my $self = {} ;
    my @args = @_ ;
    $self->{verbose} = 0 ;
    $self->{clean_after} = 1 ;
    $self->{compress_with} = $ostype eq "windows" ? "zip" : "tar.gz" ;
    while (@args) {
        my $arg = shift @args ;
	# print "  arg=$arg\n";
        if ($arg =~ /^(-h|--help)$/) {
            syntax() ;
            exit() ;
        } elsif ($arg =~ /^(-p|--parent)$/) {
            $arg = shift @args or die "No directory name after --parent argument" ;
            $self->{parent_dir} = abs_path($arg) ;
        } elsif ($arg =~ /^(-n|--name)$/) {
            $arg = shift @args or syntax("No package name after --name argument") ;
            $self->{name} = $arg ;
        } elsif ($arg =~ /^(-d|--dir)$/) {
            $arg = shift @args or syntax("No directory name after --directory argument") ;
            $self->{dir} = abs_path($arg) ;
        } elsif ($arg =~ /^(-t|--type)$/) {
            ( $arg = shift @args and $arg =~ /^(source|binary)$/ ) or syntax("Need 'source' or 'binary' after --type argument") ;
            $self->{type} = $arg ;
        } elsif ($arg =~ /^--compress-with$/) {
            ( $arg = shift @args and $arg =~ /^(zip|tar.gz)$/ ) or syntax("Need 'zip' or 'tar.gz' after --compress-with argument") ;
            $self->{compress_with} = $arg ;
        } elsif ($arg =~ /^(-v|--verbose)$/) {
            $self->{verbose} = 1 ;
        } elsif ($arg =~ /^--no-clean-after$/) {
            $self->{clean_after} = 0 ;
        } elsif ($arg =~ /^--clean-after$/) {
            $self->{clean_after} = 1 ;
        } elsif ($arg =~ /^--binary-tag$/) {
            $arg = shift @args or syntax("No name after --binary_tag argument") ;
            $self->{binary_tag} = $arg ;
        } else {
            syntax("Argument $arg not recognized") ;
        }
    }
    # now take care of defaults
    if ($self->{name}) {
         $self->{parent_dir} or $self->{parent_dir} = abs_path(".") ;
         if ($self->{dir}) {
             syntax("Gave both --directory and --name, did you mean --parent and --name?");
         }
         $self->{dir} = abs_path("$self->{parent_dir}/$self->{name}") ;
    }
    $self->{dir} or $self->{dir} = abs_path(".") ;
    $self->{name} or $self->{name} = basename($self->{dir}) ;
    $self->{parent_dir} or $self->{parent_dir} = dirname($self->{dir}) ;
    # sanity checks
    -d $self->{dir} or die "$self->{dir} is not a directory" ;
    -e "$self->{dir}/DESCRIPTION" or warn "No DESCRIPTION file found in $self->{dir}" ;
    my $dcf = Splus::Dcf->new( "$self->{dir}/DESCRIPTION" );
    $self->{version} = $dcf->{Version} or $self->{version} = "0.0-0" ;
    if ($self->{type} eq "binary") {
        if (! $self->{binary_tag} ) {
            $self->{binary_tag} = Splus_getenv("S_PLATFORM", "") ;
        }
        if (! $self->{binary_tag} ) {
            $self->{binary_tag} = $ostype eq "windows" ? "WIN386" : "unknown-unix" ;
        }
        $self->{compressed_name} = dosify(abs_path(".") . "/$self->{name}_$self->{version}_S_$self->{binary_tag}.$self->{compress_with}") ;
    } else {
        $self->{compressed_name} = dosify(abs_path(".") . "/$self->{name}_$self->{version}.$self->{compress_with}") ;
    }
    if ($self->{verbose}) {
        print "name=$self->{name}\n";
        print "parent=$self->{parent_dir}\n";
        print "dir=$self->{dir}\n" ;
    }
    # bless things and return
    bless $self, $class ;
}

sub make
{
    my $self = shift ;
    # print "First copy relevant parts of $self->{dir} to a tmpdir\n";
    $self->{tmpdir} = Splus_tempfile("Spkg") unless $self->{tmpdir} ;
    eval { $self->copy_package() ; } ;
    if ($@) {
        $self->clean_after_if_desired() ;
        die "Problem in copy_package: $@" if $@ ;
    }
    if ($self->{type} eq "binary") {
        foreach my $subdir qw(inst man R src data java swingui) {
           my $d = $self->{tmpdir} . "/" . $self->{name} . "/" . $subdir ;
           if (-e $d) {
               File::Path::rmtree($d) ;
           }
        }
    }
    # Add Packaged: line to DESCRIPTION
    my $DESCRIPTION = $self->{tmpdir} . "/" . $self->{name} . "/DESCRIPTION" ;
    Splus::SplusUtils::add_packaged_stamp_to_description_file($DESCRIPTION) ;
    # print "Then zip it up\n";
    $self->compress() ;
    $self->clean_after_if_desired() ;
    $self->{compressed_name} ;
}

sub compress
{
    my $self = shift ;
    my $cwd = getcwd() ;
    my $cmd ;
    eval {
        chdir "$self->{tmpdir}" or die "Cannot chdir to $self->{tmpdir} ($!)" ;
        if ($self->{compress_with} eq "zip") {
            # zip updates zip file if it exists, but we want brand new one.
            unlink $self->{compressed_name} if -e $self->{compressed_name} ;
            $cmd = "zip -r $self->{compressed_name} $self->{name}" ;
        } elsif ($self->{compress_with} eq "tar.gz") {
            $cmd = "tar cf - $self->{name} | gzip > $self->{compressed_name}" ;
        } else {
            die "Unrecognized compress-with type, $self->{compress_with}";
        }
        # print "$cmd\n";
        my $status = system("$cmd");
        $status eq 0 or die "Operating system problem while compressing package" ;
    } ;
    die "Problem in making compressed package: $@" if $@ ;
    chdir $cwd or die "Cannot change directory back to $cwd" ;
    print "Compressed package is $self->{compressed_name}\n"
}

sub clean_after_if_desired
{
    my $self = shift ;
    if ($self->{tmpdir} && -d $self->{tmpdir}) {
        if ($self->{clean_after}) {
            print "Removing temporary directory $self->{tmpdir}\n" ;
            File::Path::rmtree($self->{tmpdir}) ;
        } else {
            warn "Not removing temporary directory $self->{tmpdir}" ;
        }
    }
}


sub copy_package
{
    my $self = shift ;
    -d $self->{tmpdir} or
        mkdir $self->{tmpdir} or
        die "Cannot make temporary directory $self->{tmpdir} ($!)" ;
    _copy_recursive($self->{dir}, $self->{tmpdir}) ;
}

# _copy_file will copy a binary file, but translate
# a non-binary file to have unix line endings.  Override
# by supplying non-null 3 argument, copy_as_binary.
sub _copy_file
{
    my $from = shift or die "Missing from argument to _copy_file" ;
    my $to = shift or die "Missing to argument to _copy_file" ;
    -e $from or die "Source file $from does not exist" ;
    -d $from and die "Source \"file\" $from is a directory" ;
    -r $from or die "Cannot read source file $from to copy" ;
    -e $to and die "Destination file $to already exists (will not overwrite)" ;
    my $copy_as_binary = -B $from ;
    print "_copy_file from $from to $to (copy_as_binary=$copy_as_binary)\n" if $debug ;
    if ( $copy_as_binary ) {
        # Binary files are just copied.
        File::Copy::copy("$from","$to") or die "Could not copy file $from to $to ($!)" ;
    } else {
        # Text files will get Unix line endings, \n, at end of all lines, including last.
        open my $handle_from, "< $from" or die "Cannot open file $from for reading ($!)" ;
        # $handle's file will be closed when $handle goes out of scope
        open my $handle_to, "> $to" or die "Cannot create file $to ($!)" ;
        binmode $handle_to ; # so perl on Windows doesn't add \r before \n
        while (<$handle_from>) {
            chomp ;
            s/\r$// ;
            print $handle_to "$_\n" ;
        }
    }
}

sub _copy_recursive
{
    my $from = shift or die "Missing from argument in call to _copy_recursive" ;
    my $to = shift or die "Missing to argument in call to _copy_recursive" ;
    my $copy_as_binary = shift ; # ok for this arg to be missing, will be undef
    # Expect from and to both be directories
    # to must exist and will be parent directory of new copy.
    -d $from or die "from argument to _copy_recursive, \"$from\", is not a directory";
    -d $to or die "to argument to _copy_recursive, \"$to\", is not a directory";
    print "_copy_recursive from $from to $to\n" if $debug ;
    $to = "$to/" . basename("$from") ;
    mkdir $to or die "Cannot make directory $to ($!)" ;
    # print "  made directory $to\n";
    opendir my $dir_handle, "$from" or die "Cannot read from directory $from ($!)" ;
    foreach my $f (readdir $dir_handle) {
        next if ($f eq "." || $f eq "..") ;
        $f = "$from/$f" ;
        # print "  copying $f\n" if $debug ;
        if (-l $f) { warn "Will not copy symblic link $f" ; }
        elsif (-f $f) { _copy_file($f, "$to/" . basename("$f"), $copy_as_binary) ; }
        elsif (-d $f) { _copy_recursive($f, $to, $copy_as_binary) ; }
        else { warn "$f is neither a file nor a directory, will not try to copy"; }
    }
    closedir $dir_handle ;
}

1;
