/*
 imkmeans.c implementation of interfaces for IMinerCoxRegressionModel objects.
*/

#include <math.h>

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <memory.h>

#include "IMCoxRegression.h"


/* alloc memory for type IMinerCoxRegressionModel. */
long IMCSRC_STDCALL IMinerCoxRegressionModel_create(
 IMinerObject* pModel,		    /* out: data object */
 IMinerObject* md,			    /* in:  meta data object */
 long nInputs,                  /* in:  number of inputs */
 long nBaselines,				/* in:  number of baselines */
 const double* pdCoefficients,  /* in:  model coefficients of length nInputs */
 const double* pdCenters,       /* in:  model centers of length nInputs */
 const double* pdSurvival,      /* in:  survival matrix of length (5 * 62) */
 const double* pdCholeski        /* in:  choleski matrix of length (nInputs * nInputs) */
)
{ 
	char *ppszMemberNames[] = {"metaData", "coefficients", "centers", "survival", "choleski"};
	long nStatus=IMINER_SUCCESS;

	if(pModel == NULL || nInputs < 1L)
		return IMINER_BAD_INPUT_ARGS;

	/* a CoxRegressionModel always have IMINER_COX_REGR_HESSIAN_NUM+1 elements as list */
	nStatus = IMinerList_create(pModel, IMINER_COX_REGR_HESSIAN_NUM+1, IMINER_MODE_MODEL); 
	if(nStatus != IMINER_SUCCESS)
		return nStatus;

	/* member: meta data */
	nStatus = IMinerList_clone(IMINER_COX_REGR_META_DATA(pModel), md);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;

	/* member: coefficients */
	nStatus = IMinerVector_create(IMINER_COX_REGR_COEFFICIENTS(pModel), nInputs, IMINER_MODE_DOUBLE, pdCoefficients);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;

	/* member: centers */
	nStatus = IMinerVector_create(IMINER_COX_REGR_CENTERS(pModel), nInputs, IMINER_MODE_DOUBLE, pdCenters);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;

	/* member: survival matrix */
	nStatus = IMinerDoubleMatrix_create(IMINER_COX_REGR_SURVIVAL(pModel), 5, nBaselines, pdSurvival, NULL, NULL);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;

	/* member: choleski matrix */
	nStatus = IMinerDoubleMatrix_create(IMINER_COX_REGR_HESSIAN(pModel), nInputs, nInputs, pdCholeski, NULL, NULL);
	if(nStatus != IMINER_SUCCESS)
		return nStatus;

	/*member 4 (special) : member names */
	nStatus = IMinerList_setNamesFromStrings(pModel, (const char**)ppszMemberNames);

	/* set the class ID */
	pModel->m_nMode = IMINER_MODE_MODEL;
	return nStatus;
}

long IMCSRC_STDCALL IMinerCoxRegressionModel_isValid(const IMinerObject* pModel)
{
	long nStatus=IMINER_SUCCESS;
	IMinerObject* pElement;

	if(pModel==NULL  || !IMinerList_isValid(pModel))
		return 0L;

	if (!IMinerList_isValid(IMINER_COX_REGR_META_DATA(pModel))) return 0L;

	pElement = IMINER_COX_REGR_COEFFICIENTS(pModel);
	if(!IMinerVector_isValid(pElement) )
		return 0L;

	pElement = IMINER_COX_REGR_CENTERS(pModel);
	if(!IMinerVector_isValid(pElement) )
		return 0L;
	
	pElement = IMINER_COX_REGR_SURVIVAL(pModel);
	if(!IMinerDoubleMatrix_isValid(pElement) )
		return 0L;
	
	pElement = IMINER_COX_REGR_HESSIAN(pModel);
	if(!IMinerDoubleMatrix_isValid(pElement) )
		return 0L;

	/* special member : member names mut also be valid and has the same length as this object*/
	pElement = IMINER_LIST_NAMES_PTR(pModel);
	if(!IMinerVector_isString(pElement) && pElement->m_nLen == pModel->m_nLen)
		return 0L;

	return 1L;
}

/* free memory */
long IMCSRC_STDCALL IMinerCoxRegressionModel_destroy(IMinerObject* pModel )
{
	return IMinerObject_destroy(pModel);
}


/* write to stdout */
long IMCSRC_STDCALL IMinerCoxRegressionModel_print(const IMinerObject* pModel)
{
	long nStatus;

	/* printf("Begin IMinerCoxRegressionModel ...\n"); */
	if(!IMinerCoxRegressionModel_isValid(pModel))
	{
		IMiner_error("%s(%d) : ", __FILE__, __LINE__);
		IMiner_error("Invalid pModel\n");
		return IMINER_BAD_INPUT_ARGS;
	}

	printf("Meta Data:\n");
	if (IMinerMetaData_print(IMINER_COX_REGR_META_DATA(pModel))!=IMINER_SUCCESS) {
		return IMINER_BAD_INPUT_ARGS;
	}

	printf("Coefficients:\n");
	nStatus = IMinerVector_print(IMINER_COX_REGR_COEFFICIENTS(pModel));
	if(nStatus != IMINER_SUCCESS)
		return IMINER_BAD_INPUT_ARGS;

	printf("Centers:\n");
	nStatus = IMinerVector_print(IMINER_COX_REGR_CENTERS(pModel));
	if(nStatus != IMINER_SUCCESS)
		return IMINER_BAD_INPUT_ARGS;

	printf("Survival:\n");
	nStatus = IMinerDoubleMatrix_print(IMINER_COX_REGR_SURVIVAL(pModel));
	if(nStatus != IMINER_SUCCESS)
		return IMINER_BAD_INPUT_ARGS;

	printf("Choleski:\n");
	nStatus = IMinerDoubleMatrix_print(IMINER_COX_REGR_HESSIAN(pModel));
	if(nStatus != IMINER_SUCCESS)
		return IMINER_BAD_INPUT_ARGS;

	return IMINER_SUCCESS;
}

/* predict cluster memberships */
long IMCSRC_STDCALL IMinerCoxRegressionModel_predict(
 IMinerObject* pOutput,      /* out: output rectangular data */
 const IMinerObject* input,  /* in: input rectangular data */
 IMinerObject* pDescr,		 /* in: input description (if NULL, description will be 
                                    created from input data) */
 const IMinerObject* pModel  /* in: the model */
)
{
	IMinerObject pInput, *eCol, *seCol, *pdvCoeffs, *pdvCenters, *pdmSurv, *pdmCholeski, A, RX;
	long nStatus=IMINER_SUCCESS, nInputColumns, nInputRows, nOutputColumns, nMode;
	long i,j,row, nInputs, nOutputs, *pnColumnsModes;
	double se ,e, dTemp;

	char *risk_name="PREDICT.risk", *risk_se_name="PREDICT.risk.se", **columnNames;

	if(pOutput==NULL || !IMinerDataSet_isValid(input) || !IMinerCoxRegressionModel_isValid(pModel))	
		return IMINER_BAD_INPUT_ARGS;

	/* convert input if needed*/
	nStatus = IMinerMetaData_InputConvert(&pInput, IMINER_COX_REGR_META_DATA(pModel), input, pDescr);
	if(nStatus != IMINER_SUCCESS) return IMINER_FAIL;

	pdvCoeffs			= IMINER_COX_REGR_COEFFICIENTS(pModel);
	pdvCenters			= IMINER_COX_REGR_CENTERS(pModel);
	pdmSurv				= IMINER_COX_REGR_SURVIVAL(pModel);
	pdmCholeski			= IMINER_COX_REGR_HESSIAN(pModel);

	nInputColumns		= IMINER_DATASET_NCOLUMNS(&pInput);
	nInputRows			= IMINER_DATASET_NROWS(&pInput);
	nInputs				= IMINER_OBJECT_LENGTH(pdvCoeffs);
	nOutputs			= 2;


	/* create a matrix double to store the input as double */
	nStatus = IMinerDoubleMatrix_create(&A, nInputRows, nInputColumns, NULL, NULL, NULL);
	if(nStatus != IMINER_SUCCESS) return IMINER_FAIL;
	nStatus = IMinerDoubleMatrix_create(&RX, nInputRows, nInputColumns, NULL, NULL, NULL);
	if(nStatus != IMINER_SUCCESS) return IMINER_FAIL;

	/* create the output object of the same size as input + one column per component */
	nOutputColumns = /*nInputColumns + */nOutputs;
	pnColumnsModes = (long*)malloc(nOutputColumns*sizeof(long));
	columnNames = (char**)malloc(nOutputs*sizeof(char*));
	columnNames[0] = risk_name;
	columnNames[1] = risk_se_name;
	for(i=0L; i<nOutputColumns; ++i) pnColumnsModes[i] = IMINER_MODE_DOUBLE;
	nStatus = IMinerDataSet_create(pOutput, nInputRows, nOutputColumns, pnColumnsModes);
	if(nStatus != IMINER_SUCCESS) return nStatus;
	nStatus = IMinerDataSet_setColumnNamesFromStrings(pOutput, (const char**)columnNames);
	if(nStatus != IMINER_SUCCESS) return nStatus;
	free(pnColumnsModes);
	free(columnNames);

	/* Set the input data to the matrix
	TODO: apply coding factor expansion of categorical data */
	for(i=0L; i<nInputColumns; ++i) {
		nStatus = IMinerDataSet_getColumnMode(&nMode, &pInput, i);
		if(nStatus != IMINER_SUCCESS) return IMINER_BAD_INPUT_ARGS;
		/*if(nMode != IMINER_MODE_DOUBLE) return IMINER_BAD_INPUT_ARGS;*/
	}

	for(row=0L; row<nInputRows; ++row) {
		dTemp = 0.0;
		for(i=0L; i<nInputColumns; ++i) {
			double val = IMinerDataSet_getNonStringValue(&pInput, i, row);
			val -= IMINER_DOUBLE_VALUE(pdvCenters, i);
			IMINER_Aij(&A, row, i) = val;
			
			dTemp += val * IMINER_DOUBLE_VALUE(pdvCoeffs, i);
		}

		IMINER_DOUBLE_VALUE(IMINER_DATASET_VECTOR_PTR(pOutput, 0), row) = dTemp;
	}

	for (row=0L; row<nInputRows; row++) {
		for (i=0; i<nInputs; i++) {
			dTemp = 0.0;
			for (j=0; j<=i; j++) {
				double coef = IMINER_Aij(pdmCholeski, i, j);
				if (j<i) {
					double val = IMINER_Aij(&RX, row, j);
					dTemp += val*coef;
				} else {
					double val = IMINER_Aij(&A, row, j);
					dTemp = (val-dTemp)/coef;
				}
			}
			IMINER_Aij(&RX, row, i) = dTemp;
		}
	}

	/* copy content from the input object */
	for(j=nOutputs,i=0; j<nOutputColumns; ++j, ++i)
	{
		nStatus = IMinerDataSet_getColumnMode(&nMode, &pInput, i);
		if(nStatus != IMINER_SUCCESS) return IMINER_BAD_INPUT_ARGS;
		/* copy content of pInput into jth column of pOutput */
		nStatus = IMinerDataSet_setColumnAt(pOutput, j, IMINER_DATASET_COLUMN_PTR(&pInput, i)); 
		if(nStatus != IMINER_SUCCESS) return nStatus;
	}
		
	/* for each row (object), find the component score */
	eCol = IMINER_DATASET_VECTOR_PTR(pOutput, 0);
	seCol = IMINER_DATASET_VECTOR_PTR(pOutput, 1);
	for (row=0L; row<nInputRows; row++) {
		e = exp(IMINER_DOUBLE_VALUE(eCol, row));
		for (i=0L, se=0.0; i<nInputs; i++) {
			double se_i = IMINER_Aij(&RX, row, i);
			se += (se_i*se_i);
		}

		IMINER_DOUBLE_VALUE(eCol, row) = e;
		IMINER_DOUBLE_VALUE(seCol, row) = sqrt(e*se);
	}

	IMinerObject_destroy(&pInput);
	IMinerObject_destroy(&A);
	IMinerObject_destroy(&RX);

	return IMINER_SUCCESS;
}
