<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform">
	<!-- XSL style sheet for creating PMML from IMML -->
	<xsl:output method="xml" standalone="yes" indent="yes"/>
	<xsl:template match="IMML">
		<!-- Specify the DOCTYPE.  Use text to create DOCTYPE element rather than xsl:output to support internal element definitions. -->
		<xsl:choose>
			<!-- Use a simple DOCTYPE specification if only using standard PMML elements -->
			<xsl:when test="count(./TreeList) = 0 and count(./PrincipalComponentsModel)=0 and count(./SurvivalModel)=0 and count(./NeuralNetwork)=0 and count(./ClusteringModel) = 0 and count(./GeneralRegressionModel)=0">
				<xsl:text disable-output-escaping="yes">&lt;!DOCTYPE PMML PUBLIC &quot;PMML 2.0&quot; &quot;http://www.dmg.org/v2-0/pmml_v2_0.dtd&quot;&gt; 
</xsl:text>
			</xsl:when>
			<xsl:otherwise>
				<!-- Define elements extending PMML to support models not defined in PMML -->
				<!-- Open the DOCTYPE element -->
				<xsl:text disable-output-escaping="yes">&lt;!DOCTYPE PMML PUBLIC &quot;PMML 2.0&quot; &quot;http://www.dmg.org/v2-0/pmml_v2_0.dtd&quot; 
[
		</xsl:text>
				<xsl:choose>
					<xsl:when test="count(./GeneralRegressionModel)>0">
						<xsl:text disable-output-escaping="yes">	
	&lt;!-- Storing attribute values not in standard PMML --&gt;
	&lt;!ELEMENT X-IMML-XTProps (X-IMML-Property*)&gt;
	&lt;!ELEMENT X-IMML-Property (X-IMML-Property* | X-IMML-Complex)&gt;
	&lt;!ATTLIST X-IMML-Property
		name CDATA #REQUIRED
		value CDATA #IMPLIED
		complex (true | false) "false"
	&gt;
	&lt;!ELEMENT X-IMML-Complex (#PCDATA)&gt;
	&lt;!ELEMENT X-IMML-DevMatrix (X-IMML-DevCell+)&gt;
	&lt;!ATTLIST X-IMML-DevMatrix
		count CDATA #REQUIRED
		targetCategory CDATA #REQUIRED
		accuracy CDATA #IMPLIED
		R2 CDATA #IMPLIED
	&gt;
	&lt;!ELEMENT X-IMML-DevCell EMPTY&gt;
	&lt;!ATTLIST X-IMML-DevCell
		source CDATA #REQUIRED
		df CDATA #IMPLIED
		dev CDATA #REQUIRED
		scaledDev CDATA #IMPLIED
		stat CDATA #IMPLIED
		Pr CDATA #IMPLIED
	&gt;
	&lt;!ELEMENT X-IMML-GeneralRegressionModel-ParamMatrixExtraAttributes (X-IMML-XTProps)&gt;
	&lt;!ELEMENT X-IMML-GeneralRegressionModel-CoefCorr (X-IMML-GeneralRegressionModel-CorrCell+)&gt;
	&lt;!ATTLIST X-IMML-GeneralRegressionModel-CoefCorr
		count CDATA #REQUIRED
		threshold CDATA "0.5"
	&gt;
	&lt;!ELEMENT X-IMML-GeneralRegressionModel-CorrCell EMPTY&gt;
	&lt;!ATTLIST X-IMML-GeneralRegressionModel-CorrCell
		coef CDATA #REQUIRED
		value CDATA #REQUIRED
	&gt;
	&lt;!ELEMENT X-IMML-GeneralRegressionModel-Importance (X-IMML-GeneralRegressionModel-Effect+)&gt;
	&lt;!ATTLIST X-IMML-GeneralRegressionModel-Importance
		count CDATA #REQUIRED
		targetCategory CDATA #REQUIRED
	&gt;
	&lt;!ELEMENT X-IMML-GeneralRegressionModel-Effect EMPTY&gt;
	&lt;!ATTLIST X-IMML-GeneralRegressionModel-Effect
		name CDATA #REQUIRED
		value CDATA #REQUIRED
		df CDATA #IMPLIED
		stat CDATA #IMPLIED
		Pr CDATA #IMPLIED
	&gt;
	&lt;!ELEMENT X-IMML-GeneralRegressionModel-RMatrix (X-IMML-GeneralRegressionModel-Row+)&gt;
	&lt;!ATTLIST X-IMML-GeneralRegressionModel-RMatrix
		nCols  CDATA #REQUIRED
		nRows  CDATA #REQUIRED
	&gt;
	&lt;!ELEMENT X-IMML-GeneralRegressionModel-QTYMatrix (X-IMML-GeneralRegressionModel-Row+)&gt;
	&lt;!ATTLIST X-IMML-GeneralRegressionModel-QTYMatrix
		nCols  CDATA #REQUIRED
		nRows  CDATA #REQUIRED
	&gt;
	&lt;!ELEMENT X-IMML-GeneralRegressionModel-Row (X-IMML-GeneralRegressionModel-Col+)&gt;
	&lt;!ATTLIST X-IMML-GeneralRegressionModel-Row
		num CDATA #REQUIRED
	&gt;
	&lt;!ELEMENT X-IMML-GeneralRegressionModel-Col EMPTY&gt;
	&lt;!ATTLIST X-IMML-GeneralRegressionModel-Col
		num CDATA #REQUIRED
		val CDATA #REQUIRED
	&gt;
	&lt;!ELEMENT X-IMML-GeneralRegressionModel-ModelExtraInfo (
							X-IMML-GeneralRegressionModel-Family?,
							X-IMML-GeneralRegressionModel-Link?,
							X-IMML-GeneralRegressionModel-Mean?,
							X-IMML-GeneralRegressionModel-Var?,
							X-IMML-GeneralRegressionModel-Iterations?)&gt;
	&lt;!ELEMENT X-IMML-GeneralRegressionModel-Family EMPTY&gt;
	&lt;!ATTLIST X-IMML-GeneralRegressionModel-Family
		val  CDATA #REQUIRED
	&gt;
	&lt;!ELEMENT X-IMML-GeneralRegressionModel-Link EMPTY&gt;
	&lt;!ATTLIST X-IMML-GeneralRegressionModel-Link
		val  CDATA #REQUIRED
	&gt;
	&lt;!ELEMENT X-IMML-GeneralRegressionModel-Mean EMPTY&gt;
	&lt;!ATTLIST X-IMML-GeneralRegressionModel-Mean
		val  CDATA #REQUIRED
	&gt;
	&lt;!ELEMENT X-IMML-GeneralRegressionModel-Var EMPTY&gt;
	&lt;!ATTLIST X-IMML-GeneralRegressionModel-Var
		val  CDATA #REQUIRED
	&gt;
	&lt;!ELEMENT X-IMML-GeneralRegressionModel-Iterations EMPTY&gt;
	&lt;!ATTLIST X-IMML-GeneralRegressionModel-Iterations
		val  CDATA #REQUIRED
	&gt;
						</xsl:text>
					</xsl:when>
					<xsl:when test="count(./TreeList)>0">
						<!-- Ensemble trees -->
						<xsl:text disable-output-escaping="yes">	
	&lt;!-- Ensemble tree model --&gt;
	&lt;!ELEMENT X-IMML-TreeModel-TreeList (TreeModel+)&gt;
	&lt;!-- Storing attribute values not in standard PMML --&gt;
	&lt;!ELEMENT X-IMML-XTProps (X-IMML-Property*)&gt;
	&lt;!ELEMENT X-IMML-Property (X-IMML-Property* | X-IMML-Complex)&gt;
	&lt;!ATTLIST X-IMML-Property
		name CDATA #REQUIRED
		value CDATA #IMPLIED
		complex (true | false) "false"
	&gt;
	&lt;!ELEMENT X-IMML-Complex (#PCDATA)&gt;
						</xsl:text>
					</xsl:when>
					<xsl:when test="count(./ClusteringModel)>0">
						<xsl:text disable-output-escaping="yes">	
	&lt;!-- Storing attribute values not in standard PMML --&gt;
	&lt;!ELEMENT X-IMML-XTProps (X-IMML-Property*)&gt;
	&lt;!ELEMENT X-IMML-Property (X-IMML-Property* | X-IMML-Complex)&gt;
	&lt;!ATTLIST X-IMML-Property
		name CDATA #REQUIRED
		value CDATA #IMPLIED
		complex (true | false) "false"
	&gt;
	&lt;!ELEMENT X-IMML-Complex (#PCDATA)&gt;
				</xsl:text>
					</xsl:when>
					<xsl:when test="count(./PrincipalComponentsModel)>0 ">
						<!-- Principal components analysis -->
						<xsl:text disable-output-escaping="yes">	
	&lt;!-- Principal components --&gt;
	&lt;!ELEMENT X-IMML-PrincipalComponentsModel (
		MiningSchema, 
		X-IMML-PrincipalComponentsModel-Parameters, 
		X-IMML-PrincipalComponentsModel-TermIndicies, 
		X-IMML-PrincipalComponentsModel-PrinComp)&gt;
	&lt;!ELEMENT X-IMML-PrincipalComponentsModel-Parameters (
		X-IMML-PrincipalComponentsModel-Param+)&gt;
	&lt;!ATTLIST X-IMML-PrincipalComponentsModel-Parameters
		size CDATA #REQUIRED
	&gt;
	&lt;!ELEMENT X-IMML-PrincipalComponentsModel-Param (#PCDATA)&gt;
	&lt;!ATTLIST X-IMML-PrincipalComponentsModel-Param
		variable CDATA #REQUIRED
		center CDATA #REQUIRED
		scale CDATA #REQUIRED
	&gt;
	&lt;!ELEMENT X-IMML-PrincipalComponentsModel-TermIndicies (
		X-IMML-PrincipalComponentsModel-Index+)&gt;
	&lt;!ATTLIST X-IMML-PrincipalComponentsModel-TermIndicies
		size CDATA #REQUIRED
	&gt;
	&lt;!ELEMENT X-IMML-PrincipalComponentsModel-Index (#PCDATA)&gt;
	&lt;!ATTLIST X-IMML-PrincipalComponentsModel-Index
		term CDATA #REQUIRED
		value CDATA #REQUIRED
	&gt;
	&lt;!ELEMENT X-IMML-PrincipalComponentsModel-PrinComp (
		X-IMML-PrincipalComponentsModel-Component+)&gt;
	&lt;!ATTLIST X-IMML-PrincipalComponentsModel-PrinComp
		size CDATA #REQUIRED
		totalVar CDATA #REQUIRED
	&gt;
	&lt;!ELEMENT X-IMML-PrincipalComponentsModel-Component (
		X-IMML-PrincipalComponentsModel-Loading+)&gt;
	&lt;!ATTLIST X-IMML-PrincipalComponentsModel-Component
		name CDATA #REQUIRED
		size CDATA #REQUIRED
		variance CDATA #REQUIRED
		cumPercent CDATA #REQUIRED
	&gt;
	&lt;!ELEMENT X-IMML-PrincipalComponentsModel-Loading (#PCDATA)&gt;
	&lt;!ATTLIST X-IMML-PrincipalComponentsModel-Loading
		variable CDATA #REQUIRED
		value CDATA #REQUIRED
	&gt;
				</xsl:text>
					</xsl:when>
					<xsl:when test="count(./SurvivalModel)>0 ">
						<!-- Survival analysis -->
						<xsl:text disable-output-escaping="yes">			
	&lt;!-- Survival model --&gt;
	&lt;!ELEMENT X-IMML-SurvivalModel (
		CovariateList?, FactorList?, 
		X-IMML-SurvivalModel-SurvResponse, 
		X-IMML-SurvivalModel-LinearModel, 
		X-IMML-SurvivalModel-Coefficients, 
		X-IMML-SurvivalModel-Hessian, 
		X-IMML-SurvivalModel-SurvModelStats?, 
		X-IMML-SurvivalModel-BaselineSurvival?)&gt;
	&lt;!ATTLIST X-IMML-SurvivalModel
		type 	(CoxRegression) 	#REQUIRED
	&gt;
	&lt;!ELEMENT X-IMML-SurvivalModel-SurvResponse (X-IMML-SurvivalModel-EventCodes)&gt;
	&lt;!ATTLIST X-IMML-SurvivalModel-SurvResponse
		event 	CDATA	#REQUIRED
		stop 	CDATA 	#REQUIRED
		start 	CDATA 	#IMPLIED
	&gt;
	&lt;!ELEMENT X-IMML-SurvivalModel-EventCodes (#PCDATA)&gt;
	&lt;!ATTLIST X-IMML-SurvivalModel-EventCodes
		failure 		CDATA 	#REQUIRED
		rightCensor 	CDATA 	#REQUIRED
	&gt;
	&lt;!ELEMENT X-IMML-SurvivalModel-LinearModel (
		X-IMML-SurvivalModel-Term*, 
		X-IMML-SurvivalModel-Intercept?, 
		X-IMML-SurvivalModel-Weights?, 
		X-IMML-SurvivalModel-Strata?)&gt;
	&lt;!ATTLIST X-IMML-SurvivalModel-LinearModel
		nterms	CDATA 	#REQUIRED
	&gt;
	&lt;!ELEMENT X-IMML-SurvivalModel-Term (
		X-IMML-SurvivalModel-TermVar+)&gt;
	&lt;!ATTLIST X-IMML-SurvivalModel-Term
		name 	CDATA 	#REQUIRED
	&gt;
	&lt;!ELEMENT X-IMML-SurvivalModel-TermVar (#PCDATA)&gt;
	&lt;!ATTLIST X-IMML-SurvivalModel-TermVar
		value 	CDATA 	#REQUIRED
	&gt;
	&lt;!ELEMENT X-IMML-SurvivalModel-Intercept (#PCDATA)&gt;
	&lt;!ATTLIST X-IMML-SurvivalModel-Intercept
		value 	(0 | 1) 	#REQUIRED
	&gt;
	&lt;!ELEMENT X-IMML-SurvivalModel-Weights (#PCDATA)&gt;
	&lt;!ATTLIST X-IMML-SurvivalModel-Weights
		value 	CDATA	#REQUIRED
	&gt;
	&lt;!ELEMENT X-IMML-SurvivalModel-Strata (#PCDATA)&gt;
	&lt;!ATTLIST X-IMML-SurvivalModel-Strata
		value 	CDATA 	#IMPLIED
		values	CDATA #IMPLIED
	&gt;
	&lt;!ELEMENT X-IMML-SurvivalModel-Coefficients (
		X-IMML-SurvivalModel-Coef+)&gt;
	&lt;!ATTLIST X-IMML-SurvivalModel-Coefficients
		count 	CDATA 	#REQUIRED
	&gt;
	&lt;!ELEMENT X-IMML-SurvivalModel-Coef (#PCDATA)&gt;
	&lt;!ATTLIST X-IMML-SurvivalModel-Coef
		name 		CDATA 		#REQUIRED
		value 		CDATA		#REQUIRED
		expCoef	CDATA	 	#IMPLIED
		stdErr 		CDATA 	#IMPLIED
		z 			CDATA 	#IMPLIED
		Pr 			CDATA 	#IMPLIED
		center 		CDATA 	#REQUIRED
	&gt;
	&lt;!ELEMENT X-IMML-SurvivalModel-SurvModelStats (
		X-IMML-SurvivalModel-LikelihoodRatio?)&gt;
	&lt;!ATTLIST X-IMML-SurvivalModel-SurvModelStats
		nobs CDATA #REQUIRED
		missing CDATA #REQUIRED
		logLike CDATA #REQUIRED
		nullLike CDATA #REQUIRED
	&gt;
	&lt;!ELEMENT X-IMML-SurvivalModel-LikelihoodRatio (#PCDATA)&gt;
	&lt;!ATTLIST X-IMML-SurvivalModel-LikelihoodRatio
		value 	CDATA 	#REQUIRED
		df 		CDATA 	#REQUIRED
		Pr 		CDATA 	#REQUIRED
		N 		CDATA	 #REQUIRED
	&gt;
	&lt;!ELEMENT X-IMML-SurvivalModel-Hessian (#PCDATA)&gt;
	&lt;!ATTLIST X-IMML-SurvivalModel-Hessian
		dimension CDATA #REQUIRED
		storage (upperSymmetric | lowerSymmetric) #REQUIRED
		major (column | row) #REQUIRED
		delimiter CDATA #REQUIRED
		values CDATA #REQUIRED
	&gt;
	&lt;!ELEMENT X-IMML-SurvivalModel-BaselineSurvival (
		X-IMML-SurvivalModel-Time, 
		X-IMML-SurvivalModel-Failures, 
		X-IMML-SurvivalModel-RiskSet, 
		X-IMML-SurvivalModel-Hazard, 
		X-IMML-SurvivalModel-VarHazard,
		X-IMML-SurvivalModel-Strata?)&gt;
	&lt;!ATTLIST X-IMML-SurvivalModel-BaselineSurvival
		count CDATA #REQUIRED
		delimiter CDATA #REQUIRED
	&gt;
	&lt;!ELEMENT X-IMML-SurvivalModel-Time (#PCDATA)&gt;
	&lt;!ATTLIST X-IMML-SurvivalModel-Time
		values CDATA #REQUIRED
	&gt;
	&lt;!ELEMENT X-IMML-SurvivalModel-Failures (#PCDATA)&gt;
	&lt;!ATTLIST X-IMML-SurvivalModel-Failures
		values CDATA #REQUIRED
	&gt;
	&lt;!ELEMENT X-IMML-SurvivalModel-RiskSet (#PCDATA)&gt;
	&lt;!ATTLIST X-IMML-SurvivalModel-RiskSet
		values CDATA #REQUIRED
	&gt;
	&lt;!ELEMENT X-IMML-SurvivalModel-Hazard (#PCDATA)&gt;
	&lt;!ATTLIST X-IMML-SurvivalModel-Hazard
		values CDATA #REQUIRED
	&gt;
	&lt;!ELEMENT X-IMML-SurvivalModel-VarHazard (#PCDATA)&gt;
	&lt;!ATTLIST X-IMML-SurvivalModel-VarHazard
		values CDATA #REQUIRED
	&gt;
						</xsl:text>
					</xsl:when>
					<xsl:when test="count(./NeuralNetwork)>0">
						<!-- Neural Network -->
						<xsl:text disable-output-escaping="yes">	
	&lt;!ELEMENT X-IMML-NeuralNetwork-NeuralParameters EMPTY&gt;
	&lt;!ATTLIST X-IMML-NeuralNetwork-NeuralParameters
		method (rprop | quick_prop | delta_bar_delta | online | conjugate_gradient) #REQUIRED
		etaPlus CDATA #IMPLIED
		etaMinus CDATA #IMPLIED
		maxDelta CDATA #IMPLIED
		minDelta CDATA #IMPLIED
		mu CDATA #IMPLIED
		increment CDATA #IMPLIED
		decrement CDATA #IMPLIED
		theta CDATA #IMPLIED
		seed CDATA #IMPLIED
		score CDATA #IMPLIED
		nobs CDATA #IMPLIED
		maxAbsWeight CDATA #IMPLIED
		inputJitter CDATA #IMPLIED
		weightDecay CDATA #IMPLIED
	&gt;
	&lt;!ELEMENT X-IMML-DevMatrix (X-IMML-DevCell+)&gt;
	&lt;!ATTLIST X-IMML-DevMatrix
		count CDATA #REQUIRED
		targetCategory CDATA #REQUIRED
		accuracy CDATA #IMPLIED
		R2 CDATA #IMPLIED
	&gt;
	&lt;!ELEMENT X-IMML-DevCell EMPTY&gt;
	&lt;!ATTLIST X-IMML-DevCell
		source CDATA #REQUIRED
		df CDATA #IMPLIED
		dev CDATA #REQUIRED
		scaledDev CDATA #IMPLIED
		stat CDATA #IMPLIED
		Pr CDATA #IMPLIED
	&gt;
	&lt;!ELEMENT X-IMML-NeuralNetwork-ErrorWeights (X-IMML-XTProps)&gt;
	&lt;!ELEMENT X-IMML-XTProps (X-IMML-Property*)&gt;
	&lt;!ELEMENT X-IMML-Property (X-IMML-Property* | X-IMML-Complex)&gt;
	&lt;!ATTLIST X-IMML-Property
		name CDATA #REQUIRED
		value CDATA #IMPLIED
		complex (true | false) "false"
	&gt;
	&lt;!ELEMENT X-IMML-Complex (#PCDATA)&gt;
							</xsl:text>					
					</xsl:when>
				</xsl:choose>
				<!-- Close the DOCTYPE tag -->
				<xsl:text disable-output-escaping="yes">
] &gt;
</xsl:text>
			</xsl:otherwise>
		</xsl:choose>
		<xsl:element name="PMML">
			<xsl:attribute name="version"><xsl:text>2.0</xsl:text></xsl:attribute>
			<!-- Header information -->
			<xsl:apply-templates select="Header"/>
			<!-- Description of the data -->
			<xsl:apply-templates select="XTMetaData"/>
			<!-- Models -->
			<xsl:apply-templates select="TreeList"/>
			<xsl:apply-templates select="TreeModel"/>
			<xsl:apply-templates select="GeneralRegressionModel"/>
			<xsl:apply-templates select="NeuralNetwork"/>
			<xsl:apply-templates select="ClusteringModel"/>
			<xsl:apply-templates select="NaiveBayesModel"/>
			<xsl:apply-templates select="SurvivalModel"/>
			<xsl:apply-templates select="PrincipalComponentsModel"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Header">
		<!-- Get info from the IMML Header element for Application and Timestamp -->
		<xsl:element name="Header">
			<xsl:attribute name="copyright"><xsl:value-of select="@copyright"/></xsl:attribute>
			<xsl:attribute name="description"><xsl:value-of select="@name"/></xsl:attribute>
			<xsl:apply-templates select="Application"/>
			<xsl:apply-templates select="Timestamp"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Application">
		<xsl:element name="Application">
			<xsl:attribute name="name"><xsl:value-of select="@name"/></xsl:attribute>
			<xsl:attribute name="version"><xsl:value-of select="@version"/></xsl:attribute>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Timestamp">
		<xsl:element name="Timestamp">
			<xsl:value-of select="normalize-space(./text())"/>
		</xsl:element>
	</xsl:template>
	<!-- DataDictionary -->
	<xsl:template match="XTMetaData">
		<xsl:apply-templates select="ColumnDefinitions"/>
	</xsl:template>
	<xsl:template match="ColumnDefinitions">
		<xsl:element name="DataDictionary">
			<xsl:attribute name="numberOfFields"><xsl:value-of select="@fieldCount"/></xsl:attribute>
			<xsl:apply-templates select="ColumnInfo"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="ColumnInfo">
		<xsl:element name="DataField">
			<xsl:attribute name="name"><xsl:value-of select="@name"/></xsl:attribute>
			<xsl:attribute name="optype"><xsl:value-of select="@type"/></xsl:attribute>
			<xsl:apply-templates select="Level"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Level">
		<xsl:element name="Value">
			<xsl:attribute name="value"><xsl:value-of select="@value"/></xsl:attribute>
		</xsl:element>
	</xsl:template>
	<!-- MiningSchema -->
	<!-- TODO: Need something on the mapping of categoricals to multiple fields -->
	<xsl:template match="MiningSchema">
		<xsl:element name="MiningSchema">
			<xsl:apply-templates select="MiningField"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="MiningField">
		<xsl:element name="MiningField">
			<xsl:copy-of select="@name"/>
			<xsl:copy-of select="@usageType"/>
		</xsl:element>
	</xsl:template>
	<!-- TreeModel -->
	<!-- Create an X-IMML-TreeList if multiple trees, or a PMML compliant TreeModel if a single tree. -->
	<xsl:template match="TreeList">
		<xsl:choose>
			<xsl:when test="count(TreeModel)>1">
				<xsl:element name="Extension">
					<xsl:attribute name="extender"><xsl:text>Insightful</xsl:text></xsl:attribute>
					<xsl:attribute name="name"><xsl:text>X-IMML-TreeModel-TreeList</xsl:text></xsl:attribute>
					<xsl:element name="X-IMML-TreeModel-TreeList">
						<xsl:apply-templates select="TreeModel"/>
					</xsl:element>
				</xsl:element>
			</xsl:when>
			<xsl:otherwise>
				<xsl:apply-templates select="TreeModel"/>
			</xsl:otherwise>
		</xsl:choose>
	</xsl:template>
	<xsl:template match="TreeModel">
		<xsl:element name="TreeModel">
			<xsl:attribute name="modelName"><xsl:value-of select="@name"/></xsl:attribute>
			<xsl:attribute name="functionName"><xsl:value-of select="../@modelType"/></xsl:attribute>
			<xsl:attribute name="splitCharacteristic"><xsl:value-of select="@splitCharacteristic"/></xsl:attribute>
			
			<xsl:element name="Extension">
				<xsl:attribute name="extender"><xsl:text>Insightful</xsl:text></xsl:attribute>
				<xsl:attribute name="name"><xsl:text>X-IMML-XTProps</xsl:text></xsl:attribute>
				<xsl:element name="X-IMML-XTProps">
					<xsl:element name="X-IMML-Property">
						<xsl:attribute name="name">criterion</xsl:attribute>
						<xsl:attribute name="value"><xsl:value-of select="@criterion"/></xsl:attribute>
					</xsl:element>
				</xsl:element>
			</xsl:element>
			
			<!-- MiningSchema is one level up in the TreeList -->			
			<xsl:apply-templates select="../MiningSchema"/>
			<xsl:apply-templates select="Node"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Node">
		<xsl:element name="Node">
			<xsl:attribute name="score"><xsl:value-of select="@score"/></xsl:attribute>
			<xsl:attribute name="recordCount"><xsl:value-of select="@recordCount"/></xsl:attribute>
			<xsl:element name="Extension">
				<xsl:attribute name="extender"><xsl:text>Insightful</xsl:text></xsl:attribute>
				<xsl:attribute name="name"><xsl:text>X-IMML-XTProps</xsl:text></xsl:attribute>
				<xsl:element name="X-IMML-XTProps">
					<xsl:element name="X-IMML-Property">
						<xsl:attribute name="name">id</xsl:attribute>
						<xsl:attribute name="value"><xsl:value-of select="@id"/></xsl:attribute>
					</xsl:element>
					<xsl:element name="X-IMML-Property">
						<xsl:attribute name="name">group</xsl:attribute>
						<xsl:attribute name="value"><xsl:value-of select="@group"/></xsl:attribute>
					</xsl:element>
					<xsl:element name="X-IMML-Property">
						<xsl:attribute name="name">deviance</xsl:attribute>
						<xsl:attribute name="value"><xsl:value-of select="@deviance"/></xsl:attribute>
					</xsl:element>
					<xsl:element name="X-IMML-Property">
						<xsl:attribute name="name">entropy</xsl:attribute>
						<xsl:attribute name="value"><xsl:value-of select="@entropy"/></xsl:attribute>
					</xsl:element>
					<xsl:element name="X-IMML-Property">
						<xsl:attribute name="name">gini</xsl:attribute>
						<xsl:attribute name="value"><xsl:value-of select="@gini"/></xsl:attribute>
					</xsl:element>
					<xsl:element name="X-IMML-Property">
						<xsl:attribute name="name">risk</xsl:attribute>
						<xsl:attribute name="value"><xsl:value-of select="@risk"/></xsl:attribute>
					</xsl:element>
					<xsl:if test="string-length(@yprob) > 0">					
						<xsl:element name="X-IMML-Property">
							<xsl:attribute name="name">yprob</xsl:attribute>
							<xsl:attribute name="value"><xsl:value-of select="@yprob"/></xsl:attribute>
						</xsl:element>
					</xsl:if>
					<xsl:element name="X-IMML-Property">
						<xsl:attribute name="name">improvement</xsl:attribute>
							<xsl:attribute name="value"><xsl:value-of select="child::node()/@improvement"/></xsl:attribute>
					</xsl:element>
				</xsl:element>
			</xsl:element>
			
			<xsl:apply-templates select="SimplePredicate"/>
			<xsl:apply-templates select="CompoundPredicate"/>
			<xsl:apply-templates select="True"/>
			<xsl:apply-templates select="False"/>
			<xsl:apply-templates select="Node"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="SimplePredicate">
		<xsl:element name="SimplePredicate">
			<xsl:attribute name="field"><xsl:value-of select="@field"/></xsl:attribute>
			<xsl:attribute name="operator"><xsl:value-of select="@operator"/></xsl:attribute>
			<xsl:attribute name="value"><xsl:value-of select="@value"/></xsl:attribute>
		</xsl:element>
	</xsl:template>
	<xsl:template match="CompoundPredicate">
		<xsl:element name="CompoundPredicate">
			<xsl:attribute name="booleanOperator"><xsl:value-of select="@booleanOperator"/></xsl:attribute>
			<xsl:apply-templates select="SimplePredicate"/>
			<xsl:apply-templates select="CompoundPredicate"/>
			<xsl:apply-templates select="True"/>
			<xsl:apply-templates select="False"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="True">
		<xsl:element name="True"/>
	</xsl:template>
	<xsl:template match="False">
		<xsl:element name="False"/>
	</xsl:template>
	<!-- NeuralNetwork -->
	<xsl:template match="NeuralNetwork">
		<xsl:element name="NeuralNetwork">
			<xsl:attribute name="modelName"><xsl:value-of select="@modelName"/></xsl:attribute>
			<xsl:attribute name="functionName"><xsl:value-of select="@modelType"/></xsl:attribute>
			<xsl:attribute name="activationFunction"><xsl:value-of select="@activationFunction"/></xsl:attribute>

			<xsl:apply-templates select="NeuralParameters"/>
			<xsl:apply-templates select="DevMatrix"/>
			<xsl:apply-templates select="ErrorWeights"/>	
			<xsl:apply-templates select="MiningSchema"/>
			<xsl:apply-templates select="NeuralInputs"/>
			<xsl:apply-templates select="NeuralLayer"/>
			<xsl:apply-templates select="NeuralOutputs"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="NeuralInputs">
		<xsl:element name="NeuralInputs">
			<xsl:apply-templates select="NeuralInput"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="NeuralInput">
		<xsl:element name="NeuralInput">
			<xsl:attribute name="id"><xsl:value-of select="@id"/></xsl:attribute>
			<!-- Add other DerivedField types if they are implemented -->
			<xsl:apply-templates select="NormContinuous"/>
			<xsl:apply-templates select="NormDiscrete"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="NormContinuous">
		<xsl:element name="DerivedField">
			<xsl:element name="NormContinuous">
				<xsl:attribute name="field"><xsl:value-of select="@field"/></xsl:attribute>
				<xsl:apply-templates select="LinearNorm"/>
			</xsl:element>
		</xsl:element>
	</xsl:template>
	<xsl:template match="LinearNorm">
		<xsl:element name="LinearNorm">
			<xsl:attribute name="orig"><xsl:value-of select="@orig"/></xsl:attribute>
			<xsl:attribute name="norm"><xsl:value-of select="@norm"/></xsl:attribute>
		</xsl:element>
	</xsl:template>
	<xsl:template match="NormDiscrete">
		<xsl:element name="DerivedField">
			<xsl:element name="NormDiscrete">
				<xsl:attribute name="field"><xsl:value-of select="@field"/></xsl:attribute>
				<xsl:attribute name="value"><xsl:value-of select="@value"/></xsl:attribute>
			</xsl:element>
		</xsl:element>
	</xsl:template>
	<xsl:template match="NeuralLayer">
		<xsl:element name="NeuralLayer">
			<xsl:apply-templates select="Neuron"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Neuron">
		<xsl:element name="Neuron">
			<xsl:attribute name="id"><xsl:value-of select="@id"/></xsl:attribute>
			<xsl:apply-templates select="Con"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Con">
		<xsl:element name="Con">
			<xsl:attribute name="from"><xsl:value-of select="@from"/></xsl:attribute>
			<xsl:attribute name="weight"><xsl:value-of select="@weight"/></xsl:attribute>
			
			<xsl:element name="Extension">
				<xsl:attribute name="extender"><xsl:text>Insightful</xsl:text></xsl:attribute>
				<xsl:attribute name="name"><xsl:text>X-IMML-XTProps</xsl:text></xsl:attribute>
				<xsl:element name="X-IMML-XTProps">
					<xsl:element name="X-IMML-Property">
						<xsl:attribute name="name">learnRate</xsl:attribute>
						<xsl:attribute name="value"><xsl:value-of select="@learnRate"/></xsl:attribute>
					</xsl:element>
					<xsl:element name="X-IMML-Property">
						<xsl:attribute name="name">momentum</xsl:attribute>
						<xsl:attribute name="value"><xsl:value-of select="@momentum"/></xsl:attribute>
					</xsl:element>
					<xsl:element name="X-IMML-Property">
						<xsl:attribute name="name">deltaWeight</xsl:attribute>
						<xsl:attribute name="value"><xsl:value-of select="@deltaWeight"/></xsl:attribute>
					</xsl:element>
					<xsl:element name="X-IMML-Property">
						<xsl:attribute name="name">weightDelta</xsl:attribute>
						<xsl:attribute name="value"><xsl:value-of select="@weightDelta"/></xsl:attribute>
					</xsl:element>
				</xsl:element>
			</xsl:element>
		</xsl:element>
	</xsl:template>
	<xsl:template match="NeuralOutputs">
		<xsl:element name="NeuralOutputs">
			<xsl:apply-templates select="NeuralOutput"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="NeuralOutput">
		<xsl:element name="NeuralOutput">
			<!-- Prior to IM 3.0 the attribute name was inconsistent in the IMML between classification and regression -->
			<xsl:attribute name="outputNeuron">
				<xsl:choose>
					<xsl:when test="string-length(@id)>0">
						<xsl:value-of select="@id"/>				
					</xsl:when>
					<xsl:otherwise>
						<xsl:value-of select="@outputNeuron"/>
					</xsl:otherwise>
				</xsl:choose>
			</xsl:attribute>	
			<!-- Add other DerivedField types if they are implemented -->
			<xsl:apply-templates select="NormContinuous"/>
			<xsl:apply-templates select="NormDiscrete"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="NeuralParameters">
		<xsl:element name="Extension">
			<xsl:attribute name="extender"><xsl:text>Insightful</xsl:text></xsl:attribute>
			<xsl:attribute name="name"><xsl:text>X-IMML-NeuralNetwork-NeuralParameters</xsl:text></xsl:attribute>
			<xsl:element name="X-IMML-NeuralNetwork-NeuralParameters">
				<xsl:copy-of select="@*"/>
			</xsl:element>
		</xsl:element>
	</xsl:template>
	<xsl:template match="DevMatrix">
		<xsl:element name="Extension">
			<xsl:attribute name="extender"><xsl:text>Insightful</xsl:text></xsl:attribute>
			<xsl:attribute name="name"><xsl:text>X-IMML-DevMatrix</xsl:text></xsl:attribute>
			<xsl:element name="X-IMML-DevMatrix">
				<xsl:copy-of select="@*"/>
				<xsl:apply-templates select="DevCell"/>
			</xsl:element>
		</xsl:element>	
	</xsl:template>
	<xsl:template match="DevCell">
		<xsl:element name="X-IMML-DevCell">
			<xsl:copy-of select="@*"/>			
		</xsl:element>
	</xsl:template>
	<xsl:template match="ErrorWeights">
		<xsl:element name="Extension">
			<xsl:attribute name="extender"><xsl:text>Insightful</xsl:text></xsl:attribute>
			<xsl:attribute name="name"><xsl:text>X-IMML-NeuralNetwork-ErrorWeights</xsl:text></xsl:attribute>
			<xsl:element name="X-IMML-NeuralNetwork-ErrorWeights">
				<xsl:apply-templates select="XTProps"/>			
			</xsl:element>
		</xsl:element>
	</xsl:template>
	<xsl:template match="XTProps">
		<xsl:element name="X-IMML-XTProps">
			<xsl:apply-templates select="Property"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Property">
		<xsl:element name="X-IMML-Property">
			<xsl:copy-of select="@*"/>
			<xsl:apply-templates select="Property"/>
			<xsl:apply-templates select="Complex"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Complex">
		<xsl:element name="X-IMML-Complex">
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
	<!-- ClusteringModel -->
	<xsl:template match="ClusteringModel">
		<!-- Define the categorical dummy variables -->
		<xsl:element name="TransformationDictionary">
			<xsl:for-each select="../XTMetaData/ColumnDefinitions/ColumnInfo">
				<xsl:if test="@type='categorical'">
					<xsl:for-each select="Level">
						<xsl:element name="DerivedField">
							<xsl:attribute name="name"><xsl:value-of select="../@name"/><xsl:text>(</xsl:text><xsl:value-of select="@value"/><xsl:text>)</xsl:text></xsl:attribute>
							<xsl:element name="NormDiscrete">
								<xsl:attribute name="field"><xsl:value-of select="../@name"/></xsl:attribute>
								<xsl:attribute name="value"><xsl:value-of select="@value"/></xsl:attribute>
							</xsl:element>
						</xsl:element>
					</xsl:for-each>
				</xsl:if>
			</xsl:for-each>
		</xsl:element>
		<xsl:element name="ClusteringModel">
			<xsl:attribute name="modelName"><xsl:value-of select="@modelName"/></xsl:attribute>
			<xsl:attribute name="functionName"><xsl:text>clustering</xsl:text></xsl:attribute>
			<xsl:attribute name="modelClass"><xsl:text>centerBased</xsl:text></xsl:attribute>
			<xsl:attribute name="numberOfClusters"><xsl:value-of select="@numberOfClusters"/></xsl:attribute>
			<xsl:apply-templates select="MiningSchema"/>
			<!-- Note the comparison measure is hard-coded.  Change if multiple measures implemented. -->
			<xsl:element name="ComparisonMeasure">
				<xsl:attribute name="kind"><xsl:text>distance</xsl:text></xsl:attribute>
				<xsl:element name="squaredEuclidean"/>
			</xsl:element>
			<xsl:apply-templates select="ClusteringField"/>
			<xsl:apply-templates select="Cluster"/>
			
			<xsl:apply-templates select="Total"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="ClusteringField">
		<xsl:variable name="index" select="position()"/>
		<xsl:element name="ClusteringField">
			<xsl:attribute name="field"><xsl:value-of select="@field"/></xsl:attribute>
			<!-- Columns are normalized by the stdDev stored in the IMML InputScaling element -->
			<!-- Should perhaps use DerivedFields rather than storing the stdDev as the similarityScale -->
			<xsl:attribute name="similarityScale">
				<xsl:value-of select="../InputScaling/InputScalingField[$index]/@data"/>
			</xsl:attribute>
			<!-- Put in compareFunction attribute if multiple measures implemented. -->
		</xsl:element>
	</xsl:template>
	<xsl:template match="Cluster">
		<xsl:element name="Cluster">
			<xsl:attribute name="name"><xsl:value-of select="@name"/></xsl:attribute>

			<xsl:element name="Extension">
				<xsl:attribute name="extender"><xsl:text>Insightful</xsl:text></xsl:attribute>
				<xsl:attribute name="name"><xsl:text>X-IMML-XTProps</xsl:text></xsl:attribute>
				<xsl:element name="X-IMML-XTProps">
					<xsl:element name="X-IMML-Property">
						<xsl:attribute name="name">sizeInPercent</xsl:attribute>
						<xsl:attribute name="value"><xsl:value-of select="@sizeInPercent"/></xsl:attribute>
					</xsl:element>
					<xsl:element name="X-IMML-Property">
						<xsl:attribute name="name">size</xsl:attribute>
						<xsl:attribute name="value"><xsl:value-of select="@size"/></xsl:attribute>
					</xsl:element>	
					<xsl:element name="X-IMML-Property">
						<xsl:attribute name="name">withinSS</xsl:attribute>
						<xsl:attribute name="value"><xsl:value-of select="@withinSS"/></xsl:attribute>
					</xsl:element>				
					<xsl:element name="X-IMML-Property">
						<xsl:attribute name="name">withinSSOverSize</xsl:attribute>
						<xsl:attribute name="value"><xsl:value-of select="@withinSSOverSize"/></xsl:attribute>
					</xsl:element>
				</xsl:element>
			</xsl:element>

			<xsl:apply-templates select="Center"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Center">
		<xsl:call-template name="CreateClusterArray"/>
	</xsl:template>
	<xsl:template match="Total">
			<xsl:element name="Extension">
				<xsl:attribute name="extender"><xsl:text>Insightful</xsl:text></xsl:attribute>
				<xsl:attribute name="name"><xsl:text>X-IMML-XTProps</xsl:text></xsl:attribute>
				<xsl:element name="X-IMML-XTProps">
					<xsl:element name="X-IMML-Property">
						<xsl:attribute name="name">Total</xsl:attribute>
						<xsl:element name="X-IMML-Property">
							<xsl:attribute name="name">size</xsl:attribute>
							<xsl:attribute name="value"><xsl:value-of select="@size"/></xsl:attribute>
						</xsl:element>	
						<xsl:element name="X-IMML-Property">
							<xsl:attribute name="name">sumSq</xsl:attribute>
							<xsl:attribute name="value"><xsl:value-of select="@sumSq"/></xsl:attribute>
						</xsl:element>				
						<xsl:element name="X-IMML-Property">
							<xsl:attribute name="name">sumSqOverSize</xsl:attribute>
							<xsl:attribute name="value"><xsl:value-of select="@sumSqOverSize"/></xsl:attribute>
						</xsl:element>
					</xsl:element>
				</xsl:element>
			</xsl:element>		
	</xsl:template>	
	
	<!-- Template to create the Array of Cluster centers from separate CenterField data attributes -->
	<xsl:template name="CreateClusterArray">
		<xsl:param name="position" select="1"/>
		<xsl:param name="string" select="''"/>
		<xsl:variable name="numFields" select="count(./CenterField)"/>
		<xsl:variable name="separator" select="' '"/>
		<xsl:choose>
			<xsl:when test="$position &lt;= $numFields">
				<xsl:call-template name="CreateClusterArray">
					<xsl:with-param name="position">
						<xsl:value-of select="$position + 1"/>
					</xsl:with-param>
					<xsl:with-param name="string">
						<xsl:value-of select="concat($string, $separator, ./CenterField[position()=$position]/@data)"/>
					</xsl:with-param>
				</xsl:call-template>
			</xsl:when>
			<xsl:otherwise>
				<!-- Have appended all values to the string -->
				<xsl:element name="Array">
					<xsl:attribute name="n"><xsl:value-of select="$numFields"/></xsl:attribute>
					<xsl:attribute name="type">real</xsl:attribute>
					<xsl:value-of select="normalize-space($string)"/>
				</xsl:element>
			</xsl:otherwise>
		</xsl:choose>
	</xsl:template>
	<!-- GeneralRegressionModel -->
	<xsl:template match="GeneralRegressionModel">
		<xsl:element name="GeneralRegressionModel">
			<xsl:attribute name="targetVariableName"><xsl:value-of select="@targetVariableName"/></xsl:attribute>
			<xsl:attribute name="modelType"><xsl:value-of select="@modelType"/></xsl:attribute>
			<!-- For now, assume generalLinear is regression and anything else is classification -->
			<xsl:attribute name="functionName"><xsl:choose><xsl:when test="@modelType='generalLinear'"><xsl:text>regression</xsl:text></xsl:when><xsl:otherwise><xsl:text>classification</xsl:text></xsl:otherwise></xsl:choose></xsl:attribute>

			<xsl:apply-templates select="ParameterList"/>
			<xsl:apply-templates select="FactorList"/>
			<xsl:apply-templates select="CovariateList"/>
			<xsl:apply-templates select="PPMatrix"/>
			<xsl:apply-templates select="PCovMatrix"/>
			<xsl:apply-templates select="ParamMatrix"/>
			
			<xsl:apply-templates select="DevMatrix"/>
			<xsl:apply-templates select="CoefCorr"/>
			<xsl:apply-templates select="Importance"/>
			<xsl:apply-templates select="RMatrix"/>
			<xsl:apply-templates select="QTYMatrix"/>
			<xsl:apply-templates select="ModelExtraInfo"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="ParameterList">
		<xsl:element name="ParameterList">
			<xsl:apply-templates select="Parameter"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Parameter">
		<xsl:element name="Parameter">
			<xsl:attribute name="name"><xsl:value-of select="@name"/></xsl:attribute>
		</xsl:element>
	</xsl:template>
	<xsl:template match="FactorList">
		<xsl:element name="FactorList">
			<xsl:apply-templates select="Predictor"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Predictor">
		<xsl:element name="Predictor">
			<xsl:attribute name="name"><xsl:value-of select="@name"/></xsl:attribute>
		</xsl:element>
	</xsl:template>
	<xsl:template match="CovariateList">
		<xsl:element name="CovariateList">
			<xsl:apply-templates select="Predictor"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="PPMatrix">
		<xsl:element name="PPMatrix">
			<xsl:apply-templates select="PPCell"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="PPCell">
		<xsl:element name="PPCell">
			<xsl:attribute name="value"><xsl:value-of select="@value"/></xsl:attribute>
			<xsl:attribute name="predictorName"><xsl:value-of select="@predictorName"/></xsl:attribute>
			<xsl:attribute name="parameterName"><xsl:value-of select="@variableName"/></xsl:attribute>
		</xsl:element>
	</xsl:template>
	<xsl:template match="PCovMatrix">
		<xsl:element name="PCovMatrix">
			<xsl:apply-templates select="PCovCell"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="PCovCell">
		<xsl:element name="PCovCell">
			<xsl:attribute name="pRow"><xsl:value-of select="@pRow"/></xsl:attribute>
			<xsl:attribute name="pCol"><xsl:value-of select="@pCol"/></xsl:attribute>
			<xsl:attribute name="tRow"><xsl:value-of select="@tRow"/></xsl:attribute>
			<xsl:attribute name="tCol"><xsl:value-of select="@tCol"/></xsl:attribute>
			<xsl:attribute name="value"><xsl:value-of select="@value"/></xsl:attribute>
			<xsl:attribute name="targetCategory"><xsl:value-of select="@targetCategory"/></xsl:attribute>
		</xsl:element>
	</xsl:template>
	<xsl:template match="ParamMatrix">
		<xsl:element name="ParamMatrix">
			<xsl:apply-templates select="PCell"/>
		</xsl:element>
		<xsl:element name="Extension">
			<xsl:attribute name="extender"><xsl:text>Insightful</xsl:text></xsl:attribute>
			<xsl:attribute name="name"><xsl:text>X-IMML-GeneralRegressionModel-ParamMatrixExtraAttributes</xsl:text></xsl:attribute>
			<xsl:element name="X-IMML-GeneralRegressionModel-ParamMatrixExtraAttributes">
				<xsl:element name="X-IMML-XTProps">
					<xsl:for-each select="PCell">
						<xsl:element name="X-IMML-Property">
							<xsl:attribute name="name">parameterName</xsl:attribute>
							<xsl:attribute name="value"><xsl:value-of select="@parameterName"/></xsl:attribute>
							<xsl:element name="X-IMML-Property">
								<xsl:attribute name="name">stdErr</xsl:attribute>
								<xsl:attribute name="value"><xsl:value-of select="@stdErr"/></xsl:attribute>
							</xsl:element>
							<xsl:element name="X-IMML-Property">
								<xsl:attribute name="name">t</xsl:attribute>
								<xsl:attribute name="value"><xsl:value-of select="@t"/></xsl:attribute>
							</xsl:element>	
							<xsl:element name="X-IMML-Property">
								<xsl:attribute name="name">Pr</xsl:attribute>
								<xsl:attribute name="value"><xsl:value-of select="@Pr"/></xsl:attribute>
							</xsl:element>				
						</xsl:element>
					</xsl:for-each>
				</xsl:element>
			</xsl:element>
		</xsl:element>
	</xsl:template>
	<xsl:template match="PCell">
		<xsl:element name="PCell">
			<xsl:attribute name="targetCategory"><xsl:value-of select="@targetCategory"/></xsl:attribute>
			<xsl:attribute name="parameterName"><xsl:value-of select="@parameterName"/></xsl:attribute>
			<xsl:attribute name="beta"><xsl:value-of select="@beta"/></xsl:attribute>
			<xsl:attribute name="df"><xsl:value-of select="@df"/></xsl:attribute>
		</xsl:element>
	</xsl:template>
	<xsl:template match="CoefCorr">
		<xsl:element name="Extension">
			<xsl:attribute name="extender"><xsl:text>Insightful</xsl:text></xsl:attribute>
			<xsl:attribute name="name"><xsl:text>X-IMML-GeneralRegressionModel-CoefCorr</xsl:text></xsl:attribute>
			<xsl:element name="X-IMML-GeneralRegressionModel-CoefCorr">
				<xsl:copy-of select="@*"/>
				<xsl:apply-templates select="CorrCell"/>
			</xsl:element>
		</xsl:element>	
	</xsl:template>
	<xsl:template match="CorrCell">
		<xsl:element name="X-IMML-GeneralRegressionModel-CorrCell">
			<xsl:copy-of select="@*"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Importance">
		<xsl:element name="Extension">
			<xsl:attribute name="extender"><xsl:text>Insightful</xsl:text></xsl:attribute>
			<xsl:attribute name="name"><xsl:text>X-IMML-GeneralRegressionModel-Importance</xsl:text></xsl:attribute>
			<xsl:element name="X-IMML-GeneralRegressionModel-Importance">
				<xsl:copy-of select="@*"/>
				<xsl:apply-templates select="Effect"/>
			</xsl:element>
		</xsl:element>	
	</xsl:template>
	<xsl:template match="Effect">
		<xsl:element name="X-IMML-GeneralRegressionModel-Effect">
			<xsl:copy-of select="@*"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="RMatrix">
		<xsl:element name="Extension">
			<xsl:attribute name="extender"><xsl:text>Insightful</xsl:text></xsl:attribute>
			<xsl:attribute name="name"><xsl:text>X-IMML-GeneralRegressionModel-RMatrix</xsl:text></xsl:attribute>
			<xsl:element name="X-IMML-GeneralRegressionModel-RMatrix">
				<xsl:copy-of select="@*"/>
			 	<xsl:call-template name="OutputMatrix"><xsl:with-param name="matrix" select="."/></xsl:call-template>
			 </xsl:element>
		</xsl:element>	
	</xsl:template>
	<xsl:template match="QTYMatrix">
		<xsl:element name="Extension">
			<xsl:attribute name="extender"><xsl:text>Insightful</xsl:text></xsl:attribute>
			<xsl:attribute name="name"><xsl:text>X-IMML-GeneralRegressionModel-QTYMatrix</xsl:text></xsl:attribute>
			<xsl:element name="X-IMML-GeneralRegressionModel-QTYMatrix">
				<xsl:copy-of select="@*"/>
			 	<xsl:call-template name="OutputMatrix"><xsl:with-param name="matrix" select="."/></xsl:call-template>
			 </xsl:element>
		</xsl:element>	
	</xsl:template>
	<xsl:template name="OutputMatrix">
		<xsl:param name="matrix" select="''"/>

		<xsl:for-each select="$matrix/Row">
			<xsl:variable name="row" select="." />
			<xsl:element name="X-IMML-GeneralRegressionModel-Row">
				<xsl:copy-of select="@*"/>
				<xsl:for-each select="$row/Col">
					<xsl:variable name="col" select="." />
					<xsl:element name="X-IMML-GeneralRegressionModel-Col">
						<xsl:copy-of select="@*"/>
					</xsl:element>
				</xsl:for-each>
			</xsl:element>
		</xsl:for-each>
	</xsl:template>
	<xsl:template match="ModelExtraInfo">
		<xsl:element name="Extension">
			<xsl:attribute name="extender"><xsl:text>Insightful</xsl:text></xsl:attribute>
			<xsl:attribute name="name"><xsl:text>X-IMML-GeneralRegressionModel-ModelExtraInfo</xsl:text></xsl:attribute>
			<xsl:element name="X-IMML-GeneralRegressionModel-ModelExtraInfo">
				<xsl:apply-templates select="Family"/>
				<xsl:apply-templates select="Link"/>
				<xsl:apply-templates select="Mean"/>
				<xsl:apply-templates select="Var"/>
				<xsl:apply-templates select="Iterations"/>
			 </xsl:element>
		</xsl:element>	
	</xsl:template>
	<xsl:template match="Family">
		<xsl:element name="X-IMML-GeneralRegressionModel-Family">
			<xsl:copy-of select="@*"/>
		 </xsl:element>
	</xsl:template>
	<xsl:template match="Link">
		<xsl:element name="X-IMML-GeneralRegressionModel-Link">
			<xsl:copy-of select="@*"/>
		 </xsl:element>
	</xsl:template>
	<xsl:template match="Mean">
		<xsl:element name="X-IMML-GeneralRegressionModel-Mean">
			<xsl:copy-of select="@*"/>
		 </xsl:element>
	</xsl:template>
	<xsl:template match="Var">
		<xsl:element name="X-IMML-GeneralRegressionModel-Var">
			<xsl:copy-of select="@*"/>
		 </xsl:element>
	</xsl:template>
	<xsl:template match="Iterations">
		<xsl:element name="X-IMML-GeneralRegressionModel-Iterations">
			<xsl:copy-of select="@*"/>
		 </xsl:element>
	</xsl:template>
	<!-- NaiveBayes -->
	<xsl:template match="NaiveBayesModel">
		<xsl:element name="NaiveBayesModel">
			<!-- TODO: Add modelName and threshold to IMML version instead of hardcoding them. -->
			<!--	<xsl:attribute name="modelName">
				<xsl:text>Naive Bayes</xsl:text>
			</xsl:attribute> -->
			<xsl:attribute name="functionName"><xsl:text>classification</xsl:text></xsl:attribute>
			<xsl:attribute name="threshold"><xsl:text>0.001</xsl:text></xsl:attribute>
			<xsl:for-each select="XTProps[1]">
				<!-- The MiningSchema can be constructed from the Property name values -->
				<xsl:element name="MiningSchema">
					<xsl:for-each select="Property[@name='counts']/Property">
						<xsl:element name="MiningField">
							<xsl:attribute name="name"><xsl:value-of select="@name"/></xsl:attribute>
						</xsl:element>
					</xsl:for-each>
					<xsl:for-each select="Property[@name='dependentColumn'][1]">
						<xsl:element name="MiningField">
							<xsl:attribute name="name"><xsl:value-of select="@value"/></xsl:attribute>
						</xsl:element>
					</xsl:for-each>
				</xsl:element>
				<!-- The "counts" Property stores the predictor information -->
				<xsl:element name="BayesInputs">
					<xsl:for-each select="Property[@name='counts']/Property">
						<xsl:element name="BayesInput">
							<xsl:attribute name="fieldName"><xsl:value-of select="@name"/></xsl:attribute>
							<xsl:for-each select="Property">
								<xsl:element name="PairCounts">
									<xsl:attribute name="value"><xsl:value-of select="@name"/></xsl:attribute>
									<xsl:element name="TargetValueCounts">
										<xsl:for-each select="Property">
											<xsl:element name="TargetValueCount">
												<xsl:attribute name="value"><xsl:value-of select="@name"/></xsl:attribute>
												<xsl:attribute name="count"><xsl:value-of select="@value"/></xsl:attribute>
											</xsl:element>
										</xsl:for-each>
									</xsl:element>
								</xsl:element>
							</xsl:for-each>
						</xsl:element>
					</xsl:for-each>
				</xsl:element>
				<!-- The "dependentColumn" Property stores the response information -->
				<xsl:for-each select="Property[@name='dependentColumn']">
					<xsl:element name="BayesOutput">
						<xsl:attribute name="fieldName"><xsl:value-of select="@value"/></xsl:attribute>
						<xsl:element name="TargetValueCounts">
							<!-- Don't include 'total'. -->
							<xsl:for-each select="Property[@name!='total']">
								<xsl:element name="TargetValueCount">
									<xsl:attribute name="value"><xsl:value-of select="@name"/></xsl:attribute>
									<xsl:attribute name="count"><xsl:value-of select="@value"/></xsl:attribute>
								</xsl:element>
							</xsl:for-each>
						</xsl:element>
					</xsl:element>
				</xsl:for-each>
			</xsl:for-each>
		</xsl:element>
	</xsl:template>
	<!-- PrincipalComponents -->
	<!-- PMML has no PrincipalComponentsModel specification.  This is the XML representation in Insightful Miner. -->
	<xsl:template match="PrincipalComponentsModel">
		<xsl:element name="Extension">
			<xsl:attribute name="extender"><xsl:text>Insightful</xsl:text></xsl:attribute>
			<xsl:attribute name="name"><xsl:text>X-IMML-PrincipalComponentsModel</xsl:text></xsl:attribute>
			<xsl:element name="X-IMML-PrincipalComponentsModel">
				<xsl:copy-of select="@*"/>
				<xsl:apply-templates select="MiningSchema"/>
				<xsl:apply-templates select="Parameters"/>
				<xsl:apply-templates select="TermIndicies"/>
				<xsl:apply-templates select="PrinComp"/>
			</xsl:element>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Parameters">
		<xsl:element name="X-IMML-PrincipalComponentsModel-Parameters">
			<xsl:copy-of select="@*"/>
			<xsl:apply-templates select="Param"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="TermIndicies">
		<xsl:element name="X-IMML-PrincipalComponentsModel-TermIndicies">
			<xsl:copy-of select="@*"/>
			<xsl:apply-templates select="Index"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="PrinComp">
		<xsl:element name="X-IMML-PrincipalComponentsModel-PrinComp">
			<xsl:copy-of select="@*"/>
			<xsl:apply-templates select="Component"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Param">
		<xsl:element name="X-IMML-PrincipalComponentsModel-Param">
			<xsl:copy-of select="@*"/>
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Index">
		<xsl:element name="X-IMML-PrincipalComponentsModel-Index">
			<xsl:copy-of select="@*"/>
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Component">
		<xsl:element name="X-IMML-PrincipalComponentsModel-Component">
			<xsl:copy-of select="@*"/>
			<xsl:apply-templates select="Loading"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Loading">
		<xsl:element name="X-IMML-PrincipalComponentsModel-Loading">
			<xsl:copy-of select="@*"/>
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
	<!-- SurvivalModel -->
	<!-- PMML has no SurvivalModel specification.  This is the XML representation in Insightful Miner. -->
	<xsl:template match="SurvivalModel">
		<xsl:element name="Extension">
			<xsl:attribute name="extender"><xsl:text>Insightful</xsl:text></xsl:attribute>
			<xsl:attribute name="name"><xsl:text>X-IMML-SurvivalModel</xsl:text></xsl:attribute>
			<xsl:element name="X-IMML-SurvivalModel">
				<xsl:attribute name="type"><xsl:value-of select="@type"/></xsl:attribute>
				
				<xsl:apply-templates select="MiningSchema"/>
				<xsl:apply-templates select="*"/>
			</xsl:element>
		</xsl:element>
	</xsl:template>
	<xsl:template match="SurvResponse">
		<xsl:element name="X-IMML-SurvivalModel-SurvResponse">
			<xsl:copy-of select="@*"/>
			<xsl:apply-templates select="*"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="EventCodes">
		<xsl:element name="X-IMML-SurvivalModel-EventCodes">
			<xsl:copy-of select="@*"/>
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="LinearModel">
		<xsl:element name="X-IMML-SurvivalModel-LinearModel">
			<xsl:copy-of select="@*"/>
			<xsl:apply-templates select="*"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Term">
		<xsl:element name="X-IMML-SurvivalModel-Term">
			<xsl:copy-of select="@*"/>
			<xsl:apply-templates select="*"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="TermVar">
		<xsl:element name="X-IMML-SurvivalModel-TermVar">
			<xsl:copy-of select="@*"/>
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Intercept">
		<xsl:element name="X-IMML-SurvivalModel-Intercept">
			<xsl:copy-of select="@*"/>
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Weights">
		<xsl:element name="X-IMML-SurvivalModel-Weights">
			<xsl:copy-of select="@*"/>
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Strata">
		<xsl:element name="X-IMML-SurvivalModel-Strata">
			<xsl:copy-of select="@*"/>
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Coefficients">
		<xsl:element name="X-IMML-SurvivalModel-Coefficients">
			<xsl:copy-of select="@*"/>
			<xsl:apply-templates select="*"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Coef">
		<xsl:element name="X-IMML-SurvivalModel-Coef">
			<xsl:copy-of select="@*"/>
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Hessian">
		<xsl:element name="X-IMML-SurvivalModel-Hessian">
			<xsl:copy-of select="@*"/>
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="SurvModelStats">
		<xsl:element name="X-IMML-SurvivalModel-SurvModelStats">
			<xsl:copy-of select="@*"/>
			<xsl:apply-templates select="*"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="LikelihoodRatio">
		<xsl:element name="X-IMML-SurvivalModel-LikelihoodRatio">
			<xsl:copy-of select="@*"/>
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="BaselineSurvival">
		<xsl:element name="X-IMML-SurvivalModel-BaselineSurvival">
			<xsl:copy-of select="@*"/>
			<xsl:apply-templates select="*"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Time">
		<xsl:element name="X-IMML-SurvivalModel-Time">
			<xsl:copy-of select="@*"/>
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Failures">
		<xsl:element name="X-IMML-SurvivalModel-Failures">
			<xsl:copy-of select="@*"/>
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="RiskSet">
		<xsl:element name="X-IMML-SurvivalModel-RiskSet">
			<xsl:copy-of select="@*"/>
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="Hazard">
		<xsl:element name="X-IMML-SurvivalModel-Hazard">
			<xsl:copy-of select="@*"/>
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="VarHazard">
		<xsl:element name="X-IMML-SurvivalModel-VarHazard">
			<xsl:copy-of select="@*"/>
			<xsl:copy-of select="text()"/>
		</xsl:element>
	</xsl:template>
</xsl:stylesheet>
