StatesDemo <- function(vartoplot="Population")

{

###########################################################################
###
### StatesDemo: Produces a colormap graphlet of the 48 contiguous states.
###              
###             
### Arguments:
###
### vartoplot: Column name from the states dataframe. This is the variable
###            that will be plotted in the colorpmap. Note. The dataset for
###            the colormap is currently hardwired to the states dataset
###            which is included in data.sdd 
###
###########################################################################


  ## Set the java color scheme to trellis colors
  java.set.colorscheme(java.colorscheme.trellis)

  ## Strip out any "%20" string components that may be present in vartoplot.
  ## These originate from white spaces in labels sent to java.identify()
  vartoplot <- token(source=vartoplot)
  
  ##Get the names of the states from mapping the "state" map.
  name.polys <- map("state", namesonly=T,plot=F)

  ## Coding required to strip out the state names from name.polys
  idx <- regexpr(pat=":",name.polys) == -1
  name.polys[idx] <- paste(name.polys[idx],":x",sep="")
  tmp <- unpaste(name.polys, sep=":")
  tmp <- unpaste(tmp[[1]], sep=",")

  ## Match the state names from the map with the row names in states
  poly.county.id <- match(tmp[[1]],as.character(casefold(unlist(dimnames(states)[1]))))
  
  ## Grab the value to be plotted in the colormap
  dat <- states[,vartoplot]
  
  ## Fix color levels
  dat.scaled <- (dat-min(dat, na.rm=T))/diff(range(dat, na.rm=T))
  cols <- trellis.par.get("regions")$col
  ncols <- length(cols)
  thecols <- cols[round(1+dat.scaled*(ncols-1))]
  thecols[is.na(thecols)] <- 0   
  cols.to.plot <- thecols[poly.county.id]


  ## Produce the colormap
  map("state", fill=T, color=cols.to.plot, resolution=0)

  ## Allow new plot to overlay this plot.
  par(new=T)
  
  ## Second call to map creates "nicer" state boundaries
  map("state", resolution=0)

  ## Plot points at state capitals
  points(capitals$X, capitals$Y, pch=16,col=8)

  ## Add a tilte to the plot
  title(main=paste(vartoplot, " by State",sep=""))

  ## Add a legend to the plot 
  image.legend(dat,x=c(-125,-100),y=c(22,25), hor=T)
   
  ## Produce the "change colormap box".
  ## Hardcoded xvars and yvar values for locating objects
  ## in the "change colormap box".
  xvars <- c(-78,-78)
  yvars <- c(28,30)
  polygon(x=c(-79,-65,-65,-79),y=c(27,27,33,33),col=12,border=T)
  text(x=-78, y=32,labels="Change Map to:",adj= 0)
  pos.z <- c("Population","Life Exp", "Income")
  not.z.idx <- !is.element(pos.z,vartoplot)
  zplots <- pos.z[not.z.idx]
  text(x=(xvars + 2) , y=yvars,labels=zplots, adj=0)	
  points(xvars, yvars, pch=16,cex=1.0, col=1)


## Set up the points that can selected within the graphlet
## and create the commands that will produce the susequent 
## analysis request. Two components: The actions that will 
## be available from selecting a state capital and the actions
## for requests to change the colormap variable. 
  java.identify(capitals$X, capitals$Y,	labels=capitals$State.name,
		actions = paste(
		  '<menu title="', 'States of the USA', '">',
		  '<menuitem label="County & City Plots">',
		  MakeAction(map.counties,state=as.character(capitals$State.name)),
		  '</menuitem>',
		  '<menuitem label="City Only Plots">',
		  MakeAction(map.cities,state=as.character(capitals$State.name)),
		  '</menuitem>',
		  '</menu>'))

  java.identify(xvars, yvars,	labels=zplots,
		actions = MakeAction(StatesDemo,vartoplot=zplots))

 ## Comment pointing to the Floria Vote Analysis 
  lines(x=c(-84.28,-85),y=c(30.46,24))
  polygon(x=c(-90,-80,-80,-90),y=c(22,22,24,24),col=10,border=T)
  text(x=-85,y=23,labels="Vote Analysis",adj=.5)



}

