/**
 * SpConnection wraps an SplusSession interface.
 * Most of the code is logon code, to obtain a session.
 * The other methods provide access to the session that
 *   a servlet typically needs.  The access functions
 *   tend to be at a somewhat higher level than the
 *   functions in the SplusSession interface.
 *
 * author Gary Nelson, gnelson@insightful.com
 * version 24 April 2001
 */

import java.util.*;
import java.io.*;
import java.rmi.RemoteException;
import javax.swing.event.*;      // ChangeListener
import com.insightful.splus.*;

public class SpConnection implements ChangeListener
{
	protected SplusSession m_session = null;

	// friendly name for this connection
	protected String m_strFriendlyName;

	protected String m_strLastError = null;

	// login parameters
	protected String m_strHost = null;
	protected String m_strUsername = null;
	protected String m_strPassword = null;
	protected String m_strRMIPort = "1099";
	protected String m_strWorkingDirectory = "";
	protected String m_strXDisplay = "";
	protected String m_strPrompt = "";
	protected String m_strScript = "";

	/* constructor */
	public SpConnection(String strFriendlyName) {
		m_strFriendlyName = (strFriendlyName != null) ?
			strFriendlyName : "";
	}

	//==============================================================
	// Set and get methods
	//==============================================================

	public void setFriendlyName(String strFriendlyName)   
		{ m_strFriendlyName = strFriendlyName; }
	public String getFriendlyName() { return m_strFriendlyName; }

	public String getLastError() { return m_strLastError; }

	public void setHost(String strHost)           
		{ m_strHost = strHost; }
	public void setUsername(String strUsername)   
		{ m_strUsername = strUsername; }
	public void setPassword(String strPassword)           
		{ m_strPassword = strPassword; }
	public void setRMIPort(String strRMIPort)           
		{ m_strRMIPort = strRMIPort; }
	public void setWorkingDirectory(String strWorkingDirectory)
		{ m_strWorkingDirectory = strWorkingDirectory; }
	public void setXDisplay(String strXDisplay)           
		{ m_strXDisplay = strXDisplay; }
	public void setPrompt(String strPrompt)           
		{ m_strPrompt = strPrompt; }
	public void setScript(String strScript)           
		{ m_strScript = strScript; }


	//==============================================================
	// Connect, disconnect, and reconnect code
	//==============================================================

	/* Connect to a remote session. */
	protected synchronized boolean connect() {
		if (m_session != null) { // already connected
			return true;
		}

		// Check completeness of the initialization information.
		boolean bInfoOK = (m_strHost != null) &&
			(m_strUsername != null) && (m_strPassword != null);
		if (!bInfoOK) {
			onConnectFailure("Host, username or password missing");
			return false;
		}

		if (m_strRMIPort == null || m_strRMIPort.length() == 0) {
			m_strRMIPort = "1099";
		}

		int nRMIPort;
		try {
			nRMIPort = Integer.parseInt(m_strRMIPort);
		}
		catch (NumberFormatException ex) {
			onConnectFailure("Invalid RMI port");
			return false;
		}

		// According to Richard Cook, 
		//   setting this property is necessary.
		System.setProperty("splus.client.mode", "true");

		// Create the session.
		try {
			SplusLogin login = new SplusLogin();
			m_session = login.newSession(
				m_strHost, m_strUsername, m_strPassword,
				nRMIPort, m_strWorkingDirectory, m_strXDisplay,
				m_strPrompt, m_strScript, new String[0]);
		}
		catch (SplusLogin.SplusLoginException ex) {
			onConnectFailure(ex.getMessage());
			return false;
		}
		catch (SplusLogin.SplusLoginFatalException ex) {
			onConnectFailure(ex.getMessage());
			return false;
		}

		// Register a client object factory.
		try {
			ClientObjFactoryGeneric cofg =
				new ClientObjFactoryGeneric();
			cofg.addChangeListener(this);
			m_session.setClientObjectFactory(cofg);
		}
		catch (RemoteException ex) {
			onConnectFailure("Failed to create or register " +
				"client object factory");
			return false;
		}

		System.out.println("SpConnection \""
			+ m_strFriendlyName + "\" connected");
		return true;
	}

	/* Record an error that occurred when logging on. */
	protected void onConnectFailure(String strError) {
		System.err.println("SpConnection " + m_strFriendlyName + ":");
		System.err.println("\tSplusLogin.newSession failed: " +
			strError);
		m_strLastError = strError;
	}

	/* Handle a signal that the server is shutting down. */
	public void stateChanged(ChangeEvent ev) {
		m_session = null;
		m_strLastError = "Connection shut down by server";
	}

	/* Try to reconnect using all the parameters from the last try. */
	public boolean reconnect() {
		disconnect();
		return connect();
	}

	/* Disconnect. */
	public void disconnect() {
		if (m_session == null) {
			return;
		}

		try {
			m_session.sessionExit(false);
		}
		catch (RemoteException ex) {}

		System.out.println("SpConnection \"" +
			m_strFriendlyName + "\" disconnected");

		m_session = null;
	}


	//==============================================================
	// methods for accessing the session
	//==============================================================

	/**
	 * Call S-PLUS to evaluate an S-PLUS expression.
	 * If the request times out, this method will resubmit the
	 *   expression several times before giving up.
	 **/
	public SplusDataResult eval(String strExp,
		boolean bGetOutput, boolean bGetResult,
		boolean bGetErrors, boolean bGetWarnings, boolean bGetExpr)
	throws RemoteException
	{
		if (m_session == null) {
			throw new RemoteException("Invalid S-PLUS connection.");
		}

		return callSplus("{" + strExp + "}",
			bGetOutput, bGetResult, bGetErrors,
			bGetWarnings, bGetExpr);
	}

	// Call S-PLUS to evaluate an expression.
	// If the call times out, try again.
	synchronized protected SplusDataResult callSplus(String strExp,
		boolean bGetOutput, boolean bGetResult,
		boolean bGetErrors, boolean bGetWarnings, boolean bGetExpr)
	throws RemoteException
	{
		// Timeouts were a problem with Analytic Server 1.0, so I
		//   added this code to resubmit queries that timed out.
		//   Timeouts seem less frequent now, e.g. with Analytic
		//   Server 2.0, so perhaps the looping can be removed.
		int nNumTries;
		final int nMaxTries = 5;
		for (nNumTries = 0; nNumTries < nMaxTries; nNumTries++) {
			try {
				SplusDataResult spdr = m_session.evalDataQuery(
					strExp, bGetOutput, bGetResult, bGetErrors,
					bGetWarnings, bGetExpr);
				return spdr;
			}
			// I think an incomplete expression is handled best as 
			//   an S-PLUS error rather than an exception.
			catch (SplusIncompleteExpressionException ex) {
				return new SplusDataResult("",
					"Incomplete expression.\n" +
				    "Check for mismatched quotes or parentheses.",
					strExp, new String[0]);
			}
			catch (SplusEngineBusyException ex) {
				// Handle a timeout by waiting and trying again.
				System.out.println("");
				System.out.println("+-----------------------+");
				System.out.println("| Resubmitting query... |");
				System.out.println("|   (Try #" + (nNumTries+1) + 
					" failed)     |");
				System.out.println("+-----------------------+");
				System.out.println("");
				// Wait a few seconds before trying again.
				try {
					Thread.sleep(5000);
				}
				catch (InterruptedException ex2) {}
			}
		}
		String strErrMsg = "Engine is busy after " +
			nNumTries + "attempts";
		System.out.println(strErrMsg);
		throw new RemoteException(strErrMsg);
	}

	/**
	 * Move a file from the server to the local computer.
	 * If bDeleteRemote is true, delete the server copy.
	 **/
	public void getRemoteFile(String strRemoteFilename, 
		String strLocalFilename, boolean bDeleteRemote)
	throws RemoteException, IOException
	{
		// Quit if the S-PLUS session is not valid.
		if (m_session == null) {
			throw new RemoteException("Invalid S-PLUS connection.");
		}

		// A FileInputStreamProxy has the same methods as a 
		//   FileInputStream; however,
		//   * Some read() method calls need to be followed by
		//     calls to getReadByteValues() to actually retrieve
		//     the bytes, and
		//   * All methods throw RemoteExceptions in addition to 
		//     other IOExceptions.
		FileInputStreamProxy strmIn = null;
		FileOutputStream strmOut = null;

		try {
			// Open streams for the transfer.
			strmIn = m_session.getFileInputStream(strRemoteFilename);
			strmOut = new FileOutputStream(strLocalFilename);

			// Perform the transfer.
			final int MAX_BYTES = 4096;
			byte[] buf = strmIn.read(MAX_BYTES);
			int nNumRead = buf.length;
			while (nNumRead > 0) {
				strmOut.write(buf, 0, nNumRead);
				buf = strmIn.read(MAX_BYTES);
				nNumRead = buf.length;
			}

			// Delete the remote file.
			if (bDeleteRemote) {
				eval("unlink(\"" + strRemoteFilename + "\")",
					false, false, false, false, false);
			}
		}
		finally { // Close the streams.
			if (strmOut != null) {
				try { strmOut.close(); }
				catch (IOException ex) {}
			}
			if (strmIn != null) {
				try { strmIn.close(); }
				catch (IOException ex) {}
			}
		}
	}
}
