/*
	Connect/C++ : Copyright (c) 2001, 2006 Insightful Corp.
	All rights reserved.
	Version 6.0: 2001
*/

// spobject.cxx: implementation of the CSPobject class.
//
//////////////////////////////////////////////////////////////////////

#include <map>
#include <string>
#include <set>

#ifdef __hppa
#include <rw/exception.h>
#include <stdlib.h>
#endif

#include "spobject.h"
#include "spexcept.h"
#include "spengcon.h"
#include "S_y_tab.h"
#include "notifyc.h" //Engine notification
#include "sqpe.h"

#ifndef __hppa
#pragma warning(disable:4786)
#endif

#ifndef WIN32
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#endif

using namespace std;

BOOL SpValidEvaluator(void)
{
	return (S_evaluator != NULL && 
			  SqpeIsValidAddress((void*)S_evaluator, sizeof(s_evaluator)) );
}

// Access to the evaluator data
long SpGetCurrentFrame()
{
	if ( !SpValidEvaluator() )
		SCONNECT_ThrowException(SCONNECT_INVALID_ENGINE_CONNECTION);
	return S_evaluator->_Nframe;
}
//Low level debug tool: 
//if you have access to sqpe/spdebug.h define TRACE_OBJ 
//to define TraceObj() to trace an object in the MSDEV's Debug Windows.

#if defined(WIN32) && defined(_DEBUG)
void TraceObj(s_object* ps_object)
{
	SpTraceObj(ps_object);
}
#endif

//Get frame number for the bucket containing this header
long SCONNECT_get_header_frame(s_object* ps_object, s_evaluator* S_evaluator)
{
	vector_in_bucket* pHeaderInBucket = (vector_in_bucket*) ps_object;
	S_bucket* ps_bucket =  pHeaderInBucket->my_bucket;
	if(!SqpeIsValidAddress(ps_bucket, sizeof(S_bucket)))
	//	|| (ps_bucket->Type != BUCKET_TYPE))//261==BUCKET_TYPE
		SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);
	
	return ps_bucket->my_frame;
}

//Get frame number for the bucket containing this header
long CSPobject::GetHeaderFrame(BOOL bValidate) const
{
	if(bValidate)
	 Validate();
	return ::SCONNECT_get_header_frame(m_ps_object, S_evaluator);
}

/////////////////////////////////////////////////////////////////////////////////////////
// Object map support for use in S_notification to call virtual fn on notification 

class CSPobjectmap : public map<string, CSPobject *, less<string> >
{
public:
	CSPobjectmap() { clear(); };
	virtual ~CSPobjectmap() {};

	void Insert(const char *pszObjectName, CSPobject *pCSPobject)
	{
		insert(value_type(pszObjectName, pCSPobject));
	}
};

CSPobjectmap global_mapPersistentObjects;

CSPobject * CSPobject::ObjectMap_Get( const char *pszObjectName )
{
	CSPobject *pCSPobjectFound = NULL;
	if ( pszObjectName && *pszObjectName )
	{
		CSPobjectmap::iterator iter = global_mapPersistentObjects.find( pszObjectName );
		if ( iter != global_mapPersistentObjects.end() )
			pCSPobjectFound = (*iter).second;
	}
	return pCSPobjectFound;
}

BOOL CSPobject::ObjectMap_Add( const char *pszObjectName, CSPobject *pCSPobject )
{
	BOOL bSuccess = FALSE;

	if ( pszObjectName && *pszObjectName )
	{
		global_mapPersistentObjects.Insert(pszObjectName, pCSPobject);
		bSuccess = TRUE;
	}
	return bSuccess;
}

BOOL CSPobject::ObjectMap_Remove( const char *pszObjectName )
{
	BOOL bSuccess = FALSE;
	if ( global_mapPersistentObjects.empty() )
		return bSuccess;

	if ( pszObjectName && *pszObjectName )
	{
		bSuccess = ( global_mapPersistentObjects.erase( pszObjectName ) > 0 ) ? TRUE:FALSE;
		//TODO: remove from the engine's hash table?
	}
	return bSuccess;
}

BOOL CSPobject::ObjectMap_RemoveAll( void )
{
	BOOL bSuccess = TRUE;
	if ( global_mapPersistentObjects.empty() )
		return bSuccess;

	//TODO: remove from the engine's hash table?

	global_mapPersistentObjects.clear();
	return bSuccess;
}

// Return TRUE if one of objects in the "name" map is parent of ps_child
BOOL CSPobject::ObjectMap_Exists(const s_object* ps_child) 
{
	BOOL bFound = FALSE;
	if ( !global_mapPersistentObjects.empty() )
	{
		CSPobjectmap::iterator iter;
		iter=global_mapPersistentObjects.begin();
		while(!bFound && (iter!=global_mapPersistentObjects.end()))
		{
			CSPobject* pCSPobject = (CSPobject *) (*iter).second;
			if ( pCSPobject && pCSPobject->IsValid())
				bFound = pCSPobject->IsParentOf(ps_child);
			iter++;
		}
	}
	return bFound;
}

// End of object map support
/////////////////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////////////////
// Unnamed object set support for use in CSPengineConnect::CloseTopLevelEval() to
// invalidate CSPobject member m_ps_object 

class CSPunnamedObjectMap : public set<long>
{
public:
	CSPunnamedObjectMap() { clear(); };
	virtual ~CSPunnamedObjectMap() {};

	void Insert(CSPobject *pCSPobject)
	{
		insert( value_type( (long)((void *)pCSPobject) ) );
	}
};

CSPunnamedObjectMap global_mapUnnamedObjects;
CSPunnamedObjectMap global_mapUnnamedLocalObjects;

BOOL CSPobject::UnnamedObjectMap_Add( CSPobject *pCSPobject )
{
	BOOL bSuccess = FALSE;

	if ( pCSPobject )
	{
		if ( pCSPobject->IsValid() )
		{
			if((pCSPobject->GetHeaderFrame(FALSE) > 0) //Local object?
			||(pCSPobject->GetStorageFrame(FALSE) > 0))
				global_mapUnnamedLocalObjects.Insert(pCSPobject);
			else
				global_mapUnnamedObjects.Insert(pCSPobject);
			bSuccess = TRUE;
		}
	}
	return bSuccess;
}

BOOL CSPobject::UnnamedObjectMap_Remove( CSPobject *pCSPobject )
{
	BOOL bSuccess = FALSE;

	if ( pCSPobject )
	{
		if ( !global_mapUnnamedLocalObjects.empty() )
		{
			CSPunnamedObjectMap::iterator iter = global_mapUnnamedLocalObjects.find( (long)((void *)pCSPobject) );
			if ( iter != global_mapUnnamedLocalObjects.end() )
				global_mapUnnamedLocalObjects.erase( iter );
		}

		if ( !global_mapUnnamedObjects.empty() )
		{
			CSPunnamedObjectMap::iterator iter = global_mapUnnamedObjects.find( (long)((void *)pCSPobject) );
			if ( iter != global_mapUnnamedObjects.end() )
				global_mapUnnamedObjects.erase( iter );
		}
	}
	return bSuccess;
}

BOOL CSPobject::UnnamedObjectMap_RemoveAll( void )
{
	BOOL bSuccess = TRUE;

	if (!global_mapUnnamedLocalObjects.empty() )
		global_mapUnnamedLocalObjects.clear();

	if (!global_mapUnnamedObjects.empty() )
		global_mapUnnamedObjects.clear();
	return bSuccess;
}


BOOL CSPobject::UnnamedObjectMap_InvalidateAll( void )
{
	BOOL bSuccess = TRUE;

	if ( !global_mapUnnamedLocalObjects.empty() )
	{
		CSPunnamedObjectMap::iterator iter;
		for (iter=global_mapUnnamedLocalObjects.begin();iter!=global_mapUnnamedLocalObjects.end();iter++)
		{
			void *vpElement = (void *)(*iter);
			if ( vpElement )
			{
				CSPobject *pCSPobject = (CSPobject *)vpElement;
				pCSPobject->m_ps_object = NULL;
			}
		}
		global_mapUnnamedLocalObjects.clear();
	}

	if ( !global_mapUnnamedObjects.empty() )
	{
		CSPunnamedObjectMap::iterator iter;
		for (iter=global_mapUnnamedObjects.begin();iter!=global_mapUnnamedObjects.end();iter++)
		{
			void *vpElement = (void *)(*iter);
			if ( vpElement )
			{
				CSPobject *pCSPobject = (CSPobject *)vpElement;
				pCSPobject->m_ps_object = NULL;
			}
		}
		global_mapUnnamedObjects.clear();
	}

	return bSuccess;
}

//Remove local objects from the unnamed map or make them permanent if needed.
BOOL CSPobject::UnnamedObjectMap_InvalidateLocalObjects( BOOL bError )
{
	BOOL bSuccess = TRUE;
	if ( global_mapUnnamedLocalObjects.empty() )
		return bSuccess;

	CSPunnamedObjectMap::iterator iter;
	iter=global_mapUnnamedLocalObjects.begin();
	while(iter!=global_mapUnnamedLocalObjects.end())
	{
		CSPobject *pCSPobject = (CSPobject *) (*iter);
		if ( pCSPobject )
		{
			if(bError) //error? 
			{ //some local objects could have been corrupted.
				pCSPobject->m_ps_object = NULL;
			}
			else if ( !pCSPobject->IsValid() )
					pCSPobject->m_ps_object = NULL;
			else if((pCSPobject->GetHeaderFrame(FALSE) > 0) ||(pCSPobject->GetStorageFrame(FALSE) > 0))
			{	//no error?
				//make the S local object permanent so that its corresponding CSPobject be valid until its destructor is called.
				s_object* ps_new_object = pCSPobject->ClonePermanent() ;
				pCSPobject->DecrRef();
				pCSPobject->m_ps_object = ps_new_object;
				pCSPobject->SetTryToFreeOnDetach(TRUE);
				pCSPobject->IncrRef();
				global_mapUnnamedObjects.Insert(pCSPobject);
			}
		}
//  The erase() method return value differs between MSVC++ and other 
//  STL implementations.
#ifdef WIN32
		iter = global_mapUnnamedLocalObjects.erase( iter ); //iter gets next elem via .erase()
#else
		CSPunnamedObjectMap::iterator tmp_iter(iter);
		++iter;
		global_mapUnnamedLocalObjects.erase( tmp_iter );
#endif
	}
	return bSuccess;
}

//Remove local objects from the unnamed map or make them permanent if needed.
BOOL CSPobject::UnnamedObjectMap_PopFrame( long lParentFrame )
{
	BOOL bSuccess = TRUE;
	if ( global_mapUnnamedLocalObjects.empty() )
		return bSuccess;

	long lPopFrame = set_alloc(lParentFrame, S_evaluator);
	if(lPopFrame <= 1L)
		CSPobject::UnnamedObjectMap_InvalidateLocalObjects( FALSE );
	else if(lParentFrame<lPopFrame)
	{
		CSPunnamedObjectMap::iterator iter;
		iter=global_mapUnnamedLocalObjects.begin();
		while(iter!=global_mapUnnamedLocalObjects.end())
		{
			CSPobject *pCSPobject = (CSPobject *) (*iter);
			if ( pCSPobject )
			{
				if ( !pCSPobject->IsValid() )
					pCSPobject->m_ps_object = NULL;
				else if((pCSPobject->GetHeaderFrame(FALSE) >= lPopFrame)||(pCSPobject->GetStorageFrame(FALSE) >= lPopFrame))
				{	//no error?
					//Move the object to its parent frame
					s_object* ps_new_object = ::move_data(&(*pCSPobject), lPopFrame, S_evaluator);
					pCSPobject->DecrRef();
					pCSPobject->m_ps_object = ps_new_object;
					pCSPobject->IncrRef();
					global_mapUnnamedObjects.Insert(pCSPobject);
				}
			}
//  The erase() method return value differs between MSVC++ and other 
//  STL implementations.
#ifdef WIN32
			iter = global_mapUnnamedLocalObjects.erase( iter ); //iter gets next elem via .erase()	
#else
			CSPunnamedObjectMap::iterator tmp_iter(iter);
			++iter;
			global_mapUnnamedLocalObjects.erase( tmp_iter );
#endif
		}
	}
	else
	{
		::set_alloc(lPopFrame, S_evaluator); //Reset
		SCONNECT_ThrowException("internal error: lParentFrame=%d >= lPopFame=%d", lParentFrame, lPopFrame);
	}
	
	::set_alloc(lPopFrame, S_evaluator); //Reset
	return bSuccess;
}


// Return TRUE if one of objects in the "name" map is parent of ps_child
BOOL CSPobject::UnnamedObjectMap_Exists(const s_object* ps_child) 
{
	BOOL bFound = FALSE;
	if ( !global_mapUnnamedObjects.empty() )
	{
		CSPunnamedObjectMap::iterator iter;
		iter=global_mapUnnamedObjects.begin();
		while(!bFound && (iter!=global_mapUnnamedObjects.end()))
		{
			CSPobject *pCSPobject = (CSPobject *) (*iter);
			if ( pCSPobject && pCSPobject->IsValid())
				bFound = pCSPobject->IsParentOf(ps_child);
			iter++;
		}
	}
	return bFound;
}

// End of unnamed object set support
/////////////////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////////////////
// Single session engine connection support

CSPengineConnect * CSPobject::m_pEngineConnection = NULL;

CSPengineConnect * CSPobject::GetEngineConnection( void )
{
	return (CSPengineConnect*)m_pEngineConnection;
}

void CSPobject::SetEngineConnection( CSPengineConnect *pEngineConnection )
{
	m_pEngineConnection = pEngineConnection;
}

// End of engine connection support
/////////////////////////////////////////////////////////////////////////////////////////


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CSPobject::	CSPobject()
{
	Init();
}

//Construct from a valid S-expression
CSPobject::CSPobject(const char* pszExpression)
{
	Init();
	Create(pszExpression);
	m_bTryToFreeOnDetach = TRUE;
}

//Construct from an S object
CSPobject::	CSPobject(s_object* ps_object, BOOL bTryToFreeOnDetach)
{
	Init();
	Attach(ps_object, bTryToFreeOnDetach);
}

CSPobject::CSPobject(const CSPobject& sObject)
{
	Init();
	Attach(&sObject, sObject.GetTryToFreeOnDetach()); //Copy appropriate data member of sobject
}

void CSPobject::Init( void )
{
	m_ps_object = NULL;
	m_bTryToFreeOnDetach=FALSE;
	m_pszObjectName = NULL;
	m_bEnableOnModify = TRUE;
	m_bEnableOnRemove = TRUE;
}

CSPobject::~CSPobject()
{
	// This CSPobject is going to be smashed: be sure to remove it
	// from any unnamed maps that contain it. (bug#24097)
	CSPobject::UnnamedObjectMap_Remove( this );	

	if(m_bTryToFreeOnDetach )
		Release(); //free object if possible when count drops to zero
	else
		Detach();

	// Dealloc name
	if ( m_pszObjectName )
	{
		CSPobject::ObjectMap_Remove(m_pszObjectName);
		free( m_pszObjectName );
		m_pszObjectName = NULL;
		if(m_ps_object)
			::s_unhashed_add(m_ps_object);
	}
}

/////////////////////////////////////////////////////////////////////////////////////////
// Notification support

// Override these in your inherited CSPobject classes for S_MSG_MODIFY and 
//  S_MSG_REMOVE messages from the engine
// Last argument, ps_attached, is a S-PLUS attached object that uniquely 
//  identfies the database where the object resides
int CSPobject::OnModify( s_object *ps_objectOld, s_object *ps_objectNew, s_object* ps_attached )
{
	Attach( ps_objectNew );
	return 1;
}

int CSPobject::OnRemove( s_object *ps_objectOld, s_object* ps_attached )
{
	Release();
	return 1;
}

int CSPobject::OnPreModify( s_object **pps_object )
{
	SqpeTrace0( "WARNING: in CSPobject::OnPreModify()\n" );
	return 1;
}

long CSPobject::OnNotifyOrQuery(
	long lMsg,        /* message Id */
	long lArgs,       /* number of args */
	void** ppArgs    /* content depends on lMsg with length lArgs */
)
{
	SqpeTrace0( "WARNING: in CSPobject::OnNotifyOrQuery()\n" );
	return 0;
}

long  S_EngineMessage(
	long lMsg,        /* message Id */
	long lArgs,       /* number of args */
	void** ppArgs)    /* content depends on lMsg with length lArgs */
{
	long lResult = 0;

	try
	{
		if(lMsg == S_MSG_UNCACHED && lArgs == 2 )
		{
			//Tell the Engine don't uncache (destroy) this object if it is needed by the client.
			long lNeeded= CSPobject::ObjectMap_Exists((const s_object*) ppArgs[1])? 1L : 0L;
			if(!lNeeded)
				lNeeded = CSPobject::UnnamedObjectMap_Exists((const s_object*) ppArgs[1])? 1L : 0L;
					
			return lNeeded;
		}
		else if (lMsg <= S_MSG_SETOPT || S_MSG_SENDBASE <= lMsg && lMsg <= S_MSG_SENDLAST)
		{
			CSPobject *psCSPobject = CSPobject::GetEngineConnection();
			if ( psCSPobject )
			{
				return psCSPobject->OnNotifyOrQuery(lMsg, lArgs, ppArgs);
			}
			else
			{	// Notify SPL objects
				switch(lMsg)
				{
					case S_MSG_EXCEPTION:
					{
						/* Engine is about to do a long jump.  Set s_object pointers
							 to NULL in all unnamed CSpObjects
						*/	
						CSPobject::UnnamedObjectMap_InvalidateLocalObjects(TRUE);
					}	
					break;

					case S_MSG_MODIFY:
					{
						if ( lArgs != 4 )
							break;

						CSPobject *psCSPobject = CSPobject::ObjectMap_Get( (const char*)ppArgs[0] );
						if ( psCSPobject )
							lResult = psCSPobject->OnModify( (s_object*)ppArgs[1], (s_object*)ppArgs[2], (s_object*)ppArgs[3] );
					}
					break;

					case S_MSG_PREMODIFY:
					{
						if ( lArgs != 2 )
							break;
								
						// Notify the SPL object attached to the s_object
						CSPobject *psCSPobject = CSPobject::ObjectMap_Get( (const char*)ppArgs[0] );
						if ( psCSPobject )
							lResult = psCSPobject->OnPreModify( (s_object**)ppArgs[1] );
					}
					break;

					case S_MSG_REMOVE:
					{
						if ( lArgs != 3 )
							break;
						
						CSPobject *psCSPobject = CSPobject::ObjectMap_Get( (const char*)ppArgs[0] );
						if ( psCSPobject )
							lResult = psCSPobject->OnRemove( (s_object*)ppArgs[1], (s_object*)ppArgs[2] );
					}
					break;
				}
			}
		}
	}
	catch(...)
	{
	}
	return lResult;
}

// End of notification support
/////////////////////////////////////////////////////////////////////////////////////////


//////////////////////////////////////////////////////////////////////
//S_SyncParseEval: Parse and Evaluate an S expression
//This global function call does not notify client.
//To notify client use CSPobject::SyncEval()
//////////////////////////////////////////////////////////////////////

s_object* S_SyncParseEval_GetWarnings(const char* pszExpression, s_object **warningList)
{
	return SPL_ParseEval_GetWarnings(pszExpression, warningList);
}

s_object* S_SyncParseEval(const char* pszExpression)
{
	return SPL_ParseEval(pszExpression);
}

// Clone an object
CSPobject CSPobject::Clone(BOOL bValidate) const
{
	if(bValidate)
	 Validate();
	CSPobject sObject(SPL_Clone( m_ps_object, FALSE ));
	sObject.m_bTryToFreeOnDetach = TRUE; //Let it be free-able when ref-count drops to zero.
	return sObject;
}

// Clone an object and make permanent
CSPobject CSPobject::ClonePermanent( BOOL bValidate) const
{
	if(bValidate)
	 Validate();

	CSPobject sObject(SPL_Clone( m_ps_object, TRUE ));
	sObject.m_bTryToFreeOnDetach = TRUE; //Let it be free-able when ref-count drops to zero.
	return sObject;
}

//////////////////////////////////////////////////////
// Attributes
//////////////////////////////////////////////////////

long CSPobject::GetNCol(BOOL bValidate) const
{
	if(bValidate)
	 Validate();
	return 0;
}

long CSPobject::GetNRow(BOOL bValidate) const
{
	if(bValidate)
	 Validate();
	return 0;
}

BOOL CSPobject::IsValid(void) const
{
	return (!SPL_NotThere(m_ps_object) && ::SqpeIsValidAddress(m_ps_object, 1) && Sqpe_GetCurrentEvaluator() != NULL );
}

BOOL CSPobject::IsValidAddress(void) const
{
	return (m_ps_object && ::SqpeIsValidAddress(m_ps_object, 1));
}

void CSPobject::Validate(void) const
{
	if (!IsValid()) 
		throw SCONNECT_INVALID_SOBJECT;
}

s_object* CSPobject::GetVector(long lIndex) const
{
	return GetPtr(); // call GetPtr() to ensure we don't destroy object when ref. count drops to zero
} 

const CSPobject& CSPobject::operator= (const CSPobject& sobject)
{
	DecrRef();
	Attach(sobject);
	return *this;
}

long CSPobject::GetStorageFrame(BOOL bValidate) const
{ 
	return ::get_storage_frame(m_ps_object, S_evaluator);
}

//Return S_MODE: double, integer, char, etc..
int CSPobject::GetMode(BOOL bValidate) const
{
	if(bValidate)
	 Validate();
	return (int) m_ps_object->mode;
}

long CSPobject::GetLength(BOOL bValidate) const
{
	if(bValidate)
	 Validate();
	return m_ps_object->length;
}

BOOL CSPobject::IsSharing(BOOL bValidate) const
{
	if(bValidate)
	 Validate();
	return ::multiple_ref_count(const_cast<s_object*>(m_ps_object), S_evaluator);
}

// Increment reference counting
#ifdef _DEBUG
long CSPobject::m_lTotalRefCount=0;
#endif

void CSPobject::IncrRef(void) const
{
	if(CSPobject::IsValid())
	{
		m_ps_object->Type = S_REF_TYPE; //Confirm our assumption that CSPobject is always ref. counting
		::incr_ref_count(m_ps_object, S_TRUE, m_ps_object->frame, S_evaluator);
	}
	#ifdef _DEBUG
	++m_lTotalRefCount;
	#endif
}

//decrement reference counting
//   On Linux, the "const" in the method signature prevents modification
//   of m_ps_object.
#ifdef WIN32
void CSPobject::DecrRef(BOOL bTryToFree) const
#else
void CSPobject::DecrRef(BOOL bTryToFree)
#endif
{
	#ifdef _DEBUG
	--m_lTotalRefCount;
	#endif
	try
	{
		if(CSPobject::IsValid())
		{
			s_object* ps_frame = NULL;
			if(bTryToFree) 
			{
				if(m_ps_object->frame)
					ps_frame = m_ps_object->frame;
				else
				{
					CSPevaluator sEvaluator;
					ps_frame = sEvaluator.GetCurrentEvalFramePtr();
					//Try to free!
					m_ps_object->frame = ps_frame;
					::decr_ref_count(m_ps_object, S_TRUE, ps_frame, S_evaluator);
					if(m_ps_object->Type == S_FREED_TYPE)
#ifdef WIN32
						const_cast<s_object*>(m_ps_object) = NULL;
#else
						m_ps_object = NULL;
#endif
					else
						m_ps_object->frame = NULL; //reset.
					return;
				}
			}
			::decr_ref_count(m_ps_object, S_TRUE, ps_frame, S_evaluator);
			if(m_ps_object->Type == S_FREED_TYPE)
#ifdef WIN32
				const_cast<s_object*>(m_ps_object) = NULL;
#else
				m_ps_object = NULL;
#endif
		}
	}
	catch(...)
	{
	}
}
int CSPobject::GetRefCount(void) const
{
	return ::get_ref_count(m_ps_object, S_evaluator);
}

//Attach: 1) release the old object pointer if it is valid, 
//        2) get new header for the object 
//        3) incr. ref . counting
void CSPobject::Attach(s_object *ps_object, BOOL bTryToFreeOnDetach)
{
	if ( ps_object == m_ps_object )
		return;
	if(m_ps_object != NULL)
		DecrRef(); //decr ref count, but don't every destroy it and its chidren
	if(ps_object == NULL)
	{
		m_ps_object = NULL;
		return;
	}
	//It is probably better to create a new header, e.g.
	//m_ps_object = ::s_header_for_object(ps_object, &lFrame, S_evaluator);
	//but it would requires ObjectMap_Add() to use the Value member pointer of s_object.
	m_ps_object = ps_object;
	m_bTryToFreeOnDetach = bTryToFreeOnDetach;
	IncrRef();
	//The object must have perm. storage before we can put it in the Named Map
	long lStorageFrame = ::get_storage_frame(ps_object, S_evaluator);
	long lHeaderFrame = ::SCONNECT_get_header_frame(ps_object, S_evaluator);
	if(m_pszObjectName && (*m_pszObjectName) && ( lHeaderFrame < 0 && lStorageFrame<1))
	{
		//Make sure we remove it from the unnamed maps
		if( CSPobject::UnnamedObjectMap_Exists((const s_object*) this->m_ps_object))
			CSPobject::UnnamedObjectMap_Remove(this);
		//Add this object to the named map 
		m_ps_object = ps_object;
		CSPobject::ObjectMap_Add( m_pszObjectName, this );
	}
	else if( !CSPobject::ObjectMap_Exists((const s_object*) this->m_ps_object)) //not already in the named map?
		CSPobject::UnnamedObjectMap_Add( this );
	return;
}
//Detach: dec ref counting, return the s_object pointer.
s_object* CSPobject::Detach(BOOL bTryToFree)
{
	//Remove this object from whatever map that contains it.
	CSPobject::UnnamedObjectMap_Remove( this );		
	CSPobject::ObjectMap_Remove( m_pszObjectName );

	s_object* ps_object = NULL;
	if(IsValid())
	{
		DecrRef(bTryToFree); //Decr. ref. but do not delete the object
		if(IsValid())
			ps_object = m_ps_object;
	}
	else if(m_ps_object != NULL && ( m_ps_object->mode==S_MODE_NULL || m_ps_object->mode==S_MODE_MISSING))
		ps_object = m_ps_object; //OK to return an S NULL or MISSING object

	m_ps_object = NULL;
	m_bTryToFreeOnDetach = TRUE; //nothing to free anyway
	return ps_object;
}

//Release: Detach and try delete the object if requested
s_object* CSPobject::Release(BOOL bTryToFree)
{
	if(!m_ps_object)
		return NULL;
	return Detach(bTryToFree);
}


//Copy if needed but always returns a new header.
s_object* CSPobject::CopyForWrite(BOOL bValidate) 
{
	if(bValidate)
		Validate();
	CSPevaluator sEvaluator;
	return sEvaluator.CloneIfNeeded(sEvaluator.CopyForWrite(m_ps_object));
}
s_object* CSPobject::CreateNewHeader(BOOL bValidate) const
{
	if(bValidate)
		Validate();
	CSPevaluator sEvaluator;
	return sEvaluator.CreateNewHeader(m_ps_object);
}

//////////////////////////////////////////////////////////////////////
//Print: Evaluate method Show for an S expression
//m_ps_object is expected to be a valid S expression
//////////////////////////////////////////////////////////////////////

CSPobject CSPobject::Print(BOOL bAsText) const
{
	CSPobject sAns;

	// IsValid checks for missing and null class
	if(!IsValid())
		return sAns;

	try
	{
		CSPevaluator sEvaluator; //Open frame 1 and set error jump right-here if needed.
		s_object* ps_obj = const_cast<s_object*>(m_ps_object);
		if ( (m_ps_object->name != NULL) && (*m_ps_object->name != '\0') )
			ps_obj = alc_name(m_ps_object->name, S_evaluator);

		s_object* ps_call = NULL;

		if ( bAsText )
		{
			ps_call = alcvec(S_MODE_FUN_CALL, 3L, S_evaluator);
			LIST_POINTER(ps_call)[0] = alc_name("as.character.or.error", S_evaluator); 
			LIST_POINTER(ps_call)[1] = ps_obj;

			s_object* ps_doprint = NEW_LOGICAL(1);
			LOGICAL_POINTER(ps_doprint)[0] = 0;
			ps_doprint->name = (char*)make_name("doprint", S_evaluator)->text;
			LIST_POINTER(ps_call)[2] = ps_doprint;
		}
		else
		{
			ps_call = alcf((char*)"print", ps_obj, S_evaluator);
		}
		s_object* ps_eval = alcf((char*)"eval", ps_call, S_evaluator);
		s_object* ps_ans = sEvaluator.Eval(ps_eval);			 
		
#ifndef WIN32
		CSPengineConnect* pConnection = GetEngineConnection();
			
		if( pConnection )
		{
			struct stat buf;
			long nStdOutLength = 0;
			long nStdErrLength = 0;
			// unused: BOOL bHaveOutput = FALSE;
			
			usleep(1);
			
			long lOutput = pConnection->GetOSHdStdout();
			if( lOutput > 0 )
			{
				if(fstat(lOutput, &buf) == 0)
					nStdOutLength = buf.st_size;
				else
					nStdOutLength = 0;
			}
	
			lOutput = pConnection->GetOSHdStderr();
			if( lOutput > 0 )
			{
				if(fstat(lOutput, &buf) == 0)
					nStdErrLength = buf.st_size;
				else
					nStdErrLength = 0;
			}
			
			if( nStdOutLength || nStdErrLength )
			{
				pConnection->OnOutput(0);
			}
		}
#endif
		
		sAns.Attach(ps_ans);
		sAns.m_bTryToFreeOnDetach = TRUE; //Let it be free-able when ref-count drops to zero.
	}
	catch(...)
	{
		sAns.m_ps_object = NULL;
	}	
	return sAns;
}

//Create an S object and coerce it to the specified class
BOOL CSPobject::BaseCreate(
	s_class* ps_class,         //class name 
	const char* pszExpression, //==NULL, uses the default constructor, else use the expression to instantiate it.
	const char* pszName,       //object name
	long lDataBase )				   //database
{
	BOOL bSuccess = TRUE;
	s_object* ps_object = NULL;
	try
	{
		CSPevaluator sEvaluator;
		if(pszExpression != NULL && (*pszExpression!=0))
		{
			//Eval the expression to S object (without output and warning list return) 
			ps_object = SPL_DirectParseEval( pszExpression );
			if( !ps_object )
				SCONNECT_ThrowException("Failed to evaluate expression: %s", pszExpression); 
			//Coerce if the result is not exactly match the desired class
			if(ps_class && (!ps_object->Class || (ps_object->Class->index != ps_class->index)))
			{
					ps_object = AS(ps_object, ps_class);
					if ( !ps_object )
						SCONNECT_ThrowException("Failed to coerce object of class %s to class %s", GET_CLASS_NAME(ps_object), ps_class);
			}
		}
		else if(ps_class)
		{
			//Construct a new object with the default constructor of the SV4 class
			ps_object = ::new_S_object(ps_class, S_evaluator);
			if ( !ps_object )
				SCONNECT_ThrowException("Failed to create object of class %s", ps_class->name);
		}
		else
		{
			Attach(NULL);
			return FALSE;
		}
		//If persistent object, assign to disk, else call Attach
		if( pszName != NULL && *pszName && lDataBase >= 0 )
		{
			Attach(ps_object, TRUE); //coerece and attach
			CSPobject::UnnamedObjectMap_Remove( this ); //remove from the map because Assign will puts this into its map
			Assign(pszName, lDataBase);
		}
		else
		{
			//Clone to perm frame if needed
			ps_object = sEvaluator.CloneIfNeeded(ps_object);
			Attach(ps_object, TRUE);
		}
	}
	catch(CSPexception& pExcept)
	{
		pExcept.Print();
		bSuccess = FALSE;
	}
	catch(...)
	{
		bSuccess = FALSE;
	}
	return bSuccess;
}

// Create an S object in a perm. frame
BOOL CSPobject::BaseCreate(
	const char* pszClass,      //class name 
	const char* pszExpression, //==NULL, uses the default constructor, else use the expression to instantiate it.
	const char* pszName,       //object name
	long lDataBase )				   //database
{
	if ( !pszClass || !*pszClass )
		SCONNECT_ThrowException("Invalid class specification");
	s_class* ps_class = ::make_s_class(pszClass, 0, S_evaluator);
	return BaseCreate(ps_class, pszExpression, pszName, lDataBase);
}
//Create an S object and attach via the virtual member function Attach() which expect to coerce to required class
BOOL CSPobject::Create(const char* pszExpression,	const char* pszName,	long lDataBase )
{
	//Let the virtual methods Attach() coerces the object to the derive class
	return BaseCreate((s_class*)NULL, pszExpression, pszName, lDataBase);
}
//Get the persistant object in the S databases: the first one in the search list will be returned.
//Return TRUE if succeded else FALSE.
BOOL CSPobject::get(const char* pszName /* object name */)
{
	if( !pszName || !*pszName)
		return FALSE;

	CSPobject* pObject = CSPobject::ObjectMap_Get(pszName);
	if(pObject != NULL)
	{
		if(pObject != this) 
			return FALSE; //Already connected to antoher CSPobject
		else 
			return TRUE;  //Already connect to this
	}

	CSPevaluator sEvaluator;
	s_object* ps_object = sEvaluator.get(pszName, FALSE, TRUE);
	
	// Release previous obj 
	Release();

	// Dealloc previous name 
	if ( m_pszObjectName )
	{
		free( m_pszObjectName );
		m_pszObjectName = NULL;
	}

	m_pszObjectName = strdup( pszName );
	Attach(ps_object);

	if(IsValid())
		return TRUE;
	else
	{
		Release();	
		return FALSE;
	}
}

// Commit permanent object to database after any modifications 
// NOTE: Assigns modified object to database 1 by default !
BOOL CSPobject::Commit( long lDataBase )
{
	if ( !IsValid() )
		return FALSE;
	if ( !m_pszObjectName || !*m_pszObjectName )
		return FALSE;
	Assign( m_pszObjectName, lDataBase );
	return TRUE;
}

// Remove permanent object from database 
BOOL CSPobject::Remove( void )
{
	try
	{
		CSPevaluator sEvaluator;

		if ( !CSPobject::IsValid() )
			return FALSE;
		if ( !m_pszObjectName || !*m_pszObjectName )
		{
			//If this is permanent, unnamed object, try to remove it.
			if(GetHeaderFrame(FALSE) <= 0)
			{
				try_to_free(m_ps_object, -1, Nframe, S_evaluator);	
				m_ps_object = NULL;
				return TRUE;
			}
			else
				return FALSE;
		}
		int db_found;
		if ( !db_remove_named(&db_found, m_ps_object, m_pszObjectName, S_evaluator) )
			return FALSE;

		// Dealloc name 
		if ( m_pszObjectName )
		{
			free( m_pszObjectName );
			m_pszObjectName = NULL;
		}
		return TRUE;
	}
	catch(...)
	{
	}
	return FALSE;
}

//////////////////////////////////////////////////////////////////////
//Assign/reassign and commit to disk an S object to a perm. frame assoc. with a name.
// 1. the given object name will be assigned/reassigned  
// 2. the object is commited to disk
// 3. it is cached in the permanent frame assoc with lDataBase
//
void CSPobject::Assign(const char* pszName, long lDataBase)
{
	Validate();

	if ( (pszName == NULL) && (lDataBase > 0) )
		SCONNECT_ThrowException("A name is required in order to assign an object to a database");		

	//Remove me from the maps
	CSPobject* psExist = CSPobject::ObjectMap_Get(pszName);
	if(psExist!=NULL && psExist != this)
	{
		if(psExist->IsValid())
			SCONNECT_ThrowException("A Connect/C++ object is already attached to object %s. Only one Connect/C++ object can be attached to an S object at a time.", pszName, pszName);
		else
			CSPobject::ObjectMap_Remove(pszName);
	}
	try
	{
		CSPevaluator sEvaluator;
		// To ensure that there is one-to-one mapping between name and s_object* in case of
		// re-assignment of an existing object name,  
		// 1. db_put_data() will write to disk using psName.
		// 2. S_Notification() will be called from the engine to sychronize with the client.
		
		//LOCK_DB_LIST
		
		//Assign it to a perm. database.  
		//Note that the return value should have a perm. storage frame.
		long lSearchPos = -1;
		if(lDataBase > 0)
			lSearchPos = S_evaluator->_search_data_index[lDataBase-1];
		else if(lDataBase < 0)
			SCONNECT_ThrowException("Invalid database %d", lDataBase);			
		s_object* ps_object = db_put_data(pszName, lSearchPos, this->m_ps_object, S_evaluator);		
		//UNLOCK_DB_LIST				
		// Store the object name in this class for later use in Attach() and Detach() 
		if ( m_pszObjectName && strcmp(m_pszObjectName,pszName)!=0 )
		{
			CSPobject::ObjectMap_Remove(m_pszObjectName);
			free( m_pszObjectName );
			m_pszObjectName = NULL;
		}
		if(m_pszObjectName == NULL)
			m_pszObjectName = strdup( pszName );
		//Assume correct class and just call the base class method.  
		//Don't call the derive class method.  It would lead to recursion.
		CSPobject::Attach(ps_object); 
	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	catch(...)
	{
	}
}

//Re-attach and re-assign if needed.
void CSPobject::AttachAndAssign(s_object *ps_object, const char* pszName, long lDataBase)
{
	//Assign and/or attach
	if(pszName && (*pszName))
	{
		m_ps_object	= ps_object;
		Assign(pszName, lDataBase);
	}
	else
		Attach(ps_object);
}

//Re-attach and re-assign if needed.
void CSPobject::ReAttachAndAssign(s_object *ps_object, long lDataBase)
{
	if(m_ps_object == ps_object) //same object header?
		return;
	try
	{
		Attach(ps_object, TRUE); //Coerce and attach/reattach
		if(m_pszObjectName && (*m_pszObjectName))
		{
			CSPevaluator sEvaluator;
			long lSearchPos = -1;
			if(lDataBase > 0)
				lSearchPos = S_evaluator->_search_data_index[lDataBase-1];
			else if(lDataBase < 0)
				SCONNECT_ThrowException("Invalid database %d", lDataBase);
			//Remove from unnamedMap because it was added via Attach() above.
			CSPobject::ObjectMap_Add(m_pszObjectName, this);

			if(ps_object == NULL)
			{
				OnModify(m_ps_object, NULL, NULL); 
				m_ps_object = NULL;
			}
			else
			{
				s_object* ps_new_object = db_put_data(m_pszObjectName, lSearchPos, ps_object, S_evaluator);
				//expect notification to be called and fixup the member pointer to s_object*
				if(ps_new_object != m_ps_object)
					SCONNECT_ThrowException("Notification failure"); 
			}
			//no need to call CSPobject::ObjectMap_Add(m_pszObjectName ) because Attach() will call it.
		}
	}
	catch(CSPexception& except)
	{
		// re-throw
		throw except;
	}
	catch(...)
	{
		// re-throw
		SCONNECT_ThrowException("Failed to reattach and assign S-Plus object"); 
	}
}

//Coerce the data to class given by pszClass and attach
BOOL CSPobject::BaseCoerceAttach(s_object* ps_object, s_class* ps_class, BOOL bTryToFreeOnDetach)
{
	BOOL bSuccess = TRUE;

	if(ps_object == m_ps_object)
		return bSuccess;

	if(ps_object == NULL) //Attach a NULL pointer?
	{
		Detach();
		return bSuccess;
	}
	s_object* ps_newObject = ps_object;
	if ( GET_CLASS(ps_object) != ps_class )
	{
		try
		{
			//Must protect object before write
			CSPevaluator sEvaluator;
			if(sEvaluator.GetRefCount(ps_object) != 0 || PRECIOUS(ps_object))
				ps_newObject = sEvaluator.CopyForWrite(ps_object);
			ps_newObject = sEvaluator.CloneIfNeeded(sEvaluator.As(ps_object, ps_class));
		}
		catch(...)
		{
			ps_newObject = NULL;
			bSuccess = FALSE;
		}
	}

	CSPobject::Attach(ps_newObject, bTryToFreeOnDetach);
	return bSuccess;
}

//Coerce the data to class given by pszClass and attach
BOOL CSPobject::BaseCoerceAttach(s_object* ps_object, const char* pszClass, BOOL bTryToFreeOnDetach)
{
	return BaseCoerceAttach(ps_object, MAKE_CLASS(pszClass), bTryToFreeOnDetach);
}

//Coerce the data to class given by pszClass	
BOOL CSPobject::BaseCoerce(const char* pszClass)
{
	return BaseCoerceAttach(m_ps_object, pszClass);
}

char * CSPobject::AllocateStringInFrame( const char *pszString, BOOL bValidate )
{
	if(bValidate)
	 Validate();
	return SPL_AllocateStringInFrame( m_ps_object, pszString );
}

const s_class* CSPobject::GetClass(BOOL bValidate) const
{
	if(bValidate)
	 Validate();
	return m_ps_object->Class;
}

// Test for class membership.
s_boolean CSPobject::Is(const char* pszClass) const
{
	// unused: s_boolean bRet = S_FALSE;
	if ( !IsValid() )
		return S_FALSE;
	if ( !pszClass || strlen(pszClass) == 0 )
		return S_FALSE;

	s_class* ps_class = make_s_class(pszClass, 0, S_evaluator);
	long index = -1;
	if ( ps_class )
		index = ps_class->index;
	return ((s_boolean)isObjectIndex(m_ps_object, index,  m_ps_object->Class, S_evaluator));
}
	
//Return TRUE if this object is a parent of ps_child
BOOL SPL_IsParentOf(const s_object* ps_parent, const s_object* ps_child)
{
	BOOL bFound= FALSE;

	if((ps_parent == NULL) || (ps_child == NULL)) //undetermine?
		bFound = FALSE;
	if(ps_child == ps_parent) //are we the same guy?
		bFound = TRUE;
	else if(!S_NOT_RECURSIVE(ps_parent->mode)) // are you a parent?
	{
		for(long nTree = 0; nTree < ps_parent->length; nTree++)
		{
			bFound = SPL_IsParentOf(ps_parent->value.tree[nTree], ps_child);
			if(bFound)
				break;
		}
	}
	return bFound;
}

//Return TRUE if this object is a parent of ps_child
BOOL CSPobject::IsParentOf(const s_object* ps_child) const
{
	return SPL_IsParentOf(m_ps_object, ps_child);
}

//Return TRUE if this object is a child of ps_parent
BOOL CSPobject::IsChildOf(const s_object* ps_parent) const
{
	return SPL_IsParentOf(ps_parent, m_ps_object);
}

void CSPobject::SetObjectName( const char *pszObjectName, BOOL bAddToMap )
{
	if ( m_pszObjectName )
	{
#ifdef WIN32
		if ( stricmp( m_pszObjectName, pszObjectName ) != 0 )
#else
		if ( strcasecmp( m_pszObjectName, pszObjectName ) != 0 )
#endif
		{
			free( m_pszObjectName );
			m_pszObjectName = NULL;
		}
	}
	if(m_pszObjectName == NULL)
	{
		if ( bAddToMap )
		{
			CSPobject* pObject = CSPobject::ObjectMap_Get(pszObjectName);
			if ( pObject && pObject != this )
				SCONNECT_ThrowException("A Connect/C++ object already exists in the name map", pszObjectName);

			CSPobject::UnnamedObjectMap_Remove( this );
			CSPobject::ObjectMap_Add(pszObjectName, this);
		}

		m_pszObjectName = strdup( pszObjectName );
	}
}

//GetSearchPathPosition() returns database position (same id as S-function search())
//If non-permanent object, it returns S_NOT_A_DB.
long CSPobject::GetSearchPathPosition( BOOL bValidate ) const
{
	if(bValidate)
	 Validate();
	return SPL_GetSearchPathPosition( const_cast<s_object*>(m_ps_object) );
}

BOOL CSPobject::EnableOnModify( BOOL bEnable )
{
	BOOL bPrevEnable = m_bEnableOnModify;
	m_bEnableOnModify = bEnable;
	return bPrevEnable;
}

BOOL CSPobject::EnableOnRemove( BOOL bEnable )
{
	BOOL bPrevEnable = m_bEnableOnRemove;
	m_bEnableOnRemove = bEnable;
	return bPrevEnable;
}

// Returns a character object containing the deparse
//  of the object
CSPobject CSPobject::Deparse(BOOL bValidate)
{
	if(bValidate)
	 Validate();
	CSPobject sAns;
	try
	{
		CSPevaluator sEvaluator;

		s_object* ps_arg = s_header_for_object(m_ps_object, NULL, S_evaluator);
		s_copy_if_needed(ps_arg, S_FALSE, S_evaluator);
		ps_arg->name = (char *)make_name("expr", S_evaluator)->text;
		s_object* ps_deparse = alcf((char*)"deparse", ps_arg, S_evaluator);
		s_object* ps_ans =  sEvaluator.Eval(ps_deparse);

		sEvaluator.TryToFree(ps_deparse);

		sAns.Attach(sEvaluator.CloneIfNeeded(ps_ans));
		sAns.m_bTryToFreeOnDetach = TRUE; //Let it be free-able when ref-count drops to zero.
	}
	catch(...)
	{
		sAns.m_ps_object = NULL;
	}	
	return sAns;
}

BOOL SPL_NotThere(const s_object* ps_object )
{
	return (ps_object==NULL || !SqpeIsValidAddress((void*)ps_object, 1)
				|| ps_object->Type == S_FREED_TYPE || ps_object->mode==S_MODE_NULL || 
			  ps_object->mode==S_MODE_MISSING);
}
BOOL SPL_IsAtomicVector(const s_object* ps_object )
{
	return ( ps_object?ps_object->mode>0 && ps_object->mode<S_MODE_ANY && 
				ps_object->mode!=S_MODE_LIST && ps_object->mode!=S_MODE_RAW:FALSE );
}

// Gets new header and copies data if needed
s_object* SPL_ProtectVector(s_object* ps_vector, BOOL bCopy)
{
	long lFrame = S_evaluator->_cur_frame;
	ps_vector = ::s_header_for_object(ps_vector, NULL, S_evaluator);
	if ( bCopy )
	{
		// force a copy
		int iType = ps_vector->Type;
	   *ps_vector = *copy_lev(ps_vector, 0, NULL_ENTRY, S_evaluator);
		ps_vector->Type = iType;
	}
	else
		s_copy_if_needed(ps_vector, S_FALSE, S_evaluator);
	if ( lFrame != S_evaluator->_cur_frame )

		set_alloc(lFrame,  S_evaluator);
	return ps_vector;
}
// Gets new header increments reference count. Used when inserting an object into another
s_object* SPL_NewHeaderForInsertion(s_object* ps_vector)
{
	ps_vector = s_header_for_object(ps_vector, NULL, S_evaluator);
	// Set frame pointer to NULL and let ReAttachAndAssign determine frame
	incr_ref_count(ps_vector, S_TRUE, NULL, S_evaluator);
	return ps_vector;
}

s_object* SPL_NewHeaderForInsertion2(s_object* ps_vector)
{
	long lFrame = ::get_storage_frame(ps_vector, S_evaluator);
	ps_vector = s_header_for_object(ps_vector, &lFrame, S_evaluator);
	// Set frame pointer to NULL and let ReAttachAndAssign determine frame
	incr_ref_count(ps_vector, S_TRUE, FRAME_POINTER(lFrame), S_evaluator);
	return ps_vector;
}

