# Standard Perl modules:
use Getopt::Long ;
use Cwd qw(cwd abs_path);
use File::Basename ;
use File::Copy ;
use File::Path ;

# Splus Perl modules:
use Splus::Vars ;
use Splus::src2bin ;
use Splus::packup ;
use Splus::SplusUtils ;
use Splus::Utils ;

use strict;

my $opt_help;
my $opt_library;
my $opt_global;
GetOptions (
    "library|l:s" => \$opt_library,
    "help|h"      => \$opt_help,
    'global!'     => \$opt_global
    ) || usage() ;

if($opt_help) {
    usage() ;
    exit() ;
}

my $SHOME = ${Splus::Vars::SHOME} ;
my $S_TMPDIR = ${Splus::Vars::S_TMPDIR} ;
my $S_PATHSEP = ${Splus::Vars::S_PATHSEP} ;
# my $TARCMD = "c:/cygwin/bin/tar" ;
# my $UNZIPCMD = "c:/cygwin/bin/unzip" ;
my @UNGZTARCMD = qw(tar -xzf) ;
my @UNZIPCMD = qw(unzip -q) ;

my $cwd = cwd() ;
my $dirpath = dirname($cwd) ;

my $dest ;
if($opt_library) {
    $dest = $opt_library ;
    my $S_LIBS=$ENV{'S_LIBS'};
    # print "Changing S_LIBS from $S_LIBS\n" ;
    my @slibs_list = split($S_PATHSEP, $S_LIBS) ;
    push(@slibs_list, abs_path($dest)) ;
    $S_LIBS=join($S_PATHSEP, @slibs_list) ;
    $ENV{'S_LIBS'} = $S_LIBS ;
    # print "  to " . $ENV{'S_LIBS'} . "\n";
} else {
    my @libraries=get_libraries_list($opt_global) ; # in SplusUtils.pm
    $dest = @libraries[0];
}

if(! $ARGV[0]) {
    die "Error: No pkg archive or directory supplied\n" ;
}

foreach my $pkg (@ARGV) {
    my $pkgname  ;
    my $pkg_was_archive = 0 ;
    my @install_tmp_dirs = () ;
    print "Installing package \"$cwd/$pkg\" to library directory \"$dest\"\n" ;
    chdir($cwd) or die "Cannot change directory to \"$cwd\": $!" ;

    if(! -d $pkg) {
        # Must be an archive, unpack in $S_TMPDIR
        die "Cannot read archive file \"$pkg\"\n" unless (-r $pkg) ;
        # Figure out archive type:
        my ($pkgbase, $dirname, $suffix) = fileparse($pkg, (".zip", ".tar.gz")) ;
        die "Do not recognize archive file extension $suffix\n" unless ($suffix) ;
        # Extract package name:
        $pkgname = (split(/_/, $pkgbase))[0] ;
        # Copy archive to $S_TMPDIR before unpacking:
        copy($pkg, $S_TMPDIR . "/" . $pkg) ;
        chdir($S_TMPDIR) or die "Cannot change directory to \"$S_TMPDIR\"\n" ;
        # If directory with package name exists, rename it to a unique name
        #   made from time and process id:
        if(-d $pkgname) {
	    rename($pkgname, $pkgname . ".$^T$$") ;
        }
        my @cmd ;
        if($suffix eq ".zip") {
            @cmd = @UNZIPCMD ;
        } elsif ($suffix eq ".tar.gz") {
            @cmd = @UNGZTARCMD ;
        } else {
	    die "Internal error: suffix not recognized in $pkg (expect 'zip' or 'tar.gz')\n" ;
        }
        push @cmd, $pkg ;
        system(@cmd) == 0 or die "Could not successfully run \"@cmd\": $!\n" ;
        $pkg_was_archive = 1 ;
    } else {
        # pkg is package directory, name should be package name:
        $pkgname = $pkg;
        -e $pkgname . "/DESCRIPTION" or die "No DESCRIPTION file found in $pkgname\n" ;
        my $tmpdcf = Splus::Dcf->new($pkgname . "/DESCRIPTION") ;
        if (! $tmpdcf->{Built} ) {
            # First copy to tmp directory for building.
            my $blddir = Splus_tempfile("Sbld") ;
            mkdir $blddir or die "Cannot make temporary build directory $blddir ($!)" ;
	    push @install_tmp_dirs, $blddir ;
            Splus::SplusUtils::copy_source_package($pkgname, $blddir, "native") ;
            my $internal_pkgname = $tmpdcf->{"Package"} ;
            if ("$internal_pkgname" ne basename($pkgname)) {
                rename($blddir."/".basename($pkgname), $blddir."/".$internal_pkgname) or die "Cannot rename misnamed package directory to its internal name ($!)" ;
                $pkgname = $blddir . "/" . $internal_pkgname ;
            } else {
                $pkgname = $blddir . "/" . basename($pkgname) ;
            }
        }
    }

    # Look for DESCRIPTION file:
    -e $pkgname . "/DESCRIPTION" or die "No DESCRIPTION file found in $pkgname\n" ;
    my $dcf = Splus::Dcf->new($pkgname . "/DESCRIPTION") ;
    
    # Check if package is binary i.e. has a Built tag in the DESCRIPTION file,
    #   if not, build it with src2bin:
    my $built = $dcf->{Built} ;
    if(! $built) {
        my @arglist = ("--name", basename($pkgname),
                       "--parent", dirname($pkgname),
                       "--clean-after", "--update-description") ; 
        my $sb = Splus::src2bin->new(@arglist) ;
        $sb->make() ;
    }
    
    my $fullarchivename ;
    my $archivename ;
    my $pkg_was_binary_archive = 0 ;
    if ($pkg_was_archive && $built) {
        # no need to pack up again, use original zip/tar.gz file
        $pkg_was_binary_archive = 1 ;
        $fullarchivename = $pkg ;
    } else {
        # Now have a built package in $pkgname, packup into a new archive:
        my @arglist = ("--name", basename($pkgname),
                       "--parent", dirname($pkgname),
                       "--type", "binary");
        my $pu = Splus::packup->new(@arglist) ;
        $pu->make() ;
        $fullarchivename = $pu->{compressed_name} ;
        $pkg_was_binary_archive = 0 ;
    }
    $archivename = basename($fullarchivename) ;
    
    # Check if dest is OK, first making sure we are where we started:
    chdir($cwd) ;
    -d $dest or die "Installation Library location, $dest, is not a directory\n" ;
    -w $dest or die "Cannot write to installation library location, $dest\n" ;
    
    # Copy $archivename to $dest, unpack and then delete $archivename:
    copy($fullarchivename, $dest . "/" . $archivename) or die "Cannot copy $fullarchivename to $dest/$archivename: $!\n" ;
    chdir($dest) or die "Cannot change directory to $dest: $!\n" ;
    my @cmd ;
    if ($archivename =~ /\.zip$/i) {
	    @cmd = @UNZIPCMD ;
    } elsif ($archivename =~ /\.tar\.gz$/i) {
            @cmd = @UNGZTARCMD ;
    } else {
	    die "Internal error: suffix not recognized in $archivename (expect 'zip' or 'tar.gz')\n" ;
    }
    push @cmd, $archivename ;
    
    my $base_pkgname = basename($pkgname) ; # !!! should redo the variable names
       # The package name should be the thing in DESCRIPTION
    my $pkgname_already_existed = -e $base_pkgname ;
    my $pkgname_backup ;
    if ($pkgname_already_existed) {
        $pkgname_backup = "$base_pkgname-old-$^T$$" ;
        File::Path::rmtree("$pkgname_backup") if -e "$pkgname_backup" ;
        print "Renaming old $base_pkgname to $pkgname_backup in directory $dest\n" ;
        rename($base_pkgname, $pkgname_backup) or die "Cannot rename old $base_pkgname to $pkgname_backup in directory $dest" ;
    }
    # print "About to do: @cmd\n";
    if (system(@cmd) != 0) {
        if ($pkgname_already_existed) {
            print "Restoring old $base_pkgname from $pkgname_backup in directory $dest\n";
            rename($pkgname_backup, $base_pkgname) or die "Cannot restore old $base_pkgname from $pkgname_backup in directory $dest\n" ;
        }
        die "Could not successfully run \"@cmd\": $!\n" ;
    }
    if ($pkgname_already_existed) {
        File::Path::rmtree("$pkgname_backup") ;
        print "Removed old $pkgname_backup from $dest\n";
    }
    unlink($archivename) ;
    foreach my $dir (@install_tmp_dirs) {
        File::Path::rmtree("$dir")
    }
}
    
sub usage() {
    print "Usage: Splus CMD INSTALL [options] pkg1 [pkg2 ...]\n" ;
    print "\nInstall a Spotfire S+ package from an archive or a directory.\n" ;
    print "pkg can be a relative or absolute path to a directory with the\n" ;
    print "package source, or to a zipped package archive.  The library\n" ;
    print "tree to install to can be specified via '--library'.  By default,\n" ;
    print "packages are installed in the library tree rooted at the first\n" ;
    print "directory given in the environment variable S_LIBS if this is \n" ;
    print "set and non-null.   Otherwise, if --global is specified \n" ;
    print "\$SHOME/local/library and by default or if --no-global is\n" ;
    print "specified, a library directory under the user's home directory\n";
    print "(", get_default_user_library(), ").\n" ;

    print "\nOptions:\n" ;
    print "-h,--help     # print this help message\n" ;
    print "-l,--library  # location to install package in\n" ;
    print "   --global   # If \$S_LIBS not defined, default library is\n";
    print "              # \$SHOME/local/library.\n" ;
    print "   --noglobal # (default) If \$S_LIBS not defined, default\n";
    print "              # library is ", get_default_user_library(), "\n" ;

    @_ && die join(", ", @_) ;
    exit() ;
}
