/*
 * restrict_fork.so -- for secure kiosk mode firefox
 *
 * A. Gebhardt <agebhard@uni-klu.ac.at>
 *
 * was originally installwatch.c:
 *
 * Copyright (C) 1998-9 Pancrazio `Ezio' de Mauro <p@demauro.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 * compile:
 * cc -c -fpic restrict_fork.c && ld -shared restrict_fork.o -o restrict_fork.so -ldl
 *
 * use:
 * add 
 *    export LD_PRELOAD=`pwd`/restrict_fork.so 
 * to the firefox start script.
 */


#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <stdlib.h>
#define __USE_GNU
#include <dlfcn.h>
#include <errno.h>
#include <unistd.h>
#include <stdarg.h>


#define MAX_BUFLEN 256
#define MAX_ARGS   100
#define DEBUG 1  /* 0, 1, 2 , 3 */

#define LOG(_call)							   \
  snprintf(buf, MAX_BUFLEN, "about to execute an %s command ...\n", _call);\
  write(fd, buf, strlen(buf));						   \
  fsync(fd); 

#define LOG_COMMAND(_cmd,_call)						 \
  snprintf(buf, MAX_BUFLEN, "%s command: %s\n", _call, _cmd);		 \
  write(fd, buf, strlen(buf));						 \
  fsync(fd);

#define LOG_ALLOW(_cmd,_call)						 \
  snprintf(buf, MAX_BUFLEN, "... %s command allowed: %s\n",_call, _cmd);	 \
  write(fd, buf, strlen(buf));						 \
  fsync(fd);

#define LOG_DENY(_cmd, _call)	       				         \
  snprintf(buf, MAX_BUFLEN, "*** %s command denied: %s\n",_call, _cmd);	 \
  write(fd, buf, strlen(buf));						 \
  fsync(fd);


static char* paths[]={
  "/bin/",
  "/usr/bin/",
  "/usr/bin/X11/",
  "/usr/local/bin/",
  NULL
};
int n_paths;

static char* commands[]={
  "ls",
  "dirname",
  "basename",
  "expr",
  "uname",
  "id", 
  "grep",
  "pgrep",
  "cat",
  "sed",
  "lpr",
  "xdvi",
  "evince",
  "test",
  "mplayer",
  "chromium-browser",
  "sleep",
  "readlink",
  "xdg-settings",
  "xdg-open",
  "xdg-mime",
  "mimeopen",
  "dbus-send",
  "xprop",
  NULL
};
int n_commands;

static char* full_named_commands[]={
  "/usr/lib/chromium-browser/chromium-browser",
  "/usr/lib/chromium-browser/chromium-browser-sandbox",
  "/proc/self/exe",
  "/usr/lib/chromium-browser/libpdf.so",
  "/usr/lib/x86_64-linux-gnu/gstreamer0.10/gstreamer-0.10/gst-plugin-scanner",
  NULL
};
int n_full_named_commands;

static int(*true_popen)(const char *command, const char *type);
static int(*true_system)(const char *command);
static int(*true_execv)(const char *path, char *const argv[]);
static int(*true_execve)(char *filename, char *const argv [], char *const envp[]); 
static int(*true_execvp)(const char *file, char *const argv[]);
static int(*true_execl)(const char *path, const char *arg, ...);
static int(*true_execlp)(const char *file, const char *arg, ...);
static int(*true_execle)(const  char  *path,  const  char  *arg  , ...);

void _init(void)
{
  void *libc_handle;
  int fd;
  char buf[MAX_BUFLEN];


#if (DEBUG>0)
  fd=open("/tmp/restrict-fork.log", O_WRONLY | O_APPEND | O_CREAT, 0644);
#endif
#if (DEBUG>=2)
  snprintf(buf, MAX_BUFLEN, "intialization started\n");
  write(fd, buf, strlen(buf));
  fsync(fd);
#endif	

  libc_handle = RTLD_NEXT;

#if (DEBUG>=2)
  snprintf(buf, MAX_BUFLEN, "libc dlopened\n");
  write(fd, buf, strlen(buf));
  fsync(fd);
#endif	
	
  true_popen  = dlsym(libc_handle, "popen");
  true_system = dlsym(libc_handle, "system");
  true_execv  = dlsym(libc_handle, "execv");
  true_execve = dlsym(libc_handle, "execve"); 
  true_execvp = dlsym(libc_handle, "execvp"); 
  true_execl  = dlsym(libc_handle, "execl"); 
  true_execlp = dlsym(libc_handle, "execlp"); 
  true_execle = dlsym(libc_handle, "execle"); 

  n_paths=0;
  while(paths[n_paths]){
#if (DEBUG>=2)
    snprintf(buf, MAX_BUFLEN, "path: %s\n",paths[n_paths]);
    write(fd, buf, strlen(buf));
    fsync(fd);
#endif	
    n_paths++;
  }

  n_commands=0;
  while(commands[n_commands]){
#if (DEBUG>=2)
    snprintf(buf, MAX_BUFLEN, "command: %s\n",commands[n_commands]);
    write(fd, buf, strlen(buf));
    fsync(fd);
#endif
    n_commands++;
  }

  n_full_named_commands=0;
  while(full_named_commands[n_full_named_commands]){
#if (DEBUG>=2)
    snprintf(buf, MAX_BUFLEN, "full_named_command: %s\n",full_named_commands[n_full_named_commands]);
    write(fd, buf, strlen(buf));
    fsync(fd);
#endif
    n_full_named_commands++;
  }

#if (DEBUG>=2)
  snprintf(buf, MAX_BUFLEN, "intialization finished\n");
  write(fd, buf, strlen(buf));
#endif	
#if (DEBUG>0)
  fsync(fd);
  close(fd);
#endif	
}


static int allow_command(char* command, char* callname){ 

  int fd;
  char buf[MAX_BUFLEN];
  int ret;
  int i, j, cmd_len, test_cmd_len, test_path_len;
  char *test_command;



#if (DEBUG>0)
  fd=open("/tmp/restrict-fork.log", O_WRONLY | O_APPEND | O_CREAT, 0644);
#endif
#if (DEBUG>=1)
  LOG_COMMAND(command,callname);
#endif
  cmd_len=strlen(command);


  /* full specified commands */
#if (DEBUG>=2)
  snprintf(buf, MAX_BUFLEN, "test full names:\n");
  write(fd, buf, strlen(buf));
  fsync(fd);
#endif  
  for (i=0;i<n_full_named_commands;i++){
    test_cmd_len=strlen(full_named_commands[i]);
#if (DEBUG>=2)
      snprintf(buf, MAX_BUFLEN, "test_command: %s\n",full_named_commands[i]);
      write(fd, buf, strlen(buf));
      fsync(fd);
#endif  
    if(cmd_len == test_cmd_len){
      if(strncmp (command, full_named_commands[i], cmd_len)==0){
	goto allow;
      } 
    }
  }


  /* short named commands */
#if (DEBUG>=2)
  snprintf(buf, MAX_BUFLEN, "test short names:\n");
  write(fd, buf, strlen(buf));
  fsync(fd);
#endif  
  for (i=0;i<n_commands;i++){
    test_cmd_len=strlen(commands[i]);
#if (DEBUG>=2)
      snprintf(buf, MAX_BUFLEN, "test_command: %s\n",commands[i]);
      write(fd, buf, strlen(buf));
      fsync(fd);
#endif  
    if(cmd_len == test_cmd_len){
      if(strncmp (command, commands[i], cmd_len)==0){
	goto allow;
      }
    }
  }

  /* short named commands combined with pathnames */
#if (DEBUG>=2)
  snprintf(buf, MAX_BUFLEN, "test short names + paths:\n");
  write(fd, buf, strlen(buf));
  fsync(fd);
#endif  
 
  for (i=0;i<n_commands;i++){
    for (j=0;j<n_paths;j++){
      test_path_len=strlen(paths[j]);
      test_cmd_len=strlen(commands[i]);
      test_command=(char*)malloc((test_path_len+test_cmd_len+1)*sizeof(char));
      bzero(test_command,test_path_len+test_cmd_len+1);
      strncpy(test_command,paths[j],test_path_len);
      strncat(test_command,commands[i],test_cmd_len);
#if (DEBUG>=3)
      snprintf(buf, MAX_BUFLEN, "testing: (%i,%i) %s(%i) %s(%i)\n--> %s\n",i,j,paths[j],test_path_len,commands[i],test_cmd_len, test_command);
      write(fd, buf, strlen(buf));
      fsync(fd);
#endif


#if (DEBUG>=2)
      snprintf(buf, MAX_BUFLEN, "test_command: %s\n",test_command);
      write(fd, buf, strlen(buf));
      fsync(fd);
#endif  
      if(cmd_len == test_cmd_len+test_path_len){
	if(strncmp (command, test_command, test_cmd_len+test_path_len)==0){
	  goto allow;
	}
      }
      free(test_command);
    }
  }
  
 deny:
#if (DEBUG>=1)
  LOG_DENY(command,callname);
  close(fd);
#endif	
  errno = EACCES;
  return 0;
  
 allow:
#if (DEBUG>=1)
  LOG_ALLOW(command,callname);
  close(fd);
#endif	
  return 1;
}

FILE *popen(const char *command, const char *type){
  const char *call="popen";
  int fd;
  FILE* ret;
  char buf[MAX_BUFLEN];
  
#if (DEBUG>=2)
  fd=open("/tmp/restrict-fork.log", O_WRONLY | O_APPEND | O_CREAT, 0644);
  LOG(call);
  close(fd);
#endif

  if(allow_command(command,call)==1){
    ret = (FILE*) true_popen(command, type);
  } else {
    ret = (FILE*)0;
    errno = EACCES;
  }
  return ret;
}

int system(const char *command){ 
  const char *call="system";
  int fd;
  int ret;
  char buf[MAX_BUFLEN];
  
#if (DEBUG>=2)
  fd=open("/tmp/restrict-fork.log", O_WRONLY | O_APPEND | O_CREAT, 0644);
  LOG(call);
  close(fd);
#endif

  if(allow_command(command,call)==1){
    ret = (int) true_system(command);
  } else {
    ret = -1;
    errno = EACCES;
  }
  return ret;
}

int execv(const char *path, char *const argv[]){
  const char *call="execv";
  int fd;
  int ret;
  char buf[MAX_BUFLEN];
  
#if (DEBUG>=2)
  fd=open("/tmp/restrict-fork.log", O_WRONLY | O_APPEND | O_CREAT, 0644);
  LOG(call);
  close(fd);
#endif

  if(allow_command(path,call)==1){
    ret = (int) true_execv(path, argv);
  } else {
    ret = -1;
    errno = EACCES;
  }
  return ret;
}

int execve(const char *path, char *const argv[], char *const envp[]){
  const char *call="execve";
  int fd;
  int ret;
  char buf[MAX_BUFLEN];
  
#if (DEBUG>=2)
  fd=open("/tmp/restrict-fork.log", O_WRONLY | O_APPEND | O_CREAT, 0644);
  LOG(call);
  close(fd);
#endif

  if(allow_command(path,call)==1){
    ret = (int) true_execve(path, argv, envp);
  } else {
    ret = -1;
    errno = EACCES;
  }
  return ret;
}

int execvp(const char *file, char *const argv[]){
  const char *call="execvp";
  int fd;
  int ret;
  char buf[MAX_BUFLEN];
  
#if (DEBUG>=2)
  fd=open("/tmp/restrict-fork.log", O_WRONLY | O_APPEND | O_CREAT, 0644);
  LOG(call);
  close(fd);
#endif

  if(allow_command(file,call)==1){
    ret = (int) true_execvp(file, argv);
  } else {
    ret = -1;
    errno = EACCES;
  }
  return ret;
}

int execl(const char *path, const char *arg, ...){
  const char *call="execl";
  int fd;
  int ret;
  char buf[MAX_BUFLEN];
  char *argv[MAX_ARGS];
  int argno=0;
  va_list ap;

#if (DEBUG>=2)
  fd=open("/tmp/restrict-fork.log", O_WRONLY | O_APPEND | O_CREAT, 0644);
  LOG(call);
  close(fd);
#endif

  if(allow_command(path,call)==1){
    /* translate execl to execv: */
    va_start(ap, arg);
    argv[0]=arg;
    while ((argv[argno++] = va_arg(ap, char *)) != (char *)0)
      ;
    va_end(ap);
    ret = (int) true_execv(path, argv );

  } else {
    ret = -1;
    errno = EACCES;
  }
  return ret;
}

int execle(const  char  *path,  const  char  *arg  , ...){
  const char *call="execle";
  int fd;
  int ret;
  char buf[MAX_BUFLEN];
  char *argv[MAX_ARGS];
  char **envp;
  int argno=0;
  va_list ap;

#if (DEBUG>=2)
  fd=open("/tmp/restrict-fork.log", O_WRONLY | O_APPEND | O_CREAT, 0644);
  LOG(call);
  close(fd);
#endif

  if(allow_command(path,call)==1){
    /* translate execle to execve: */
    va_start(ap, arg);
    argv[0] = arg;
    while ((argv[argno++] = va_arg(ap, char *)) != (char *)0)
      ;
    envp = va_arg(ap, char **);
    va_end(ap);
    ret = (int) true_execve(path, argv, envp);

  } else {
    ret = -1;
    errno = EACCES;
  }
  return ret;
}
