(*******************************************************************************)
(* GeoProof, an interactive geometry tool writen in OCaml.                    *)
(* Copyright (C) 2005 Julien Narboux                                           *)
(*                                                                             *)
(* This program is free software; you can redistribute it and/or               *)
(* modify it under the terms of the GNU General Public License                 *)
(* as published by the Free Software Foundation; either version 2              *)
(* of the License, or (at your option) any later version.                      *)
(*                                                                             *)
(* This program is distributed in the hope that it will be useful,             *)
(* but WITHOUT ANY WARRANTY; without even the implied warranty of              *)
(* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               *)
(* GNU General Public License for more details.                                *)
(*                                                                             *)
(* You should have received a copy of the GNU General Public License           *)
(* along with this program; if not, write to the Free Software                 *)
(* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. *)
(*******************************************************************************)

(* $Id: objets_repere.ml,v 1.27 2005/11/16 17:19:35 jnarboux Exp $ *)

open Geometric_functions.GeometryFunctions
open Types_de_base
open Fonctions
open Repere
open Options
open Optionsdeconfiguration
open Objets_graphiques
open List

class type t_objet_repere =
  object
    inherit t_objet_graphique
    method def_langue_naturelle_corps : unit -> string
    method recalcule : unit
    method depend_de : t_objet_graphique -> bool
    method dependance : t_objet_graphique list
    method objet_type :  t_type_objet 
    method paint : Cairo.t -> Repere.t -> unit
    method est_proche : Repere.t -> t_coords -> bool
    method movable : unit -> bool

    method output_car_body : unit -> Xml.xml
    method output_kig_body : unit -> Xml.xml list
    method output_drg_body : unit -> Xml.xml
    method output_coq_body : unit -> Fol.fol Formulas.formula option
    method output_eukleides_body : unit -> string
    method output_pst_eucl_body : unit -> string
    method predicate_form : unit -> Fol.fol Formulas.formula option
    method defining_points : unit -> (string * string) option
  end
 
class objet_repere =
  object (self)
    inherit objet_graphique

    val precision = 2 (* Number of digits of the grid labels *)

    method private grid_lines_list () =  
      let rep = Repere.get() in
      let i n = papier_vers_ecran_t rep {x = n; y = of_int 0} in
      let j n = papier_vers_ecran_t rep {x = of_int 0; y = n} in
      let l = ref [] in
      
      
      let deltax = x_max() -! x_min() in
      let deltay = y_max() -! y_min() in
      let delta = min deltax deltay in
      let step = delta /!  (of_int 10) in

      let linex k = 
	(((i k).x,(of_int 0)),
	 ((i k).x,!screen_width),
	 ((i k).x +! two,(i k).y -! (of_int 3)),
	 (to_string k precision)) 
      in
      let liney k = 
	((of_int 0,(j k).y),
	 (!screen_width,(j k).y),
	 ((j k).x +! (of_int 3),(j k).y -! two),
	 (to_string k precision)) 
      in

      let k = ref step in
      while (!k <! (x_max()))
      do
	l:= (linex !k)::!l;
	k := !k +! step
      done;
      let k = ref (neg step) in
      while (!k >! (x_min()))
      do
	l := (linex !k)::!l;
	k := !k -! step
      done;
      
      let k = ref step in
      while (!k <! (y_max()))
      do
	l := (liney !k)::!l;
	k := !k +! step
      done;
      let k = ref (neg step) in
      while (!k >! (y_min()))
      do
	l := (liney !k)::!l;
	k := !k -! step
      done;
      !l

    method paint cr rep =    
      let origine =  origine_ecran_float rep in 
      let screen_width_float = to_float !screen_width 5 in
      let screen_height_float = to_float !screen_height 5 in

      
      (* grid drawing *)
	(match self#objet_style with
	    STYLE_REPERE style -> if style.grid_shown then
	      begin
		Cairo.save cr;
		Cairo.set_source_rgb cr 0.8 0.8 0.8;
		Cairo.set_line_width cr 1.;
		Cairo.set_dash cr [|7.;3.|] 0.; 
		List.iter 
		  (fun (p,q,r,s) ->	
		     Cairo.move_to cr (to_float (fst r) 5) (to_float (snd r) 5);
		     Cairo.show_text cr s;
		     Cairo.move_to cr (to_float (fst p) 5) (to_float (snd p) 5);
		     Cairo.line_to cr (to_float (fst q) 5) (to_float (snd q) 5)
		  ) 
		  
		  (self#grid_lines_list ());
		Cairo.stroke cr;
		Cairo.restore cr
	      end
	   | _ -> ()) (* assert false TODO *);
	
	(* axis drawing *)
	Cairo.move_to cr 0. origine.yfloat;
	Cairo.line_to cr ~x:screen_width_float ~y:origine.yfloat;
	Cairo.move_to cr ~x:origine.xfloat ~y:0.;
	Cairo.line_to cr origine.xfloat screen_height_float;
	Cairo.stroke cr

    method est_proche rep c =
      let origine = origine_ecran_t rep in
      let proche = (of_int  !!pixels_proche)  /! two in
      abs (c.x -! origine.x) <! proche ||
      abs (c.y -! origine.y) <! proche	

    method objet_type = TYPE_REPERE

    method dependance = []

    method recalcule = ()

    method def_langue_naturelle_corps () = " some axes"^
      match self#objet_style with
	STYLE_REPERE style ->  if style.grid_shown then
	  " with a grid"
	else ""
      | _ -> "" (* assert false TODO  *)   
	    
    method output_car_body () =
      Xml.Element("",[],[])

    method output_kig_body () =
      [Xml.Element("",[],[])]
    
    method output_coq_body () = None

    method output_drg_body () =
      let grid_shown =
	match self#objet_style with
	    STYLE_REPERE style ->  (string_of_bool style.grid_shown) 
	  | _ -> assert false
      in
	
	Xml.Element("Axes",[
		      ("name",self#objet_nom);
                      ("grid_shown",grid_shown)
		    ],[])
	  

    method output_eukleides_body () = ""
    method output_pst_eucl_body () = ""

    method movable () = false
    method predicate_form () = None
    method defining_points () = None

    initializer self#recalcule
  end
